use std::cmp::{Ord, Ordering, PartialOrd};
use std::collections::{BTreeSet, HashSet, VecDeque};

use std::fmt;
use std::ops;

use anyhow::Result;
use pest::iterators::{Pair, Pairs};
use pest::Parser;
use serde::de;
use serde::Deserialize;

use crate::errors;
use crate::grammar::{ExpressionIdentifier, Scenario};
use crate::variants::model::AlleleFreq;

#[derive(Shrinkwrap, Clone, Copy, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub(crate) struct IUPAC(u8);

impl IUPAC {
    pub(crate) fn contains(&self, base: u8) -> bool {
        if base == **self {
            return true;
        }
        match **self {
            b'R' if base == b'A' || base == b'G' => true,
            b'Y' if base == b'C' || base == b'T' => true,
            b'S' if base == b'G' || base == b'C' => true,
            b'W' if base == b'A' || base == b'T' => true,
            b'K' if base == b'G' || base == b'T' => true,
            b'M' if base == b'A' || base == b'C' => true,
            b'B' if base == b'C' || base == b'G' || base == b'T' => true,
            b'D' if base == b'A' || base == b'G' || base == b'T' => true,
            b'H' if base == b'A' || base == b'C' || base == b'T' => true,
            b'V' if base == b'A' || base == b'C' || base == b'G' => true,
            b'N' => true,
            _ => false,
        }
    }
}

#[derive(Parser)]
#[grammar = "grammar/formula.pest"]
pub(crate) struct FormulaParser;

#[derive(PartialEq, Eq, Clone, Debug, Hash)]
pub(crate) enum Formula {
    Conjunction {
        operands: Vec<Formula>,
    },
    Disjunction {
        operands: Vec<Formula>,
    },
    Negation {
        operand: Box<Formula>,
    },
    Atom {
        sample: String,
        vafs: VAFSpectrum,
    },
    Variant {
        positive: bool,
        refbase: IUPAC,
        altbase: IUPAC,
    },
    Expression {
        identifier: ExpressionIdentifier,
        negated: bool,
    },
}

#[derive(PartialEq, Eq, Clone, Debug, Hash)]
pub(crate) enum NormalizedFormula {
    Conjunction {
        operands: Vec<NormalizedFormula>,
    },
    Disjunction {
        operands: Vec<NormalizedFormula>,
    },
    Atom {
        sample: String,
        vafs: VAFSpectrum,
    },
    Variant {
        positive: bool,
        refbase: IUPAC,
        altbase: IUPAC,
    },
}

impl Formula {
    /// Negate formula.
    pub(crate) fn negate(&self, scenario: &Scenario, contig: &str) -> Result<Formula> {
        Ok(match self {
            Formula::Conjunction { operands } => Formula::Disjunction {
                operands: operands
                    .iter()
                    .map(|o| Ok(o.negate(scenario, contig)?))
                    .collect::<Result<Vec<Formula>>>()?,
            },
            Formula::Disjunction { operands } => Formula::Conjunction {
                operands: operands
                    .iter()
                    .map(|o| Ok(o.negate(scenario, contig)?))
                    .collect::<Result<Vec<Formula>>>()?,
            },
            Formula::Negation { operand } => operand.as_ref().clone(),
            &Formula::Variant {
                positive,
                refbase,
                altbase,
            } => Formula::Variant {
                positive: !positive,
                refbase,
                altbase,
            },
            Formula::Expression {
                identifier,
                negated,
            } => Formula::Expression {
                identifier: identifier.clone(),
                negated: !negated,
            },
            Formula::Atom { sample, vafs } => {
                let universe = scenario
                    .samples()
                    .get(sample)
                    .ok_or_else(|| errors::Error::InvalidSampleName {
                        name: sample.to_owned(),
                    })?
                    .contig_universe(contig, scenario.species())?;

                let mut disjunction = Vec::new();
                match vafs {
                    VAFSpectrum::Set(vafs) => {
                        let mut uvaf_stack: VecDeque<_> = universe.iter().cloned().collect();
                        while let Some(uvafs) = uvaf_stack.pop_front() {
                            match uvafs {
                                VAFSpectrum::Set(uvafs) => {
                                    let difference: BTreeSet<_> =
                                        uvafs.difference(&vafs).cloned().collect();
                                    if !difference.is_empty() {
                                        disjunction.push(VAFSpectrum::Set(difference));
                                    }
                                }
                                VAFSpectrum::Range(urange) => {
                                    for &vaf in vafs {
                                        if urange.contains(vaf) {
                                            let (left_urange, right_urange) = urange.split_at(vaf);
                                            if let Some(right_urange) = right_urange {
                                                uvaf_stack.push_back(right_urange);
                                            }
                                            if let Some(left_urange) = left_urange {
                                                disjunction.push(left_urange);
                                            }
                                        } else {
                                            disjunction.push(VAFSpectrum::Range(urange.clone()));
                                        }
                                    }
                                }
                            }
                        }
                    }
                    VAFSpectrum::Range(range) => {
                        for uvafs in universe.iter() {
                            match uvafs {
                                VAFSpectrum::Set(uvafs) => {
                                    let set: BTreeSet<_> = uvafs
                                        .iter()
                                        .filter(|uvaf| !range.contains(**uvaf))
                                        .cloned()
                                        .collect();
                                    if !set.is_empty() {
                                        disjunction.push(VAFSpectrum::Set(set));
                                    }
                                }
                                VAFSpectrum::Range(urange) => match range.overlap(urange) {
                                    VAFRangeOverlap::Contained => {
                                        if let Some(left) = urange.split_at(range.start).0 {
                                            disjunction.push(left);
                                        }
                                        if let Some(right) = urange.split_at(range.end).1 {
                                            disjunction.push(right);
                                        }
                                    }
                                    VAFRangeOverlap::End => {
                                        if let Some(spec) = urange.split_at(range.end).1 {
                                            disjunction.push(spec);
                                        }
                                    }
                                    VAFRangeOverlap::Start => {
                                        if let Some(spec) = urange.split_at(range.start).0 {
                                            disjunction.push(spec);
                                        }
                                    }
                                    VAFRangeOverlap::None => {
                                        disjunction.push(VAFSpectrum::Range(urange.clone()))
                                    }
                                    VAFRangeOverlap::Contains => (),
                                },
                            }
                        }
                    }
                }
                Formula::Disjunction {
                    operands: disjunction
                        .into_iter()
                        .map(|vafs| Formula::Atom {
                            sample: sample.clone(),
                            vafs,
                        })
                        .collect(),
                }
            }
        })
    }

    pub(crate) fn normalize(&self, scenario: &Scenario, contig: &str) -> Result<NormalizedFormula> {
        Ok(match self {
            Formula::Negation { operand } => operand
                .negate(scenario, contig)?
                .normalize(scenario, contig)?,
            Formula::Atom { sample, vafs } => NormalizedFormula::Atom {
                sample: sample.to_owned(),
                vafs: vafs.to_owned(),
            },
            Formula::Conjunction { operands } => NormalizedFormula::Conjunction {
                operands: operands
                    .iter()
                    .map(|o| Ok(o.normalize(scenario, contig)?))
                    .collect::<Result<Vec<NormalizedFormula>>>()?,
            },
            Formula::Disjunction { operands } => NormalizedFormula::Disjunction {
                operands: operands
                    .iter()
                    .map(|o| Ok(o.normalize(scenario, contig)?))
                    .collect::<Result<Vec<NormalizedFormula>>>()?,
            },
            &Formula::Variant {
                positive,
                refbase,
                altbase,
            } => NormalizedFormula::Variant {
                positive,
                refbase,
                altbase,
            },
            &Formula::Expression {
                ref identifier,
                negated,
            } => {
                if let Some(formula) = scenario.expressions().get(identifier) {
                    if negated {
                        formula
                            .negate(scenario, contig)?
                            .normalize(scenario, contig)?
                    } else {
                        formula.normalize(scenario, contig)?
                    }
                } else {
                    Err(errors::Error::UndefinedExpression {
                        identifier: identifier.to_string(),
                    })?;
                    unreachable!();
                }
            }
        })
    }
}

#[derive(Clone, Debug, PartialEq, Eq, Hash)]
pub(crate) enum VAFSpectrum {
    Set(BTreeSet<AlleleFreq>),
    Range(VAFRange),
}

impl VAFSpectrum {
    pub(crate) fn singleton(vaf: AlleleFreq) -> Self {
        let mut set = BTreeSet::new();
        set.insert(vaf);
        VAFSpectrum::Set(set)
    }

    pub(crate) fn contains(&self, vaf: AlleleFreq) -> bool {
        match self {
            VAFSpectrum::Set(ref set) => set.contains(&vaf),
            VAFSpectrum::Range(ref range) => range.contains(vaf),
        }
    }
}

#[derive(Clone, Debug, PartialEq, Eq, TypedBuilder, Hash)]
pub(crate) struct VAFRange {
    inner: ops::Range<AlleleFreq>,
    left_exclusive: bool,
    right_exclusive: bool,
}

pub(crate) enum VAFRangeOverlap {
    Contained,
    Contains,
    End,
    Start,
    None,
}

impl VAFRange {
    pub(crate) fn contains(&self, vaf: AlleleFreq) -> bool {
        match (self.left_exclusive, self.right_exclusive) {
            (true, true) => self.start < vaf && self.end > vaf,
            (true, false) => self.start < vaf && self.end >= vaf,
            (false, true) => self.start <= vaf && self.end > vaf,
            (false, false) => self.start <= vaf && self.end >= vaf,
        }
    }

    pub(crate) fn split_at(&self, vaf: AlleleFreq) -> (Option<VAFSpectrum>, Option<VAFSpectrum>) {
        assert!(
            self.contains(vaf),
            "bug: split_at is only defined if given VAF is contained in range"
        );
        let left = VAFRange {
            inner: self.start..vaf,
            left_exclusive: self.left_exclusive,
            right_exclusive: true,
        };
        let right = VAFRange {
            inner: vaf..self.end,
            left_exclusive: true,
            right_exclusive: self.right_exclusive,
        };

        let to_spectrum = |range: VAFRange| {
            if range.start == range.end {
                if !(range.left_exclusive && self.right_exclusive) {
                    Some(VAFSpectrum::singleton(range.start))
                } else {
                    None
                }
            } else {
                Some(VAFSpectrum::Range(range))
            }
        };

        (to_spectrum(left), to_spectrum(right))
    }

    pub(crate) fn overlap(&self, vafs: &VAFRange) -> VAFRangeOverlap {
        let range = self;
        let other_range = vafs;
        let start_is_right_of_start = match (self.left_exclusive, self.right_exclusive) {
            (true, true) => range.start >= other_range.start,
            (true, false) => range.start >= other_range.start,
            (false, true) => range.start > other_range.start,
            (false, false) => range.start >= other_range.start,
        };
        let end_is_left_of_end = match (self.left_exclusive, self.right_exclusive) {
            (true, true) => range.end <= other_range.end,
            (true, false) => range.end <= other_range.end,
            (false, true) => range.end < other_range.end,
            (false, false) => range.end <= other_range.end,
        };
        if range.end < other_range.start || range.start >= other_range.end {
            VAFRangeOverlap::None
        } else {
            match (start_is_right_of_start, end_is_left_of_end) {
                (true, true) => VAFRangeOverlap::Contained,
                (true, false) => VAFRangeOverlap::Start,
                (false, true) => VAFRangeOverlap::End,
                (false, false) => VAFRangeOverlap::Contains,
            }
        }
    }

    pub(crate) fn observable_min(&self, n_obs: usize) -> AlleleFreq {
        if n_obs < 10 {
            self.start
        } else {
            let obs_count = Self::expected_observation_count(self.start, n_obs);
            let adjust_allelefreq = |obs_count: f64| AlleleFreq(obs_count.ceil() / n_obs as f64);

            if self.left_exclusive && obs_count % 1.0 == 0.0 {
                // We are left exclusive and need to find a supremum from the right.

                let adjusted_end = self.observable_max(n_obs);

                for offset in &[1.0, 0.0] {
                    let adjusted_obs_count = obs_count + offset;
                    let adjusted_start = adjust_allelefreq(adjusted_obs_count);
                    if *adjusted_start <= 1.0 && adjusted_start <= adjusted_end {
                        return adjusted_start;
                    }
                }
            }

            adjust_allelefreq(obs_count)
        }
    }

    pub(crate) fn observable_max(&self, n_obs: usize) -> AlleleFreq {
        assert!(
            *self.end != 0.0,
            "bug: observable_max may not be called if end=0.0."
        );
        if n_obs < 10 {
            self.end
        } else {
            let mut obs_count = Self::expected_observation_count(self.end, n_obs);
            if self.right_exclusive && obs_count % 1.0 == 0.0 {
                obs_count -= 1.0;
            }
            AlleleFreq(obs_count.floor() / n_obs as f64)
        }
    }

    fn expected_observation_count(freq: AlleleFreq, n_obs: usize) -> f64 {
        n_obs as f64 * *freq
    }
}

impl ops::Deref for VAFRange {
    type Target = ops::Range<AlleleFreq>;

    fn deref(&self) -> &ops::Range<AlleleFreq> {
        &self.inner
    }
}

impl Ord for VAFRange {
    fn cmp(&self, other: &Self) -> Ordering {
        match self.start.cmp(&other.start) {
            Ordering::Equal => self.end.cmp(&other.end),
            ord => ord,
        }
    }
}

impl PartialOrd for VAFRange {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

#[derive(Debug, Clone, Default, Derefable)]
pub(crate) struct VAFUniverse(#[deref(mutable)] HashSet<VAFSpectrum>);

impl VAFUniverse {
    pub(crate) fn contains(&self, vaf: AlleleFreq) -> bool {
        for atom in &**self {
            if atom.contains(vaf) {
                return true;
            }
        }
        false
    }
}

impl<'de> Deserialize<'de> for VAFUniverse {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: de::Deserializer<'de>,
    {
        deserializer.deserialize_string(VAFUniverseVisitor)
    }
}

struct VAFUniverseVisitor;

impl<'de> de::Visitor<'de> for VAFUniverseVisitor {
    type Value = VAFUniverse;

    fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        formatter.write_str(
            "a disjunction of possible VAFs (see https://varlociraptor.github.io/docs/calling)",
        )
    }

    fn visit_str<E>(self, v: &str) -> Result<Self::Value, E>
    where
        E: de::Error,
    {
        let res = FormulaParser::parse(Rule::universe, v);
        match res {
            Ok(pairs) => {
                let mut operands = HashSet::new();
                for pair in pairs {
                    match pair.as_rule() {
                        Rule::vaf => {
                            operands.insert(parse_vaf(pair));
                        }
                        Rule::vafrange => {
                            let inner = pair.into_inner();
                            operands.insert(parse_vafrange(inner));
                        }
                        Rule::EOI => (),
                        _ => unreachable!(),
                    }
                }
                Ok(VAFUniverse(operands))
            }
            Err(e) => {
                eprintln!("{}", e);
                Err(de::Error::invalid_value(
                    serde::de::Unexpected::Other("invalid VAF formula"),
                    &self,
                ))
            }
        }
    }
}

impl<'de> Deserialize<'de> for Formula {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: de::Deserializer<'de>,
    {
        deserializer.deserialize_string(FormulaVisitor)
    }
}

struct FormulaVisitor;

impl<'de> de::Visitor<'de> for FormulaVisitor {
    type Value = Formula;

    fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        formatter
            .write_str("a valid VAF formula (see https://varlociraptor.github.io/docs/calling)")
    }

    fn visit_str<E>(self, v: &str) -> Result<Self::Value, E>
    where
        E: de::Error,
    {
        let res = FormulaParser::parse(Rule::formula, v);
        match res {
            Ok(mut pairs) => {
                let pair = pairs.next().expect("bug: expecting formula");
                parse_formula(pair)
            }
            Err(e) => Err(de::Error::invalid_value(
                serde::de::Unexpected::Other(&format!("invalid VAF formula:\n{}", e)),
                &self,
            )),
        }
    }
}

fn parse_vaf(pair: Pair<Rule>) -> VAFSpectrum {
    let vaf = pair.as_str().parse().expect("bug: unable to parse VAF");
    VAFSpectrum::singleton(AlleleFreq(vaf))
}

fn parse_vafrange(mut inner: Pairs<Rule>) -> VAFSpectrum {
    let left = inner.next().unwrap().as_str();
    let lower = inner.next().unwrap().as_str().parse().unwrap();
    let upper = inner.next().unwrap().as_str().parse().unwrap();
    let right = inner.next().unwrap().as_str();

    let range = VAFRange {
        inner: lower..upper,
        left_exclusive: left == "]",
        right_exclusive: right == "[",
    };

    VAFSpectrum::Range(range)
}

fn parse_formula<E>(pair: Pair<Rule>) -> Result<Formula, E>
where
    E: de::Error,
{
    Ok(match pair.as_rule() {
        Rule::expression => {
            let mut inner = pair.into_inner();
            let identifier = inner.next().unwrap().as_str();
            Formula::Expression {
                identifier: ExpressionIdentifier(identifier.to_owned()),
                negated: false,
            }
        }
        Rule::variant => {
            let mut inner = pair.into_inner();
            let refbase = inner.next().unwrap().as_str().as_bytes()[0];
            let altbase = inner.next().unwrap().as_str().as_bytes()[0];
            Formula::Variant {
                refbase: IUPAC(refbase),
                altbase: IUPAC(altbase),
                positive: true,
            }
        }
        Rule::sample_vaf => {
            let mut inner = pair.into_inner();
            let sample = inner.next().unwrap().as_str().to_owned();
            Formula::Atom {
                sample,
                vafs: parse_vaf(inner.next().unwrap()),
            }
        }
        Rule::sample_vafrange => {
            let mut inner = pair.into_inner();
            let sample = inner.next().unwrap().as_str().to_owned();
            Formula::Atom {
                sample,
                vafs: parse_vafrange(inner.next().unwrap().into_inner()),
            }
        }
        Rule::conjunction => {
            let inner = pair.into_inner();
            let mut operands = Vec::new();
            for operand in inner {
                operands.push(parse_formula(operand)?);
            }
            Formula::Conjunction { operands }
        }
        Rule::disjunction => {
            let inner = pair.into_inner();
            let mut operands = Vec::new();
            for operand in inner {
                operands.push(parse_formula(operand)?);
            }
            Formula::Disjunction { operands }
        }
        Rule::negation => {
            let mut inner = pair.into_inner();
            Formula::Negation {
                operand: Box::new(parse_formula(inner.next().unwrap())?),
            }
        }
        Rule::formula => unreachable!(),
        Rule::subformula => unreachable!(),
        Rule::vafdef => unreachable!(),
        Rule::bound => unreachable!(),
        Rule::universe => unreachable!(),
        Rule::vafrange => unreachable!(),
        Rule::identifier => unreachable!(),
        Rule::vaf => unreachable!(),
        Rule::sample_vafdef => unreachable!(),
        Rule::EOI => unreachable!(),
        Rule::WHITESPACE => unreachable!(),
        Rule::COMMENT => unreachable!(),
        Rule::iupac => unreachable!(),
    })
}
