#![warn(clippy::all)]
#![deny(missing_docs)]

//! Encode the validity of your data in the type system
//! 
//! Example usage:
//! 
//! ```
//! # use validity::*;
//! #[derive(Debug)]
//! struct Email(String);
//! 
//! impl Validate for Email {
//!     type Error = ();
//! 
//!     fn validate(&self) -> Result<(), Self::Error> {
//!         // perform some actual email validation
//!         Ok(()) 
//!     }
//! }
//! 
//! // This requires a Valid<Email>, which guarantees the data has been validated
//! fn send_email(email: Valid<Email>) {
//!     println!("Sending email to {:?}", email.0);
//! }
//! 
//! fn main() {
//!    let email = Email("example@example.com".into()); 
//!    let valid_email = Valid::new(email).unwrap();
//!    // at this point, validation must have succeeded
//!    send_email(valid_email);
//! }
//! ```



use std::ops::Deref;

/// Definition of what it means for a value to be "valid"
pub trait Validate {
    /// The type of errors that can be returned from a validation function
    type Error;

    /// Check if data is valid, returning `Ok(())` if it is, or a custom `Error` otherwise
    fn validate(&self) -> Result<(), Self::Error>;
}

/// A wrapper struct that guarantees that the inner value has been checked with its [`Validate`]
/// implementation
///
/// The only way to obtain a `Valid<T>` is to call `Valid::new` and handle the returned `Result`
///
/// By requiring a `Valid<T>` in your API, you guarantee at compile time that unvalidated data
/// cannot passed into your function.
#[derive(Debug, Clone, PartialEq, Eq, Hash, Copy)]
pub struct Valid<T>(T);

impl<T: Validate> Valid<T> {
    /// Consume `self`, returning the wrapped data
    pub fn into_inner(self) -> T {
        self.0
    }

    /// Attempt to create a `Valid<T>`, returning a `Result`
    ///
    /// If validation fails, an `Err` is returned
    pub fn new(t: T) -> Result<Self, <T as Validate>::Error> {
        match t.validate() {
            Ok(()) => Ok(Valid(t)),
            Err(e) => Err(e),
        }
    }
}

impl<T> AsRef<T> for Valid<T> {
    fn as_ref(&self) -> &T {
        &self.0
    }
}

impl<T> Deref for Valid<T> {
    type Target = T;

    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

