#[cfg(feature = "struct_loader")]
pub use struct_commands::*;
#[cfg(feature = "struct_loader")]
pub mod struct_commands {
    //! Contains the [`GlFns`] type for using the struct GL loader.
    use super::*;
    impl GlFns {
        /// Constructs a new struct with all pointers loaded by the `get_proc_address` given.
        pub unsafe fn load_with<F>(mut get_proc_address: F) -> Self
        where
            F: FnMut(*const c_char) -> *mut c_void,
        {
            // Safety: The `GlFns` struct is nothing but `AtomicPtr` fields,
            // which can be safely constructed with `zeroed`.
            let out: Self = core::mem::zeroed();
            out.load_all_with_dyn(&mut get_proc_address);
            out
        }

        #[cfg(feature = "debug_automatic_glGetError")]
        #[inline(never)]
        unsafe fn automatic_glGetError(&self, name: &str) {
            let mut err = self.GetError();
            while err != GL_NO_ERROR {
                report_error_code_from(name, err);
                err = self.GetError();
            }
        }

        /// Loads all pointers using the `get_proc_address` given.
        #[doc(hidden)]
        #[inline(never)]
        pub unsafe fn load_all_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) {
            #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
            {
                self.AcquireKeyedMutexWin32EXT_load_with_dyn(get_proc_address);
            }
            self.ActiveShaderProgram_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ActiveShaderProgramEXT_load_with_dyn(get_proc_address);
            }
            self.ActiveTexture_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_QCOM_alpha_test"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_alpha_test"))))]
            {
                self.AlphaFuncQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))))]
            {
                self.ApplyFramebufferAttachmentCMAAINTEL_load_with_dyn(get_proc_address);
            }
            self.AttachShader_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_conditional_render"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
            {
                self.BeginConditionalRenderNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.BeginPerfMonitorAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.BeginPerfQueryINTEL_load_with_dyn(get_proc_address);
            }
            self.BeginQuery_load_with_dyn(get_proc_address);
            #[cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_disjoint_timer_query",
                    feature = "GL_EXT_occlusion_query_boolean"
                )))
            )]
            {
                self.BeginQueryEXT_load_with_dyn(get_proc_address);
            }
            self.BeginTransformFeedback_load_with_dyn(get_proc_address);
            self.BindAttribLocation_load_with_dyn(get_proc_address);
            self.BindBuffer_load_with_dyn(get_proc_address);
            self.BindBufferBase_load_with_dyn(get_proc_address);
            self.BindBufferRange_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
            {
                self.BindFragDataLocationEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
            {
                self.BindFragDataLocationIndexedEXT_load_with_dyn(get_proc_address);
            }
            self.BindFramebuffer_load_with_dyn(get_proc_address);
            self.BindImageTexture_load_with_dyn(get_proc_address);
            self.BindProgramPipeline_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.BindProgramPipelineEXT_load_with_dyn(get_proc_address);
            }
            self.BindRenderbuffer_load_with_dyn(get_proc_address);
            self.BindSampler_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_shading_rate_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
            {
                self.BindShadingRateImageNV_load_with_dyn(get_proc_address);
            }
            self.BindTexture_load_with_dyn(get_proc_address);
            self.BindTransformFeedback_load_with_dyn(get_proc_address);
            self.BindVertexArray_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_vertex_array_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
            {
                self.BindVertexArrayOES_load_with_dyn(get_proc_address);
            }
            self.BindVertexBuffer_load_with_dyn(get_proc_address);
            self.BlendBarrier_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_blend_equation_advanced"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_blend_equation_advanced"))))]
            {
                self.BlendBarrierKHR_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
            {
                self.BlendBarrierNV_load_with_dyn(get_proc_address);
            }
            self.BlendColor_load_with_dyn(get_proc_address);
            self.BlendEquation_load_with_dyn(get_proc_address);
            self.BlendEquationSeparate_load_with_dyn(get_proc_address);
            self.BlendEquationSeparatei_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
            {
                self.BlendEquationSeparateiEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
            {
                self.BlendEquationSeparateiOES_load_with_dyn(get_proc_address);
            }
            self.BlendEquationi_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
            {
                self.BlendEquationiEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
            {
                self.BlendEquationiOES_load_with_dyn(get_proc_address);
            }
            self.BlendFunc_load_with_dyn(get_proc_address);
            self.BlendFuncSeparate_load_with_dyn(get_proc_address);
            self.BlendFuncSeparatei_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
            {
                self.BlendFuncSeparateiEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
            {
                self.BlendFuncSeparateiOES_load_with_dyn(get_proc_address);
            }
            self.BlendFunci_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
            {
                self.BlendFunciEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
            {
                self.BlendFunciOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
            {
                self.BlendParameteriNV_load_with_dyn(get_proc_address);
            }
            self.BlitFramebuffer_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_ANGLE_framebuffer_blit"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_blit"))))]
            {
                self.BlitFramebufferANGLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_framebuffer_blit"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_blit"))))]
            {
                self.BlitFramebufferNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_memory_attachment"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
            {
                self.BufferAttachMemoryNV_load_with_dyn(get_proc_address);
            }
            self.BufferData_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
            {
                self.BufferPageCommitmentMemNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_buffer_storage"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_buffer_storage"))))]
            {
                self.BufferStorageEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_external_buffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
            {
                self.BufferStorageExternalEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.BufferStorageMemEXT_load_with_dyn(get_proc_address);
            }
            self.BufferSubData_load_with_dyn(get_proc_address);
            self.CheckFramebufferStatus_load_with_dyn(get_proc_address);
            self.Clear_load_with_dyn(get_proc_address);
            self.ClearBufferfi_load_with_dyn(get_proc_address);
            self.ClearBufferfv_load_with_dyn(get_proc_address);
            self.ClearBufferiv_load_with_dyn(get_proc_address);
            self.ClearBufferuiv_load_with_dyn(get_proc_address);
            self.ClearColor_load_with_dyn(get_proc_address);
            self.ClearDepthf_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
            {
                self.ClearPixelLocalStorageuiEXT_load_with_dyn(get_proc_address);
            }
            self.ClearStencil_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_clear_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
            {
                self.ClearTexImageEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_clear_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
            {
                self.ClearTexSubImageEXT_load_with_dyn(get_proc_address);
            }
            self.ClientWaitSync_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_APPLE_sync"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
            {
                self.ClientWaitSyncAPPLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_clip_control"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clip_control"))))]
            {
                self.ClipControlEXT_load_with_dyn(get_proc_address);
            }
            self.ColorMask_load_with_dyn(get_proc_address);
            self.ColorMaski_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
            {
                self.ColorMaskiEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
            {
                self.ColorMaskiOES_load_with_dyn(get_proc_address);
            }
            self.CompileShader_load_with_dyn(get_proc_address);
            self.CompressedTexImage2D_load_with_dyn(get_proc_address);
            self.CompressedTexImage3D_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_texture_3D"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
            {
                self.CompressedTexImage3DOES_load_with_dyn(get_proc_address);
            }
            self.CompressedTexSubImage2D_load_with_dyn(get_proc_address);
            self.CompressedTexSubImage3D_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_texture_3D"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
            {
                self.CompressedTexSubImage3DOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles")))
            )]
            {
                self.ConservativeRasterParameteriNV_load_with_dyn(get_proc_address);
            }
            self.CopyBufferSubData_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_copy_buffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_copy_buffer"))))]
            {
                self.CopyBufferSubDataNV_load_with_dyn(get_proc_address);
            }
            self.CopyImageSubData_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_copy_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_copy_image"))))]
            {
                self.CopyImageSubDataEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_copy_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_copy_image"))))]
            {
                self.CopyImageSubDataOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.CopyPathNV_load_with_dyn(get_proc_address);
            }
            self.CopyTexImage2D_load_with_dyn(get_proc_address);
            self.CopyTexSubImage2D_load_with_dyn(get_proc_address);
            self.CopyTexSubImage3D_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_texture_3D"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
            {
                self.CopyTexSubImage3DOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_APPLE_copy_texture_levels"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_copy_texture_levels"))))]
            {
                self.CopyTextureLevelsAPPLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.CoverFillPathInstancedNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.CoverFillPathNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.CoverStrokePathInstancedNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.CoverStrokePathNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_coverage_sample"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
            {
                self.CoverageMaskNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
            {
                self.CoverageModulationNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
            {
                self.CoverageModulationTableNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_coverage_sample"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
            {
                self.CoverageOperationNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.CreateMemoryObjectsEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.CreatePerfQueryINTEL_load_with_dyn(get_proc_address);
            }
            self.CreateProgram_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
            {
                self.CreateSemaphoresNV_load_with_dyn(get_proc_address);
            }
            self.CreateShader_load_with_dyn(get_proc_address);
            self.CreateShaderProgramv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.CreateShaderProgramvEXT_load_with_dyn(get_proc_address);
            }
            self.CullFace_load_with_dyn(get_proc_address);
            self.DebugMessageCallback_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.DebugMessageCallbackKHR_load_with_dyn(get_proc_address);
            }
            self.DebugMessageControl_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.DebugMessageControlKHR_load_with_dyn(get_proc_address);
            }
            self.DebugMessageInsert_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.DebugMessageInsertKHR_load_with_dyn(get_proc_address);
            }
            self.DeleteBuffers_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_fence"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
            {
                self.DeleteFencesNV_load_with_dyn(get_proc_address);
            }
            self.DeleteFramebuffers_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.DeleteMemoryObjectsEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.DeletePathsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.DeletePerfMonitorsAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.DeletePerfQueryINTEL_load_with_dyn(get_proc_address);
            }
            self.DeleteProgram_load_with_dyn(get_proc_address);
            self.DeleteProgramPipelines_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.DeleteProgramPipelinesEXT_load_with_dyn(get_proc_address);
            }
            self.DeleteQueries_load_with_dyn(get_proc_address);
            #[cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_disjoint_timer_query",
                    feature = "GL_EXT_occlusion_query_boolean"
                )))
            )]
            {
                self.DeleteQueriesEXT_load_with_dyn(get_proc_address);
            }
            self.DeleteRenderbuffers_load_with_dyn(get_proc_address);
            self.DeleteSamplers_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
            {
                self.DeleteSemaphoresEXT_load_with_dyn(get_proc_address);
            }
            self.DeleteShader_load_with_dyn(get_proc_address);
            self.DeleteSync_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_APPLE_sync"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
            {
                self.DeleteSyncAPPLE_load_with_dyn(get_proc_address);
            }
            self.DeleteTextures_load_with_dyn(get_proc_address);
            self.DeleteTransformFeedbacks_load_with_dyn(get_proc_address);
            self.DeleteVertexArrays_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_vertex_array_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
            {
                self.DeleteVertexArraysOES_load_with_dyn(get_proc_address);
            }
            self.DepthFunc_load_with_dyn(get_proc_address);
            self.DepthMask_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.DepthRangeArrayfvNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
            {
                self.DepthRangeArrayfvOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.DepthRangeIndexedfNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
            {
                self.DepthRangeIndexedfOES_load_with_dyn(get_proc_address);
            }
            self.DepthRangef_load_with_dyn(get_proc_address);
            self.DetachShader_load_with_dyn(get_proc_address);
            self.Disable_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_QCOM_driver_control"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
            {
                self.DisableDriverControlQCOM_load_with_dyn(get_proc_address);
            }
            self.DisableVertexAttribArray_load_with_dyn(get_proc_address);
            self.Disablei_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
            {
                self.DisableiEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.DisableiNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_OES_draw_buffers_indexed",
                    feature = "GL_OES_viewport_array"
                )))
            )]
            {
                self.DisableiOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_discard_framebuffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_discard_framebuffer"))))]
            {
                self.DiscardFramebufferEXT_load_with_dyn(get_proc_address);
            }
            self.DispatchCompute_load_with_dyn(get_proc_address);
            self.DispatchComputeIndirect_load_with_dyn(get_proc_address);
            self.DrawArrays_load_with_dyn(get_proc_address);
            self.DrawArraysIndirect_load_with_dyn(get_proc_address);
            self.DrawArraysInstanced_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
            {
                self.DrawArraysInstancedANGLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_base_instance"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
            {
                self.DrawArraysInstancedBaseInstanceEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_draw_instanced",
                    feature = "GL_EXT_instanced_arrays"
                )))
            )]
            {
                self.DrawArraysInstancedEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_draw_instanced"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
            {
                self.DrawArraysInstancedNV_load_with_dyn(get_proc_address);
            }
            self.DrawBuffers_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_buffers"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers"))))]
            {
                self.DrawBuffersEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
            {
                self.DrawBuffersIndexedEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_draw_buffers"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_buffers"))))]
            {
                self.DrawBuffersNV_load_with_dyn(get_proc_address);
            }
            self.DrawElements_load_with_dyn(get_proc_address);
            self.DrawElementsBaseVertex_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
            {
                self.DrawElementsBaseVertexEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
            {
                self.DrawElementsBaseVertexOES_load_with_dyn(get_proc_address);
            }
            self.DrawElementsIndirect_load_with_dyn(get_proc_address);
            self.DrawElementsInstanced_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
            {
                self.DrawElementsInstancedANGLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_base_instance"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
            {
                self.DrawElementsInstancedBaseInstanceEXT_load_with_dyn(get_proc_address);
            }
            self.DrawElementsInstancedBaseVertex_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_base_instance"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
            {
                self.DrawElementsInstancedBaseVertexBaseInstanceEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
            {
                self.DrawElementsInstancedBaseVertexEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
            {
                self.DrawElementsInstancedBaseVertexOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_draw_instanced",
                    feature = "GL_EXT_instanced_arrays"
                )))
            )]
            {
                self.DrawElementsInstancedEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_draw_instanced"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
            {
                self.DrawElementsInstancedNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_mesh_shader"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
            {
                self.DrawMeshTasksIndirectNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_mesh_shader"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
            {
                self.DrawMeshTasksNV_load_with_dyn(get_proc_address);
            }
            self.DrawRangeElements_load_with_dyn(get_proc_address);
            self.DrawRangeElementsBaseVertex_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
            {
                self.DrawRangeElementsBaseVertexEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
            {
                self.DrawRangeElementsBaseVertexOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
            {
                self.DrawTransformFeedbackEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
            {
                self.DrawTransformFeedbackInstancedEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
            {
                self.DrawVkImageNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_EGL_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
            {
                self.EGLImageTargetRenderbufferStorageOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
            {
                self.EGLImageTargetTexStorageEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_EGL_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
            {
                self.EGLImageTargetTexture2DOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
            {
                self.EGLImageTargetTextureStorageEXT_load_with_dyn(get_proc_address);
            }
            self.Enable_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_QCOM_driver_control"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
            {
                self.EnableDriverControlQCOM_load_with_dyn(get_proc_address);
            }
            self.EnableVertexAttribArray_load_with_dyn(get_proc_address);
            self.Enablei_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
            {
                self.EnableiEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.EnableiNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_OES_draw_buffers_indexed",
                    feature = "GL_OES_viewport_array"
                )))
            )]
            {
                self.EnableiOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_conditional_render"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
            {
                self.EndConditionalRenderNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.EndPerfMonitorAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.EndPerfQueryINTEL_load_with_dyn(get_proc_address);
            }
            self.EndQuery_load_with_dyn(get_proc_address);
            #[cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_disjoint_timer_query",
                    feature = "GL_EXT_occlusion_query_boolean"
                )))
            )]
            {
                self.EndQueryEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
            {
                self.EndTilingQCOM_load_with_dyn(get_proc_address);
            }
            self.EndTransformFeedback_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_QCOM_extended_get"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
            {
                self.ExtGetBufferPointervQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
            {
                self.ExtGetBuffersQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
            {
                self.ExtGetFramebuffersQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get2"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
            {
                self.ExtGetProgramBinarySourceQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get2"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
            {
                self.ExtGetProgramsQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
            {
                self.ExtGetRenderbuffersQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get2"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
            {
                self.ExtGetShadersQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
            {
                self.ExtGetTexLevelParameterivQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
            {
                self.ExtGetTexSubImageQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
            {
                self.ExtGetTexturesQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get2"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
            {
                self.ExtIsProgramBinaryQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_extended_get"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
            {
                self.ExtTexObjectStateOverrideiQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_frame_extrapolation"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_frame_extrapolation"))))]
            {
                self.ExtrapolateTex2DQCOM_load_with_dyn(get_proc_address);
            }
            self.FenceSync_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_APPLE_sync"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
            {
                self.FenceSyncAPPLE_load_with_dyn(get_proc_address);
            }
            self.Finish_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_fence"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
            {
                self.FinishFenceNV_load_with_dyn(get_proc_address);
            }
            self.Flush_load_with_dyn(get_proc_address);
            self.FlushMappedBufferRange_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
            {
                self.FlushMappedBufferRangeEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_fragment_coverage_to_color"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fragment_coverage_to_color"))))]
            {
                self.FragmentCoverageColorNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent")))
            )]
            {
                self.FramebufferFetchBarrierEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent")))
            )]
            {
                self.FramebufferFetchBarrierQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
            {
                self.FramebufferFoveationConfigQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
            {
                self.FramebufferFoveationParametersQCOM_load_with_dyn(get_proc_address);
            }
            self.FramebufferParameteri_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
            {
                self.FramebufferParameteriMESA_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
            {
                self.FramebufferPixelLocalStorageSizeEXT_load_with_dyn(get_proc_address);
            }
            self.FramebufferRenderbuffer_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_sample_locations"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
            {
                self.FramebufferSampleLocationsfvNV_load_with_dyn(get_proc_address);
            }
            self.FramebufferTexture_load_with_dyn(get_proc_address);
            self.FramebufferTexture2D_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
            {
                self.FramebufferTexture2DDownsampleIMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
            )]
            {
                self.FramebufferTexture2DMultisampleEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
            )]
            {
                self.FramebufferTexture2DMultisampleIMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_3D"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
            {
                self.FramebufferTexture3DOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_geometry_shader"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_geometry_shader"))))]
            {
                self.FramebufferTextureEXT_load_with_dyn(get_proc_address);
            }
            self.FramebufferTextureLayer_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
            {
                self.FramebufferTextureLayerDownsampleIMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture")))
            )]
            {
                self.FramebufferTextureMultisampleMultiviewOVR_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OVR_multiview"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OVR_multiview"))))]
            {
                self.FramebufferTextureMultiviewOVR_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_geometry_shader"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_geometry_shader"))))]
            {
                self.FramebufferTextureOES_load_with_dyn(get_proc_address);
            }
            self.FrontFace_load_with_dyn(get_proc_address);
            self.GenBuffers_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_fence"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
            {
                self.GenFencesNV_load_with_dyn(get_proc_address);
            }
            self.GenFramebuffers_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GenPathsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.GenPerfMonitorsAMD_load_with_dyn(get_proc_address);
            }
            self.GenProgramPipelines_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.GenProgramPipelinesEXT_load_with_dyn(get_proc_address);
            }
            self.GenQueries_load_with_dyn(get_proc_address);
            #[cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_disjoint_timer_query",
                    feature = "GL_EXT_occlusion_query_boolean"
                )))
            )]
            {
                self.GenQueriesEXT_load_with_dyn(get_proc_address);
            }
            self.GenRenderbuffers_load_with_dyn(get_proc_address);
            self.GenSamplers_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
            {
                self.GenSemaphoresEXT_load_with_dyn(get_proc_address);
            }
            self.GenTextures_load_with_dyn(get_proc_address);
            self.GenTransformFeedbacks_load_with_dyn(get_proc_address);
            self.GenVertexArrays_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_vertex_array_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
            {
                self.GenVertexArraysOES_load_with_dyn(get_proc_address);
            }
            self.GenerateMipmap_load_with_dyn(get_proc_address);
            self.GetActiveAttrib_load_with_dyn(get_proc_address);
            self.GetActiveUniform_load_with_dyn(get_proc_address);
            self.GetActiveUniformBlockName_load_with_dyn(get_proc_address);
            self.GetActiveUniformBlockiv_load_with_dyn(get_proc_address);
            self.GetActiveUniformsiv_load_with_dyn(get_proc_address);
            self.GetAttachedShaders_load_with_dyn(get_proc_address);
            self.GetAttribLocation_load_with_dyn(get_proc_address);
            self.GetBooleani_v_load_with_dyn(get_proc_address);
            self.GetBooleanv_load_with_dyn(get_proc_address);
            self.GetBufferParameteri64v_load_with_dyn(get_proc_address);
            self.GetBufferParameteriv_load_with_dyn(get_proc_address);
            self.GetBufferPointerv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_mapbuffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
            {
                self.GetBufferPointervOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
            {
                self.GetCoverageModulationTableNV_load_with_dyn(get_proc_address);
            }
            self.GetDebugMessageLog_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.GetDebugMessageLogKHR_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_driver_control"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
            {
                self.GetDriverControlStringQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_driver_control"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
            {
                self.GetDriverControlsQCOM_load_with_dyn(get_proc_address);
            }
            self.GetError_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_fence"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
            {
                self.GetFenceivNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.GetFirstPerfQueryIdINTEL_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.GetFloati_vNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
            {
                self.GetFloati_vOES_load_with_dyn(get_proc_address);
            }
            self.GetFloatv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
            {
                self.GetFragDataIndexEXT_load_with_dyn(get_proc_address);
            }
            self.GetFragDataLocation_load_with_dyn(get_proc_address);
            self.GetFramebufferAttachmentParameteriv_load_with_dyn(get_proc_address);
            self.GetFramebufferParameteriv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
            {
                self.GetFramebufferParameterivMESA_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
            {
                self.GetFramebufferPixelLocalStorageSizeEXT_load_with_dyn(get_proc_address);
            }
            self.GetGraphicsResetStatus_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_robustness"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
            {
                self.GetGraphicsResetStatusEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_KHR_robustness"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
            {
                self.GetGraphicsResetStatusKHR_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.GetImageHandleNV_load_with_dyn(get_proc_address);
            }
            self.GetInteger64i_v_load_with_dyn(get_proc_address);
            self.GetInteger64v_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_APPLE_sync"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
            {
                self.GetInteger64vAPPLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
            {
                self.GetInteger64vEXT_load_with_dyn(get_proc_address);
            }
            self.GetIntegeri_v_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
            {
                self.GetIntegeri_vEXT_load_with_dyn(get_proc_address);
            }
            self.GetIntegerv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_internalformat_sample_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_internalformat_sample_query"))))]
            {
                self.GetInternalformatSampleivNV_load_with_dyn(get_proc_address);
            }
            self.GetInternalformativ_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_memory_attachment"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
            {
                self.GetMemoryObjectDetachedResourcesuivNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.GetMemoryObjectParameterivEXT_load_with_dyn(get_proc_address);
            }
            self.GetMultisamplefv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.GetNextPerfQueryIdINTEL_load_with_dyn(get_proc_address);
            }
            self.GetObjectLabel_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_debug_label"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
            {
                self.GetObjectLabelEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.GetObjectLabelKHR_load_with_dyn(get_proc_address);
            }
            self.GetObjectPtrLabel_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.GetObjectPtrLabelKHR_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetPathCommandsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetPathCoordsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetPathDashArrayNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetPathLengthNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetPathMetricRangeNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetPathMetricsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetPathParameterfvNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetPathParameterivNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetPathSpacingNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.GetPerfCounterInfoINTEL_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.GetPerfMonitorCounterDataAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.GetPerfMonitorCounterInfoAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.GetPerfMonitorCounterStringAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.GetPerfMonitorCountersAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.GetPerfMonitorGroupStringAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.GetPerfMonitorGroupsAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.GetPerfQueryDataINTEL_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.GetPerfQueryIdByNameINTEL_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_INTEL_performance_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
            {
                self.GetPerfQueryInfoINTEL_load_with_dyn(get_proc_address);
            }
            self.GetPointerv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.GetPointervKHR_load_with_dyn(get_proc_address);
            }
            self.GetProgramBinary_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_get_program_binary"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
            {
                self.GetProgramBinaryOES_load_with_dyn(get_proc_address);
            }
            self.GetProgramInfoLog_load_with_dyn(get_proc_address);
            self.GetProgramInterfaceiv_load_with_dyn(get_proc_address);
            self.GetProgramPipelineInfoLog_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.GetProgramPipelineInfoLogEXT_load_with_dyn(get_proc_address);
            }
            self.GetProgramPipelineiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.GetProgramPipelineivEXT_load_with_dyn(get_proc_address);
            }
            self.GetProgramResourceIndex_load_with_dyn(get_proc_address);
            self.GetProgramResourceLocation_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
            {
                self.GetProgramResourceLocationIndexEXT_load_with_dyn(get_proc_address);
            }
            self.GetProgramResourceName_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.GetProgramResourcefvNV_load_with_dyn(get_proc_address);
            }
            self.GetProgramResourceiv_load_with_dyn(get_proc_address);
            self.GetProgramiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
            {
                self.GetQueryObjecti64vEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
            {
                self.GetQueryObjectivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
            {
                self.GetQueryObjectui64vEXT_load_with_dyn(get_proc_address);
            }
            self.GetQueryObjectuiv_load_with_dyn(get_proc_address);
            #[cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_disjoint_timer_query",
                    feature = "GL_EXT_occlusion_query_boolean"
                )))
            )]
            {
                self.GetQueryObjectuivEXT_load_with_dyn(get_proc_address);
            }
            self.GetQueryiv_load_with_dyn(get_proc_address);
            #[cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_disjoint_timer_query",
                    feature = "GL_EXT_occlusion_query_boolean"
                )))
            )]
            {
                self.GetQueryivEXT_load_with_dyn(get_proc_address);
            }
            self.GetRenderbufferParameteriv_load_with_dyn(get_proc_address);
            self.GetSamplerParameterIiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
            {
                self.GetSamplerParameterIivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
            {
                self.GetSamplerParameterIivOES_load_with_dyn(get_proc_address);
            }
            self.GetSamplerParameterIuiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
            {
                self.GetSamplerParameterIuivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
            {
                self.GetSamplerParameterIuivOES_load_with_dyn(get_proc_address);
            }
            self.GetSamplerParameterfv_load_with_dyn(get_proc_address);
            self.GetSamplerParameteriv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
            {
                self.GetSemaphoreParameterivNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
            {
                self.GetSemaphoreParameterui64vEXT_load_with_dyn(get_proc_address);
            }
            self.GetShaderInfoLog_load_with_dyn(get_proc_address);
            self.GetShaderPrecisionFormat_load_with_dyn(get_proc_address);
            self.GetShaderSource_load_with_dyn(get_proc_address);
            self.GetShaderiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_shading_rate_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
            {
                self.GetShadingRateImagePaletteNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_shading_rate_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
            {
                self.GetShadingRateSampleLocationivNV_load_with_dyn(get_proc_address);
            }
            self.GetString_load_with_dyn(get_proc_address);
            self.GetStringi_load_with_dyn(get_proc_address);
            self.GetSynciv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_APPLE_sync"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
            {
                self.GetSyncivAPPLE_load_with_dyn(get_proc_address);
            }
            self.GetTexLevelParameterfv_load_with_dyn(get_proc_address);
            self.GetTexLevelParameteriv_load_with_dyn(get_proc_address);
            self.GetTexParameterIiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
            {
                self.GetTexParameterIivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
            {
                self.GetTexParameterIivOES_load_with_dyn(get_proc_address);
            }
            self.GetTexParameterIuiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
            {
                self.GetTexParameterIuivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
            {
                self.GetTexParameterIuivOES_load_with_dyn(get_proc_address);
            }
            self.GetTexParameterfv_load_with_dyn(get_proc_address);
            self.GetTexParameteriv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_IMG_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
            {
                self.GetTextureHandleIMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.GetTextureHandleNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_IMG_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
            {
                self.GetTextureSamplerHandleIMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.GetTextureSamplerHandleNV_load_with_dyn(get_proc_address);
            }
            self.GetTransformFeedbackVarying_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_ANGLE_translated_shader_source"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_translated_shader_source"))))]
            {
                self.GetTranslatedShaderSourceANGLE_load_with_dyn(get_proc_address);
            }
            self.GetUniformBlockIndex_load_with_dyn(get_proc_address);
            self.GetUniformIndices_load_with_dyn(get_proc_address);
            self.GetUniformLocation_load_with_dyn(get_proc_address);
            self.GetUniformfv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.GetUniformi64vNV_load_with_dyn(get_proc_address);
            }
            self.GetUniformiv_load_with_dyn(get_proc_address);
            self.GetUniformuiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
            )]
            {
                self.GetUnsignedBytei_vEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
            )]
            {
                self.GetUnsignedBytevEXT_load_with_dyn(get_proc_address);
            }
            self.GetVertexAttribIiv_load_with_dyn(get_proc_address);
            self.GetVertexAttribIuiv_load_with_dyn(get_proc_address);
            self.GetVertexAttribPointerv_load_with_dyn(get_proc_address);
            self.GetVertexAttribfv_load_with_dyn(get_proc_address);
            self.GetVertexAttribiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
            {
                self.GetVkProcAddrNV_load_with_dyn(get_proc_address);
            }
            self.GetnUniformfv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_robustness"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
            {
                self.GetnUniformfvEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_KHR_robustness"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
            {
                self.GetnUniformfvKHR_load_with_dyn(get_proc_address);
            }
            self.GetnUniformiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_robustness"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
            {
                self.GetnUniformivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_KHR_robustness"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
            {
                self.GetnUniformivKHR_load_with_dyn(get_proc_address);
            }
            self.GetnUniformuiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_robustness"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
            {
                self.GetnUniformuivKHR_load_with_dyn(get_proc_address);
            }
            self.Hint_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_memory_object_fd"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object_fd"))))]
            {
                self.ImportMemoryFdEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object_win32"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object_win32"))))]
            {
                self.ImportMemoryWin32HandleEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object_win32"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object_win32"))))]
            {
                self.ImportMemoryWin32NameEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_semaphore_fd"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore_fd"))))]
            {
                self.ImportSemaphoreFdEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_semaphore_win32"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore_win32"))))]
            {
                self.ImportSemaphoreWin32HandleEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_semaphore_win32"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore_win32"))))]
            {
                self.ImportSemaphoreWin32NameEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_debug_marker"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_marker"))))]
            {
                self.InsertEventMarkerEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.InterpolatePathsNV_load_with_dyn(get_proc_address);
            }
            self.InvalidateFramebuffer_load_with_dyn(get_proc_address);
            self.InvalidateSubFramebuffer_load_with_dyn(get_proc_address);
            self.IsBuffer_load_with_dyn(get_proc_address);
            self.IsEnabled_load_with_dyn(get_proc_address);
            self.IsEnabledi_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
            {
                self.IsEnablediEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.IsEnablediNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_OES_draw_buffers_indexed",
                    feature = "GL_OES_viewport_array"
                )))
            )]
            {
                self.IsEnablediOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_fence"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
            {
                self.IsFenceNV_load_with_dyn(get_proc_address);
            }
            self.IsFramebuffer_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.IsImageHandleResidentNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.IsMemoryObjectEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.IsPathNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.IsPointInFillPathNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.IsPointInStrokePathNV_load_with_dyn(get_proc_address);
            }
            self.IsProgram_load_with_dyn(get_proc_address);
            self.IsProgramPipeline_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.IsProgramPipelineEXT_load_with_dyn(get_proc_address);
            }
            self.IsQuery_load_with_dyn(get_proc_address);
            #[cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_disjoint_timer_query",
                    feature = "GL_EXT_occlusion_query_boolean"
                )))
            )]
            {
                self.IsQueryEXT_load_with_dyn(get_proc_address);
            }
            self.IsRenderbuffer_load_with_dyn(get_proc_address);
            self.IsSampler_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
            {
                self.IsSemaphoreEXT_load_with_dyn(get_proc_address);
            }
            self.IsShader_load_with_dyn(get_proc_address);
            self.IsSync_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_APPLE_sync"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
            {
                self.IsSyncAPPLE_load_with_dyn(get_proc_address);
            }
            self.IsTexture_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.IsTextureHandleResidentNV_load_with_dyn(get_proc_address);
            }
            self.IsTransformFeedback_load_with_dyn(get_proc_address);
            self.IsVertexArray_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_vertex_array_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
            {
                self.IsVertexArrayOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_debug_label"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
            {
                self.LabelObjectEXT_load_with_dyn(get_proc_address);
            }
            self.LineWidth_load_with_dyn(get_proc_address);
            self.LinkProgram_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.MakeImageHandleNonResidentNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.MakeImageHandleResidentNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.MakeTextureHandleNonResidentNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.MakeTextureHandleResidentNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_mapbuffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
            {
                self.MapBufferOES_load_with_dyn(get_proc_address);
            }
            self.MapBufferRange_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
            {
                self.MapBufferRangeEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixFrustumEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixLoad3x2fNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixLoad3x3fNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixLoadIdentityEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixLoadTranspose3x3fNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixLoadTransposedEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixLoadTransposefEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixLoaddEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixLoadfEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixMult3x2fNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixMult3x3fNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixMultTranspose3x3fNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixMultTransposedEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixMultTransposefEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixMultdEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixMultfEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixOrthoEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixPopEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixPushEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixRotatedEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixRotatefEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixScaledEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixScalefEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixTranslatedEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.MatrixTranslatefEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_KHR_parallel_shader_compile"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_parallel_shader_compile"))))]
            {
                self.MaxShaderCompilerThreadsKHR_load_with_dyn(get_proc_address);
            }
            self.MemoryBarrier_load_with_dyn(get_proc_address);
            self.MemoryBarrierByRegion_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.MemoryObjectParameterivEXT_load_with_dyn(get_proc_address);
            }
            self.MinSampleShading_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_sample_shading"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_sample_shading"))))]
            {
                self.MinSampleShadingOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_multi_draw_arrays"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multi_draw_arrays"))))]
            {
                self.MultiDrawArraysEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_multi_draw_indirect"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multi_draw_indirect"))))]
            {
                self.MultiDrawArraysIndirectEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(
                feature = "GL_EXT_draw_elements_base_vertex",
                feature = "GL_OES_draw_elements_base_vertex"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_draw_elements_base_vertex",
                    feature = "GL_OES_draw_elements_base_vertex"
                )))
            )]
            {
                self.MultiDrawElementsBaseVertexEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_multi_draw_arrays"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multi_draw_arrays"))))]
            {
                self.MultiDrawElementsEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_multi_draw_indirect"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multi_draw_indirect"))))]
            {
                self.MultiDrawElementsIndirectEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_mesh_shader"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
            {
                self.MultiDrawMeshTasksIndirectCountNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_mesh_shader"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
            {
                self.MultiDrawMeshTasksIndirectNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_memory_attachment"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
            {
                self.NamedBufferAttachMemoryNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
            {
                self.NamedBufferPageCommitmentMemNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_external_buffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
            {
                self.NamedBufferStorageExternalEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.NamedBufferStorageMemEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_sample_locations"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
            {
                self.NamedFramebufferSampleLocationsfvNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_framebuffer_multisample_advanced"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_AMD_framebuffer_multisample_advanced")))
            )]
            {
                self.NamedRenderbufferStorageMultisampleAdvancedAMD_load_with_dyn(get_proc_address);
            }
            self.ObjectLabel_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.ObjectLabelKHR_load_with_dyn(get_proc_address);
            }
            self.ObjectPtrLabel_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.ObjectPtrLabelKHR_load_with_dyn(get_proc_address);
            }
            self.PatchParameteri_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_tessellation_shader"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_tessellation_shader"))))]
            {
                self.PatchParameteriEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_tessellation_shader"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_tessellation_shader"))))]
            {
                self.PatchParameteriOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathCommandsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathCoordsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathCoverDepthFuncNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathDashArrayNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathGlyphIndexArrayNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathGlyphIndexRangeNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathGlyphRangeNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathGlyphsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathMemoryGlyphIndexArrayNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathParameterfNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathParameterfvNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathParameteriNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathParameterivNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathStencilDepthOffsetNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathStencilFuncNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathStringNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathSubCommandsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PathSubCoordsNV_load_with_dyn(get_proc_address);
            }
            self.PauseTransformFeedback_load_with_dyn(get_proc_address);
            self.PixelStorei_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.PointAlongPathNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_polygon_mode"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_polygon_mode"))))]
            {
                self.PolygonModeNV_load_with_dyn(get_proc_address);
            }
            self.PolygonOffset_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_polygon_offset_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_polygon_offset_clamp"))))]
            {
                self.PolygonOffsetClampEXT_load_with_dyn(get_proc_address);
            }
            self.PopDebugGroup_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.PopDebugGroupKHR_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_debug_marker"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_marker"))))]
            {
                self.PopGroupMarkerEXT_load_with_dyn(get_proc_address);
            }
            self.PrimitiveBoundingBox_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_primitive_bounding_box"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_primitive_bounding_box"))))]
            {
                self.PrimitiveBoundingBoxEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_primitive_bounding_box"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_primitive_bounding_box"))))]
            {
                self.PrimitiveBoundingBoxOES_load_with_dyn(get_proc_address);
            }
            self.ProgramBinary_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_get_program_binary"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
            {
                self.ProgramBinaryOES_load_with_dyn(get_proc_address);
            }
            self.ProgramParameteri_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramParameteriEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.ProgramPathFragmentInputGenNV_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform1f_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform1fEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform1fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform1fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform1i_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform1i64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform1i64vNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform1iEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform1iv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform1ivEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform1ui_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform1ui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform1ui64vNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform1uiEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform1uiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform1uivEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform2f_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform2fEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform2fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform2fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform2i_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform2i64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform2i64vNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform2iEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform2iv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform2ivEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform2ui_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform2ui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform2ui64vNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform2uiEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform2uiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform2uivEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform3f_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform3fEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform3fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform3fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform3i_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform3i64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform3i64vNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform3iEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform3iv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform3ivEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform3ui_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform3ui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform3ui64vNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform3uiEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform3uiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform3uivEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform4f_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform4fEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform4fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform4fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform4i_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform4i64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform4i64vNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform4iEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform4iv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform4ivEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform4ui_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform4ui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.ProgramUniform4ui64vNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform4uiEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniform4uiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniform4uivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_IMG_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
            {
                self.ProgramUniformHandleui64IMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.ProgramUniformHandleui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_IMG_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
            {
                self.ProgramUniformHandleui64vIMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.ProgramUniformHandleui64vNV_load_with_dyn(get_proc_address);
            }
            self.ProgramUniformMatrix2fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniformMatrix2fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniformMatrix2x3fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniformMatrix2x3fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniformMatrix2x4fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniformMatrix2x4fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniformMatrix3fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniformMatrix3fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniformMatrix3x2fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniformMatrix3x2fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniformMatrix3x4fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniformMatrix3x4fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniformMatrix4fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniformMatrix4fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniformMatrix4x2fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniformMatrix4x2fvEXT_load_with_dyn(get_proc_address);
            }
            self.ProgramUniformMatrix4x3fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ProgramUniformMatrix4x3fvEXT_load_with_dyn(get_proc_address);
            }
            self.PushDebugGroup_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_KHR_debug"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
            {
                self.PushDebugGroupKHR_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_debug_marker"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_marker"))))]
            {
                self.PushGroupMarkerEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
            {
                self.QueryCounterEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(
                feature = "GL_EXT_raster_multisample",
                feature = "GL_NV_framebuffer_mixed_samples"
            ))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(
                    feature = "GL_EXT_raster_multisample",
                    feature = "GL_NV_framebuffer_mixed_samples"
                )))
            )]
            {
                self.RasterSamplesEXT_load_with_dyn(get_proc_address);
            }
            self.ReadBuffer_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
            {
                self.ReadBufferIndexedEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_read_buffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_read_buffer"))))]
            {
                self.ReadBufferNV_load_with_dyn(get_proc_address);
            }
            self.ReadPixels_load_with_dyn(get_proc_address);
            self.ReadnPixels_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_robustness"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
            {
                self.ReadnPixelsEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_KHR_robustness"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
            {
                self.ReadnPixelsKHR_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
            {
                self.ReleaseKeyedMutexWin32EXT_load_with_dyn(get_proc_address);
            }
            self.ReleaseShaderCompiler_load_with_dyn(get_proc_address);
            self.RenderbufferStorage_load_with_dyn(get_proc_address);
            self.RenderbufferStorageMultisample_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_ANGLE_framebuffer_multisample"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_multisample"))))]
            {
                self.RenderbufferStorageMultisampleANGLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_APPLE_framebuffer_multisample"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_framebuffer_multisample"))))]
            {
                self.RenderbufferStorageMultisampleAPPLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_framebuffer_multisample_advanced"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_AMD_framebuffer_multisample_advanced")))
            )]
            {
                self.RenderbufferStorageMultisampleAdvancedAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
            )]
            {
                self.RenderbufferStorageMultisampleEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
            )]
            {
                self.RenderbufferStorageMultisampleIMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_framebuffer_multisample"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_multisample"))))]
            {
                self.RenderbufferStorageMultisampleNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_memory_attachment"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
            {
                self.ResetMemoryObjectParameterNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_sample_locations"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
            {
                self.ResolveDepthValuesNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_APPLE_framebuffer_multisample"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_framebuffer_multisample"))))]
            {
                self.ResolveMultisampleFramebufferAPPLE_load_with_dyn(get_proc_address);
            }
            self.ResumeTransformFeedback_load_with_dyn(get_proc_address);
            self.SampleCoverage_load_with_dyn(get_proc_address);
            self.SampleMaski_load_with_dyn(get_proc_address);
            self.SamplerParameterIiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
            {
                self.SamplerParameterIivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
            {
                self.SamplerParameterIivOES_load_with_dyn(get_proc_address);
            }
            self.SamplerParameterIuiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
            {
                self.SamplerParameterIuivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
            {
                self.SamplerParameterIuivOES_load_with_dyn(get_proc_address);
            }
            self.SamplerParameterf_load_with_dyn(get_proc_address);
            self.SamplerParameterfv_load_with_dyn(get_proc_address);
            self.SamplerParameteri_load_with_dyn(get_proc_address);
            self.SamplerParameteriv_load_with_dyn(get_proc_address);
            self.Scissor_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.ScissorArrayvNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
            {
                self.ScissorArrayvOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_scissor_exclusive"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_scissor_exclusive"))))]
            {
                self.ScissorExclusiveArrayvNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_scissor_exclusive"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_scissor_exclusive"))))]
            {
                self.ScissorExclusiveNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.ScissorIndexedNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
            {
                self.ScissorIndexedOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.ScissorIndexedvNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
            {
                self.ScissorIndexedvOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_AMD_performance_monitor"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
            {
                self.SelectPerfMonitorCountersAMD_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
            {
                self.SemaphoreParameterivNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
            {
                self.SemaphoreParameterui64vEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_fence"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
            {
                self.SetFenceNV_load_with_dyn(get_proc_address);
            }
            self.ShaderBinary_load_with_dyn(get_proc_address);
            self.ShaderSource_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_shading_rate_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
            {
                self.ShadingRateImageBarrierNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_shading_rate_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
            {
                self.ShadingRateImagePaletteNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_shading_rate"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_shading_rate"))))]
            {
                self.ShadingRateQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_shading_rate_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
            {
                self.ShadingRateSampleOrderCustomNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_shading_rate_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
            {
                self.ShadingRateSampleOrderNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
            {
                self.SignalSemaphoreEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
            {
                self.SignalVkFenceNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
            {
                self.SignalVkSemaphoreNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
            {
                self.StartTilingQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.StencilFillPathInstancedNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.StencilFillPathNV_load_with_dyn(get_proc_address);
            }
            self.StencilFunc_load_with_dyn(get_proc_address);
            self.StencilFuncSeparate_load_with_dyn(get_proc_address);
            self.StencilMask_load_with_dyn(get_proc_address);
            self.StencilMaskSeparate_load_with_dyn(get_proc_address);
            self.StencilOp_load_with_dyn(get_proc_address);
            self.StencilOpSeparate_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.StencilStrokePathInstancedNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.StencilStrokePathNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.StencilThenCoverFillPathInstancedNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.StencilThenCoverFillPathNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.StencilThenCoverStrokePathInstancedNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.StencilThenCoverStrokePathNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_conservative_raster"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conservative_raster"))))]
            {
                self.SubpixelPrecisionBiasNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_fence"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
            {
                self.TestFenceNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_memory_attachment"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
            {
                self.TexAttachMemoryNV_load_with_dyn(get_proc_address);
            }
            self.TexBuffer_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_buffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_buffer"))))]
            {
                self.TexBufferEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_buffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_buffer"))))]
            {
                self.TexBufferOES_load_with_dyn(get_proc_address);
            }
            self.TexBufferRange_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_buffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_buffer"))))]
            {
                self.TexBufferRangeEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_buffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_buffer"))))]
            {
                self.TexBufferRangeOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_motion_estimation"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_motion_estimation"))))]
            {
                self.TexEstimateMotionQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_motion_estimation"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_motion_estimation"))))]
            {
                self.TexEstimateMotionRegionsQCOM_load_with_dyn(get_proc_address);
            }
            self.TexImage2D_load_with_dyn(get_proc_address);
            self.TexImage3D_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_texture_3D"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
            {
                self.TexImage3DOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_sparse_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_sparse_texture"))))]
            {
                self.TexPageCommitmentEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
            {
                self.TexPageCommitmentMemNV_load_with_dyn(get_proc_address);
            }
            self.TexParameterIiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
            {
                self.TexParameterIivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
            {
                self.TexParameterIivOES_load_with_dyn(get_proc_address);
            }
            self.TexParameterIuiv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
            {
                self.TexParameterIuivEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
            {
                self.TexParameterIuivOES_load_with_dyn(get_proc_address);
            }
            self.TexParameterf_load_with_dyn(get_proc_address);
            self.TexParameterfv_load_with_dyn(get_proc_address);
            self.TexParameteri_load_with_dyn(get_proc_address);
            self.TexParameteriv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_storage"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
            {
                self.TexStorage1DEXT_load_with_dyn(get_proc_address);
            }
            self.TexStorage2D_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_storage"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
            {
                self.TexStorage2DEXT_load_with_dyn(get_proc_address);
            }
            self.TexStorage2DMultisample_load_with_dyn(get_proc_address);
            self.TexStorage3D_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_texture_storage"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
            {
                self.TexStorage3DEXT_load_with_dyn(get_proc_address);
            }
            self.TexStorage3DMultisample_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_texture_storage_multisample_2d_array"))]
            #[cfg_attr(
                docs_rs,
                doc(cfg(any(feature = "GL_OES_texture_storage_multisample_2d_array")))
            )]
            {
                self.TexStorage3DMultisampleOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.TexStorageMem2DEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.TexStorageMem2DMultisampleEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.TexStorageMem3DEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.TexStorageMem3DMultisampleEXT_load_with_dyn(get_proc_address);
            }
            self.TexSubImage2D_load_with_dyn(get_proc_address);
            self.TexSubImage3D_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_texture_3D"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
            {
                self.TexSubImage3DOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_memory_attachment"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
            {
                self.TextureAttachMemoryNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_QCOM_texture_foveated"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_texture_foveated"))))]
            {
                self.TextureFoveationParametersQCOM_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
            {
                self.TexturePageCommitmentMemNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_texture_storage"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
            {
                self.TextureStorage1DEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_texture_storage"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
            {
                self.TextureStorage2DEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_texture_storage"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
            {
                self.TextureStorage3DEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.TextureStorageMem2DEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.TextureStorageMem2DMultisampleEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.TextureStorageMem3DEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_memory_object"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
            {
                self.TextureStorageMem3DMultisampleEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_texture_view"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_view"))))]
            {
                self.TextureViewEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_texture_view"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_view"))))]
            {
                self.TextureViewOES_load_with_dyn(get_proc_address);
            }
            self.TransformFeedbackVaryings_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.TransformPathNV_load_with_dyn(get_proc_address);
            }
            self.Uniform1f_load_with_dyn(get_proc_address);
            self.Uniform1fv_load_with_dyn(get_proc_address);
            self.Uniform1i_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform1i64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform1i64vNV_load_with_dyn(get_proc_address);
            }
            self.Uniform1iv_load_with_dyn(get_proc_address);
            self.Uniform1ui_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform1ui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform1ui64vNV_load_with_dyn(get_proc_address);
            }
            self.Uniform1uiv_load_with_dyn(get_proc_address);
            self.Uniform2f_load_with_dyn(get_proc_address);
            self.Uniform2fv_load_with_dyn(get_proc_address);
            self.Uniform2i_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform2i64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform2i64vNV_load_with_dyn(get_proc_address);
            }
            self.Uniform2iv_load_with_dyn(get_proc_address);
            self.Uniform2ui_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform2ui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform2ui64vNV_load_with_dyn(get_proc_address);
            }
            self.Uniform2uiv_load_with_dyn(get_proc_address);
            self.Uniform3f_load_with_dyn(get_proc_address);
            self.Uniform3fv_load_with_dyn(get_proc_address);
            self.Uniform3i_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform3i64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform3i64vNV_load_with_dyn(get_proc_address);
            }
            self.Uniform3iv_load_with_dyn(get_proc_address);
            self.Uniform3ui_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform3ui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform3ui64vNV_load_with_dyn(get_proc_address);
            }
            self.Uniform3uiv_load_with_dyn(get_proc_address);
            self.Uniform4f_load_with_dyn(get_proc_address);
            self.Uniform4fv_load_with_dyn(get_proc_address);
            self.Uniform4i_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform4i64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform4i64vNV_load_with_dyn(get_proc_address);
            }
            self.Uniform4iv_load_with_dyn(get_proc_address);
            self.Uniform4ui_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform4ui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_gpu_shader5"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
            {
                self.Uniform4ui64vNV_load_with_dyn(get_proc_address);
            }
            self.Uniform4uiv_load_with_dyn(get_proc_address);
            self.UniformBlockBinding_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_IMG_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
            {
                self.UniformHandleui64IMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.UniformHandleui64NV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_IMG_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
            {
                self.UniformHandleui64vIMG_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_bindless_texture"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
            {
                self.UniformHandleui64vNV_load_with_dyn(get_proc_address);
            }
            self.UniformMatrix2fv_load_with_dyn(get_proc_address);
            self.UniformMatrix2x3fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_non_square_matrices"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
            {
                self.UniformMatrix2x3fvNV_load_with_dyn(get_proc_address);
            }
            self.UniformMatrix2x4fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_non_square_matrices"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
            {
                self.UniformMatrix2x4fvNV_load_with_dyn(get_proc_address);
            }
            self.UniformMatrix3fv_load_with_dyn(get_proc_address);
            self.UniformMatrix3x2fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_non_square_matrices"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
            {
                self.UniformMatrix3x2fvNV_load_with_dyn(get_proc_address);
            }
            self.UniformMatrix3x4fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_non_square_matrices"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
            {
                self.UniformMatrix3x4fvNV_load_with_dyn(get_proc_address);
            }
            self.UniformMatrix4fv_load_with_dyn(get_proc_address);
            self.UniformMatrix4x2fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_non_square_matrices"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
            {
                self.UniformMatrix4x2fvNV_load_with_dyn(get_proc_address);
            }
            self.UniformMatrix4x3fv_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_non_square_matrices"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
            {
                self.UniformMatrix4x3fvNV_load_with_dyn(get_proc_address);
            }
            self.UnmapBuffer_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_OES_mapbuffer"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
            {
                self.UnmapBufferOES_load_with_dyn(get_proc_address);
            }
            self.UseProgram_load_with_dyn(get_proc_address);
            self.UseProgramStages_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.UseProgramStagesEXT_load_with_dyn(get_proc_address);
            }
            self.ValidateProgram_load_with_dyn(get_proc_address);
            self.ValidateProgramPipeline_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
            {
                self.ValidateProgramPipelineEXT_load_with_dyn(get_proc_address);
            }
            self.VertexAttrib1f_load_with_dyn(get_proc_address);
            self.VertexAttrib1fv_load_with_dyn(get_proc_address);
            self.VertexAttrib2f_load_with_dyn(get_proc_address);
            self.VertexAttrib2fv_load_with_dyn(get_proc_address);
            self.VertexAttrib3f_load_with_dyn(get_proc_address);
            self.VertexAttrib3fv_load_with_dyn(get_proc_address);
            self.VertexAttrib4f_load_with_dyn(get_proc_address);
            self.VertexAttrib4fv_load_with_dyn(get_proc_address);
            self.VertexAttribBinding_load_with_dyn(get_proc_address);
            self.VertexAttribDivisor_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
            {
                self.VertexAttribDivisorANGLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_instanced_arrays"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_instanced_arrays"))))]
            {
                self.VertexAttribDivisorEXT_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_instanced_arrays"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_instanced_arrays"))))]
            {
                self.VertexAttribDivisorNV_load_with_dyn(get_proc_address);
            }
            self.VertexAttribFormat_load_with_dyn(get_proc_address);
            self.VertexAttribI4i_load_with_dyn(get_proc_address);
            self.VertexAttribI4iv_load_with_dyn(get_proc_address);
            self.VertexAttribI4ui_load_with_dyn(get_proc_address);
            self.VertexAttribI4uiv_load_with_dyn(get_proc_address);
            self.VertexAttribIFormat_load_with_dyn(get_proc_address);
            self.VertexAttribIPointer_load_with_dyn(get_proc_address);
            self.VertexAttribPointer_load_with_dyn(get_proc_address);
            self.VertexBindingDivisor_load_with_dyn(get_proc_address);
            self.Viewport_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.ViewportArrayvNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
            {
                self.ViewportArrayvOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.ViewportIndexedfNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
            {
                self.ViewportIndexedfOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
            {
                self.ViewportIndexedfvNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_OES_viewport_array"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
            {
                self.ViewportIndexedfvOES_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_clip_space_w_scaling"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_clip_space_w_scaling"))))]
            {
                self.ViewportPositionWScaleNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_viewport_swizzle"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_swizzle"))))]
            {
                self.ViewportSwizzleNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_semaphore"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
            {
                self.WaitSemaphoreEXT_load_with_dyn(get_proc_address);
            }
            self.WaitSync_load_with_dyn(get_proc_address);
            #[cfg(any(feature = "GL_APPLE_sync"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
            {
                self.WaitSyncAPPLE_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
            {
                self.WaitVkSemaphoreNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_NV_path_rendering"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
            {
                self.WeightPathsNV_load_with_dyn(get_proc_address);
            }
            #[cfg(any(feature = "GL_EXT_window_rectangles"))]
            #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_window_rectangles"))))]
            {
                self.WindowRectanglesEXT_load_with_dyn(get_proc_address);
            }
        }
        /// [glAcquireKeyedMutexWin32EXT](http://docs.gl/es3/glAcquireKeyedMutexWin32EXT)(memory, key, timeout)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
        pub unsafe fn AcquireKeyedMutexWin32EXT(
            &self,
            memory: GLuint,
            key: GLuint64,
            timeout: GLuint,
        ) -> GLboolean {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.AcquireKeyedMutexWin32EXT({:?}, {:?}, {:?});",
                    memory,
                    key,
                    timeout
                );
            }
            let out = call_atomic_ptr_3arg(
                "glAcquireKeyedMutexWin32EXT",
                &self.glAcquireKeyedMutexWin32EXT_p,
                memory,
                key,
                timeout,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glAcquireKeyedMutexWin32EXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
        #[doc(hidden)]
        pub unsafe fn AcquireKeyedMutexWin32EXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glAcquireKeyedMutexWin32EXT\0",
                &self.glAcquireKeyedMutexWin32EXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
        pub fn AcquireKeyedMutexWin32EXT_is_loaded(&self) -> bool {
            !self.glAcquireKeyedMutexWin32EXT_p.load(RELAX).is_null()
        }
        /// [glActiveShaderProgram](http://docs.gl/es3/glActiveShaderProgram)(pipeline, program)
        /// * `pipeline` class: program pipeline
        /// * `program` class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ActiveShaderProgram(&self, pipeline: GLuint, program: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ActiveShaderProgram({:?}, {:?});",
                    pipeline,
                    program
                );
            }
            let out = call_atomic_ptr_2arg(
                "glActiveShaderProgram",
                &self.glActiveShaderProgram_p,
                pipeline,
                program,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glActiveShaderProgram");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ActiveShaderProgram_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glActiveShaderProgram\0",
                &self.glActiveShaderProgram_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn ActiveShaderProgram_is_loaded(&self) -> bool {
            !self.glActiveShaderProgram_p.load(RELAX).is_null()
        }
        /// [glActiveShaderProgramEXT](http://docs.gl/es3/glActiveShaderProgramEXT)(pipeline, program)
        /// * `pipeline` class: program pipeline
        /// * `program` class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub unsafe fn ActiveShaderProgramEXT(&self, pipeline: GLuint, program: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ActiveShaderProgramEXT({:?}, {:?});",
                    pipeline,
                    program
                );
            }
            let out = call_atomic_ptr_2arg(
                "glActiveShaderProgramEXT",
                &self.glActiveShaderProgramEXT_p,
                pipeline,
                program,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glActiveShaderProgramEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        #[doc(hidden)]
        pub unsafe fn ActiveShaderProgramEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glActiveShaderProgramEXT\0",
                &self.glActiveShaderProgramEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub fn ActiveShaderProgramEXT_is_loaded(&self) -> bool {
            !self.glActiveShaderProgramEXT_p.load(RELAX).is_null()
        }
        /// [glActiveTexture](http://docs.gl/es3/glActiveTexture)(texture)
        /// * `texture` group: TextureUnit
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ActiveTexture(&self, texture: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.ActiveTexture({:#X});", texture);
            }
            let out = call_atomic_ptr_1arg("glActiveTexture", &self.glActiveTexture_p, texture);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glActiveTexture");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ActiveTexture_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glActiveTexture\0",
                &self.glActiveTexture_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn ActiveTexture_is_loaded(&self) -> bool {
            !self.glActiveTexture_p.load(RELAX).is_null()
        }
        /// [glAlphaFuncQCOM](http://docs.gl/es3/glAlphaFuncQCOM)(func, ref_)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_alpha_test"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_alpha_test"))))]
        pub unsafe fn AlphaFuncQCOM(&self, func: GLenum, ref_: GLclampf) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.AlphaFuncQCOM({:#X}, {:?});", func, ref_);
            }
            let out = call_atomic_ptr_2arg("glAlphaFuncQCOM", &self.glAlphaFuncQCOM_p, func, ref_);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glAlphaFuncQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_alpha_test"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_alpha_test"))))]
        #[doc(hidden)]
        pub unsafe fn AlphaFuncQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glAlphaFuncQCOM\0",
                &self.glAlphaFuncQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_alpha_test"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_alpha_test"))))]
        pub fn AlphaFuncQCOM_is_loaded(&self) -> bool {
            !self.glAlphaFuncQCOM_p.load(RELAX).is_null()
        }
        /// [glApplyFramebufferAttachmentCMAAINTEL](http://docs.gl/es3/glApplyFramebufferAttachmentCMAAINTEL)()
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))))]
        pub unsafe fn ApplyFramebufferAttachmentCMAAINTEL(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.ApplyFramebufferAttachmentCMAAINTEL();",);
            }
            let out = call_atomic_ptr_0arg(
                "glApplyFramebufferAttachmentCMAAINTEL",
                &self.glApplyFramebufferAttachmentCMAAINTEL_p,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glApplyFramebufferAttachmentCMAAINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))))]
        #[doc(hidden)]
        pub unsafe fn ApplyFramebufferAttachmentCMAAINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glApplyFramebufferAttachmentCMAAINTEL\0",
                &self.glApplyFramebufferAttachmentCMAAINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))))]
        pub fn ApplyFramebufferAttachmentCMAAINTEL_is_loaded(&self) -> bool {
            !self
                .glApplyFramebufferAttachmentCMAAINTEL_p
                .load(RELAX)
                .is_null()
        }
        /// [glAttachShader](http://docs.gl/es3/glAttachShader)(program, shader)
        /// * `program` class: program
        /// * `shader` class: shader
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn AttachShader(&self, program: GLuint, shader: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.AttachShader({:?}, {:?});", program, shader);
            }
            let out =
                call_atomic_ptr_2arg("glAttachShader", &self.glAttachShader_p, program, shader);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glAttachShader");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn AttachShader_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glAttachShader\0",
                &self.glAttachShader_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn AttachShader_is_loaded(&self) -> bool {
            !self.glAttachShader_p.load(RELAX).is_null()
        }
        /// [glBeginConditionalRenderNV](http://docs.gl/es3/glBeginConditionalRenderNV)(id, mode)
        /// * `mode` group: ConditionalRenderMode
        /// * alias of: [`glBeginConditionalRender`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_conditional_render"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
        pub unsafe fn BeginConditionalRenderNV(&self, id: GLuint, mode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BeginConditionalRenderNV({:?}, {:#X});",
                    id,
                    mode
                );
            }
            let out = call_atomic_ptr_2arg(
                "glBeginConditionalRenderNV",
                &self.glBeginConditionalRenderNV_p,
                id,
                mode,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBeginConditionalRenderNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_conditional_render"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
        #[doc(hidden)]
        pub unsafe fn BeginConditionalRenderNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBeginConditionalRenderNV\0",
                &self.glBeginConditionalRenderNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_conditional_render"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
        pub fn BeginConditionalRenderNV_is_loaded(&self) -> bool {
            !self.glBeginConditionalRenderNV_p.load(RELAX).is_null()
        }
        /// [glBeginPerfMonitorAMD](http://docs.gl/es3/glBeginPerfMonitorAMD)(monitor)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn BeginPerfMonitorAMD(&self, monitor: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BeginPerfMonitorAMD({:?});", monitor);
            }
            let out = call_atomic_ptr_1arg(
                "glBeginPerfMonitorAMD",
                &self.glBeginPerfMonitorAMD_p,
                monitor,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBeginPerfMonitorAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn BeginPerfMonitorAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBeginPerfMonitorAMD\0",
                &self.glBeginPerfMonitorAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn BeginPerfMonitorAMD_is_loaded(&self) -> bool {
            !self.glBeginPerfMonitorAMD_p.load(RELAX).is_null()
        }
        /// [glBeginPerfQueryINTEL](http://docs.gl/es3/glBeginPerfQueryINTEL)(queryHandle)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn BeginPerfQueryINTEL(&self, queryHandle: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BeginPerfQueryINTEL({:?});", queryHandle);
            }
            let out = call_atomic_ptr_1arg(
                "glBeginPerfQueryINTEL",
                &self.glBeginPerfQueryINTEL_p,
                queryHandle,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBeginPerfQueryINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn BeginPerfQueryINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBeginPerfQueryINTEL\0",
                &self.glBeginPerfQueryINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn BeginPerfQueryINTEL_is_loaded(&self) -> bool {
            !self.glBeginPerfQueryINTEL_p.load(RELAX).is_null()
        }
        /// [glBeginQuery](http://docs.gl/es3/glBeginQuery)(target, id)
        /// * `target` group: QueryTarget
        /// * `id` class: query
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BeginQuery(&self, target: GLenum, id: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BeginQuery({:#X}, {:?});", target, id);
            }
            let out = call_atomic_ptr_2arg("glBeginQuery", &self.glBeginQuery_p, target, id);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBeginQuery");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BeginQuery_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glBeginQuery\0", &self.glBeginQuery_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn BeginQuery_is_loaded(&self) -> bool {
            !self.glBeginQuery_p.load(RELAX).is_null()
        }
        /// [glBeginQueryEXT](http://docs.gl/es3/glBeginQueryEXT)(target, id)
        /// * `target` group: QueryTarget
        /// * `id` class: query
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub unsafe fn BeginQueryEXT(&self, target: GLenum, id: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BeginQueryEXT({:#X}, {:?});", target, id);
            }
            let out = call_atomic_ptr_2arg("glBeginQueryEXT", &self.glBeginQueryEXT_p, target, id);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBeginQueryEXT");
            }
            out
        }
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn BeginQueryEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBeginQueryEXT\0",
                &self.glBeginQueryEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub fn BeginQueryEXT_is_loaded(&self) -> bool {
            !self.glBeginQueryEXT_p.load(RELAX).is_null()
        }
        /// [glBeginTransformFeedback](http://docs.gl/es3/glBeginTransformFeedback)(primitiveMode)
        /// * `primitiveMode` group: PrimitiveType
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BeginTransformFeedback(&self, primitiveMode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BeginTransformFeedback({:#X});", primitiveMode);
            }
            let out = call_atomic_ptr_1arg(
                "glBeginTransformFeedback",
                &self.glBeginTransformFeedback_p,
                primitiveMode,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBeginTransformFeedback");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BeginTransformFeedback_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBeginTransformFeedback\0",
                &self.glBeginTransformFeedback_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BeginTransformFeedback_is_loaded(&self) -> bool {
            !self.glBeginTransformFeedback_p.load(RELAX).is_null()
        }
        /// [glBindAttribLocation](http://docs.gl/es3/glBindAttribLocation)(program, index, name)
        /// * `program` class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindAttribLocation(
            &self,
            program: GLuint,
            index: GLuint,
            name: *const GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BindAttribLocation({:?}, {:?}, {:p});",
                    program,
                    index,
                    name
                );
            }
            let out = call_atomic_ptr_3arg(
                "glBindAttribLocation",
                &self.glBindAttribLocation_p,
                program,
                index,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindAttribLocation");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindAttribLocation_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindAttribLocation\0",
                &self.glBindAttribLocation_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindAttribLocation_is_loaded(&self) -> bool {
            !self.glBindAttribLocation_p.load(RELAX).is_null()
        }
        /// [glBindBuffer](http://docs.gl/es3/glBindBuffer)(target, buffer)
        /// * `target` group: BufferTargetARB
        /// * `buffer` class: buffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindBuffer(&self, target: GLenum, buffer: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BindBuffer({:#X}, {:?});", target, buffer);
            }
            let out = call_atomic_ptr_2arg("glBindBuffer", &self.glBindBuffer_p, target, buffer);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindBuffer");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindBuffer_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glBindBuffer\0", &self.glBindBuffer_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindBuffer_is_loaded(&self) -> bool {
            !self.glBindBuffer_p.load(RELAX).is_null()
        }
        /// [glBindBufferBase](http://docs.gl/es3/glBindBufferBase)(target, index, buffer)
        /// * `target` group: BufferTargetARB
        /// * `buffer` class: buffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindBufferBase(&self, target: GLenum, index: GLuint, buffer: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BindBufferBase({:#X}, {:?}, {:?});",
                    target,
                    index,
                    buffer
                );
            }
            let out = call_atomic_ptr_3arg(
                "glBindBufferBase",
                &self.glBindBufferBase_p,
                target,
                index,
                buffer,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindBufferBase");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindBufferBase_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindBufferBase\0",
                &self.glBindBufferBase_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindBufferBase_is_loaded(&self) -> bool {
            !self.glBindBufferBase_p.load(RELAX).is_null()
        }
        /// [glBindBufferRange](http://docs.gl/es3/glBindBufferRange)(target, index, buffer, offset, size)
        /// * `target` group: BufferTargetARB
        /// * `buffer` class: buffer
        /// * `offset` group: BufferOffset
        /// * `size` group: BufferSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindBufferRange(
            &self,
            target: GLenum,
            index: GLuint,
            buffer: GLuint,
            offset: GLintptr,
            size: GLsizeiptr,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BindBufferRange({:#X}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    index,
                    buffer,
                    offset,
                    size
                );
            }
            let out = call_atomic_ptr_5arg(
                "glBindBufferRange",
                &self.glBindBufferRange_p,
                target,
                index,
                buffer,
                offset,
                size,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindBufferRange");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindBufferRange_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindBufferRange\0",
                &self.glBindBufferRange_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindBufferRange_is_loaded(&self) -> bool {
            !self.glBindBufferRange_p.load(RELAX).is_null()
        }
        /// [glBindFragDataLocationEXT](http://docs.gl/es3/glBindFragDataLocationEXT)(program, color, name)
        /// * `program` class: program
        /// * `name` len: COMPSIZE(name)
        /// * alias of: [`glBindFragDataLocation`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        pub unsafe fn BindFragDataLocationEXT(
            &self,
            program: GLuint,
            color: GLuint,
            name: *const GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BindFragDataLocationEXT({:?}, {:?}, {:p});",
                    program,
                    color,
                    name
                );
            }
            let out = call_atomic_ptr_3arg(
                "glBindFragDataLocationEXT",
                &self.glBindFragDataLocationEXT_p,
                program,
                color,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindFragDataLocationEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        #[doc(hidden)]
        pub unsafe fn BindFragDataLocationEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindFragDataLocationEXT\0",
                &self.glBindFragDataLocationEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        pub fn BindFragDataLocationEXT_is_loaded(&self) -> bool {
            !self.glBindFragDataLocationEXT_p.load(RELAX).is_null()
        }
        /// [glBindFragDataLocationIndexedEXT](http://docs.gl/es3/glBindFragDataLocationIndexedEXT)(program, colorNumber, index, name)
        /// * `program` class: program
        /// * alias of: [`glBindFragDataLocationIndexed`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        pub unsafe fn BindFragDataLocationIndexedEXT(
            &self,
            program: GLuint,
            colorNumber: GLuint,
            index: GLuint,
            name: *const GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BindFragDataLocationIndexedEXT({:?}, {:?}, {:?}, {:p});",
                    program,
                    colorNumber,
                    index,
                    name
                );
            }
            let out = call_atomic_ptr_4arg(
                "glBindFragDataLocationIndexedEXT",
                &self.glBindFragDataLocationIndexedEXT_p,
                program,
                colorNumber,
                index,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindFragDataLocationIndexedEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        #[doc(hidden)]
        pub unsafe fn BindFragDataLocationIndexedEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindFragDataLocationIndexedEXT\0",
                &self.glBindFragDataLocationIndexedEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        pub fn BindFragDataLocationIndexedEXT_is_loaded(&self) -> bool {
            !self
                .glBindFragDataLocationIndexedEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glBindFramebuffer](http://docs.gl/es3/glBindFramebuffer)(target, framebuffer)
        /// * `target` group: FramebufferTarget
        /// * `framebuffer` class: framebuffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindFramebuffer(&self, target: GLenum, framebuffer: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BindFramebuffer({:#X}, {:?});",
                    target,
                    framebuffer
                );
            }
            let out = call_atomic_ptr_2arg(
                "glBindFramebuffer",
                &self.glBindFramebuffer_p,
                target,
                framebuffer,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindFramebuffer");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindFramebuffer_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindFramebuffer\0",
                &self.glBindFramebuffer_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindFramebuffer_is_loaded(&self) -> bool {
            !self.glBindFramebuffer_p.load(RELAX).is_null()
        }
        /// [glBindImageTexture](http://docs.gl/es3/glBindImageTexture)(unit, texture, level, layered, layer, access, format)
        /// * `texture` class: texture
        /// * `access` group: BufferAccessARB
        /// * `format` group: InternalFormat
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindImageTexture(
            &self,
            unit: GLuint,
            texture: GLuint,
            level: GLint,
            layered: GLboolean,
            layer: GLint,
            access: GLenum,
            format: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BindImageTexture({:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:#X});",
                    unit,
                    texture,
                    level,
                    layered,
                    layer,
                    access,
                    format
                );
            }
            let out = call_atomic_ptr_7arg(
                "glBindImageTexture",
                &self.glBindImageTexture_p,
                unit,
                texture,
                level,
                layered,
                layer,
                access,
                format,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindImageTexture");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindImageTexture_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindImageTexture\0",
                &self.glBindImageTexture_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindImageTexture_is_loaded(&self) -> bool {
            !self.glBindImageTexture_p.load(RELAX).is_null()
        }
        /// [glBindProgramPipeline](http://docs.gl/es3/glBindProgramPipeline)(pipeline)
        /// * `pipeline` class: program pipeline
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindProgramPipeline(&self, pipeline: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BindProgramPipeline({:?});", pipeline);
            }
            let out = call_atomic_ptr_1arg(
                "glBindProgramPipeline",
                &self.glBindProgramPipeline_p,
                pipeline,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindProgramPipeline");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindProgramPipeline_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindProgramPipeline\0",
                &self.glBindProgramPipeline_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindProgramPipeline_is_loaded(&self) -> bool {
            !self.glBindProgramPipeline_p.load(RELAX).is_null()
        }
        /// [glBindProgramPipelineEXT](http://docs.gl/es3/glBindProgramPipelineEXT)(pipeline)
        /// * `pipeline` class: program pipeline
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub unsafe fn BindProgramPipelineEXT(&self, pipeline: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BindProgramPipelineEXT({:?});", pipeline);
            }
            let out = call_atomic_ptr_1arg(
                "glBindProgramPipelineEXT",
                &self.glBindProgramPipelineEXT_p,
                pipeline,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindProgramPipelineEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        #[doc(hidden)]
        pub unsafe fn BindProgramPipelineEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindProgramPipelineEXT\0",
                &self.glBindProgramPipelineEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub fn BindProgramPipelineEXT_is_loaded(&self) -> bool {
            !self.glBindProgramPipelineEXT_p.load(RELAX).is_null()
        }
        /// [glBindRenderbuffer](http://docs.gl/es3/glBindRenderbuffer)(target, renderbuffer)
        /// * `target` group: RenderbufferTarget
        /// * `renderbuffer` class: renderbuffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindRenderbuffer(&self, target: GLenum, renderbuffer: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BindRenderbuffer({:#X}, {:?});",
                    target,
                    renderbuffer
                );
            }
            let out = call_atomic_ptr_2arg(
                "glBindRenderbuffer",
                &self.glBindRenderbuffer_p,
                target,
                renderbuffer,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindRenderbuffer");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindRenderbuffer_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindRenderbuffer\0",
                &self.glBindRenderbuffer_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindRenderbuffer_is_loaded(&self) -> bool {
            !self.glBindRenderbuffer_p.load(RELAX).is_null()
        }
        /// [glBindSampler](http://docs.gl/es3/glBindSampler)(unit, sampler)
        /// * `sampler` class: sampler
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindSampler(&self, unit: GLuint, sampler: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BindSampler({:?}, {:?});", unit, sampler);
            }
            let out = call_atomic_ptr_2arg("glBindSampler", &self.glBindSampler_p, unit, sampler);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindSampler");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindSampler_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glBindSampler\0", &self.glBindSampler_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindSampler_is_loaded(&self) -> bool {
            !self.glBindSampler_p.load(RELAX).is_null()
        }
        /// [glBindShadingRateImageNV](http://docs.gl/es3/glBindShadingRateImageNV)(texture)
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        pub unsafe fn BindShadingRateImageNV(&self, texture: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BindShadingRateImageNV({:?});", texture);
            }
            let out = call_atomic_ptr_1arg(
                "glBindShadingRateImageNV",
                &self.glBindShadingRateImageNV_p,
                texture,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindShadingRateImageNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        #[doc(hidden)]
        pub unsafe fn BindShadingRateImageNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindShadingRateImageNV\0",
                &self.glBindShadingRateImageNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        pub fn BindShadingRateImageNV_is_loaded(&self) -> bool {
            !self.glBindShadingRateImageNV_p.load(RELAX).is_null()
        }
        /// [glBindTexture](http://docs.gl/es3/glBindTexture)(target, texture)
        /// * `target` group: TextureTarget
        /// * `texture` group: Texture
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindTexture(&self, target: GLenum, texture: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BindTexture({:#X}, {:?});", target, texture);
            }
            let out = call_atomic_ptr_2arg("glBindTexture", &self.glBindTexture_p, target, texture);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindTexture");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindTexture_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glBindTexture\0", &self.glBindTexture_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindTexture_is_loaded(&self) -> bool {
            !self.glBindTexture_p.load(RELAX).is_null()
        }
        /// [glBindTransformFeedback](http://docs.gl/es3/glBindTransformFeedback)(target, id)
        /// * `target` group: BindTransformFeedbackTarget
        /// * `id` class: transform feedback
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindTransformFeedback(&self, target: GLenum, id: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BindTransformFeedback({:#X}, {:?});", target, id);
            }
            let out = call_atomic_ptr_2arg(
                "glBindTransformFeedback",
                &self.glBindTransformFeedback_p,
                target,
                id,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindTransformFeedback");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindTransformFeedback_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindTransformFeedback\0",
                &self.glBindTransformFeedback_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindTransformFeedback_is_loaded(&self) -> bool {
            !self.glBindTransformFeedback_p.load(RELAX).is_null()
        }
        /// [glBindVertexArray](http://docs.gl/es3/glBindVertexArray)(array)
        /// * `array` class: vertex array
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindVertexArray(&self, array: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BindVertexArray({:?});", array);
            }
            let out = call_atomic_ptr_1arg("glBindVertexArray", &self.glBindVertexArray_p, array);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindVertexArray");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindVertexArray_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindVertexArray\0",
                &self.glBindVertexArray_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindVertexArray_is_loaded(&self) -> bool {
            !self.glBindVertexArray_p.load(RELAX).is_null()
        }
        /// [glBindVertexArrayOES](http://docs.gl/es3/glBindVertexArrayOES)(array)
        /// * `array` class: vertex array
        /// * alias of: [`glBindVertexArray`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        pub unsafe fn BindVertexArrayOES(&self, array: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BindVertexArrayOES({:?});", array);
            }
            let out =
                call_atomic_ptr_1arg("glBindVertexArrayOES", &self.glBindVertexArrayOES_p, array);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindVertexArrayOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        #[doc(hidden)]
        pub unsafe fn BindVertexArrayOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindVertexArrayOES\0",
                &self.glBindVertexArrayOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        pub fn BindVertexArrayOES_is_loaded(&self) -> bool {
            !self.glBindVertexArrayOES_p.load(RELAX).is_null()
        }
        /// [glBindVertexBuffer](http://docs.gl/es3/glBindVertexBuffer)(bindingindex, buffer, offset, stride)
        /// * `buffer` class: buffer
        /// * `offset` group: BufferOffset
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BindVertexBuffer(
            &self,
            bindingindex: GLuint,
            buffer: GLuint,
            offset: GLintptr,
            stride: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BindVertexBuffer({:?}, {:?}, {:?}, {:?});",
                    bindingindex,
                    buffer,
                    offset,
                    stride
                );
            }
            let out = call_atomic_ptr_4arg(
                "glBindVertexBuffer",
                &self.glBindVertexBuffer_p,
                bindingindex,
                buffer,
                offset,
                stride,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBindVertexBuffer");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BindVertexBuffer_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBindVertexBuffer\0",
                &self.glBindVertexBuffer_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BindVertexBuffer_is_loaded(&self) -> bool {
            !self.glBindVertexBuffer_p.load(RELAX).is_null()
        }
        /// [glBlendBarrier](http://docs.gl/es3/glBlendBarrier)()
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendBarrier(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendBarrier();",);
            }
            let out = call_atomic_ptr_0arg("glBlendBarrier", &self.glBlendBarrier_p);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendBarrier");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendBarrier_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendBarrier\0",
                &self.glBlendBarrier_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendBarrier_is_loaded(&self) -> bool {
            !self.glBlendBarrier_p.load(RELAX).is_null()
        }
        /// [glBlendBarrierKHR](http://docs.gl/es3/glBlendBarrierKHR)()
        /// * alias of: [`glBlendBarrier`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_KHR_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_blend_equation_advanced"))))]
        pub unsafe fn BlendBarrierKHR(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendBarrierKHR();",);
            }
            let out = call_atomic_ptr_0arg("glBlendBarrierKHR", &self.glBlendBarrierKHR_p);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendBarrierKHR");
            }
            out
        }
        #[cfg(any(feature = "GL_KHR_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_blend_equation_advanced"))))]
        #[doc(hidden)]
        pub unsafe fn BlendBarrierKHR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendBarrierKHR\0",
                &self.glBlendBarrierKHR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_KHR_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_blend_equation_advanced"))))]
        pub fn BlendBarrierKHR_is_loaded(&self) -> bool {
            !self.glBlendBarrierKHR_p.load(RELAX).is_null()
        }
        /// [glBlendBarrierNV](http://docs.gl/es3/glBlendBarrierNV)()
        /// * alias of: [`glBlendBarrier`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
        pub unsafe fn BlendBarrierNV(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendBarrierNV();",);
            }
            let out = call_atomic_ptr_0arg("glBlendBarrierNV", &self.glBlendBarrierNV_p);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendBarrierNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
        #[doc(hidden)]
        pub unsafe fn BlendBarrierNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendBarrierNV\0",
                &self.glBlendBarrierNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
        pub fn BlendBarrierNV_is_loaded(&self) -> bool {
            !self.glBlendBarrierNV_p.load(RELAX).is_null()
        }
        /// [glBlendColor](http://docs.gl/es3/glBlendColor)(red, green, blue, alpha)
        /// * `red` group: ColorF
        /// * `green` group: ColorF
        /// * `blue` group: ColorF
        /// * `alpha` group: ColorF
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendColor(
            &self,
            red: GLfloat,
            green: GLfloat,
            blue: GLfloat,
            alpha: GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendColor({:?}, {:?}, {:?}, {:?});",
                    red,
                    green,
                    blue,
                    alpha
                );
            }
            let out = call_atomic_ptr_4arg(
                "glBlendColor",
                &self.glBlendColor_p,
                red,
                green,
                blue,
                alpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendColor");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendColor_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glBlendColor\0", &self.glBlendColor_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendColor_is_loaded(&self) -> bool {
            !self.glBlendColor_p.load(RELAX).is_null()
        }
        /// [glBlendEquation](http://docs.gl/es3/glBlendEquation)(mode)
        /// * `mode` group: BlendEquationModeEXT
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendEquation(&self, mode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendEquation({:#X});", mode);
            }
            let out = call_atomic_ptr_1arg("glBlendEquation", &self.glBlendEquation_p, mode);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendEquation");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendEquation_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendEquation\0",
                &self.glBlendEquation_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendEquation_is_loaded(&self) -> bool {
            !self.glBlendEquation_p.load(RELAX).is_null()
        }
        /// [glBlendEquationSeparate](http://docs.gl/es3/glBlendEquationSeparate)(modeRGB, modeAlpha)
        /// * `modeRGB` group: BlendEquationModeEXT
        /// * `modeAlpha` group: BlendEquationModeEXT
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendEquationSeparate(&self, modeRGB: GLenum, modeAlpha: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendEquationSeparate({:#X}, {:#X});",
                    modeRGB,
                    modeAlpha
                );
            }
            let out = call_atomic_ptr_2arg(
                "glBlendEquationSeparate",
                &self.glBlendEquationSeparate_p,
                modeRGB,
                modeAlpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendEquationSeparate");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendEquationSeparate_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendEquationSeparate\0",
                &self.glBlendEquationSeparate_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendEquationSeparate_is_loaded(&self) -> bool {
            !self.glBlendEquationSeparate_p.load(RELAX).is_null()
        }
        /// [glBlendEquationSeparatei](http://docs.gl/es3/glBlendEquationSeparate)(buf, modeRGB, modeAlpha)
        /// * `modeRGB` group: BlendEquationModeEXT
        /// * `modeAlpha` group: BlendEquationModeEXT
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendEquationSeparatei(
            &self,
            buf: GLuint,
            modeRGB: GLenum,
            modeAlpha: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendEquationSeparatei({:?}, {:#X}, {:#X});",
                    buf,
                    modeRGB,
                    modeAlpha
                );
            }
            let out = call_atomic_ptr_3arg(
                "glBlendEquationSeparatei",
                &self.glBlendEquationSeparatei_p,
                buf,
                modeRGB,
                modeAlpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendEquationSeparatei");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendEquationSeparatei_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendEquationSeparatei\0",
                &self.glBlendEquationSeparatei_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendEquationSeparatei_is_loaded(&self) -> bool {
            !self.glBlendEquationSeparatei_p.load(RELAX).is_null()
        }
        /// [glBlendEquationSeparateiEXT](http://docs.gl/es3/glBlendEquationSeparateiEXT)(buf, modeRGB, modeAlpha)
        /// * `modeRGB` group: BlendEquationModeEXT
        /// * `modeAlpha` group: BlendEquationModeEXT
        /// * alias of: [`glBlendEquationSeparatei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub unsafe fn BlendEquationSeparateiEXT(
            &self,
            buf: GLuint,
            modeRGB: GLenum,
            modeAlpha: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendEquationSeparateiEXT({:?}, {:#X}, {:#X});",
                    buf,
                    modeRGB,
                    modeAlpha
                );
            }
            let out = call_atomic_ptr_3arg(
                "glBlendEquationSeparateiEXT",
                &self.glBlendEquationSeparateiEXT_p,
                buf,
                modeRGB,
                modeAlpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendEquationSeparateiEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn BlendEquationSeparateiEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendEquationSeparateiEXT\0",
                &self.glBlendEquationSeparateiEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub fn BlendEquationSeparateiEXT_is_loaded(&self) -> bool {
            !self.glBlendEquationSeparateiEXT_p.load(RELAX).is_null()
        }
        /// [glBlendEquationSeparateiOES](http://docs.gl/es3/glBlendEquationSeparateiOES)(buf, modeRGB, modeAlpha)
        /// * `modeRGB` group: BlendEquationModeEXT
        /// * `modeAlpha` group: BlendEquationModeEXT
        /// * alias of: [`glBlendEquationSeparatei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub unsafe fn BlendEquationSeparateiOES(
            &self,
            buf: GLuint,
            modeRGB: GLenum,
            modeAlpha: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendEquationSeparateiOES({:?}, {:#X}, {:#X});",
                    buf,
                    modeRGB,
                    modeAlpha
                );
            }
            let out = call_atomic_ptr_3arg(
                "glBlendEquationSeparateiOES",
                &self.glBlendEquationSeparateiOES_p,
                buf,
                modeRGB,
                modeAlpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendEquationSeparateiOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn BlendEquationSeparateiOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendEquationSeparateiOES\0",
                &self.glBlendEquationSeparateiOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub fn BlendEquationSeparateiOES_is_loaded(&self) -> bool {
            !self.glBlendEquationSeparateiOES_p.load(RELAX).is_null()
        }
        /// [glBlendEquationi](http://docs.gl/es3/glBlendEquation)(buf, mode)
        /// * `mode` group: BlendEquationModeEXT
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendEquationi(&self, buf: GLuint, mode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendEquationi({:?}, {:#X});", buf, mode);
            }
            let out = call_atomic_ptr_2arg("glBlendEquationi", &self.glBlendEquationi_p, buf, mode);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendEquationi");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendEquationi_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendEquationi\0",
                &self.glBlendEquationi_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendEquationi_is_loaded(&self) -> bool {
            !self.glBlendEquationi_p.load(RELAX).is_null()
        }
        /// [glBlendEquationiEXT](http://docs.gl/es3/glBlendEquationiEXT)(buf, mode)
        /// * `mode` group: BlendEquationModeEXT
        /// * alias of: [`glBlendEquationi`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub unsafe fn BlendEquationiEXT(&self, buf: GLuint, mode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendEquationiEXT({:?}, {:#X});", buf, mode);
            }
            let out = call_atomic_ptr_2arg(
                "glBlendEquationiEXT",
                &self.glBlendEquationiEXT_p,
                buf,
                mode,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendEquationiEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn BlendEquationiEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendEquationiEXT\0",
                &self.glBlendEquationiEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub fn BlendEquationiEXT_is_loaded(&self) -> bool {
            !self.glBlendEquationiEXT_p.load(RELAX).is_null()
        }
        /// [glBlendEquationiOES](http://docs.gl/es3/glBlendEquationiOES)(buf, mode)
        /// * `mode` group: BlendEquationModeEXT
        /// * alias of: [`glBlendEquationi`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub unsafe fn BlendEquationiOES(&self, buf: GLuint, mode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendEquationiOES({:?}, {:#X});", buf, mode);
            }
            let out = call_atomic_ptr_2arg(
                "glBlendEquationiOES",
                &self.glBlendEquationiOES_p,
                buf,
                mode,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendEquationiOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn BlendEquationiOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendEquationiOES\0",
                &self.glBlendEquationiOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub fn BlendEquationiOES_is_loaded(&self) -> bool {
            !self.glBlendEquationiOES_p.load(RELAX).is_null()
        }
        /// [glBlendFunc](http://docs.gl/es3/glBlendFunc)(sfactor, dfactor)
        /// * `sfactor` group: BlendingFactor
        /// * `dfactor` group: BlendingFactor
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendFunc(&self, sfactor: GLenum, dfactor: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendFunc({:#X}, {:#X});", sfactor, dfactor);
            }
            let out = call_atomic_ptr_2arg("glBlendFunc", &self.glBlendFunc_p, sfactor, dfactor);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendFunc");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendFunc_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glBlendFunc\0", &self.glBlendFunc_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendFunc_is_loaded(&self) -> bool {
            !self.glBlendFunc_p.load(RELAX).is_null()
        }
        /// [glBlendFuncSeparate](http://docs.gl/es3/glBlendFuncSeparate)(sfactorRGB, dfactorRGB, sfactorAlpha, dfactorAlpha)
        /// * `sfactorRGB` group: BlendingFactor
        /// * `dfactorRGB` group: BlendingFactor
        /// * `sfactorAlpha` group: BlendingFactor
        /// * `dfactorAlpha` group: BlendingFactor
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendFuncSeparate(
            &self,
            sfactorRGB: GLenum,
            dfactorRGB: GLenum,
            sfactorAlpha: GLenum,
            dfactorAlpha: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendFuncSeparate({:#X}, {:#X}, {:#X}, {:#X});",
                    sfactorRGB,
                    dfactorRGB,
                    sfactorAlpha,
                    dfactorAlpha
                );
            }
            let out = call_atomic_ptr_4arg(
                "glBlendFuncSeparate",
                &self.glBlendFuncSeparate_p,
                sfactorRGB,
                dfactorRGB,
                sfactorAlpha,
                dfactorAlpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendFuncSeparate");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendFuncSeparate_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendFuncSeparate\0",
                &self.glBlendFuncSeparate_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendFuncSeparate_is_loaded(&self) -> bool {
            !self.glBlendFuncSeparate_p.load(RELAX).is_null()
        }
        /// [glBlendFuncSeparatei](http://docs.gl/es3/glBlendFuncSeparate)(buf, srcRGB, dstRGB, srcAlpha, dstAlpha)
        /// * `srcRGB` group: BlendingFactor
        /// * `dstRGB` group: BlendingFactor
        /// * `srcAlpha` group: BlendingFactor
        /// * `dstAlpha` group: BlendingFactor
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendFuncSeparatei(
            &self,
            buf: GLuint,
            srcRGB: GLenum,
            dstRGB: GLenum,
            srcAlpha: GLenum,
            dstAlpha: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendFuncSeparatei({:?}, {:#X}, {:#X}, {:#X}, {:#X});",
                    buf,
                    srcRGB,
                    dstRGB,
                    srcAlpha,
                    dstAlpha
                );
            }
            let out = call_atomic_ptr_5arg(
                "glBlendFuncSeparatei",
                &self.glBlendFuncSeparatei_p,
                buf,
                srcRGB,
                dstRGB,
                srcAlpha,
                dstAlpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendFuncSeparatei");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendFuncSeparatei_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendFuncSeparatei\0",
                &self.glBlendFuncSeparatei_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendFuncSeparatei_is_loaded(&self) -> bool {
            !self.glBlendFuncSeparatei_p.load(RELAX).is_null()
        }
        /// [glBlendFuncSeparateiEXT](http://docs.gl/es3/glBlendFuncSeparateiEXT)(buf, srcRGB, dstRGB, srcAlpha, dstAlpha)
        /// * `srcRGB` group: BlendingFactor
        /// * `dstRGB` group: BlendingFactor
        /// * `srcAlpha` group: BlendingFactor
        /// * `dstAlpha` group: BlendingFactor
        /// * alias of: [`glBlendFuncSeparatei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub unsafe fn BlendFuncSeparateiEXT(
            &self,
            buf: GLuint,
            srcRGB: GLenum,
            dstRGB: GLenum,
            srcAlpha: GLenum,
            dstAlpha: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendFuncSeparateiEXT({:?}, {:#X}, {:#X}, {:#X}, {:#X});",
                    buf,
                    srcRGB,
                    dstRGB,
                    srcAlpha,
                    dstAlpha
                );
            }
            let out = call_atomic_ptr_5arg(
                "glBlendFuncSeparateiEXT",
                &self.glBlendFuncSeparateiEXT_p,
                buf,
                srcRGB,
                dstRGB,
                srcAlpha,
                dstAlpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendFuncSeparateiEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn BlendFuncSeparateiEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendFuncSeparateiEXT\0",
                &self.glBlendFuncSeparateiEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub fn BlendFuncSeparateiEXT_is_loaded(&self) -> bool {
            !self.glBlendFuncSeparateiEXT_p.load(RELAX).is_null()
        }
        /// [glBlendFuncSeparateiOES](http://docs.gl/es3/glBlendFuncSeparateiOES)(buf, srcRGB, dstRGB, srcAlpha, dstAlpha)
        /// * `srcRGB` group: BlendingFactor
        /// * `dstRGB` group: BlendingFactor
        /// * `srcAlpha` group: BlendingFactor
        /// * `dstAlpha` group: BlendingFactor
        /// * alias of: [`glBlendFuncSeparatei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub unsafe fn BlendFuncSeparateiOES(
            &self,
            buf: GLuint,
            srcRGB: GLenum,
            dstRGB: GLenum,
            srcAlpha: GLenum,
            dstAlpha: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendFuncSeparateiOES({:?}, {:#X}, {:#X}, {:#X}, {:#X});",
                    buf,
                    srcRGB,
                    dstRGB,
                    srcAlpha,
                    dstAlpha
                );
            }
            let out = call_atomic_ptr_5arg(
                "glBlendFuncSeparateiOES",
                &self.glBlendFuncSeparateiOES_p,
                buf,
                srcRGB,
                dstRGB,
                srcAlpha,
                dstAlpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendFuncSeparateiOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn BlendFuncSeparateiOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendFuncSeparateiOES\0",
                &self.glBlendFuncSeparateiOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub fn BlendFuncSeparateiOES_is_loaded(&self) -> bool {
            !self.glBlendFuncSeparateiOES_p.load(RELAX).is_null()
        }
        /// [glBlendFunci](http://docs.gl/es3/glBlendFunc)(buf, src, dst)
        /// * `src` group: BlendingFactor
        /// * `dst` group: BlendingFactor
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlendFunci(&self, buf: GLuint, src: GLenum, dst: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendFunci({:?}, {:#X}, {:#X});", buf, src, dst);
            }
            let out = call_atomic_ptr_3arg("glBlendFunci", &self.glBlendFunci_p, buf, src, dst);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendFunci");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlendFunci_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glBlendFunci\0", &self.glBlendFunci_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlendFunci_is_loaded(&self) -> bool {
            !self.glBlendFunci_p.load(RELAX).is_null()
        }
        /// [glBlendFunciEXT](http://docs.gl/es3/glBlendFunciEXT)(buf, src, dst)
        /// * `src` group: BlendingFactor
        /// * `dst` group: BlendingFactor
        /// * alias of: [`glBlendFunci`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub unsafe fn BlendFunciEXT(&self, buf: GLuint, src: GLenum, dst: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendFunciEXT({:?}, {:#X}, {:#X});",
                    buf,
                    src,
                    dst
                );
            }
            let out =
                call_atomic_ptr_3arg("glBlendFunciEXT", &self.glBlendFunciEXT_p, buf, src, dst);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendFunciEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn BlendFunciEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendFunciEXT\0",
                &self.glBlendFunciEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub fn BlendFunciEXT_is_loaded(&self) -> bool {
            !self.glBlendFunciEXT_p.load(RELAX).is_null()
        }
        /// [glBlendFunciOES](http://docs.gl/es3/glBlendFunciOES)(buf, src, dst)
        /// * `src` group: BlendingFactor
        /// * `dst` group: BlendingFactor
        /// * alias of: [`glBlendFunci`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub unsafe fn BlendFunciOES(&self, buf: GLuint, src: GLenum, dst: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlendFunciOES({:?}, {:#X}, {:#X});",
                    buf,
                    src,
                    dst
                );
            }
            let out =
                call_atomic_ptr_3arg("glBlendFunciOES", &self.glBlendFunciOES_p, buf, src, dst);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendFunciOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn BlendFunciOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendFunciOES\0",
                &self.glBlendFunciOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub fn BlendFunciOES_is_loaded(&self) -> bool {
            !self.glBlendFunciOES_p.load(RELAX).is_null()
        }
        /// [glBlendParameteriNV](http://docs.gl/es3/glBlendParameteriNV)(pname, value)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
        pub unsafe fn BlendParameteriNV(&self, pname: GLenum, value: GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.BlendParameteriNV({:#X}, {:?});", pname, value);
            }
            let out = call_atomic_ptr_2arg(
                "glBlendParameteriNV",
                &self.glBlendParameteriNV_p,
                pname,
                value,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlendParameteriNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
        #[doc(hidden)]
        pub unsafe fn BlendParameteriNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlendParameteriNV\0",
                &self.glBlendParameteriNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
        pub fn BlendParameteriNV_is_loaded(&self) -> bool {
            !self.glBlendParameteriNV_p.load(RELAX).is_null()
        }
        /// [glBlitFramebuffer](http://docs.gl/es3/glBlitFramebuffer)(srcX0, srcY0, srcX1, srcY1, dstX0, dstY0, dstX1, dstY1, mask, filter)
        /// * `mask` group: ClearBufferMask
        /// * `filter` group: BlitFramebufferFilter
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BlitFramebuffer(
            &self,
            srcX0: GLint,
            srcY0: GLint,
            srcX1: GLint,
            srcY1: GLint,
            dstX0: GLint,
            dstY0: GLint,
            dstX1: GLint,
            dstY1: GLint,
            mask: GLbitfield,
            filter: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlitFramebuffer({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X});",
                    srcX0,
                    srcY0,
                    srcX1,
                    srcY1,
                    dstX0,
                    dstY0,
                    dstX1,
                    dstY1,
                    mask,
                    filter
                );
            }
            let out = call_atomic_ptr_10arg(
                "glBlitFramebuffer",
                &self.glBlitFramebuffer_p,
                srcX0,
                srcY0,
                srcX1,
                srcY1,
                dstX0,
                dstY0,
                dstX1,
                dstY1,
                mask,
                filter,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlitFramebuffer");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BlitFramebuffer_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlitFramebuffer\0",
                &self.glBlitFramebuffer_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BlitFramebuffer_is_loaded(&self) -> bool {
            !self.glBlitFramebuffer_p.load(RELAX).is_null()
        }
        /// [glBlitFramebufferANGLE](http://docs.gl/es3/glBlitFramebufferANGLE)(srcX0, srcY0, srcX1, srcY1, dstX0, dstY0, dstX1, dstY1, mask, filter)
        /// * `mask` group: ClearBufferMask
        /// * `filter` group: BlitFramebufferFilter
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_ANGLE_framebuffer_blit"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_blit"))))]
        pub unsafe fn BlitFramebufferANGLE(
            &self,
            srcX0: GLint,
            srcY0: GLint,
            srcX1: GLint,
            srcY1: GLint,
            dstX0: GLint,
            dstY0: GLint,
            dstX1: GLint,
            dstY1: GLint,
            mask: GLbitfield,
            filter: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlitFramebufferANGLE({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X});",
                    srcX0,
                    srcY0,
                    srcX1,
                    srcY1,
                    dstX0,
                    dstY0,
                    dstX1,
                    dstY1,
                    mask,
                    filter
                );
            }
            let out = call_atomic_ptr_10arg(
                "glBlitFramebufferANGLE",
                &self.glBlitFramebufferANGLE_p,
                srcX0,
                srcY0,
                srcX1,
                srcY1,
                dstX0,
                dstY0,
                dstX1,
                dstY1,
                mask,
                filter,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlitFramebufferANGLE");
            }
            out
        }
        #[cfg(any(feature = "GL_ANGLE_framebuffer_blit"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_blit"))))]
        #[doc(hidden)]
        pub unsafe fn BlitFramebufferANGLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlitFramebufferANGLE\0",
                &self.glBlitFramebufferANGLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_ANGLE_framebuffer_blit"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_blit"))))]
        pub fn BlitFramebufferANGLE_is_loaded(&self) -> bool {
            !self.glBlitFramebufferANGLE_p.load(RELAX).is_null()
        }
        /// [glBlitFramebufferNV](http://docs.gl/es3/glBlitFramebufferNV)(srcX0, srcY0, srcX1, srcY1, dstX0, dstY0, dstX1, dstY1, mask, filter)
        /// * `mask` group: ClearBufferMask
        /// * `filter` group: BlitFramebufferFilter
        /// * alias of: [`glBlitFramebuffer`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_framebuffer_blit"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_blit"))))]
        pub unsafe fn BlitFramebufferNV(
            &self,
            srcX0: GLint,
            srcY0: GLint,
            srcX1: GLint,
            srcY1: GLint,
            dstX0: GLint,
            dstY0: GLint,
            dstX1: GLint,
            dstY1: GLint,
            mask: GLbitfield,
            filter: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BlitFramebufferNV({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X});",
                    srcX0,
                    srcY0,
                    srcX1,
                    srcY1,
                    dstX0,
                    dstY0,
                    dstX1,
                    dstY1,
                    mask,
                    filter
                );
            }
            let out = call_atomic_ptr_10arg(
                "glBlitFramebufferNV",
                &self.glBlitFramebufferNV_p,
                srcX0,
                srcY0,
                srcX1,
                srcY1,
                dstX0,
                dstY0,
                dstX1,
                dstY1,
                mask,
                filter,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBlitFramebufferNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_framebuffer_blit"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_blit"))))]
        #[doc(hidden)]
        pub unsafe fn BlitFramebufferNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBlitFramebufferNV\0",
                &self.glBlitFramebufferNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_framebuffer_blit"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_blit"))))]
        pub fn BlitFramebufferNV_is_loaded(&self) -> bool {
            !self.glBlitFramebufferNV_p.load(RELAX).is_null()
        }
        /// [glBufferAttachMemoryNV](http://docs.gl/es3/glBufferAttachMemoryNV)(target, memory, offset)
        /// * `target` group: BufferTargetARB
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        pub unsafe fn BufferAttachMemoryNV(
            &self,
            target: GLenum,
            memory: GLuint,
            offset: GLuint64,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BufferAttachMemoryNV({:#X}, {:?}, {:?});",
                    target,
                    memory,
                    offset
                );
            }
            let out = call_atomic_ptr_3arg(
                "glBufferAttachMemoryNV",
                &self.glBufferAttachMemoryNV_p,
                target,
                memory,
                offset,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBufferAttachMemoryNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        #[doc(hidden)]
        pub unsafe fn BufferAttachMemoryNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBufferAttachMemoryNV\0",
                &self.glBufferAttachMemoryNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        pub fn BufferAttachMemoryNV_is_loaded(&self) -> bool {
            !self.glBufferAttachMemoryNV_p.load(RELAX).is_null()
        }
        /// [glBufferData](http://docs.gl/es3/glBufferData)(target, size, data, usage)
        /// * `target` group: BufferTargetARB
        /// * `size` group: BufferSize
        /// * `data` len: size
        /// * `usage` group: BufferUsageARB
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BufferData(
            &self,
            target: GLenum,
            size: GLsizeiptr,
            data: *const c_void,
            usage: GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BufferData({:#X}, {:?}, {:p}, {:#X});",
                    target,
                    size,
                    data,
                    usage
                );
            }
            let out = call_atomic_ptr_4arg(
                "glBufferData",
                &self.glBufferData_p,
                target,
                size,
                data,
                usage,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBufferData");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BufferData_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glBufferData\0", &self.glBufferData_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn BufferData_is_loaded(&self) -> bool {
            !self.glBufferData_p.load(RELAX).is_null()
        }
        /// [glBufferPageCommitmentMemNV](http://docs.gl/es3/glBufferPageCommitmentMemNV)(target, offset, size, memory, memOffset, commit)
        /// * `target` group: BufferStorageTarget
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
        pub unsafe fn BufferPageCommitmentMemNV(
            &self,
            target: GLenum,
            offset: GLintptr,
            size: GLsizeiptr,
            memory: GLuint,
            memOffset: GLuint64,
            commit: GLboolean,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BufferPageCommitmentMemNV({:#X}, {:?}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    offset,
                    size,
                    memory,
                    memOffset,
                    commit
                );
            }
            let out = call_atomic_ptr_6arg(
                "glBufferPageCommitmentMemNV",
                &self.glBufferPageCommitmentMemNV_p,
                target,
                offset,
                size,
                memory,
                memOffset,
                commit,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBufferPageCommitmentMemNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
        #[doc(hidden)]
        pub unsafe fn BufferPageCommitmentMemNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBufferPageCommitmentMemNV\0",
                &self.glBufferPageCommitmentMemNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
        pub fn BufferPageCommitmentMemNV_is_loaded(&self) -> bool {
            !self.glBufferPageCommitmentMemNV_p.load(RELAX).is_null()
        }
        /// [glBufferStorageEXT](http://docs.gl/es3/glBufferStorageEXT)(target, size, data, flags)
        /// * `target` group: BufferStorageTarget
        /// * `data` len: size
        /// * `flags` group: BufferStorageMask
        /// * alias of: [`glBufferStorage`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_buffer_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_buffer_storage"))))]
        pub unsafe fn BufferStorageEXT(
            &self,
            target: GLenum,
            size: GLsizeiptr,
            data: *const c_void,
            flags: GLbitfield,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BufferStorageEXT({:#X}, {:?}, {:p}, {:?});",
                    target,
                    size,
                    data,
                    flags
                );
            }
            let out = call_atomic_ptr_4arg(
                "glBufferStorageEXT",
                &self.glBufferStorageEXT_p,
                target,
                size,
                data,
                flags,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBufferStorageEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_buffer_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_buffer_storage"))))]
        #[doc(hidden)]
        pub unsafe fn BufferStorageEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBufferStorageEXT\0",
                &self.glBufferStorageEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_buffer_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_buffer_storage"))))]
        pub fn BufferStorageEXT_is_loaded(&self) -> bool {
            !self.glBufferStorageEXT_p.load(RELAX).is_null()
        }
        /// [glBufferStorageExternalEXT](http://docs.gl/es3/glBufferStorageExternalEXT)(target, offset, size, clientBuffer, flags)
        /// * `flags` group: BufferStorageMask
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_external_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
        pub unsafe fn BufferStorageExternalEXT(
            &self,
            target: GLenum,
            offset: GLintptr,
            size: GLsizeiptr,
            clientBuffer: GLeglClientBufferEXT,
            flags: GLbitfield,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BufferStorageExternalEXT({:#X}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    offset,
                    size,
                    clientBuffer,
                    flags
                );
            }
            let out = call_atomic_ptr_5arg(
                "glBufferStorageExternalEXT",
                &self.glBufferStorageExternalEXT_p,
                target,
                offset,
                size,
                clientBuffer,
                flags,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBufferStorageExternalEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_external_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
        #[doc(hidden)]
        pub unsafe fn BufferStorageExternalEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBufferStorageExternalEXT\0",
                &self.glBufferStorageExternalEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_external_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
        pub fn BufferStorageExternalEXT_is_loaded(&self) -> bool {
            !self.glBufferStorageExternalEXT_p.load(RELAX).is_null()
        }
        /// [glBufferStorageMemEXT](http://docs.gl/es3/glBufferStorageMemEXT)(target, size, memory, offset)
        /// * `target` group: BufferTargetARB
        /// * `size` group: BufferSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        pub unsafe fn BufferStorageMemEXT(
            &self,
            target: GLenum,
            size: GLsizeiptr,
            memory: GLuint,
            offset: GLuint64,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BufferStorageMemEXT({:#X}, {:?}, {:?}, {:?});",
                    target,
                    size,
                    memory,
                    offset
                );
            }
            let out = call_atomic_ptr_4arg(
                "glBufferStorageMemEXT",
                &self.glBufferStorageMemEXT_p,
                target,
                size,
                memory,
                offset,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBufferStorageMemEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        #[doc(hidden)]
        pub unsafe fn BufferStorageMemEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBufferStorageMemEXT\0",
                &self.glBufferStorageMemEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        pub fn BufferStorageMemEXT_is_loaded(&self) -> bool {
            !self.glBufferStorageMemEXT_p.load(RELAX).is_null()
        }
        /// [glBufferSubData](http://docs.gl/es3/glBufferSubData)(target, offset, size, data)
        /// * `target` group: BufferTargetARB
        /// * `offset` group: BufferOffset
        /// * `size` group: BufferSize
        /// * `data` len: size
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn BufferSubData(
            &self,
            target: GLenum,
            offset: GLintptr,
            size: GLsizeiptr,
            data: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.BufferSubData({:#X}, {:?}, {:?}, {:p});",
                    target,
                    offset,
                    size,
                    data
                );
            }
            let out = call_atomic_ptr_4arg(
                "glBufferSubData",
                &self.glBufferSubData_p,
                target,
                offset,
                size,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glBufferSubData");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn BufferSubData_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glBufferSubData\0",
                &self.glBufferSubData_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn BufferSubData_is_loaded(&self) -> bool {
            !self.glBufferSubData_p.load(RELAX).is_null()
        }
        /// [glCheckFramebufferStatus](http://docs.gl/es3/glCheckFramebufferStatus)(target)
        /// * `target` group: FramebufferTarget
        /// * return value group: FramebufferStatus
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CheckFramebufferStatus(&self, target: GLenum) -> GLenum {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CheckFramebufferStatus({:#X});", target);
            }
            let out = call_atomic_ptr_1arg(
                "glCheckFramebufferStatus",
                &self.glCheckFramebufferStatus_p,
                target,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCheckFramebufferStatus");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CheckFramebufferStatus_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCheckFramebufferStatus\0",
                &self.glCheckFramebufferStatus_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CheckFramebufferStatus_is_loaded(&self) -> bool {
            !self.glCheckFramebufferStatus_p.load(RELAX).is_null()
        }
        /// [glClear](http://docs.gl/es3/glClear)(mask)
        /// * `mask` group: ClearBufferMask
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn Clear(&self, mask: GLbitfield) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.Clear({:?});", mask);
            }
            let out = call_atomic_ptr_1arg("glClear", &self.glClear_p, mask);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClear");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn Clear_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glClear\0", &self.glClear_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn Clear_is_loaded(&self) -> bool {
            !self.glClear_p.load(RELAX).is_null()
        }
        /// [glClearBufferfi](http://docs.gl/es3/glClearBuffer)(buffer, drawbuffer, depth, stencil)
        /// * `buffer` group: Buffer
        /// * `drawbuffer` group: DrawBufferName
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ClearBufferfi(
            &self,
            buffer: GLenum,
            drawbuffer: GLint,
            depth: GLfloat,
            stencil: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ClearBufferfi({:#X}, {:?}, {:?}, {:?});",
                    buffer,
                    drawbuffer,
                    depth,
                    stencil
                );
            }
            let out = call_atomic_ptr_4arg(
                "glClearBufferfi",
                &self.glClearBufferfi_p,
                buffer,
                drawbuffer,
                depth,
                stencil,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearBufferfi");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ClearBufferfi_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClearBufferfi\0",
                &self.glClearBufferfi_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn ClearBufferfi_is_loaded(&self) -> bool {
            !self.glClearBufferfi_p.load(RELAX).is_null()
        }
        /// [glClearBufferfv](http://docs.gl/es3/glClearBuffer)(buffer, drawbuffer, value)
        /// * `buffer` group: Buffer
        /// * `drawbuffer` group: DrawBufferName
        /// * `value` len: COMPSIZE(buffer)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ClearBufferfv(
            &self,
            buffer: GLenum,
            drawbuffer: GLint,
            value: *const GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ClearBufferfv({:#X}, {:?}, {:p});",
                    buffer,
                    drawbuffer,
                    value
                );
            }
            let out = call_atomic_ptr_3arg(
                "glClearBufferfv",
                &self.glClearBufferfv_p,
                buffer,
                drawbuffer,
                value,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearBufferfv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ClearBufferfv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClearBufferfv\0",
                &self.glClearBufferfv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn ClearBufferfv_is_loaded(&self) -> bool {
            !self.glClearBufferfv_p.load(RELAX).is_null()
        }
        /// [glClearBufferiv](http://docs.gl/es3/glClearBuffer)(buffer, drawbuffer, value)
        /// * `buffer` group: Buffer
        /// * `drawbuffer` group: DrawBufferName
        /// * `value` len: COMPSIZE(buffer)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ClearBufferiv(&self, buffer: GLenum, drawbuffer: GLint, value: *const GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ClearBufferiv({:#X}, {:?}, {:p});",
                    buffer,
                    drawbuffer,
                    value
                );
            }
            let out = call_atomic_ptr_3arg(
                "glClearBufferiv",
                &self.glClearBufferiv_p,
                buffer,
                drawbuffer,
                value,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearBufferiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ClearBufferiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClearBufferiv\0",
                &self.glClearBufferiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn ClearBufferiv_is_loaded(&self) -> bool {
            !self.glClearBufferiv_p.load(RELAX).is_null()
        }
        /// [glClearBufferuiv](http://docs.gl/es3/glClearBuffer)(buffer, drawbuffer, value)
        /// * `buffer` group: Buffer
        /// * `drawbuffer` group: DrawBufferName
        /// * `value` len: COMPSIZE(buffer)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ClearBufferuiv(
            &self,
            buffer: GLenum,
            drawbuffer: GLint,
            value: *const GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ClearBufferuiv({:#X}, {:?}, {:p});",
                    buffer,
                    drawbuffer,
                    value
                );
            }
            let out = call_atomic_ptr_3arg(
                "glClearBufferuiv",
                &self.glClearBufferuiv_p,
                buffer,
                drawbuffer,
                value,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearBufferuiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ClearBufferuiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClearBufferuiv\0",
                &self.glClearBufferuiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn ClearBufferuiv_is_loaded(&self) -> bool {
            !self.glClearBufferuiv_p.load(RELAX).is_null()
        }
        /// [glClearColor](http://docs.gl/es3/glClearColor)(red, green, blue, alpha)
        /// * `red` group: ColorF
        /// * `green` group: ColorF
        /// * `blue` group: ColorF
        /// * `alpha` group: ColorF
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ClearColor(
            &self,
            red: GLfloat,
            green: GLfloat,
            blue: GLfloat,
            alpha: GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ClearColor({:?}, {:?}, {:?}, {:?});",
                    red,
                    green,
                    blue,
                    alpha
                );
            }
            let out = call_atomic_ptr_4arg(
                "glClearColor",
                &self.glClearColor_p,
                red,
                green,
                blue,
                alpha,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearColor");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ClearColor_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glClearColor\0", &self.glClearColor_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn ClearColor_is_loaded(&self) -> bool {
            !self.glClearColor_p.load(RELAX).is_null()
        }
        /// [glClearDepthf](http://docs.gl/es3/glClearDepth)(d)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ClearDepthf(&self, d: GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.ClearDepthf({:?});", d);
            }
            let out = call_atomic_ptr_1arg("glClearDepthf", &self.glClearDepthf_p, d);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearDepthf");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ClearDepthf_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glClearDepthf\0", &self.glClearDepthf_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn ClearDepthf_is_loaded(&self) -> bool {
            !self.glClearDepthf_p.load(RELAX).is_null()
        }
        /// [glClearPixelLocalStorageuiEXT](http://docs.gl/es3/glClearPixelLocalStorageuiEXT)(offset, n, values)
        /// * `values` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        pub unsafe fn ClearPixelLocalStorageuiEXT(
            &self,
            offset: GLsizei,
            n: GLsizei,
            values: *const GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ClearPixelLocalStorageuiEXT({:?}, {:?}, {:p});",
                    offset,
                    n,
                    values
                );
            }
            let out = call_atomic_ptr_3arg(
                "glClearPixelLocalStorageuiEXT",
                &self.glClearPixelLocalStorageuiEXT_p,
                offset,
                n,
                values,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearPixelLocalStorageuiEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        #[doc(hidden)]
        pub unsafe fn ClearPixelLocalStorageuiEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClearPixelLocalStorageuiEXT\0",
                &self.glClearPixelLocalStorageuiEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        pub fn ClearPixelLocalStorageuiEXT_is_loaded(&self) -> bool {
            !self.glClearPixelLocalStorageuiEXT_p.load(RELAX).is_null()
        }
        /// [glClearStencil](http://docs.gl/es3/glClearStencil)(s)
        /// * `s` group: StencilValue
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ClearStencil(&self, s: GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.ClearStencil({:?});", s);
            }
            let out = call_atomic_ptr_1arg("glClearStencil", &self.glClearStencil_p, s);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearStencil");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ClearStencil_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClearStencil\0",
                &self.glClearStencil_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn ClearStencil_is_loaded(&self) -> bool {
            !self.glClearStencil_p.load(RELAX).is_null()
        }
        /// [glClearTexImageEXT](http://docs.gl/es3/glClearTexImageEXT)(texture, level, format, type_, data)
        /// * `texture` class: texture
        /// * `format` group: PixelFormat
        /// * `type_` group: PixelType
        /// * `data` len: COMPSIZE(format,type)
        /// * alias of: [`glClearTexImage`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_clear_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
        pub unsafe fn ClearTexImageEXT(
            &self,
            texture: GLuint,
            level: GLint,
            format: GLenum,
            type_: GLenum,
            data: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ClearTexImageEXT({:?}, {:?}, {:#X}, {:#X}, {:p});",
                    texture,
                    level,
                    format,
                    type_,
                    data
                );
            }
            let out = call_atomic_ptr_5arg(
                "glClearTexImageEXT",
                &self.glClearTexImageEXT_p,
                texture,
                level,
                format,
                type_,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearTexImageEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_clear_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
        #[doc(hidden)]
        pub unsafe fn ClearTexImageEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClearTexImageEXT\0",
                &self.glClearTexImageEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_clear_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
        pub fn ClearTexImageEXT_is_loaded(&self) -> bool {
            !self.glClearTexImageEXT_p.load(RELAX).is_null()
        }
        /// [glClearTexSubImageEXT](http://docs.gl/es3/glClearTexSubImageEXT)(texture, level, xoffset, yoffset, zoffset, width, height, depth, format, type_, data)
        /// * `texture` class: texture
        /// * `format` group: PixelFormat
        /// * `type_` group: PixelType
        /// * `data` len: COMPSIZE(format,type)
        /// * alias of: [`glClearTexSubImage`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_clear_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
        pub unsafe fn ClearTexSubImageEXT(
            &self,
            texture: GLuint,
            level: GLint,
            xoffset: GLint,
            yoffset: GLint,
            zoffset: GLint,
            width: GLsizei,
            height: GLsizei,
            depth: GLsizei,
            format: GLenum,
            type_: GLenum,
            data: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.ClearTexSubImageEXT({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:#X}, {:p});", texture, level, xoffset, yoffset, zoffset, width, height, depth, format, type_, data);
            }
            let out = call_atomic_ptr_11arg(
                "glClearTexSubImageEXT",
                &self.glClearTexSubImageEXT_p,
                texture,
                level,
                xoffset,
                yoffset,
                zoffset,
                width,
                height,
                depth,
                format,
                type_,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClearTexSubImageEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_clear_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
        #[doc(hidden)]
        pub unsafe fn ClearTexSubImageEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClearTexSubImageEXT\0",
                &self.glClearTexSubImageEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_clear_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
        pub fn ClearTexSubImageEXT_is_loaded(&self) -> bool {
            !self.glClearTexSubImageEXT_p.load(RELAX).is_null()
        }
        /// [glClientWaitSync](http://docs.gl/es3/glClientWaitSync)(sync, flags, timeout)
        /// * `sync` group: sync
        /// * `sync` class: sync
        /// * `flags` group: SyncObjectMask
        /// * return value group: SyncStatus
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ClientWaitSync(
            &self,
            sync: GLsync,
            flags: GLbitfield,
            timeout: GLuint64,
        ) -> GLenum {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ClientWaitSync({:p}, {:?}, {:?});",
                    sync,
                    flags,
                    timeout
                );
            }
            let out = call_atomic_ptr_3arg(
                "glClientWaitSync",
                &self.glClientWaitSync_p,
                sync,
                flags,
                timeout,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClientWaitSync");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ClientWaitSync_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClientWaitSync\0",
                &self.glClientWaitSync_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn ClientWaitSync_is_loaded(&self) -> bool {
            !self.glClientWaitSync_p.load(RELAX).is_null()
        }
        /// [glClientWaitSyncAPPLE](http://docs.gl/es3/glClientWaitSyncAPPLE)(sync, flags, timeout)
        /// * `sync` group: sync
        /// * `sync` class: sync
        /// * `flags` group: SyncObjectMask
        /// * return value group: SyncStatus
        /// * alias of: [`glClientWaitSync`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub unsafe fn ClientWaitSyncAPPLE(
            &self,
            sync: GLsync,
            flags: GLbitfield,
            timeout: GLuint64,
        ) -> GLenum {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ClientWaitSyncAPPLE({:p}, {:?}, {:?});",
                    sync,
                    flags,
                    timeout
                );
            }
            let out = call_atomic_ptr_3arg(
                "glClientWaitSyncAPPLE",
                &self.glClientWaitSyncAPPLE_p,
                sync,
                flags,
                timeout,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClientWaitSyncAPPLE");
            }
            out
        }
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        #[doc(hidden)]
        pub unsafe fn ClientWaitSyncAPPLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClientWaitSyncAPPLE\0",
                &self.glClientWaitSyncAPPLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub fn ClientWaitSyncAPPLE_is_loaded(&self) -> bool {
            !self.glClientWaitSyncAPPLE_p.load(RELAX).is_null()
        }
        /// [glClipControlEXT](http://docs.gl/es3/glClipControlEXT)(origin, depth)
        /// * alias of: [`glClipControl`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_clip_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clip_control"))))]
        pub unsafe fn ClipControlEXT(&self, origin: GLenum, depth: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.ClipControlEXT({:#X}, {:#X});", origin, depth);
            }
            let out =
                call_atomic_ptr_2arg("glClipControlEXT", &self.glClipControlEXT_p, origin, depth);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glClipControlEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_clip_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clip_control"))))]
        #[doc(hidden)]
        pub unsafe fn ClipControlEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glClipControlEXT\0",
                &self.glClipControlEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_clip_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clip_control"))))]
        pub fn ClipControlEXT_is_loaded(&self) -> bool {
            !self.glClipControlEXT_p.load(RELAX).is_null()
        }
        /// [glColorMask](http://docs.gl/es3/glColorMask)(red, green, blue, alpha)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ColorMask(
            &self,
            red: GLboolean,
            green: GLboolean,
            blue: GLboolean,
            alpha: GLboolean,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ColorMask({:?}, {:?}, {:?}, {:?});",
                    red,
                    green,
                    blue,
                    alpha
                );
            }
            let out =
                call_atomic_ptr_4arg("glColorMask", &self.glColorMask_p, red, green, blue, alpha);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glColorMask");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ColorMask_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glColorMask\0", &self.glColorMask_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn ColorMask_is_loaded(&self) -> bool {
            !self.glColorMask_p.load(RELAX).is_null()
        }
        /// [glColorMaski](http://docs.gl/es3/glColorMask)(index, r, g, b, a)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn ColorMaski(
            &self,
            index: GLuint,
            r: GLboolean,
            g: GLboolean,
            b: GLboolean,
            a: GLboolean,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ColorMaski({:?}, {:?}, {:?}, {:?}, {:?});",
                    index,
                    r,
                    g,
                    b,
                    a
                );
            }
            let out = call_atomic_ptr_5arg("glColorMaski", &self.glColorMaski_p, index, r, g, b, a);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glColorMaski");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn ColorMaski_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glColorMaski\0", &self.glColorMaski_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn ColorMaski_is_loaded(&self) -> bool {
            !self.glColorMaski_p.load(RELAX).is_null()
        }
        /// [glColorMaskiEXT](http://docs.gl/es3/glColorMaskiEXT)(index, r, g, b, a)
        /// * alias of: [`glColorMaski`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub unsafe fn ColorMaskiEXT(
            &self,
            index: GLuint,
            r: GLboolean,
            g: GLboolean,
            b: GLboolean,
            a: GLboolean,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ColorMaskiEXT({:?}, {:?}, {:?}, {:?}, {:?});",
                    index,
                    r,
                    g,
                    b,
                    a
                );
            }
            let out = call_atomic_ptr_5arg(
                "glColorMaskiEXT",
                &self.glColorMaskiEXT_p,
                index,
                r,
                g,
                b,
                a,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glColorMaskiEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn ColorMaskiEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glColorMaskiEXT\0",
                &self.glColorMaskiEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub fn ColorMaskiEXT_is_loaded(&self) -> bool {
            !self.glColorMaskiEXT_p.load(RELAX).is_null()
        }
        /// [glColorMaskiOES](http://docs.gl/es3/glColorMaskiOES)(index, r, g, b, a)
        /// * alias of: [`glColorMaski`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub unsafe fn ColorMaskiOES(
            &self,
            index: GLuint,
            r: GLboolean,
            g: GLboolean,
            b: GLboolean,
            a: GLboolean,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ColorMaskiOES({:?}, {:?}, {:?}, {:?}, {:?});",
                    index,
                    r,
                    g,
                    b,
                    a
                );
            }
            let out = call_atomic_ptr_5arg(
                "glColorMaskiOES",
                &self.glColorMaskiOES_p,
                index,
                r,
                g,
                b,
                a,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glColorMaskiOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn ColorMaskiOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glColorMaskiOES\0",
                &self.glColorMaskiOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        pub fn ColorMaskiOES_is_loaded(&self) -> bool {
            !self.glColorMaskiOES_p.load(RELAX).is_null()
        }
        /// [glCompileShader](http://docs.gl/es3/glCompileShader)(shader)
        /// * `shader` class: shader
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CompileShader(&self, shader: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CompileShader({:?});", shader);
            }
            let out = call_atomic_ptr_1arg("glCompileShader", &self.glCompileShader_p, shader);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCompileShader");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CompileShader_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCompileShader\0",
                &self.glCompileShader_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CompileShader_is_loaded(&self) -> bool {
            !self.glCompileShader_p.load(RELAX).is_null()
        }
        /// [glCompressedTexImage2D](http://docs.gl/es3/glCompressedTexImage2D)(target, level, internalformat, width, height, border, imageSize, data)
        /// * `target` group: TextureTarget
        /// * `level` group: CheckedInt32
        /// * `internalformat` group: InternalFormat
        /// * `border` group: CheckedInt32
        /// * `data` group: CompressedTextureARB
        /// * `data` len: imageSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CompressedTexImage2D(
            &self,
            target: GLenum,
            level: GLint,
            internalformat: GLenum,
            width: GLsizei,
            height: GLsizei,
            border: GLint,
            imageSize: GLsizei,
            data: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CompressedTexImage2D({:#X}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:p});",
                    target,
                    level,
                    internalformat,
                    width,
                    height,
                    border,
                    imageSize,
                    data
                );
            }
            let out = call_atomic_ptr_8arg(
                "glCompressedTexImage2D",
                &self.glCompressedTexImage2D_p,
                target,
                level,
                internalformat,
                width,
                height,
                border,
                imageSize,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCompressedTexImage2D");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CompressedTexImage2D_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCompressedTexImage2D\0",
                &self.glCompressedTexImage2D_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CompressedTexImage2D_is_loaded(&self) -> bool {
            !self.glCompressedTexImage2D_p.load(RELAX).is_null()
        }
        /// [glCompressedTexImage3D](http://docs.gl/es3/glCompressedTexImage3D)(target, level, internalformat, width, height, depth, border, imageSize, data)
        /// * `target` group: TextureTarget
        /// * `level` group: CheckedInt32
        /// * `internalformat` group: InternalFormat
        /// * `border` group: CheckedInt32
        /// * `data` group: CompressedTextureARB
        /// * `data` len: imageSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CompressedTexImage3D(
            &self,
            target: GLenum,
            level: GLint,
            internalformat: GLenum,
            width: GLsizei,
            height: GLsizei,
            depth: GLsizei,
            border: GLint,
            imageSize: GLsizei,
            data: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CompressedTexImage3D({:#X}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:p});",
                    target,
                    level,
                    internalformat,
                    width,
                    height,
                    depth,
                    border,
                    imageSize,
                    data
                );
            }
            let out = call_atomic_ptr_9arg(
                "glCompressedTexImage3D",
                &self.glCompressedTexImage3D_p,
                target,
                level,
                internalformat,
                width,
                height,
                depth,
                border,
                imageSize,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCompressedTexImage3D");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CompressedTexImage3D_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCompressedTexImage3D\0",
                &self.glCompressedTexImage3D_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CompressedTexImage3D_is_loaded(&self) -> bool {
            !self.glCompressedTexImage3D_p.load(RELAX).is_null()
        }
        /// [glCompressedTexImage3DOES](http://docs.gl/es3/glCompressedTexImage3DOES)(target, level, internalformat, width, height, depth, border, imageSize, data)
        /// * `target` group: TextureTarget
        /// * `internalformat` group: InternalFormat
        /// * `data` len: imageSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        pub unsafe fn CompressedTexImage3DOES(
            &self,
            target: GLenum,
            level: GLint,
            internalformat: GLenum,
            width: GLsizei,
            height: GLsizei,
            depth: GLsizei,
            border: GLint,
            imageSize: GLsizei,
            data: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CompressedTexImage3DOES({:#X}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:p});",
                    target,
                    level,
                    internalformat,
                    width,
                    height,
                    depth,
                    border,
                    imageSize,
                    data
                );
            }
            let out = call_atomic_ptr_9arg(
                "glCompressedTexImage3DOES",
                &self.glCompressedTexImage3DOES_p,
                target,
                level,
                internalformat,
                width,
                height,
                depth,
                border,
                imageSize,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCompressedTexImage3DOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        #[doc(hidden)]
        pub unsafe fn CompressedTexImage3DOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCompressedTexImage3DOES\0",
                &self.glCompressedTexImage3DOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        pub fn CompressedTexImage3DOES_is_loaded(&self) -> bool {
            !self.glCompressedTexImage3DOES_p.load(RELAX).is_null()
        }
        /// [glCompressedTexSubImage2D](http://docs.gl/es3/glCompressedTexSubImage2D)(target, level, xoffset, yoffset, width, height, format, imageSize, data)
        /// * `target` group: TextureTarget
        /// * `level` group: CheckedInt32
        /// * `xoffset` group: CheckedInt32
        /// * `yoffset` group: CheckedInt32
        /// * `format` group: PixelFormat
        /// * `data` group: CompressedTextureARB
        /// * `data` len: imageSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CompressedTexSubImage2D(
            &self,
            target: GLenum,
            level: GLint,
            xoffset: GLint,
            yoffset: GLint,
            width: GLsizei,
            height: GLsizei,
            format: GLenum,
            imageSize: GLsizei,
            data: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CompressedTexSubImage2D({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:p});",
                    target,
                    level,
                    xoffset,
                    yoffset,
                    width,
                    height,
                    format,
                    imageSize,
                    data
                );
            }
            let out = call_atomic_ptr_9arg(
                "glCompressedTexSubImage2D",
                &self.glCompressedTexSubImage2D_p,
                target,
                level,
                xoffset,
                yoffset,
                width,
                height,
                format,
                imageSize,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCompressedTexSubImage2D");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CompressedTexSubImage2D_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCompressedTexSubImage2D\0",
                &self.glCompressedTexSubImage2D_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CompressedTexSubImage2D_is_loaded(&self) -> bool {
            !self.glCompressedTexSubImage2D_p.load(RELAX).is_null()
        }
        /// [glCompressedTexSubImage3D](http://docs.gl/es3/glCompressedTexSubImage3D)(target, level, xoffset, yoffset, zoffset, width, height, depth, format, imageSize, data)
        /// * `target` group: TextureTarget
        /// * `level` group: CheckedInt32
        /// * `xoffset` group: CheckedInt32
        /// * `yoffset` group: CheckedInt32
        /// * `zoffset` group: CheckedInt32
        /// * `format` group: PixelFormat
        /// * `data` group: CompressedTextureARB
        /// * `data` len: imageSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CompressedTexSubImage3D(
            &self,
            target: GLenum,
            level: GLint,
            xoffset: GLint,
            yoffset: GLint,
            zoffset: GLint,
            width: GLsizei,
            height: GLsizei,
            depth: GLsizei,
            format: GLenum,
            imageSize: GLsizei,
            data: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CompressedTexSubImage3D({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:p});", target, level, xoffset, yoffset, zoffset, width, height, depth, format, imageSize, data);
            }
            let out = call_atomic_ptr_11arg(
                "glCompressedTexSubImage3D",
                &self.glCompressedTexSubImage3D_p,
                target,
                level,
                xoffset,
                yoffset,
                zoffset,
                width,
                height,
                depth,
                format,
                imageSize,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCompressedTexSubImage3D");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CompressedTexSubImage3D_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCompressedTexSubImage3D\0",
                &self.glCompressedTexSubImage3D_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CompressedTexSubImage3D_is_loaded(&self) -> bool {
            !self.glCompressedTexSubImage3D_p.load(RELAX).is_null()
        }
        /// [glCompressedTexSubImage3DOES](http://docs.gl/es3/glCompressedTexSubImage3DOES)(target, level, xoffset, yoffset, zoffset, width, height, depth, format, imageSize, data)
        /// * `target` group: TextureTarget
        /// * `format` group: PixelFormat
        /// * `data` len: imageSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        pub unsafe fn CompressedTexSubImage3DOES(
            &self,
            target: GLenum,
            level: GLint,
            xoffset: GLint,
            yoffset: GLint,
            zoffset: GLint,
            width: GLsizei,
            height: GLsizei,
            depth: GLsizei,
            format: GLenum,
            imageSize: GLsizei,
            data: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CompressedTexSubImage3DOES({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:p});", target, level, xoffset, yoffset, zoffset, width, height, depth, format, imageSize, data);
            }
            let out = call_atomic_ptr_11arg(
                "glCompressedTexSubImage3DOES",
                &self.glCompressedTexSubImage3DOES_p,
                target,
                level,
                xoffset,
                yoffset,
                zoffset,
                width,
                height,
                depth,
                format,
                imageSize,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCompressedTexSubImage3DOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        #[doc(hidden)]
        pub unsafe fn CompressedTexSubImage3DOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCompressedTexSubImage3DOES\0",
                &self.glCompressedTexSubImage3DOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        pub fn CompressedTexSubImage3DOES_is_loaded(&self) -> bool {
            !self.glCompressedTexSubImage3DOES_p.load(RELAX).is_null()
        }
        /// [glConservativeRasterParameteriNV](http://docs.gl/es3/glConservativeRasterParameteriNV)(pname, param)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles")))
        )]
        pub unsafe fn ConservativeRasterParameteriNV(&self, pname: GLenum, param: GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ConservativeRasterParameteriNV({:#X}, {:?});",
                    pname,
                    param
                );
            }
            let out = call_atomic_ptr_2arg(
                "glConservativeRasterParameteriNV",
                &self.glConservativeRasterParameteriNV_p,
                pname,
                param,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glConservativeRasterParameteriNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles")))
        )]
        #[doc(hidden)]
        pub unsafe fn ConservativeRasterParameteriNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glConservativeRasterParameteriNV\0",
                &self.glConservativeRasterParameteriNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles")))
        )]
        pub fn ConservativeRasterParameteriNV_is_loaded(&self) -> bool {
            !self
                .glConservativeRasterParameteriNV_p
                .load(RELAX)
                .is_null()
        }
        /// [glCopyBufferSubData](http://docs.gl/es3/glCopyBufferSubData)(readTarget, writeTarget, readOffset, writeOffset, size)
        /// * `readTarget` group: CopyBufferSubDataTarget
        /// * `writeTarget` group: CopyBufferSubDataTarget
        /// * `readOffset` group: BufferOffset
        /// * `writeOffset` group: BufferOffset
        /// * `size` group: BufferSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CopyBufferSubData(
            &self,
            readTarget: GLenum,
            writeTarget: GLenum,
            readOffset: GLintptr,
            writeOffset: GLintptr,
            size: GLsizeiptr,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CopyBufferSubData({:#X}, {:#X}, {:?}, {:?}, {:?});",
                    readTarget,
                    writeTarget,
                    readOffset,
                    writeOffset,
                    size
                );
            }
            let out = call_atomic_ptr_5arg(
                "glCopyBufferSubData",
                &self.glCopyBufferSubData_p,
                readTarget,
                writeTarget,
                readOffset,
                writeOffset,
                size,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyBufferSubData");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CopyBufferSubData_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyBufferSubData\0",
                &self.glCopyBufferSubData_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CopyBufferSubData_is_loaded(&self) -> bool {
            !self.glCopyBufferSubData_p.load(RELAX).is_null()
        }
        /// [glCopyBufferSubDataNV](http://docs.gl/es3/glCopyBufferSubDataNV)(readTarget, writeTarget, readOffset, writeOffset, size)
        /// * `readTarget` group: CopyBufferSubDataTarget
        /// * `writeTarget` group: CopyBufferSubDataTarget
        /// * `readOffset` group: BufferOffset
        /// * `writeOffset` group: BufferOffset
        /// * `size` group: BufferSize
        /// * alias of: [`glCopyBufferSubData`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_copy_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_copy_buffer"))))]
        pub unsafe fn CopyBufferSubDataNV(
            &self,
            readTarget: GLenum,
            writeTarget: GLenum,
            readOffset: GLintptr,
            writeOffset: GLintptr,
            size: GLsizeiptr,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CopyBufferSubDataNV({:#X}, {:#X}, {:?}, {:?}, {:?});",
                    readTarget,
                    writeTarget,
                    readOffset,
                    writeOffset,
                    size
                );
            }
            let out = call_atomic_ptr_5arg(
                "glCopyBufferSubDataNV",
                &self.glCopyBufferSubDataNV_p,
                readTarget,
                writeTarget,
                readOffset,
                writeOffset,
                size,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyBufferSubDataNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_copy_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_copy_buffer"))))]
        #[doc(hidden)]
        pub unsafe fn CopyBufferSubDataNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyBufferSubDataNV\0",
                &self.glCopyBufferSubDataNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_copy_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_copy_buffer"))))]
        pub fn CopyBufferSubDataNV_is_loaded(&self) -> bool {
            !self.glCopyBufferSubDataNV_p.load(RELAX).is_null()
        }
        /// [glCopyImageSubData](http://docs.gl/es3/glCopyImageSubData)(srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth)
        /// * `srcTarget` group: CopyImageSubDataTarget
        /// * `dstTarget` group: CopyImageSubDataTarget
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CopyImageSubData(
            &self,
            srcName: GLuint,
            srcTarget: GLenum,
            srcLevel: GLint,
            srcX: GLint,
            srcY: GLint,
            srcZ: GLint,
            dstName: GLuint,
            dstTarget: GLenum,
            dstLevel: GLint,
            dstX: GLint,
            dstY: GLint,
            dstZ: GLint,
            srcWidth: GLsizei,
            srcHeight: GLsizei,
            srcDepth: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CopyImageSubData({:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});", srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth);
            }
            let out = call_atomic_ptr_15arg(
                "glCopyImageSubData",
                &self.glCopyImageSubData_p,
                srcName,
                srcTarget,
                srcLevel,
                srcX,
                srcY,
                srcZ,
                dstName,
                dstTarget,
                dstLevel,
                dstX,
                dstY,
                dstZ,
                srcWidth,
                srcHeight,
                srcDepth,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyImageSubData");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CopyImageSubData_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyImageSubData\0",
                &self.glCopyImageSubData_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CopyImageSubData_is_loaded(&self) -> bool {
            !self.glCopyImageSubData_p.load(RELAX).is_null()
        }
        /// [glCopyImageSubDataEXT](http://docs.gl/es3/glCopyImageSubDataEXT)(srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth)
        /// * `srcTarget` group: CopyBufferSubDataTarget
        /// * `dstTarget` group: CopyBufferSubDataTarget
        /// * alias of: [`glCopyImageSubData`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_copy_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_copy_image"))))]
        pub unsafe fn CopyImageSubDataEXT(
            &self,
            srcName: GLuint,
            srcTarget: GLenum,
            srcLevel: GLint,
            srcX: GLint,
            srcY: GLint,
            srcZ: GLint,
            dstName: GLuint,
            dstTarget: GLenum,
            dstLevel: GLint,
            dstX: GLint,
            dstY: GLint,
            dstZ: GLint,
            srcWidth: GLsizei,
            srcHeight: GLsizei,
            srcDepth: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CopyImageSubDataEXT({:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});", srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth);
            }
            let out = call_atomic_ptr_15arg(
                "glCopyImageSubDataEXT",
                &self.glCopyImageSubDataEXT_p,
                srcName,
                srcTarget,
                srcLevel,
                srcX,
                srcY,
                srcZ,
                dstName,
                dstTarget,
                dstLevel,
                dstX,
                dstY,
                dstZ,
                srcWidth,
                srcHeight,
                srcDepth,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyImageSubDataEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_copy_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_copy_image"))))]
        #[doc(hidden)]
        pub unsafe fn CopyImageSubDataEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyImageSubDataEXT\0",
                &self.glCopyImageSubDataEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_copy_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_copy_image"))))]
        pub fn CopyImageSubDataEXT_is_loaded(&self) -> bool {
            !self.glCopyImageSubDataEXT_p.load(RELAX).is_null()
        }
        /// [glCopyImageSubDataOES](http://docs.gl/es3/glCopyImageSubDataOES)(srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth)
        /// * `srcTarget` group: CopyBufferSubDataTarget
        /// * `dstTarget` group: CopyBufferSubDataTarget
        /// * alias of: [`glCopyImageSubData`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_copy_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_copy_image"))))]
        pub unsafe fn CopyImageSubDataOES(
            &self,
            srcName: GLuint,
            srcTarget: GLenum,
            srcLevel: GLint,
            srcX: GLint,
            srcY: GLint,
            srcZ: GLint,
            dstName: GLuint,
            dstTarget: GLenum,
            dstLevel: GLint,
            dstX: GLint,
            dstY: GLint,
            dstZ: GLint,
            srcWidth: GLsizei,
            srcHeight: GLsizei,
            srcDepth: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CopyImageSubDataOES({:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});", srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth);
            }
            let out = call_atomic_ptr_15arg(
                "glCopyImageSubDataOES",
                &self.glCopyImageSubDataOES_p,
                srcName,
                srcTarget,
                srcLevel,
                srcX,
                srcY,
                srcZ,
                dstName,
                dstTarget,
                dstLevel,
                dstX,
                dstY,
                dstZ,
                srcWidth,
                srcHeight,
                srcDepth,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyImageSubDataOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_copy_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_copy_image"))))]
        #[doc(hidden)]
        pub unsafe fn CopyImageSubDataOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyImageSubDataOES\0",
                &self.glCopyImageSubDataOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_copy_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_copy_image"))))]
        pub fn CopyImageSubDataOES_is_loaded(&self) -> bool {
            !self.glCopyImageSubDataOES_p.load(RELAX).is_null()
        }
        /// [glCopyPathNV](http://docs.gl/es3/glCopyPathNV)(resultPath, srcPath)
        /// * `resultPath` group: Path
        /// * `srcPath` group: Path
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn CopyPathNV(&self, resultPath: GLuint, srcPath: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CopyPathNV({:?}, {:?});", resultPath, srcPath);
            }
            let out =
                call_atomic_ptr_2arg("glCopyPathNV", &self.glCopyPathNV_p, resultPath, srcPath);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyPathNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn CopyPathNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glCopyPathNV\0", &self.glCopyPathNV_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn CopyPathNV_is_loaded(&self) -> bool {
            !self.glCopyPathNV_p.load(RELAX).is_null()
        }
        /// [glCopyTexImage2D](http://docs.gl/es3/glCopyTexImage2D)(target, level, internalformat, x, y, width, height, border)
        /// * `target` group: TextureTarget
        /// * `level` group: CheckedInt32
        /// * `internalformat` group: InternalFormat
        /// * `x` group: WinCoord
        /// * `y` group: WinCoord
        /// * `border` group: CheckedInt32
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CopyTexImage2D(
            &self,
            target: GLenum,
            level: GLint,
            internalformat: GLenum,
            x: GLint,
            y: GLint,
            width: GLsizei,
            height: GLsizei,
            border: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CopyTexImage2D({:#X}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    level,
                    internalformat,
                    x,
                    y,
                    width,
                    height,
                    border
                );
            }
            let out = call_atomic_ptr_8arg(
                "glCopyTexImage2D",
                &self.glCopyTexImage2D_p,
                target,
                level,
                internalformat,
                x,
                y,
                width,
                height,
                border,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyTexImage2D");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CopyTexImage2D_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyTexImage2D\0",
                &self.glCopyTexImage2D_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CopyTexImage2D_is_loaded(&self) -> bool {
            !self.glCopyTexImage2D_p.load(RELAX).is_null()
        }
        /// [glCopyTexSubImage2D](http://docs.gl/es3/glCopyTexSubImage2D)(target, level, xoffset, yoffset, x, y, width, height)
        /// * `target` group: TextureTarget
        /// * `level` group: CheckedInt32
        /// * `xoffset` group: CheckedInt32
        /// * `yoffset` group: CheckedInt32
        /// * `x` group: WinCoord
        /// * `y` group: WinCoord
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CopyTexSubImage2D(
            &self,
            target: GLenum,
            level: GLint,
            xoffset: GLint,
            yoffset: GLint,
            x: GLint,
            y: GLint,
            width: GLsizei,
            height: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CopyTexSubImage2D({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    level,
                    xoffset,
                    yoffset,
                    x,
                    y,
                    width,
                    height
                );
            }
            let out = call_atomic_ptr_8arg(
                "glCopyTexSubImage2D",
                &self.glCopyTexSubImage2D_p,
                target,
                level,
                xoffset,
                yoffset,
                x,
                y,
                width,
                height,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyTexSubImage2D");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CopyTexSubImage2D_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyTexSubImage2D\0",
                &self.glCopyTexSubImage2D_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CopyTexSubImage2D_is_loaded(&self) -> bool {
            !self.glCopyTexSubImage2D_p.load(RELAX).is_null()
        }
        /// [glCopyTexSubImage3D](http://docs.gl/es3/glCopyTexSubImage3D)(target, level, xoffset, yoffset, zoffset, x, y, width, height)
        /// * `target` group: TextureTarget
        /// * `level` group: CheckedInt32
        /// * `xoffset` group: CheckedInt32
        /// * `yoffset` group: CheckedInt32
        /// * `zoffset` group: CheckedInt32
        /// * `x` group: WinCoord
        /// * `y` group: WinCoord
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CopyTexSubImage3D(
            &self,
            target: GLenum,
            level: GLint,
            xoffset: GLint,
            yoffset: GLint,
            zoffset: GLint,
            x: GLint,
            y: GLint,
            width: GLsizei,
            height: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CopyTexSubImage3D({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    level,
                    xoffset,
                    yoffset,
                    zoffset,
                    x,
                    y,
                    width,
                    height
                );
            }
            let out = call_atomic_ptr_9arg(
                "glCopyTexSubImage3D",
                &self.glCopyTexSubImage3D_p,
                target,
                level,
                xoffset,
                yoffset,
                zoffset,
                x,
                y,
                width,
                height,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyTexSubImage3D");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CopyTexSubImage3D_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyTexSubImage3D\0",
                &self.glCopyTexSubImage3D_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CopyTexSubImage3D_is_loaded(&self) -> bool {
            !self.glCopyTexSubImage3D_p.load(RELAX).is_null()
        }
        /// [glCopyTexSubImage3DOES](http://docs.gl/es3/glCopyTexSubImage3DOES)(target, level, xoffset, yoffset, zoffset, x, y, width, height)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        pub unsafe fn CopyTexSubImage3DOES(
            &self,
            target: GLenum,
            level: GLint,
            xoffset: GLint,
            yoffset: GLint,
            zoffset: GLint,
            x: GLint,
            y: GLint,
            width: GLsizei,
            height: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CopyTexSubImage3DOES({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    level,
                    xoffset,
                    yoffset,
                    zoffset,
                    x,
                    y,
                    width,
                    height
                );
            }
            let out = call_atomic_ptr_9arg(
                "glCopyTexSubImage3DOES",
                &self.glCopyTexSubImage3DOES_p,
                target,
                level,
                xoffset,
                yoffset,
                zoffset,
                x,
                y,
                width,
                height,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyTexSubImage3DOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        #[doc(hidden)]
        pub unsafe fn CopyTexSubImage3DOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyTexSubImage3DOES\0",
                &self.glCopyTexSubImage3DOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        pub fn CopyTexSubImage3DOES_is_loaded(&self) -> bool {
            !self.glCopyTexSubImage3DOES_p.load(RELAX).is_null()
        }
        /// [glCopyTextureLevelsAPPLE](http://docs.gl/es3/glCopyTextureLevelsAPPLE)(destinationTexture, sourceTexture, sourceBaseLevel, sourceLevelCount)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_APPLE_copy_texture_levels"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_copy_texture_levels"))))]
        pub unsafe fn CopyTextureLevelsAPPLE(
            &self,
            destinationTexture: GLuint,
            sourceTexture: GLuint,
            sourceBaseLevel: GLint,
            sourceLevelCount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CopyTextureLevelsAPPLE({:?}, {:?}, {:?}, {:?});",
                    destinationTexture,
                    sourceTexture,
                    sourceBaseLevel,
                    sourceLevelCount
                );
            }
            let out = call_atomic_ptr_4arg(
                "glCopyTextureLevelsAPPLE",
                &self.glCopyTextureLevelsAPPLE_p,
                destinationTexture,
                sourceTexture,
                sourceBaseLevel,
                sourceLevelCount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCopyTextureLevelsAPPLE");
            }
            out
        }
        #[cfg(any(feature = "GL_APPLE_copy_texture_levels"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_copy_texture_levels"))))]
        #[doc(hidden)]
        pub unsafe fn CopyTextureLevelsAPPLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCopyTextureLevelsAPPLE\0",
                &self.glCopyTextureLevelsAPPLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_APPLE_copy_texture_levels"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_copy_texture_levels"))))]
        pub fn CopyTextureLevelsAPPLE_is_loaded(&self) -> bool {
            !self.glCopyTextureLevelsAPPLE_p.load(RELAX).is_null()
        }
        /// [glCoverFillPathInstancedNV](http://docs.gl/es3/glCoverFillPathInstancedNV)(numPaths, pathNameType, paths, pathBase, coverMode, transformType, transformValues)
        /// * `pathNameType` group: PathElementType
        /// * `paths` group: PathElement
        /// * `paths` len: COMPSIZE(numPaths,pathNameType,paths)
        /// * `pathBase` group: Path
        /// * `coverMode` group: PathCoverMode
        /// * `transformType` group: PathTransformType
        /// * `transformValues` len: COMPSIZE(numPaths,transformType)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn CoverFillPathInstancedNV(
            &self,
            numPaths: GLsizei,
            pathNameType: GLenum,
            paths: *const c_void,
            pathBase: GLuint,
            coverMode: GLenum,
            transformType: GLenum,
            transformValues: *const GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CoverFillPathInstancedNV({:?}, {:#X}, {:p}, {:?}, {:#X}, {:#X}, {:p});",
                    numPaths,
                    pathNameType,
                    paths,
                    pathBase,
                    coverMode,
                    transformType,
                    transformValues
                );
            }
            let out = call_atomic_ptr_7arg(
                "glCoverFillPathInstancedNV",
                &self.glCoverFillPathInstancedNV_p,
                numPaths,
                pathNameType,
                paths,
                pathBase,
                coverMode,
                transformType,
                transformValues,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCoverFillPathInstancedNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn CoverFillPathInstancedNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCoverFillPathInstancedNV\0",
                &self.glCoverFillPathInstancedNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn CoverFillPathInstancedNV_is_loaded(&self) -> bool {
            !self.glCoverFillPathInstancedNV_p.load(RELAX).is_null()
        }
        /// [glCoverFillPathNV](http://docs.gl/es3/glCoverFillPathNV)(path, coverMode)
        /// * `path` group: Path
        /// * `coverMode` group: PathCoverMode
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn CoverFillPathNV(&self, path: GLuint, coverMode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CoverFillPathNV({:?}, {:#X});", path, coverMode);
            }
            let out = call_atomic_ptr_2arg(
                "glCoverFillPathNV",
                &self.glCoverFillPathNV_p,
                path,
                coverMode,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCoverFillPathNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn CoverFillPathNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCoverFillPathNV\0",
                &self.glCoverFillPathNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn CoverFillPathNV_is_loaded(&self) -> bool {
            !self.glCoverFillPathNV_p.load(RELAX).is_null()
        }
        /// [glCoverStrokePathInstancedNV](http://docs.gl/es3/glCoverStrokePathInstancedNV)(numPaths, pathNameType, paths, pathBase, coverMode, transformType, transformValues)
        /// * `pathNameType` group: PathElementType
        /// * `paths` group: PathElement
        /// * `paths` len: COMPSIZE(numPaths,pathNameType,paths)
        /// * `pathBase` group: Path
        /// * `coverMode` group: PathCoverMode
        /// * `transformType` group: PathTransformType
        /// * `transformValues` len: COMPSIZE(numPaths,transformType)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn CoverStrokePathInstancedNV(
            &self,
            numPaths: GLsizei,
            pathNameType: GLenum,
            paths: *const c_void,
            pathBase: GLuint,
            coverMode: GLenum,
            transformType: GLenum,
            transformValues: *const GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CoverStrokePathInstancedNV({:?}, {:#X}, {:p}, {:?}, {:#X}, {:#X}, {:p});",
                    numPaths,
                    pathNameType,
                    paths,
                    pathBase,
                    coverMode,
                    transformType,
                    transformValues
                );
            }
            let out = call_atomic_ptr_7arg(
                "glCoverStrokePathInstancedNV",
                &self.glCoverStrokePathInstancedNV_p,
                numPaths,
                pathNameType,
                paths,
                pathBase,
                coverMode,
                transformType,
                transformValues,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCoverStrokePathInstancedNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn CoverStrokePathInstancedNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCoverStrokePathInstancedNV\0",
                &self.glCoverStrokePathInstancedNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn CoverStrokePathInstancedNV_is_loaded(&self) -> bool {
            !self.glCoverStrokePathInstancedNV_p.load(RELAX).is_null()
        }
        /// [glCoverStrokePathNV](http://docs.gl/es3/glCoverStrokePathNV)(path, coverMode)
        /// * `path` group: Path
        /// * `coverMode` group: PathCoverMode
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn CoverStrokePathNV(&self, path: GLuint, coverMode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CoverStrokePathNV({:?}, {:#X});",
                    path,
                    coverMode
                );
            }
            let out = call_atomic_ptr_2arg(
                "glCoverStrokePathNV",
                &self.glCoverStrokePathNV_p,
                path,
                coverMode,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCoverStrokePathNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn CoverStrokePathNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCoverStrokePathNV\0",
                &self.glCoverStrokePathNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn CoverStrokePathNV_is_loaded(&self) -> bool {
            !self.glCoverStrokePathNV_p.load(RELAX).is_null()
        }
        /// [glCoverageMaskNV](http://docs.gl/es3/glCoverageMaskNV)(mask)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_coverage_sample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
        pub unsafe fn CoverageMaskNV(&self, mask: GLboolean) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CoverageMaskNV({:?});", mask);
            }
            let out = call_atomic_ptr_1arg("glCoverageMaskNV", &self.glCoverageMaskNV_p, mask);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCoverageMaskNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_coverage_sample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
        #[doc(hidden)]
        pub unsafe fn CoverageMaskNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCoverageMaskNV\0",
                &self.glCoverageMaskNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_coverage_sample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
        pub fn CoverageMaskNV_is_loaded(&self) -> bool {
            !self.glCoverageMaskNV_p.load(RELAX).is_null()
        }
        /// [glCoverageModulationNV](http://docs.gl/es3/glCoverageModulationNV)(components)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        pub unsafe fn CoverageModulationNV(&self, components: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CoverageModulationNV({:#X});", components);
            }
            let out = call_atomic_ptr_1arg(
                "glCoverageModulationNV",
                &self.glCoverageModulationNV_p,
                components,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCoverageModulationNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        #[doc(hidden)]
        pub unsafe fn CoverageModulationNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCoverageModulationNV\0",
                &self.glCoverageModulationNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        pub fn CoverageModulationNV_is_loaded(&self) -> bool {
            !self.glCoverageModulationNV_p.load(RELAX).is_null()
        }
        /// [glCoverageModulationTableNV](http://docs.gl/es3/glCoverageModulationTableNV)(n, v)
        /// * `v` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        pub unsafe fn CoverageModulationTableNV(&self, n: GLsizei, v: *const GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CoverageModulationTableNV({:?}, {:p});", n, v);
            }
            let out = call_atomic_ptr_2arg(
                "glCoverageModulationTableNV",
                &self.glCoverageModulationTableNV_p,
                n,
                v,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCoverageModulationTableNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        #[doc(hidden)]
        pub unsafe fn CoverageModulationTableNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCoverageModulationTableNV\0",
                &self.glCoverageModulationTableNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        pub fn CoverageModulationTableNV_is_loaded(&self) -> bool {
            !self.glCoverageModulationTableNV_p.load(RELAX).is_null()
        }
        /// [glCoverageOperationNV](http://docs.gl/es3/glCoverageOperationNV)(operation)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_coverage_sample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
        pub unsafe fn CoverageOperationNV(&self, operation: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CoverageOperationNV({:#X});", operation);
            }
            let out = call_atomic_ptr_1arg(
                "glCoverageOperationNV",
                &self.glCoverageOperationNV_p,
                operation,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCoverageOperationNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_coverage_sample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
        #[doc(hidden)]
        pub unsafe fn CoverageOperationNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCoverageOperationNV\0",
                &self.glCoverageOperationNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_coverage_sample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
        pub fn CoverageOperationNV_is_loaded(&self) -> bool {
            !self.glCoverageOperationNV_p.load(RELAX).is_null()
        }
        /// [glCreateMemoryObjectsEXT](http://docs.gl/es3/glCreateMemoryObjectsEXT)(n, memoryObjects)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        pub unsafe fn CreateMemoryObjectsEXT(&self, n: GLsizei, memoryObjects: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CreateMemoryObjectsEXT({:?}, {:p});",
                    n,
                    memoryObjects
                );
            }
            let out = call_atomic_ptr_2arg(
                "glCreateMemoryObjectsEXT",
                &self.glCreateMemoryObjectsEXT_p,
                n,
                memoryObjects,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCreateMemoryObjectsEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        #[doc(hidden)]
        pub unsafe fn CreateMemoryObjectsEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCreateMemoryObjectsEXT\0",
                &self.glCreateMemoryObjectsEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        pub fn CreateMemoryObjectsEXT_is_loaded(&self) -> bool {
            !self.glCreateMemoryObjectsEXT_p.load(RELAX).is_null()
        }
        /// [glCreatePerfQueryINTEL](http://docs.gl/es3/glCreatePerfQueryINTEL)(queryId, queryHandle)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn CreatePerfQueryINTEL(&self, queryId: GLuint, queryHandle: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CreatePerfQueryINTEL({:?}, {:p});",
                    queryId,
                    queryHandle
                );
            }
            let out = call_atomic_ptr_2arg(
                "glCreatePerfQueryINTEL",
                &self.glCreatePerfQueryINTEL_p,
                queryId,
                queryHandle,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCreatePerfQueryINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn CreatePerfQueryINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCreatePerfQueryINTEL\0",
                &self.glCreatePerfQueryINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn CreatePerfQueryINTEL_is_loaded(&self) -> bool {
            !self.glCreatePerfQueryINTEL_p.load(RELAX).is_null()
        }
        /// [glCreateProgram](http://docs.gl/es3/glCreateProgram)()
        /// * return value class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CreateProgram(&self) -> GLuint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CreateProgram();",);
            }
            let out = call_atomic_ptr_0arg("glCreateProgram", &self.glCreateProgram_p);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCreateProgram");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CreateProgram_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCreateProgram\0",
                &self.glCreateProgram_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CreateProgram_is_loaded(&self) -> bool {
            !self.glCreateProgram_p.load(RELAX).is_null()
        }
        /// [glCreateSemaphoresNV](http://docs.gl/es3/glCreateSemaphoresNV)(n, semaphores)
        /// * `semaphores` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
        pub unsafe fn CreateSemaphoresNV(&self, n: GLsizei, semaphores: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CreateSemaphoresNV({:?}, {:p});", n, semaphores);
            }
            let out = call_atomic_ptr_2arg(
                "glCreateSemaphoresNV",
                &self.glCreateSemaphoresNV_p,
                n,
                semaphores,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCreateSemaphoresNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
        #[doc(hidden)]
        pub unsafe fn CreateSemaphoresNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCreateSemaphoresNV\0",
                &self.glCreateSemaphoresNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
        pub fn CreateSemaphoresNV_is_loaded(&self) -> bool {
            !self.glCreateSemaphoresNV_p.load(RELAX).is_null()
        }
        /// [glCreateShader](http://docs.gl/es3/glCreateShader)(type_)
        /// * `type_` group: ShaderType
        /// * return value class: shader
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CreateShader(&self, type_: GLenum) -> GLuint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CreateShader({:#X});", type_);
            }
            let out = call_atomic_ptr_1arg("glCreateShader", &self.glCreateShader_p, type_);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCreateShader");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CreateShader_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCreateShader\0",
                &self.glCreateShader_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CreateShader_is_loaded(&self) -> bool {
            !self.glCreateShader_p.load(RELAX).is_null()
        }
        /// [glCreateShaderProgramv](http://docs.gl/es3/glCreateShaderProgramv)(type_, count, strings)
        /// * `type_` group: ShaderType
        /// * `strings` len: count
        /// * return value class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CreateShaderProgramv(
            &self,
            type_: GLenum,
            count: GLsizei,
            strings: *const *const GLchar,
        ) -> GLuint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CreateShaderProgramv({:#X}, {:?}, {:p});",
                    type_,
                    count,
                    strings
                );
            }
            let out = call_atomic_ptr_3arg(
                "glCreateShaderProgramv",
                &self.glCreateShaderProgramv_p,
                type_,
                count,
                strings,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCreateShaderProgramv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CreateShaderProgramv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCreateShaderProgramv\0",
                &self.glCreateShaderProgramv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn CreateShaderProgramv_is_loaded(&self) -> bool {
            !self.glCreateShaderProgramv_p.load(RELAX).is_null()
        }
        /// [glCreateShaderProgramvEXT](http://docs.gl/es3/glCreateShaderProgramvEXT)(type_, count, strings)
        /// * `type_` group: ShaderType
        /// * `strings` len: count
        /// * return value class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub unsafe fn CreateShaderProgramvEXT(
            &self,
            type_: GLenum,
            count: GLsizei,
            strings: *const GLchar,
        ) -> GLuint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.CreateShaderProgramvEXT({:#X}, {:?}, {:p});",
                    type_,
                    count,
                    strings
                );
            }
            let out = call_atomic_ptr_3arg(
                "glCreateShaderProgramvEXT",
                &self.glCreateShaderProgramvEXT_p,
                type_,
                count,
                strings,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCreateShaderProgramvEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        #[doc(hidden)]
        pub unsafe fn CreateShaderProgramvEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glCreateShaderProgramvEXT\0",
                &self.glCreateShaderProgramvEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub fn CreateShaderProgramvEXT_is_loaded(&self) -> bool {
            !self.glCreateShaderProgramvEXT_p.load(RELAX).is_null()
        }
        /// [glCullFace](http://docs.gl/es3/glCullFace)(mode)
        /// * `mode` group: CullFaceMode
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn CullFace(&self, mode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.CullFace({:#X});", mode);
            }
            let out = call_atomic_ptr_1arg("glCullFace", &self.glCullFace_p, mode);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glCullFace");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn CullFace_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glCullFace\0", &self.glCullFace_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn CullFace_is_loaded(&self) -> bool {
            !self.glCullFace_p.load(RELAX).is_null()
        }
        /// [glDebugMessageCallback](http://docs.gl/es3/glDebugMessageCallback)(callback, userParam)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DebugMessageCallback(&self, callback: GLDEBUGPROC, userParam: *const c_void) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DebugMessageCallback({:?}, {:p});",
                    transmute::<_, Option<fn()>>(callback),
                    userParam
                );
            }
            let out = call_atomic_ptr_2arg(
                "glDebugMessageCallback",
                &self.glDebugMessageCallback_p,
                callback,
                userParam,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDebugMessageCallback");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DebugMessageCallback_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDebugMessageCallback\0",
                &self.glDebugMessageCallback_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DebugMessageCallback_is_loaded(&self) -> bool {
            !self.glDebugMessageCallback_p.load(RELAX).is_null()
        }
        /// [glDebugMessageCallbackKHR](http://docs.gl/es3/glDebugMessageCallbackKHR)(callback, userParam)
        /// * alias of: [`glDebugMessageCallback`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub unsafe fn DebugMessageCallbackKHR(
            &self,
            callback: GLDEBUGPROCKHR,
            userParam: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DebugMessageCallbackKHR({:?}, {:p});",
                    transmute::<_, Option<fn()>>(callback),
                    userParam
                );
            }
            let out = call_atomic_ptr_2arg(
                "glDebugMessageCallbackKHR",
                &self.glDebugMessageCallbackKHR_p,
                callback,
                userParam,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDebugMessageCallbackKHR");
            }
            out
        }
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        #[doc(hidden)]
        pub unsafe fn DebugMessageCallbackKHR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDebugMessageCallbackKHR\0",
                &self.glDebugMessageCallbackKHR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub fn DebugMessageCallbackKHR_is_loaded(&self) -> bool {
            !self.glDebugMessageCallbackKHR_p.load(RELAX).is_null()
        }
        /// [glDebugMessageControl](http://docs.gl/es3/glDebugMessageControl)(source, type_, severity, count, ids, enabled)
        /// * `source` group: DebugSource
        /// * `type_` group: DebugType
        /// * `severity` group: DebugSeverity
        /// * `ids` len: count
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DebugMessageControl(
            &self,
            source: GLenum,
            type_: GLenum,
            severity: GLenum,
            count: GLsizei,
            ids: *const GLuint,
            enabled: GLboolean,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DebugMessageControl({:#X}, {:#X}, {:#X}, {:?}, {:p}, {:?});",
                    source,
                    type_,
                    severity,
                    count,
                    ids,
                    enabled
                );
            }
            let out = call_atomic_ptr_6arg(
                "glDebugMessageControl",
                &self.glDebugMessageControl_p,
                source,
                type_,
                severity,
                count,
                ids,
                enabled,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDebugMessageControl");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DebugMessageControl_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDebugMessageControl\0",
                &self.glDebugMessageControl_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DebugMessageControl_is_loaded(&self) -> bool {
            !self.glDebugMessageControl_p.load(RELAX).is_null()
        }
        /// [glDebugMessageControlKHR](http://docs.gl/es3/glDebugMessageControlKHR)(source, type_, severity, count, ids, enabled)
        /// * `source` group: DebugSource
        /// * `type_` group: DebugType
        /// * `severity` group: DebugSeverity
        /// * alias of: [`glDebugMessageControl`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub unsafe fn DebugMessageControlKHR(
            &self,
            source: GLenum,
            type_: GLenum,
            severity: GLenum,
            count: GLsizei,
            ids: *const GLuint,
            enabled: GLboolean,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DebugMessageControlKHR({:#X}, {:#X}, {:#X}, {:?}, {:p}, {:?});",
                    source,
                    type_,
                    severity,
                    count,
                    ids,
                    enabled
                );
            }
            let out = call_atomic_ptr_6arg(
                "glDebugMessageControlKHR",
                &self.glDebugMessageControlKHR_p,
                source,
                type_,
                severity,
                count,
                ids,
                enabled,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDebugMessageControlKHR");
            }
            out
        }
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        #[doc(hidden)]
        pub unsafe fn DebugMessageControlKHR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDebugMessageControlKHR\0",
                &self.glDebugMessageControlKHR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub fn DebugMessageControlKHR_is_loaded(&self) -> bool {
            !self.glDebugMessageControlKHR_p.load(RELAX).is_null()
        }
        /// [glDebugMessageInsert](http://docs.gl/es3/glDebugMessageInsert)(source, type_, id, severity, length, buf)
        /// * `source` group: DebugSource
        /// * `type_` group: DebugType
        /// * `severity` group: DebugSeverity
        /// * `buf` len: COMPSIZE(buf,length)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DebugMessageInsert(
            &self,
            source: GLenum,
            type_: GLenum,
            id: GLuint,
            severity: GLenum,
            length: GLsizei,
            buf: *const GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DebugMessageInsert({:#X}, {:#X}, {:?}, {:#X}, {:?}, {:p});",
                    source,
                    type_,
                    id,
                    severity,
                    length,
                    buf
                );
            }
            let out = call_atomic_ptr_6arg(
                "glDebugMessageInsert",
                &self.glDebugMessageInsert_p,
                source,
                type_,
                id,
                severity,
                length,
                buf,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDebugMessageInsert");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DebugMessageInsert_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDebugMessageInsert\0",
                &self.glDebugMessageInsert_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DebugMessageInsert_is_loaded(&self) -> bool {
            !self.glDebugMessageInsert_p.load(RELAX).is_null()
        }
        /// [glDebugMessageInsertKHR](http://docs.gl/es3/glDebugMessageInsertKHR)(source, type_, id, severity, length, buf)
        /// * `source` group: DebugSource
        /// * `type_` group: DebugType
        /// * `severity` group: DebugSeverity
        /// * alias of: [`glDebugMessageInsert`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub unsafe fn DebugMessageInsertKHR(
            &self,
            source: GLenum,
            type_: GLenum,
            id: GLuint,
            severity: GLenum,
            length: GLsizei,
            buf: *const GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DebugMessageInsertKHR({:#X}, {:#X}, {:?}, {:#X}, {:?}, {:p});",
                    source,
                    type_,
                    id,
                    severity,
                    length,
                    buf
                );
            }
            let out = call_atomic_ptr_6arg(
                "glDebugMessageInsertKHR",
                &self.glDebugMessageInsertKHR_p,
                source,
                type_,
                id,
                severity,
                length,
                buf,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDebugMessageInsertKHR");
            }
            out
        }
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        #[doc(hidden)]
        pub unsafe fn DebugMessageInsertKHR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDebugMessageInsertKHR\0",
                &self.glDebugMessageInsertKHR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub fn DebugMessageInsertKHR_is_loaded(&self) -> bool {
            !self.glDebugMessageInsertKHR_p.load(RELAX).is_null()
        }
        /// [glDeleteBuffers](http://docs.gl/es3/glDeleteBuffers)(n, buffers)
        /// * `buffers` len: n
        /// * `buffers` class: buffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteBuffers(&self, n: GLsizei, buffers: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteBuffers({:?}, {:p});", n, buffers);
            }
            let out = call_atomic_ptr_2arg("glDeleteBuffers", &self.glDeleteBuffers_p, n, buffers);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteBuffers");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteBuffers_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteBuffers\0",
                &self.glDeleteBuffers_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteBuffers_is_loaded(&self) -> bool {
            !self.glDeleteBuffers_p.load(RELAX).is_null()
        }
        /// [glDeleteFencesNV](http://docs.gl/es3/glDeleteFencesNV)(n, fences)
        /// * `fences` group: FenceNV
        /// * `fences` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        pub unsafe fn DeleteFencesNV(&self, n: GLsizei, fences: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteFencesNV({:?}, {:p});", n, fences);
            }
            let out = call_atomic_ptr_2arg("glDeleteFencesNV", &self.glDeleteFencesNV_p, n, fences);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteFencesNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        #[doc(hidden)]
        pub unsafe fn DeleteFencesNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteFencesNV\0",
                &self.glDeleteFencesNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        pub fn DeleteFencesNV_is_loaded(&self) -> bool {
            !self.glDeleteFencesNV_p.load(RELAX).is_null()
        }
        /// [glDeleteFramebuffers](http://docs.gl/es3/glDeleteFramebuffers)(n, framebuffers)
        /// * `framebuffers` len: n
        /// * `framebuffers` class: framebuffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteFramebuffers(&self, n: GLsizei, framebuffers: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DeleteFramebuffers({:?}, {:p});",
                    n,
                    framebuffers
                );
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteFramebuffers",
                &self.glDeleteFramebuffers_p,
                n,
                framebuffers,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteFramebuffers");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteFramebuffers_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteFramebuffers\0",
                &self.glDeleteFramebuffers_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteFramebuffers_is_loaded(&self) -> bool {
            !self.glDeleteFramebuffers_p.load(RELAX).is_null()
        }
        /// [glDeleteMemoryObjectsEXT](http://docs.gl/es3/glDeleteMemoryObjectsEXT)(n, memoryObjects)
        /// * `memoryObjects` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        pub unsafe fn DeleteMemoryObjectsEXT(&self, n: GLsizei, memoryObjects: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DeleteMemoryObjectsEXT({:?}, {:p});",
                    n,
                    memoryObjects
                );
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteMemoryObjectsEXT",
                &self.glDeleteMemoryObjectsEXT_p,
                n,
                memoryObjects,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteMemoryObjectsEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        #[doc(hidden)]
        pub unsafe fn DeleteMemoryObjectsEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteMemoryObjectsEXT\0",
                &self.glDeleteMemoryObjectsEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        pub fn DeleteMemoryObjectsEXT_is_loaded(&self) -> bool {
            !self.glDeleteMemoryObjectsEXT_p.load(RELAX).is_null()
        }
        /// [glDeletePathsNV](http://docs.gl/es3/glDeletePathsNV)(path, range)
        /// * `path` group: Path
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn DeletePathsNV(&self, path: GLuint, range: GLsizei) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeletePathsNV({:?}, {:?});", path, range);
            }
            let out = call_atomic_ptr_2arg("glDeletePathsNV", &self.glDeletePathsNV_p, path, range);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeletePathsNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn DeletePathsNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeletePathsNV\0",
                &self.glDeletePathsNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn DeletePathsNV_is_loaded(&self) -> bool {
            !self.glDeletePathsNV_p.load(RELAX).is_null()
        }
        /// [glDeletePerfMonitorsAMD](http://docs.gl/es3/glDeletePerfMonitorsAMD)(n, monitors)
        /// * `monitors` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn DeletePerfMonitorsAMD(&self, n: GLsizei, monitors: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeletePerfMonitorsAMD({:?}, {:p});", n, monitors);
            }
            let out = call_atomic_ptr_2arg(
                "glDeletePerfMonitorsAMD",
                &self.glDeletePerfMonitorsAMD_p,
                n,
                monitors,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeletePerfMonitorsAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn DeletePerfMonitorsAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeletePerfMonitorsAMD\0",
                &self.glDeletePerfMonitorsAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn DeletePerfMonitorsAMD_is_loaded(&self) -> bool {
            !self.glDeletePerfMonitorsAMD_p.load(RELAX).is_null()
        }
        /// [glDeletePerfQueryINTEL](http://docs.gl/es3/glDeletePerfQueryINTEL)(queryHandle)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn DeletePerfQueryINTEL(&self, queryHandle: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeletePerfQueryINTEL({:?});", queryHandle);
            }
            let out = call_atomic_ptr_1arg(
                "glDeletePerfQueryINTEL",
                &self.glDeletePerfQueryINTEL_p,
                queryHandle,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeletePerfQueryINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn DeletePerfQueryINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeletePerfQueryINTEL\0",
                &self.glDeletePerfQueryINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn DeletePerfQueryINTEL_is_loaded(&self) -> bool {
            !self.glDeletePerfQueryINTEL_p.load(RELAX).is_null()
        }
        /// [glDeleteProgram](http://docs.gl/es3/glDeleteProgram)(program)
        /// * `program` class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteProgram(&self, program: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteProgram({:?});", program);
            }
            let out = call_atomic_ptr_1arg("glDeleteProgram", &self.glDeleteProgram_p, program);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteProgram");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteProgram_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteProgram\0",
                &self.glDeleteProgram_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteProgram_is_loaded(&self) -> bool {
            !self.glDeleteProgram_p.load(RELAX).is_null()
        }
        /// [glDeleteProgramPipelines](http://docs.gl/es3/glDeleteProgramPipelines)(n, pipelines)
        /// * `pipelines` len: n
        /// * `pipelines` class: program pipeline
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteProgramPipelines(&self, n: GLsizei, pipelines: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DeleteProgramPipelines({:?}, {:p});",
                    n,
                    pipelines
                );
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteProgramPipelines",
                &self.glDeleteProgramPipelines_p,
                n,
                pipelines,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteProgramPipelines");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteProgramPipelines_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteProgramPipelines\0",
                &self.glDeleteProgramPipelines_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteProgramPipelines_is_loaded(&self) -> bool {
            !self.glDeleteProgramPipelines_p.load(RELAX).is_null()
        }
        /// [glDeleteProgramPipelinesEXT](http://docs.gl/es3/glDeleteProgramPipelinesEXT)(n, pipelines)
        /// * `pipelines` len: n
        /// * `pipelines` class: program pipeline
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub unsafe fn DeleteProgramPipelinesEXT(&self, n: GLsizei, pipelines: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DeleteProgramPipelinesEXT({:?}, {:p});",
                    n,
                    pipelines
                );
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteProgramPipelinesEXT",
                &self.glDeleteProgramPipelinesEXT_p,
                n,
                pipelines,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteProgramPipelinesEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        #[doc(hidden)]
        pub unsafe fn DeleteProgramPipelinesEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteProgramPipelinesEXT\0",
                &self.glDeleteProgramPipelinesEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub fn DeleteProgramPipelinesEXT_is_loaded(&self) -> bool {
            !self.glDeleteProgramPipelinesEXT_p.load(RELAX).is_null()
        }
        /// [glDeleteQueries](http://docs.gl/es3/glDeleteQueries)(n, ids)
        /// * `ids` len: n
        /// * `ids` class: query
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteQueries(&self, n: GLsizei, ids: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteQueries({:?}, {:p});", n, ids);
            }
            let out = call_atomic_ptr_2arg("glDeleteQueries", &self.glDeleteQueries_p, n, ids);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteQueries");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteQueries_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteQueries\0",
                &self.glDeleteQueries_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteQueries_is_loaded(&self) -> bool {
            !self.glDeleteQueries_p.load(RELAX).is_null()
        }
        /// [glDeleteQueriesEXT](http://docs.gl/es3/glDeleteQueriesEXT)(n, ids)
        /// * `ids` len: n
        /// * `ids` class: query
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub unsafe fn DeleteQueriesEXT(&self, n: GLsizei, ids: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteQueriesEXT({:?}, {:p});", n, ids);
            }
            let out =
                call_atomic_ptr_2arg("glDeleteQueriesEXT", &self.glDeleteQueriesEXT_p, n, ids);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteQueriesEXT");
            }
            out
        }
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn DeleteQueriesEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteQueriesEXT\0",
                &self.glDeleteQueriesEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub fn DeleteQueriesEXT_is_loaded(&self) -> bool {
            !self.glDeleteQueriesEXT_p.load(RELAX).is_null()
        }
        /// [glDeleteRenderbuffers](http://docs.gl/es3/glDeleteRenderbuffers)(n, renderbuffers)
        /// * `renderbuffers` len: n
        /// * `renderbuffers` class: renderbuffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteRenderbuffers(&self, n: GLsizei, renderbuffers: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DeleteRenderbuffers({:?}, {:p});",
                    n,
                    renderbuffers
                );
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteRenderbuffers",
                &self.glDeleteRenderbuffers_p,
                n,
                renderbuffers,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteRenderbuffers");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteRenderbuffers_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteRenderbuffers\0",
                &self.glDeleteRenderbuffers_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteRenderbuffers_is_loaded(&self) -> bool {
            !self.glDeleteRenderbuffers_p.load(RELAX).is_null()
        }
        /// [glDeleteSamplers](http://docs.gl/es3/glDeleteSamplers)(count, samplers)
        /// * `samplers` len: count
        /// * `samplers` class: sampler
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteSamplers(&self, count: GLsizei, samplers: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteSamplers({:?}, {:p});", count, samplers);
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteSamplers",
                &self.glDeleteSamplers_p,
                count,
                samplers,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteSamplers");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteSamplers_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteSamplers\0",
                &self.glDeleteSamplers_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteSamplers_is_loaded(&self) -> bool {
            !self.glDeleteSamplers_p.load(RELAX).is_null()
        }
        /// [glDeleteSemaphoresEXT](http://docs.gl/es3/glDeleteSemaphoresEXT)(n, semaphores)
        /// * `semaphores` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        pub unsafe fn DeleteSemaphoresEXT(&self, n: GLsizei, semaphores: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteSemaphoresEXT({:?}, {:p});", n, semaphores);
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteSemaphoresEXT",
                &self.glDeleteSemaphoresEXT_p,
                n,
                semaphores,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteSemaphoresEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        #[doc(hidden)]
        pub unsafe fn DeleteSemaphoresEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteSemaphoresEXT\0",
                &self.glDeleteSemaphoresEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        pub fn DeleteSemaphoresEXT_is_loaded(&self) -> bool {
            !self.glDeleteSemaphoresEXT_p.load(RELAX).is_null()
        }
        /// [glDeleteShader](http://docs.gl/es3/glDeleteShader)(shader)
        /// * `shader` class: shader
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteShader(&self, shader: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteShader({:?});", shader);
            }
            let out = call_atomic_ptr_1arg("glDeleteShader", &self.glDeleteShader_p, shader);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteShader");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteShader_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteShader\0",
                &self.glDeleteShader_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteShader_is_loaded(&self) -> bool {
            !self.glDeleteShader_p.load(RELAX).is_null()
        }
        /// [glDeleteSync](http://docs.gl/es3/glDeleteSync)(sync)
        /// * `sync` group: sync
        /// * `sync` class: sync
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteSync(&self, sync: GLsync) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteSync({:p});", sync);
            }
            let out = call_atomic_ptr_1arg("glDeleteSync", &self.glDeleteSync_p, sync);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteSync");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteSync_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDeleteSync\0", &self.glDeleteSync_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteSync_is_loaded(&self) -> bool {
            !self.glDeleteSync_p.load(RELAX).is_null()
        }
        /// [glDeleteSyncAPPLE](http://docs.gl/es3/glDeleteSyncAPPLE)(sync)
        /// * `sync` class: sync
        /// * alias of: [`glDeleteSync`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub unsafe fn DeleteSyncAPPLE(&self, sync: GLsync) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteSyncAPPLE({:p});", sync);
            }
            let out = call_atomic_ptr_1arg("glDeleteSyncAPPLE", &self.glDeleteSyncAPPLE_p, sync);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteSyncAPPLE");
            }
            out
        }
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        #[doc(hidden)]
        pub unsafe fn DeleteSyncAPPLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteSyncAPPLE\0",
                &self.glDeleteSyncAPPLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub fn DeleteSyncAPPLE_is_loaded(&self) -> bool {
            !self.glDeleteSyncAPPLE_p.load(RELAX).is_null()
        }
        /// [glDeleteTextures](http://docs.gl/es3/glDeleteTextures)(n, textures)
        /// * `textures` group: Texture
        /// * `textures` len: n
        /// * `textures` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteTextures(&self, n: GLsizei, textures: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteTextures({:?}, {:p});", n, textures);
            }
            let out =
                call_atomic_ptr_2arg("glDeleteTextures", &self.glDeleteTextures_p, n, textures);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteTextures");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteTextures_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteTextures\0",
                &self.glDeleteTextures_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteTextures_is_loaded(&self) -> bool {
            !self.glDeleteTextures_p.load(RELAX).is_null()
        }
        /// [glDeleteTransformFeedbacks](http://docs.gl/es3/glDeleteTransformFeedbacks)(n, ids)
        /// * `ids` len: n
        /// * `ids` class: transform feedback
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteTransformFeedbacks(&self, n: GLsizei, ids: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteTransformFeedbacks({:?}, {:p});", n, ids);
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteTransformFeedbacks",
                &self.glDeleteTransformFeedbacks_p,
                n,
                ids,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteTransformFeedbacks");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteTransformFeedbacks_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteTransformFeedbacks\0",
                &self.glDeleteTransformFeedbacks_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteTransformFeedbacks_is_loaded(&self) -> bool {
            !self.glDeleteTransformFeedbacks_p.load(RELAX).is_null()
        }
        /// [glDeleteVertexArrays](http://docs.gl/es3/glDeleteVertexArrays)(n, arrays)
        /// * `arrays` len: n
        /// * `arrays` class: vertex array
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DeleteVertexArrays(&self, n: GLsizei, arrays: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteVertexArrays({:?}, {:p});", n, arrays);
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteVertexArrays",
                &self.glDeleteVertexArrays_p,
                n,
                arrays,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteVertexArrays");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DeleteVertexArrays_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteVertexArrays\0",
                &self.glDeleteVertexArrays_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DeleteVertexArrays_is_loaded(&self) -> bool {
            !self.glDeleteVertexArrays_p.load(RELAX).is_null()
        }
        /// [glDeleteVertexArraysOES](http://docs.gl/es3/glDeleteVertexArraysOES)(n, arrays)
        /// * `arrays` len: n
        /// * `arrays` class: vertex array
        /// * alias of: [`glDeleteVertexArrays`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        pub unsafe fn DeleteVertexArraysOES(&self, n: GLsizei, arrays: *const GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DeleteVertexArraysOES({:?}, {:p});", n, arrays);
            }
            let out = call_atomic_ptr_2arg(
                "glDeleteVertexArraysOES",
                &self.glDeleteVertexArraysOES_p,
                n,
                arrays,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDeleteVertexArraysOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        #[doc(hidden)]
        pub unsafe fn DeleteVertexArraysOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDeleteVertexArraysOES\0",
                &self.glDeleteVertexArraysOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        pub fn DeleteVertexArraysOES_is_loaded(&self) -> bool {
            !self.glDeleteVertexArraysOES_p.load(RELAX).is_null()
        }
        /// [glDepthFunc](http://docs.gl/es3/glDepthFunc)(func)
        /// * `func` group: DepthFunction
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DepthFunc(&self, func: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DepthFunc({:#X});", func);
            }
            let out = call_atomic_ptr_1arg("glDepthFunc", &self.glDepthFunc_p, func);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDepthFunc");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DepthFunc_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDepthFunc\0", &self.glDepthFunc_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn DepthFunc_is_loaded(&self) -> bool {
            !self.glDepthFunc_p.load(RELAX).is_null()
        }
        /// [glDepthMask](http://docs.gl/es3/glDepthMask)(flag)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DepthMask(&self, flag: GLboolean) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DepthMask({:?});", flag);
            }
            let out = call_atomic_ptr_1arg("glDepthMask", &self.glDepthMask_p, flag);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDepthMask");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DepthMask_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDepthMask\0", &self.glDepthMask_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn DepthMask_is_loaded(&self) -> bool {
            !self.glDepthMask_p.load(RELAX).is_null()
        }
        /// [glDepthRangeArrayfvNV](http://docs.gl/es3/glDepthRangeArrayfvNV)(first, count, v)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub unsafe fn DepthRangeArrayfvNV(&self, first: GLuint, count: GLsizei, v: *const GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DepthRangeArrayfvNV({:?}, {:?}, {:p});",
                    first,
                    count,
                    v
                );
            }
            let out = call_atomic_ptr_3arg(
                "glDepthRangeArrayfvNV",
                &self.glDepthRangeArrayfvNV_p,
                first,
                count,
                v,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDepthRangeArrayfvNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        #[doc(hidden)]
        pub unsafe fn DepthRangeArrayfvNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDepthRangeArrayfvNV\0",
                &self.glDepthRangeArrayfvNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub fn DepthRangeArrayfvNV_is_loaded(&self) -> bool {
            !self.glDepthRangeArrayfvNV_p.load(RELAX).is_null()
        }
        /// [glDepthRangeArrayfvOES](http://docs.gl/es3/glDepthRangeArrayfvOES)(first, count, v)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        pub unsafe fn DepthRangeArrayfvOES(
            &self,
            first: GLuint,
            count: GLsizei,
            v: *const GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DepthRangeArrayfvOES({:?}, {:?}, {:p});",
                    first,
                    count,
                    v
                );
            }
            let out = call_atomic_ptr_3arg(
                "glDepthRangeArrayfvOES",
                &self.glDepthRangeArrayfvOES_p,
                first,
                count,
                v,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDepthRangeArrayfvOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        #[doc(hidden)]
        pub unsafe fn DepthRangeArrayfvOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDepthRangeArrayfvOES\0",
                &self.glDepthRangeArrayfvOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        pub fn DepthRangeArrayfvOES_is_loaded(&self) -> bool {
            !self.glDepthRangeArrayfvOES_p.load(RELAX).is_null()
        }
        /// [glDepthRangeIndexedfNV](http://docs.gl/es3/glDepthRangeIndexedfNV)(index, n, f)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub unsafe fn DepthRangeIndexedfNV(&self, index: GLuint, n: GLfloat, f: GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DepthRangeIndexedfNV({:?}, {:?}, {:?});",
                    index,
                    n,
                    f
                );
            }
            let out = call_atomic_ptr_3arg(
                "glDepthRangeIndexedfNV",
                &self.glDepthRangeIndexedfNV_p,
                index,
                n,
                f,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDepthRangeIndexedfNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        #[doc(hidden)]
        pub unsafe fn DepthRangeIndexedfNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDepthRangeIndexedfNV\0",
                &self.glDepthRangeIndexedfNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub fn DepthRangeIndexedfNV_is_loaded(&self) -> bool {
            !self.glDepthRangeIndexedfNV_p.load(RELAX).is_null()
        }
        /// [glDepthRangeIndexedfOES](http://docs.gl/es3/glDepthRangeIndexedfOES)(index, n, f)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        pub unsafe fn DepthRangeIndexedfOES(&self, index: GLuint, n: GLfloat, f: GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DepthRangeIndexedfOES({:?}, {:?}, {:?});",
                    index,
                    n,
                    f
                );
            }
            let out = call_atomic_ptr_3arg(
                "glDepthRangeIndexedfOES",
                &self.glDepthRangeIndexedfOES_p,
                index,
                n,
                f,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDepthRangeIndexedfOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        #[doc(hidden)]
        pub unsafe fn DepthRangeIndexedfOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDepthRangeIndexedfOES\0",
                &self.glDepthRangeIndexedfOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        pub fn DepthRangeIndexedfOES_is_loaded(&self) -> bool {
            !self.glDepthRangeIndexedfOES_p.load(RELAX).is_null()
        }
        /// [glDepthRangef](http://docs.gl/es3/glDepthRange)(n, f)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DepthRangef(&self, n: GLfloat, f: GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DepthRangef({:?}, {:?});", n, f);
            }
            let out = call_atomic_ptr_2arg("glDepthRangef", &self.glDepthRangef_p, n, f);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDepthRangef");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DepthRangef_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDepthRangef\0", &self.glDepthRangef_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn DepthRangef_is_loaded(&self) -> bool {
            !self.glDepthRangef_p.load(RELAX).is_null()
        }
        /// [glDetachShader](http://docs.gl/es3/glDetachShader)(program, shader)
        /// * `program` class: program
        /// * `shader` class: shader
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DetachShader(&self, program: GLuint, shader: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DetachShader({:?}, {:?});", program, shader);
            }
            let out =
                call_atomic_ptr_2arg("glDetachShader", &self.glDetachShader_p, program, shader);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDetachShader");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DetachShader_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDetachShader\0",
                &self.glDetachShader_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DetachShader_is_loaded(&self) -> bool {
            !self.glDetachShader_p.load(RELAX).is_null()
        }
        /// [glDisable](http://docs.gl/es3/glDisable)(cap)
        /// * `cap` group: EnableCap
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn Disable(&self, cap: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.Disable({:#X});", cap);
            }
            let out = call_atomic_ptr_1arg("glDisable", &self.glDisable_p, cap);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDisable");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn Disable_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDisable\0", &self.glDisable_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn Disable_is_loaded(&self) -> bool {
            !self.glDisable_p.load(RELAX).is_null()
        }
        /// [glDisableDriverControlQCOM](http://docs.gl/es3/glDisableDriverControlQCOM)(driverControl)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        pub unsafe fn DisableDriverControlQCOM(&self, driverControl: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DisableDriverControlQCOM({:?});", driverControl);
            }
            let out = call_atomic_ptr_1arg(
                "glDisableDriverControlQCOM",
                &self.glDisableDriverControlQCOM_p,
                driverControl,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDisableDriverControlQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        #[doc(hidden)]
        pub unsafe fn DisableDriverControlQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDisableDriverControlQCOM\0",
                &self.glDisableDriverControlQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        pub fn DisableDriverControlQCOM_is_loaded(&self) -> bool {
            !self.glDisableDriverControlQCOM_p.load(RELAX).is_null()
        }
        /// [glDisableVertexAttribArray](http://docs.gl/es3/glDisableVertexAttribArray)(index)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DisableVertexAttribArray(&self, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DisableVertexAttribArray({:?});", index);
            }
            let out = call_atomic_ptr_1arg(
                "glDisableVertexAttribArray",
                &self.glDisableVertexAttribArray_p,
                index,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDisableVertexAttribArray");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DisableVertexAttribArray_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDisableVertexAttribArray\0",
                &self.glDisableVertexAttribArray_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DisableVertexAttribArray_is_loaded(&self) -> bool {
            !self.glDisableVertexAttribArray_p.load(RELAX).is_null()
        }
        /// [glDisablei](http://docs.gl/es3/glDisable)(target, index)
        /// * `target` group: EnableCap
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn Disablei(&self, target: GLenum, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.Disablei({:#X}, {:?});", target, index);
            }
            let out = call_atomic_ptr_2arg("glDisablei", &self.glDisablei_p, target, index);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDisablei");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn Disablei_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDisablei\0", &self.glDisablei_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn Disablei_is_loaded(&self) -> bool {
            !self.glDisablei_p.load(RELAX).is_null()
        }
        /// [glDisableiEXT](http://docs.gl/es3/glDisableiEXT)(target, index)
        /// * `target` group: EnableCap
        /// * alias of: [`glDisablei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub unsafe fn DisableiEXT(&self, target: GLenum, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DisableiEXT({:#X}, {:?});", target, index);
            }
            let out = call_atomic_ptr_2arg("glDisableiEXT", &self.glDisableiEXT_p, target, index);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDisableiEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn DisableiEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDisableiEXT\0", &self.glDisableiEXT_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub fn DisableiEXT_is_loaded(&self) -> bool {
            !self.glDisableiEXT_p.load(RELAX).is_null()
        }
        /// [glDisableiNV](http://docs.gl/es3/glDisableiNV)(target, index)
        /// * `target` group: EnableCap
        /// * alias of: [`glDisablei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub unsafe fn DisableiNV(&self, target: GLenum, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DisableiNV({:#X}, {:?});", target, index);
            }
            let out = call_atomic_ptr_2arg("glDisableiNV", &self.glDisableiNV_p, target, index);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDisableiNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        #[doc(hidden)]
        pub unsafe fn DisableiNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDisableiNV\0", &self.glDisableiNV_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub fn DisableiNV_is_loaded(&self) -> bool {
            !self.glDisableiNV_p.load(RELAX).is_null()
        }
        /// [glDisableiOES](http://docs.gl/es3/glDisableiOES)(target, index)
        /// * `target` group: EnableCap
        /// * alias of: [`glDisablei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            )))
        )]
        pub unsafe fn DisableiOES(&self, target: GLenum, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DisableiOES({:#X}, {:?});", target, index);
            }
            let out = call_atomic_ptr_2arg("glDisableiOES", &self.glDisableiOES_p, target, index);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDisableiOES");
            }
            out
        }
        #[cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn DisableiOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDisableiOES\0", &self.glDisableiOES_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            )))
        )]
        pub fn DisableiOES_is_loaded(&self) -> bool {
            !self.glDisableiOES_p.load(RELAX).is_null()
        }
        /// [glDiscardFramebufferEXT](http://docs.gl/es3/glDiscardFramebufferEXT)(target, numAttachments, attachments)
        /// * `target` group: FramebufferTarget
        /// * `attachments` group: InvalidateFramebufferAttachment
        /// * `attachments` len: numAttachments
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_discard_framebuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_discard_framebuffer"))))]
        pub unsafe fn DiscardFramebufferEXT(
            &self,
            target: GLenum,
            numAttachments: GLsizei,
            attachments: *const GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DiscardFramebufferEXT({:#X}, {:?}, {:p});",
                    target,
                    numAttachments,
                    attachments
                );
            }
            let out = call_atomic_ptr_3arg(
                "glDiscardFramebufferEXT",
                &self.glDiscardFramebufferEXT_p,
                target,
                numAttachments,
                attachments,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDiscardFramebufferEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_discard_framebuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_discard_framebuffer"))))]
        #[doc(hidden)]
        pub unsafe fn DiscardFramebufferEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDiscardFramebufferEXT\0",
                &self.glDiscardFramebufferEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_discard_framebuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_discard_framebuffer"))))]
        pub fn DiscardFramebufferEXT_is_loaded(&self) -> bool {
            !self.glDiscardFramebufferEXT_p.load(RELAX).is_null()
        }
        /// [glDispatchCompute](http://docs.gl/es3/glDispatchCompute)(num_groups_x, num_groups_y, num_groups_z)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DispatchCompute(
            &self,
            num_groups_x: GLuint,
            num_groups_y: GLuint,
            num_groups_z: GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DispatchCompute({:?}, {:?}, {:?});",
                    num_groups_x,
                    num_groups_y,
                    num_groups_z
                );
            }
            let out = call_atomic_ptr_3arg(
                "glDispatchCompute",
                &self.glDispatchCompute_p,
                num_groups_x,
                num_groups_y,
                num_groups_z,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDispatchCompute");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DispatchCompute_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDispatchCompute\0",
                &self.glDispatchCompute_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DispatchCompute_is_loaded(&self) -> bool {
            !self.glDispatchCompute_p.load(RELAX).is_null()
        }
        /// [glDispatchComputeIndirect](http://docs.gl/es3/glDispatchComputeIndirect)(indirect)
        /// * `indirect` group: BufferOffset
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DispatchComputeIndirect(&self, indirect: GLintptr) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DispatchComputeIndirect({:?});", indirect);
            }
            let out = call_atomic_ptr_1arg(
                "glDispatchComputeIndirect",
                &self.glDispatchComputeIndirect_p,
                indirect,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDispatchComputeIndirect");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DispatchComputeIndirect_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDispatchComputeIndirect\0",
                &self.glDispatchComputeIndirect_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DispatchComputeIndirect_is_loaded(&self) -> bool {
            !self.glDispatchComputeIndirect_p.load(RELAX).is_null()
        }
        /// [glDrawArrays](http://docs.gl/es3/glDrawArrays)(mode, first, count)
        /// * `mode` group: PrimitiveType
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawArrays(&self, mode: GLenum, first: GLint, count: GLsizei) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawArrays({:#X}, {:?}, {:?});",
                    mode,
                    first,
                    count
                );
            }
            let out =
                call_atomic_ptr_3arg("glDrawArrays", &self.glDrawArrays_p, mode, first, count);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawArrays");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawArrays_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDrawArrays\0", &self.glDrawArrays_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawArrays_is_loaded(&self) -> bool {
            !self.glDrawArrays_p.load(RELAX).is_null()
        }
        /// [glDrawArraysIndirect](http://docs.gl/es3/glDrawArraysIndirect)(mode, indirect)
        /// * `mode` group: PrimitiveType
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawArraysIndirect(&self, mode: GLenum, indirect: *const c_void) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawArraysIndirect({:#X}, {:p});",
                    mode,
                    indirect
                );
            }
            let out = call_atomic_ptr_2arg(
                "glDrawArraysIndirect",
                &self.glDrawArraysIndirect_p,
                mode,
                indirect,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawArraysIndirect");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawArraysIndirect_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawArraysIndirect\0",
                &self.glDrawArraysIndirect_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawArraysIndirect_is_loaded(&self) -> bool {
            !self.glDrawArraysIndirect_p.load(RELAX).is_null()
        }
        /// [glDrawArraysInstanced](http://docs.gl/es3/glDrawArraysInstanced)(mode, first, count, instancecount)
        /// * `mode` group: PrimitiveType
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawArraysInstanced(
            &self,
            mode: GLenum,
            first: GLint,
            count: GLsizei,
            instancecount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawArraysInstanced({:#X}, {:?}, {:?}, {:?});",
                    mode,
                    first,
                    count,
                    instancecount
                );
            }
            let out = call_atomic_ptr_4arg(
                "glDrawArraysInstanced",
                &self.glDrawArraysInstanced_p,
                mode,
                first,
                count,
                instancecount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawArraysInstanced");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawArraysInstanced_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawArraysInstanced\0",
                &self.glDrawArraysInstanced_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawArraysInstanced_is_loaded(&self) -> bool {
            !self.glDrawArraysInstanced_p.load(RELAX).is_null()
        }
        /// [glDrawArraysInstancedANGLE](http://docs.gl/es3/glDrawArraysInstancedANGLE)(mode, first, count, primcount)
        /// * `mode` group: PrimitiveType
        /// * alias of: [`glDrawArraysInstanced`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
        pub unsafe fn DrawArraysInstancedANGLE(
            &self,
            mode: GLenum,
            first: GLint,
            count: GLsizei,
            primcount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawArraysInstancedANGLE({:#X}, {:?}, {:?}, {:?});",
                    mode,
                    first,
                    count,
                    primcount
                );
            }
            let out = call_atomic_ptr_4arg(
                "glDrawArraysInstancedANGLE",
                &self.glDrawArraysInstancedANGLE_p,
                mode,
                first,
                count,
                primcount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawArraysInstancedANGLE");
            }
            out
        }
        #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
        #[doc(hidden)]
        pub unsafe fn DrawArraysInstancedANGLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawArraysInstancedANGLE\0",
                &self.glDrawArraysInstancedANGLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
        pub fn DrawArraysInstancedANGLE_is_loaded(&self) -> bool {
            !self.glDrawArraysInstancedANGLE_p.load(RELAX).is_null()
        }
        /// [glDrawArraysInstancedBaseInstanceEXT](http://docs.gl/es3/glDrawArraysInstancedBaseInstanceEXT)(mode, first, count, instancecount, baseinstance)
        /// * `mode` group: PrimitiveType
        /// * alias of: [`glDrawArraysInstancedBaseInstance`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        pub unsafe fn DrawArraysInstancedBaseInstanceEXT(
            &self,
            mode: GLenum,
            first: GLint,
            count: GLsizei,
            instancecount: GLsizei,
            baseinstance: GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawArraysInstancedBaseInstanceEXT({:#X}, {:?}, {:?}, {:?}, {:?});",
                    mode,
                    first,
                    count,
                    instancecount,
                    baseinstance
                );
            }
            let out = call_atomic_ptr_5arg(
                "glDrawArraysInstancedBaseInstanceEXT",
                &self.glDrawArraysInstancedBaseInstanceEXT_p,
                mode,
                first,
                count,
                instancecount,
                baseinstance,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawArraysInstancedBaseInstanceEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        #[doc(hidden)]
        pub unsafe fn DrawArraysInstancedBaseInstanceEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawArraysInstancedBaseInstanceEXT\0",
                &self.glDrawArraysInstancedBaseInstanceEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        pub fn DrawArraysInstancedBaseInstanceEXT_is_loaded(&self) -> bool {
            !self
                .glDrawArraysInstancedBaseInstanceEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glDrawArraysInstancedEXT](http://docs.gl/es3/glDrawArraysInstancedEXT)(mode, start, count, primcount)
        /// * `mode` group: PrimitiveType
        /// * alias of: [`glDrawArraysInstanced`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_draw_instanced",
                feature = "GL_EXT_instanced_arrays"
            )))
        )]
        pub unsafe fn DrawArraysInstancedEXT(
            &self,
            mode: GLenum,
            start: GLint,
            count: GLsizei,
            primcount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawArraysInstancedEXT({:#X}, {:?}, {:?}, {:?});",
                    mode,
                    start,
                    count,
                    primcount
                );
            }
            let out = call_atomic_ptr_4arg(
                "glDrawArraysInstancedEXT",
                &self.glDrawArraysInstancedEXT_p,
                mode,
                start,
                count,
                primcount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawArraysInstancedEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_draw_instanced",
                feature = "GL_EXT_instanced_arrays"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn DrawArraysInstancedEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawArraysInstancedEXT\0",
                &self.glDrawArraysInstancedEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_draw_instanced",
                feature = "GL_EXT_instanced_arrays"
            )))
        )]
        pub fn DrawArraysInstancedEXT_is_loaded(&self) -> bool {
            !self.glDrawArraysInstancedEXT_p.load(RELAX).is_null()
        }
        /// [glDrawArraysInstancedNV](http://docs.gl/es3/glDrawArraysInstancedNV)(mode, first, count, primcount)
        /// * `mode` group: PrimitiveType
        /// * alias of: [`glDrawArraysInstanced`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_draw_instanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
        pub unsafe fn DrawArraysInstancedNV(
            &self,
            mode: GLenum,
            first: GLint,
            count: GLsizei,
            primcount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawArraysInstancedNV({:#X}, {:?}, {:?}, {:?});",
                    mode,
                    first,
                    count,
                    primcount
                );
            }
            let out = call_atomic_ptr_4arg(
                "glDrawArraysInstancedNV",
                &self.glDrawArraysInstancedNV_p,
                mode,
                first,
                count,
                primcount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawArraysInstancedNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_draw_instanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
        #[doc(hidden)]
        pub unsafe fn DrawArraysInstancedNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawArraysInstancedNV\0",
                &self.glDrawArraysInstancedNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_draw_instanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
        pub fn DrawArraysInstancedNV_is_loaded(&self) -> bool {
            !self.glDrawArraysInstancedNV_p.load(RELAX).is_null()
        }
        /// [glDrawBuffers](http://docs.gl/es3/glDrawBuffers)(n, bufs)
        /// * `bufs` group: DrawBufferMode
        /// * `bufs` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawBuffers(&self, n: GLsizei, bufs: *const GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DrawBuffers({:?}, {:p});", n, bufs);
            }
            let out = call_atomic_ptr_2arg("glDrawBuffers", &self.glDrawBuffers_p, n, bufs);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawBuffers");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawBuffers_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glDrawBuffers\0", &self.glDrawBuffers_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawBuffers_is_loaded(&self) -> bool {
            !self.glDrawBuffers_p.load(RELAX).is_null()
        }
        /// [glDrawBuffersEXT](http://docs.gl/es3/glDrawBuffersEXT)(n, bufs)
        /// * `bufs` len: n
        /// * alias of: [`glDrawBuffers`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers"))))]
        pub unsafe fn DrawBuffersEXT(&self, n: GLsizei, bufs: *const GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DrawBuffersEXT({:?}, {:p});", n, bufs);
            }
            let out = call_atomic_ptr_2arg("glDrawBuffersEXT", &self.glDrawBuffersEXT_p, n, bufs);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawBuffersEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers"))))]
        #[doc(hidden)]
        pub unsafe fn DrawBuffersEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawBuffersEXT\0",
                &self.glDrawBuffersEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers"))))]
        pub fn DrawBuffersEXT_is_loaded(&self) -> bool {
            !self.glDrawBuffersEXT_p.load(RELAX).is_null()
        }
        /// [glDrawBuffersIndexedEXT](http://docs.gl/es3/glDrawBuffersIndexedEXT)(n, location, indices)
        /// * `location` len: n
        /// * `indices` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
        pub unsafe fn DrawBuffersIndexedEXT(
            &self,
            n: GLint,
            location: *const GLenum,
            indices: *const GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawBuffersIndexedEXT({:?}, {:p}, {:p});",
                    n,
                    location,
                    indices
                );
            }
            let out = call_atomic_ptr_3arg(
                "glDrawBuffersIndexedEXT",
                &self.glDrawBuffersIndexedEXT_p,
                n,
                location,
                indices,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawBuffersIndexedEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
        #[doc(hidden)]
        pub unsafe fn DrawBuffersIndexedEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawBuffersIndexedEXT\0",
                &self.glDrawBuffersIndexedEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
        pub fn DrawBuffersIndexedEXT_is_loaded(&self) -> bool {
            !self.glDrawBuffersIndexedEXT_p.load(RELAX).is_null()
        }
        /// [glDrawBuffersNV](http://docs.gl/es3/glDrawBuffersNV)(n, bufs)
        /// * `bufs` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_buffers"))))]
        pub unsafe fn DrawBuffersNV(&self, n: GLsizei, bufs: *const GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DrawBuffersNV({:?}, {:p});", n, bufs);
            }
            let out = call_atomic_ptr_2arg("glDrawBuffersNV", &self.glDrawBuffersNV_p, n, bufs);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawBuffersNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_buffers"))))]
        #[doc(hidden)]
        pub unsafe fn DrawBuffersNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawBuffersNV\0",
                &self.glDrawBuffersNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_buffers"))))]
        pub fn DrawBuffersNV_is_loaded(&self) -> bool {
            !self.glDrawBuffersNV_p.load(RELAX).is_null()
        }
        /// [glDrawElements](http://docs.gl/es3/glDrawElements)(mode, count, type_, indices)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawElements(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElements({:#X}, {:?}, {:#X}, {:p});",
                    mode,
                    count,
                    type_,
                    indices
                );
            }
            let out = call_atomic_ptr_4arg(
                "glDrawElements",
                &self.glDrawElements_p,
                mode,
                count,
                type_,
                indices,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElements");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawElements_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElements\0",
                &self.glDrawElements_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawElements_is_loaded(&self) -> bool {
            !self.glDrawElements_p.load(RELAX).is_null()
        }
        /// [glDrawElementsBaseVertex](http://docs.gl/es3/glDrawElementsBaseVertex)(mode, count, type_, indices, basevertex)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawElementsBaseVertex(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            basevertex: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsBaseVertex({:#X}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    basevertex
                );
            }
            let out = call_atomic_ptr_5arg(
                "glDrawElementsBaseVertex",
                &self.glDrawElementsBaseVertex_p,
                mode,
                count,
                type_,
                indices,
                basevertex,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsBaseVertex");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawElementsBaseVertex_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsBaseVertex\0",
                &self.glDrawElementsBaseVertex_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawElementsBaseVertex_is_loaded(&self) -> bool {
            !self.glDrawElementsBaseVertex_p.load(RELAX).is_null()
        }
        /// [glDrawElementsBaseVertexEXT](http://docs.gl/es3/glDrawElementsBaseVertexEXT)(mode, count, type_, indices, basevertex)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        /// * alias of: [`glDrawElementsBaseVertex`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        pub unsafe fn DrawElementsBaseVertexEXT(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            basevertex: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsBaseVertexEXT({:#X}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    basevertex
                );
            }
            let out = call_atomic_ptr_5arg(
                "glDrawElementsBaseVertexEXT",
                &self.glDrawElementsBaseVertexEXT_p,
                mode,
                count,
                type_,
                indices,
                basevertex,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsBaseVertexEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        #[doc(hidden)]
        pub unsafe fn DrawElementsBaseVertexEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsBaseVertexEXT\0",
                &self.glDrawElementsBaseVertexEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        pub fn DrawElementsBaseVertexEXT_is_loaded(&self) -> bool {
            !self.glDrawElementsBaseVertexEXT_p.load(RELAX).is_null()
        }
        /// [glDrawElementsBaseVertexOES](http://docs.gl/es3/glDrawElementsBaseVertexOES)(mode, count, type_, indices, basevertex)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        /// * alias of: [`glDrawElementsBaseVertex`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        pub unsafe fn DrawElementsBaseVertexOES(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            basevertex: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsBaseVertexOES({:#X}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    basevertex
                );
            }
            let out = call_atomic_ptr_5arg(
                "glDrawElementsBaseVertexOES",
                &self.glDrawElementsBaseVertexOES_p,
                mode,
                count,
                type_,
                indices,
                basevertex,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsBaseVertexOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        #[doc(hidden)]
        pub unsafe fn DrawElementsBaseVertexOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsBaseVertexOES\0",
                &self.glDrawElementsBaseVertexOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        pub fn DrawElementsBaseVertexOES_is_loaded(&self) -> bool {
            !self.glDrawElementsBaseVertexOES_p.load(RELAX).is_null()
        }
        /// [glDrawElementsIndirect](http://docs.gl/es3/glDrawElementsIndirect)(mode, type_, indirect)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawElementsIndirect(
            &self,
            mode: GLenum,
            type_: GLenum,
            indirect: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsIndirect({:#X}, {:#X}, {:p});",
                    mode,
                    type_,
                    indirect
                );
            }
            let out = call_atomic_ptr_3arg(
                "glDrawElementsIndirect",
                &self.glDrawElementsIndirect_p,
                mode,
                type_,
                indirect,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsIndirect");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawElementsIndirect_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsIndirect\0",
                &self.glDrawElementsIndirect_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawElementsIndirect_is_loaded(&self) -> bool {
            !self.glDrawElementsIndirect_p.load(RELAX).is_null()
        }
        /// [glDrawElementsInstanced](http://docs.gl/es3/glDrawElementsInstanced)(mode, count, type_, indices, instancecount)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawElementsInstanced(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            instancecount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsInstanced({:#X}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    instancecount
                );
            }
            let out = call_atomic_ptr_5arg(
                "glDrawElementsInstanced",
                &self.glDrawElementsInstanced_p,
                mode,
                count,
                type_,
                indices,
                instancecount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsInstanced");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawElementsInstanced_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsInstanced\0",
                &self.glDrawElementsInstanced_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawElementsInstanced_is_loaded(&self) -> bool {
            !self.glDrawElementsInstanced_p.load(RELAX).is_null()
        }
        /// [glDrawElementsInstancedANGLE](http://docs.gl/es3/glDrawElementsInstancedANGLE)(mode, count, type_, indices, primcount)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: PrimitiveType
        /// * `indices` len: COMPSIZE(count,type)
        /// * alias of: [`glDrawElementsInstanced`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
        pub unsafe fn DrawElementsInstancedANGLE(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            primcount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsInstancedANGLE({:#X}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    primcount
                );
            }
            let out = call_atomic_ptr_5arg(
                "glDrawElementsInstancedANGLE",
                &self.glDrawElementsInstancedANGLE_p,
                mode,
                count,
                type_,
                indices,
                primcount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsInstancedANGLE");
            }
            out
        }
        #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
        #[doc(hidden)]
        pub unsafe fn DrawElementsInstancedANGLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsInstancedANGLE\0",
                &self.glDrawElementsInstancedANGLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
        pub fn DrawElementsInstancedANGLE_is_loaded(&self) -> bool {
            !self.glDrawElementsInstancedANGLE_p.load(RELAX).is_null()
        }
        /// [glDrawElementsInstancedBaseInstanceEXT](http://docs.gl/es3/glDrawElementsInstancedBaseInstanceEXT)(mode, count, type_, indices, instancecount, baseinstance)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: PrimitiveType
        /// * `indices` len: count
        /// * alias of: [`glDrawElementsInstancedBaseInstance`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        pub unsafe fn DrawElementsInstancedBaseInstanceEXT(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            instancecount: GLsizei,
            baseinstance: GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsInstancedBaseInstanceEXT({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    instancecount,
                    baseinstance
                );
            }
            let out = call_atomic_ptr_6arg(
                "glDrawElementsInstancedBaseInstanceEXT",
                &self.glDrawElementsInstancedBaseInstanceEXT_p,
                mode,
                count,
                type_,
                indices,
                instancecount,
                baseinstance,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsInstancedBaseInstanceEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        #[doc(hidden)]
        pub unsafe fn DrawElementsInstancedBaseInstanceEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsInstancedBaseInstanceEXT\0",
                &self.glDrawElementsInstancedBaseInstanceEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        pub fn DrawElementsInstancedBaseInstanceEXT_is_loaded(&self) -> bool {
            !self
                .glDrawElementsInstancedBaseInstanceEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glDrawElementsInstancedBaseVertex](http://docs.gl/es3/glDrawElementsInstancedBaseVertex)(mode, count, type_, indices, instancecount, basevertex)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawElementsInstancedBaseVertex(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            instancecount: GLsizei,
            basevertex: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsInstancedBaseVertex({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    instancecount,
                    basevertex
                );
            }
            let out = call_atomic_ptr_6arg(
                "glDrawElementsInstancedBaseVertex",
                &self.glDrawElementsInstancedBaseVertex_p,
                mode,
                count,
                type_,
                indices,
                instancecount,
                basevertex,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsInstancedBaseVertex");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawElementsInstancedBaseVertex_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsInstancedBaseVertex\0",
                &self.glDrawElementsInstancedBaseVertex_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawElementsInstancedBaseVertex_is_loaded(&self) -> bool {
            !self
                .glDrawElementsInstancedBaseVertex_p
                .load(RELAX)
                .is_null()
        }
        /// [glDrawElementsInstancedBaseVertexBaseInstanceEXT](http://docs.gl/es3/glDrawElementsInstancedBaseVertexBaseInstanceEXT)(mode, count, type_, indices, instancecount, basevertex, baseinstance)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: count
        /// * alias of: [`glDrawElementsInstancedBaseVertexBaseInstance`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        pub unsafe fn DrawElementsInstancedBaseVertexBaseInstanceEXT(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            instancecount: GLsizei,
            basevertex: GLint,
            baseinstance: GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DrawElementsInstancedBaseVertexBaseInstanceEXT({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?}, {:?});", mode, count, type_, indices, instancecount, basevertex, baseinstance);
            }
            let out = call_atomic_ptr_7arg(
                "glDrawElementsInstancedBaseVertexBaseInstanceEXT",
                &self.glDrawElementsInstancedBaseVertexBaseInstanceEXT_p,
                mode,
                count,
                type_,
                indices,
                instancecount,
                basevertex,
                baseinstance,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsInstancedBaseVertexBaseInstanceEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        #[doc(hidden)]
        pub unsafe fn DrawElementsInstancedBaseVertexBaseInstanceEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsInstancedBaseVertexBaseInstanceEXT\0",
                &self.glDrawElementsInstancedBaseVertexBaseInstanceEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        pub fn DrawElementsInstancedBaseVertexBaseInstanceEXT_is_loaded(&self) -> bool {
            !self
                .glDrawElementsInstancedBaseVertexBaseInstanceEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glDrawElementsInstancedBaseVertexEXT](http://docs.gl/es3/glDrawElementsInstancedBaseVertexEXT)(mode, count, type_, indices, instancecount, basevertex)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        /// * alias of: [`glDrawElementsInstancedBaseVertex`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        pub unsafe fn DrawElementsInstancedBaseVertexEXT(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            instancecount: GLsizei,
            basevertex: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsInstancedBaseVertexEXT({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    instancecount,
                    basevertex
                );
            }
            let out = call_atomic_ptr_6arg(
                "glDrawElementsInstancedBaseVertexEXT",
                &self.glDrawElementsInstancedBaseVertexEXT_p,
                mode,
                count,
                type_,
                indices,
                instancecount,
                basevertex,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsInstancedBaseVertexEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        #[doc(hidden)]
        pub unsafe fn DrawElementsInstancedBaseVertexEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsInstancedBaseVertexEXT\0",
                &self.glDrawElementsInstancedBaseVertexEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        pub fn DrawElementsInstancedBaseVertexEXT_is_loaded(&self) -> bool {
            !self
                .glDrawElementsInstancedBaseVertexEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glDrawElementsInstancedBaseVertexOES](http://docs.gl/es3/glDrawElementsInstancedBaseVertexOES)(mode, count, type_, indices, instancecount, basevertex)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        /// * alias of: [`glDrawElementsInstancedBaseVertex`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        pub unsafe fn DrawElementsInstancedBaseVertexOES(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            instancecount: GLsizei,
            basevertex: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsInstancedBaseVertexOES({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    instancecount,
                    basevertex
                );
            }
            let out = call_atomic_ptr_6arg(
                "glDrawElementsInstancedBaseVertexOES",
                &self.glDrawElementsInstancedBaseVertexOES_p,
                mode,
                count,
                type_,
                indices,
                instancecount,
                basevertex,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsInstancedBaseVertexOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        #[doc(hidden)]
        pub unsafe fn DrawElementsInstancedBaseVertexOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsInstancedBaseVertexOES\0",
                &self.glDrawElementsInstancedBaseVertexOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        pub fn DrawElementsInstancedBaseVertexOES_is_loaded(&self) -> bool {
            !self
                .glDrawElementsInstancedBaseVertexOES_p
                .load(RELAX)
                .is_null()
        }
        /// [glDrawElementsInstancedEXT](http://docs.gl/es3/glDrawElementsInstancedEXT)(mode, count, type_, indices, primcount)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        /// * alias of: [`glDrawElementsInstanced`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_draw_instanced",
                feature = "GL_EXT_instanced_arrays"
            )))
        )]
        pub unsafe fn DrawElementsInstancedEXT(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            primcount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsInstancedEXT({:#X}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    primcount
                );
            }
            let out = call_atomic_ptr_5arg(
                "glDrawElementsInstancedEXT",
                &self.glDrawElementsInstancedEXT_p,
                mode,
                count,
                type_,
                indices,
                primcount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsInstancedEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_draw_instanced",
                feature = "GL_EXT_instanced_arrays"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn DrawElementsInstancedEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsInstancedEXT\0",
                &self.glDrawElementsInstancedEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_draw_instanced",
                feature = "GL_EXT_instanced_arrays"
            )))
        )]
        pub fn DrawElementsInstancedEXT_is_loaded(&self) -> bool {
            !self.glDrawElementsInstancedEXT_p.load(RELAX).is_null()
        }
        /// [glDrawElementsInstancedNV](http://docs.gl/es3/glDrawElementsInstancedNV)(mode, count, type_, indices, primcount)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: PrimitiveType
        /// * `indices` len: COMPSIZE(count,type)
        /// * alias of: [`glDrawElementsInstanced`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_draw_instanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
        pub unsafe fn DrawElementsInstancedNV(
            &self,
            mode: GLenum,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            primcount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawElementsInstancedNV({:#X}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    count,
                    type_,
                    indices,
                    primcount
                );
            }
            let out = call_atomic_ptr_5arg(
                "glDrawElementsInstancedNV",
                &self.glDrawElementsInstancedNV_p,
                mode,
                count,
                type_,
                indices,
                primcount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawElementsInstancedNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_draw_instanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
        #[doc(hidden)]
        pub unsafe fn DrawElementsInstancedNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawElementsInstancedNV\0",
                &self.glDrawElementsInstancedNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_draw_instanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
        pub fn DrawElementsInstancedNV_is_loaded(&self) -> bool {
            !self.glDrawElementsInstancedNV_p.load(RELAX).is_null()
        }
        /// [glDrawMeshTasksIndirectNV](http://docs.gl/es3/glDrawMeshTasksIndirectNV)(indirect)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        pub unsafe fn DrawMeshTasksIndirectNV(&self, indirect: GLintptr) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DrawMeshTasksIndirectNV({:?});", indirect);
            }
            let out = call_atomic_ptr_1arg(
                "glDrawMeshTasksIndirectNV",
                &self.glDrawMeshTasksIndirectNV_p,
                indirect,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawMeshTasksIndirectNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        #[doc(hidden)]
        pub unsafe fn DrawMeshTasksIndirectNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawMeshTasksIndirectNV\0",
                &self.glDrawMeshTasksIndirectNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        pub fn DrawMeshTasksIndirectNV_is_loaded(&self) -> bool {
            !self.glDrawMeshTasksIndirectNV_p.load(RELAX).is_null()
        }
        /// [glDrawMeshTasksNV](http://docs.gl/es3/glDrawMeshTasksNV)(first, count)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        pub unsafe fn DrawMeshTasksNV(&self, first: GLuint, count: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.DrawMeshTasksNV({:?}, {:?});", first, count);
            }
            let out =
                call_atomic_ptr_2arg("glDrawMeshTasksNV", &self.glDrawMeshTasksNV_p, first, count);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawMeshTasksNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        #[doc(hidden)]
        pub unsafe fn DrawMeshTasksNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawMeshTasksNV\0",
                &self.glDrawMeshTasksNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        pub fn DrawMeshTasksNV_is_loaded(&self) -> bool {
            !self.glDrawMeshTasksNV_p.load(RELAX).is_null()
        }
        /// [glDrawRangeElements](http://docs.gl/es3/glDrawRangeElements)(mode, start, end, count, type_, indices)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawRangeElements(
            &self,
            mode: GLenum,
            start: GLuint,
            end: GLuint,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawRangeElements({:#X}, {:?}, {:?}, {:?}, {:#X}, {:p});",
                    mode,
                    start,
                    end,
                    count,
                    type_,
                    indices
                );
            }
            let out = call_atomic_ptr_6arg(
                "glDrawRangeElements",
                &self.glDrawRangeElements_p,
                mode,
                start,
                end,
                count,
                type_,
                indices,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawRangeElements");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawRangeElements_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawRangeElements\0",
                &self.glDrawRangeElements_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawRangeElements_is_loaded(&self) -> bool {
            !self.glDrawRangeElements_p.load(RELAX).is_null()
        }
        /// [glDrawRangeElementsBaseVertex](http://docs.gl/es3/glDrawRangeElementsBaseVertex)(mode, start, end, count, type_, indices, basevertex)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn DrawRangeElementsBaseVertex(
            &self,
            mode: GLenum,
            start: GLuint,
            end: GLuint,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            basevertex: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawRangeElementsBaseVertex({:#X}, {:?}, {:?}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    start,
                    end,
                    count,
                    type_,
                    indices,
                    basevertex
                );
            }
            let out = call_atomic_ptr_7arg(
                "glDrawRangeElementsBaseVertex",
                &self.glDrawRangeElementsBaseVertex_p,
                mode,
                start,
                end,
                count,
                type_,
                indices,
                basevertex,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawRangeElementsBaseVertex");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn DrawRangeElementsBaseVertex_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawRangeElementsBaseVertex\0",
                &self.glDrawRangeElementsBaseVertex_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn DrawRangeElementsBaseVertex_is_loaded(&self) -> bool {
            !self.glDrawRangeElementsBaseVertex_p.load(RELAX).is_null()
        }
        /// [glDrawRangeElementsBaseVertexEXT](http://docs.gl/es3/glDrawRangeElementsBaseVertexEXT)(mode, start, end, count, type_, indices, basevertex)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        /// * alias of: [`glDrawRangeElementsBaseVertex`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        pub unsafe fn DrawRangeElementsBaseVertexEXT(
            &self,
            mode: GLenum,
            start: GLuint,
            end: GLuint,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            basevertex: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawRangeElementsBaseVertexEXT({:#X}, {:?}, {:?}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    start,
                    end,
                    count,
                    type_,
                    indices,
                    basevertex
                );
            }
            let out = call_atomic_ptr_7arg(
                "glDrawRangeElementsBaseVertexEXT",
                &self.glDrawRangeElementsBaseVertexEXT_p,
                mode,
                start,
                end,
                count,
                type_,
                indices,
                basevertex,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawRangeElementsBaseVertexEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        #[doc(hidden)]
        pub unsafe fn DrawRangeElementsBaseVertexEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawRangeElementsBaseVertexEXT\0",
                &self.glDrawRangeElementsBaseVertexEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        pub fn DrawRangeElementsBaseVertexEXT_is_loaded(&self) -> bool {
            !self
                .glDrawRangeElementsBaseVertexEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glDrawRangeElementsBaseVertexOES](http://docs.gl/es3/glDrawRangeElementsBaseVertexOES)(mode, start, end, count, type_, indices, basevertex)
        /// * `mode` group: PrimitiveType
        /// * `type_` group: DrawElementsType
        /// * `indices` len: COMPSIZE(count,type)
        /// * alias of: [`glDrawRangeElementsBaseVertex`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        pub unsafe fn DrawRangeElementsBaseVertexOES(
            &self,
            mode: GLenum,
            start: GLuint,
            end: GLuint,
            count: GLsizei,
            type_: GLenum,
            indices: *const c_void,
            basevertex: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawRangeElementsBaseVertexOES({:#X}, {:?}, {:?}, {:?}, {:#X}, {:p}, {:?});",
                    mode,
                    start,
                    end,
                    count,
                    type_,
                    indices,
                    basevertex
                );
            }
            let out = call_atomic_ptr_7arg(
                "glDrawRangeElementsBaseVertexOES",
                &self.glDrawRangeElementsBaseVertexOES_p,
                mode,
                start,
                end,
                count,
                type_,
                indices,
                basevertex,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawRangeElementsBaseVertexOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        #[doc(hidden)]
        pub unsafe fn DrawRangeElementsBaseVertexOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawRangeElementsBaseVertexOES\0",
                &self.glDrawRangeElementsBaseVertexOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        pub fn DrawRangeElementsBaseVertexOES_is_loaded(&self) -> bool {
            !self
                .glDrawRangeElementsBaseVertexOES_p
                .load(RELAX)
                .is_null()
        }
        /// [glDrawTransformFeedbackEXT](http://docs.gl/es3/glDrawTransformFeedbackEXT)(mode, id)
        /// * `mode` group: PrimitiveType
        /// * `id` class: transform feedback
        /// * alias of: [`glDrawTransformFeedback`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
        pub unsafe fn DrawTransformFeedbackEXT(&self, mode: GLenum, id: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawTransformFeedbackEXT({:#X}, {:?});",
                    mode,
                    id
                );
            }
            let out = call_atomic_ptr_2arg(
                "glDrawTransformFeedbackEXT",
                &self.glDrawTransformFeedbackEXT_p,
                mode,
                id,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawTransformFeedbackEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
        #[doc(hidden)]
        pub unsafe fn DrawTransformFeedbackEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawTransformFeedbackEXT\0",
                &self.glDrawTransformFeedbackEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
        pub fn DrawTransformFeedbackEXT_is_loaded(&self) -> bool {
            !self.glDrawTransformFeedbackEXT_p.load(RELAX).is_null()
        }
        /// [glDrawTransformFeedbackInstancedEXT](http://docs.gl/es3/glDrawTransformFeedbackInstancedEXT)(mode, id, instancecount)
        /// * `mode` group: PrimitiveType
        /// * `id` class: transform feedback
        /// * alias of: [`glDrawTransformFeedbackInstanced`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
        pub unsafe fn DrawTransformFeedbackInstancedEXT(
            &self,
            mode: GLenum,
            id: GLuint,
            instancecount: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawTransformFeedbackInstancedEXT({:#X}, {:?}, {:?});",
                    mode,
                    id,
                    instancecount
                );
            }
            let out = call_atomic_ptr_3arg(
                "glDrawTransformFeedbackInstancedEXT",
                &self.glDrawTransformFeedbackInstancedEXT_p,
                mode,
                id,
                instancecount,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawTransformFeedbackInstancedEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
        #[doc(hidden)]
        pub unsafe fn DrawTransformFeedbackInstancedEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawTransformFeedbackInstancedEXT\0",
                &self.glDrawTransformFeedbackInstancedEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
        pub fn DrawTransformFeedbackInstancedEXT_is_loaded(&self) -> bool {
            !self
                .glDrawTransformFeedbackInstancedEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glDrawVkImageNV](http://docs.gl/es3/glDrawVkImageNV)(vkImage, sampler, x0, y0, x1, y1, z, s0, t0, s1, t1)
        /// * `sampler` class: sampler
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        pub unsafe fn DrawVkImageNV(
            &self,
            vkImage: GLuint64,
            sampler: GLuint,
            x0: GLfloat,
            y0: GLfloat,
            x1: GLfloat,
            y1: GLfloat,
            z: GLfloat,
            s0: GLfloat,
            t0: GLfloat,
            s1: GLfloat,
            t1: GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.DrawVkImageNV({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                    vkImage,
                    sampler,
                    x0,
                    y0,
                    x1,
                    y1,
                    z,
                    s0,
                    t0,
                    s1,
                    t1
                );
            }
            let out = call_atomic_ptr_11arg(
                "glDrawVkImageNV",
                &self.glDrawVkImageNV_p,
                vkImage,
                sampler,
                x0,
                y0,
                x1,
                y1,
                z,
                s0,
                t0,
                s1,
                t1,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glDrawVkImageNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        #[doc(hidden)]
        pub unsafe fn DrawVkImageNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glDrawVkImageNV\0",
                &self.glDrawVkImageNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        pub fn DrawVkImageNV_is_loaded(&self) -> bool {
            !self.glDrawVkImageNV_p.load(RELAX).is_null()
        }
        /// [glEGLImageTargetRenderbufferStorageOES](http://docs.gl/es3/glEGLImageTargetRenderbufferStorageOES)(target, image)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_EGL_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
        pub unsafe fn EGLImageTargetRenderbufferStorageOES(
            &self,
            target: GLenum,
            image: GLeglImageOES,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.EGLImageTargetRenderbufferStorageOES({:#X}, {:?});",
                    target,
                    image
                );
            }
            let out = call_atomic_ptr_2arg(
                "glEGLImageTargetRenderbufferStorageOES",
                &self.glEGLImageTargetRenderbufferStorageOES_p,
                target,
                image,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEGLImageTargetRenderbufferStorageOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_EGL_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
        #[doc(hidden)]
        pub unsafe fn EGLImageTargetRenderbufferStorageOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEGLImageTargetRenderbufferStorageOES\0",
                &self.glEGLImageTargetRenderbufferStorageOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_EGL_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
        pub fn EGLImageTargetRenderbufferStorageOES_is_loaded(&self) -> bool {
            !self
                .glEGLImageTargetRenderbufferStorageOES_p
                .load(RELAX)
                .is_null()
        }
        /// [glEGLImageTargetTexStorageEXT](http://docs.gl/es3/glEGLImageTargetTexStorageEXT)(target, image, attrib_list)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
        pub unsafe fn EGLImageTargetTexStorageEXT(
            &self,
            target: GLenum,
            image: GLeglImageOES,
            attrib_list: *const GLin,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.EGLImageTargetTexStorageEXT({:#X}, {:?}, {:p});",
                    target,
                    image,
                    attrib_list
                );
            }
            let out = call_atomic_ptr_3arg(
                "glEGLImageTargetTexStorageEXT",
                &self.glEGLImageTargetTexStorageEXT_p,
                target,
                image,
                attrib_list,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEGLImageTargetTexStorageEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
        #[doc(hidden)]
        pub unsafe fn EGLImageTargetTexStorageEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEGLImageTargetTexStorageEXT\0",
                &self.glEGLImageTargetTexStorageEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
        pub fn EGLImageTargetTexStorageEXT_is_loaded(&self) -> bool {
            !self.glEGLImageTargetTexStorageEXT_p.load(RELAX).is_null()
        }
        /// [glEGLImageTargetTexture2DOES](http://docs.gl/es3/glEGLImageTargetTexture2DOES)(target, image)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_EGL_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
        pub unsafe fn EGLImageTargetTexture2DOES(&self, target: GLenum, image: GLeglImageOES) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.EGLImageTargetTexture2DOES({:#X}, {:?});",
                    target,
                    image
                );
            }
            let out = call_atomic_ptr_2arg(
                "glEGLImageTargetTexture2DOES",
                &self.glEGLImageTargetTexture2DOES_p,
                target,
                image,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEGLImageTargetTexture2DOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_EGL_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
        #[doc(hidden)]
        pub unsafe fn EGLImageTargetTexture2DOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEGLImageTargetTexture2DOES\0",
                &self.glEGLImageTargetTexture2DOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_EGL_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
        pub fn EGLImageTargetTexture2DOES_is_loaded(&self) -> bool {
            !self.glEGLImageTargetTexture2DOES_p.load(RELAX).is_null()
        }
        /// [glEGLImageTargetTextureStorageEXT](http://docs.gl/es3/glEGLImageTargetTextureStorageEXT)(texture, image, attrib_list)
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
        pub unsafe fn EGLImageTargetTextureStorageEXT(
            &self,
            texture: GLuint,
            image: GLeglImageOES,
            attrib_list: *const GLin,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.EGLImageTargetTextureStorageEXT({:?}, {:?}, {:p});",
                    texture,
                    image,
                    attrib_list
                );
            }
            let out = call_atomic_ptr_3arg(
                "glEGLImageTargetTextureStorageEXT",
                &self.glEGLImageTargetTextureStorageEXT_p,
                texture,
                image,
                attrib_list,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEGLImageTargetTextureStorageEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
        #[doc(hidden)]
        pub unsafe fn EGLImageTargetTextureStorageEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEGLImageTargetTextureStorageEXT\0",
                &self.glEGLImageTargetTextureStorageEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
        pub fn EGLImageTargetTextureStorageEXT_is_loaded(&self) -> bool {
            !self
                .glEGLImageTargetTextureStorageEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glEnable](http://docs.gl/es3/glEnable)(cap)
        /// * `cap` group: EnableCap
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn Enable(&self, cap: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.Enable({:#X});", cap);
            }
            let out = call_atomic_ptr_1arg("glEnable", &self.glEnable_p, cap);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEnable");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn Enable_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glEnable\0", &self.glEnable_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn Enable_is_loaded(&self) -> bool {
            !self.glEnable_p.load(RELAX).is_null()
        }
        /// [glEnableDriverControlQCOM](http://docs.gl/es3/glEnableDriverControlQCOM)(driverControl)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        pub unsafe fn EnableDriverControlQCOM(&self, driverControl: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EnableDriverControlQCOM({:?});", driverControl);
            }
            let out = call_atomic_ptr_1arg(
                "glEnableDriverControlQCOM",
                &self.glEnableDriverControlQCOM_p,
                driverControl,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEnableDriverControlQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        #[doc(hidden)]
        pub unsafe fn EnableDriverControlQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEnableDriverControlQCOM\0",
                &self.glEnableDriverControlQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        pub fn EnableDriverControlQCOM_is_loaded(&self) -> bool {
            !self.glEnableDriverControlQCOM_p.load(RELAX).is_null()
        }
        /// [glEnableVertexAttribArray](http://docs.gl/es3/glEnableVertexAttribArray)(index)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn EnableVertexAttribArray(&self, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EnableVertexAttribArray({:?});", index);
            }
            let out = call_atomic_ptr_1arg(
                "glEnableVertexAttribArray",
                &self.glEnableVertexAttribArray_p,
                index,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEnableVertexAttribArray");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn EnableVertexAttribArray_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEnableVertexAttribArray\0",
                &self.glEnableVertexAttribArray_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn EnableVertexAttribArray_is_loaded(&self) -> bool {
            !self.glEnableVertexAttribArray_p.load(RELAX).is_null()
        }
        /// [glEnablei](http://docs.gl/es3/glEnable)(target, index)
        /// * `target` group: EnableCap
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn Enablei(&self, target: GLenum, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.Enablei({:#X}, {:?});", target, index);
            }
            let out = call_atomic_ptr_2arg("glEnablei", &self.glEnablei_p, target, index);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEnablei");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn Enablei_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glEnablei\0", &self.glEnablei_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn Enablei_is_loaded(&self) -> bool {
            !self.glEnablei_p.load(RELAX).is_null()
        }
        /// [glEnableiEXT](http://docs.gl/es3/glEnableiEXT)(target, index)
        /// * `target` group: EnableCap
        /// * alias of: [`glEnablei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub unsafe fn EnableiEXT(&self, target: GLenum, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EnableiEXT({:#X}, {:?});", target, index);
            }
            let out = call_atomic_ptr_2arg("glEnableiEXT", &self.glEnableiEXT_p, target, index);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEnableiEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        #[doc(hidden)]
        pub unsafe fn EnableiEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glEnableiEXT\0", &self.glEnableiEXT_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        pub fn EnableiEXT_is_loaded(&self) -> bool {
            !self.glEnableiEXT_p.load(RELAX).is_null()
        }
        /// [glEnableiNV](http://docs.gl/es3/glEnableiNV)(target, index)
        /// * `target` group: EnableCap
        /// * alias of: [`glEnablei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub unsafe fn EnableiNV(&self, target: GLenum, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EnableiNV({:#X}, {:?});", target, index);
            }
            let out = call_atomic_ptr_2arg("glEnableiNV", &self.glEnableiNV_p, target, index);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEnableiNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        #[doc(hidden)]
        pub unsafe fn EnableiNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glEnableiNV\0", &self.glEnableiNV_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub fn EnableiNV_is_loaded(&self) -> bool {
            !self.glEnableiNV_p.load(RELAX).is_null()
        }
        /// [glEnableiOES](http://docs.gl/es3/glEnableiOES)(target, index)
        /// * `target` group: EnableCap
        /// * alias of: [`glEnablei`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            )))
        )]
        pub unsafe fn EnableiOES(&self, target: GLenum, index: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EnableiOES({:#X}, {:?});", target, index);
            }
            let out = call_atomic_ptr_2arg("glEnableiOES", &self.glEnableiOES_p, target, index);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEnableiOES");
            }
            out
        }
        #[cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn EnableiOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glEnableiOES\0", &self.glEnableiOES_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            )))
        )]
        pub fn EnableiOES_is_loaded(&self) -> bool {
            !self.glEnableiOES_p.load(RELAX).is_null()
        }
        /// [glEndConditionalRenderNV](http://docs.gl/es3/glEndConditionalRenderNV)()
        /// * alias of: [`glEndConditionalRender`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_conditional_render"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
        pub unsafe fn EndConditionalRenderNV(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EndConditionalRenderNV();",);
            }
            let out =
                call_atomic_ptr_0arg("glEndConditionalRenderNV", &self.glEndConditionalRenderNV_p);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEndConditionalRenderNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_conditional_render"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
        #[doc(hidden)]
        pub unsafe fn EndConditionalRenderNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEndConditionalRenderNV\0",
                &self.glEndConditionalRenderNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_conditional_render"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
        pub fn EndConditionalRenderNV_is_loaded(&self) -> bool {
            !self.glEndConditionalRenderNV_p.load(RELAX).is_null()
        }
        /// [glEndPerfMonitorAMD](http://docs.gl/es3/glEndPerfMonitorAMD)(monitor)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn EndPerfMonitorAMD(&self, monitor: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EndPerfMonitorAMD({:?});", monitor);
            }
            let out =
                call_atomic_ptr_1arg("glEndPerfMonitorAMD", &self.glEndPerfMonitorAMD_p, monitor);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEndPerfMonitorAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn EndPerfMonitorAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEndPerfMonitorAMD\0",
                &self.glEndPerfMonitorAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn EndPerfMonitorAMD_is_loaded(&self) -> bool {
            !self.glEndPerfMonitorAMD_p.load(RELAX).is_null()
        }
        /// [glEndPerfQueryINTEL](http://docs.gl/es3/glEndPerfQueryINTEL)(queryHandle)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn EndPerfQueryINTEL(&self, queryHandle: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EndPerfQueryINTEL({:?});", queryHandle);
            }
            let out = call_atomic_ptr_1arg(
                "glEndPerfQueryINTEL",
                &self.glEndPerfQueryINTEL_p,
                queryHandle,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEndPerfQueryINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn EndPerfQueryINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEndPerfQueryINTEL\0",
                &self.glEndPerfQueryINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn EndPerfQueryINTEL_is_loaded(&self) -> bool {
            !self.glEndPerfQueryINTEL_p.load(RELAX).is_null()
        }
        /// [glEndQuery](http://docs.gl/es3/glEndQuery)(target)
        /// * `target` group: QueryTarget
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn EndQuery(&self, target: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EndQuery({:#X});", target);
            }
            let out = call_atomic_ptr_1arg("glEndQuery", &self.glEndQuery_p, target);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEndQuery");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn EndQuery_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glEndQuery\0", &self.glEndQuery_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn EndQuery_is_loaded(&self) -> bool {
            !self.glEndQuery_p.load(RELAX).is_null()
        }
        /// [glEndQueryEXT](http://docs.gl/es3/glEndQueryEXT)(target)
        /// * `target` group: QueryTarget
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub unsafe fn EndQueryEXT(&self, target: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EndQueryEXT({:#X});", target);
            }
            let out = call_atomic_ptr_1arg("glEndQueryEXT", &self.glEndQueryEXT_p, target);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEndQueryEXT");
            }
            out
        }
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn EndQueryEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glEndQueryEXT\0", &self.glEndQueryEXT_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub fn EndQueryEXT_is_loaded(&self) -> bool {
            !self.glEndQueryEXT_p.load(RELAX).is_null()
        }
        /// [glEndTilingQCOM](http://docs.gl/es3/glEndTilingQCOM)(preserveMask)
        /// * `preserveMask` group: BufferBitQCOM
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
        pub unsafe fn EndTilingQCOM(&self, preserveMask: GLbitfield) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EndTilingQCOM({:?});", preserveMask);
            }
            let out =
                call_atomic_ptr_1arg("glEndTilingQCOM", &self.glEndTilingQCOM_p, preserveMask);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEndTilingQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn EndTilingQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEndTilingQCOM\0",
                &self.glEndTilingQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
        pub fn EndTilingQCOM_is_loaded(&self) -> bool {
            !self.glEndTilingQCOM_p.load(RELAX).is_null()
        }
        /// [glEndTransformFeedback](http://docs.gl/es3/glEndTransformFeedback)()
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn EndTransformFeedback(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.EndTransformFeedback();",);
            }
            let out =
                call_atomic_ptr_0arg("glEndTransformFeedback", &self.glEndTransformFeedback_p);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glEndTransformFeedback");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn EndTransformFeedback_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glEndTransformFeedback\0",
                &self.glEndTransformFeedback_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn EndTransformFeedback_is_loaded(&self) -> bool {
            !self.glEndTransformFeedback_p.load(RELAX).is_null()
        }
        /// [glExtGetBufferPointervQCOM](http://docs.gl/es3/glExtGetBufferPointervQCOM)(target, params)
        /// * `params` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub unsafe fn ExtGetBufferPointervQCOM(&self, target: GLenum, params: *mut *mut c_void) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtGetBufferPointervQCOM({:#X}, {:p});",
                    target,
                    params
                );
            }
            let out = call_atomic_ptr_2arg(
                "glExtGetBufferPointervQCOM",
                &self.glExtGetBufferPointervQCOM_p,
                target,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetBufferPointervQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetBufferPointervQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetBufferPointervQCOM\0",
                &self.glExtGetBufferPointervQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub fn ExtGetBufferPointervQCOM_is_loaded(&self) -> bool {
            !self.glExtGetBufferPointervQCOM_p.load(RELAX).is_null()
        }
        /// [glExtGetBuffersQCOM](http://docs.gl/es3/glExtGetBuffersQCOM)(buffers, maxBuffers, numBuffers)
        /// * `buffers` len: maxBuffers
        /// * `buffers` class: buffer
        /// * `numBuffers` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub unsafe fn ExtGetBuffersQCOM(
            &self,
            buffers: *mut GLuint,
            maxBuffers: GLint,
            numBuffers: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtGetBuffersQCOM({:p}, {:?}, {:p});",
                    buffers,
                    maxBuffers,
                    numBuffers
                );
            }
            let out = call_atomic_ptr_3arg(
                "glExtGetBuffersQCOM",
                &self.glExtGetBuffersQCOM_p,
                buffers,
                maxBuffers,
                numBuffers,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetBuffersQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetBuffersQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetBuffersQCOM\0",
                &self.glExtGetBuffersQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub fn ExtGetBuffersQCOM_is_loaded(&self) -> bool {
            !self.glExtGetBuffersQCOM_p.load(RELAX).is_null()
        }
        /// [glExtGetFramebuffersQCOM](http://docs.gl/es3/glExtGetFramebuffersQCOM)(framebuffers, maxFramebuffers, numFramebuffers)
        /// * `framebuffers` len: maxFramebuffers
        /// * `framebuffers` class: framebuffer
        /// * `numFramebuffers` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub unsafe fn ExtGetFramebuffersQCOM(
            &self,
            framebuffers: *mut GLuint,
            maxFramebuffers: GLint,
            numFramebuffers: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtGetFramebuffersQCOM({:p}, {:?}, {:p});",
                    framebuffers,
                    maxFramebuffers,
                    numFramebuffers
                );
            }
            let out = call_atomic_ptr_3arg(
                "glExtGetFramebuffersQCOM",
                &self.glExtGetFramebuffersQCOM_p,
                framebuffers,
                maxFramebuffers,
                numFramebuffers,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetFramebuffersQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetFramebuffersQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetFramebuffersQCOM\0",
                &self.glExtGetFramebuffersQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub fn ExtGetFramebuffersQCOM_is_loaded(&self) -> bool {
            !self.glExtGetFramebuffersQCOM_p.load(RELAX).is_null()
        }
        /// [glExtGetProgramBinarySourceQCOM](http://docs.gl/es3/glExtGetProgramBinarySourceQCOM)(program, shadertype, source, length)
        /// * `program` class: program
        /// * `shadertype` group: ShaderType
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        pub unsafe fn ExtGetProgramBinarySourceQCOM(
            &self,
            program: GLuint,
            shadertype: GLenum,
            source: *mut GLchar,
            length: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtGetProgramBinarySourceQCOM({:?}, {:#X}, {:p}, {:p});",
                    program,
                    shadertype,
                    source,
                    length
                );
            }
            let out = call_atomic_ptr_4arg(
                "glExtGetProgramBinarySourceQCOM",
                &self.glExtGetProgramBinarySourceQCOM_p,
                program,
                shadertype,
                source,
                length,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetProgramBinarySourceQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetProgramBinarySourceQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetProgramBinarySourceQCOM\0",
                &self.glExtGetProgramBinarySourceQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        pub fn ExtGetProgramBinarySourceQCOM_is_loaded(&self) -> bool {
            !self.glExtGetProgramBinarySourceQCOM_p.load(RELAX).is_null()
        }
        /// [glExtGetProgramsQCOM](http://docs.gl/es3/glExtGetProgramsQCOM)(programs, maxPrograms, numPrograms)
        /// * `programs` len: maxPrograms
        /// * `programs` class: program
        /// * `numPrograms` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        pub unsafe fn ExtGetProgramsQCOM(
            &self,
            programs: *mut GLuint,
            maxPrograms: GLint,
            numPrograms: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtGetProgramsQCOM({:p}, {:?}, {:p});",
                    programs,
                    maxPrograms,
                    numPrograms
                );
            }
            let out = call_atomic_ptr_3arg(
                "glExtGetProgramsQCOM",
                &self.glExtGetProgramsQCOM_p,
                programs,
                maxPrograms,
                numPrograms,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetProgramsQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetProgramsQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetProgramsQCOM\0",
                &self.glExtGetProgramsQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        pub fn ExtGetProgramsQCOM_is_loaded(&self) -> bool {
            !self.glExtGetProgramsQCOM_p.load(RELAX).is_null()
        }
        /// [glExtGetRenderbuffersQCOM](http://docs.gl/es3/glExtGetRenderbuffersQCOM)(renderbuffers, maxRenderbuffers, numRenderbuffers)
        /// * `renderbuffers` len: maxRenderbuffers
        /// * `renderbuffers` class: renderbuffer
        /// * `numRenderbuffers` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub unsafe fn ExtGetRenderbuffersQCOM(
            &self,
            renderbuffers: *mut GLuint,
            maxRenderbuffers: GLint,
            numRenderbuffers: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtGetRenderbuffersQCOM({:p}, {:?}, {:p});",
                    renderbuffers,
                    maxRenderbuffers,
                    numRenderbuffers
                );
            }
            let out = call_atomic_ptr_3arg(
                "glExtGetRenderbuffersQCOM",
                &self.glExtGetRenderbuffersQCOM_p,
                renderbuffers,
                maxRenderbuffers,
                numRenderbuffers,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetRenderbuffersQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetRenderbuffersQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetRenderbuffersQCOM\0",
                &self.glExtGetRenderbuffersQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub fn ExtGetRenderbuffersQCOM_is_loaded(&self) -> bool {
            !self.glExtGetRenderbuffersQCOM_p.load(RELAX).is_null()
        }
        /// [glExtGetShadersQCOM](http://docs.gl/es3/glExtGetShadersQCOM)(shaders, maxShaders, numShaders)
        /// * `shaders` len: maxShaders
        /// * `shaders` class: shader
        /// * `numShaders` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        pub unsafe fn ExtGetShadersQCOM(
            &self,
            shaders: *mut GLuint,
            maxShaders: GLint,
            numShaders: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtGetShadersQCOM({:p}, {:?}, {:p});",
                    shaders,
                    maxShaders,
                    numShaders
                );
            }
            let out = call_atomic_ptr_3arg(
                "glExtGetShadersQCOM",
                &self.glExtGetShadersQCOM_p,
                shaders,
                maxShaders,
                numShaders,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetShadersQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetShadersQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetShadersQCOM\0",
                &self.glExtGetShadersQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        pub fn ExtGetShadersQCOM_is_loaded(&self) -> bool {
            !self.glExtGetShadersQCOM_p.load(RELAX).is_null()
        }
        /// [glExtGetTexLevelParameterivQCOM](http://docs.gl/es3/glExtGetTexLevelParameterivQCOM)(texture, face, level, pname, params)
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub unsafe fn ExtGetTexLevelParameterivQCOM(
            &self,
            texture: GLuint,
            face: GLenum,
            level: GLint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtGetTexLevelParameterivQCOM({:?}, {:#X}, {:?}, {:#X}, {:p});",
                    texture,
                    face,
                    level,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_5arg(
                "glExtGetTexLevelParameterivQCOM",
                &self.glExtGetTexLevelParameterivQCOM_p,
                texture,
                face,
                level,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetTexLevelParameterivQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetTexLevelParameterivQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetTexLevelParameterivQCOM\0",
                &self.glExtGetTexLevelParameterivQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub fn ExtGetTexLevelParameterivQCOM_is_loaded(&self) -> bool {
            !self.glExtGetTexLevelParameterivQCOM_p.load(RELAX).is_null()
        }
        /// [glExtGetTexSubImageQCOM](http://docs.gl/es3/glExtGetTexSubImageQCOM)(target, level, xoffset, yoffset, zoffset, width, height, depth, format, type_, texels)
        /// * `format` group: PixelFormat
        /// * `type_` group: PixelType
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub unsafe fn ExtGetTexSubImageQCOM(
            &self,
            target: GLenum,
            level: GLint,
            xoffset: GLint,
            yoffset: GLint,
            zoffset: GLint,
            width: GLsizei,
            height: GLsizei,
            depth: GLsizei,
            format: GLenum,
            type_: GLenum,
            texels: *mut c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.ExtGetTexSubImageQCOM({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:#X}, {:p});", target, level, xoffset, yoffset, zoffset, width, height, depth, format, type_, texels);
            }
            let out = call_atomic_ptr_11arg(
                "glExtGetTexSubImageQCOM",
                &self.glExtGetTexSubImageQCOM_p,
                target,
                level,
                xoffset,
                yoffset,
                zoffset,
                width,
                height,
                depth,
                format,
                type_,
                texels,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetTexSubImageQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetTexSubImageQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetTexSubImageQCOM\0",
                &self.glExtGetTexSubImageQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub fn ExtGetTexSubImageQCOM_is_loaded(&self) -> bool {
            !self.glExtGetTexSubImageQCOM_p.load(RELAX).is_null()
        }
        /// [glExtGetTexturesQCOM](http://docs.gl/es3/glExtGetTexturesQCOM)(textures, maxTextures, numTextures)
        /// * `textures` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub unsafe fn ExtGetTexturesQCOM(
            &self,
            textures: *mut GLuint,
            maxTextures: GLint,
            numTextures: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtGetTexturesQCOM({:p}, {:?}, {:p});",
                    textures,
                    maxTextures,
                    numTextures
                );
            }
            let out = call_atomic_ptr_3arg(
                "glExtGetTexturesQCOM",
                &self.glExtGetTexturesQCOM_p,
                textures,
                maxTextures,
                numTextures,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtGetTexturesQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        #[doc(hidden)]
        pub unsafe fn ExtGetTexturesQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtGetTexturesQCOM\0",
                &self.glExtGetTexturesQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub fn ExtGetTexturesQCOM_is_loaded(&self) -> bool {
            !self.glExtGetTexturesQCOM_p.load(RELAX).is_null()
        }
        /// [glExtIsProgramBinaryQCOM](http://docs.gl/es3/glExtIsProgramBinaryQCOM)(program)
        /// * `program` class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        pub unsafe fn ExtIsProgramBinaryQCOM(&self, program: GLuint) -> GLboolean {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.ExtIsProgramBinaryQCOM({:?});", program);
            }
            let out = call_atomic_ptr_1arg(
                "glExtIsProgramBinaryQCOM",
                &self.glExtIsProgramBinaryQCOM_p,
                program,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtIsProgramBinaryQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        #[doc(hidden)]
        pub unsafe fn ExtIsProgramBinaryQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtIsProgramBinaryQCOM\0",
                &self.glExtIsProgramBinaryQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        pub fn ExtIsProgramBinaryQCOM_is_loaded(&self) -> bool {
            !self.glExtIsProgramBinaryQCOM_p.load(RELAX).is_null()
        }
        /// [glExtTexObjectStateOverrideiQCOM](http://docs.gl/es3/glExtTexObjectStateOverrideiQCOM)(target, pname, param)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub unsafe fn ExtTexObjectStateOverrideiQCOM(
            &self,
            target: GLenum,
            pname: GLenum,
            param: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtTexObjectStateOverrideiQCOM({:#X}, {:#X}, {:?});",
                    target,
                    pname,
                    param
                );
            }
            let out = call_atomic_ptr_3arg(
                "glExtTexObjectStateOverrideiQCOM",
                &self.glExtTexObjectStateOverrideiQCOM_p,
                target,
                pname,
                param,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtTexObjectStateOverrideiQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        #[doc(hidden)]
        pub unsafe fn ExtTexObjectStateOverrideiQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtTexObjectStateOverrideiQCOM\0",
                &self.glExtTexObjectStateOverrideiQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        pub fn ExtTexObjectStateOverrideiQCOM_is_loaded(&self) -> bool {
            !self
                .glExtTexObjectStateOverrideiQCOM_p
                .load(RELAX)
                .is_null()
        }
        /// [glExtrapolateTex2DQCOM](http://docs.gl/es3/glExtrapolateTex2DQCOM)(src1, src2, output, scaleFactor)
        /// * `src1` class: texture
        /// * `src2` class: texture
        /// * `output` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_frame_extrapolation"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_frame_extrapolation"))))]
        pub unsafe fn ExtrapolateTex2DQCOM(
            &self,
            src1: GLuint,
            src2: GLuint,
            output: GLuint,
            scaleFactor: GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.ExtrapolateTex2DQCOM({:?}, {:?}, {:?}, {:?});",
                    src1,
                    src2,
                    output,
                    scaleFactor
                );
            }
            let out = call_atomic_ptr_4arg(
                "glExtrapolateTex2DQCOM",
                &self.glExtrapolateTex2DQCOM_p,
                src1,
                src2,
                output,
                scaleFactor,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glExtrapolateTex2DQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_frame_extrapolation"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_frame_extrapolation"))))]
        #[doc(hidden)]
        pub unsafe fn ExtrapolateTex2DQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glExtrapolateTex2DQCOM\0",
                &self.glExtrapolateTex2DQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_frame_extrapolation"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_frame_extrapolation"))))]
        pub fn ExtrapolateTex2DQCOM_is_loaded(&self) -> bool {
            !self.glExtrapolateTex2DQCOM_p.load(RELAX).is_null()
        }
        /// [glFenceSync](http://docs.gl/es3/glFenceSync)(condition, flags)
        /// * `condition` group: SyncCondition
        /// * `flags` group: SyncBehaviorFlags
        /// * return value group: sync
        /// * return value class: sync
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn FenceSync(&self, condition: GLenum, flags: GLbitfield) -> GLsync {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.FenceSync({:#X}, {:?});", condition, flags);
            }
            let out = call_atomic_ptr_2arg("glFenceSync", &self.glFenceSync_p, condition, flags);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFenceSync");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn FenceSync_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glFenceSync\0", &self.glFenceSync_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn FenceSync_is_loaded(&self) -> bool {
            !self.glFenceSync_p.load(RELAX).is_null()
        }
        /// [glFenceSyncAPPLE](http://docs.gl/es3/glFenceSyncAPPLE)(condition, flags)
        /// * `condition` group: SyncCondition
        /// * `flags` group: SyncBehaviorFlags
        /// * return value group: sync
        /// * return value class: sync
        /// * alias of: [`glFenceSync`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub unsafe fn FenceSyncAPPLE(&self, condition: GLenum, flags: GLbitfield) -> GLsync {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.FenceSyncAPPLE({:#X}, {:?});", condition, flags);
            }
            let out = call_atomic_ptr_2arg(
                "glFenceSyncAPPLE",
                &self.glFenceSyncAPPLE_p,
                condition,
                flags,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFenceSyncAPPLE");
            }
            out
        }
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        #[doc(hidden)]
        pub unsafe fn FenceSyncAPPLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFenceSyncAPPLE\0",
                &self.glFenceSyncAPPLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub fn FenceSyncAPPLE_is_loaded(&self) -> bool {
            !self.glFenceSyncAPPLE_p.load(RELAX).is_null()
        }
        /// [glFinish](http://docs.gl/es3/glFinish)()
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn Finish(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.Finish();",);
            }
            let out = call_atomic_ptr_0arg("glFinish", &self.glFinish_p);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFinish");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn Finish_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glFinish\0", &self.glFinish_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn Finish_is_loaded(&self) -> bool {
            !self.glFinish_p.load(RELAX).is_null()
        }
        /// [glFinishFenceNV](http://docs.gl/es3/glFinishFenceNV)(fence)
        /// * `fence` group: FenceNV
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        pub unsafe fn FinishFenceNV(&self, fence: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.FinishFenceNV({:?});", fence);
            }
            let out = call_atomic_ptr_1arg("glFinishFenceNV", &self.glFinishFenceNV_p, fence);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFinishFenceNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        #[doc(hidden)]
        pub unsafe fn FinishFenceNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFinishFenceNV\0",
                &self.glFinishFenceNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        pub fn FinishFenceNV_is_loaded(&self) -> bool {
            !self.glFinishFenceNV_p.load(RELAX).is_null()
        }
        /// [glFlush](http://docs.gl/es3/glFlush)()
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn Flush(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.Flush();",);
            }
            let out = call_atomic_ptr_0arg("glFlush", &self.glFlush_p);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFlush");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn Flush_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glFlush\0", &self.glFlush_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn Flush_is_loaded(&self) -> bool {
            !self.glFlush_p.load(RELAX).is_null()
        }
        /// [glFlushMappedBufferRange](http://docs.gl/es3/glFlushMappedBufferRange)(target, offset, length)
        /// * `target` group: BufferTargetARB
        /// * `offset` group: BufferOffset
        /// * `length` group: BufferSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn FlushMappedBufferRange(
            &self,
            target: GLenum,
            offset: GLintptr,
            length: GLsizeiptr,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FlushMappedBufferRange({:#X}, {:?}, {:?});",
                    target,
                    offset,
                    length
                );
            }
            let out = call_atomic_ptr_3arg(
                "glFlushMappedBufferRange",
                &self.glFlushMappedBufferRange_p,
                target,
                offset,
                length,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFlushMappedBufferRange");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn FlushMappedBufferRange_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFlushMappedBufferRange\0",
                &self.glFlushMappedBufferRange_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn FlushMappedBufferRange_is_loaded(&self) -> bool {
            !self.glFlushMappedBufferRange_p.load(RELAX).is_null()
        }
        /// [glFlushMappedBufferRangeEXT](http://docs.gl/es3/glFlushMappedBufferRangeEXT)(target, offset, length)
        /// * `target` group: BufferTargetARB
        /// * alias of: [`glFlushMappedBufferRange`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
        pub unsafe fn FlushMappedBufferRangeEXT(
            &self,
            target: GLenum,
            offset: GLintptr,
            length: GLsizeiptr,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FlushMappedBufferRangeEXT({:#X}, {:?}, {:?});",
                    target,
                    offset,
                    length
                );
            }
            let out = call_atomic_ptr_3arg(
                "glFlushMappedBufferRangeEXT",
                &self.glFlushMappedBufferRangeEXT_p,
                target,
                offset,
                length,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFlushMappedBufferRangeEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
        #[doc(hidden)]
        pub unsafe fn FlushMappedBufferRangeEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFlushMappedBufferRangeEXT\0",
                &self.glFlushMappedBufferRangeEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
        pub fn FlushMappedBufferRangeEXT_is_loaded(&self) -> bool {
            !self.glFlushMappedBufferRangeEXT_p.load(RELAX).is_null()
        }
        /// [glFragmentCoverageColorNV](http://docs.gl/es3/glFragmentCoverageColorNV)(color)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_fragment_coverage_to_color"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fragment_coverage_to_color"))))]
        pub unsafe fn FragmentCoverageColorNV(&self, color: GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.FragmentCoverageColorNV({:?});", color);
            }
            let out = call_atomic_ptr_1arg(
                "glFragmentCoverageColorNV",
                &self.glFragmentCoverageColorNV_p,
                color,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFragmentCoverageColorNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_fragment_coverage_to_color"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fragment_coverage_to_color"))))]
        #[doc(hidden)]
        pub unsafe fn FragmentCoverageColorNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFragmentCoverageColorNV\0",
                &self.glFragmentCoverageColorNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_fragment_coverage_to_color"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fragment_coverage_to_color"))))]
        pub fn FragmentCoverageColorNV_is_loaded(&self) -> bool {
            !self.glFragmentCoverageColorNV_p.load(RELAX).is_null()
        }
        /// [glFramebufferFetchBarrierEXT](http://docs.gl/es3/glFramebufferFetchBarrierEXT)()
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent")))
        )]
        pub unsafe fn FramebufferFetchBarrierEXT(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.FramebufferFetchBarrierEXT();",);
            }
            let out = call_atomic_ptr_0arg(
                "glFramebufferFetchBarrierEXT",
                &self.glFramebufferFetchBarrierEXT_p,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferFetchBarrierEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent")))
        )]
        #[doc(hidden)]
        pub unsafe fn FramebufferFetchBarrierEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferFetchBarrierEXT\0",
                &self.glFramebufferFetchBarrierEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent")))
        )]
        pub fn FramebufferFetchBarrierEXT_is_loaded(&self) -> bool {
            !self.glFramebufferFetchBarrierEXT_p.load(RELAX).is_null()
        }
        /// [glFramebufferFetchBarrierQCOM](http://docs.gl/es3/glFramebufferFetchBarrierQCOM)()
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent")))
        )]
        pub unsafe fn FramebufferFetchBarrierQCOM(&self) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.FramebufferFetchBarrierQCOM();",);
            }
            let out = call_atomic_ptr_0arg(
                "glFramebufferFetchBarrierQCOM",
                &self.glFramebufferFetchBarrierQCOM_p,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferFetchBarrierQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent")))
        )]
        #[doc(hidden)]
        pub unsafe fn FramebufferFetchBarrierQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferFetchBarrierQCOM\0",
                &self.glFramebufferFetchBarrierQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent")))
        )]
        pub fn FramebufferFetchBarrierQCOM_is_loaded(&self) -> bool {
            !self.glFramebufferFetchBarrierQCOM_p.load(RELAX).is_null()
        }
        /// [glFramebufferFoveationConfigQCOM](http://docs.gl/es3/glFramebufferFoveationConfigQCOM)(framebuffer, numLayers, focalPointsPerLayer, requestedFeatures, providedFeatures)
        /// * `framebuffer` group: Framebuffer
        /// * `framebuffer` class: framebuffer
        /// * `providedFeatures` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
        pub unsafe fn FramebufferFoveationConfigQCOM(
            &self,
            framebuffer: GLuint,
            numLayers: GLuint,
            focalPointsPerLayer: GLuint,
            requestedFeatures: GLuint,
            providedFeatures: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferFoveationConfigQCOM({:?}, {:?}, {:?}, {:?}, {:p});",
                    framebuffer,
                    numLayers,
                    focalPointsPerLayer,
                    requestedFeatures,
                    providedFeatures
                );
            }
            let out = call_atomic_ptr_5arg(
                "glFramebufferFoveationConfigQCOM",
                &self.glFramebufferFoveationConfigQCOM_p,
                framebuffer,
                numLayers,
                focalPointsPerLayer,
                requestedFeatures,
                providedFeatures,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferFoveationConfigQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferFoveationConfigQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferFoveationConfigQCOM\0",
                &self.glFramebufferFoveationConfigQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
        pub fn FramebufferFoveationConfigQCOM_is_loaded(&self) -> bool {
            !self
                .glFramebufferFoveationConfigQCOM_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferFoveationParametersQCOM](http://docs.gl/es3/glFramebufferFoveationParametersQCOM)(framebuffer, layer, focalPoint, focalX, focalY, gainX, gainY, foveaArea)
        /// * `framebuffer` group: Framebuffer
        /// * `framebuffer` class: framebuffer
        /// * `focalX` group: CheckedFloat32
        /// * `focalY` group: CheckedFloat32
        /// * `gainX` group: CheckedFloat32
        /// * `gainY` group: CheckedFloat32
        /// * `foveaArea` group: CheckedFloat32
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
        pub unsafe fn FramebufferFoveationParametersQCOM(
            &self,
            framebuffer: GLuint,
            layer: GLuint,
            focalPoint: GLuint,
            focalX: GLfloat,
            focalY: GLfloat,
            gainX: GLfloat,
            gainY: GLfloat,
            foveaArea: GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferFoveationParametersQCOM({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                    framebuffer,
                    layer,
                    focalPoint,
                    focalX,
                    focalY,
                    gainX,
                    gainY,
                    foveaArea
                );
            }
            let out = call_atomic_ptr_8arg(
                "glFramebufferFoveationParametersQCOM",
                &self.glFramebufferFoveationParametersQCOM_p,
                framebuffer,
                layer,
                focalPoint,
                focalX,
                focalY,
                gainX,
                gainY,
                foveaArea,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferFoveationParametersQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferFoveationParametersQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferFoveationParametersQCOM\0",
                &self.glFramebufferFoveationParametersQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
        pub fn FramebufferFoveationParametersQCOM_is_loaded(&self) -> bool {
            !self
                .glFramebufferFoveationParametersQCOM_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferParameteri](http://docs.gl/es3/glFramebufferParameter)(target, pname, param)
        /// * `target` group: FramebufferTarget
        /// * `pname` group: FramebufferParameterName
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn FramebufferParameteri(&self, target: GLenum, pname: GLenum, param: GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferParameteri({:#X}, {:#X}, {:?});",
                    target,
                    pname,
                    param
                );
            }
            let out = call_atomic_ptr_3arg(
                "glFramebufferParameteri",
                &self.glFramebufferParameteri_p,
                target,
                pname,
                param,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferParameteri");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn FramebufferParameteri_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferParameteri\0",
                &self.glFramebufferParameteri_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn FramebufferParameteri_is_loaded(&self) -> bool {
            !self.glFramebufferParameteri_p.load(RELAX).is_null()
        }
        /// [glFramebufferParameteriMESA](http://docs.gl/es3/glFramebufferParameteriMESA)(target, pname, param)
        /// * `target` group: FramebufferTarget
        /// * `pname` group: FramebufferParameterName
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
        pub unsafe fn FramebufferParameteriMESA(
            &self,
            target: GLenum,
            pname: GLenum,
            param: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferParameteriMESA({:#X}, {:#X}, {:?});",
                    target,
                    pname,
                    param
                );
            }
            let out = call_atomic_ptr_3arg(
                "glFramebufferParameteriMESA",
                &self.glFramebufferParameteriMESA_p,
                target,
                pname,
                param,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferParameteriMESA");
            }
            out
        }
        #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferParameteriMESA_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferParameteriMESA\0",
                &self.glFramebufferParameteriMESA_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
        pub fn FramebufferParameteriMESA_is_loaded(&self) -> bool {
            !self.glFramebufferParameteriMESA_p.load(RELAX).is_null()
        }
        /// [glFramebufferPixelLocalStorageSizeEXT](http://docs.gl/es3/glFramebufferPixelLocalStorageSizeEXT)(target, size)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        pub unsafe fn FramebufferPixelLocalStorageSizeEXT(&self, target: GLuint, size: GLsizei) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferPixelLocalStorageSizeEXT({:?}, {:?});",
                    target,
                    size
                );
            }
            let out = call_atomic_ptr_2arg(
                "glFramebufferPixelLocalStorageSizeEXT",
                &self.glFramebufferPixelLocalStorageSizeEXT_p,
                target,
                size,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferPixelLocalStorageSizeEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferPixelLocalStorageSizeEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferPixelLocalStorageSizeEXT\0",
                &self.glFramebufferPixelLocalStorageSizeEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        pub fn FramebufferPixelLocalStorageSizeEXT_is_loaded(&self) -> bool {
            !self
                .glFramebufferPixelLocalStorageSizeEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferRenderbuffer](http://docs.gl/es3/glFramebufferRenderbuffer)(target, attachment, renderbuffertarget, renderbuffer)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `renderbuffertarget` group: RenderbufferTarget
        /// * `renderbuffer` class: renderbuffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn FramebufferRenderbuffer(
            &self,
            target: GLenum,
            attachment: GLenum,
            renderbuffertarget: GLenum,
            renderbuffer: GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferRenderbuffer({:#X}, {:#X}, {:#X}, {:?});",
                    target,
                    attachment,
                    renderbuffertarget,
                    renderbuffer
                );
            }
            let out = call_atomic_ptr_4arg(
                "glFramebufferRenderbuffer",
                &self.glFramebufferRenderbuffer_p,
                target,
                attachment,
                renderbuffertarget,
                renderbuffer,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferRenderbuffer");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn FramebufferRenderbuffer_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferRenderbuffer\0",
                &self.glFramebufferRenderbuffer_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn FramebufferRenderbuffer_is_loaded(&self) -> bool {
            !self.glFramebufferRenderbuffer_p.load(RELAX).is_null()
        }
        /// [glFramebufferSampleLocationsfvNV](http://docs.gl/es3/glFramebufferSampleLocationsfvNV)(target, start, count, v)
        /// * `target` group: FramebufferTarget
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_sample_locations"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
        pub unsafe fn FramebufferSampleLocationsfvNV(
            &self,
            target: GLenum,
            start: GLuint,
            count: GLsizei,
            v: *const GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferSampleLocationsfvNV({:#X}, {:?}, {:?}, {:p});",
                    target,
                    start,
                    count,
                    v
                );
            }
            let out = call_atomic_ptr_4arg(
                "glFramebufferSampleLocationsfvNV",
                &self.glFramebufferSampleLocationsfvNV_p,
                target,
                start,
                count,
                v,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferSampleLocationsfvNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_sample_locations"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferSampleLocationsfvNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferSampleLocationsfvNV\0",
                &self.glFramebufferSampleLocationsfvNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_sample_locations"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
        pub fn FramebufferSampleLocationsfvNV_is_loaded(&self) -> bool {
            !self
                .glFramebufferSampleLocationsfvNV_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferTexture](http://docs.gl/es3/glFramebufferTexture)(target, attachment, texture, level)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn FramebufferTexture(
            &self,
            target: GLenum,
            attachment: GLenum,
            texture: GLuint,
            level: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTexture({:#X}, {:#X}, {:?}, {:?});",
                    target,
                    attachment,
                    texture,
                    level
                );
            }
            let out = call_atomic_ptr_4arg(
                "glFramebufferTexture",
                &self.glFramebufferTexture_p,
                target,
                attachment,
                texture,
                level,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTexture");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn FramebufferTexture_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTexture\0",
                &self.glFramebufferTexture_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn FramebufferTexture_is_loaded(&self) -> bool {
            !self.glFramebufferTexture_p.load(RELAX).is_null()
        }
        /// [glFramebufferTexture2D](http://docs.gl/es3/glFramebufferTexture2D)(target, attachment, textarget, texture, level)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `textarget` group: TextureTarget
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn FramebufferTexture2D(
            &self,
            target: GLenum,
            attachment: GLenum,
            textarget: GLenum,
            texture: GLuint,
            level: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTexture2D({:#X}, {:#X}, {:#X}, {:?}, {:?});",
                    target,
                    attachment,
                    textarget,
                    texture,
                    level
                );
            }
            let out = call_atomic_ptr_5arg(
                "glFramebufferTexture2D",
                &self.glFramebufferTexture2D_p,
                target,
                attachment,
                textarget,
                texture,
                level,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTexture2D");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn FramebufferTexture2D_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTexture2D\0",
                &self.glFramebufferTexture2D_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn FramebufferTexture2D_is_loaded(&self) -> bool {
            !self.glFramebufferTexture2D_p.load(RELAX).is_null()
        }
        /// [glFramebufferTexture2DDownsampleIMG](http://docs.gl/es3/glFramebufferTexture2DDownsampleIMG)(target, attachment, textarget, texture, level, xscale, yscale)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `textarget` group: TextureTarget
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
        pub unsafe fn FramebufferTexture2DDownsampleIMG(
            &self,
            target: GLenum,
            attachment: GLenum,
            textarget: GLenum,
            texture: GLuint,
            level: GLint,
            xscale: GLint,
            yscale: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTexture2DDownsampleIMG({:#X}, {:#X}, {:#X}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    attachment,
                    textarget,
                    texture,
                    level,
                    xscale,
                    yscale
                );
            }
            let out = call_atomic_ptr_7arg(
                "glFramebufferTexture2DDownsampleIMG",
                &self.glFramebufferTexture2DDownsampleIMG_p,
                target,
                attachment,
                textarget,
                texture,
                level,
                xscale,
                yscale,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTexture2DDownsampleIMG");
            }
            out
        }
        #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferTexture2DDownsampleIMG_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTexture2DDownsampleIMG\0",
                &self.glFramebufferTexture2DDownsampleIMG_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
        pub fn FramebufferTexture2DDownsampleIMG_is_loaded(&self) -> bool {
            !self
                .glFramebufferTexture2DDownsampleIMG_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferTexture2DMultisampleEXT](http://docs.gl/es3/glFramebufferTexture2DMultisampleEXT)(target, attachment, textarget, texture, level, samples)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `textarget` group: TextureTarget
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
        )]
        pub unsafe fn FramebufferTexture2DMultisampleEXT(
            &self,
            target: GLenum,
            attachment: GLenum,
            textarget: GLenum,
            texture: GLuint,
            level: GLint,
            samples: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTexture2DMultisampleEXT({:#X}, {:#X}, {:#X}, {:?}, {:?}, {:?});",
                    target,
                    attachment,
                    textarget,
                    texture,
                    level,
                    samples
                );
            }
            let out = call_atomic_ptr_6arg(
                "glFramebufferTexture2DMultisampleEXT",
                &self.glFramebufferTexture2DMultisampleEXT_p,
                target,
                attachment,
                textarget,
                texture,
                level,
                samples,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTexture2DMultisampleEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
        )]
        #[doc(hidden)]
        pub unsafe fn FramebufferTexture2DMultisampleEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTexture2DMultisampleEXT\0",
                &self.glFramebufferTexture2DMultisampleEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
        )]
        pub fn FramebufferTexture2DMultisampleEXT_is_loaded(&self) -> bool {
            !self
                .glFramebufferTexture2DMultisampleEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferTexture2DMultisampleIMG](http://docs.gl/es3/glFramebufferTexture2DMultisampleIMG)(target, attachment, textarget, texture, level, samples)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `textarget` group: TextureTarget
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
        )]
        pub unsafe fn FramebufferTexture2DMultisampleIMG(
            &self,
            target: GLenum,
            attachment: GLenum,
            textarget: GLenum,
            texture: GLuint,
            level: GLint,
            samples: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTexture2DMultisampleIMG({:#X}, {:#X}, {:#X}, {:?}, {:?}, {:?});",
                    target,
                    attachment,
                    textarget,
                    texture,
                    level,
                    samples
                );
            }
            let out = call_atomic_ptr_6arg(
                "glFramebufferTexture2DMultisampleIMG",
                &self.glFramebufferTexture2DMultisampleIMG_p,
                target,
                attachment,
                textarget,
                texture,
                level,
                samples,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTexture2DMultisampleIMG");
            }
            out
        }
        #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
        )]
        #[doc(hidden)]
        pub unsafe fn FramebufferTexture2DMultisampleIMG_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTexture2DMultisampleIMG\0",
                &self.glFramebufferTexture2DMultisampleIMG_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
        )]
        pub fn FramebufferTexture2DMultisampleIMG_is_loaded(&self) -> bool {
            !self
                .glFramebufferTexture2DMultisampleIMG_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferTexture3DOES](http://docs.gl/es3/glFramebufferTexture3DOES)(target, attachment, textarget, texture, level, zoffset)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `textarget` group: TextureTarget
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        pub unsafe fn FramebufferTexture3DOES(
            &self,
            target: GLenum,
            attachment: GLenum,
            textarget: GLenum,
            texture: GLuint,
            level: GLint,
            zoffset: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTexture3DOES({:#X}, {:#X}, {:#X}, {:?}, {:?}, {:?});",
                    target,
                    attachment,
                    textarget,
                    texture,
                    level,
                    zoffset
                );
            }
            let out = call_atomic_ptr_6arg(
                "glFramebufferTexture3DOES",
                &self.glFramebufferTexture3DOES_p,
                target,
                attachment,
                textarget,
                texture,
                level,
                zoffset,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTexture3DOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferTexture3DOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTexture3DOES\0",
                &self.glFramebufferTexture3DOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        pub fn FramebufferTexture3DOES_is_loaded(&self) -> bool {
            !self.glFramebufferTexture3DOES_p.load(RELAX).is_null()
        }
        /// [glFramebufferTextureEXT](http://docs.gl/es3/glFramebufferTextureEXT)(target, attachment, texture, level)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `texture` group: Texture
        /// * `texture` class: texture
        /// * `level` group: CheckedInt32
        /// * alias of: [`glFramebufferTexture`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_geometry_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_geometry_shader"))))]
        pub unsafe fn FramebufferTextureEXT(
            &self,
            target: GLenum,
            attachment: GLenum,
            texture: GLuint,
            level: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTextureEXT({:#X}, {:#X}, {:?}, {:?});",
                    target,
                    attachment,
                    texture,
                    level
                );
            }
            let out = call_atomic_ptr_4arg(
                "glFramebufferTextureEXT",
                &self.glFramebufferTextureEXT_p,
                target,
                attachment,
                texture,
                level,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTextureEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_geometry_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_geometry_shader"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferTextureEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTextureEXT\0",
                &self.glFramebufferTextureEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_geometry_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_geometry_shader"))))]
        pub fn FramebufferTextureEXT_is_loaded(&self) -> bool {
            !self.glFramebufferTextureEXT_p.load(RELAX).is_null()
        }
        /// [glFramebufferTextureLayer](http://docs.gl/es3/glFramebufferTextureLayer)(target, attachment, texture, level, layer)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `texture` group: Texture
        /// * `texture` class: texture
        /// * `level` group: CheckedInt32
        /// * `layer` group: CheckedInt32
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn FramebufferTextureLayer(
            &self,
            target: GLenum,
            attachment: GLenum,
            texture: GLuint,
            level: GLint,
            layer: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTextureLayer({:#X}, {:#X}, {:?}, {:?}, {:?});",
                    target,
                    attachment,
                    texture,
                    level,
                    layer
                );
            }
            let out = call_atomic_ptr_5arg(
                "glFramebufferTextureLayer",
                &self.glFramebufferTextureLayer_p,
                target,
                attachment,
                texture,
                level,
                layer,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTextureLayer");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn FramebufferTextureLayer_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTextureLayer\0",
                &self.glFramebufferTextureLayer_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn FramebufferTextureLayer_is_loaded(&self) -> bool {
            !self.glFramebufferTextureLayer_p.load(RELAX).is_null()
        }
        /// [glFramebufferTextureLayerDownsampleIMG](http://docs.gl/es3/glFramebufferTextureLayerDownsampleIMG)(target, attachment, texture, level, layer, xscale, yscale)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `texture` group: Texture
        /// * `texture` class: texture
        /// * `level` group: CheckedInt32
        /// * `layer` group: CheckedInt32
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
        pub unsafe fn FramebufferTextureLayerDownsampleIMG(
            &self,
            target: GLenum,
            attachment: GLenum,
            texture: GLuint,
            level: GLint,
            layer: GLint,
            xscale: GLint,
            yscale: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTextureLayerDownsampleIMG({:#X}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    attachment,
                    texture,
                    level,
                    layer,
                    xscale,
                    yscale
                );
            }
            let out = call_atomic_ptr_7arg(
                "glFramebufferTextureLayerDownsampleIMG",
                &self.glFramebufferTextureLayerDownsampleIMG_p,
                target,
                attachment,
                texture,
                level,
                layer,
                xscale,
                yscale,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTextureLayerDownsampleIMG");
            }
            out
        }
        #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferTextureLayerDownsampleIMG_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTextureLayerDownsampleIMG\0",
                &self.glFramebufferTextureLayerDownsampleIMG_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
        pub fn FramebufferTextureLayerDownsampleIMG_is_loaded(&self) -> bool {
            !self
                .glFramebufferTextureLayerDownsampleIMG_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferTextureMultisampleMultiviewOVR](http://docs.gl/es3/glFramebufferTextureMultisampleMultiviewOVR)(target, attachment, texture, level, samples, baseViewIndex, numViews)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `texture` group: Texture
        /// * `texture` class: texture
        /// * `level` group: CheckedInt32
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture")))
        )]
        pub unsafe fn FramebufferTextureMultisampleMultiviewOVR(
            &self,
            target: GLenum,
            attachment: GLenum,
            texture: GLuint,
            level: GLint,
            samples: GLsizei,
            baseViewIndex: GLint,
            numViews: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTextureMultisampleMultiviewOVR({:#X}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    attachment,
                    texture,
                    level,
                    samples,
                    baseViewIndex,
                    numViews
                );
            }
            let out = call_atomic_ptr_7arg(
                "glFramebufferTextureMultisampleMultiviewOVR",
                &self.glFramebufferTextureMultisampleMultiviewOVR_p,
                target,
                attachment,
                texture,
                level,
                samples,
                baseViewIndex,
                numViews,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTextureMultisampleMultiviewOVR");
            }
            out
        }
        #[cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture")))
        )]
        #[doc(hidden)]
        pub unsafe fn FramebufferTextureMultisampleMultiviewOVR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTextureMultisampleMultiviewOVR\0",
                &self.glFramebufferTextureMultisampleMultiviewOVR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture")))
        )]
        pub fn FramebufferTextureMultisampleMultiviewOVR_is_loaded(&self) -> bool {
            !self
                .glFramebufferTextureMultisampleMultiviewOVR_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferTextureMultiviewOVR](http://docs.gl/es3/glFramebufferTextureMultiviewOVR)(target, attachment, texture, level, baseViewIndex, numViews)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `texture` group: Texture
        /// * `texture` class: texture
        /// * `level` group: CheckedInt32
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OVR_multiview"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OVR_multiview"))))]
        pub unsafe fn FramebufferTextureMultiviewOVR(
            &self,
            target: GLenum,
            attachment: GLenum,
            texture: GLuint,
            level: GLint,
            baseViewIndex: GLint,
            numViews: GLsizei,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTextureMultiviewOVR({:#X}, {:#X}, {:?}, {:?}, {:?}, {:?});",
                    target,
                    attachment,
                    texture,
                    level,
                    baseViewIndex,
                    numViews
                );
            }
            let out = call_atomic_ptr_6arg(
                "glFramebufferTextureMultiviewOVR",
                &self.glFramebufferTextureMultiviewOVR_p,
                target,
                attachment,
                texture,
                level,
                baseViewIndex,
                numViews,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTextureMultiviewOVR");
            }
            out
        }
        #[cfg(any(feature = "GL_OVR_multiview"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OVR_multiview"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferTextureMultiviewOVR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTextureMultiviewOVR\0",
                &self.glFramebufferTextureMultiviewOVR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OVR_multiview"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OVR_multiview"))))]
        pub fn FramebufferTextureMultiviewOVR_is_loaded(&self) -> bool {
            !self
                .glFramebufferTextureMultiviewOVR_p
                .load(RELAX)
                .is_null()
        }
        /// [glFramebufferTextureOES](http://docs.gl/es3/glFramebufferTextureOES)(target, attachment, texture, level)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `texture` group: Texture
        /// * `texture` class: texture
        /// * `level` group: CheckedInt32
        /// * alias of: [`glFramebufferTexture`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_geometry_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_geometry_shader"))))]
        pub unsafe fn FramebufferTextureOES(
            &self,
            target: GLenum,
            attachment: GLenum,
            texture: GLuint,
            level: GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.FramebufferTextureOES({:#X}, {:#X}, {:?}, {:?});",
                    target,
                    attachment,
                    texture,
                    level
                );
            }
            let out = call_atomic_ptr_4arg(
                "glFramebufferTextureOES",
                &self.glFramebufferTextureOES_p,
                target,
                attachment,
                texture,
                level,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFramebufferTextureOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_geometry_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_geometry_shader"))))]
        #[doc(hidden)]
        pub unsafe fn FramebufferTextureOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glFramebufferTextureOES\0",
                &self.glFramebufferTextureOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_geometry_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_geometry_shader"))))]
        pub fn FramebufferTextureOES_is_loaded(&self) -> bool {
            !self.glFramebufferTextureOES_p.load(RELAX).is_null()
        }
        /// [glFrontFace](http://docs.gl/es3/glFrontFace)(mode)
        /// * `mode` group: FrontFaceDirection
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn FrontFace(&self, mode: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.FrontFace({:#X});", mode);
            }
            let out = call_atomic_ptr_1arg("glFrontFace", &self.glFrontFace_p, mode);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glFrontFace");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn FrontFace_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glFrontFace\0", &self.glFrontFace_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn FrontFace_is_loaded(&self) -> bool {
            !self.glFrontFace_p.load(RELAX).is_null()
        }
        /// [glGenBuffers](http://docs.gl/es3/glGenBuffers)(n, buffers)
        /// * `buffers` len: n
        /// * `buffers` class: buffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenBuffers(&self, n: GLsizei, buffers: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenBuffers({:?}, {:p});", n, buffers);
            }
            let out = call_atomic_ptr_2arg("glGenBuffers", &self.glGenBuffers_p, n, buffers);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenBuffers");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenBuffers_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGenBuffers\0", &self.glGenBuffers_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenBuffers_is_loaded(&self) -> bool {
            !self.glGenBuffers_p.load(RELAX).is_null()
        }
        /// [glGenFencesNV](http://docs.gl/es3/glGenFencesNV)(n, fences)
        /// * `fences` group: FenceNV
        /// * `fences` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        pub unsafe fn GenFencesNV(&self, n: GLsizei, fences: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenFencesNV({:?}, {:p});", n, fences);
            }
            let out = call_atomic_ptr_2arg("glGenFencesNV", &self.glGenFencesNV_p, n, fences);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenFencesNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        #[doc(hidden)]
        pub unsafe fn GenFencesNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGenFencesNV\0", &self.glGenFencesNV_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        pub fn GenFencesNV_is_loaded(&self) -> bool {
            !self.glGenFencesNV_p.load(RELAX).is_null()
        }
        /// [glGenFramebuffers](http://docs.gl/es3/glGenFramebuffers)(n, framebuffers)
        /// * `framebuffers` len: n
        /// * `framebuffers` class: framebuffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenFramebuffers(&self, n: GLsizei, framebuffers: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenFramebuffers({:?}, {:p});", n, framebuffers);
            }
            let out = call_atomic_ptr_2arg(
                "glGenFramebuffers",
                &self.glGenFramebuffers_p,
                n,
                framebuffers,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenFramebuffers");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenFramebuffers_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenFramebuffers\0",
                &self.glGenFramebuffers_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenFramebuffers_is_loaded(&self) -> bool {
            !self.glGenFramebuffers_p.load(RELAX).is_null()
        }
        /// [glGenPathsNV](http://docs.gl/es3/glGenPathsNV)(range)
        /// * return value group: Path
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GenPathsNV(&self, range: GLsizei) -> GLuint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenPathsNV({:?});", range);
            }
            let out = call_atomic_ptr_1arg("glGenPathsNV", &self.glGenPathsNV_p, range);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenPathsNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GenPathsNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGenPathsNV\0", &self.glGenPathsNV_p)
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GenPathsNV_is_loaded(&self) -> bool {
            !self.glGenPathsNV_p.load(RELAX).is_null()
        }
        /// [glGenPerfMonitorsAMD](http://docs.gl/es3/glGenPerfMonitorsAMD)(n, monitors)
        /// * `monitors` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn GenPerfMonitorsAMD(&self, n: GLsizei, monitors: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenPerfMonitorsAMD({:?}, {:p});", n, monitors);
            }
            let out = call_atomic_ptr_2arg(
                "glGenPerfMonitorsAMD",
                &self.glGenPerfMonitorsAMD_p,
                n,
                monitors,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenPerfMonitorsAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn GenPerfMonitorsAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenPerfMonitorsAMD\0",
                &self.glGenPerfMonitorsAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn GenPerfMonitorsAMD_is_loaded(&self) -> bool {
            !self.glGenPerfMonitorsAMD_p.load(RELAX).is_null()
        }
        /// [glGenProgramPipelines](http://docs.gl/es3/glGenProgramPipelines)(n, pipelines)
        /// * `pipelines` len: n
        /// * `pipelines` class: program pipeline
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenProgramPipelines(&self, n: GLsizei, pipelines: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenProgramPipelines({:?}, {:p});", n, pipelines);
            }
            let out = call_atomic_ptr_2arg(
                "glGenProgramPipelines",
                &self.glGenProgramPipelines_p,
                n,
                pipelines,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenProgramPipelines");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenProgramPipelines_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenProgramPipelines\0",
                &self.glGenProgramPipelines_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenProgramPipelines_is_loaded(&self) -> bool {
            !self.glGenProgramPipelines_p.load(RELAX).is_null()
        }
        /// [glGenProgramPipelinesEXT](http://docs.gl/es3/glGenProgramPipelinesEXT)(n, pipelines)
        /// * `pipelines` len: n
        /// * `pipelines` class: program pipeline
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub unsafe fn GenProgramPipelinesEXT(&self, n: GLsizei, pipelines: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GenProgramPipelinesEXT({:?}, {:p});",
                    n,
                    pipelines
                );
            }
            let out = call_atomic_ptr_2arg(
                "glGenProgramPipelinesEXT",
                &self.glGenProgramPipelinesEXT_p,
                n,
                pipelines,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenProgramPipelinesEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        #[doc(hidden)]
        pub unsafe fn GenProgramPipelinesEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenProgramPipelinesEXT\0",
                &self.glGenProgramPipelinesEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub fn GenProgramPipelinesEXT_is_loaded(&self) -> bool {
            !self.glGenProgramPipelinesEXT_p.load(RELAX).is_null()
        }
        /// [glGenQueries](http://docs.gl/es3/glGenQueries)(n, ids)
        /// * `ids` len: n
        /// * `ids` class: query
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenQueries(&self, n: GLsizei, ids: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenQueries({:?}, {:p});", n, ids);
            }
            let out = call_atomic_ptr_2arg("glGenQueries", &self.glGenQueries_p, n, ids);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenQueries");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenQueries_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGenQueries\0", &self.glGenQueries_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenQueries_is_loaded(&self) -> bool {
            !self.glGenQueries_p.load(RELAX).is_null()
        }
        /// [glGenQueriesEXT](http://docs.gl/es3/glGenQueriesEXT)(n, ids)
        /// * `ids` len: n
        /// * `ids` class: query
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub unsafe fn GenQueriesEXT(&self, n: GLsizei, ids: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenQueriesEXT({:?}, {:p});", n, ids);
            }
            let out = call_atomic_ptr_2arg("glGenQueriesEXT", &self.glGenQueriesEXT_p, n, ids);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenQueriesEXT");
            }
            out
        }
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn GenQueriesEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenQueriesEXT\0",
                &self.glGenQueriesEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub fn GenQueriesEXT_is_loaded(&self) -> bool {
            !self.glGenQueriesEXT_p.load(RELAX).is_null()
        }
        /// [glGenRenderbuffers](http://docs.gl/es3/glGenRenderbuffers)(n, renderbuffers)
        /// * `renderbuffers` len: n
        /// * `renderbuffers` class: renderbuffer
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenRenderbuffers(&self, n: GLsizei, renderbuffers: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenRenderbuffers({:?}, {:p});", n, renderbuffers);
            }
            let out = call_atomic_ptr_2arg(
                "glGenRenderbuffers",
                &self.glGenRenderbuffers_p,
                n,
                renderbuffers,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenRenderbuffers");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenRenderbuffers_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenRenderbuffers\0",
                &self.glGenRenderbuffers_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenRenderbuffers_is_loaded(&self) -> bool {
            !self.glGenRenderbuffers_p.load(RELAX).is_null()
        }
        /// [glGenSamplers](http://docs.gl/es3/glGenSamplers)(count, samplers)
        /// * `samplers` len: count
        /// * `samplers` class: sampler
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenSamplers(&self, count: GLsizei, samplers: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenSamplers({:?}, {:p});", count, samplers);
            }
            let out = call_atomic_ptr_2arg("glGenSamplers", &self.glGenSamplers_p, count, samplers);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenSamplers");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenSamplers_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGenSamplers\0", &self.glGenSamplers_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenSamplers_is_loaded(&self) -> bool {
            !self.glGenSamplers_p.load(RELAX).is_null()
        }
        /// [glGenSemaphoresEXT](http://docs.gl/es3/glGenSemaphoresEXT)(n, semaphores)
        /// * `semaphores` len: n
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        pub unsafe fn GenSemaphoresEXT(&self, n: GLsizei, semaphores: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenSemaphoresEXT({:?}, {:p});", n, semaphores);
            }
            let out = call_atomic_ptr_2arg(
                "glGenSemaphoresEXT",
                &self.glGenSemaphoresEXT_p,
                n,
                semaphores,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenSemaphoresEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        #[doc(hidden)]
        pub unsafe fn GenSemaphoresEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenSemaphoresEXT\0",
                &self.glGenSemaphoresEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        pub fn GenSemaphoresEXT_is_loaded(&self) -> bool {
            !self.glGenSemaphoresEXT_p.load(RELAX).is_null()
        }
        /// [glGenTextures](http://docs.gl/es3/glGenTextures)(n, textures)
        /// * `textures` group: Texture
        /// * `textures` len: n
        /// * `textures` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenTextures(&self, n: GLsizei, textures: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenTextures({:?}, {:p});", n, textures);
            }
            let out = call_atomic_ptr_2arg("glGenTextures", &self.glGenTextures_p, n, textures);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenTextures");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenTextures_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGenTextures\0", &self.glGenTextures_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenTextures_is_loaded(&self) -> bool {
            !self.glGenTextures_p.load(RELAX).is_null()
        }
        /// [glGenTransformFeedbacks](http://docs.gl/es3/glGenTransformFeedbacks)(n, ids)
        /// * `ids` len: n
        /// * `ids` class: transform feedback
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenTransformFeedbacks(&self, n: GLsizei, ids: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenTransformFeedbacks({:?}, {:p});", n, ids);
            }
            let out = call_atomic_ptr_2arg(
                "glGenTransformFeedbacks",
                &self.glGenTransformFeedbacks_p,
                n,
                ids,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenTransformFeedbacks");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenTransformFeedbacks_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenTransformFeedbacks\0",
                &self.glGenTransformFeedbacks_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenTransformFeedbacks_is_loaded(&self) -> bool {
            !self.glGenTransformFeedbacks_p.load(RELAX).is_null()
        }
        /// [glGenVertexArrays](http://docs.gl/es3/glGenVertexArrays)(n, arrays)
        /// * `arrays` len: n
        /// * `arrays` class: vertex array
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenVertexArrays(&self, n: GLsizei, arrays: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenVertexArrays({:?}, {:p});", n, arrays);
            }
            let out =
                call_atomic_ptr_2arg("glGenVertexArrays", &self.glGenVertexArrays_p, n, arrays);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenVertexArrays");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenVertexArrays_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenVertexArrays\0",
                &self.glGenVertexArrays_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenVertexArrays_is_loaded(&self) -> bool {
            !self.glGenVertexArrays_p.load(RELAX).is_null()
        }
        /// [glGenVertexArraysOES](http://docs.gl/es3/glGenVertexArraysOES)(n, arrays)
        /// * `arrays` len: n
        /// * `arrays` class: vertex array
        /// * alias of: [`glGenVertexArrays`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        pub unsafe fn GenVertexArraysOES(&self, n: GLsizei, arrays: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenVertexArraysOES({:?}, {:p});", n, arrays);
            }
            let out = call_atomic_ptr_2arg(
                "glGenVertexArraysOES",
                &self.glGenVertexArraysOES_p,
                n,
                arrays,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenVertexArraysOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        #[doc(hidden)]
        pub unsafe fn GenVertexArraysOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenVertexArraysOES\0",
                &self.glGenVertexArraysOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        pub fn GenVertexArraysOES_is_loaded(&self) -> bool {
            !self.glGenVertexArraysOES_p.load(RELAX).is_null()
        }
        /// [glGenerateMipmap](http://docs.gl/es3/glGenerateMipmap)(target)
        /// * `target` group: TextureTarget
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GenerateMipmap(&self, target: GLenum) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GenerateMipmap({:#X});", target);
            }
            let out = call_atomic_ptr_1arg("glGenerateMipmap", &self.glGenerateMipmap_p, target);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGenerateMipmap");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GenerateMipmap_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGenerateMipmap\0",
                &self.glGenerateMipmap_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GenerateMipmap_is_loaded(&self) -> bool {
            !self.glGenerateMipmap_p.load(RELAX).is_null()
        }
        /// [glGetActiveAttrib](http://docs.gl/es3/glGetActiveAttrib)(program, index, bufSize, length, size, type_, name)
        /// * `program` class: program
        /// * `length` len: 1
        /// * `size` len: 1
        /// * `type_` group: AttributeType
        /// * `type_` len: 1
        /// * `name` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetActiveAttrib(
            &self,
            program: GLuint,
            index: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            size: *mut GLint,
            type_: *mut GLenum,
            name: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetActiveAttrib({:?}, {:?}, {:?}, {:p}, {:p}, {:p}, {:p});",
                    program,
                    index,
                    bufSize,
                    length,
                    size,
                    type_,
                    name
                );
            }
            let out = call_atomic_ptr_7arg(
                "glGetActiveAttrib",
                &self.glGetActiveAttrib_p,
                program,
                index,
                bufSize,
                length,
                size,
                type_,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetActiveAttrib");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetActiveAttrib_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetActiveAttrib\0",
                &self.glGetActiveAttrib_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetActiveAttrib_is_loaded(&self) -> bool {
            !self.glGetActiveAttrib_p.load(RELAX).is_null()
        }
        /// [glGetActiveUniform](http://docs.gl/es3/glGetActiveUniform)(program, index, bufSize, length, size, type_, name)
        /// * `program` class: program
        /// * `length` len: 1
        /// * `size` len: 1
        /// * `type_` group: UniformType
        /// * `type_` len: 1
        /// * `name` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetActiveUniform(
            &self,
            program: GLuint,
            index: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            size: *mut GLint,
            type_: *mut GLenum,
            name: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetActiveUniform({:?}, {:?}, {:?}, {:p}, {:p}, {:p}, {:p});",
                    program,
                    index,
                    bufSize,
                    length,
                    size,
                    type_,
                    name
                );
            }
            let out = call_atomic_ptr_7arg(
                "glGetActiveUniform",
                &self.glGetActiveUniform_p,
                program,
                index,
                bufSize,
                length,
                size,
                type_,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetActiveUniform");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetActiveUniform_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetActiveUniform\0",
                &self.glGetActiveUniform_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetActiveUniform_is_loaded(&self) -> bool {
            !self.glGetActiveUniform_p.load(RELAX).is_null()
        }
        /// [glGetActiveUniformBlockName](http://docs.gl/es3/glGetActiveUniformBlockName)(program, uniformBlockIndex, bufSize, length, uniformBlockName)
        /// * `program` class: program
        /// * `length` len: 1
        /// * `uniformBlockName` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetActiveUniformBlockName(
            &self,
            program: GLuint,
            uniformBlockIndex: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            uniformBlockName: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetActiveUniformBlockName({:?}, {:?}, {:?}, {:p}, {:p});",
                    program,
                    uniformBlockIndex,
                    bufSize,
                    length,
                    uniformBlockName
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetActiveUniformBlockName",
                &self.glGetActiveUniformBlockName_p,
                program,
                uniformBlockIndex,
                bufSize,
                length,
                uniformBlockName,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetActiveUniformBlockName");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetActiveUniformBlockName_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetActiveUniformBlockName\0",
                &self.glGetActiveUniformBlockName_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetActiveUniformBlockName_is_loaded(&self) -> bool {
            !self.glGetActiveUniformBlockName_p.load(RELAX).is_null()
        }
        /// [glGetActiveUniformBlockiv](http://docs.gl/es3/glGetActiveUniformBlockiv)(program, uniformBlockIndex, pname, params)
        /// * `program` class: program
        /// * `pname` group: UniformBlockPName
        /// * `params` len: COMPSIZE(program,uniformBlockIndex,pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetActiveUniformBlockiv(
            &self,
            program: GLuint,
            uniformBlockIndex: GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetActiveUniformBlockiv({:?}, {:?}, {:#X}, {:p});",
                    program,
                    uniformBlockIndex,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetActiveUniformBlockiv",
                &self.glGetActiveUniformBlockiv_p,
                program,
                uniformBlockIndex,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetActiveUniformBlockiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetActiveUniformBlockiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetActiveUniformBlockiv\0",
                &self.glGetActiveUniformBlockiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetActiveUniformBlockiv_is_loaded(&self) -> bool {
            !self.glGetActiveUniformBlockiv_p.load(RELAX).is_null()
        }
        /// [glGetActiveUniformsiv](http://docs.gl/es3/glGetActiveUniformsiv)(program, uniformCount, uniformIndices, pname, params)
        /// * `program` class: program
        /// * `uniformIndices` len: uniformCount
        /// * `pname` group: UniformPName
        /// * `params` len: COMPSIZE(uniformCount,pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetActiveUniformsiv(
            &self,
            program: GLuint,
            uniformCount: GLsizei,
            uniformIndices: *const GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetActiveUniformsiv({:?}, {:?}, {:p}, {:#X}, {:p});",
                    program,
                    uniformCount,
                    uniformIndices,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetActiveUniformsiv",
                &self.glGetActiveUniformsiv_p,
                program,
                uniformCount,
                uniformIndices,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetActiveUniformsiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetActiveUniformsiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetActiveUniformsiv\0",
                &self.glGetActiveUniformsiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetActiveUniformsiv_is_loaded(&self) -> bool {
            !self.glGetActiveUniformsiv_p.load(RELAX).is_null()
        }
        /// [glGetAttachedShaders](http://docs.gl/es3/glGetAttachedShaders)(program, maxCount, count, shaders)
        /// * `program` class: program
        /// * `count` len: 1
        /// * `shaders` len: maxCount
        /// * `shaders` class: shader
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetAttachedShaders(
            &self,
            program: GLuint,
            maxCount: GLsizei,
            count: *mut GLsizei,
            shaders: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetAttachedShaders({:?}, {:?}, {:p}, {:p});",
                    program,
                    maxCount,
                    count,
                    shaders
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetAttachedShaders",
                &self.glGetAttachedShaders_p,
                program,
                maxCount,
                count,
                shaders,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetAttachedShaders");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetAttachedShaders_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetAttachedShaders\0",
                &self.glGetAttachedShaders_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetAttachedShaders_is_loaded(&self) -> bool {
            !self.glGetAttachedShaders_p.load(RELAX).is_null()
        }
        /// [glGetAttribLocation](http://docs.gl/es3/glGetAttribLocation)(program, name)
        /// * `program` class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetAttribLocation(&self, program: GLuint, name: *const GLchar) -> GLint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetAttribLocation({:?}, {:p});", program, name);
            }
            let out = call_atomic_ptr_2arg(
                "glGetAttribLocation",
                &self.glGetAttribLocation_p,
                program,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetAttribLocation");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetAttribLocation_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetAttribLocation\0",
                &self.glGetAttribLocation_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetAttribLocation_is_loaded(&self) -> bool {
            !self.glGetAttribLocation_p.load(RELAX).is_null()
        }
        /// [glGetBooleani_v](http://docs.gl/es3/glGet)(target, index, data)
        /// * `target` group: BufferTargetARB
        /// * `data` len: COMPSIZE(target)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetBooleani_v(&self, target: GLenum, index: GLuint, data: *mut GLboolean) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetBooleani_v({:#X}, {:?}, {:p});",
                    target,
                    index,
                    data
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetBooleani_v",
                &self.glGetBooleani_v_p,
                target,
                index,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetBooleani_v");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetBooleani_v_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetBooleani_v\0",
                &self.glGetBooleani_v_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetBooleani_v_is_loaded(&self) -> bool {
            !self.glGetBooleani_v_p.load(RELAX).is_null()
        }
        /// [glGetBooleanv](http://docs.gl/es3/glGet)(pname, data)
        /// * `pname` group: GetPName
        /// * `data` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetBooleanv(&self, pname: GLenum, data: *mut GLboolean) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetBooleanv({:#X}, {:p});", pname, data);
            }
            let out = call_atomic_ptr_2arg("glGetBooleanv", &self.glGetBooleanv_p, pname, data);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetBooleanv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetBooleanv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetBooleanv\0", &self.glGetBooleanv_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetBooleanv_is_loaded(&self) -> bool {
            !self.glGetBooleanv_p.load(RELAX).is_null()
        }
        /// [glGetBufferParameteri64v](http://docs.gl/es3/glGetBufferParameter)(target, pname, params)
        /// * `target` group: BufferTargetARB
        /// * `pname` group: BufferPNameARB
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetBufferParameteri64v(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLint64,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetBufferParameteri64v({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetBufferParameteri64v",
                &self.glGetBufferParameteri64v_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetBufferParameteri64v");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetBufferParameteri64v_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetBufferParameteri64v\0",
                &self.glGetBufferParameteri64v_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetBufferParameteri64v_is_loaded(&self) -> bool {
            !self.glGetBufferParameteri64v_p.load(RELAX).is_null()
        }
        /// [glGetBufferParameteriv](http://docs.gl/es3/glGetBufferParameter)(target, pname, params)
        /// * `target` group: BufferTargetARB
        /// * `pname` group: BufferPNameARB
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetBufferParameteriv(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetBufferParameteriv({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetBufferParameteriv",
                &self.glGetBufferParameteriv_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetBufferParameteriv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetBufferParameteriv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetBufferParameteriv\0",
                &self.glGetBufferParameteriv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetBufferParameteriv_is_loaded(&self) -> bool {
            !self.glGetBufferParameteriv_p.load(RELAX).is_null()
        }
        /// [glGetBufferPointerv](http://docs.gl/es3/glGetBufferPointerv)(target, pname, params)
        /// * `target` group: BufferTargetARB
        /// * `pname` group: BufferPointerNameARB
        /// * `params` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetBufferPointerv(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut *mut c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetBufferPointerv({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetBufferPointerv",
                &self.glGetBufferPointerv_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetBufferPointerv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetBufferPointerv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetBufferPointerv\0",
                &self.glGetBufferPointerv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetBufferPointerv_is_loaded(&self) -> bool {
            !self.glGetBufferPointerv_p.load(RELAX).is_null()
        }
        /// [glGetBufferPointervOES](http://docs.gl/es3/glGetBufferPointervOES)(target, pname, params)
        /// * `target` group: BufferTargetARB
        /// * `pname` group: BufferPointerNameARB
        /// * `params` len: 1
        /// * alias of: [`glGetBufferPointerv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_mapbuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
        pub unsafe fn GetBufferPointervOES(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut *mut c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetBufferPointervOES({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetBufferPointervOES",
                &self.glGetBufferPointervOES_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetBufferPointervOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_mapbuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
        #[doc(hidden)]
        pub unsafe fn GetBufferPointervOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetBufferPointervOES\0",
                &self.glGetBufferPointervOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_mapbuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
        pub fn GetBufferPointervOES_is_loaded(&self) -> bool {
            !self.glGetBufferPointervOES_p.load(RELAX).is_null()
        }
        /// [glGetCoverageModulationTableNV](http://docs.gl/es3/glGetCoverageModulationTableNV)(bufSize, v)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        pub unsafe fn GetCoverageModulationTableNV(&self, bufSize: GLsizei, v: *mut GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetCoverageModulationTableNV({:?}, {:p});",
                    bufSize,
                    v
                );
            }
            let out = call_atomic_ptr_2arg(
                "glGetCoverageModulationTableNV",
                &self.glGetCoverageModulationTableNV_p,
                bufSize,
                v,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetCoverageModulationTableNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        #[doc(hidden)]
        pub unsafe fn GetCoverageModulationTableNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetCoverageModulationTableNV\0",
                &self.glGetCoverageModulationTableNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        pub fn GetCoverageModulationTableNV_is_loaded(&self) -> bool {
            !self.glGetCoverageModulationTableNV_p.load(RELAX).is_null()
        }
        /// [glGetDebugMessageLog](http://docs.gl/es3/glGetDebugMessageLog)(count, bufSize, sources, types, ids, severities, lengths, messageLog)
        /// * `sources` group: DebugSource
        /// * `sources` len: count
        /// * `types` group: DebugType
        /// * `types` len: count
        /// * `ids` len: count
        /// * `severities` group: DebugSeverity
        /// * `severities` len: count
        /// * `lengths` len: count
        /// * `messageLog` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetDebugMessageLog(
            &self,
            count: GLuint,
            bufSize: GLsizei,
            sources: *mut GLenum,
            types: *mut GLenum,
            ids: *mut GLuint,
            severities: *mut GLenum,
            lengths: *mut GLsizei,
            messageLog: *mut GLchar,
        ) -> GLuint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetDebugMessageLog({:?}, {:?}, {:p}, {:p}, {:p}, {:p}, {:p}, {:p});",
                    count,
                    bufSize,
                    sources,
                    types,
                    ids,
                    severities,
                    lengths,
                    messageLog
                );
            }
            let out = call_atomic_ptr_8arg(
                "glGetDebugMessageLog",
                &self.glGetDebugMessageLog_p,
                count,
                bufSize,
                sources,
                types,
                ids,
                severities,
                lengths,
                messageLog,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetDebugMessageLog");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetDebugMessageLog_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetDebugMessageLog\0",
                &self.glGetDebugMessageLog_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetDebugMessageLog_is_loaded(&self) -> bool {
            !self.glGetDebugMessageLog_p.load(RELAX).is_null()
        }
        /// [glGetDebugMessageLogKHR](http://docs.gl/es3/glGetDebugMessageLogKHR)(count, bufSize, sources, types, ids, severities, lengths, messageLog)
        /// * `sources` group: DebugSource
        /// * `sources` len: count
        /// * `types` group: DebugType
        /// * `types` len: count
        /// * `ids` len: count
        /// * `severities` group: DebugSeverity
        /// * `severities` len: count
        /// * `lengths` len: count
        /// * `messageLog` len: bufSize
        /// * alias of: [`glGetDebugMessageLog`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub unsafe fn GetDebugMessageLogKHR(
            &self,
            count: GLuint,
            bufSize: GLsizei,
            sources: *mut GLenum,
            types: *mut GLenum,
            ids: *mut GLuint,
            severities: *mut GLenum,
            lengths: *mut GLsizei,
            messageLog: *mut GLchar,
        ) -> GLuint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetDebugMessageLogKHR({:?}, {:?}, {:p}, {:p}, {:p}, {:p}, {:p}, {:p});",
                    count,
                    bufSize,
                    sources,
                    types,
                    ids,
                    severities,
                    lengths,
                    messageLog
                );
            }
            let out = call_atomic_ptr_8arg(
                "glGetDebugMessageLogKHR",
                &self.glGetDebugMessageLogKHR_p,
                count,
                bufSize,
                sources,
                types,
                ids,
                severities,
                lengths,
                messageLog,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetDebugMessageLogKHR");
            }
            out
        }
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        #[doc(hidden)]
        pub unsafe fn GetDebugMessageLogKHR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetDebugMessageLogKHR\0",
                &self.glGetDebugMessageLogKHR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub fn GetDebugMessageLogKHR_is_loaded(&self) -> bool {
            !self.glGetDebugMessageLogKHR_p.load(RELAX).is_null()
        }
        /// [glGetDriverControlStringQCOM](http://docs.gl/es3/glGetDriverControlStringQCOM)(driverControl, bufSize, length, driverControlString)
        /// * `driverControlString` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        pub unsafe fn GetDriverControlStringQCOM(
            &self,
            driverControl: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            driverControlString: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetDriverControlStringQCOM({:?}, {:?}, {:p}, {:p});",
                    driverControl,
                    bufSize,
                    length,
                    driverControlString
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetDriverControlStringQCOM",
                &self.glGetDriverControlStringQCOM_p,
                driverControl,
                bufSize,
                length,
                driverControlString,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetDriverControlStringQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        #[doc(hidden)]
        pub unsafe fn GetDriverControlStringQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetDriverControlStringQCOM\0",
                &self.glGetDriverControlStringQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        pub fn GetDriverControlStringQCOM_is_loaded(&self) -> bool {
            !self.glGetDriverControlStringQCOM_p.load(RELAX).is_null()
        }
        /// [glGetDriverControlsQCOM](http://docs.gl/es3/glGetDriverControlsQCOM)(num, size, driverControls)
        /// * `driverControls` len: size
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        pub unsafe fn GetDriverControlsQCOM(
            &self,
            num: *mut GLint,
            size: GLsizei,
            driverControls: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetDriverControlsQCOM({:p}, {:?}, {:p});",
                    num,
                    size,
                    driverControls
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetDriverControlsQCOM",
                &self.glGetDriverControlsQCOM_p,
                num,
                size,
                driverControls,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetDriverControlsQCOM");
            }
            out
        }
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        #[doc(hidden)]
        pub unsafe fn GetDriverControlsQCOM_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetDriverControlsQCOM\0",
                &self.glGetDriverControlsQCOM_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        pub fn GetDriverControlsQCOM_is_loaded(&self) -> bool {
            !self.glGetDriverControlsQCOM_p.load(RELAX).is_null()
        }
        /// [glGetError](http://docs.gl/es3/glGetError)()
        /// * return value group: ErrorCode
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetError(&self) -> GLenum {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetError();",);
            }
            let out = call_atomic_ptr_0arg("glGetError", &self.glGetError_p);

            out
        }
        #[doc(hidden)]
        pub unsafe fn GetError_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetError\0", &self.glGetError_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetError_is_loaded(&self) -> bool {
            !self.glGetError_p.load(RELAX).is_null()
        }
        /// [glGetFenceivNV](http://docs.gl/es3/glGetFenceivNV)(fence, pname, params)
        /// * `fence` group: FenceNV
        /// * `pname` group: FenceParameterNameNV
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        pub unsafe fn GetFenceivNV(&self, fence: GLuint, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetFenceivNV({:?}, {:#X}, {:p});",
                    fence,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetFenceivNV",
                &self.glGetFenceivNV_p,
                fence,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFenceivNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        #[doc(hidden)]
        pub unsafe fn GetFenceivNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFenceivNV\0",
                &self.glGetFenceivNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        pub fn GetFenceivNV_is_loaded(&self) -> bool {
            !self.glGetFenceivNV_p.load(RELAX).is_null()
        }
        /// [glGetFirstPerfQueryIdINTEL](http://docs.gl/es3/glGetFirstPerfQueryIdINTEL)(queryId)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn GetFirstPerfQueryIdINTEL(&self, queryId: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetFirstPerfQueryIdINTEL({:p});", queryId);
            }
            let out = call_atomic_ptr_1arg(
                "glGetFirstPerfQueryIdINTEL",
                &self.glGetFirstPerfQueryIdINTEL_p,
                queryId,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFirstPerfQueryIdINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetFirstPerfQueryIdINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFirstPerfQueryIdINTEL\0",
                &self.glGetFirstPerfQueryIdINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn GetFirstPerfQueryIdINTEL_is_loaded(&self) -> bool {
            !self.glGetFirstPerfQueryIdINTEL_p.load(RELAX).is_null()
        }
        /// [glGetFloati_vNV](http://docs.gl/es3/glGetFloati_vNV)(target, index, data)
        /// * `target` group: GetPName
        /// * `data` len: COMPSIZE(target)
        /// * alias of: [`glGetFloati_v`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub unsafe fn GetFloati_vNV(&self, target: GLenum, index: GLuint, data: *mut GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetFloati_vNV({:#X}, {:?}, {:p});",
                    target,
                    index,
                    data
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetFloati_vNV",
                &self.glGetFloati_vNV_p,
                target,
                index,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFloati_vNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        #[doc(hidden)]
        pub unsafe fn GetFloati_vNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFloati_vNV\0",
                &self.glGetFloati_vNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        pub fn GetFloati_vNV_is_loaded(&self) -> bool {
            !self.glGetFloati_vNV_p.load(RELAX).is_null()
        }
        /// [glGetFloati_vOES](http://docs.gl/es3/glGetFloati_vOES)(target, index, data)
        /// * `target` group: GetPName
        /// * `data` len: COMPSIZE(target)
        /// * alias of: [`glGetFloati_v`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        pub unsafe fn GetFloati_vOES(&self, target: GLenum, index: GLuint, data: *mut GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetFloati_vOES({:#X}, {:?}, {:p});",
                    target,
                    index,
                    data
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetFloati_vOES",
                &self.glGetFloati_vOES_p,
                target,
                index,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFloati_vOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        #[doc(hidden)]
        pub unsafe fn GetFloati_vOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFloati_vOES\0",
                &self.glGetFloati_vOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        pub fn GetFloati_vOES_is_loaded(&self) -> bool {
            !self.glGetFloati_vOES_p.load(RELAX).is_null()
        }
        /// [glGetFloatv](http://docs.gl/es3/glGet)(pname, data)
        /// * `pname` group: GetPName
        /// * `data` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetFloatv(&self, pname: GLenum, data: *mut GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetFloatv({:#X}, {:p});", pname, data);
            }
            let out = call_atomic_ptr_2arg("glGetFloatv", &self.glGetFloatv_p, pname, data);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFloatv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetFloatv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetFloatv\0", &self.glGetFloatv_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetFloatv_is_loaded(&self) -> bool {
            !self.glGetFloatv_p.load(RELAX).is_null()
        }
        /// [glGetFragDataIndexEXT](http://docs.gl/es3/glGetFragDataIndexEXT)(program, name)
        /// * `program` class: program
        /// * alias of: [`glGetFragDataIndex`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        pub unsafe fn GetFragDataIndexEXT(&self, program: GLuint, name: *const GLchar) -> GLint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetFragDataIndexEXT({:?}, {:p});", program, name);
            }
            let out = call_atomic_ptr_2arg(
                "glGetFragDataIndexEXT",
                &self.glGetFragDataIndexEXT_p,
                program,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFragDataIndexEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        #[doc(hidden)]
        pub unsafe fn GetFragDataIndexEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFragDataIndexEXT\0",
                &self.glGetFragDataIndexEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        pub fn GetFragDataIndexEXT_is_loaded(&self) -> bool {
            !self.glGetFragDataIndexEXT_p.load(RELAX).is_null()
        }
        /// [glGetFragDataLocation](http://docs.gl/es3/glGetFragDataLocation)(program, name)
        /// * `program` class: program
        /// * `name` len: COMPSIZE(name)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetFragDataLocation(&self, program: GLuint, name: *const GLchar) -> GLint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetFragDataLocation({:?}, {:p});", program, name);
            }
            let out = call_atomic_ptr_2arg(
                "glGetFragDataLocation",
                &self.glGetFragDataLocation_p,
                program,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFragDataLocation");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetFragDataLocation_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFragDataLocation\0",
                &self.glGetFragDataLocation_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetFragDataLocation_is_loaded(&self) -> bool {
            !self.glGetFragDataLocation_p.load(RELAX).is_null()
        }
        /// [glGetFramebufferAttachmentParameteriv](http://docs.gl/es3/glGetFramebufferAttachmentParameter)(target, attachment, pname, params)
        /// * `target` group: FramebufferTarget
        /// * `attachment` group: FramebufferAttachment
        /// * `pname` group: FramebufferAttachmentParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetFramebufferAttachmentParameteriv(
            &self,
            target: GLenum,
            attachment: GLenum,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetFramebufferAttachmentParameteriv({:#X}, {:#X}, {:#X}, {:p});",
                    target,
                    attachment,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetFramebufferAttachmentParameteriv",
                &self.glGetFramebufferAttachmentParameteriv_p,
                target,
                attachment,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFramebufferAttachmentParameteriv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetFramebufferAttachmentParameteriv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFramebufferAttachmentParameteriv\0",
                &self.glGetFramebufferAttachmentParameteriv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetFramebufferAttachmentParameteriv_is_loaded(&self) -> bool {
            !self
                .glGetFramebufferAttachmentParameteriv_p
                .load(RELAX)
                .is_null()
        }
        /// [glGetFramebufferParameteriv](http://docs.gl/es3/glGetFramebufferParameter)(target, pname, params)
        /// * `target` group: FramebufferTarget
        /// * `pname` group: FramebufferAttachmentParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetFramebufferParameteriv(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetFramebufferParameteriv({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetFramebufferParameteriv",
                &self.glGetFramebufferParameteriv_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFramebufferParameteriv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetFramebufferParameteriv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFramebufferParameteriv\0",
                &self.glGetFramebufferParameteriv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetFramebufferParameteriv_is_loaded(&self) -> bool {
            !self.glGetFramebufferParameteriv_p.load(RELAX).is_null()
        }
        /// [glGetFramebufferParameterivMESA](http://docs.gl/es3/glGetFramebufferParameterivMESA)(target, pname, params)
        /// * `target` group: FramebufferTarget
        /// * `pname` group: FramebufferAttachmentParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
        pub unsafe fn GetFramebufferParameterivMESA(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetFramebufferParameterivMESA({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetFramebufferParameterivMESA",
                &self.glGetFramebufferParameterivMESA_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFramebufferParameterivMESA");
            }
            out
        }
        #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
        #[doc(hidden)]
        pub unsafe fn GetFramebufferParameterivMESA_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFramebufferParameterivMESA\0",
                &self.glGetFramebufferParameterivMESA_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
        pub fn GetFramebufferParameterivMESA_is_loaded(&self) -> bool {
            !self.glGetFramebufferParameterivMESA_p.load(RELAX).is_null()
        }
        /// [glGetFramebufferPixelLocalStorageSizeEXT](http://docs.gl/es3/glGetFramebufferPixelLocalStorageSizeEXT)(target)
        /// * `target` group: FramebufferTarget
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        pub unsafe fn GetFramebufferPixelLocalStorageSizeEXT(&self, target: GLuint) -> GLsizei {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetFramebufferPixelLocalStorageSizeEXT({:?});",
                    target
                );
            }
            let out = call_atomic_ptr_1arg(
                "glGetFramebufferPixelLocalStorageSizeEXT",
                &self.glGetFramebufferPixelLocalStorageSizeEXT_p,
                target,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetFramebufferPixelLocalStorageSizeEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        #[doc(hidden)]
        pub unsafe fn GetFramebufferPixelLocalStorageSizeEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetFramebufferPixelLocalStorageSizeEXT\0",
                &self.glGetFramebufferPixelLocalStorageSizeEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        pub fn GetFramebufferPixelLocalStorageSizeEXT_is_loaded(&self) -> bool {
            !self
                .glGetFramebufferPixelLocalStorageSizeEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glGetGraphicsResetStatus](http://docs.gl/es3/glGetGraphicsResetStatus)()
        /// * return value group: GraphicsResetStatus
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetGraphicsResetStatus(&self) -> GLenum {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetGraphicsResetStatus();",);
            }
            let out =
                call_atomic_ptr_0arg("glGetGraphicsResetStatus", &self.glGetGraphicsResetStatus_p);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetGraphicsResetStatus");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetGraphicsResetStatus_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetGraphicsResetStatus\0",
                &self.glGetGraphicsResetStatus_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetGraphicsResetStatus_is_loaded(&self) -> bool {
            !self.glGetGraphicsResetStatus_p.load(RELAX).is_null()
        }
        /// [glGetGraphicsResetStatusEXT](http://docs.gl/es3/glGetGraphicsResetStatusEXT)()
        /// * return value group: GraphicsResetStatus
        /// * alias of: [`glGetGraphicsResetStatus`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        pub unsafe fn GetGraphicsResetStatusEXT(&self) -> GLenum {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetGraphicsResetStatusEXT();",);
            }
            let out = call_atomic_ptr_0arg(
                "glGetGraphicsResetStatusEXT",
                &self.glGetGraphicsResetStatusEXT_p,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetGraphicsResetStatusEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        #[doc(hidden)]
        pub unsafe fn GetGraphicsResetStatusEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetGraphicsResetStatusEXT\0",
                &self.glGetGraphicsResetStatusEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        pub fn GetGraphicsResetStatusEXT_is_loaded(&self) -> bool {
            !self.glGetGraphicsResetStatusEXT_p.load(RELAX).is_null()
        }
        /// [glGetGraphicsResetStatusKHR](http://docs.gl/es3/glGetGraphicsResetStatusKHR)()
        /// * return value group: GraphicsResetStatus
        /// * alias of: [`glGetGraphicsResetStatus`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_KHR_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
        pub unsafe fn GetGraphicsResetStatusKHR(&self) -> GLenum {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetGraphicsResetStatusKHR();",);
            }
            let out = call_atomic_ptr_0arg(
                "glGetGraphicsResetStatusKHR",
                &self.glGetGraphicsResetStatusKHR_p,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetGraphicsResetStatusKHR");
            }
            out
        }
        #[cfg(any(feature = "GL_KHR_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
        #[doc(hidden)]
        pub unsafe fn GetGraphicsResetStatusKHR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetGraphicsResetStatusKHR\0",
                &self.glGetGraphicsResetStatusKHR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_KHR_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
        pub fn GetGraphicsResetStatusKHR_is_loaded(&self) -> bool {
            !self.glGetGraphicsResetStatusKHR_p.load(RELAX).is_null()
        }
        /// [glGetImageHandleNV](http://docs.gl/es3/glGetImageHandleNV)(texture, level, layered, layer, format)
        /// * `texture` class: texture
        /// * `format` group: PixelFormat
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        pub unsafe fn GetImageHandleNV(
            &self,
            texture: GLuint,
            level: GLint,
            layered: GLboolean,
            layer: GLint,
            format: GLenum,
        ) -> GLuint64 {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetImageHandleNV({:?}, {:?}, {:?}, {:?}, {:#X});",
                    texture,
                    level,
                    layered,
                    layer,
                    format
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetImageHandleNV",
                &self.glGetImageHandleNV_p,
                texture,
                level,
                layered,
                layer,
                format,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetImageHandleNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        #[doc(hidden)]
        pub unsafe fn GetImageHandleNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetImageHandleNV\0",
                &self.glGetImageHandleNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        pub fn GetImageHandleNV_is_loaded(&self) -> bool {
            !self.glGetImageHandleNV_p.load(RELAX).is_null()
        }
        /// [glGetInteger64i_v](http://docs.gl/es3/glGet)(target, index, data)
        /// * `target` group: GetPName
        /// * `data` len: COMPSIZE(target)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetInteger64i_v(&self, target: GLenum, index: GLuint, data: *mut GLint64) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetInteger64i_v({:#X}, {:?}, {:p});",
                    target,
                    index,
                    data
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetInteger64i_v",
                &self.glGetInteger64i_v_p,
                target,
                index,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetInteger64i_v");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetInteger64i_v_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetInteger64i_v\0",
                &self.glGetInteger64i_v_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetInteger64i_v_is_loaded(&self) -> bool {
            !self.glGetInteger64i_v_p.load(RELAX).is_null()
        }
        /// [glGetInteger64v](http://docs.gl/es3/glGet)(pname, data)
        /// * `pname` group: GetPName
        /// * `data` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetInteger64v(&self, pname: GLenum, data: *mut GLint64) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetInteger64v({:#X}, {:p});", pname, data);
            }
            let out = call_atomic_ptr_2arg("glGetInteger64v", &self.glGetInteger64v_p, pname, data);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetInteger64v");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetInteger64v_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetInteger64v\0",
                &self.glGetInteger64v_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetInteger64v_is_loaded(&self) -> bool {
            !self.glGetInteger64v_p.load(RELAX).is_null()
        }
        /// [glGetInteger64vAPPLE](http://docs.gl/es3/glGetInteger64vAPPLE)(pname, params)
        /// * `pname` group: GetPName
        /// * alias of: [`glGetInteger64v`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub unsafe fn GetInteger64vAPPLE(&self, pname: GLenum, params: *mut GLint64) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetInteger64vAPPLE({:#X}, {:p});", pname, params);
            }
            let out = call_atomic_ptr_2arg(
                "glGetInteger64vAPPLE",
                &self.glGetInteger64vAPPLE_p,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetInteger64vAPPLE");
            }
            out
        }
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        #[doc(hidden)]
        pub unsafe fn GetInteger64vAPPLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetInteger64vAPPLE\0",
                &self.glGetInteger64vAPPLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub fn GetInteger64vAPPLE_is_loaded(&self) -> bool {
            !self.glGetInteger64vAPPLE_p.load(RELAX).is_null()
        }
        /// [glGetInteger64vEXT](http://docs.gl/es3/glGetInteger64vEXT)(pname, data)
        /// * `pname` group: GetPName
        /// * `data` len: COMPSIZE(pname)
        /// * alias of: [`glGetInteger64v`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        pub unsafe fn GetInteger64vEXT(&self, pname: GLenum, data: *mut GLint64) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetInteger64vEXT({:#X}, {:p});", pname, data);
            }
            let out = call_atomic_ptr_2arg(
                "glGetInteger64vEXT",
                &self.glGetInteger64vEXT_p,
                pname,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetInteger64vEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetInteger64vEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetInteger64vEXT\0",
                &self.glGetInteger64vEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        pub fn GetInteger64vEXT_is_loaded(&self) -> bool {
            !self.glGetInteger64vEXT_p.load(RELAX).is_null()
        }
        /// [glGetIntegeri_v](http://docs.gl/es3/glGet)(target, index, data)
        /// * `target` group: GetPName
        /// * `data` len: COMPSIZE(target)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetIntegeri_v(&self, target: GLenum, index: GLuint, data: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetIntegeri_v({:#X}, {:?}, {:p});",
                    target,
                    index,
                    data
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetIntegeri_v",
                &self.glGetIntegeri_v_p,
                target,
                index,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetIntegeri_v");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetIntegeri_v_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetIntegeri_v\0",
                &self.glGetIntegeri_v_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetIntegeri_v_is_loaded(&self) -> bool {
            !self.glGetIntegeri_v_p.load(RELAX).is_null()
        }
        /// [glGetIntegeri_vEXT](http://docs.gl/es3/glGetIntegeri_vEXT)(target, index, data)
        /// * `target` group: GetPName
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
        pub unsafe fn GetIntegeri_vEXT(&self, target: GLenum, index: GLuint, data: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetIntegeri_vEXT({:#X}, {:?}, {:p});",
                    target,
                    index,
                    data
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetIntegeri_vEXT",
                &self.glGetIntegeri_vEXT_p,
                target,
                index,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetIntegeri_vEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
        #[doc(hidden)]
        pub unsafe fn GetIntegeri_vEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetIntegeri_vEXT\0",
                &self.glGetIntegeri_vEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
        pub fn GetIntegeri_vEXT_is_loaded(&self) -> bool {
            !self.glGetIntegeri_vEXT_p.load(RELAX).is_null()
        }
        /// [glGetIntegerv](http://docs.gl/es3/glGet)(pname, data)
        /// * `pname` group: GetPName
        /// * `data` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetIntegerv(&self, pname: GLenum, data: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetIntegerv({:#X}, {:p});", pname, data);
            }
            let out = call_atomic_ptr_2arg("glGetIntegerv", &self.glGetIntegerv_p, pname, data);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetIntegerv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetIntegerv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetIntegerv\0", &self.glGetIntegerv_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetIntegerv_is_loaded(&self) -> bool {
            !self.glGetIntegerv_p.load(RELAX).is_null()
        }
        /// [glGetInternalformatSampleivNV](http://docs.gl/es3/glGetInternalformatSampleivNV)(target, internalformat, samples, pname, count, params)
        /// * `target` group: TextureTarget
        /// * `internalformat` group: InternalFormat
        /// * `pname` group: InternalFormatPName
        /// * `params` len: count
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_internalformat_sample_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_internalformat_sample_query"))))]
        pub unsafe fn GetInternalformatSampleivNV(
            &self,
            target: GLenum,
            internalformat: GLenum,
            samples: GLsizei,
            pname: GLenum,
            count: GLsizei,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetInternalformatSampleivNV({:#X}, {:#X}, {:?}, {:#X}, {:?}, {:p});",
                    target,
                    internalformat,
                    samples,
                    pname,
                    count,
                    params
                );
            }
            let out = call_atomic_ptr_6arg(
                "glGetInternalformatSampleivNV",
                &self.glGetInternalformatSampleivNV_p,
                target,
                internalformat,
                samples,
                pname,
                count,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetInternalformatSampleivNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_internalformat_sample_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_internalformat_sample_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetInternalformatSampleivNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetInternalformatSampleivNV\0",
                &self.glGetInternalformatSampleivNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_internalformat_sample_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_internalformat_sample_query"))))]
        pub fn GetInternalformatSampleivNV_is_loaded(&self) -> bool {
            !self.glGetInternalformatSampleivNV_p.load(RELAX).is_null()
        }
        /// [glGetInternalformativ](http://docs.gl/es3/glGetInternalformativ)(target, internalformat, pname, count, params)
        /// * `target` group: TextureTarget
        /// * `internalformat` group: InternalFormat
        /// * `pname` group: InternalFormatPName
        /// * `params` len: count
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetInternalformativ(
            &self,
            target: GLenum,
            internalformat: GLenum,
            pname: GLenum,
            count: GLsizei,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetInternalformativ({:#X}, {:#X}, {:#X}, {:?}, {:p});",
                    target,
                    internalformat,
                    pname,
                    count,
                    params
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetInternalformativ",
                &self.glGetInternalformativ_p,
                target,
                internalformat,
                pname,
                count,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetInternalformativ");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetInternalformativ_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetInternalformativ\0",
                &self.glGetInternalformativ_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetInternalformativ_is_loaded(&self) -> bool {
            !self.glGetInternalformativ_p.load(RELAX).is_null()
        }
        /// [glGetMemoryObjectDetachedResourcesuivNV](http://docs.gl/es3/glGetMemoryObjectDetachedResourcesuivNV)(memory, pname, first, count, params)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        pub unsafe fn GetMemoryObjectDetachedResourcesuivNV(
            &self,
            memory: GLuint,
            pname: GLenum,
            first: GLint,
            count: GLsizei,
            params: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetMemoryObjectDetachedResourcesuivNV({:?}, {:#X}, {:?}, {:?}, {:p});",
                    memory,
                    pname,
                    first,
                    count,
                    params
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetMemoryObjectDetachedResourcesuivNV",
                &self.glGetMemoryObjectDetachedResourcesuivNV_p,
                memory,
                pname,
                first,
                count,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetMemoryObjectDetachedResourcesuivNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        #[doc(hidden)]
        pub unsafe fn GetMemoryObjectDetachedResourcesuivNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetMemoryObjectDetachedResourcesuivNV\0",
                &self.glGetMemoryObjectDetachedResourcesuivNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        pub fn GetMemoryObjectDetachedResourcesuivNV_is_loaded(&self) -> bool {
            !self
                .glGetMemoryObjectDetachedResourcesuivNV_p
                .load(RELAX)
                .is_null()
        }
        /// [glGetMemoryObjectParameterivEXT](http://docs.gl/es3/glGetMemoryObjectParameterivEXT)(memoryObject, pname, params)
        /// * `pname` group: MemoryObjectParameterName
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        pub unsafe fn GetMemoryObjectParameterivEXT(
            &self,
            memoryObject: GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetMemoryObjectParameterivEXT({:?}, {:#X}, {:p});",
                    memoryObject,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetMemoryObjectParameterivEXT",
                &self.glGetMemoryObjectParameterivEXT_p,
                memoryObject,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetMemoryObjectParameterivEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        #[doc(hidden)]
        pub unsafe fn GetMemoryObjectParameterivEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetMemoryObjectParameterivEXT\0",
                &self.glGetMemoryObjectParameterivEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        pub fn GetMemoryObjectParameterivEXT_is_loaded(&self) -> bool {
            !self.glGetMemoryObjectParameterivEXT_p.load(RELAX).is_null()
        }
        /// [glGetMultisamplefv](http://docs.gl/es3/glGetMultisample)(pname, index, val)
        /// * `pname` group: GetMultisamplePNameNV
        /// * `val` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetMultisamplefv(&self, pname: GLenum, index: GLuint, val: *mut GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetMultisamplefv({:#X}, {:?}, {:p});",
                    pname,
                    index,
                    val
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetMultisamplefv",
                &self.glGetMultisamplefv_p,
                pname,
                index,
                val,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetMultisamplefv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetMultisamplefv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetMultisamplefv\0",
                &self.glGetMultisamplefv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetMultisamplefv_is_loaded(&self) -> bool {
            !self.glGetMultisamplefv_p.load(RELAX).is_null()
        }
        /// [glGetNextPerfQueryIdINTEL](http://docs.gl/es3/glGetNextPerfQueryIdINTEL)(queryId, nextQueryId)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn GetNextPerfQueryIdINTEL(&self, queryId: GLuint, nextQueryId: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetNextPerfQueryIdINTEL({:?}, {:p});",
                    queryId,
                    nextQueryId
                );
            }
            let out = call_atomic_ptr_2arg(
                "glGetNextPerfQueryIdINTEL",
                &self.glGetNextPerfQueryIdINTEL_p,
                queryId,
                nextQueryId,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetNextPerfQueryIdINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetNextPerfQueryIdINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetNextPerfQueryIdINTEL\0",
                &self.glGetNextPerfQueryIdINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn GetNextPerfQueryIdINTEL_is_loaded(&self) -> bool {
            !self.glGetNextPerfQueryIdINTEL_p.load(RELAX).is_null()
        }
        /// [glGetObjectLabel](http://docs.gl/es3/glGetObjectLabel)(identifier, name, bufSize, length, label)
        /// * `identifier` group: ObjectIdentifier
        /// * `length` len: 1
        /// * `label` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetObjectLabel(
            &self,
            identifier: GLenum,
            name: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            label: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetObjectLabel({:#X}, {:?}, {:?}, {:p}, {:p});",
                    identifier,
                    name,
                    bufSize,
                    length,
                    label
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetObjectLabel",
                &self.glGetObjectLabel_p,
                identifier,
                name,
                bufSize,
                length,
                label,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetObjectLabel");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetObjectLabel_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetObjectLabel\0",
                &self.glGetObjectLabel_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetObjectLabel_is_loaded(&self) -> bool {
            !self.glGetObjectLabel_p.load(RELAX).is_null()
        }
        /// [glGetObjectLabelEXT](http://docs.gl/es3/glGetObjectLabelEXT)(type_, object, bufSize, length, label)
        /// * `length` len: 1
        /// * `label` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_debug_label"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
        pub unsafe fn GetObjectLabelEXT(
            &self,
            type_: GLenum,
            object: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            label: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetObjectLabelEXT({:#X}, {:?}, {:?}, {:p}, {:p});",
                    type_,
                    object,
                    bufSize,
                    length,
                    label
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetObjectLabelEXT",
                &self.glGetObjectLabelEXT_p,
                type_,
                object,
                bufSize,
                length,
                label,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetObjectLabelEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_debug_label"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
        #[doc(hidden)]
        pub unsafe fn GetObjectLabelEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetObjectLabelEXT\0",
                &self.glGetObjectLabelEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_debug_label"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
        pub fn GetObjectLabelEXT_is_loaded(&self) -> bool {
            !self.glGetObjectLabelEXT_p.load(RELAX).is_null()
        }
        /// [glGetObjectLabelKHR](http://docs.gl/es3/glGetObjectLabelKHR)(identifier, name, bufSize, length, label)
        /// * `label` len: bufSize
        /// * alias of: [`glGetObjectLabel`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub unsafe fn GetObjectLabelKHR(
            &self,
            identifier: GLenum,
            name: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            label: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetObjectLabelKHR({:#X}, {:?}, {:?}, {:p}, {:p});",
                    identifier,
                    name,
                    bufSize,
                    length,
                    label
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetObjectLabelKHR",
                &self.glGetObjectLabelKHR_p,
                identifier,
                name,
                bufSize,
                length,
                label,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetObjectLabelKHR");
            }
            out
        }
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        #[doc(hidden)]
        pub unsafe fn GetObjectLabelKHR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetObjectLabelKHR\0",
                &self.glGetObjectLabelKHR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub fn GetObjectLabelKHR_is_loaded(&self) -> bool {
            !self.glGetObjectLabelKHR_p.load(RELAX).is_null()
        }
        /// [glGetObjectPtrLabel](http://docs.gl/es3/glGetObjectPtrLabel)(ptr, bufSize, length, label)
        /// * `length` len: 1
        /// * `label` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetObjectPtrLabel(
            &self,
            ptr: *const c_void,
            bufSize: GLsizei,
            length: *mut GLsizei,
            label: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetObjectPtrLabel({:p}, {:?}, {:p}, {:p});",
                    ptr,
                    bufSize,
                    length,
                    label
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetObjectPtrLabel",
                &self.glGetObjectPtrLabel_p,
                ptr,
                bufSize,
                length,
                label,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetObjectPtrLabel");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetObjectPtrLabel_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetObjectPtrLabel\0",
                &self.glGetObjectPtrLabel_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetObjectPtrLabel_is_loaded(&self) -> bool {
            !self.glGetObjectPtrLabel_p.load(RELAX).is_null()
        }
        /// [glGetObjectPtrLabelKHR](http://docs.gl/es3/glGetObjectPtrLabelKHR)(ptr, bufSize, length, label)
        /// * `length` len: 1
        /// * `label` len: bufSize
        /// * alias of: [`glGetObjectPtrLabel`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub unsafe fn GetObjectPtrLabelKHR(
            &self,
            ptr: *const c_void,
            bufSize: GLsizei,
            length: *mut GLsizei,
            label: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetObjectPtrLabelKHR({:p}, {:?}, {:p}, {:p});",
                    ptr,
                    bufSize,
                    length,
                    label
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetObjectPtrLabelKHR",
                &self.glGetObjectPtrLabelKHR_p,
                ptr,
                bufSize,
                length,
                label,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetObjectPtrLabelKHR");
            }
            out
        }
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        #[doc(hidden)]
        pub unsafe fn GetObjectPtrLabelKHR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetObjectPtrLabelKHR\0",
                &self.glGetObjectPtrLabelKHR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub fn GetObjectPtrLabelKHR_is_loaded(&self) -> bool {
            !self.glGetObjectPtrLabelKHR_p.load(RELAX).is_null()
        }
        /// [glGetPathCommandsNV](http://docs.gl/es3/glGetPathCommandsNV)(path, commands)
        /// * `path` group: Path
        /// * `commands` group: PathCommand
        /// * `commands` len: COMPSIZE(path)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetPathCommandsNV(&self, path: GLuint, commands: *mut GLubyte) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetPathCommandsNV({:?}, {:p});", path, commands);
            }
            let out = call_atomic_ptr_2arg(
                "glGetPathCommandsNV",
                &self.glGetPathCommandsNV_p,
                path,
                commands,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPathCommandsNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetPathCommandsNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPathCommandsNV\0",
                &self.glGetPathCommandsNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetPathCommandsNV_is_loaded(&self) -> bool {
            !self.glGetPathCommandsNV_p.load(RELAX).is_null()
        }
        /// [glGetPathCoordsNV](http://docs.gl/es3/glGetPathCoordsNV)(path, coords)
        /// * `path` group: Path
        /// * `coords` len: COMPSIZE(path)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetPathCoordsNV(&self, path: GLuint, coords: *mut GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetPathCoordsNV({:?}, {:p});", path, coords);
            }
            let out =
                call_atomic_ptr_2arg("glGetPathCoordsNV", &self.glGetPathCoordsNV_p, path, coords);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPathCoordsNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetPathCoordsNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPathCoordsNV\0",
                &self.glGetPathCoordsNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetPathCoordsNV_is_loaded(&self) -> bool {
            !self.glGetPathCoordsNV_p.load(RELAX).is_null()
        }
        /// [glGetPathDashArrayNV](http://docs.gl/es3/glGetPathDashArrayNV)(path, dashArray)
        /// * `path` group: Path
        /// * `dashArray` len: COMPSIZE(path)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetPathDashArrayNV(&self, path: GLuint, dashArray: *mut GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPathDashArrayNV({:?}, {:p});",
                    path,
                    dashArray
                );
            }
            let out = call_atomic_ptr_2arg(
                "glGetPathDashArrayNV",
                &self.glGetPathDashArrayNV_p,
                path,
                dashArray,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPathDashArrayNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetPathDashArrayNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPathDashArrayNV\0",
                &self.glGetPathDashArrayNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetPathDashArrayNV_is_loaded(&self) -> bool {
            !self.glGetPathDashArrayNV_p.load(RELAX).is_null()
        }
        /// [glGetPathLengthNV](http://docs.gl/es3/glGetPathLengthNV)(path, startSegment, numSegments)
        /// * `path` group: Path
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetPathLengthNV(
            &self,
            path: GLuint,
            startSegment: GLsizei,
            numSegments: GLsizei,
        ) -> GLfloat {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPathLengthNV({:?}, {:?}, {:?});",
                    path,
                    startSegment,
                    numSegments
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetPathLengthNV",
                &self.glGetPathLengthNV_p,
                path,
                startSegment,
                numSegments,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPathLengthNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetPathLengthNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPathLengthNV\0",
                &self.glGetPathLengthNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetPathLengthNV_is_loaded(&self) -> bool {
            !self.glGetPathLengthNV_p.load(RELAX).is_null()
        }
        /// [glGetPathMetricRangeNV](http://docs.gl/es3/glGetPathMetricRangeNV)(metricQueryMask, firstPathName, numPaths, stride, metrics)
        /// * `metricQueryMask` group: PathMetricMask
        /// * `firstPathName` group: Path
        /// * `metrics` len: COMPSIZE(metricQueryMask,numPaths,stride)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetPathMetricRangeNV(
            &self,
            metricQueryMask: GLbitfield,
            firstPathName: GLuint,
            numPaths: GLsizei,
            stride: GLsizei,
            metrics: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPathMetricRangeNV({:?}, {:?}, {:?}, {:?}, {:p});",
                    metricQueryMask,
                    firstPathName,
                    numPaths,
                    stride,
                    metrics
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetPathMetricRangeNV",
                &self.glGetPathMetricRangeNV_p,
                metricQueryMask,
                firstPathName,
                numPaths,
                stride,
                metrics,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPathMetricRangeNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetPathMetricRangeNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPathMetricRangeNV\0",
                &self.glGetPathMetricRangeNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetPathMetricRangeNV_is_loaded(&self) -> bool {
            !self.glGetPathMetricRangeNV_p.load(RELAX).is_null()
        }
        /// [glGetPathMetricsNV](http://docs.gl/es3/glGetPathMetricsNV)(metricQueryMask, numPaths, pathNameType, paths, pathBase, stride, metrics)
        /// * `metricQueryMask` group: PathMetricMask
        /// * `pathNameType` group: PathElementType
        /// * `paths` group: PathElement
        /// * `paths` len: COMPSIZE(numPaths,pathNameType,paths)
        /// * `pathBase` group: Path
        /// * `metrics` len: COMPSIZE(metricQueryMask,numPaths,stride)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetPathMetricsNV(
            &self,
            metricQueryMask: GLbitfield,
            numPaths: GLsizei,
            pathNameType: GLenum,
            paths: *const c_void,
            pathBase: GLuint,
            stride: GLsizei,
            metrics: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPathMetricsNV({:?}, {:?}, {:#X}, {:p}, {:?}, {:?}, {:p});",
                    metricQueryMask,
                    numPaths,
                    pathNameType,
                    paths,
                    pathBase,
                    stride,
                    metrics
                );
            }
            let out = call_atomic_ptr_7arg(
                "glGetPathMetricsNV",
                &self.glGetPathMetricsNV_p,
                metricQueryMask,
                numPaths,
                pathNameType,
                paths,
                pathBase,
                stride,
                metrics,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPathMetricsNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetPathMetricsNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPathMetricsNV\0",
                &self.glGetPathMetricsNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetPathMetricsNV_is_loaded(&self) -> bool {
            !self.glGetPathMetricsNV_p.load(RELAX).is_null()
        }
        /// [glGetPathParameterfvNV](http://docs.gl/es3/glGetPathParameterfvNV)(path, pname, value)
        /// * `path` group: Path
        /// * `pname` group: PathParameter
        /// * `value` len: 4
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetPathParameterfvNV(
            &self,
            path: GLuint,
            pname: GLenum,
            value: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPathParameterfvNV({:?}, {:#X}, {:p});",
                    path,
                    pname,
                    value
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetPathParameterfvNV",
                &self.glGetPathParameterfvNV_p,
                path,
                pname,
                value,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPathParameterfvNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetPathParameterfvNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPathParameterfvNV\0",
                &self.glGetPathParameterfvNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetPathParameterfvNV_is_loaded(&self) -> bool {
            !self.glGetPathParameterfvNV_p.load(RELAX).is_null()
        }
        /// [glGetPathParameterivNV](http://docs.gl/es3/glGetPathParameterivNV)(path, pname, value)
        /// * `path` group: Path
        /// * `pname` group: PathParameter
        /// * `value` len: 4
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetPathParameterivNV(&self, path: GLuint, pname: GLenum, value: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPathParameterivNV({:?}, {:#X}, {:p});",
                    path,
                    pname,
                    value
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetPathParameterivNV",
                &self.glGetPathParameterivNV_p,
                path,
                pname,
                value,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPathParameterivNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetPathParameterivNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPathParameterivNV\0",
                &self.glGetPathParameterivNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetPathParameterivNV_is_loaded(&self) -> bool {
            !self.glGetPathParameterivNV_p.load(RELAX).is_null()
        }
        /// [glGetPathSpacingNV](http://docs.gl/es3/glGetPathSpacingNV)(pathListMode, numPaths, pathNameType, paths, pathBase, advanceScale, kerningScale, transformType, returnedSpacing)
        /// * `pathListMode` group: PathListMode
        /// * `pathNameType` group: PathElementType
        /// * `paths` group: PathElement
        /// * `paths` len: COMPSIZE(numPaths,pathNameType,paths)
        /// * `pathBase` group: Path
        /// * `transformType` group: PathTransformType
        /// * `returnedSpacing` len: COMPSIZE(pathListMode,numPaths)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetPathSpacingNV(
            &self,
            pathListMode: GLenum,
            numPaths: GLsizei,
            pathNameType: GLenum,
            paths: *const c_void,
            pathBase: GLuint,
            advanceScale: GLfloat,
            kerningScale: GLfloat,
            transformType: GLenum,
            returnedSpacing: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPathSpacingNV({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?}, {:?}, {:#X}, {:p});",
                    pathListMode,
                    numPaths,
                    pathNameType,
                    paths,
                    pathBase,
                    advanceScale,
                    kerningScale,
                    transformType,
                    returnedSpacing
                );
            }
            let out = call_atomic_ptr_9arg(
                "glGetPathSpacingNV",
                &self.glGetPathSpacingNV_p,
                pathListMode,
                numPaths,
                pathNameType,
                paths,
                pathBase,
                advanceScale,
                kerningScale,
                transformType,
                returnedSpacing,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPathSpacingNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetPathSpacingNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPathSpacingNV\0",
                &self.glGetPathSpacingNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetPathSpacingNV_is_loaded(&self) -> bool {
            !self.glGetPathSpacingNV_p.load(RELAX).is_null()
        }
        /// [glGetPerfCounterInfoINTEL](http://docs.gl/es3/glGetPerfCounterInfoINTEL)(queryId, counterId, counterNameLength, counterName, counterDescLength, counterDesc, counterOffset, counterDataSize, counterTypeEnum, counterDataTypeEnum, rawCounterMaxValue)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn GetPerfCounterInfoINTEL(
            &self,
            queryId: GLuint,
            counterId: GLuint,
            counterNameLength: GLuint,
            counterName: *mut GLchar,
            counterDescLength: GLuint,
            counterDesc: *mut GLchar,
            counterOffset: *mut GLuint,
            counterDataSize: *mut GLuint,
            counterTypeEnum: *mut GLuint,
            counterDataTypeEnum: *mut GLuint,
            rawCounterMaxValue: *mut GLuint64,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetPerfCounterInfoINTEL({:?}, {:?}, {:?}, {:p}, {:?}, {:p}, {:p}, {:p}, {:p}, {:p}, {:p});", queryId, counterId, counterNameLength, counterName, counterDescLength, counterDesc, counterOffset, counterDataSize, counterTypeEnum, counterDataTypeEnum, rawCounterMaxValue);
            }
            let out = call_atomic_ptr_11arg(
                "glGetPerfCounterInfoINTEL",
                &self.glGetPerfCounterInfoINTEL_p,
                queryId,
                counterId,
                counterNameLength,
                counterName,
                counterDescLength,
                counterDesc,
                counterOffset,
                counterDataSize,
                counterTypeEnum,
                counterDataTypeEnum,
                rawCounterMaxValue,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfCounterInfoINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfCounterInfoINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfCounterInfoINTEL\0",
                &self.glGetPerfCounterInfoINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn GetPerfCounterInfoINTEL_is_loaded(&self) -> bool {
            !self.glGetPerfCounterInfoINTEL_p.load(RELAX).is_null()
        }
        /// [glGetPerfMonitorCounterDataAMD](http://docs.gl/es3/glGetPerfMonitorCounterDataAMD)(monitor, pname, dataSize, data, bytesWritten)
        /// * `data` len: dataSize / 4
        /// * `bytesWritten` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn GetPerfMonitorCounterDataAMD(
            &self,
            monitor: GLuint,
            pname: GLenum,
            dataSize: GLsizei,
            data: *mut GLuint,
            bytesWritten: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPerfMonitorCounterDataAMD({:?}, {:#X}, {:?}, {:p}, {:p});",
                    monitor,
                    pname,
                    dataSize,
                    data,
                    bytesWritten
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetPerfMonitorCounterDataAMD",
                &self.glGetPerfMonitorCounterDataAMD_p,
                monitor,
                pname,
                dataSize,
                data,
                bytesWritten,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfMonitorCounterDataAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfMonitorCounterDataAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfMonitorCounterDataAMD\0",
                &self.glGetPerfMonitorCounterDataAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn GetPerfMonitorCounterDataAMD_is_loaded(&self) -> bool {
            !self.glGetPerfMonitorCounterDataAMD_p.load(RELAX).is_null()
        }
        /// [glGetPerfMonitorCounterInfoAMD](http://docs.gl/es3/glGetPerfMonitorCounterInfoAMD)(group, counter, pname, data)
        /// * `data` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn GetPerfMonitorCounterInfoAMD(
            &self,
            group: GLuint,
            counter: GLuint,
            pname: GLenum,
            data: *mut c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPerfMonitorCounterInfoAMD({:?}, {:?}, {:#X}, {:p});",
                    group,
                    counter,
                    pname,
                    data
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetPerfMonitorCounterInfoAMD",
                &self.glGetPerfMonitorCounterInfoAMD_p,
                group,
                counter,
                pname,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfMonitorCounterInfoAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfMonitorCounterInfoAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfMonitorCounterInfoAMD\0",
                &self.glGetPerfMonitorCounterInfoAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn GetPerfMonitorCounterInfoAMD_is_loaded(&self) -> bool {
            !self.glGetPerfMonitorCounterInfoAMD_p.load(RELAX).is_null()
        }
        /// [glGetPerfMonitorCounterStringAMD](http://docs.gl/es3/glGetPerfMonitorCounterStringAMD)(group, counter, bufSize, length, counterString)
        /// * `length` len: 1
        /// * `counterString` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn GetPerfMonitorCounterStringAMD(
            &self,
            group: GLuint,
            counter: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            counterString: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPerfMonitorCounterStringAMD({:?}, {:?}, {:?}, {:p}, {:p});",
                    group,
                    counter,
                    bufSize,
                    length,
                    counterString
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetPerfMonitorCounterStringAMD",
                &self.glGetPerfMonitorCounterStringAMD_p,
                group,
                counter,
                bufSize,
                length,
                counterString,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfMonitorCounterStringAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfMonitorCounterStringAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfMonitorCounterStringAMD\0",
                &self.glGetPerfMonitorCounterStringAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn GetPerfMonitorCounterStringAMD_is_loaded(&self) -> bool {
            !self
                .glGetPerfMonitorCounterStringAMD_p
                .load(RELAX)
                .is_null()
        }
        /// [glGetPerfMonitorCountersAMD](http://docs.gl/es3/glGetPerfMonitorCountersAMD)(group, numCounters, maxActiveCounters, counterSize, counters)
        /// * `numCounters` len: 1
        /// * `maxActiveCounters` len: 1
        /// * `counters` len: counterSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn GetPerfMonitorCountersAMD(
            &self,
            group: GLuint,
            numCounters: *mut GLint,
            maxActiveCounters: *mut GLint,
            counterSize: GLsizei,
            counters: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPerfMonitorCountersAMD({:?}, {:p}, {:p}, {:?}, {:p});",
                    group,
                    numCounters,
                    maxActiveCounters,
                    counterSize,
                    counters
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetPerfMonitorCountersAMD",
                &self.glGetPerfMonitorCountersAMD_p,
                group,
                numCounters,
                maxActiveCounters,
                counterSize,
                counters,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfMonitorCountersAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfMonitorCountersAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfMonitorCountersAMD\0",
                &self.glGetPerfMonitorCountersAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn GetPerfMonitorCountersAMD_is_loaded(&self) -> bool {
            !self.glGetPerfMonitorCountersAMD_p.load(RELAX).is_null()
        }
        /// [glGetPerfMonitorGroupStringAMD](http://docs.gl/es3/glGetPerfMonitorGroupStringAMD)(group, bufSize, length, groupString)
        /// * `length` len: 1
        /// * `groupString` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn GetPerfMonitorGroupStringAMD(
            &self,
            group: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            groupString: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPerfMonitorGroupStringAMD({:?}, {:?}, {:p}, {:p});",
                    group,
                    bufSize,
                    length,
                    groupString
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetPerfMonitorGroupStringAMD",
                &self.glGetPerfMonitorGroupStringAMD_p,
                group,
                bufSize,
                length,
                groupString,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfMonitorGroupStringAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfMonitorGroupStringAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfMonitorGroupStringAMD\0",
                &self.glGetPerfMonitorGroupStringAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn GetPerfMonitorGroupStringAMD_is_loaded(&self) -> bool {
            !self.glGetPerfMonitorGroupStringAMD_p.load(RELAX).is_null()
        }
        /// [glGetPerfMonitorGroupsAMD](http://docs.gl/es3/glGetPerfMonitorGroupsAMD)(numGroups, groupsSize, groups)
        /// * `numGroups` len: 1
        /// * `groups` len: groupsSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub unsafe fn GetPerfMonitorGroupsAMD(
            &self,
            numGroups: *mut GLint,
            groupsSize: GLsizei,
            groups: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPerfMonitorGroupsAMD({:p}, {:?}, {:p});",
                    numGroups,
                    groupsSize,
                    groups
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetPerfMonitorGroupsAMD",
                &self.glGetPerfMonitorGroupsAMD_p,
                numGroups,
                groupsSize,
                groups,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfMonitorGroupsAMD");
            }
            out
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfMonitorGroupsAMD_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfMonitorGroupsAMD\0",
                &self.glGetPerfMonitorGroupsAMD_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        pub fn GetPerfMonitorGroupsAMD_is_loaded(&self) -> bool {
            !self.glGetPerfMonitorGroupsAMD_p.load(RELAX).is_null()
        }
        /// [glGetPerfQueryDataINTEL](http://docs.gl/es3/glGetPerfQueryDataINTEL)(queryHandle, flags, dataSize, data, bytesWritten)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn GetPerfQueryDataINTEL(
            &self,
            queryHandle: GLuint,
            flags: GLuint,
            dataSize: GLsizei,
            data: *mut c_void,
            bytesWritten: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPerfQueryDataINTEL({:?}, {:?}, {:?}, {:p}, {:p});",
                    queryHandle,
                    flags,
                    dataSize,
                    data,
                    bytesWritten
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetPerfQueryDataINTEL",
                &self.glGetPerfQueryDataINTEL_p,
                queryHandle,
                flags,
                dataSize,
                data,
                bytesWritten,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfQueryDataINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfQueryDataINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfQueryDataINTEL\0",
                &self.glGetPerfQueryDataINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn GetPerfQueryDataINTEL_is_loaded(&self) -> bool {
            !self.glGetPerfQueryDataINTEL_p.load(RELAX).is_null()
        }
        /// [glGetPerfQueryIdByNameINTEL](http://docs.gl/es3/glGetPerfQueryIdByNameINTEL)(queryName, queryId)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn GetPerfQueryIdByNameINTEL(
            &self,
            queryName: *mut GLchar,
            queryId: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPerfQueryIdByNameINTEL({:p}, {:p});",
                    queryName,
                    queryId
                );
            }
            let out = call_atomic_ptr_2arg(
                "glGetPerfQueryIdByNameINTEL",
                &self.glGetPerfQueryIdByNameINTEL_p,
                queryName,
                queryId,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfQueryIdByNameINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfQueryIdByNameINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfQueryIdByNameINTEL\0",
                &self.glGetPerfQueryIdByNameINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn GetPerfQueryIdByNameINTEL_is_loaded(&self) -> bool {
            !self.glGetPerfQueryIdByNameINTEL_p.load(RELAX).is_null()
        }
        /// [glGetPerfQueryInfoINTEL](http://docs.gl/es3/glGetPerfQueryInfoINTEL)(queryId, queryNameLength, queryName, dataSize, noCounters, noInstances, capsMask)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub unsafe fn GetPerfQueryInfoINTEL(
            &self,
            queryId: GLuint,
            queryNameLength: GLuint,
            queryName: *mut GLchar,
            dataSize: *mut GLuint,
            noCounters: *mut GLuint,
            noInstances: *mut GLuint,
            capsMask: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetPerfQueryInfoINTEL({:?}, {:?}, {:p}, {:p}, {:p}, {:p}, {:p});",
                    queryId,
                    queryNameLength,
                    queryName,
                    dataSize,
                    noCounters,
                    noInstances,
                    capsMask
                );
            }
            let out = call_atomic_ptr_7arg(
                "glGetPerfQueryInfoINTEL",
                &self.glGetPerfQueryInfoINTEL_p,
                queryId,
                queryNameLength,
                queryName,
                dataSize,
                noCounters,
                noInstances,
                capsMask,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPerfQueryInfoINTEL");
            }
            out
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetPerfQueryInfoINTEL_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPerfQueryInfoINTEL\0",
                &self.glGetPerfQueryInfoINTEL_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        pub fn GetPerfQueryInfoINTEL_is_loaded(&self) -> bool {
            !self.glGetPerfQueryInfoINTEL_p.load(RELAX).is_null()
        }
        /// [glGetPointerv](http://docs.gl/es3/glGetPointerv)(pname, params)
        /// * `pname` group: GetPointervPName
        /// * `params` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetPointerv(&self, pname: GLenum, params: *mut *mut c_void) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetPointerv({:#X}, {:p});", pname, params);
            }
            let out = call_atomic_ptr_2arg("glGetPointerv", &self.glGetPointerv_p, pname, params);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPointerv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetPointerv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetPointerv\0", &self.glGetPointerv_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetPointerv_is_loaded(&self) -> bool {
            !self.glGetPointerv_p.load(RELAX).is_null()
        }
        /// [glGetPointervKHR](http://docs.gl/es3/glGetPointervKHR)(pname, params)
        /// * `params` len: 1
        /// * alias of: [`glGetPointerv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub unsafe fn GetPointervKHR(&self, pname: GLenum, params: *mut *mut c_void) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetPointervKHR({:#X}, {:p});", pname, params);
            }
            let out =
                call_atomic_ptr_2arg("glGetPointervKHR", &self.glGetPointervKHR_p, pname, params);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetPointervKHR");
            }
            out
        }
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        #[doc(hidden)]
        pub unsafe fn GetPointervKHR_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetPointervKHR\0",
                &self.glGetPointervKHR_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        pub fn GetPointervKHR_is_loaded(&self) -> bool {
            !self.glGetPointervKHR_p.load(RELAX).is_null()
        }
        /// [glGetProgramBinary](http://docs.gl/es3/glGetProgramBinary)(program, bufSize, length, binaryFormat, binary)
        /// * `program` class: program
        /// * `length` len: 1
        /// * `binaryFormat` len: 1
        /// * `binary` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramBinary(
            &self,
            program: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            binaryFormat: *mut GLenum,
            binary: *mut c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramBinary({:?}, {:?}, {:p}, {:p}, {:p});",
                    program,
                    bufSize,
                    length,
                    binaryFormat,
                    binary
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetProgramBinary",
                &self.glGetProgramBinary_p,
                program,
                bufSize,
                length,
                binaryFormat,
                binary,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramBinary");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramBinary_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramBinary\0",
                &self.glGetProgramBinary_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramBinary_is_loaded(&self) -> bool {
            !self.glGetProgramBinary_p.load(RELAX).is_null()
        }
        /// [glGetProgramBinaryOES](http://docs.gl/es3/glGetProgramBinaryOES)(program, bufSize, length, binaryFormat, binary)
        /// * `program` class: program
        /// * `length` len: 1
        /// * `binaryFormat` len: 1
        /// * `binary` len: bufSize
        /// * alias of: [`glGetProgramBinary`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_get_program_binary"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
        pub unsafe fn GetProgramBinaryOES(
            &self,
            program: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            binaryFormat: *mut GLenum,
            binary: *mut c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramBinaryOES({:?}, {:?}, {:p}, {:p}, {:p});",
                    program,
                    bufSize,
                    length,
                    binaryFormat,
                    binary
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetProgramBinaryOES",
                &self.glGetProgramBinaryOES_p,
                program,
                bufSize,
                length,
                binaryFormat,
                binary,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramBinaryOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_get_program_binary"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
        #[doc(hidden)]
        pub unsafe fn GetProgramBinaryOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramBinaryOES\0",
                &self.glGetProgramBinaryOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_get_program_binary"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
        pub fn GetProgramBinaryOES_is_loaded(&self) -> bool {
            !self.glGetProgramBinaryOES_p.load(RELAX).is_null()
        }
        /// [glGetProgramInfoLog](http://docs.gl/es3/glGetProgramInfoLog)(program, bufSize, length, infoLog)
        /// * `program` class: program
        /// * `length` len: 1
        /// * `infoLog` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramInfoLog(
            &self,
            program: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            infoLog: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramInfoLog({:?}, {:?}, {:p}, {:p});",
                    program,
                    bufSize,
                    length,
                    infoLog
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetProgramInfoLog",
                &self.glGetProgramInfoLog_p,
                program,
                bufSize,
                length,
                infoLog,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramInfoLog");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramInfoLog_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramInfoLog\0",
                &self.glGetProgramInfoLog_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramInfoLog_is_loaded(&self) -> bool {
            !self.glGetProgramInfoLog_p.load(RELAX).is_null()
        }
        /// [glGetProgramInterfaceiv](http://docs.gl/es3/glGetProgramInterface)(program, programInterface, pname, params)
        /// * `program` class: program
        /// * `programInterface` group: ProgramInterface
        /// * `pname` group: ProgramInterfacePName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramInterfaceiv(
            &self,
            program: GLuint,
            programInterface: GLenum,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramInterfaceiv({:?}, {:#X}, {:#X}, {:p});",
                    program,
                    programInterface,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetProgramInterfaceiv",
                &self.glGetProgramInterfaceiv_p,
                program,
                programInterface,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramInterfaceiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramInterfaceiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramInterfaceiv\0",
                &self.glGetProgramInterfaceiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramInterfaceiv_is_loaded(&self) -> bool {
            !self.glGetProgramInterfaceiv_p.load(RELAX).is_null()
        }
        /// [glGetProgramPipelineInfoLog](http://docs.gl/es3/glGetProgramPipelineInfoLog)(pipeline, bufSize, length, infoLog)
        /// * `pipeline` class: program pipeline
        /// * `length` len: 1
        /// * `infoLog` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramPipelineInfoLog(
            &self,
            pipeline: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            infoLog: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramPipelineInfoLog({:?}, {:?}, {:p}, {:p});",
                    pipeline,
                    bufSize,
                    length,
                    infoLog
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetProgramPipelineInfoLog",
                &self.glGetProgramPipelineInfoLog_p,
                pipeline,
                bufSize,
                length,
                infoLog,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramPipelineInfoLog");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramPipelineInfoLog_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramPipelineInfoLog\0",
                &self.glGetProgramPipelineInfoLog_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramPipelineInfoLog_is_loaded(&self) -> bool {
            !self.glGetProgramPipelineInfoLog_p.load(RELAX).is_null()
        }
        /// [glGetProgramPipelineInfoLogEXT](http://docs.gl/es3/glGetProgramPipelineInfoLogEXT)(pipeline, bufSize, length, infoLog)
        /// * `pipeline` class: program pipeline
        /// * `length` len: 1
        /// * `infoLog` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub unsafe fn GetProgramPipelineInfoLogEXT(
            &self,
            pipeline: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            infoLog: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramPipelineInfoLogEXT({:?}, {:?}, {:p}, {:p});",
                    pipeline,
                    bufSize,
                    length,
                    infoLog
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetProgramPipelineInfoLogEXT",
                &self.glGetProgramPipelineInfoLogEXT_p,
                pipeline,
                bufSize,
                length,
                infoLog,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramPipelineInfoLogEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        #[doc(hidden)]
        pub unsafe fn GetProgramPipelineInfoLogEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramPipelineInfoLogEXT\0",
                &self.glGetProgramPipelineInfoLogEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub fn GetProgramPipelineInfoLogEXT_is_loaded(&self) -> bool {
            !self.glGetProgramPipelineInfoLogEXT_p.load(RELAX).is_null()
        }
        /// [glGetProgramPipelineiv](http://docs.gl/es3/glGetProgramPipeline)(pipeline, pname, params)
        /// * `pipeline` class: program pipeline
        /// * `pname` group: PipelineParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramPipelineiv(
            &self,
            pipeline: GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramPipelineiv({:?}, {:#X}, {:p});",
                    pipeline,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetProgramPipelineiv",
                &self.glGetProgramPipelineiv_p,
                pipeline,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramPipelineiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramPipelineiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramPipelineiv\0",
                &self.glGetProgramPipelineiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramPipelineiv_is_loaded(&self) -> bool {
            !self.glGetProgramPipelineiv_p.load(RELAX).is_null()
        }
        /// [glGetProgramPipelineivEXT](http://docs.gl/es3/glGetProgramPipelineivEXT)(pipeline, pname, params)
        /// * `pipeline` class: program pipeline
        /// * `pname` group: PipelineParameterName
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub unsafe fn GetProgramPipelineivEXT(
            &self,
            pipeline: GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramPipelineivEXT({:?}, {:#X}, {:p});",
                    pipeline,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetProgramPipelineivEXT",
                &self.glGetProgramPipelineivEXT_p,
                pipeline,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramPipelineivEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        #[doc(hidden)]
        pub unsafe fn GetProgramPipelineivEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramPipelineivEXT\0",
                &self.glGetProgramPipelineivEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        pub fn GetProgramPipelineivEXT_is_loaded(&self) -> bool {
            !self.glGetProgramPipelineivEXT_p.load(RELAX).is_null()
        }
        /// [glGetProgramResourceIndex](http://docs.gl/es3/glGetProgramResourceIndex)(program, programInterface, name)
        /// * `program` class: program
        /// * `programInterface` group: ProgramInterface
        /// * `name` len: COMPSIZE(name)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramResourceIndex(
            &self,
            program: GLuint,
            programInterface: GLenum,
            name: *const GLchar,
        ) -> GLuint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramResourceIndex({:?}, {:#X}, {:p});",
                    program,
                    programInterface,
                    name
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetProgramResourceIndex",
                &self.glGetProgramResourceIndex_p,
                program,
                programInterface,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramResourceIndex");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramResourceIndex_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramResourceIndex\0",
                &self.glGetProgramResourceIndex_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramResourceIndex_is_loaded(&self) -> bool {
            !self.glGetProgramResourceIndex_p.load(RELAX).is_null()
        }
        /// [glGetProgramResourceLocation](http://docs.gl/es3/glGetProgramResourceLocation)(program, programInterface, name)
        /// * `program` class: program
        /// * `programInterface` group: ProgramInterface
        /// * `name` len: COMPSIZE(name)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramResourceLocation(
            &self,
            program: GLuint,
            programInterface: GLenum,
            name: *const GLchar,
        ) -> GLint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramResourceLocation({:?}, {:#X}, {:p});",
                    program,
                    programInterface,
                    name
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetProgramResourceLocation",
                &self.glGetProgramResourceLocation_p,
                program,
                programInterface,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramResourceLocation");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramResourceLocation_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramResourceLocation\0",
                &self.glGetProgramResourceLocation_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramResourceLocation_is_loaded(&self) -> bool {
            !self.glGetProgramResourceLocation_p.load(RELAX).is_null()
        }
        /// [glGetProgramResourceLocationIndexEXT](http://docs.gl/es3/glGetProgramResourceLocationIndexEXT)(program, programInterface, name)
        /// * `program` class: program
        /// * `programInterface` group: ProgramInterface
        /// * `name` len: COMPSIZE(name)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        pub unsafe fn GetProgramResourceLocationIndexEXT(
            &self,
            program: GLuint,
            programInterface: GLenum,
            name: *const GLchar,
        ) -> GLint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramResourceLocationIndexEXT({:?}, {:#X}, {:p});",
                    program,
                    programInterface,
                    name
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetProgramResourceLocationIndexEXT",
                &self.glGetProgramResourceLocationIndexEXT_p,
                program,
                programInterface,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramResourceLocationIndexEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        #[doc(hidden)]
        pub unsafe fn GetProgramResourceLocationIndexEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramResourceLocationIndexEXT\0",
                &self.glGetProgramResourceLocationIndexEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        pub fn GetProgramResourceLocationIndexEXT_is_loaded(&self) -> bool {
            !self
                .glGetProgramResourceLocationIndexEXT_p
                .load(RELAX)
                .is_null()
        }
        /// [glGetProgramResourceName](http://docs.gl/es3/glGetProgramResourceName)(program, programInterface, index, bufSize, length, name)
        /// * `program` class: program
        /// * `programInterface` group: ProgramInterface
        /// * `length` len: 1
        /// * `name` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramResourceName(
            &self,
            program: GLuint,
            programInterface: GLenum,
            index: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            name: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramResourceName({:?}, {:#X}, {:?}, {:?}, {:p}, {:p});",
                    program,
                    programInterface,
                    index,
                    bufSize,
                    length,
                    name
                );
            }
            let out = call_atomic_ptr_6arg(
                "glGetProgramResourceName",
                &self.glGetProgramResourceName_p,
                program,
                programInterface,
                index,
                bufSize,
                length,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramResourceName");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramResourceName_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramResourceName\0",
                &self.glGetProgramResourceName_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramResourceName_is_loaded(&self) -> bool {
            !self.glGetProgramResourceName_p.load(RELAX).is_null()
        }
        /// [glGetProgramResourcefvNV](http://docs.gl/es3/glGetProgramResourcefvNV)(program, programInterface, index, propCount, props, count, length, params)
        /// * `program` class: program
        /// * `programInterface` group: ProgramInterface
        /// * `length` len: 1
        /// * `params` len: count
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub unsafe fn GetProgramResourcefvNV(
            &self,
            program: GLuint,
            programInterface: GLenum,
            index: GLuint,
            propCount: GLsizei,
            props: *const GLenum,
            count: GLsizei,
            length: *mut GLsizei,
            params: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramResourcefvNV({:?}, {:#X}, {:?}, {:?}, {:p}, {:?}, {:p}, {:p});",
                    program,
                    programInterface,
                    index,
                    propCount,
                    props,
                    count,
                    length,
                    params
                );
            }
            let out = call_atomic_ptr_8arg(
                "glGetProgramResourcefvNV",
                &self.glGetProgramResourcefvNV_p,
                program,
                programInterface,
                index,
                propCount,
                props,
                count,
                length,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramResourcefvNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        #[doc(hidden)]
        pub unsafe fn GetProgramResourcefvNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramResourcefvNV\0",
                &self.glGetProgramResourcefvNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        pub fn GetProgramResourcefvNV_is_loaded(&self) -> bool {
            !self.glGetProgramResourcefvNV_p.load(RELAX).is_null()
        }
        /// [glGetProgramResourceiv](http://docs.gl/es3/glGetProgramResource)(program, programInterface, index, propCount, props, count, length, params)
        /// * `program` class: program
        /// * `programInterface` group: ProgramInterface
        /// * `props` group: ProgramResourceProperty
        /// * `props` len: propCount
        /// * `length` len: 1
        /// * `params` len: count
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramResourceiv(
            &self,
            program: GLuint,
            programInterface: GLenum,
            index: GLuint,
            propCount: GLsizei,
            props: *const GLenum,
            count: GLsizei,
            length: *mut GLsizei,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramResourceiv({:?}, {:#X}, {:?}, {:?}, {:p}, {:?}, {:p}, {:p});",
                    program,
                    programInterface,
                    index,
                    propCount,
                    props,
                    count,
                    length,
                    params
                );
            }
            let out = call_atomic_ptr_8arg(
                "glGetProgramResourceiv",
                &self.glGetProgramResourceiv_p,
                program,
                programInterface,
                index,
                propCount,
                props,
                count,
                length,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramResourceiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramResourceiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramResourceiv\0",
                &self.glGetProgramResourceiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramResourceiv_is_loaded(&self) -> bool {
            !self.glGetProgramResourceiv_p.load(RELAX).is_null()
        }
        /// [glGetProgramiv](http://docs.gl/es3/glGetProgram)(program, pname, params)
        /// * `program` class: program
        /// * `pname` group: ProgramPropertyARB
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetProgramiv(&self, program: GLuint, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetProgramiv({:?}, {:#X}, {:p});",
                    program,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetProgramiv",
                &self.glGetProgramiv_p,
                program,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetProgramiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetProgramiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetProgramiv\0",
                &self.glGetProgramiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetProgramiv_is_loaded(&self) -> bool {
            !self.glGetProgramiv_p.load(RELAX).is_null()
        }
        /// [glGetQueryObjecti64vEXT](http://docs.gl/es3/glGetQueryObjecti64vEXT)(id, pname, params)
        /// * `id` class: query
        /// * `pname` group: QueryObjectParameterName
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetQueryObjecti64v`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        pub unsafe fn GetQueryObjecti64vEXT(
            &self,
            id: GLuint,
            pname: GLenum,
            params: *mut GLint64,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetQueryObjecti64vEXT({:?}, {:#X}, {:p});",
                    id,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetQueryObjecti64vEXT",
                &self.glGetQueryObjecti64vEXT_p,
                id,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetQueryObjecti64vEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetQueryObjecti64vEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetQueryObjecti64vEXT\0",
                &self.glGetQueryObjecti64vEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        pub fn GetQueryObjecti64vEXT_is_loaded(&self) -> bool {
            !self.glGetQueryObjecti64vEXT_p.load(RELAX).is_null()
        }
        /// [glGetQueryObjectivEXT](http://docs.gl/es3/glGetQueryObjectivEXT)(id, pname, params)
        /// * `id` class: query
        /// * `pname` group: QueryObjectParameterName
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetQueryObjectiv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        pub unsafe fn GetQueryObjectivEXT(&self, id: GLuint, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetQueryObjectivEXT({:?}, {:#X}, {:p});",
                    id,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetQueryObjectivEXT",
                &self.glGetQueryObjectivEXT_p,
                id,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetQueryObjectivEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetQueryObjectivEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetQueryObjectivEXT\0",
                &self.glGetQueryObjectivEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        pub fn GetQueryObjectivEXT_is_loaded(&self) -> bool {
            !self.glGetQueryObjectivEXT_p.load(RELAX).is_null()
        }
        /// [glGetQueryObjectui64vEXT](http://docs.gl/es3/glGetQueryObjectui64vEXT)(id, pname, params)
        /// * `id` class: query
        /// * `pname` group: QueryObjectParameterName
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetQueryObjectui64v`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        pub unsafe fn GetQueryObjectui64vEXT(
            &self,
            id: GLuint,
            pname: GLenum,
            params: *mut GLuint64,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetQueryObjectui64vEXT({:?}, {:#X}, {:p});",
                    id,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetQueryObjectui64vEXT",
                &self.glGetQueryObjectui64vEXT_p,
                id,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetQueryObjectui64vEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        #[doc(hidden)]
        pub unsafe fn GetQueryObjectui64vEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetQueryObjectui64vEXT\0",
                &self.glGetQueryObjectui64vEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        pub fn GetQueryObjectui64vEXT_is_loaded(&self) -> bool {
            !self.glGetQueryObjectui64vEXT_p.load(RELAX).is_null()
        }
        /// [glGetQueryObjectuiv](http://docs.gl/es3/glGetQueryObject)(id, pname, params)
        /// * `id` class: query
        /// * `pname` group: QueryObjectParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetQueryObjectuiv(&self, id: GLuint, pname: GLenum, params: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetQueryObjectuiv({:?}, {:#X}, {:p});",
                    id,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetQueryObjectuiv",
                &self.glGetQueryObjectuiv_p,
                id,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetQueryObjectuiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetQueryObjectuiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetQueryObjectuiv\0",
                &self.glGetQueryObjectuiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetQueryObjectuiv_is_loaded(&self) -> bool {
            !self.glGetQueryObjectuiv_p.load(RELAX).is_null()
        }
        /// [glGetQueryObjectuivEXT](http://docs.gl/es3/glGetQueryObjectuivEXT)(id, pname, params)
        /// * `id` class: query
        /// * `pname` group: QueryObjectParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub unsafe fn GetQueryObjectuivEXT(&self, id: GLuint, pname: GLenum, params: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetQueryObjectuivEXT({:?}, {:#X}, {:p});",
                    id,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetQueryObjectuivEXT",
                &self.glGetQueryObjectuivEXT_p,
                id,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetQueryObjectuivEXT");
            }
            out
        }
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn GetQueryObjectuivEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetQueryObjectuivEXT\0",
                &self.glGetQueryObjectuivEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub fn GetQueryObjectuivEXT_is_loaded(&self) -> bool {
            !self.glGetQueryObjectuivEXT_p.load(RELAX).is_null()
        }
        /// [glGetQueryiv](http://docs.gl/es3/glGetQuery)(target, pname, params)
        /// * `target` group: QueryTarget
        /// * `pname` group: QueryParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetQueryiv(&self, target: GLenum, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetQueryiv({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out =
                call_atomic_ptr_3arg("glGetQueryiv", &self.glGetQueryiv_p, target, pname, params);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetQueryiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetQueryiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetQueryiv\0", &self.glGetQueryiv_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetQueryiv_is_loaded(&self) -> bool {
            !self.glGetQueryiv_p.load(RELAX).is_null()
        }
        /// [glGetQueryivEXT](http://docs.gl/es3/glGetQueryivEXT)(target, pname, params)
        /// * `target` group: QueryTarget
        /// * `pname` group: QueryParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub unsafe fn GetQueryivEXT(&self, target: GLenum, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetQueryivEXT({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetQueryivEXT",
                &self.glGetQueryivEXT_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetQueryivEXT");
            }
            out
        }
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        #[doc(hidden)]
        pub unsafe fn GetQueryivEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetQueryivEXT\0",
                &self.glGetQueryivEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        pub fn GetQueryivEXT_is_loaded(&self) -> bool {
            !self.glGetQueryivEXT_p.load(RELAX).is_null()
        }
        /// [glGetRenderbufferParameteriv](http://docs.gl/es3/glGetRenderbufferParameter)(target, pname, params)
        /// * `target` group: RenderbufferTarget
        /// * `pname` group: RenderbufferParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetRenderbufferParameteriv(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetRenderbufferParameteriv({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetRenderbufferParameteriv",
                &self.glGetRenderbufferParameteriv_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetRenderbufferParameteriv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetRenderbufferParameteriv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetRenderbufferParameteriv\0",
                &self.glGetRenderbufferParameteriv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetRenderbufferParameteriv_is_loaded(&self) -> bool {
            !self.glGetRenderbufferParameteriv_p.load(RELAX).is_null()
        }
        /// [glGetSamplerParameterIiv](http://docs.gl/es3/glGetSamplerParameter)(sampler, pname, params)
        /// * `sampler` class: sampler
        /// * `pname` group: SamplerParameterI
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetSamplerParameterIiv(
            &self,
            sampler: GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSamplerParameterIiv({:?}, {:#X}, {:p});",
                    sampler,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSamplerParameterIiv",
                &self.glGetSamplerParameterIiv_p,
                sampler,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSamplerParameterIiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetSamplerParameterIiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSamplerParameterIiv\0",
                &self.glGetSamplerParameterIiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetSamplerParameterIiv_is_loaded(&self) -> bool {
            !self.glGetSamplerParameterIiv_p.load(RELAX).is_null()
        }
        /// [glGetSamplerParameterIivEXT](http://docs.gl/es3/glGetSamplerParameterIivEXT)(sampler, pname, params)
        /// * `sampler` class: sampler
        /// * `pname` group: SamplerParameterI
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetSamplerParameterIiv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        pub unsafe fn GetSamplerParameterIivEXT(
            &self,
            sampler: GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSamplerParameterIivEXT({:?}, {:#X}, {:p});",
                    sampler,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSamplerParameterIivEXT",
                &self.glGetSamplerParameterIivEXT_p,
                sampler,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSamplerParameterIivEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        #[doc(hidden)]
        pub unsafe fn GetSamplerParameterIivEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSamplerParameterIivEXT\0",
                &self.glGetSamplerParameterIivEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        pub fn GetSamplerParameterIivEXT_is_loaded(&self) -> bool {
            !self.glGetSamplerParameterIivEXT_p.load(RELAX).is_null()
        }
        /// [glGetSamplerParameterIivOES](http://docs.gl/es3/glGetSamplerParameterIivOES)(sampler, pname, params)
        /// * `sampler` class: sampler
        /// * `pname` group: SamplerParameterI
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetSamplerParameterIiv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        pub unsafe fn GetSamplerParameterIivOES(
            &self,
            sampler: GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSamplerParameterIivOES({:?}, {:#X}, {:p});",
                    sampler,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSamplerParameterIivOES",
                &self.glGetSamplerParameterIivOES_p,
                sampler,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSamplerParameterIivOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        #[doc(hidden)]
        pub unsafe fn GetSamplerParameterIivOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSamplerParameterIivOES\0",
                &self.glGetSamplerParameterIivOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        pub fn GetSamplerParameterIivOES_is_loaded(&self) -> bool {
            !self.glGetSamplerParameterIivOES_p.load(RELAX).is_null()
        }
        /// [glGetSamplerParameterIuiv](http://docs.gl/es3/glGetSamplerParameter)(sampler, pname, params)
        /// * `sampler` class: sampler
        /// * `pname` group: SamplerParameterI
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetSamplerParameterIuiv(
            &self,
            sampler: GLuint,
            pname: GLenum,
            params: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSamplerParameterIuiv({:?}, {:#X}, {:p});",
                    sampler,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSamplerParameterIuiv",
                &self.glGetSamplerParameterIuiv_p,
                sampler,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSamplerParameterIuiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetSamplerParameterIuiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSamplerParameterIuiv\0",
                &self.glGetSamplerParameterIuiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetSamplerParameterIuiv_is_loaded(&self) -> bool {
            !self.glGetSamplerParameterIuiv_p.load(RELAX).is_null()
        }
        /// [glGetSamplerParameterIuivEXT](http://docs.gl/es3/glGetSamplerParameterIuivEXT)(sampler, pname, params)
        /// * `sampler` class: sampler
        /// * `pname` group: SamplerParameterI
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetSamplerParameterIuiv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        pub unsafe fn GetSamplerParameterIuivEXT(
            &self,
            sampler: GLuint,
            pname: GLenum,
            params: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSamplerParameterIuivEXT({:?}, {:#X}, {:p});",
                    sampler,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSamplerParameterIuivEXT",
                &self.glGetSamplerParameterIuivEXT_p,
                sampler,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSamplerParameterIuivEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        #[doc(hidden)]
        pub unsafe fn GetSamplerParameterIuivEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSamplerParameterIuivEXT\0",
                &self.glGetSamplerParameterIuivEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        pub fn GetSamplerParameterIuivEXT_is_loaded(&self) -> bool {
            !self.glGetSamplerParameterIuivEXT_p.load(RELAX).is_null()
        }
        /// [glGetSamplerParameterIuivOES](http://docs.gl/es3/glGetSamplerParameterIuivOES)(sampler, pname, params)
        /// * `sampler` class: sampler
        /// * `pname` group: SamplerParameterI
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetSamplerParameterIuiv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        pub unsafe fn GetSamplerParameterIuivOES(
            &self,
            sampler: GLuint,
            pname: GLenum,
            params: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSamplerParameterIuivOES({:?}, {:#X}, {:p});",
                    sampler,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSamplerParameterIuivOES",
                &self.glGetSamplerParameterIuivOES_p,
                sampler,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSamplerParameterIuivOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        #[doc(hidden)]
        pub unsafe fn GetSamplerParameterIuivOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSamplerParameterIuivOES\0",
                &self.glGetSamplerParameterIuivOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        pub fn GetSamplerParameterIuivOES_is_loaded(&self) -> bool {
            !self.glGetSamplerParameterIuivOES_p.load(RELAX).is_null()
        }
        /// [glGetSamplerParameterfv](http://docs.gl/es3/glGetSamplerParameter)(sampler, pname, params)
        /// * `sampler` class: sampler
        /// * `pname` group: SamplerParameterF
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetSamplerParameterfv(
            &self,
            sampler: GLuint,
            pname: GLenum,
            params: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSamplerParameterfv({:?}, {:#X}, {:p});",
                    sampler,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSamplerParameterfv",
                &self.glGetSamplerParameterfv_p,
                sampler,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSamplerParameterfv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetSamplerParameterfv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSamplerParameterfv\0",
                &self.glGetSamplerParameterfv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetSamplerParameterfv_is_loaded(&self) -> bool {
            !self.glGetSamplerParameterfv_p.load(RELAX).is_null()
        }
        /// [glGetSamplerParameteriv](http://docs.gl/es3/glGetSamplerParameter)(sampler, pname, params)
        /// * `sampler` class: sampler
        /// * `pname` group: SamplerParameterI
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetSamplerParameteriv(
            &self,
            sampler: GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSamplerParameteriv({:?}, {:#X}, {:p});",
                    sampler,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSamplerParameteriv",
                &self.glGetSamplerParameteriv_p,
                sampler,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSamplerParameteriv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetSamplerParameteriv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSamplerParameteriv\0",
                &self.glGetSamplerParameteriv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetSamplerParameteriv_is_loaded(&self) -> bool {
            !self.glGetSamplerParameteriv_p.load(RELAX).is_null()
        }
        /// [glGetSemaphoreParameterivNV](http://docs.gl/es3/glGetSemaphoreParameterivNV)(semaphore, pname, params)
        /// * `pname` group: SemaphoreParameterName
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
        pub unsafe fn GetSemaphoreParameterivNV(
            &self,
            semaphore: GLuint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSemaphoreParameterivNV({:?}, {:#X}, {:p});",
                    semaphore,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSemaphoreParameterivNV",
                &self.glGetSemaphoreParameterivNV_p,
                semaphore,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSemaphoreParameterivNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
        #[doc(hidden)]
        pub unsafe fn GetSemaphoreParameterivNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSemaphoreParameterivNV\0",
                &self.glGetSemaphoreParameterivNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
        pub fn GetSemaphoreParameterivNV_is_loaded(&self) -> bool {
            !self.glGetSemaphoreParameterivNV_p.load(RELAX).is_null()
        }
        /// [glGetSemaphoreParameterui64vEXT](http://docs.gl/es3/glGetSemaphoreParameterui64vEXT)(semaphore, pname, params)
        /// * `pname` group: SemaphoreParameterName
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        pub unsafe fn GetSemaphoreParameterui64vEXT(
            &self,
            semaphore: GLuint,
            pname: GLenum,
            params: *mut GLuint64,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSemaphoreParameterui64vEXT({:?}, {:#X}, {:p});",
                    semaphore,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetSemaphoreParameterui64vEXT",
                &self.glGetSemaphoreParameterui64vEXT_p,
                semaphore,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSemaphoreParameterui64vEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        #[doc(hidden)]
        pub unsafe fn GetSemaphoreParameterui64vEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSemaphoreParameterui64vEXT\0",
                &self.glGetSemaphoreParameterui64vEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        pub fn GetSemaphoreParameterui64vEXT_is_loaded(&self) -> bool {
            !self.glGetSemaphoreParameterui64vEXT_p.load(RELAX).is_null()
        }
        /// [glGetShaderInfoLog](http://docs.gl/es3/glGetShaderInfoLog)(shader, bufSize, length, infoLog)
        /// * `shader` class: shader
        /// * `length` len: 1
        /// * `infoLog` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetShaderInfoLog(
            &self,
            shader: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            infoLog: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetShaderInfoLog({:?}, {:?}, {:p}, {:p});",
                    shader,
                    bufSize,
                    length,
                    infoLog
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetShaderInfoLog",
                &self.glGetShaderInfoLog_p,
                shader,
                bufSize,
                length,
                infoLog,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetShaderInfoLog");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetShaderInfoLog_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetShaderInfoLog\0",
                &self.glGetShaderInfoLog_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetShaderInfoLog_is_loaded(&self) -> bool {
            !self.glGetShaderInfoLog_p.load(RELAX).is_null()
        }
        /// [glGetShaderPrecisionFormat](http://docs.gl/es3/glGetShaderPrecisionFormat)(shadertype, precisiontype, range, precision)
        /// * `shadertype` group: ShaderType
        /// * `precisiontype` group: PrecisionType
        /// * `range` len: 2
        /// * `precision` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetShaderPrecisionFormat(
            &self,
            shadertype: GLenum,
            precisiontype: GLenum,
            range: *mut GLint,
            precision: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetShaderPrecisionFormat({:#X}, {:#X}, {:p}, {:p});",
                    shadertype,
                    precisiontype,
                    range,
                    precision
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetShaderPrecisionFormat",
                &self.glGetShaderPrecisionFormat_p,
                shadertype,
                precisiontype,
                range,
                precision,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetShaderPrecisionFormat");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetShaderPrecisionFormat_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetShaderPrecisionFormat\0",
                &self.glGetShaderPrecisionFormat_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetShaderPrecisionFormat_is_loaded(&self) -> bool {
            !self.glGetShaderPrecisionFormat_p.load(RELAX).is_null()
        }
        /// [glGetShaderSource](http://docs.gl/es3/glGetShaderSource)(shader, bufSize, length, source)
        /// * `shader` class: shader
        /// * `length` len: 1
        /// * `source` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetShaderSource(
            &self,
            shader: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            source: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetShaderSource({:?}, {:?}, {:p}, {:p});",
                    shader,
                    bufSize,
                    length,
                    source
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetShaderSource",
                &self.glGetShaderSource_p,
                shader,
                bufSize,
                length,
                source,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetShaderSource");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetShaderSource_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetShaderSource\0",
                &self.glGetShaderSource_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetShaderSource_is_loaded(&self) -> bool {
            !self.glGetShaderSource_p.load(RELAX).is_null()
        }
        /// [glGetShaderiv](http://docs.gl/es3/glGetShaderiv)(shader, pname, params)
        /// * `shader` class: shader
        /// * `pname` group: ShaderParameterName
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetShaderiv(&self, shader: GLuint, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetShaderiv({:?}, {:#X}, {:p});",
                    shader,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetShaderiv",
                &self.glGetShaderiv_p,
                shader,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetShaderiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetShaderiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetShaderiv\0", &self.glGetShaderiv_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetShaderiv_is_loaded(&self) -> bool {
            !self.glGetShaderiv_p.load(RELAX).is_null()
        }
        /// [glGetShadingRateImagePaletteNV](http://docs.gl/es3/glGetShadingRateImagePaletteNV)(viewport, entry, rate)
        /// * `rate` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        pub unsafe fn GetShadingRateImagePaletteNV(
            &self,
            viewport: GLuint,
            entry: GLuint,
            rate: *mut GLenum,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetShadingRateImagePaletteNV({:?}, {:?}, {:p});",
                    viewport,
                    entry,
                    rate
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetShadingRateImagePaletteNV",
                &self.glGetShadingRateImagePaletteNV_p,
                viewport,
                entry,
                rate,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetShadingRateImagePaletteNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        #[doc(hidden)]
        pub unsafe fn GetShadingRateImagePaletteNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetShadingRateImagePaletteNV\0",
                &self.glGetShadingRateImagePaletteNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        pub fn GetShadingRateImagePaletteNV_is_loaded(&self) -> bool {
            !self.glGetShadingRateImagePaletteNV_p.load(RELAX).is_null()
        }
        /// [glGetShadingRateSampleLocationivNV](http://docs.gl/es3/glGetShadingRateSampleLocationivNV)(rate, samples, index, location)
        /// * `location` len: 3
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        pub unsafe fn GetShadingRateSampleLocationivNV(
            &self,
            rate: GLenum,
            samples: GLuint,
            index: GLuint,
            location: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetShadingRateSampleLocationivNV({:#X}, {:?}, {:?}, {:p});",
                    rate,
                    samples,
                    index,
                    location
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetShadingRateSampleLocationivNV",
                &self.glGetShadingRateSampleLocationivNV_p,
                rate,
                samples,
                index,
                location,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetShadingRateSampleLocationivNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        #[doc(hidden)]
        pub unsafe fn GetShadingRateSampleLocationivNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetShadingRateSampleLocationivNV\0",
                &self.glGetShadingRateSampleLocationivNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        pub fn GetShadingRateSampleLocationivNV_is_loaded(&self) -> bool {
            !self
                .glGetShadingRateSampleLocationivNV_p
                .load(RELAX)
                .is_null()
        }
        /// [glGetString](http://docs.gl/es3/glGetString)(name)
        /// * `name` group: StringName
        /// * return value group: String
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetString(&self, name: GLenum) -> *const GLubyte {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetString({:#X});", name);
            }
            let out = call_atomic_ptr_1arg("glGetString", &self.glGetString_p, name);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetString");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetString_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetString\0", &self.glGetString_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetString_is_loaded(&self) -> bool {
            !self.glGetString_p.load(RELAX).is_null()
        }
        /// [glGetStringi](http://docs.gl/es3/glGetString)(name, index)
        /// * `name` group: StringName
        /// * return value group: String
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetStringi(&self, name: GLenum, index: GLuint) -> *const GLubyte {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetStringi({:#X}, {:?});", name, index);
            }
            let out = call_atomic_ptr_2arg("glGetStringi", &self.glGetStringi_p, name, index);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetStringi");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetStringi_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetStringi\0", &self.glGetStringi_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetStringi_is_loaded(&self) -> bool {
            !self.glGetStringi_p.load(RELAX).is_null()
        }
        /// [glGetSynciv](http://docs.gl/es3/glGetSync)(sync, pname, count, length, values)
        /// * `sync` group: sync
        /// * `sync` class: sync
        /// * `pname` group: SyncParameterName
        /// * `length` len: 1
        /// * `values` len: count
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetSynciv(
            &self,
            sync: GLsync,
            pname: GLenum,
            count: GLsizei,
            length: *mut GLsizei,
            values: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSynciv({:p}, {:#X}, {:?}, {:p}, {:p});",
                    sync,
                    pname,
                    count,
                    length,
                    values
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetSynciv",
                &self.glGetSynciv_p,
                sync,
                pname,
                count,
                length,
                values,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSynciv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetSynciv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(get_proc_address, b"glGetSynciv\0", &self.glGetSynciv_p)
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetSynciv_is_loaded(&self) -> bool {
            !self.glGetSynciv_p.load(RELAX).is_null()
        }
        /// [glGetSyncivAPPLE](http://docs.gl/es3/glGetSyncivAPPLE)(sync, pname, count, length, values)
        /// * `sync` class: sync
        /// * `pname` group: SyncParameterName
        /// * `values` len: count
        /// * alias of: [`glGetSynciv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub unsafe fn GetSyncivAPPLE(
            &self,
            sync: GLsync,
            pname: GLenum,
            count: GLsizei,
            length: *mut GLsizei,
            values: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetSyncivAPPLE({:p}, {:#X}, {:?}, {:p}, {:p});",
                    sync,
                    pname,
                    count,
                    length,
                    values
                );
            }
            let out = call_atomic_ptr_5arg(
                "glGetSyncivAPPLE",
                &self.glGetSyncivAPPLE_p,
                sync,
                pname,
                count,
                length,
                values,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetSyncivAPPLE");
            }
            out
        }
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        #[doc(hidden)]
        pub unsafe fn GetSyncivAPPLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetSyncivAPPLE\0",
                &self.glGetSyncivAPPLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        pub fn GetSyncivAPPLE_is_loaded(&self) -> bool {
            !self.glGetSyncivAPPLE_p.load(RELAX).is_null()
        }
        /// [glGetTexLevelParameterfv](http://docs.gl/es3/glGetTexLevelParameter)(target, level, pname, params)
        /// * `target` group: TextureTarget
        /// * `level` group: CheckedInt32
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetTexLevelParameterfv(
            &self,
            target: GLenum,
            level: GLint,
            pname: GLenum,
            params: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexLevelParameterfv({:#X}, {:?}, {:#X}, {:p});",
                    target,
                    level,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetTexLevelParameterfv",
                &self.glGetTexLevelParameterfv_p,
                target,
                level,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexLevelParameterfv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetTexLevelParameterfv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexLevelParameterfv\0",
                &self.glGetTexLevelParameterfv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetTexLevelParameterfv_is_loaded(&self) -> bool {
            !self.glGetTexLevelParameterfv_p.load(RELAX).is_null()
        }
        /// [glGetTexLevelParameteriv](http://docs.gl/es3/glGetTexLevelParameter)(target, level, pname, params)
        /// * `target` group: TextureTarget
        /// * `level` group: CheckedInt32
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetTexLevelParameteriv(
            &self,
            target: GLenum,
            level: GLint,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexLevelParameteriv({:#X}, {:?}, {:#X}, {:p});",
                    target,
                    level,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetTexLevelParameteriv",
                &self.glGetTexLevelParameteriv_p,
                target,
                level,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexLevelParameteriv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetTexLevelParameteriv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexLevelParameteriv\0",
                &self.glGetTexLevelParameteriv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetTexLevelParameteriv_is_loaded(&self) -> bool {
            !self.glGetTexLevelParameteriv_p.load(RELAX).is_null()
        }
        /// [glGetTexParameterIiv](http://docs.gl/es3/glGetTexParameter)(target, pname, params)
        /// * `target` group: TextureTarget
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetTexParameterIiv(&self, target: GLenum, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexParameterIiv({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetTexParameterIiv",
                &self.glGetTexParameterIiv_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexParameterIiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetTexParameterIiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexParameterIiv\0",
                &self.glGetTexParameterIiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetTexParameterIiv_is_loaded(&self) -> bool {
            !self.glGetTexParameterIiv_p.load(RELAX).is_null()
        }
        /// [glGetTexParameterIivEXT](http://docs.gl/es3/glGetTexParameterIivEXT)(target, pname, params)
        /// * `target` group: TextureTarget
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetTexParameterIiv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        pub unsafe fn GetTexParameterIivEXT(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexParameterIivEXT({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetTexParameterIivEXT",
                &self.glGetTexParameterIivEXT_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexParameterIivEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        #[doc(hidden)]
        pub unsafe fn GetTexParameterIivEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexParameterIivEXT\0",
                &self.glGetTexParameterIivEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        pub fn GetTexParameterIivEXT_is_loaded(&self) -> bool {
            !self.glGetTexParameterIivEXT_p.load(RELAX).is_null()
        }
        /// [glGetTexParameterIivOES](http://docs.gl/es3/glGetTexParameterIivOES)(target, pname, params)
        /// * `target` group: TextureTarget
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetTexParameterIiv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        pub unsafe fn GetTexParameterIivOES(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexParameterIivOES({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetTexParameterIivOES",
                &self.glGetTexParameterIivOES_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexParameterIivOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        #[doc(hidden)]
        pub unsafe fn GetTexParameterIivOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexParameterIivOES\0",
                &self.glGetTexParameterIivOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        pub fn GetTexParameterIivOES_is_loaded(&self) -> bool {
            !self.glGetTexParameterIivOES_p.load(RELAX).is_null()
        }
        /// [glGetTexParameterIuiv](http://docs.gl/es3/glGetTexParameter)(target, pname, params)
        /// * `target` group: TextureTarget
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetTexParameterIuiv(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexParameterIuiv({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetTexParameterIuiv",
                &self.glGetTexParameterIuiv_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexParameterIuiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetTexParameterIuiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexParameterIuiv\0",
                &self.glGetTexParameterIuiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetTexParameterIuiv_is_loaded(&self) -> bool {
            !self.glGetTexParameterIuiv_p.load(RELAX).is_null()
        }
        /// [glGetTexParameterIuivEXT](http://docs.gl/es3/glGetTexParameterIuivEXT)(target, pname, params)
        /// * `target` group: TextureTarget
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetTexParameterIuiv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        pub unsafe fn GetTexParameterIuivEXT(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexParameterIuivEXT({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetTexParameterIuivEXT",
                &self.glGetTexParameterIuivEXT_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexParameterIuivEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        #[doc(hidden)]
        pub unsafe fn GetTexParameterIuivEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexParameterIuivEXT\0",
                &self.glGetTexParameterIuivEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        pub fn GetTexParameterIuivEXT_is_loaded(&self) -> bool {
            !self.glGetTexParameterIuivEXT_p.load(RELAX).is_null()
        }
        /// [glGetTexParameterIuivOES](http://docs.gl/es3/glGetTexParameterIuivOES)(target, pname, params)
        /// * `target` group: TextureTarget
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        /// * alias of: [`glGetTexParameterIuiv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        pub unsafe fn GetTexParameterIuivOES(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexParameterIuivOES({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetTexParameterIuivOES",
                &self.glGetTexParameterIuivOES_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexParameterIuivOES");
            }
            out
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        #[doc(hidden)]
        pub unsafe fn GetTexParameterIuivOES_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexParameterIuivOES\0",
                &self.glGetTexParameterIuivOES_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        pub fn GetTexParameterIuivOES_is_loaded(&self) -> bool {
            !self.glGetTexParameterIuivOES_p.load(RELAX).is_null()
        }
        /// [glGetTexParameterfv](http://docs.gl/es3/glGetTexParameter)(target, pname, params)
        /// * `target` group: TextureTarget
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetTexParameterfv(
            &self,
            target: GLenum,
            pname: GLenum,
            params: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexParameterfv({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetTexParameterfv",
                &self.glGetTexParameterfv_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexParameterfv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetTexParameterfv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexParameterfv\0",
                &self.glGetTexParameterfv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetTexParameterfv_is_loaded(&self) -> bool {
            !self.glGetTexParameterfv_p.load(RELAX).is_null()
        }
        /// [glGetTexParameteriv](http://docs.gl/es3/glGetTexParameter)(target, pname, params)
        /// * `target` group: TextureTarget
        /// * `pname` group: GetTextureParameter
        /// * `params` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetTexParameteriv(&self, target: GLenum, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTexParameteriv({:#X}, {:#X}, {:p});",
                    target,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetTexParameteriv",
                &self.glGetTexParameteriv_p,
                target,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTexParameteriv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetTexParameteriv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTexParameteriv\0",
                &self.glGetTexParameteriv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetTexParameteriv_is_loaded(&self) -> bool {
            !self.glGetTexParameteriv_p.load(RELAX).is_null()
        }
        /// [glGetTextureHandleIMG](http://docs.gl/es3/glGetTextureHandleIMG)(texture)
        /// * `texture` class: texture
        /// * alias of: [`glGetTextureHandleARB`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        pub unsafe fn GetTextureHandleIMG(&self, texture: GLuint) -> GLuint64 {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetTextureHandleIMG({:?});", texture);
            }
            let out = call_atomic_ptr_1arg(
                "glGetTextureHandleIMG",
                &self.glGetTextureHandleIMG_p,
                texture,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTextureHandleIMG");
            }
            out
        }
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        #[doc(hidden)]
        pub unsafe fn GetTextureHandleIMG_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTextureHandleIMG\0",
                &self.glGetTextureHandleIMG_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        pub fn GetTextureHandleIMG_is_loaded(&self) -> bool {
            !self.glGetTextureHandleIMG_p.load(RELAX).is_null()
        }
        /// [glGetTextureHandleNV](http://docs.gl/es3/glGetTextureHandleNV)(texture)
        /// * `texture` class: texture
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        pub unsafe fn GetTextureHandleNV(&self, texture: GLuint) -> GLuint64 {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetTextureHandleNV({:?});", texture);
            }
            let out = call_atomic_ptr_1arg(
                "glGetTextureHandleNV",
                &self.glGetTextureHandleNV_p,
                texture,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTextureHandleNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        #[doc(hidden)]
        pub unsafe fn GetTextureHandleNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTextureHandleNV\0",
                &self.glGetTextureHandleNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        pub fn GetTextureHandleNV_is_loaded(&self) -> bool {
            !self.glGetTextureHandleNV_p.load(RELAX).is_null()
        }
        /// [glGetTextureSamplerHandleIMG](http://docs.gl/es3/glGetTextureSamplerHandleIMG)(texture, sampler)
        /// * `texture` class: texture
        /// * `sampler` class: sampler
        /// * alias of: [`glGetTextureSamplerHandleARB`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        pub unsafe fn GetTextureSamplerHandleIMG(
            &self,
            texture: GLuint,
            sampler: GLuint,
        ) -> GLuint64 {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTextureSamplerHandleIMG({:?}, {:?});",
                    texture,
                    sampler
                );
            }
            let out = call_atomic_ptr_2arg(
                "glGetTextureSamplerHandleIMG",
                &self.glGetTextureSamplerHandleIMG_p,
                texture,
                sampler,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTextureSamplerHandleIMG");
            }
            out
        }
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        #[doc(hidden)]
        pub unsafe fn GetTextureSamplerHandleIMG_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTextureSamplerHandleIMG\0",
                &self.glGetTextureSamplerHandleIMG_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        pub fn GetTextureSamplerHandleIMG_is_loaded(&self) -> bool {
            !self.glGetTextureSamplerHandleIMG_p.load(RELAX).is_null()
        }
        /// [glGetTextureSamplerHandleNV](http://docs.gl/es3/glGetTextureSamplerHandleNV)(texture, sampler)
        /// * `texture` class: texture
        /// * `sampler` class: sampler
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        pub unsafe fn GetTextureSamplerHandleNV(
            &self,
            texture: GLuint,
            sampler: GLuint,
        ) -> GLuint64 {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTextureSamplerHandleNV({:?}, {:?});",
                    texture,
                    sampler
                );
            }
            let out = call_atomic_ptr_2arg(
                "glGetTextureSamplerHandleNV",
                &self.glGetTextureSamplerHandleNV_p,
                texture,
                sampler,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTextureSamplerHandleNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        #[doc(hidden)]
        pub unsafe fn GetTextureSamplerHandleNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTextureSamplerHandleNV\0",
                &self.glGetTextureSamplerHandleNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        pub fn GetTextureSamplerHandleNV_is_loaded(&self) -> bool {
            !self.glGetTextureSamplerHandleNV_p.load(RELAX).is_null()
        }
        /// [glGetTransformFeedbackVarying](http://docs.gl/es3/glGetTransformFeedbackVarying)(program, index, bufSize, length, size, type_, name)
        /// * `program` class: program
        /// * `length` len: 1
        /// * `size` len: 1
        /// * `type_` group: AttributeType
        /// * `type_` len: 1
        /// * `name` len: bufSize
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetTransformFeedbackVarying(
            &self,
            program: GLuint,
            index: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            size: *mut GLsizei,
            type_: *mut GLenum,
            name: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTransformFeedbackVarying({:?}, {:?}, {:?}, {:p}, {:p}, {:p}, {:p});",
                    program,
                    index,
                    bufSize,
                    length,
                    size,
                    type_,
                    name
                );
            }
            let out = call_atomic_ptr_7arg(
                "glGetTransformFeedbackVarying",
                &self.glGetTransformFeedbackVarying_p,
                program,
                index,
                bufSize,
                length,
                size,
                type_,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTransformFeedbackVarying");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetTransformFeedbackVarying_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTransformFeedbackVarying\0",
                &self.glGetTransformFeedbackVarying_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetTransformFeedbackVarying_is_loaded(&self) -> bool {
            !self.glGetTransformFeedbackVarying_p.load(RELAX).is_null()
        }
        /// [glGetTranslatedShaderSourceANGLE](http://docs.gl/es3/glGetTranslatedShaderSourceANGLE)(shader, bufSize, length, source)
        /// * `shader` class: shader
        /// * `length` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_ANGLE_translated_shader_source"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_translated_shader_source"))))]
        pub unsafe fn GetTranslatedShaderSourceANGLE(
            &self,
            shader: GLuint,
            bufSize: GLsizei,
            length: *mut GLsizei,
            source: *mut GLchar,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetTranslatedShaderSourceANGLE({:?}, {:?}, {:p}, {:p});",
                    shader,
                    bufSize,
                    length,
                    source
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetTranslatedShaderSourceANGLE",
                &self.glGetTranslatedShaderSourceANGLE_p,
                shader,
                bufSize,
                length,
                source,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetTranslatedShaderSourceANGLE");
            }
            out
        }
        #[cfg(any(feature = "GL_ANGLE_translated_shader_source"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_translated_shader_source"))))]
        #[doc(hidden)]
        pub unsafe fn GetTranslatedShaderSourceANGLE_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetTranslatedShaderSourceANGLE\0",
                &self.glGetTranslatedShaderSourceANGLE_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_ANGLE_translated_shader_source"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_translated_shader_source"))))]
        pub fn GetTranslatedShaderSourceANGLE_is_loaded(&self) -> bool {
            !self
                .glGetTranslatedShaderSourceANGLE_p
                .load(RELAX)
                .is_null()
        }
        /// [glGetUniformBlockIndex](http://docs.gl/es3/glGetUniformBlockIndex)(program, uniformBlockName)
        /// * `program` class: program
        /// * `uniformBlockName` len: COMPSIZE()
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetUniformBlockIndex(
            &self,
            program: GLuint,
            uniformBlockName: *const GLchar,
        ) -> GLuint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetUniformBlockIndex({:?}, {:p});",
                    program,
                    uniformBlockName
                );
            }
            let out = call_atomic_ptr_2arg(
                "glGetUniformBlockIndex",
                &self.glGetUniformBlockIndex_p,
                program,
                uniformBlockName,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetUniformBlockIndex");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetUniformBlockIndex_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetUniformBlockIndex\0",
                &self.glGetUniformBlockIndex_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetUniformBlockIndex_is_loaded(&self) -> bool {
            !self.glGetUniformBlockIndex_p.load(RELAX).is_null()
        }
        /// [glGetUniformIndices](http://docs.gl/es3/glGetUniformIndices)(program, uniformCount, uniformNames, uniformIndices)
        /// * `program` class: program
        /// * `uniformNames` len: COMPSIZE(uniformCount)
        /// * `uniformIndices` len: COMPSIZE(uniformCount)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetUniformIndices(
            &self,
            program: GLuint,
            uniformCount: GLsizei,
            uniformNames: *const *const GLchar,
            uniformIndices: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetUniformIndices({:?}, {:?}, {:p}, {:p});",
                    program,
                    uniformCount,
                    uniformNames,
                    uniformIndices
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetUniformIndices",
                &self.glGetUniformIndices_p,
                program,
                uniformCount,
                uniformNames,
                uniformIndices,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetUniformIndices");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetUniformIndices_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetUniformIndices\0",
                &self.glGetUniformIndices_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetUniformIndices_is_loaded(&self) -> bool {
            !self.glGetUniformIndices_p.load(RELAX).is_null()
        }
        /// [glGetUniformLocation](http://docs.gl/es3/glGetUniformLocation)(program, name)
        /// * `program` class: program
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetUniformLocation(&self, program: GLuint, name: *const GLchar) -> GLint {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetUniformLocation({:?}, {:p});", program, name);
            }
            let out = call_atomic_ptr_2arg(
                "glGetUniformLocation",
                &self.glGetUniformLocation_p,
                program,
                name,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetUniformLocation");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetUniformLocation_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetUniformLocation\0",
                &self.glGetUniformLocation_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetUniformLocation_is_loaded(&self) -> bool {
            !self.glGetUniformLocation_p.load(RELAX).is_null()
        }
        /// [glGetUniformfv](http://docs.gl/es3/glGetUniform)(program, location, params)
        /// * `program` class: program
        /// * `params` len: COMPSIZE(program,location)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetUniformfv(&self, program: GLuint, location: GLint, params: *mut GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetUniformfv({:?}, {:?}, {:p});",
                    program,
                    location,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetUniformfv",
                &self.glGetUniformfv_p,
                program,
                location,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetUniformfv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetUniformfv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetUniformfv\0",
                &self.glGetUniformfv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetUniformfv_is_loaded(&self) -> bool {
            !self.glGetUniformfv_p.load(RELAX).is_null()
        }
        /// [glGetUniformi64vNV](http://docs.gl/es3/glGetUniformi64vNV)(program, location, params)
        /// * `program` class: program
        /// * `params` len: COMPSIZE(program,location)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        pub unsafe fn GetUniformi64vNV(
            &self,
            program: GLuint,
            location: GLint,
            params: *mut GLint64EXT,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetUniformi64vNV({:?}, {:?}, {:p});",
                    program,
                    location,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetUniformi64vNV",
                &self.glGetUniformi64vNV_p,
                program,
                location,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetUniformi64vNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        #[doc(hidden)]
        pub unsafe fn GetUniformi64vNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetUniformi64vNV\0",
                &self.glGetUniformi64vNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        pub fn GetUniformi64vNV_is_loaded(&self) -> bool {
            !self.glGetUniformi64vNV_p.load(RELAX).is_null()
        }
        /// [glGetUniformiv](http://docs.gl/es3/glGetUniform)(program, location, params)
        /// * `program` class: program
        /// * `params` len: COMPSIZE(program,location)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetUniformiv(&self, program: GLuint, location: GLint, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetUniformiv({:?}, {:?}, {:p});",
                    program,
                    location,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetUniformiv",
                &self.glGetUniformiv_p,
                program,
                location,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetUniformiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetUniformiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetUniformiv\0",
                &self.glGetUniformiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetUniformiv_is_loaded(&self) -> bool {
            !self.glGetUniformiv_p.load(RELAX).is_null()
        }
        /// [glGetUniformuiv](http://docs.gl/es3/glGetUniform)(program, location, params)
        /// * `program` class: program
        /// * `params` len: COMPSIZE(program,location)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetUniformuiv(&self, program: GLuint, location: GLint, params: *mut GLuint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetUniformuiv({:?}, {:?}, {:p});",
                    program,
                    location,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetUniformuiv",
                &self.glGetUniformuiv_p,
                program,
                location,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetUniformuiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetUniformuiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetUniformuiv\0",
                &self.glGetUniformuiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetUniformuiv_is_loaded(&self) -> bool {
            !self.glGetUniformuiv_p.load(RELAX).is_null()
        }
        /// [glGetUnsignedBytei_vEXT](http://docs.gl/es3/glGetUnsignedBytei_vEXT)(target, index, data)
        /// * `data` len: COMPSIZE(target)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
        )]
        pub unsafe fn GetUnsignedBytei_vEXT(
            &self,
            target: GLenum,
            index: GLuint,
            data: *mut GLubyte,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetUnsignedBytei_vEXT({:#X}, {:?}, {:p});",
                    target,
                    index,
                    data
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetUnsignedBytei_vEXT",
                &self.glGetUnsignedBytei_vEXT_p,
                target,
                index,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetUnsignedBytei_vEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
        )]
        #[doc(hidden)]
        pub unsafe fn GetUnsignedBytei_vEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetUnsignedBytei_vEXT\0",
                &self.glGetUnsignedBytei_vEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
        )]
        pub fn GetUnsignedBytei_vEXT_is_loaded(&self) -> bool {
            !self.glGetUnsignedBytei_vEXT_p.load(RELAX).is_null()
        }
        /// [glGetUnsignedBytevEXT](http://docs.gl/es3/glGetUnsignedBytevEXT)(pname, data)
        /// * `pname` group: GetPName
        /// * `data` len: COMPSIZE(pname)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
        )]
        pub unsafe fn GetUnsignedBytevEXT(&self, pname: GLenum, data: *mut GLubyte) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetUnsignedBytevEXT({:#X}, {:p});", pname, data);
            }
            let out = call_atomic_ptr_2arg(
                "glGetUnsignedBytevEXT",
                &self.glGetUnsignedBytevEXT_p,
                pname,
                data,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetUnsignedBytevEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
        )]
        #[doc(hidden)]
        pub unsafe fn GetUnsignedBytevEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetUnsignedBytevEXT\0",
                &self.glGetUnsignedBytevEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
        )]
        pub fn GetUnsignedBytevEXT_is_loaded(&self) -> bool {
            !self.glGetUnsignedBytevEXT_p.load(RELAX).is_null()
        }
        /// [glGetVertexAttribIiv](http://docs.gl/es3/glGetVertexAttrib)(index, pname, params)
        /// * `pname` group: VertexAttribEnum
        /// * `params` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetVertexAttribIiv(&self, index: GLuint, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetVertexAttribIiv({:?}, {:#X}, {:p});",
                    index,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetVertexAttribIiv",
                &self.glGetVertexAttribIiv_p,
                index,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetVertexAttribIiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetVertexAttribIiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetVertexAttribIiv\0",
                &self.glGetVertexAttribIiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetVertexAttribIiv_is_loaded(&self) -> bool {
            !self.glGetVertexAttribIiv_p.load(RELAX).is_null()
        }
        /// [glGetVertexAttribIuiv](http://docs.gl/es3/glGetVertexAttrib)(index, pname, params)
        /// * `pname` group: VertexAttribEnum
        /// * `params` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetVertexAttribIuiv(
            &self,
            index: GLuint,
            pname: GLenum,
            params: *mut GLuint,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetVertexAttribIuiv({:?}, {:#X}, {:p});",
                    index,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetVertexAttribIuiv",
                &self.glGetVertexAttribIuiv_p,
                index,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetVertexAttribIuiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetVertexAttribIuiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetVertexAttribIuiv\0",
                &self.glGetVertexAttribIuiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetVertexAttribIuiv_is_loaded(&self) -> bool {
            !self.glGetVertexAttribIuiv_p.load(RELAX).is_null()
        }
        /// [glGetVertexAttribPointerv](http://docs.gl/es3/glGetVertexAttribPointerv)(index, pname, pointer)
        /// * `pname` group: VertexAttribPointerPropertyARB
        /// * `pointer` len: 1
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetVertexAttribPointerv(
            &self,
            index: GLuint,
            pname: GLenum,
            pointer: *mut *mut c_void,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetVertexAttribPointerv({:?}, {:#X}, {:p});",
                    index,
                    pname,
                    pointer
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetVertexAttribPointerv",
                &self.glGetVertexAttribPointerv_p,
                index,
                pname,
                pointer,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetVertexAttribPointerv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetVertexAttribPointerv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetVertexAttribPointerv\0",
                &self.glGetVertexAttribPointerv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetVertexAttribPointerv_is_loaded(&self) -> bool {
            !self.glGetVertexAttribPointerv_p.load(RELAX).is_null()
        }
        /// [glGetVertexAttribfv](http://docs.gl/es3/glGetVertexAttrib)(index, pname, params)
        /// * `pname` group: VertexAttribPropertyARB
        /// * `params` len: 4
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetVertexAttribfv(&self, index: GLuint, pname: GLenum, params: *mut GLfloat) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetVertexAttribfv({:?}, {:#X}, {:p});",
                    index,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetVertexAttribfv",
                &self.glGetVertexAttribfv_p,
                index,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetVertexAttribfv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetVertexAttribfv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetVertexAttribfv\0",
                &self.glGetVertexAttribfv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetVertexAttribfv_is_loaded(&self) -> bool {
            !self.glGetVertexAttribfv_p.load(RELAX).is_null()
        }
        /// [glGetVertexAttribiv](http://docs.gl/es3/glGetVertexAttrib)(index, pname, params)
        /// * `pname` group: VertexAttribPropertyARB
        /// * `params` len: 4
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetVertexAttribiv(&self, index: GLuint, pname: GLenum, params: *mut GLint) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetVertexAttribiv({:?}, {:#X}, {:p});",
                    index,
                    pname,
                    params
                );
            }
            let out = call_atomic_ptr_3arg(
                "glGetVertexAttribiv",
                &self.glGetVertexAttribiv_p,
                index,
                pname,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetVertexAttribiv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetVertexAttribiv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetVertexAttribiv\0",
                &self.glGetVertexAttribiv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetVertexAttribiv_is_loaded(&self) -> bool {
            !self.glGetVertexAttribiv_p.load(RELAX).is_null()
        }
        /// [glGetVkProcAddrNV](http://docs.gl/es3/glGetVkProcAddrNV)(name)
        /// * `name` len: COMPSIZE(name)
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        pub unsafe fn GetVkProcAddrNV(&self, name: *const GLchar) -> GLVULKANPROCNV {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!("calling gl.GetVkProcAddrNV({:p});", name);
            }
            let out = call_atomic_ptr_1arg("glGetVkProcAddrNV", &self.glGetVkProcAddrNV_p, name);
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetVkProcAddrNV");
            }
            out
        }
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        #[doc(hidden)]
        pub unsafe fn GetVkProcAddrNV_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetVkProcAddrNV\0",
                &self.glGetVkProcAddrNV_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        pub fn GetVkProcAddrNV_is_loaded(&self) -> bool {
            !self.glGetVkProcAddrNV_p.load(RELAX).is_null()
        }
        /// [glGetnUniformfv](http://docs.gl/es3/glGetnUniform)(program, location, bufSize, params)
        /// * `program` class: program
        /// * `params` len: bufSize / 4
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        pub unsafe fn GetnUniformfv(
            &self,
            program: GLuint,
            location: GLint,
            bufSize: GLsizei,
            params: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetnUniformfv({:?}, {:?}, {:?}, {:p});",
                    program,
                    location,
                    bufSize,
                    params
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetnUniformfv",
                &self.glGetnUniformfv_p,
                program,
                location,
                bufSize,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetnUniformfv");
            }
            out
        }
        #[doc(hidden)]
        pub unsafe fn GetnUniformfv_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetnUniformfv\0",
                &self.glGetnUniformfv_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        pub fn GetnUniformfv_is_loaded(&self) -> bool {
            !self.glGetnUniformfv_p.load(RELAX).is_null()
        }
        /// [glGetnUniformfvEXT](http://docs.gl/es3/glGetnUniformfvEXT)(program, location, bufSize, params)
        /// * `program` class: program
        /// * `params` len: bufSize / 4
        /// * alias of: [`glGetnUniformfv`]
        #[cfg_attr(feature = "inline", inline)]
        #[cfg_attr(feature = "inline_always", inline(always))]
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        pub unsafe fn GetnUniformfvEXT(
            &self,
            program: GLuint,
            location: GLint,
            bufSize: GLsizei,
            params: *mut GLfloat,
        ) {
            #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
            {
                trace!(
                    "calling gl.GetnUniformfvEXT({:?}, {:?}, {:?}, {:p});",
                    program,
                    location,
                    bufSize,
                    params
                );
            }
            let out = call_atomic_ptr_4arg(
                "glGetnUniformfvEXT",
                &self.glGetnUniformfvEXT_p,
                program,
                location,
                bufSize,
                params,
            );
            #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
            {
                self.automatic_glGetError("glGetnUniformfvEXT");
            }
            out
        }
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        #[doc(hidden)]
        pub unsafe fn GetnUniformfvEXT_load_with_dyn(
            &self,
            get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
        ) -> bool {
            load_dyn_name_atomic_ptr(
                get_proc_address,
                b"glGetnUniformfvEXT\0",
                &self.glGetnUniformfvEXT_p,
            )
        }
        #[inline]
        #[doc(hidden)]
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        pub fn GetnUniformfvEXT_is_loaded(&self) -> bool {
            !self.glGetnUniformfvEXT_p.load(RELAX).is_null()
        }
        /// [glGetnUniformfvKHR](http://docs.gl/es3/g