#![allow(unused_attributes)]
#![no_std]
#![allow(bad_style)]
#![deny(missing_docs)]
#![deny(missing_debug_implementations)]
#![cfg_attr(docs_rs, feature(doc_cfg))]

//! Supported Features:
//! * `global_loader`: Include all mechanisms necessary for calling GL using
//!   global functions.
//! * `struct_loader`: Include all mechanisms necessary for calling GL as
//!   methods on a struct.
//! * `debug_trace_calls`: if cfg!(debug_assertions), any call to a GL function
//!   will `trace!` what was called and with what args.
//! * `debug_automatic_glGetError`: If cfg!(debug_assertions), this will
//!   automatically call `glGetError` after every call to any *other* GL
//!   function. If an error code occurs it's shown via `error!` along with the
//!   name of the function that had the error.
//! * `log`: imports `trace!` and `error!` macros from the `log` crate.
//!   Otherwise they just call `println!` and `eprintln!` respectively.
//! * `chlorine`: gets all C types from the `chlorine` crate (which is `no_std`
//!   friendly). Otherwise they will be imported from `std::os::raw`.
//! * `bytemuck`: Adds support for the `bytemuck` crate, mostly in the form of
//!   `bytemuck::Zeroable` on `GlFns`.
//! * `inline`: Tags all GL calls as `#[inline]`.
//! * `inline_always`: Tags all GL calls as `#[inline(always)]`. This will
//!   effectively override the `inline` feature.
//!
//! The crate is `no_std` friendly by default, but features above can end up
//! requiring `std` to be available.
//!
//! # GL Loaders
//! The docs for this crate hosted on docs.rs generate **both** the
//! `global_loader` and `struct_loader` documentation for sake of completeness.
//!
//! However, you are generally expected to use **only one** loader style in any
//! particular project.
//!
//! Each loader style has its own small advantages:
//! * The `global_loader` stores the GL function pointers in static `AtomicPtr`
//!   values.
//!   * Call [`load_global_gl_with`] to initialize the pointers.
//!   * Each GL function is available as a global function under its standard
//!     name, eg `glGetError()`.
//!   * This lets you call GL functions from anywhere at all, and it's how you
//!     might expect to use GL if you have a C background.
//!   * Being able to call GL from anywhere makes it easy to write Drop impls,
//!     among other things.
// * The `struct_loader` stores all the function pointers in the fields of a
//   [`GlFns`] struct.
//   * Call [`GlFns::load_with`] to make a `GlFns` value.
//   * Each GL function is available as a method on the struct with the `gl`
//     prefix removed. It's presumed that you'll call the struct itself `gl`,
//     so calls will look something like `gl.GetError()`.
//   * This is closer to how WebGL works on WASM targets, and so this is how
//     the [`glow`](https://docs.rs/glow) crate works to maintain consistency
//     across desktop and web.
//   * Also, if you want to do any sort of "live code reloading" you'll have to
//     use the struct loader. DLLs don't share their static values with the
//     main program, so if the DLL uses the global loader functions the
//     pointers won't be loaded and calling any GL function from the DLL will
//     panic. Instead, if you just pass a `&GlFns` to your DLL it can call the
//     GL methods just fine.
//
//! In both styles, if you call a function that isn't loaded you will get a
//! panic. This generally only happens if the context doesn't fully support
//! the GL version. You can check if a GL command is loaded or not before
//! actually calling it by adding `_is_loaded` to the name of the command. In
//! other words, `glGetError_is_loaded` to check if `glGetError` is globally
//! loaded, and `gl.GetError_is_loaded` to check if it's loaded in a `GlFns`.
//! All of the "`_is_loaded`" functions are hidden in the generated docs just
//! to keep things tidy, but they're there.
//!
//! # Safety
//! In general, there's many ways that GL can go wrong.
//!
//! For the purposes of this library, it's important to focus on the fact that:
//! * Initially all functions are null pointers. If a function is called when it's in a null state then you'll get a panic (reminder: a panic is safe).
//! * You can load pointers from the current GL context (described above).
//!   * These pointers are technically context specific, though in practice different contexts for the same graphics driver often all share the same function pointers.
//!   * The loader has no way to verify that pointers it gets are actually pointers to the correct functions, it just trusts what you tell it.
//! * Since loading a function pointer transitions the world from "it will definitely (safely) panic to call that GL command" to "it might be UB to call that GL command (even with the correct arguments)", the act of simply loading a function pointer is itself considered to be `unsafe`.
//! * Individual GL commands are generally safe to use once they've been properly loaded for the current context, but this crate doesn't attempt to sort out what is safe and what's not. All GL commands are blanket marked as being `unsafe`.
//! It's up to you to try and manage this unsafety! Sorry, but this crate just does what you tell it to.

#[cfg(any(
    all(
        not(feature = "log"),
        any(feature = "debug_trace_calls", feature = "debug_automatic_glGetError")
    ),
    not(feature = "chlorine"),
))]
extern crate std;

#[cfg(feature = "chlorine")]
use chlorine::*;
#[cfg(not(feature = "chlorine"))]
use std::os::raw::*;

#[cfg(feature = "log")]
#[allow(unused)]
use log::{error, trace};
#[cfg(all(not(feature = "log"), feature = "debug_trace_calls"))]
macro_rules! trace { ($($arg:tt)*) => { std::println!($($arg)*) } }
#[cfg(all(not(feature = "log"), feature = "debug_automatic_glGetError"))]
macro_rules! error { ($($arg:tt)*) => { std::eprintln!($($arg)*) } }

use core::{
    mem::transmute,
    ptr::null_mut,
    sync::atomic::{AtomicPtr, Ordering},
};
#[allow(dead_code)]
const RELAX: Ordering = Ordering::Relaxed;
#[allow(dead_code)]
type APcv = AtomicPtr<c_void>;
#[cfg(not(feature = "struct_loader"))]
const fn ap_null() -> APcv {
    AtomicPtr::new(null_mut())
}

/// This is called to panic when a not-loaded function is attempted.
///
/// Placing the panic mechanism in this cold function generally helps code generation for the hot path.
/// Or so the sages say, at least.
#[cold]
#[inline(never)]
#[allow(dead_code)]
fn go_panic_because_fn_not_loaded(name: &str) -> ! {
    panic!("called {name} but it was not loaded.", name = name)
}

/// Loads a function pointer.
/// Rejects suggested pointer addresses which are likely to be lies.
/// This function is used by both the global loader and struct loader.
/// We mark it as `inline(never)` to favor a small binary over initialization speed.
/// Returns if there's now a non-null value in the atomic pointer.
#[inline(never)]
#[allow(dead_code)]
fn load_dyn_name_atomic_ptr(
    get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    fn_name: &[u8],
    ptr: &APcv,
) -> bool {
    // if this fails the code generator itself royally screwed up somehow,
    // and so it's only a debug assert.
    debug_assert_eq!(*fn_name.last().unwrap(), 0);
    let p: *mut c_void = get_proc_address(fn_name.as_ptr() as *const c_char);
    let p_usize = p as usize;
    // You *should* get null for failed lookups, but some systems have been
    // reported to give "error code" values such as -1 or small non-null values.
    // To help guard against this silliness, we consider these values to also
    // just be a result of null.
    if p_usize == core::usize::MAX || p_usize < 8 {
        ptr.store(null_mut(), RELAX);
        false
    } else {
        ptr.store(p, RELAX);
        true
    }
}

/// Returns if an error was printed.
#[cfg(feature = "debug_automatic_glGetError")]
#[inline(never)]
fn report_error_code_from(name: &str, err: GLenum) {
    match err {
        GL_NO_ERROR => return,
        GL_INVALID_ENUM => error!("Invalid Enum to {name}.", name = name),
        GL_INVALID_VALUE => error!("Invalid Value to {name}.", name = name),
        GL_INVALID_OPERATION => error!("Invalid Operation to {name}.", name = name),
        GL_INVALID_FRAMEBUFFER_OPERATION => {
            error!("Invalid Framebuffer Operation to {name}.", name = name)
        }
        GL_OUT_OF_MEMORY => error!("Out of Memory in {name}.", name = name),
        GL_STACK_UNDERFLOW => error!("Stack Underflow in {name}.", name = name),
        GL_STACK_OVERFLOW => error!("Stack Overflow in {name}.", name = name),
        unknown => error!(
            "Unknown error code {unknown} to {name}.",
            name = name,
            unknown = unknown
        ),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_0arg<Ret>(name: &str, ptr: &APcv) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn() -> Ret>>(p) {
        Some(fn_p) => fn_p(),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_1arg<Ret, A>(name: &str, ptr: &APcv, a: A) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A) -> Ret>>(p) {
        Some(fn_p) => fn_p(a),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_2arg<Ret, A, B>(name: &str, ptr: &APcv, a: A, b: B) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B) -> Ret>>(p) {
        Some(fn_p) => fn_p(a, b),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_3arg<Ret, A, B, C>(name: &str, ptr: &APcv, a: A, b: B, c: C) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B, C) -> Ret>>(p) {
        Some(fn_p) => fn_p(a, b, c),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_4arg<Ret, A, B, C, D>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B, C, D) -> Ret>>(p) {
        Some(fn_p) => fn_p(a, b, c, d),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_5arg<Ret, A, B, C, D, E>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
    e: E,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B, C, D, E) -> Ret>>(p) {
        Some(fn_p) => fn_p(a, b, c, d, e),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_6arg<Ret, A, B, C, D, E, F>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
    e: E,
    f: F,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B, C, D, E, F) -> Ret>>(p) {
        Some(fn_p) => fn_p(a, b, c, d, e, f),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_7arg<Ret, A, B, C, D, E, F, G>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
    e: E,
    f: F,
    g: G,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B, C, D, E, F, G) -> Ret>>(p) {
        Some(fn_p) => fn_p(a, b, c, d, e, f, g),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_8arg<Ret, A, B, C, D, E, F, G, H>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
    e: E,
    f: F,
    g: G,
    h: H,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B, C, D, E, F, G, H) -> Ret>>(p) {
        Some(fn_p) => fn_p(a, b, c, d, e, f, g, h),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_9arg<Ret, A, B, C, D, E, F, G, H, I>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
    e: E,
    f: F,
    g: G,
    h: H,
    i: I,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B, C, D, E, F, G, H, I) -> Ret>>(p)
    {
        Some(fn_p) => fn_p(a, b, c, d, e, f, g, h, i),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_10arg<Ret, A, B, C, D, E, F, G, H, I, J>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
    e: E,
    f: F,
    g: G,
    h: H,
    i: I,
    j: J,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B, C, D, E, F, G, H, I, J) -> Ret>>(
        p,
    ) {
        Some(fn_p) => fn_p(a, b, c, d, e, f, g, h, i, j),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_11arg<Ret, A, B, C, D, E, F, G, H, I, J, K>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
    e: E,
    f: F,
    g: G,
    h: H,
    i: I,
    j: J,
    k: K,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<*mut c_void, Option<extern "system" fn(A, B, C, D, E, F, G, H, I, J, K) -> Ret>>(
        p,
    ) {
        Some(fn_p) => fn_p(a, b, c, d, e, f, g, h, i, j, k),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_12arg<Ret, A, B, C, D, E, F, G, H, I, J, K, L>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
    e: E,
    f: F,
    g: G,
    h: H,
    i: I,
    j: J,
    k: K,
    l: L,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<
        *mut c_void,
        Option<extern "system" fn(A, B, C, D, E, F, G, H, I, J, K, L) -> Ret>,
    >(p)
    {
        Some(fn_p) => fn_p(a, b, c, d, e, f, g, h, i, j, k, l),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[inline(always)]
#[allow(dead_code)]
unsafe fn call_atomic_ptr_15arg<Ret, A, B, C, D, E, F, G, H, I, J, K, L, M, N, O>(
    name: &str,
    ptr: &APcv,
    a: A,
    b: B,
    c: C,
    d: D,
    e: E,
    f: F,
    g: G,
    h: H,
    i: I,
    j: J,
    k: K,
    l: L,
    m: M,
    n: N,
    o: O,
) -> Ret {
    let p = ptr.load(RELAX);
    match transmute::<
        *mut c_void,
        Option<extern "system" fn(A, B, C, D, E, F, G, H, I, J, K, L, M, N, O) -> Ret>,
    >(p)
    {
        Some(fn_p) => fn_p(a, b, c, d, e, f, g, h, i, j, k, l, m, n, o),
        None => go_panic_because_fn_not_loaded(name),
    }
}

#[cfg(not(feature = "struct_loader"))]
pub use global_commands::*;
#[cfg(not(feature = "struct_loader"))]
pub mod global_commands {
    //! Contains functions for using the global GL loader.
    use super::*;
    use crate::gles::types::*;

    #[cfg(feature = "debug_automatic_glGetError")]
    #[inline(never)]
    unsafe fn global_automatic_glGetError(name: &str) {
        let mut err = glGetError();
        while err != GL_NO_ERROR {
            report_error_code_from(name, err);
            err = glGetError();
        }
    }

    /// Loads all global functions using the `get_proc_address` given.
    ///
    /// The closure should, when given a null-terminated name of a function,
    /// return a pointer to that function. If the function isn't available, then
    /// a null pointer should be returned instead.
    ///
    /// This allows you to call [SDL_GL_GetProcAddress](https://wiki.libsdl.org/SDL_GL_GetProcAddress),
    /// [wglGetProcAddress](https://docs.microsoft.com/en-us/windows/win32/api/wingdi/nf-wingdi-wglgetprocaddress),
    /// or some similar function, depending on your OS.
    pub unsafe fn load_global_gl_with<F>(mut get_proc_address: F)
    where
        F: FnMut(*const c_char) -> *mut c_void,
    {
        #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
        {
            glAcquireKeyedMutexWin32EXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glActiveShaderProgram_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glActiveShaderProgramEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glActiveTexture_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_QCOM_alpha_test"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_alpha_test"))))]
        {
            glAlphaFuncQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))))]
        {
            glApplyFramebufferAttachmentCMAAINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        glAttachShader_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_conditional_render"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
        {
            glBeginConditionalRenderNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glBeginPerfMonitorAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glBeginPerfQueryINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBeginQuery_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        {
            glBeginQueryEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBeginTransformFeedback_load_with_dyn(&mut get_proc_address) as usize;
        glBindAttribLocation_load_with_dyn(&mut get_proc_address) as usize;
        glBindBuffer_load_with_dyn(&mut get_proc_address) as usize;
        glBindBufferBase_load_with_dyn(&mut get_proc_address) as usize;
        glBindBufferRange_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        {
            glBindFragDataLocationEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        {
            glBindFragDataLocationIndexedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBindFramebuffer_load_with_dyn(&mut get_proc_address) as usize;
        glBindImageTexture_load_with_dyn(&mut get_proc_address) as usize;
        glBindProgramPipeline_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glBindProgramPipelineEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBindRenderbuffer_load_with_dyn(&mut get_proc_address) as usize;
        glBindSampler_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        {
            glBindShadingRateImageNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBindTexture_load_with_dyn(&mut get_proc_address) as usize;
        glBindTransformFeedback_load_with_dyn(&mut get_proc_address) as usize;
        glBindVertexArray_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        {
            glBindVertexArrayOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBindVertexBuffer_load_with_dyn(&mut get_proc_address) as usize;
        glBlendBarrier_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_blend_equation_advanced"))))]
        {
            glBlendBarrierKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
        {
            glBlendBarrierNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBlendColor_load_with_dyn(&mut get_proc_address) as usize;
        glBlendEquation_load_with_dyn(&mut get_proc_address) as usize;
        glBlendEquationSeparate_load_with_dyn(&mut get_proc_address) as usize;
        glBlendEquationSeparatei_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        {
            glBlendEquationSeparateiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        {
            glBlendEquationSeparateiOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBlendEquationi_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        {
            glBlendEquationiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        {
            glBlendEquationiOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBlendFunc_load_with_dyn(&mut get_proc_address) as usize;
        glBlendFuncSeparate_load_with_dyn(&mut get_proc_address) as usize;
        glBlendFuncSeparatei_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        {
            glBlendFuncSeparateiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        {
            glBlendFuncSeparateiOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBlendFunci_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        {
            glBlendFunciEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        {
            glBlendFunciOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
        {
            glBlendParameteriNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBlitFramebuffer_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_ANGLE_framebuffer_blit"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_blit"))))]
        {
            glBlitFramebufferANGLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_framebuffer_blit"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_blit"))))]
        {
            glBlitFramebufferNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        {
            glBufferAttachMemoryNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBufferData_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
        {
            glBufferPageCommitmentMemNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_buffer_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_buffer_storage"))))]
        {
            glBufferStorageEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_external_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
        {
            glBufferStorageExternalEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glBufferStorageMemEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glBufferSubData_load_with_dyn(&mut get_proc_address) as usize;
        glCheckFramebufferStatus_load_with_dyn(&mut get_proc_address) as usize;
        glClear_load_with_dyn(&mut get_proc_address) as usize;
        glClearBufferfi_load_with_dyn(&mut get_proc_address) as usize;
        glClearBufferfv_load_with_dyn(&mut get_proc_address) as usize;
        glClearBufferiv_load_with_dyn(&mut get_proc_address) as usize;
        glClearBufferuiv_load_with_dyn(&mut get_proc_address) as usize;
        glClearColor_load_with_dyn(&mut get_proc_address) as usize;
        glClearDepthf_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        {
            glClearPixelLocalStorageuiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glClearStencil_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_clear_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
        {
            glClearTexImageEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_clear_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
        {
            glClearTexSubImageEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glClientWaitSync_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        {
            glClientWaitSyncAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_clip_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clip_control"))))]
        {
            glClipControlEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glColorMask_load_with_dyn(&mut get_proc_address) as usize;
        glColorMaski_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        {
            glColorMaskiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
        {
            glColorMaskiOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glCompileShader_load_with_dyn(&mut get_proc_address) as usize;
        glCompressedTexImage2D_load_with_dyn(&mut get_proc_address) as usize;
        glCompressedTexImage3D_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        {
            glCompressedTexImage3DOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glCompressedTexSubImage2D_load_with_dyn(&mut get_proc_address) as usize;
        glCompressedTexSubImage3D_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        {
            glCompressedTexSubImage3DOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles")))
        )]
        {
            glConservativeRasterParameteriNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glCopyBufferSubData_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_copy_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_copy_buffer"))))]
        {
            glCopyBufferSubDataNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glCopyImageSubData_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_copy_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_copy_image"))))]
        {
            glCopyImageSubDataEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_copy_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_copy_image"))))]
        {
            glCopyImageSubDataOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glCopyPathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glCopyTexImage2D_load_with_dyn(&mut get_proc_address) as usize;
        glCopyTexSubImage2D_load_with_dyn(&mut get_proc_address) as usize;
        glCopyTexSubImage3D_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        {
            glCopyTexSubImage3DOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_APPLE_copy_texture_levels"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_copy_texture_levels"))))]
        {
            glCopyTextureLevelsAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glCoverFillPathInstancedNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glCoverFillPathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glCoverStrokePathInstancedNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glCoverStrokePathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_coverage_sample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
        {
            glCoverageMaskNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        {
            glCoverageModulationNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        {
            glCoverageModulationTableNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_coverage_sample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
        {
            glCoverageOperationNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glCreateMemoryObjectsEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glCreatePerfQueryINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        glCreateProgram_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
        {
            glCreateSemaphoresNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glCreateShader_load_with_dyn(&mut get_proc_address) as usize;
        glCreateShaderProgramv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glCreateShaderProgramvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glCullFace_load_with_dyn(&mut get_proc_address) as usize;
        glDebugMessageCallback_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glDebugMessageCallbackKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDebugMessageControl_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glDebugMessageControlKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDebugMessageInsert_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glDebugMessageInsertKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDeleteBuffers_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        {
            glDeleteFencesNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDeleteFramebuffers_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glDeleteMemoryObjectsEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glDeletePathsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glDeletePerfMonitorsAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glDeletePerfQueryINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDeleteProgram_load_with_dyn(&mut get_proc_address) as usize;
        glDeleteProgramPipelines_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glDeleteProgramPipelinesEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDeleteQueries_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        {
            glDeleteQueriesEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDeleteRenderbuffers_load_with_dyn(&mut get_proc_address) as usize;
        glDeleteSamplers_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        {
            glDeleteSemaphoresEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDeleteShader_load_with_dyn(&mut get_proc_address) as usize;
        glDeleteSync_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        {
            glDeleteSyncAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDeleteTextures_load_with_dyn(&mut get_proc_address) as usize;
        glDeleteTransformFeedbacks_load_with_dyn(&mut get_proc_address) as usize;
        glDeleteVertexArrays_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        {
            glDeleteVertexArraysOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDepthFunc_load_with_dyn(&mut get_proc_address) as usize;
        glDepthMask_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glDepthRangeArrayfvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        {
            glDepthRangeArrayfvOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glDepthRangeIndexedfNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        {
            glDepthRangeIndexedfOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDepthRangef_load_with_dyn(&mut get_proc_address) as usize;
        glDetachShader_load_with_dyn(&mut get_proc_address) as usize;
        glDisable_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        {
            glDisableDriverControlQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDisableVertexAttribArray_load_with_dyn(&mut get_proc_address) as usize;
        glDisablei_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        {
            glDisableiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glDisableiNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            )))
        )]
        {
            glDisableiOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_discard_framebuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_discard_framebuffer"))))]
        {
            glDiscardFramebufferEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDispatchCompute_load_with_dyn(&mut get_proc_address) as usize;
        glDispatchComputeIndirect_load_with_dyn(&mut get_proc_address) as usize;
        glDrawArrays_load_with_dyn(&mut get_proc_address) as usize;
        glDrawArraysIndirect_load_with_dyn(&mut get_proc_address) as usize;
        glDrawArraysInstanced_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
        {
            glDrawArraysInstancedANGLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        {
            glDrawArraysInstancedBaseInstanceEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_draw_instanced",
                feature = "GL_EXT_instanced_arrays"
            )))
        )]
        {
            glDrawArraysInstancedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_draw_instanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
        {
            glDrawArraysInstancedNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDrawBuffers_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers"))))]
        {
            glDrawBuffersEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
        {
            glDrawBuffersIndexedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_buffers"))))]
        {
            glDrawBuffersNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDrawElements_load_with_dyn(&mut get_proc_address) as usize;
        glDrawElementsBaseVertex_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        {
            glDrawElementsBaseVertexEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        {
            glDrawElementsBaseVertexOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDrawElementsIndirect_load_with_dyn(&mut get_proc_address) as usize;
        glDrawElementsInstanced_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
        {
            glDrawElementsInstancedANGLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        {
            glDrawElementsInstancedBaseInstanceEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDrawElementsInstancedBaseVertex_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_base_instance"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
        {
            glDrawElementsInstancedBaseVertexBaseInstanceEXT_load_with_dyn(&mut get_proc_address)
                as usize;
        }
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        {
            glDrawElementsInstancedBaseVertexEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        {
            glDrawElementsInstancedBaseVertexOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_draw_instanced",
                feature = "GL_EXT_instanced_arrays"
            )))
        )]
        {
            glDrawElementsInstancedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_draw_instanced"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
        {
            glDrawElementsInstancedNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        {
            glDrawMeshTasksIndirectNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        {
            glDrawMeshTasksNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glDrawRangeElements_load_with_dyn(&mut get_proc_address) as usize;
        glDrawRangeElementsBaseVertex_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
        {
            glDrawRangeElementsBaseVertexEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
        {
            glDrawRangeElementsBaseVertexOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
        {
            glDrawTransformFeedbackEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
        {
            glDrawTransformFeedbackInstancedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        {
            glDrawVkImageNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_EGL_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
        {
            glEGLImageTargetRenderbufferStorageOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
        {
            glEGLImageTargetTexStorageEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_EGL_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
        {
            glEGLImageTargetTexture2DOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
        {
            glEGLImageTargetTextureStorageEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glEnable_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        {
            glEnableDriverControlQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        glEnableVertexAttribArray_load_with_dyn(&mut get_proc_address) as usize;
        glEnablei_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        {
            glEnableiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glEnableiNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            )))
        )]
        {
            glEnableiOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_conditional_render"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
        {
            glEndConditionalRenderNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glEndPerfMonitorAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glEndPerfQueryINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        glEndQuery_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        {
            glEndQueryEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
        {
            glEndTilingQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        glEndTransformFeedback_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        {
            glExtGetBufferPointervQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        {
            glExtGetBuffersQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        {
            glExtGetFramebuffersQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        {
            glExtGetProgramBinarySourceQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        {
            glExtGetProgramsQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        {
            glExtGetRenderbuffersQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        {
            glExtGetShadersQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        {
            glExtGetTexLevelParameterivQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        {
            glExtGetTexSubImageQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        {
            glExtGetTexturesQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
        {
            glExtIsProgramBinaryQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_extended_get"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
        {
            glExtTexObjectStateOverrideiQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_frame_extrapolation"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_frame_extrapolation"))))]
        {
            glExtrapolateTex2DQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        glFenceSync_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        {
            glFenceSyncAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        glFinish_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        {
            glFinishFenceNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glFlush_load_with_dyn(&mut get_proc_address) as usize;
        glFlushMappedBufferRange_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
        {
            glFlushMappedBufferRangeEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_fragment_coverage_to_color"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fragment_coverage_to_color"))))]
        {
            glFragmentCoverageColorNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent")))
        )]
        {
            glFramebufferFetchBarrierEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent")))
        )]
        {
            glFramebufferFetchBarrierQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
        {
            glFramebufferFoveationConfigQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
        {
            glFramebufferFoveationParametersQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        glFramebufferParameteri_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
        {
            glFramebufferParameteriMESA_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        {
            glFramebufferPixelLocalStorageSizeEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glFramebufferRenderbuffer_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_sample_locations"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
        {
            glFramebufferSampleLocationsfvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glFramebufferTexture_load_with_dyn(&mut get_proc_address) as usize;
        glFramebufferTexture2D_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
        {
            glFramebufferTexture2DDownsampleIMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
        )]
        {
            glFramebufferTexture2DMultisampleEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
        )]
        {
            glFramebufferTexture2DMultisampleIMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        {
            glFramebufferTexture3DOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_geometry_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_geometry_shader"))))]
        {
            glFramebufferTextureEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glFramebufferTextureLayer_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
        {
            glFramebufferTextureLayerDownsampleIMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture")))
        )]
        {
            glFramebufferTextureMultisampleMultiviewOVR_load_with_dyn(&mut get_proc_address)
                as usize;
        }
        #[cfg(any(feature = "GL_OVR_multiview"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OVR_multiview"))))]
        {
            glFramebufferTextureMultiviewOVR_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_geometry_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_geometry_shader"))))]
        {
            glFramebufferTextureOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glFrontFace_load_with_dyn(&mut get_proc_address) as usize;
        glGenBuffers_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        {
            glGenFencesNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGenFramebuffers_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGenPathsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glGenPerfMonitorsAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGenProgramPipelines_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glGenProgramPipelinesEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGenQueries_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        {
            glGenQueriesEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGenRenderbuffers_load_with_dyn(&mut get_proc_address) as usize;
        glGenSamplers_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        {
            glGenSemaphoresEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGenTextures_load_with_dyn(&mut get_proc_address) as usize;
        glGenTransformFeedbacks_load_with_dyn(&mut get_proc_address) as usize;
        glGenVertexArrays_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        {
            glGenVertexArraysOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGenerateMipmap_load_with_dyn(&mut get_proc_address) as usize;
        glGetActiveAttrib_load_with_dyn(&mut get_proc_address) as usize;
        glGetActiveUniform_load_with_dyn(&mut get_proc_address) as usize;
        glGetActiveUniformBlockName_load_with_dyn(&mut get_proc_address) as usize;
        glGetActiveUniformBlockiv_load_with_dyn(&mut get_proc_address) as usize;
        glGetActiveUniformsiv_load_with_dyn(&mut get_proc_address) as usize;
        glGetAttachedShaders_load_with_dyn(&mut get_proc_address) as usize;
        glGetAttribLocation_load_with_dyn(&mut get_proc_address) as usize;
        glGetBooleani_v_load_with_dyn(&mut get_proc_address) as usize;
        glGetBooleanv_load_with_dyn(&mut get_proc_address) as usize;
        glGetBufferParameteri64v_load_with_dyn(&mut get_proc_address) as usize;
        glGetBufferParameteriv_load_with_dyn(&mut get_proc_address) as usize;
        glGetBufferPointerv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_mapbuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
        {
            glGetBufferPointervOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
        {
            glGetCoverageModulationTableNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetDebugMessageLog_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glGetDebugMessageLogKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        {
            glGetDriverControlStringQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_driver_control"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
        {
            glGetDriverControlsQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetError_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        {
            glGetFenceivNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glGetFirstPerfQueryIdINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glGetFloati_vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        {
            glGetFloati_vOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetFloatv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        {
            glGetFragDataIndexEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetFragDataLocation_load_with_dyn(&mut get_proc_address) as usize;
        glGetFramebufferAttachmentParameteriv_load_with_dyn(&mut get_proc_address) as usize;
        glGetFramebufferParameteriv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
        {
            glGetFramebufferParameterivMESA_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
        {
            glGetFramebufferPixelLocalStorageSizeEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetGraphicsResetStatus_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        {
            glGetGraphicsResetStatusEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_KHR_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
        {
            glGetGraphicsResetStatusKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glGetImageHandleNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetInteger64i_v_load_with_dyn(&mut get_proc_address) as usize;
        glGetInteger64v_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        {
            glGetInteger64vAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        {
            glGetInteger64vEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetIntegeri_v_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
        {
            glGetIntegeri_vEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetIntegerv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_internalformat_sample_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_internalformat_sample_query"))))]
        {
            glGetInternalformatSampleivNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetInternalformativ_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        {
            glGetMemoryObjectDetachedResourcesuivNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glGetMemoryObjectParameterivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetMultisamplefv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glGetNextPerfQueryIdINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetObjectLabel_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_debug_label"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
        {
            glGetObjectLabelEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glGetObjectLabelKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetObjectPtrLabel_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glGetObjectPtrLabelKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetPathCommandsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetPathCoordsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetPathDashArrayNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetPathLengthNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetPathMetricRangeNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetPathMetricsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetPathParameterfvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetPathParameterivNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetPathSpacingNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glGetPerfCounterInfoINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glGetPerfMonitorCounterDataAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glGetPerfMonitorCounterInfoAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glGetPerfMonitorCounterStringAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glGetPerfMonitorCountersAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glGetPerfMonitorGroupStringAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glGetPerfMonitorGroupsAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glGetPerfQueryDataINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glGetPerfQueryIdByNameINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_INTEL_performance_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
        {
            glGetPerfQueryInfoINTEL_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetPointerv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glGetPointervKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetProgramBinary_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_get_program_binary"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
        {
            glGetProgramBinaryOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetProgramInfoLog_load_with_dyn(&mut get_proc_address) as usize;
        glGetProgramInterfaceiv_load_with_dyn(&mut get_proc_address) as usize;
        glGetProgramPipelineInfoLog_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glGetProgramPipelineInfoLogEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetProgramPipelineiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glGetProgramPipelineivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetProgramResourceIndex_load_with_dyn(&mut get_proc_address) as usize;
        glGetProgramResourceLocation_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
        {
            glGetProgramResourceLocationIndexEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetProgramResourceName_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glGetProgramResourcefvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetProgramResourceiv_load_with_dyn(&mut get_proc_address) as usize;
        glGetProgramiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        {
            glGetQueryObjecti64vEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        {
            glGetQueryObjectivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        {
            glGetQueryObjectui64vEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetQueryObjectuiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        {
            glGetQueryObjectuivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetQueryiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        {
            glGetQueryivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetRenderbufferParameteriv_load_with_dyn(&mut get_proc_address) as usize;
        glGetSamplerParameterIiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        {
            glGetSamplerParameterIivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        {
            glGetSamplerParameterIivOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetSamplerParameterIuiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        {
            glGetSamplerParameterIuivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        {
            glGetSamplerParameterIuivOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetSamplerParameterfv_load_with_dyn(&mut get_proc_address) as usize;
        glGetSamplerParameteriv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
        {
            glGetSemaphoreParameterivNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        {
            glGetSemaphoreParameterui64vEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetShaderInfoLog_load_with_dyn(&mut get_proc_address) as usize;
        glGetShaderPrecisionFormat_load_with_dyn(&mut get_proc_address) as usize;
        glGetShaderSource_load_with_dyn(&mut get_proc_address) as usize;
        glGetShaderiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        {
            glGetShadingRateImagePaletteNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        {
            glGetShadingRateSampleLocationivNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetString_load_with_dyn(&mut get_proc_address) as usize;
        glGetStringi_load_with_dyn(&mut get_proc_address) as usize;
        glGetSynciv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        {
            glGetSyncivAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetTexLevelParameterfv_load_with_dyn(&mut get_proc_address) as usize;
        glGetTexLevelParameteriv_load_with_dyn(&mut get_proc_address) as usize;
        glGetTexParameterIiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        {
            glGetTexParameterIivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        {
            glGetTexParameterIivOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetTexParameterIuiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        {
            glGetTexParameterIuivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        {
            glGetTexParameterIuivOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetTexParameterfv_load_with_dyn(&mut get_proc_address) as usize;
        glGetTexParameteriv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        {
            glGetTextureHandleIMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glGetTextureHandleNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        {
            glGetTextureSamplerHandleIMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glGetTextureSamplerHandleNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetTransformFeedbackVarying_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_ANGLE_translated_shader_source"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_translated_shader_source"))))]
        {
            glGetTranslatedShaderSourceANGLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetUniformBlockIndex_load_with_dyn(&mut get_proc_address) as usize;
        glGetUniformIndices_load_with_dyn(&mut get_proc_address) as usize;
        glGetUniformLocation_load_with_dyn(&mut get_proc_address) as usize;
        glGetUniformfv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glGetUniformi64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetUniformiv_load_with_dyn(&mut get_proc_address) as usize;
        glGetUniformuiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
        )]
        {
            glGetUnsignedBytei_vEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_memory_object", feature = "GL_EXT_semaphore")))
        )]
        {
            glGetUnsignedBytevEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetVertexAttribIiv_load_with_dyn(&mut get_proc_address) as usize;
        glGetVertexAttribIuiv_load_with_dyn(&mut get_proc_address) as usize;
        glGetVertexAttribPointerv_load_with_dyn(&mut get_proc_address) as usize;
        glGetVertexAttribfv_load_with_dyn(&mut get_proc_address) as usize;
        glGetVertexAttribiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        {
            glGetVkProcAddrNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetnUniformfv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        {
            glGetnUniformfvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_KHR_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
        {
            glGetnUniformfvKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetnUniformiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        {
            glGetnUniformivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_KHR_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
        {
            glGetnUniformivKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glGetnUniformuiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
        {
            glGetnUniformuivKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glHint_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_memory_object_fd"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object_fd"))))]
        {
            glImportMemoryFdEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object_win32"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object_win32"))))]
        {
            glImportMemoryWin32HandleEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object_win32"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object_win32"))))]
        {
            glImportMemoryWin32NameEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_semaphore_fd"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore_fd"))))]
        {
            glImportSemaphoreFdEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_semaphore_win32"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore_win32"))))]
        {
            glImportSemaphoreWin32HandleEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_semaphore_win32"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore_win32"))))]
        {
            glImportSemaphoreWin32NameEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_debug_marker"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_marker"))))]
        {
            glInsertEventMarkerEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glInterpolatePathsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glInvalidateFramebuffer_load_with_dyn(&mut get_proc_address) as usize;
        glInvalidateSubFramebuffer_load_with_dyn(&mut get_proc_address) as usize;
        glIsBuffer_load_with_dyn(&mut get_proc_address) as usize;
        glIsEnabled_load_with_dyn(&mut get_proc_address) as usize;
        glIsEnabledi_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
        {
            glIsEnablediEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glIsEnablediNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_OES_draw_buffers_indexed",
                feature = "GL_OES_viewport_array"
            )))
        )]
        {
            glIsEnablediOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        {
            glIsFenceNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glIsFramebuffer_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glIsImageHandleResidentNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glIsMemoryObjectEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glIsPathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glIsPointInFillPathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glIsPointInStrokePathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glIsProgram_load_with_dyn(&mut get_proc_address) as usize;
        glIsProgramPipeline_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glIsProgramPipelineEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glIsQuery_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_disjoint_timer_query",
                feature = "GL_EXT_occlusion_query_boolean"
            )))
        )]
        {
            glIsQueryEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glIsRenderbuffer_load_with_dyn(&mut get_proc_address) as usize;
        glIsSampler_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        {
            glIsSemaphoreEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glIsShader_load_with_dyn(&mut get_proc_address) as usize;
        glIsSync_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        {
            glIsSyncAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        glIsTexture_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glIsTextureHandleResidentNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glIsTransformFeedback_load_with_dyn(&mut get_proc_address) as usize;
        glIsVertexArray_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_vertex_array_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
        {
            glIsVertexArrayOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_debug_label"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
        {
            glLabelObjectEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glLineWidth_load_with_dyn(&mut get_proc_address) as usize;
        glLinkProgram_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glMakeImageHandleNonResidentNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glMakeImageHandleResidentNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glMakeTextureHandleNonResidentNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glMakeTextureHandleResidentNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_mapbuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
        {
            glMapBufferOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glMapBufferRange_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
        {
            glMapBufferRangeEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixFrustumEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixLoad3x2fNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixLoad3x3fNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixLoadIdentityEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixLoadTranspose3x3fNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixLoadTransposedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixLoadTransposefEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixLoaddEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixLoadfEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixMult3x2fNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixMult3x3fNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixMultTranspose3x3fNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixMultTransposedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixMultTransposefEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixMultdEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixMultfEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixOrthoEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixPopEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixPushEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixRotatedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixRotatefEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixScaledEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixScalefEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixTranslatedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glMatrixTranslatefEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_KHR_parallel_shader_compile"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_parallel_shader_compile"))))]
        {
            glMaxShaderCompilerThreadsKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glMemoryBarrier_load_with_dyn(&mut get_proc_address) as usize;
        glMemoryBarrierByRegion_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glMemoryObjectParameterivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glMinSampleShading_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_sample_shading"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_sample_shading"))))]
        {
            glMinSampleShadingOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_multi_draw_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multi_draw_arrays"))))]
        {
            glMultiDrawArraysEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_multi_draw_indirect"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multi_draw_indirect"))))]
        {
            glMultiDrawArraysIndirectEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(
            feature = "GL_EXT_draw_elements_base_vertex",
            feature = "GL_OES_draw_elements_base_vertex"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_draw_elements_base_vertex",
                feature = "GL_OES_draw_elements_base_vertex"
            )))
        )]
        {
            glMultiDrawElementsBaseVertexEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_multi_draw_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multi_draw_arrays"))))]
        {
            glMultiDrawElementsEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_multi_draw_indirect"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multi_draw_indirect"))))]
        {
            glMultiDrawElementsIndirectEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        {
            glMultiDrawMeshTasksIndirectCountNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_mesh_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
        {
            glMultiDrawMeshTasksIndirectNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        {
            glNamedBufferAttachMemoryNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
        {
            glNamedBufferPageCommitmentMemNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_external_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
        {
            glNamedBufferStorageExternalEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glNamedBufferStorageMemEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_sample_locations"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
        {
            glNamedFramebufferSampleLocationsfvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_framebuffer_multisample_advanced"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_AMD_framebuffer_multisample_advanced")))
        )]
        {
            glNamedRenderbufferStorageMultisampleAdvancedAMD_load_with_dyn(&mut get_proc_address)
                as usize;
        }
        glObjectLabel_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glObjectLabelKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glObjectPtrLabel_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glObjectPtrLabelKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        glPatchParameteri_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_tessellation_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_tessellation_shader"))))]
        {
            glPatchParameteriEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_tessellation_shader"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_tessellation_shader"))))]
        {
            glPatchParameteriOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathCommandsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathCoordsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathCoverDepthFuncNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathDashArrayNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathGlyphIndexArrayNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathGlyphIndexRangeNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathGlyphRangeNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathGlyphsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathMemoryGlyphIndexArrayNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathParameterfNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathParameterfvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathParameteriNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathParameterivNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathStencilDepthOffsetNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathStencilFuncNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathStringNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathSubCommandsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPathSubCoordsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glPauseTransformFeedback_load_with_dyn(&mut get_proc_address) as usize;
        glPixelStorei_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glPointAlongPathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_polygon_mode"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_polygon_mode"))))]
        {
            glPolygonModeNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glPolygonOffset_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_polygon_offset_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_polygon_offset_clamp"))))]
        {
            glPolygonOffsetClampEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glPopDebugGroup_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glPopDebugGroupKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_debug_marker"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_marker"))))]
        {
            glPopGroupMarkerEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glPrimitiveBoundingBox_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_primitive_bounding_box"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_primitive_bounding_box"))))]
        {
            glPrimitiveBoundingBoxEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_primitive_bounding_box"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_primitive_bounding_box"))))]
        {
            glPrimitiveBoundingBoxOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramBinary_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_get_program_binary"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
        {
            glProgramBinaryOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramParameteri_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramParameteriEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glProgramPathFragmentInputGenNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform1f_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform1fEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform1fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform1fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform1i_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform1i64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform1i64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform1iEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform1iv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform1ivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform1ui_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform1ui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform1ui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform1uiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform1uiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform1uivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform2f_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform2fEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform2fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform2fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform2i_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform2i64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform2i64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform2iEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform2iv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform2ivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform2ui_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform2ui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform2ui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform2uiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform2uiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform2uivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform3f_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform3fEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform3fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform3fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform3i_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform3i64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform3i64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform3iEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform3iv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform3ivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform3ui_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform3ui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform3ui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform3uiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform3uiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform3uivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform4f_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform4fEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform4fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform4fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform4i_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform4i64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform4i64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform4iEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform4iv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform4ivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform4ui_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform4ui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glProgramUniform4ui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform4uiEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniform4uiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniform4uivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        {
            glProgramUniformHandleui64IMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glProgramUniformHandleui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        {
            glProgramUniformHandleui64vIMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glProgramUniformHandleui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniformMatrix2fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniformMatrix2fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniformMatrix2x3fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniformMatrix2x3fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniformMatrix2x4fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniformMatrix2x4fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniformMatrix3fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniformMatrix3fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniformMatrix3x2fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniformMatrix3x2fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniformMatrix3x4fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniformMatrix3x4fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniformMatrix4fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniformMatrix4fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniformMatrix4x2fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniformMatrix4x2fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glProgramUniformMatrix4x3fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glProgramUniformMatrix4x3fvEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glPushDebugGroup_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_KHR_debug"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
        {
            glPushDebugGroupKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_debug_marker"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_marker"))))]
        {
            glPushGroupMarkerEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
        {
            glQueryCounterEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(
            feature = "GL_EXT_raster_multisample",
            feature = "GL_NV_framebuffer_mixed_samples"
        ))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(
                feature = "GL_EXT_raster_multisample",
                feature = "GL_NV_framebuffer_mixed_samples"
            )))
        )]
        {
            glRasterSamplesEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glReadBuffer_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
        {
            glReadBufferIndexedEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_read_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_read_buffer"))))]
        {
            glReadBufferNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glReadPixels_load_with_dyn(&mut get_proc_address) as usize;
        glReadnPixels_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
        {
            glReadnPixelsEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_KHR_robustness"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
        {
            glReadnPixelsKHR_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
        {
            glReleaseKeyedMutexWin32EXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glReleaseShaderCompiler_load_with_dyn(&mut get_proc_address) as usize;
        glRenderbufferStorage_load_with_dyn(&mut get_proc_address) as usize;
        glRenderbufferStorageMultisample_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_ANGLE_framebuffer_multisample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_multisample"))))]
        {
            glRenderbufferStorageMultisampleANGLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_APPLE_framebuffer_multisample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_framebuffer_multisample"))))]
        {
            glRenderbufferStorageMultisampleAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_framebuffer_multisample_advanced"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_AMD_framebuffer_multisample_advanced")))
        )]
        {
            glRenderbufferStorageMultisampleAdvancedAMD_load_with_dyn(&mut get_proc_address)
                as usize;
        }
        #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
        )]
        {
            glRenderbufferStorageMultisampleEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
        )]
        {
            glRenderbufferStorageMultisampleIMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_framebuffer_multisample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_multisample"))))]
        {
            glRenderbufferStorageMultisampleNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        {
            glResetMemoryObjectParameterNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_sample_locations"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
        {
            glResolveDepthValuesNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_APPLE_framebuffer_multisample"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_framebuffer_multisample"))))]
        {
            glResolveMultisampleFramebufferAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        glResumeTransformFeedback_load_with_dyn(&mut get_proc_address) as usize;
        glSampleCoverage_load_with_dyn(&mut get_proc_address) as usize;
        glSampleMaski_load_with_dyn(&mut get_proc_address) as usize;
        glSamplerParameterIiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        {
            glSamplerParameterIivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        {
            glSamplerParameterIivOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glSamplerParameterIuiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        {
            glSamplerParameterIuivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        {
            glSamplerParameterIuivOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glSamplerParameterf_load_with_dyn(&mut get_proc_address) as usize;
        glSamplerParameterfv_load_with_dyn(&mut get_proc_address) as usize;
        glSamplerParameteri_load_with_dyn(&mut get_proc_address) as usize;
        glSamplerParameteriv_load_with_dyn(&mut get_proc_address) as usize;
        glScissor_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glScissorArrayvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        {
            glScissorArrayvOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_scissor_exclusive"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_scissor_exclusive"))))]
        {
            glScissorExclusiveArrayvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_scissor_exclusive"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_scissor_exclusive"))))]
        {
            glScissorExclusiveNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glScissorIndexedNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        {
            glScissorIndexedOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glScissorIndexedvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        {
            glScissorIndexedvOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_AMD_performance_monitor"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
        {
            glSelectPerfMonitorCountersAMD_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
        {
            glSemaphoreParameterivNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        {
            glSemaphoreParameterui64vEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        {
            glSetFenceNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glShaderBinary_load_with_dyn(&mut get_proc_address) as usize;
        glShaderSource_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        {
            glShadingRateImageBarrierNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        {
            glShadingRateImagePaletteNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_shading_rate"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_shading_rate"))))]
        {
            glShadingRateQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        {
            glShadingRateSampleOrderCustomNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_shading_rate_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
        {
            glShadingRateSampleOrderNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        {
            glSignalSemaphoreEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        {
            glSignalVkFenceNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        {
            glSignalVkSemaphoreNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
        {
            glStartTilingQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glStencilFillPathInstancedNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glStencilFillPathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glStencilFunc_load_with_dyn(&mut get_proc_address) as usize;
        glStencilFuncSeparate_load_with_dyn(&mut get_proc_address) as usize;
        glStencilMask_load_with_dyn(&mut get_proc_address) as usize;
        glStencilMaskSeparate_load_with_dyn(&mut get_proc_address) as usize;
        glStencilOp_load_with_dyn(&mut get_proc_address) as usize;
        glStencilOpSeparate_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glStencilStrokePathInstancedNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glStencilStrokePathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glStencilThenCoverFillPathInstancedNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glStencilThenCoverFillPathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glStencilThenCoverStrokePathInstancedNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glStencilThenCoverStrokePathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_conservative_raster"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conservative_raster"))))]
        {
            glSubpixelPrecisionBiasNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_fence"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
        {
            glTestFenceNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        {
            glTexAttachMemoryNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexBuffer_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_buffer"))))]
        {
            glTexBufferEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_buffer"))))]
        {
            glTexBufferOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexBufferRange_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_buffer"))))]
        {
            glTexBufferRangeEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_buffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_buffer"))))]
        {
            glTexBufferRangeOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_motion_estimation"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_motion_estimation"))))]
        {
            glTexEstimateMotionQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_motion_estimation"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_motion_estimation"))))]
        {
            glTexEstimateMotionRegionsQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexImage2D_load_with_dyn(&mut get_proc_address) as usize;
        glTexImage3D_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        {
            glTexImage3DOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_sparse_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_sparse_texture"))))]
        {
            glTexPageCommitmentEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
        {
            glTexPageCommitmentMemNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexParameterIiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        {
            glTexParameterIivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        {
            glTexParameterIivOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexParameterIuiv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
        {
            glTexParameterIuivEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
        {
            glTexParameterIuivOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexParameterf_load_with_dyn(&mut get_proc_address) as usize;
        glTexParameterfv_load_with_dyn(&mut get_proc_address) as usize;
        glTexParameteri_load_with_dyn(&mut get_proc_address) as usize;
        glTexParameteriv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
        {
            glTexStorage1DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexStorage2D_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
        {
            glTexStorage2DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexStorage2DMultisample_load_with_dyn(&mut get_proc_address) as usize;
        glTexStorage3D_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_texture_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
        {
            glTexStorage3DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexStorage3DMultisample_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_texture_storage_multisample_2d_array"))]
        #[cfg_attr(
            docs_rs,
            doc(cfg(any(feature = "GL_OES_texture_storage_multisample_2d_array")))
        )]
        {
            glTexStorage3DMultisampleOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glTexStorageMem2DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glTexStorageMem2DMultisampleEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glTexStorageMem3DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glTexStorageMem3DMultisampleEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTexSubImage2D_load_with_dyn(&mut get_proc_address) as usize;
        glTexSubImage3D_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_texture_3D"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
        {
            glTexSubImage3DOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_memory_attachment"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
        {
            glTextureAttachMemoryNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_QCOM_texture_foveated"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_texture_foveated"))))]
        {
            glTextureFoveationParametersQCOM_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
        {
            glTexturePageCommitmentMemNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_texture_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
        {
            glTextureStorage1DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_texture_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
        {
            glTextureStorage2DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_texture_storage"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_storage"))))]
        {
            glTextureStorage3DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glTextureStorageMem2DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glTextureStorageMem2DMultisampleEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glTextureStorageMem3DEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_memory_object"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
        {
            glTextureStorageMem3DMultisampleEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_texture_view"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_view"))))]
        {
            glTextureViewEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_texture_view"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_view"))))]
        {
            glTextureViewOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glTransformFeedbackVaryings_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glTransformPathNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniform1f_load_with_dyn(&mut get_proc_address) as usize;
        glUniform1fv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform1i_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform1i64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform1i64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniform1iv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform1ui_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform1ui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform1ui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniform1uiv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform2f_load_with_dyn(&mut get_proc_address) as usize;
        glUniform2fv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform2i_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform2i64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform2i64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniform2iv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform2ui_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform2ui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform2ui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniform2uiv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform3f_load_with_dyn(&mut get_proc_address) as usize;
        glUniform3fv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform3i_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform3i64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform3i64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniform3iv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform3ui_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform3ui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform3ui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniform3uiv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform4f_load_with_dyn(&mut get_proc_address) as usize;
        glUniform4fv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform4i_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform4i64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform4i64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniform4iv_load_with_dyn(&mut get_proc_address) as usize;
        glUniform4ui_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform4ui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_gpu_shader5"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_gpu_shader5"))))]
        {
            glUniform4ui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniform4uiv_load_with_dyn(&mut get_proc_address) as usize;
        glUniformBlockBinding_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        {
            glUniformHandleui64IMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glUniformHandleui64NV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_IMG_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
        {
            glUniformHandleui64vIMG_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_bindless_texture"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
        {
            glUniformHandleui64vNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniformMatrix2fv_load_with_dyn(&mut get_proc_address) as usize;
        glUniformMatrix2x3fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_non_square_matrices"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
        {
            glUniformMatrix2x3fvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniformMatrix2x4fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_non_square_matrices"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
        {
            glUniformMatrix2x4fvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniformMatrix3fv_load_with_dyn(&mut get_proc_address) as usize;
        glUniformMatrix3x2fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_non_square_matrices"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
        {
            glUniformMatrix3x2fvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniformMatrix3x4fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_non_square_matrices"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
        {
            glUniformMatrix3x4fvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniformMatrix4fv_load_with_dyn(&mut get_proc_address) as usize;
        glUniformMatrix4x2fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_non_square_matrices"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
        {
            glUniformMatrix4x2fvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUniformMatrix4x3fv_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_non_square_matrices"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_non_square_matrices"))))]
        {
            glUniformMatrix4x3fvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUnmapBuffer_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_OES_mapbuffer"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
        {
            glUnmapBufferOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        glUseProgram_load_with_dyn(&mut get_proc_address) as usize;
        glUseProgramStages_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glUseProgramStagesEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glValidateProgram_load_with_dyn(&mut get_proc_address) as usize;
        glValidateProgramPipeline_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
        {
            glValidateProgramPipelineEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glVertexAttrib1f_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttrib1fv_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttrib2f_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttrib2fv_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttrib3f_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttrib3fv_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttrib4f_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttrib4fv_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttribBinding_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttribDivisor_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
        {
            glVertexAttribDivisorANGLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_instanced_arrays"))))]
        {
            glVertexAttribDivisorEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_instanced_arrays"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_instanced_arrays"))))]
        {
            glVertexAttribDivisorNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        glVertexAttribFormat_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttribI4i_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttribI4iv_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttribI4ui_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttribI4uiv_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttribIFormat_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttribIPointer_load_with_dyn(&mut get_proc_address) as usize;
        glVertexAttribPointer_load_with_dyn(&mut get_proc_address) as usize;
        glVertexBindingDivisor_load_with_dyn(&mut get_proc_address) as usize;
        glViewport_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glViewportArrayvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        {
            glViewportArrayvOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glViewportIndexedfNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        {
            glViewportIndexedfOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
        {
            glViewportIndexedfvNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_OES_viewport_array"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
        {
            glViewportIndexedfvOES_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_clip_space_w_scaling"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_clip_space_w_scaling"))))]
        {
            glViewportPositionWScaleNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_viewport_swizzle"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_swizzle"))))]
        {
            glViewportSwizzleNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_semaphore"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
        {
            glWaitSemaphoreEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
        glWaitSync_load_with_dyn(&mut get_proc_address) as usize;
        #[cfg(any(feature = "GL_APPLE_sync"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
        {
            glWaitSyncAPPLE_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
        {
            glWaitVkSemaphoreNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_NV_path_rendering"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
        {
            glWeightPathsNV_load_with_dyn(&mut get_proc_address) as usize;
        }
        #[cfg(any(feature = "GL_EXT_window_rectangles"))]
        #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_window_rectangles"))))]
        {
            glWindowRectanglesEXT_load_with_dyn(&mut get_proc_address) as usize;
        }
    }

    /// [glAcquireKeyedMutexWin32EXT](http://docs.gl/es3/glAcquireKeyedMutexWin32EXT)(memory, key, timeout)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
    pub unsafe fn glAcquireKeyedMutexWin32EXT(
        memory: GLuint,
        key: GLuint64,
        timeout: GLuint,
    ) -> GLboolean {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glAcquireKeyedMutexWin32EXT({:?}, {:?}, {:?});",
                memory,
                key,
                timeout
            );
        }
        let out = call_atomic_ptr_3arg(
            "glAcquireKeyedMutexWin32EXT",
            &glAcquireKeyedMutexWin32EXT_p,
            memory,
            key,
            timeout,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glAcquireKeyedMutexWin32EXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
    static glAcquireKeyedMutexWin32EXT_p: APcv = ap_null();
    /// Tries to load [`glAcquireKeyedMutexWin32EXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
    pub unsafe fn glAcquireKeyedMutexWin32EXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glAcquireKeyedMutexWin32EXT\0",
            &glAcquireKeyedMutexWin32EXT_p,
        )
    }
    /// Checks if the pointer for [`glAcquireKeyedMutexWin32EXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_win32_keyed_mutex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_win32_keyed_mutex"))))]
    pub fn glAcquireKeyedMutexWin32EXT_is_loaded() -> bool {
        !glAcquireKeyedMutexWin32EXT_p.load(RELAX).is_null()
    }

    /// [glActiveShaderProgram](http://docs.gl/es3/glActiveShaderProgram)(pipeline, program)
    /// * `pipeline` class: program pipeline
    /// * `program` class: program
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glActiveShaderProgram(pipeline: GLuint, program: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glActiveShaderProgram({:?}, {:?});",
                pipeline,
                program
            );
        }
        let out = call_atomic_ptr_2arg(
            "glActiveShaderProgram",
            &glActiveShaderProgram_p,
            pipeline,
            program,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glActiveShaderProgram");
        }
        out
    }
    static glActiveShaderProgram_p: APcv = ap_null();
    /// Tries to load [`glActiveShaderProgram`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glActiveShaderProgram_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glActiveShaderProgram\0",
            &glActiveShaderProgram_p,
        )
    }
    /// Checks if the pointer for [`glActiveShaderProgram`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glActiveShaderProgram_is_loaded() -> bool {
        !glActiveShaderProgram_p.load(RELAX).is_null()
    }

    /// [glActiveShaderProgramEXT](http://docs.gl/es3/glActiveShaderProgramEXT)(pipeline, program)
    /// * `pipeline` class: program pipeline
    /// * `program` class: program
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glActiveShaderProgramEXT(pipeline: GLuint, program: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glActiveShaderProgramEXT({:?}, {:?});",
                pipeline,
                program
            );
        }
        let out = call_atomic_ptr_2arg(
            "glActiveShaderProgramEXT",
            &glActiveShaderProgramEXT_p,
            pipeline,
            program,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glActiveShaderProgramEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    static glActiveShaderProgramEXT_p: APcv = ap_null();
    /// Tries to load [`glActiveShaderProgramEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glActiveShaderProgramEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glActiveShaderProgramEXT\0",
            &glActiveShaderProgramEXT_p,
        )
    }
    /// Checks if the pointer for [`glActiveShaderProgramEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub fn glActiveShaderProgramEXT_is_loaded() -> bool {
        !glActiveShaderProgramEXT_p.load(RELAX).is_null()
    }

    /// [glActiveTexture](http://docs.gl/es3/glActiveTexture)(texture)
    /// * `texture` group: TextureUnit
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glActiveTexture(texture: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glActiveTexture({:#X});", texture);
        }
        let out = call_atomic_ptr_1arg("glActiveTexture", &glActiveTexture_p, texture);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glActiveTexture");
        }
        out
    }
    static glActiveTexture_p: APcv = ap_null();
    /// Tries to load [`glActiveTexture`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glActiveTexture_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glActiveTexture\0", &glActiveTexture_p)
    }
    /// Checks if the pointer for [`glActiveTexture`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glActiveTexture_is_loaded() -> bool {
        !glActiveTexture_p.load(RELAX).is_null()
    }

    /// [glAlphaFuncQCOM](http://docs.gl/es3/glAlphaFuncQCOM)(func, ref_)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_alpha_test"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_alpha_test"))))]
    pub unsafe fn glAlphaFuncQCOM(func: GLenum, ref_: GLclampf) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glAlphaFuncQCOM({:#X}, {:?});", func, ref_);
        }
        let out = call_atomic_ptr_2arg("glAlphaFuncQCOM", &glAlphaFuncQCOM_p, func, ref_);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glAlphaFuncQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_alpha_test"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_alpha_test"))))]
    static glAlphaFuncQCOM_p: APcv = ap_null();
    /// Tries to load [`glAlphaFuncQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_alpha_test"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_alpha_test"))))]
    pub unsafe fn glAlphaFuncQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glAlphaFuncQCOM\0", &glAlphaFuncQCOM_p)
    }
    /// Checks if the pointer for [`glAlphaFuncQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_alpha_test"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_alpha_test"))))]
    pub fn glAlphaFuncQCOM_is_loaded() -> bool {
        !glAlphaFuncQCOM_p.load(RELAX).is_null()
    }

    /// [glApplyFramebufferAttachmentCMAAINTEL](http://docs.gl/es3/glApplyFramebufferAttachmentCMAAINTEL)()
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))))]
    pub unsafe fn glApplyFramebufferAttachmentCMAAINTEL() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glApplyFramebufferAttachmentCMAAINTEL();",);
        }
        let out = call_atomic_ptr_0arg(
            "glApplyFramebufferAttachmentCMAAINTEL",
            &glApplyFramebufferAttachmentCMAAINTEL_p,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glApplyFramebufferAttachmentCMAAINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))))]
    static glApplyFramebufferAttachmentCMAAINTEL_p: APcv = ap_null();
    /// Tries to load [`glApplyFramebufferAttachmentCMAAINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))))]
    pub unsafe fn glApplyFramebufferAttachmentCMAAINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glApplyFramebufferAttachmentCMAAINTEL\0",
            &glApplyFramebufferAttachmentCMAAINTEL_p,
        )
    }
    /// Checks if the pointer for [`glApplyFramebufferAttachmentCMAAINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_framebuffer_CMAA"))))]
    pub fn glApplyFramebufferAttachmentCMAAINTEL_is_loaded() -> bool {
        !glApplyFramebufferAttachmentCMAAINTEL_p
            .load(RELAX)
            .is_null()
    }

    /// [glAttachShader](http://docs.gl/es3/glAttachShader)(program, shader)
    /// * `program` class: program
    /// * `shader` class: shader
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glAttachShader(program: GLuint, shader: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glAttachShader({:?}, {:?});", program, shader);
        }
        let out = call_atomic_ptr_2arg("glAttachShader", &glAttachShader_p, program, shader);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glAttachShader");
        }
        out
    }
    static glAttachShader_p: APcv = ap_null();
    /// Tries to load [`glAttachShader`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glAttachShader_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glAttachShader\0", &glAttachShader_p)
    }
    /// Checks if the pointer for [`glAttachShader`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glAttachShader_is_loaded() -> bool {
        !glAttachShader_p.load(RELAX).is_null()
    }

    /// [glBeginConditionalRenderNV](http://docs.gl/es3/glBeginConditionalRenderNV)(id, mode)
    /// * `mode` group: ConditionalRenderMode
    /// * alias of: [`glBeginConditionalRender`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_conditional_render"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
    pub unsafe fn glBeginConditionalRenderNV(id: GLuint, mode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBeginConditionalRenderNV({:?}, {:#X});", id, mode);
        }
        let out = call_atomic_ptr_2arg(
            "glBeginConditionalRenderNV",
            &glBeginConditionalRenderNV_p,
            id,
            mode,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBeginConditionalRenderNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_conditional_render"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
    static glBeginConditionalRenderNV_p: APcv = ap_null();
    /// Tries to load [`glBeginConditionalRenderNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_conditional_render"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
    pub unsafe fn glBeginConditionalRenderNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBeginConditionalRenderNV\0",
            &glBeginConditionalRenderNV_p,
        )
    }
    /// Checks if the pointer for [`glBeginConditionalRenderNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_conditional_render"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
    pub fn glBeginConditionalRenderNV_is_loaded() -> bool {
        !glBeginConditionalRenderNV_p.load(RELAX).is_null()
    }

    /// [glBeginPerfMonitorAMD](http://docs.gl/es3/glBeginPerfMonitorAMD)(monitor)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glBeginPerfMonitorAMD(monitor: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBeginPerfMonitorAMD({:?});", monitor);
        }
        let out = call_atomic_ptr_1arg("glBeginPerfMonitorAMD", &glBeginPerfMonitorAMD_p, monitor);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBeginPerfMonitorAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glBeginPerfMonitorAMD_p: APcv = ap_null();
    /// Tries to load [`glBeginPerfMonitorAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glBeginPerfMonitorAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBeginPerfMonitorAMD\0",
            &glBeginPerfMonitorAMD_p,
        )
    }
    /// Checks if the pointer for [`glBeginPerfMonitorAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glBeginPerfMonitorAMD_is_loaded() -> bool {
        !glBeginPerfMonitorAMD_p.load(RELAX).is_null()
    }

    /// [glBeginPerfQueryINTEL](http://docs.gl/es3/glBeginPerfQueryINTEL)(queryHandle)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glBeginPerfQueryINTEL(queryHandle: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBeginPerfQueryINTEL({:?});", queryHandle);
        }
        let out = call_atomic_ptr_1arg(
            "glBeginPerfQueryINTEL",
            &glBeginPerfQueryINTEL_p,
            queryHandle,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBeginPerfQueryINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glBeginPerfQueryINTEL_p: APcv = ap_null();
    /// Tries to load [`glBeginPerfQueryINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glBeginPerfQueryINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBeginPerfQueryINTEL\0",
            &glBeginPerfQueryINTEL_p,
        )
    }
    /// Checks if the pointer for [`glBeginPerfQueryINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glBeginPerfQueryINTEL_is_loaded() -> bool {
        !glBeginPerfQueryINTEL_p.load(RELAX).is_null()
    }

    /// [glBeginQuery](http://docs.gl/es3/glBeginQuery)(target, id)
    /// * `target` group: QueryTarget
    /// * `id` class: query
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBeginQuery(target: GLenum, id: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBeginQuery({:#X}, {:?});", target, id);
        }
        let out = call_atomic_ptr_2arg("glBeginQuery", &glBeginQuery_p, target, id);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBeginQuery");
        }
        out
    }
    static glBeginQuery_p: APcv = ap_null();
    /// Tries to load [`glBeginQuery`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBeginQuery_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBeginQuery\0", &glBeginQuery_p)
    }
    /// Checks if the pointer for [`glBeginQuery`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBeginQuery_is_loaded() -> bool {
        !glBeginQuery_p.load(RELAX).is_null()
    }

    /// [glBeginQueryEXT](http://docs.gl/es3/glBeginQueryEXT)(target, id)
    /// * `target` group: QueryTarget
    /// * `id` class: query
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glBeginQueryEXT(target: GLenum, id: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBeginQueryEXT({:#X}, {:?});", target, id);
        }
        let out = call_atomic_ptr_2arg("glBeginQueryEXT", &glBeginQueryEXT_p, target, id);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBeginQueryEXT");
        }
        out
    }
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    static glBeginQueryEXT_p: APcv = ap_null();
    /// Tries to load [`glBeginQueryEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glBeginQueryEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBeginQueryEXT\0", &glBeginQueryEXT_p)
    }
    /// Checks if the pointer for [`glBeginQueryEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub fn glBeginQueryEXT_is_loaded() -> bool {
        !glBeginQueryEXT_p.load(RELAX).is_null()
    }

    /// [glBeginTransformFeedback](http://docs.gl/es3/glBeginTransformFeedback)(primitiveMode)
    /// * `primitiveMode` group: PrimitiveType
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBeginTransformFeedback(primitiveMode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBeginTransformFeedback({:#X});", primitiveMode);
        }
        let out = call_atomic_ptr_1arg(
            "glBeginTransformFeedback",
            &glBeginTransformFeedback_p,
            primitiveMode,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBeginTransformFeedback");
        }
        out
    }
    static glBeginTransformFeedback_p: APcv = ap_null();
    /// Tries to load [`glBeginTransformFeedback`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBeginTransformFeedback_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBeginTransformFeedback\0",
            &glBeginTransformFeedback_p,
        )
    }
    /// Checks if the pointer for [`glBeginTransformFeedback`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBeginTransformFeedback_is_loaded() -> bool {
        !glBeginTransformFeedback_p.load(RELAX).is_null()
    }

    /// [glBindAttribLocation](http://docs.gl/es3/glBindAttribLocation)(program, index, name)
    /// * `program` class: program
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindAttribLocation(program: GLuint, index: GLuint, name: *const GLchar) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBindAttribLocation({:?}, {:?}, {:p});",
                program,
                index,
                name
            );
        }
        let out = call_atomic_ptr_3arg(
            "glBindAttribLocation",
            &glBindAttribLocation_p,
            program,
            index,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindAttribLocation");
        }
        out
    }
    static glBindAttribLocation_p: APcv = ap_null();
    /// Tries to load [`glBindAttribLocation`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindAttribLocation_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindAttribLocation\0",
            &glBindAttribLocation_p,
        )
    }
    /// Checks if the pointer for [`glBindAttribLocation`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindAttribLocation_is_loaded() -> bool {
        !glBindAttribLocation_p.load(RELAX).is_null()
    }

    /// [glBindBuffer](http://docs.gl/es3/glBindBuffer)(target, buffer)
    /// * `target` group: BufferTargetARB
    /// * `buffer` class: buffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindBuffer(target: GLenum, buffer: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBindBuffer({:#X}, {:?});", target, buffer);
        }
        let out = call_atomic_ptr_2arg("glBindBuffer", &glBindBuffer_p, target, buffer);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindBuffer");
        }
        out
    }
    static glBindBuffer_p: APcv = ap_null();
    /// Tries to load [`glBindBuffer`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindBuffer_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBindBuffer\0", &glBindBuffer_p)
    }
    /// Checks if the pointer for [`glBindBuffer`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindBuffer_is_loaded() -> bool {
        !glBindBuffer_p.load(RELAX).is_null()
    }

    /// [glBindBufferBase](http://docs.gl/es3/glBindBufferBase)(target, index, buffer)
    /// * `target` group: BufferTargetARB
    /// * `buffer` class: buffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindBufferBase(target: GLenum, index: GLuint, buffer: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBindBufferBase({:#X}, {:?}, {:?});",
                target,
                index,
                buffer
            );
        }
        let out = call_atomic_ptr_3arg(
            "glBindBufferBase",
            &glBindBufferBase_p,
            target,
            index,
            buffer,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindBufferBase");
        }
        out
    }
    static glBindBufferBase_p: APcv = ap_null();
    /// Tries to load [`glBindBufferBase`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindBufferBase_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBindBufferBase\0", &glBindBufferBase_p)
    }
    /// Checks if the pointer for [`glBindBufferBase`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindBufferBase_is_loaded() -> bool {
        !glBindBufferBase_p.load(RELAX).is_null()
    }

    /// [glBindBufferRange](http://docs.gl/es3/glBindBufferRange)(target, index, buffer, offset, size)
    /// * `target` group: BufferTargetARB
    /// * `buffer` class: buffer
    /// * `offset` group: BufferOffset
    /// * `size` group: BufferSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindBufferRange(
        target: GLenum,
        index: GLuint,
        buffer: GLuint,
        offset: GLintptr,
        size: GLsizeiptr,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBindBufferRange({:#X}, {:?}, {:?}, {:?}, {:?});",
                target,
                index,
                buffer,
                offset,
                size
            );
        }
        let out = call_atomic_ptr_5arg(
            "glBindBufferRange",
            &glBindBufferRange_p,
            target,
            index,
            buffer,
            offset,
            size,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindBufferRange");
        }
        out
    }
    static glBindBufferRange_p: APcv = ap_null();
    /// Tries to load [`glBindBufferRange`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindBufferRange_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindBufferRange\0",
            &glBindBufferRange_p,
        )
    }
    /// Checks if the pointer for [`glBindBufferRange`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindBufferRange_is_loaded() -> bool {
        !glBindBufferRange_p.load(RELAX).is_null()
    }

    /// [glBindFragDataLocationEXT](http://docs.gl/es3/glBindFragDataLocationEXT)(program, color, name)
    /// * `program` class: program
    /// * `name` len: COMPSIZE(name)
    /// * alias of: [`glBindFragDataLocation`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub unsafe fn glBindFragDataLocationEXT(program: GLuint, color: GLuint, name: *const GLchar) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBindFragDataLocationEXT({:?}, {:?}, {:p});",
                program,
                color,
                name
            );
        }
        let out = call_atomic_ptr_3arg(
            "glBindFragDataLocationEXT",
            &glBindFragDataLocationEXT_p,
            program,
            color,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindFragDataLocationEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    static glBindFragDataLocationEXT_p: APcv = ap_null();
    /// Tries to load [`glBindFragDataLocationEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub unsafe fn glBindFragDataLocationEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindFragDataLocationEXT\0",
            &glBindFragDataLocationEXT_p,
        )
    }
    /// Checks if the pointer for [`glBindFragDataLocationEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub fn glBindFragDataLocationEXT_is_loaded() -> bool {
        !glBindFragDataLocationEXT_p.load(RELAX).is_null()
    }

    /// [glBindFragDataLocationIndexedEXT](http://docs.gl/es3/glBindFragDataLocationIndexedEXT)(program, colorNumber, index, name)
    /// * `program` class: program
    /// * alias of: [`glBindFragDataLocationIndexed`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub unsafe fn glBindFragDataLocationIndexedEXT(
        program: GLuint,
        colorNumber: GLuint,
        index: GLuint,
        name: *const GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBindFragDataLocationIndexedEXT({:?}, {:?}, {:?}, {:p});",
                program,
                colorNumber,
                index,
                name
            );
        }
        let out = call_atomic_ptr_4arg(
            "glBindFragDataLocationIndexedEXT",
            &glBindFragDataLocationIndexedEXT_p,
            program,
            colorNumber,
            index,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindFragDataLocationIndexedEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    static glBindFragDataLocationIndexedEXT_p: APcv = ap_null();
    /// Tries to load [`glBindFragDataLocationIndexedEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub unsafe fn glBindFragDataLocationIndexedEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindFragDataLocationIndexedEXT\0",
            &glBindFragDataLocationIndexedEXT_p,
        )
    }
    /// Checks if the pointer for [`glBindFragDataLocationIndexedEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub fn glBindFragDataLocationIndexedEXT_is_loaded() -> bool {
        !glBindFragDataLocationIndexedEXT_p.load(RELAX).is_null()
    }

    /// [glBindFramebuffer](http://docs.gl/es3/glBindFramebuffer)(target, framebuffer)
    /// * `target` group: FramebufferTarget
    /// * `framebuffer` class: framebuffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindFramebuffer(target: GLenum, framebuffer: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBindFramebuffer({:#X}, {:?});",
                target,
                framebuffer
            );
        }
        let out = call_atomic_ptr_2arg(
            "glBindFramebuffer",
            &glBindFramebuffer_p,
            target,
            framebuffer,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindFramebuffer");
        }
        out
    }
    static glBindFramebuffer_p: APcv = ap_null();
    /// Tries to load [`glBindFramebuffer`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindFramebuffer_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindFramebuffer\0",
            &glBindFramebuffer_p,
        )
    }
    /// Checks if the pointer for [`glBindFramebuffer`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindFramebuffer_is_loaded() -> bool {
        !glBindFramebuffer_p.load(RELAX).is_null()
    }

    /// [glBindImageTexture](http://docs.gl/es3/glBindImageTexture)(unit, texture, level, layered, layer, access, format)
    /// * `texture` class: texture
    /// * `access` group: BufferAccessARB
    /// * `format` group: InternalFormat
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindImageTexture(
        unit: GLuint,
        texture: GLuint,
        level: GLint,
        layered: GLboolean,
        layer: GLint,
        access: GLenum,
        format: GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBindImageTexture({:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:#X});",
                unit,
                texture,
                level,
                layered,
                layer,
                access,
                format
            );
        }
        let out = call_atomic_ptr_7arg(
            "glBindImageTexture",
            &glBindImageTexture_p,
            unit,
            texture,
            level,
            layered,
            layer,
            access,
            format,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindImageTexture");
        }
        out
    }
    static glBindImageTexture_p: APcv = ap_null();
    /// Tries to load [`glBindImageTexture`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindImageTexture_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindImageTexture\0",
            &glBindImageTexture_p,
        )
    }
    /// Checks if the pointer for [`glBindImageTexture`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindImageTexture_is_loaded() -> bool {
        !glBindImageTexture_p.load(RELAX).is_null()
    }

    /// [glBindProgramPipeline](http://docs.gl/es3/glBindProgramPipeline)(pipeline)
    /// * `pipeline` class: program pipeline
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindProgramPipeline(pipeline: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBindProgramPipeline({:?});", pipeline);
        }
        let out = call_atomic_ptr_1arg("glBindProgramPipeline", &glBindProgramPipeline_p, pipeline);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindProgramPipeline");
        }
        out
    }
    static glBindProgramPipeline_p: APcv = ap_null();
    /// Tries to load [`glBindProgramPipeline`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindProgramPipeline_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindProgramPipeline\0",
            &glBindProgramPipeline_p,
        )
    }
    /// Checks if the pointer for [`glBindProgramPipeline`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindProgramPipeline_is_loaded() -> bool {
        !glBindProgramPipeline_p.load(RELAX).is_null()
    }

    /// [glBindProgramPipelineEXT](http://docs.gl/es3/glBindProgramPipelineEXT)(pipeline)
    /// * `pipeline` class: program pipeline
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glBindProgramPipelineEXT(pipeline: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBindProgramPipelineEXT({:?});", pipeline);
        }
        let out = call_atomic_ptr_1arg(
            "glBindProgramPipelineEXT",
            &glBindProgramPipelineEXT_p,
            pipeline,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindProgramPipelineEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    static glBindProgramPipelineEXT_p: APcv = ap_null();
    /// Tries to load [`glBindProgramPipelineEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glBindProgramPipelineEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindProgramPipelineEXT\0",
            &glBindProgramPipelineEXT_p,
        )
    }
    /// Checks if the pointer for [`glBindProgramPipelineEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub fn glBindProgramPipelineEXT_is_loaded() -> bool {
        !glBindProgramPipelineEXT_p.load(RELAX).is_null()
    }

    /// [glBindRenderbuffer](http://docs.gl/es3/glBindRenderbuffer)(target, renderbuffer)
    /// * `target` group: RenderbufferTarget
    /// * `renderbuffer` class: renderbuffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindRenderbuffer(target: GLenum, renderbuffer: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBindRenderbuffer({:#X}, {:?});",
                target,
                renderbuffer
            );
        }
        let out = call_atomic_ptr_2arg(
            "glBindRenderbuffer",
            &glBindRenderbuffer_p,
            target,
            renderbuffer,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindRenderbuffer");
        }
        out
    }
    static glBindRenderbuffer_p: APcv = ap_null();
    /// Tries to load [`glBindRenderbuffer`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindRenderbuffer_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindRenderbuffer\0",
            &glBindRenderbuffer_p,
        )
    }
    /// Checks if the pointer for [`glBindRenderbuffer`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindRenderbuffer_is_loaded() -> bool {
        !glBindRenderbuffer_p.load(RELAX).is_null()
    }

    /// [glBindSampler](http://docs.gl/es3/glBindSampler)(unit, sampler)
    /// * `sampler` class: sampler
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindSampler(unit: GLuint, sampler: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBindSampler({:?}, {:?});", unit, sampler);
        }
        let out = call_atomic_ptr_2arg("glBindSampler", &glBindSampler_p, unit, sampler);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindSampler");
        }
        out
    }
    static glBindSampler_p: APcv = ap_null();
    /// Tries to load [`glBindSampler`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindSampler_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBindSampler\0", &glBindSampler_p)
    }
    /// Checks if the pointer for [`glBindSampler`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindSampler_is_loaded() -> bool {
        !glBindSampler_p.load(RELAX).is_null()
    }

    /// [glBindShadingRateImageNV](http://docs.gl/es3/glBindShadingRateImageNV)(texture)
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    pub unsafe fn glBindShadingRateImageNV(texture: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBindShadingRateImageNV({:?});", texture);
        }
        let out = call_atomic_ptr_1arg(
            "glBindShadingRateImageNV",
            &glBindShadingRateImageNV_p,
            texture,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindShadingRateImageNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    static glBindShadingRateImageNV_p: APcv = ap_null();
    /// Tries to load [`glBindShadingRateImageNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    pub unsafe fn glBindShadingRateImageNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindShadingRateImageNV\0",
            &glBindShadingRateImageNV_p,
        )
    }
    /// Checks if the pointer for [`glBindShadingRateImageNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    pub fn glBindShadingRateImageNV_is_loaded() -> bool {
        !glBindShadingRateImageNV_p.load(RELAX).is_null()
    }

    /// [glBindTexture](http://docs.gl/es3/glBindTexture)(target, texture)
    /// * `target` group: TextureTarget
    /// * `texture` group: Texture
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindTexture(target: GLenum, texture: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBindTexture({:#X}, {:?});", target, texture);
        }
        let out = call_atomic_ptr_2arg("glBindTexture", &glBindTexture_p, target, texture);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindTexture");
        }
        out
    }
    static glBindTexture_p: APcv = ap_null();
    /// Tries to load [`glBindTexture`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindTexture_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBindTexture\0", &glBindTexture_p)
    }
    /// Checks if the pointer for [`glBindTexture`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindTexture_is_loaded() -> bool {
        !glBindTexture_p.load(RELAX).is_null()
    }

    /// [glBindTransformFeedback](http://docs.gl/es3/glBindTransformFeedback)(target, id)
    /// * `target` group: BindTransformFeedbackTarget
    /// * `id` class: transform feedback
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindTransformFeedback(target: GLenum, id: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBindTransformFeedback({:#X}, {:?});", target, id);
        }
        let out = call_atomic_ptr_2arg(
            "glBindTransformFeedback",
            &glBindTransformFeedback_p,
            target,
            id,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindTransformFeedback");
        }
        out
    }
    static glBindTransformFeedback_p: APcv = ap_null();
    /// Tries to load [`glBindTransformFeedback`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindTransformFeedback_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindTransformFeedback\0",
            &glBindTransformFeedback_p,
        )
    }
    /// Checks if the pointer for [`glBindTransformFeedback`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindTransformFeedback_is_loaded() -> bool {
        !glBindTransformFeedback_p.load(RELAX).is_null()
    }

    /// [glBindVertexArray](http://docs.gl/es3/glBindVertexArray)(array)
    /// * `array` class: vertex array
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindVertexArray(array: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBindVertexArray({:?});", array);
        }
        let out = call_atomic_ptr_1arg("glBindVertexArray", &glBindVertexArray_p, array);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindVertexArray");
        }
        out
    }
    static glBindVertexArray_p: APcv = ap_null();
    /// Tries to load [`glBindVertexArray`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindVertexArray_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindVertexArray\0",
            &glBindVertexArray_p,
        )
    }
    /// Checks if the pointer for [`glBindVertexArray`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindVertexArray_is_loaded() -> bool {
        !glBindVertexArray_p.load(RELAX).is_null()
    }

    /// [glBindVertexArrayOES](http://docs.gl/es3/glBindVertexArrayOES)(array)
    /// * `array` class: vertex array
    /// * alias of: [`glBindVertexArray`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    pub unsafe fn glBindVertexArrayOES(array: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBindVertexArrayOES({:?});", array);
        }
        let out = call_atomic_ptr_1arg("glBindVertexArrayOES", &glBindVertexArrayOES_p, array);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindVertexArrayOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    static glBindVertexArrayOES_p: APcv = ap_null();
    /// Tries to load [`glBindVertexArrayOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    pub unsafe fn glBindVertexArrayOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindVertexArrayOES\0",
            &glBindVertexArrayOES_p,
        )
    }
    /// Checks if the pointer for [`glBindVertexArrayOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    pub fn glBindVertexArrayOES_is_loaded() -> bool {
        !glBindVertexArrayOES_p.load(RELAX).is_null()
    }

    /// [glBindVertexBuffer](http://docs.gl/es3/glBindVertexBuffer)(bindingindex, buffer, offset, stride)
    /// * `buffer` class: buffer
    /// * `offset` group: BufferOffset
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBindVertexBuffer(
        bindingindex: GLuint,
        buffer: GLuint,
        offset: GLintptr,
        stride: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBindVertexBuffer({:?}, {:?}, {:?}, {:?});",
                bindingindex,
                buffer,
                offset,
                stride
            );
        }
        let out = call_atomic_ptr_4arg(
            "glBindVertexBuffer",
            &glBindVertexBuffer_p,
            bindingindex,
            buffer,
            offset,
            stride,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBindVertexBuffer");
        }
        out
    }
    static glBindVertexBuffer_p: APcv = ap_null();
    /// Tries to load [`glBindVertexBuffer`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBindVertexBuffer_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBindVertexBuffer\0",
            &glBindVertexBuffer_p,
        )
    }
    /// Checks if the pointer for [`glBindVertexBuffer`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBindVertexBuffer_is_loaded() -> bool {
        !glBindVertexBuffer_p.load(RELAX).is_null()
    }

    /// [glBlendBarrier](http://docs.gl/es3/glBlendBarrier)()
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendBarrier() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendBarrier();",);
        }
        let out = call_atomic_ptr_0arg("glBlendBarrier", &glBlendBarrier_p);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendBarrier");
        }
        out
    }
    static glBlendBarrier_p: APcv = ap_null();
    /// Tries to load [`glBlendBarrier`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendBarrier_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBlendBarrier\0", &glBlendBarrier_p)
    }
    /// Checks if the pointer for [`glBlendBarrier`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendBarrier_is_loaded() -> bool {
        !glBlendBarrier_p.load(RELAX).is_null()
    }

    /// [glBlendBarrierKHR](http://docs.gl/es3/glBlendBarrierKHR)()
    /// * alias of: [`glBlendBarrier`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_KHR_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_blend_equation_advanced"))))]
    pub unsafe fn glBlendBarrierKHR() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendBarrierKHR();",);
        }
        let out = call_atomic_ptr_0arg("glBlendBarrierKHR", &glBlendBarrierKHR_p);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendBarrierKHR");
        }
        out
    }
    #[cfg(any(feature = "GL_KHR_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_blend_equation_advanced"))))]
    static glBlendBarrierKHR_p: APcv = ap_null();
    /// Tries to load [`glBlendBarrierKHR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_blend_equation_advanced"))))]
    pub unsafe fn glBlendBarrierKHR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendBarrierKHR\0",
            &glBlendBarrierKHR_p,
        )
    }
    /// Checks if the pointer for [`glBlendBarrierKHR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_blend_equation_advanced"))))]
    pub fn glBlendBarrierKHR_is_loaded() -> bool {
        !glBlendBarrierKHR_p.load(RELAX).is_null()
    }

    /// [glBlendBarrierNV](http://docs.gl/es3/glBlendBarrierNV)()
    /// * alias of: [`glBlendBarrier`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
    pub unsafe fn glBlendBarrierNV() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendBarrierNV();",);
        }
        let out = call_atomic_ptr_0arg("glBlendBarrierNV", &glBlendBarrierNV_p);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendBarrierNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
    static glBlendBarrierNV_p: APcv = ap_null();
    /// Tries to load [`glBlendBarrierNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
    pub unsafe fn glBlendBarrierNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBlendBarrierNV\0", &glBlendBarrierNV_p)
    }
    /// Checks if the pointer for [`glBlendBarrierNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
    pub fn glBlendBarrierNV_is_loaded() -> bool {
        !glBlendBarrierNV_p.load(RELAX).is_null()
    }

    /// [glBlendColor](http://docs.gl/es3/glBlendColor)(red, green, blue, alpha)
    /// * `red` group: ColorF
    /// * `green` group: ColorF
    /// * `blue` group: ColorF
    /// * `alpha` group: ColorF
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendColor(red: GLfloat, green: GLfloat, blue: GLfloat, alpha: GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendColor({:?}, {:?}, {:?}, {:?});",
                red,
                green,
                blue,
                alpha
            );
        }
        let out = call_atomic_ptr_4arg("glBlendColor", &glBlendColor_p, red, green, blue, alpha);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendColor");
        }
        out
    }
    static glBlendColor_p: APcv = ap_null();
    /// Tries to load [`glBlendColor`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendColor_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBlendColor\0", &glBlendColor_p)
    }
    /// Checks if the pointer for [`glBlendColor`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendColor_is_loaded() -> bool {
        !glBlendColor_p.load(RELAX).is_null()
    }

    /// [glBlendEquation](http://docs.gl/es3/glBlendEquation)(mode)
    /// * `mode` group: BlendEquationModeEXT
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendEquation(mode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendEquation({:#X});", mode);
        }
        let out = call_atomic_ptr_1arg("glBlendEquation", &glBlendEquation_p, mode);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendEquation");
        }
        out
    }
    static glBlendEquation_p: APcv = ap_null();
    /// Tries to load [`glBlendEquation`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendEquation_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBlendEquation\0", &glBlendEquation_p)
    }
    /// Checks if the pointer for [`glBlendEquation`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendEquation_is_loaded() -> bool {
        !glBlendEquation_p.load(RELAX).is_null()
    }

    /// [glBlendEquationSeparate](http://docs.gl/es3/glBlendEquationSeparate)(modeRGB, modeAlpha)
    /// * `modeRGB` group: BlendEquationModeEXT
    /// * `modeAlpha` group: BlendEquationModeEXT
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendEquationSeparate(modeRGB: GLenum, modeAlpha: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendEquationSeparate({:#X}, {:#X});",
                modeRGB,
                modeAlpha
            );
        }
        let out = call_atomic_ptr_2arg(
            "glBlendEquationSeparate",
            &glBlendEquationSeparate_p,
            modeRGB,
            modeAlpha,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendEquationSeparate");
        }
        out
    }
    static glBlendEquationSeparate_p: APcv = ap_null();
    /// Tries to load [`glBlendEquationSeparate`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendEquationSeparate_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendEquationSeparate\0",
            &glBlendEquationSeparate_p,
        )
    }
    /// Checks if the pointer for [`glBlendEquationSeparate`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendEquationSeparate_is_loaded() -> bool {
        !glBlendEquationSeparate_p.load(RELAX).is_null()
    }

    /// [glBlendEquationSeparatei](http://docs.gl/es3/glBlendEquationSeparate)(buf, modeRGB, modeAlpha)
    /// * `modeRGB` group: BlendEquationModeEXT
    /// * `modeAlpha` group: BlendEquationModeEXT
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendEquationSeparatei(buf: GLuint, modeRGB: GLenum, modeAlpha: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendEquationSeparatei({:?}, {:#X}, {:#X});",
                buf,
                modeRGB,
                modeAlpha
            );
        }
        let out = call_atomic_ptr_3arg(
            "glBlendEquationSeparatei",
            &glBlendEquationSeparatei_p,
            buf,
            modeRGB,
            modeAlpha,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendEquationSeparatei");
        }
        out
    }
    static glBlendEquationSeparatei_p: APcv = ap_null();
    /// Tries to load [`glBlendEquationSeparatei`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendEquationSeparatei_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendEquationSeparatei\0",
            &glBlendEquationSeparatei_p,
        )
    }
    /// Checks if the pointer for [`glBlendEquationSeparatei`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendEquationSeparatei_is_loaded() -> bool {
        !glBlendEquationSeparatei_p.load(RELAX).is_null()
    }

    /// [glBlendEquationSeparateiEXT](http://docs.gl/es3/glBlendEquationSeparateiEXT)(buf, modeRGB, modeAlpha)
    /// * `modeRGB` group: BlendEquationModeEXT
    /// * `modeAlpha` group: BlendEquationModeEXT
    /// * alias of: [`glBlendEquationSeparatei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glBlendEquationSeparateiEXT(buf: GLuint, modeRGB: GLenum, modeAlpha: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendEquationSeparateiEXT({:?}, {:#X}, {:#X});",
                buf,
                modeRGB,
                modeAlpha
            );
        }
        let out = call_atomic_ptr_3arg(
            "glBlendEquationSeparateiEXT",
            &glBlendEquationSeparateiEXT_p,
            buf,
            modeRGB,
            modeAlpha,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendEquationSeparateiEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    static glBlendEquationSeparateiEXT_p: APcv = ap_null();
    /// Tries to load [`glBlendEquationSeparateiEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glBlendEquationSeparateiEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendEquationSeparateiEXT\0",
            &glBlendEquationSeparateiEXT_p,
        )
    }
    /// Checks if the pointer for [`glBlendEquationSeparateiEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub fn glBlendEquationSeparateiEXT_is_loaded() -> bool {
        !glBlendEquationSeparateiEXT_p.load(RELAX).is_null()
    }

    /// [glBlendEquationSeparateiOES](http://docs.gl/es3/glBlendEquationSeparateiOES)(buf, modeRGB, modeAlpha)
    /// * `modeRGB` group: BlendEquationModeEXT
    /// * `modeAlpha` group: BlendEquationModeEXT
    /// * alias of: [`glBlendEquationSeparatei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glBlendEquationSeparateiOES(buf: GLuint, modeRGB: GLenum, modeAlpha: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendEquationSeparateiOES({:?}, {:#X}, {:#X});",
                buf,
                modeRGB,
                modeAlpha
            );
        }
        let out = call_atomic_ptr_3arg(
            "glBlendEquationSeparateiOES",
            &glBlendEquationSeparateiOES_p,
            buf,
            modeRGB,
            modeAlpha,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendEquationSeparateiOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    static glBlendEquationSeparateiOES_p: APcv = ap_null();
    /// Tries to load [`glBlendEquationSeparateiOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glBlendEquationSeparateiOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendEquationSeparateiOES\0",
            &glBlendEquationSeparateiOES_p,
        )
    }
    /// Checks if the pointer for [`glBlendEquationSeparateiOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub fn glBlendEquationSeparateiOES_is_loaded() -> bool {
        !glBlendEquationSeparateiOES_p.load(RELAX).is_null()
    }

    /// [glBlendEquationi](http://docs.gl/es3/glBlendEquation)(buf, mode)
    /// * `mode` group: BlendEquationModeEXT
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendEquationi(buf: GLuint, mode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendEquationi({:?}, {:#X});", buf, mode);
        }
        let out = call_atomic_ptr_2arg("glBlendEquationi", &glBlendEquationi_p, buf, mode);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendEquationi");
        }
        out
    }
    static glBlendEquationi_p: APcv = ap_null();
    /// Tries to load [`glBlendEquationi`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendEquationi_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBlendEquationi\0", &glBlendEquationi_p)
    }
    /// Checks if the pointer for [`glBlendEquationi`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendEquationi_is_loaded() -> bool {
        !glBlendEquationi_p.load(RELAX).is_null()
    }

    /// [glBlendEquationiEXT](http://docs.gl/es3/glBlendEquationiEXT)(buf, mode)
    /// * `mode` group: BlendEquationModeEXT
    /// * alias of: [`glBlendEquationi`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glBlendEquationiEXT(buf: GLuint, mode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendEquationiEXT({:?}, {:#X});", buf, mode);
        }
        let out = call_atomic_ptr_2arg("glBlendEquationiEXT", &glBlendEquationiEXT_p, buf, mode);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendEquationiEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    static glBlendEquationiEXT_p: APcv = ap_null();
    /// Tries to load [`glBlendEquationiEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glBlendEquationiEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendEquationiEXT\0",
            &glBlendEquationiEXT_p,
        )
    }
    /// Checks if the pointer for [`glBlendEquationiEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub fn glBlendEquationiEXT_is_loaded() -> bool {
        !glBlendEquationiEXT_p.load(RELAX).is_null()
    }

    /// [glBlendEquationiOES](http://docs.gl/es3/glBlendEquationiOES)(buf, mode)
    /// * `mode` group: BlendEquationModeEXT
    /// * alias of: [`glBlendEquationi`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glBlendEquationiOES(buf: GLuint, mode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendEquationiOES({:?}, {:#X});", buf, mode);
        }
        let out = call_atomic_ptr_2arg("glBlendEquationiOES", &glBlendEquationiOES_p, buf, mode);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendEquationiOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    static glBlendEquationiOES_p: APcv = ap_null();
    /// Tries to load [`glBlendEquationiOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glBlendEquationiOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendEquationiOES\0",
            &glBlendEquationiOES_p,
        )
    }
    /// Checks if the pointer for [`glBlendEquationiOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub fn glBlendEquationiOES_is_loaded() -> bool {
        !glBlendEquationiOES_p.load(RELAX).is_null()
    }

    /// [glBlendFunc](http://docs.gl/es3/glBlendFunc)(sfactor, dfactor)
    /// * `sfactor` group: BlendingFactor
    /// * `dfactor` group: BlendingFactor
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendFunc(sfactor: GLenum, dfactor: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendFunc({:#X}, {:#X});", sfactor, dfactor);
        }
        let out = call_atomic_ptr_2arg("glBlendFunc", &glBlendFunc_p, sfactor, dfactor);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendFunc");
        }
        out
    }
    static glBlendFunc_p: APcv = ap_null();
    /// Tries to load [`glBlendFunc`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendFunc_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBlendFunc\0", &glBlendFunc_p)
    }
    /// Checks if the pointer for [`glBlendFunc`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendFunc_is_loaded() -> bool {
        !glBlendFunc_p.load(RELAX).is_null()
    }

    /// [glBlendFuncSeparate](http://docs.gl/es3/glBlendFuncSeparate)(sfactorRGB, dfactorRGB, sfactorAlpha, dfactorAlpha)
    /// * `sfactorRGB` group: BlendingFactor
    /// * `dfactorRGB` group: BlendingFactor
    /// * `sfactorAlpha` group: BlendingFactor
    /// * `dfactorAlpha` group: BlendingFactor
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendFuncSeparate(
        sfactorRGB: GLenum,
        dfactorRGB: GLenum,
        sfactorAlpha: GLenum,
        dfactorAlpha: GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendFuncSeparate({:#X}, {:#X}, {:#X}, {:#X});",
                sfactorRGB,
                dfactorRGB,
                sfactorAlpha,
                dfactorAlpha
            );
        }
        let out = call_atomic_ptr_4arg(
            "glBlendFuncSeparate",
            &glBlendFuncSeparate_p,
            sfactorRGB,
            dfactorRGB,
            sfactorAlpha,
            dfactorAlpha,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendFuncSeparate");
        }
        out
    }
    static glBlendFuncSeparate_p: APcv = ap_null();
    /// Tries to load [`glBlendFuncSeparate`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendFuncSeparate_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendFuncSeparate\0",
            &glBlendFuncSeparate_p,
        )
    }
    /// Checks if the pointer for [`glBlendFuncSeparate`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendFuncSeparate_is_loaded() -> bool {
        !glBlendFuncSeparate_p.load(RELAX).is_null()
    }

    /// [glBlendFuncSeparatei](http://docs.gl/es3/glBlendFuncSeparate)(buf, srcRGB, dstRGB, srcAlpha, dstAlpha)
    /// * `srcRGB` group: BlendingFactor
    /// * `dstRGB` group: BlendingFactor
    /// * `srcAlpha` group: BlendingFactor
    /// * `dstAlpha` group: BlendingFactor
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendFuncSeparatei(
        buf: GLuint,
        srcRGB: GLenum,
        dstRGB: GLenum,
        srcAlpha: GLenum,
        dstAlpha: GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendFuncSeparatei({:?}, {:#X}, {:#X}, {:#X}, {:#X});",
                buf,
                srcRGB,
                dstRGB,
                srcAlpha,
                dstAlpha
            );
        }
        let out = call_atomic_ptr_5arg(
            "glBlendFuncSeparatei",
            &glBlendFuncSeparatei_p,
            buf,
            srcRGB,
            dstRGB,
            srcAlpha,
            dstAlpha,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendFuncSeparatei");
        }
        out
    }
    static glBlendFuncSeparatei_p: APcv = ap_null();
    /// Tries to load [`glBlendFuncSeparatei`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendFuncSeparatei_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendFuncSeparatei\0",
            &glBlendFuncSeparatei_p,
        )
    }
    /// Checks if the pointer for [`glBlendFuncSeparatei`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendFuncSeparatei_is_loaded() -> bool {
        !glBlendFuncSeparatei_p.load(RELAX).is_null()
    }

    /// [glBlendFuncSeparateiEXT](http://docs.gl/es3/glBlendFuncSeparateiEXT)(buf, srcRGB, dstRGB, srcAlpha, dstAlpha)
    /// * `srcRGB` group: BlendingFactor
    /// * `dstRGB` group: BlendingFactor
    /// * `srcAlpha` group: BlendingFactor
    /// * `dstAlpha` group: BlendingFactor
    /// * alias of: [`glBlendFuncSeparatei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glBlendFuncSeparateiEXT(
        buf: GLuint,
        srcRGB: GLenum,
        dstRGB: GLenum,
        srcAlpha: GLenum,
        dstAlpha: GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendFuncSeparateiEXT({:?}, {:#X}, {:#X}, {:#X}, {:#X});",
                buf,
                srcRGB,
                dstRGB,
                srcAlpha,
                dstAlpha
            );
        }
        let out = call_atomic_ptr_5arg(
            "glBlendFuncSeparateiEXT",
            &glBlendFuncSeparateiEXT_p,
            buf,
            srcRGB,
            dstRGB,
            srcAlpha,
            dstAlpha,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendFuncSeparateiEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    static glBlendFuncSeparateiEXT_p: APcv = ap_null();
    /// Tries to load [`glBlendFuncSeparateiEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glBlendFuncSeparateiEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendFuncSeparateiEXT\0",
            &glBlendFuncSeparateiEXT_p,
        )
    }
    /// Checks if the pointer for [`glBlendFuncSeparateiEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub fn glBlendFuncSeparateiEXT_is_loaded() -> bool {
        !glBlendFuncSeparateiEXT_p.load(RELAX).is_null()
    }

    /// [glBlendFuncSeparateiOES](http://docs.gl/es3/glBlendFuncSeparateiOES)(buf, srcRGB, dstRGB, srcAlpha, dstAlpha)
    /// * `srcRGB` group: BlendingFactor
    /// * `dstRGB` group: BlendingFactor
    /// * `srcAlpha` group: BlendingFactor
    /// * `dstAlpha` group: BlendingFactor
    /// * alias of: [`glBlendFuncSeparatei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glBlendFuncSeparateiOES(
        buf: GLuint,
        srcRGB: GLenum,
        dstRGB: GLenum,
        srcAlpha: GLenum,
        dstAlpha: GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendFuncSeparateiOES({:?}, {:#X}, {:#X}, {:#X}, {:#X});",
                buf,
                srcRGB,
                dstRGB,
                srcAlpha,
                dstAlpha
            );
        }
        let out = call_atomic_ptr_5arg(
            "glBlendFuncSeparateiOES",
            &glBlendFuncSeparateiOES_p,
            buf,
            srcRGB,
            dstRGB,
            srcAlpha,
            dstAlpha,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendFuncSeparateiOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    static glBlendFuncSeparateiOES_p: APcv = ap_null();
    /// Tries to load [`glBlendFuncSeparateiOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glBlendFuncSeparateiOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendFuncSeparateiOES\0",
            &glBlendFuncSeparateiOES_p,
        )
    }
    /// Checks if the pointer for [`glBlendFuncSeparateiOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub fn glBlendFuncSeparateiOES_is_loaded() -> bool {
        !glBlendFuncSeparateiOES_p.load(RELAX).is_null()
    }

    /// [glBlendFunci](http://docs.gl/es3/glBlendFunc)(buf, src, dst)
    /// * `src` group: BlendingFactor
    /// * `dst` group: BlendingFactor
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlendFunci(buf: GLuint, src: GLenum, dst: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendFunci({:?}, {:#X}, {:#X});", buf, src, dst);
        }
        let out = call_atomic_ptr_3arg("glBlendFunci", &glBlendFunci_p, buf, src, dst);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendFunci");
        }
        out
    }
    static glBlendFunci_p: APcv = ap_null();
    /// Tries to load [`glBlendFunci`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlendFunci_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBlendFunci\0", &glBlendFunci_p)
    }
    /// Checks if the pointer for [`glBlendFunci`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlendFunci_is_loaded() -> bool {
        !glBlendFunci_p.load(RELAX).is_null()
    }

    /// [glBlendFunciEXT](http://docs.gl/es3/glBlendFunciEXT)(buf, src, dst)
    /// * `src` group: BlendingFactor
    /// * `dst` group: BlendingFactor
    /// * alias of: [`glBlendFunci`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glBlendFunciEXT(buf: GLuint, src: GLenum, dst: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendFunciEXT({:?}, {:#X}, {:#X});",
                buf,
                src,
                dst
            );
        }
        let out = call_atomic_ptr_3arg("glBlendFunciEXT", &glBlendFunciEXT_p, buf, src, dst);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendFunciEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    static glBlendFunciEXT_p: APcv = ap_null();
    /// Tries to load [`glBlendFunciEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glBlendFunciEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBlendFunciEXT\0", &glBlendFunciEXT_p)
    }
    /// Checks if the pointer for [`glBlendFunciEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub fn glBlendFunciEXT_is_loaded() -> bool {
        !glBlendFunciEXT_p.load(RELAX).is_null()
    }

    /// [glBlendFunciOES](http://docs.gl/es3/glBlendFunciOES)(buf, src, dst)
    /// * `src` group: BlendingFactor
    /// * `dst` group: BlendingFactor
    /// * alias of: [`glBlendFunci`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glBlendFunciOES(buf: GLuint, src: GLenum, dst: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlendFunciOES({:?}, {:#X}, {:#X});",
                buf,
                src,
                dst
            );
        }
        let out = call_atomic_ptr_3arg("glBlendFunciOES", &glBlendFunciOES_p, buf, src, dst);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendFunciOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    static glBlendFunciOES_p: APcv = ap_null();
    /// Tries to load [`glBlendFunciOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glBlendFunciOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBlendFunciOES\0", &glBlendFunciOES_p)
    }
    /// Checks if the pointer for [`glBlendFunciOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub fn glBlendFunciOES_is_loaded() -> bool {
        !glBlendFunciOES_p.load(RELAX).is_null()
    }

    /// [glBlendParameteriNV](http://docs.gl/es3/glBlendParameteriNV)(pname, value)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
    pub unsafe fn glBlendParameteriNV(pname: GLenum, value: GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glBlendParameteriNV({:#X}, {:?});", pname, value);
        }
        let out = call_atomic_ptr_2arg("glBlendParameteriNV", &glBlendParameteriNV_p, pname, value);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlendParameteriNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
    static glBlendParameteriNV_p: APcv = ap_null();
    /// Tries to load [`glBlendParameteriNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
    pub unsafe fn glBlendParameteriNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlendParameteriNV\0",
            &glBlendParameteriNV_p,
        )
    }
    /// Checks if the pointer for [`glBlendParameteriNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_blend_equation_advanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_blend_equation_advanced"))))]
    pub fn glBlendParameteriNV_is_loaded() -> bool {
        !glBlendParameteriNV_p.load(RELAX).is_null()
    }

    /// [glBlitFramebuffer](http://docs.gl/es3/glBlitFramebuffer)(srcX0, srcY0, srcX1, srcY1, dstX0, dstY0, dstX1, dstY1, mask, filter)
    /// * `mask` group: ClearBufferMask
    /// * `filter` group: BlitFramebufferFilter
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBlitFramebuffer(
        srcX0: GLint,
        srcY0: GLint,
        srcX1: GLint,
        srcY1: GLint,
        dstX0: GLint,
        dstY0: GLint,
        dstX1: GLint,
        dstY1: GLint,
        mask: GLbitfield,
        filter: GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlitFramebuffer({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X});",
                srcX0,
                srcY0,
                srcX1,
                srcY1,
                dstX0,
                dstY0,
                dstX1,
                dstY1,
                mask,
                filter
            );
        }
        let out = call_atomic_ptr_10arg(
            "glBlitFramebuffer",
            &glBlitFramebuffer_p,
            srcX0,
            srcY0,
            srcX1,
            srcY1,
            dstX0,
            dstY0,
            dstX1,
            dstY1,
            mask,
            filter,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlitFramebuffer");
        }
        out
    }
    static glBlitFramebuffer_p: APcv = ap_null();
    /// Tries to load [`glBlitFramebuffer`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBlitFramebuffer_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlitFramebuffer\0",
            &glBlitFramebuffer_p,
        )
    }
    /// Checks if the pointer for [`glBlitFramebuffer`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBlitFramebuffer_is_loaded() -> bool {
        !glBlitFramebuffer_p.load(RELAX).is_null()
    }

    /// [glBlitFramebufferANGLE](http://docs.gl/es3/glBlitFramebufferANGLE)(srcX0, srcY0, srcX1, srcY1, dstX0, dstY0, dstX1, dstY1, mask, filter)
    /// * `mask` group: ClearBufferMask
    /// * `filter` group: BlitFramebufferFilter
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_ANGLE_framebuffer_blit"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_blit"))))]
    pub unsafe fn glBlitFramebufferANGLE(
        srcX0: GLint,
        srcY0: GLint,
        srcX1: GLint,
        srcY1: GLint,
        dstX0: GLint,
        dstY0: GLint,
        dstX1: GLint,
        dstY1: GLint,
        mask: GLbitfield,
        filter: GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlitFramebufferANGLE({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X});",
                srcX0,
                srcY0,
                srcX1,
                srcY1,
                dstX0,
                dstY0,
                dstX1,
                dstY1,
                mask,
                filter
            );
        }
        let out = call_atomic_ptr_10arg(
            "glBlitFramebufferANGLE",
            &glBlitFramebufferANGLE_p,
            srcX0,
            srcY0,
            srcX1,
            srcY1,
            dstX0,
            dstY0,
            dstX1,
            dstY1,
            mask,
            filter,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlitFramebufferANGLE");
        }
        out
    }
    #[cfg(any(feature = "GL_ANGLE_framebuffer_blit"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_blit"))))]
    static glBlitFramebufferANGLE_p: APcv = ap_null();
    /// Tries to load [`glBlitFramebufferANGLE`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_ANGLE_framebuffer_blit"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_blit"))))]
    pub unsafe fn glBlitFramebufferANGLE_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlitFramebufferANGLE\0",
            &glBlitFramebufferANGLE_p,
        )
    }
    /// Checks if the pointer for [`glBlitFramebufferANGLE`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_ANGLE_framebuffer_blit"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_framebuffer_blit"))))]
    pub fn glBlitFramebufferANGLE_is_loaded() -> bool {
        !glBlitFramebufferANGLE_p.load(RELAX).is_null()
    }

    /// [glBlitFramebufferNV](http://docs.gl/es3/glBlitFramebufferNV)(srcX0, srcY0, srcX1, srcY1, dstX0, dstY0, dstX1, dstY1, mask, filter)
    /// * `mask` group: ClearBufferMask
    /// * `filter` group: BlitFramebufferFilter
    /// * alias of: [`glBlitFramebuffer`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_framebuffer_blit"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_blit"))))]
    pub unsafe fn glBlitFramebufferNV(
        srcX0: GLint,
        srcY0: GLint,
        srcX1: GLint,
        srcY1: GLint,
        dstX0: GLint,
        dstY0: GLint,
        dstX1: GLint,
        dstY1: GLint,
        mask: GLbitfield,
        filter: GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBlitFramebufferNV({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X});",
                srcX0,
                srcY0,
                srcX1,
                srcY1,
                dstX0,
                dstY0,
                dstX1,
                dstY1,
                mask,
                filter
            );
        }
        let out = call_atomic_ptr_10arg(
            "glBlitFramebufferNV",
            &glBlitFramebufferNV_p,
            srcX0,
            srcY0,
            srcX1,
            srcY1,
            dstX0,
            dstY0,
            dstX1,
            dstY1,
            mask,
            filter,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBlitFramebufferNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_framebuffer_blit"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_blit"))))]
    static glBlitFramebufferNV_p: APcv = ap_null();
    /// Tries to load [`glBlitFramebufferNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_framebuffer_blit"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_blit"))))]
    pub unsafe fn glBlitFramebufferNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBlitFramebufferNV\0",
            &glBlitFramebufferNV_p,
        )
    }
    /// Checks if the pointer for [`glBlitFramebufferNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_framebuffer_blit"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_blit"))))]
    pub fn glBlitFramebufferNV_is_loaded() -> bool {
        !glBlitFramebufferNV_p.load(RELAX).is_null()
    }

    /// [glBufferAttachMemoryNV](http://docs.gl/es3/glBufferAttachMemoryNV)(target, memory, offset)
    /// * `target` group: BufferTargetARB
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_memory_attachment"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
    pub unsafe fn glBufferAttachMemoryNV(target: GLenum, memory: GLuint, offset: GLuint64) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBufferAttachMemoryNV({:#X}, {:?}, {:?});",
                target,
                memory,
                offset
            );
        }
        let out = call_atomic_ptr_3arg(
            "glBufferAttachMemoryNV",
            &glBufferAttachMemoryNV_p,
            target,
            memory,
            offset,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBufferAttachMemoryNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_memory_attachment"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
    static glBufferAttachMemoryNV_p: APcv = ap_null();
    /// Tries to load [`glBufferAttachMemoryNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_memory_attachment"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
    pub unsafe fn glBufferAttachMemoryNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBufferAttachMemoryNV\0",
            &glBufferAttachMemoryNV_p,
        )
    }
    /// Checks if the pointer for [`glBufferAttachMemoryNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_memory_attachment"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
    pub fn glBufferAttachMemoryNV_is_loaded() -> bool {
        !glBufferAttachMemoryNV_p.load(RELAX).is_null()
    }

    /// [glBufferData](http://docs.gl/es3/glBufferData)(target, size, data, usage)
    /// * `target` group: BufferTargetARB
    /// * `size` group: BufferSize
    /// * `data` len: size
    /// * `usage` group: BufferUsageARB
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBufferData(
        target: GLenum,
        size: GLsizeiptr,
        data: *const c_void,
        usage: GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBufferData({:#X}, {:?}, {:p}, {:#X});",
                target,
                size,
                data,
                usage
            );
        }
        let out = call_atomic_ptr_4arg("glBufferData", &glBufferData_p, target, size, data, usage);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBufferData");
        }
        out
    }
    static glBufferData_p: APcv = ap_null();
    /// Tries to load [`glBufferData`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBufferData_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBufferData\0", &glBufferData_p)
    }
    /// Checks if the pointer for [`glBufferData`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBufferData_is_loaded() -> bool {
        !glBufferData_p.load(RELAX).is_null()
    }

    /// [glBufferPageCommitmentMemNV](http://docs.gl/es3/glBufferPageCommitmentMemNV)(target, offset, size, memory, memOffset, commit)
    /// * `target` group: BufferStorageTarget
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
    pub unsafe fn glBufferPageCommitmentMemNV(
        target: GLenum,
        offset: GLintptr,
        size: GLsizeiptr,
        memory: GLuint,
        memOffset: GLuint64,
        commit: GLboolean,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBufferPageCommitmentMemNV({:#X}, {:?}, {:?}, {:?}, {:?}, {:?});",
                target,
                offset,
                size,
                memory,
                memOffset,
                commit
            );
        }
        let out = call_atomic_ptr_6arg(
            "glBufferPageCommitmentMemNV",
            &glBufferPageCommitmentMemNV_p,
            target,
            offset,
            size,
            memory,
            memOffset,
            commit,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBufferPageCommitmentMemNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
    static glBufferPageCommitmentMemNV_p: APcv = ap_null();
    /// Tries to load [`glBufferPageCommitmentMemNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
    pub unsafe fn glBufferPageCommitmentMemNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBufferPageCommitmentMemNV\0",
            &glBufferPageCommitmentMemNV_p,
        )
    }
    /// Checks if the pointer for [`glBufferPageCommitmentMemNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_memory_object_sparse"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_object_sparse"))))]
    pub fn glBufferPageCommitmentMemNV_is_loaded() -> bool {
        !glBufferPageCommitmentMemNV_p.load(RELAX).is_null()
    }

    /// [glBufferStorageEXT](http://docs.gl/es3/glBufferStorageEXT)(target, size, data, flags)
    /// * `target` group: BufferStorageTarget
    /// * `data` len: size
    /// * `flags` group: BufferStorageMask
    /// * alias of: [`glBufferStorage`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_buffer_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_buffer_storage"))))]
    pub unsafe fn glBufferStorageEXT(
        target: GLenum,
        size: GLsizeiptr,
        data: *const c_void,
        flags: GLbitfield,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBufferStorageEXT({:#X}, {:?}, {:p}, {:?});",
                target,
                size,
                data,
                flags
            );
        }
        let out = call_atomic_ptr_4arg(
            "glBufferStorageEXT",
            &glBufferStorageEXT_p,
            target,
            size,
            data,
            flags,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBufferStorageEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_buffer_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_buffer_storage"))))]
    static glBufferStorageEXT_p: APcv = ap_null();
    /// Tries to load [`glBufferStorageEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_buffer_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_buffer_storage"))))]
    pub unsafe fn glBufferStorageEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBufferStorageEXT\0",
            &glBufferStorageEXT_p,
        )
    }
    /// Checks if the pointer for [`glBufferStorageEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_buffer_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_buffer_storage"))))]
    pub fn glBufferStorageEXT_is_loaded() -> bool {
        !glBufferStorageEXT_p.load(RELAX).is_null()
    }

    /// [glBufferStorageExternalEXT](http://docs.gl/es3/glBufferStorageExternalEXT)(target, offset, size, clientBuffer, flags)
    /// * `flags` group: BufferStorageMask
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_external_buffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
    pub unsafe fn glBufferStorageExternalEXT(
        target: GLenum,
        offset: GLintptr,
        size: GLsizeiptr,
        clientBuffer: GLeglClientBufferEXT,
        flags: GLbitfield,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBufferStorageExternalEXT({:#X}, {:?}, {:?}, {:?}, {:?});",
                target,
                offset,
                size,
                clientBuffer,
                flags
            );
        }
        let out = call_atomic_ptr_5arg(
            "glBufferStorageExternalEXT",
            &glBufferStorageExternalEXT_p,
            target,
            offset,
            size,
            clientBuffer,
            flags,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBufferStorageExternalEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_external_buffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
    static glBufferStorageExternalEXT_p: APcv = ap_null();
    /// Tries to load [`glBufferStorageExternalEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_external_buffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
    pub unsafe fn glBufferStorageExternalEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBufferStorageExternalEXT\0",
            &glBufferStorageExternalEXT_p,
        )
    }
    /// Checks if the pointer for [`glBufferStorageExternalEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_external_buffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_external_buffer"))))]
    pub fn glBufferStorageExternalEXT_is_loaded() -> bool {
        !glBufferStorageExternalEXT_p.load(RELAX).is_null()
    }

    /// [glBufferStorageMemEXT](http://docs.gl/es3/glBufferStorageMemEXT)(target, size, memory, offset)
    /// * `target` group: BufferTargetARB
    /// * `size` group: BufferSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub unsafe fn glBufferStorageMemEXT(
        target: GLenum,
        size: GLsizeiptr,
        memory: GLuint,
        offset: GLuint64,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBufferStorageMemEXT({:#X}, {:?}, {:?}, {:?});",
                target,
                size,
                memory,
                offset
            );
        }
        let out = call_atomic_ptr_4arg(
            "glBufferStorageMemEXT",
            &glBufferStorageMemEXT_p,
            target,
            size,
            memory,
            offset,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBufferStorageMemEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    static glBufferStorageMemEXT_p: APcv = ap_null();
    /// Tries to load [`glBufferStorageMemEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub unsafe fn glBufferStorageMemEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glBufferStorageMemEXT\0",
            &glBufferStorageMemEXT_p,
        )
    }
    /// Checks if the pointer for [`glBufferStorageMemEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub fn glBufferStorageMemEXT_is_loaded() -> bool {
        !glBufferStorageMemEXT_p.load(RELAX).is_null()
    }

    /// [glBufferSubData](http://docs.gl/es3/glBufferSubData)(target, offset, size, data)
    /// * `target` group: BufferTargetARB
    /// * `offset` group: BufferOffset
    /// * `size` group: BufferSize
    /// * `data` len: size
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glBufferSubData(
        target: GLenum,
        offset: GLintptr,
        size: GLsizeiptr,
        data: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glBufferSubData({:#X}, {:?}, {:?}, {:p});",
                target,
                offset,
                size,
                data
            );
        }
        let out = call_atomic_ptr_4arg(
            "glBufferSubData",
            &glBufferSubData_p,
            target,
            offset,
            size,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glBufferSubData");
        }
        out
    }
    static glBufferSubData_p: APcv = ap_null();
    /// Tries to load [`glBufferSubData`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glBufferSubData_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glBufferSubData\0", &glBufferSubData_p)
    }
    /// Checks if the pointer for [`glBufferSubData`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glBufferSubData_is_loaded() -> bool {
        !glBufferSubData_p.load(RELAX).is_null()
    }

    /// [glCheckFramebufferStatus](http://docs.gl/es3/glCheckFramebufferStatus)(target)
    /// * `target` group: FramebufferTarget
    /// * return value group: FramebufferStatus
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCheckFramebufferStatus(target: GLenum) -> GLenum {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCheckFramebufferStatus({:#X});", target);
        }
        let out = call_atomic_ptr_1arg(
            "glCheckFramebufferStatus",
            &glCheckFramebufferStatus_p,
            target,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCheckFramebufferStatus");
        }
        out
    }
    static glCheckFramebufferStatus_p: APcv = ap_null();
    /// Tries to load [`glCheckFramebufferStatus`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCheckFramebufferStatus_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCheckFramebufferStatus\0",
            &glCheckFramebufferStatus_p,
        )
    }
    /// Checks if the pointer for [`glCheckFramebufferStatus`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCheckFramebufferStatus_is_loaded() -> bool {
        !glCheckFramebufferStatus_p.load(RELAX).is_null()
    }

    /// [glClear](http://docs.gl/es3/glClear)(mask)
    /// * `mask` group: ClearBufferMask
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glClear(mask: GLbitfield) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glClear({:?});", mask);
        }
        let out = call_atomic_ptr_1arg("glClear", &glClear_p, mask);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClear");
        }
        out
    }
    static glClear_p: APcv = ap_null();
    /// Tries to load [`glClear`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glClear_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClear\0", &glClear_p)
    }
    /// Checks if the pointer for [`glClear`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glClear_is_loaded() -> bool {
        !glClear_p.load(RELAX).is_null()
    }

    /// [glClearBufferfi](http://docs.gl/es3/glClearBuffer)(buffer, drawbuffer, depth, stencil)
    /// * `buffer` group: Buffer
    /// * `drawbuffer` group: DrawBufferName
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glClearBufferfi(
        buffer: GLenum,
        drawbuffer: GLint,
        depth: GLfloat,
        stencil: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClearBufferfi({:#X}, {:?}, {:?}, {:?});",
                buffer,
                drawbuffer,
                depth,
                stencil
            );
        }
        let out = call_atomic_ptr_4arg(
            "glClearBufferfi",
            &glClearBufferfi_p,
            buffer,
            drawbuffer,
            depth,
            stencil,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearBufferfi");
        }
        out
    }
    static glClearBufferfi_p: APcv = ap_null();
    /// Tries to load [`glClearBufferfi`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glClearBufferfi_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClearBufferfi\0", &glClearBufferfi_p)
    }
    /// Checks if the pointer for [`glClearBufferfi`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glClearBufferfi_is_loaded() -> bool {
        !glClearBufferfi_p.load(RELAX).is_null()
    }

    /// [glClearBufferfv](http://docs.gl/es3/glClearBuffer)(buffer, drawbuffer, value)
    /// * `buffer` group: Buffer
    /// * `drawbuffer` group: DrawBufferName
    /// * `value` len: COMPSIZE(buffer)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glClearBufferfv(buffer: GLenum, drawbuffer: GLint, value: *const GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClearBufferfv({:#X}, {:?}, {:p});",
                buffer,
                drawbuffer,
                value
            );
        }
        let out = call_atomic_ptr_3arg(
            "glClearBufferfv",
            &glClearBufferfv_p,
            buffer,
            drawbuffer,
            value,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearBufferfv");
        }
        out
    }
    static glClearBufferfv_p: APcv = ap_null();
    /// Tries to load [`glClearBufferfv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glClearBufferfv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClearBufferfv\0", &glClearBufferfv_p)
    }
    /// Checks if the pointer for [`glClearBufferfv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glClearBufferfv_is_loaded() -> bool {
        !glClearBufferfv_p.load(RELAX).is_null()
    }

    /// [glClearBufferiv](http://docs.gl/es3/glClearBuffer)(buffer, drawbuffer, value)
    /// * `buffer` group: Buffer
    /// * `drawbuffer` group: DrawBufferName
    /// * `value` len: COMPSIZE(buffer)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glClearBufferiv(buffer: GLenum, drawbuffer: GLint, value: *const GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClearBufferiv({:#X}, {:?}, {:p});",
                buffer,
                drawbuffer,
                value
            );
        }
        let out = call_atomic_ptr_3arg(
            "glClearBufferiv",
            &glClearBufferiv_p,
            buffer,
            drawbuffer,
            value,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearBufferiv");
        }
        out
    }
    static glClearBufferiv_p: APcv = ap_null();
    /// Tries to load [`glClearBufferiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glClearBufferiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClearBufferiv\0", &glClearBufferiv_p)
    }
    /// Checks if the pointer for [`glClearBufferiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glClearBufferiv_is_loaded() -> bool {
        !glClearBufferiv_p.load(RELAX).is_null()
    }

    /// [glClearBufferuiv](http://docs.gl/es3/glClearBuffer)(buffer, drawbuffer, value)
    /// * `buffer` group: Buffer
    /// * `drawbuffer` group: DrawBufferName
    /// * `value` len: COMPSIZE(buffer)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glClearBufferuiv(buffer: GLenum, drawbuffer: GLint, value: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClearBufferuiv({:#X}, {:?}, {:p});",
                buffer,
                drawbuffer,
                value
            );
        }
        let out = call_atomic_ptr_3arg(
            "glClearBufferuiv",
            &glClearBufferuiv_p,
            buffer,
            drawbuffer,
            value,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearBufferuiv");
        }
        out
    }
    static glClearBufferuiv_p: APcv = ap_null();
    /// Tries to load [`glClearBufferuiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glClearBufferuiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClearBufferuiv\0", &glClearBufferuiv_p)
    }
    /// Checks if the pointer for [`glClearBufferuiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glClearBufferuiv_is_loaded() -> bool {
        !glClearBufferuiv_p.load(RELAX).is_null()
    }

    /// [glClearColor](http://docs.gl/es3/glClearColor)(red, green, blue, alpha)
    /// * `red` group: ColorF
    /// * `green` group: ColorF
    /// * `blue` group: ColorF
    /// * `alpha` group: ColorF
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glClearColor(red: GLfloat, green: GLfloat, blue: GLfloat, alpha: GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClearColor({:?}, {:?}, {:?}, {:?});",
                red,
                green,
                blue,
                alpha
            );
        }
        let out = call_atomic_ptr_4arg("glClearColor", &glClearColor_p, red, green, blue, alpha);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearColor");
        }
        out
    }
    static glClearColor_p: APcv = ap_null();
    /// Tries to load [`glClearColor`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glClearColor_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClearColor\0", &glClearColor_p)
    }
    /// Checks if the pointer for [`glClearColor`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glClearColor_is_loaded() -> bool {
        !glClearColor_p.load(RELAX).is_null()
    }

    /// [glClearDepthf](http://docs.gl/es3/glClearDepth)(d)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glClearDepthf(d: GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glClearDepthf({:?});", d);
        }
        let out = call_atomic_ptr_1arg("glClearDepthf", &glClearDepthf_p, d);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearDepthf");
        }
        out
    }
    static glClearDepthf_p: APcv = ap_null();
    /// Tries to load [`glClearDepthf`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glClearDepthf_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClearDepthf\0", &glClearDepthf_p)
    }
    /// Checks if the pointer for [`glClearDepthf`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glClearDepthf_is_loaded() -> bool {
        !glClearDepthf_p.load(RELAX).is_null()
    }

    /// [glClearPixelLocalStorageuiEXT](http://docs.gl/es3/glClearPixelLocalStorageuiEXT)(offset, n, values)
    /// * `values` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    pub unsafe fn glClearPixelLocalStorageuiEXT(
        offset: GLsizei,
        n: GLsizei,
        values: *const GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClearPixelLocalStorageuiEXT({:?}, {:?}, {:p});",
                offset,
                n,
                values
            );
        }
        let out = call_atomic_ptr_3arg(
            "glClearPixelLocalStorageuiEXT",
            &glClearPixelLocalStorageuiEXT_p,
            offset,
            n,
            values,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearPixelLocalStorageuiEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    static glClearPixelLocalStorageuiEXT_p: APcv = ap_null();
    /// Tries to load [`glClearPixelLocalStorageuiEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    pub unsafe fn glClearPixelLocalStorageuiEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glClearPixelLocalStorageuiEXT\0",
            &glClearPixelLocalStorageuiEXT_p,
        )
    }
    /// Checks if the pointer for [`glClearPixelLocalStorageuiEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    pub fn glClearPixelLocalStorageuiEXT_is_loaded() -> bool {
        !glClearPixelLocalStorageuiEXT_p.load(RELAX).is_null()
    }

    /// [glClearStencil](http://docs.gl/es3/glClearStencil)(s)
    /// * `s` group: StencilValue
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glClearStencil(s: GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glClearStencil({:?});", s);
        }
        let out = call_atomic_ptr_1arg("glClearStencil", &glClearStencil_p, s);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearStencil");
        }
        out
    }
    static glClearStencil_p: APcv = ap_null();
    /// Tries to load [`glClearStencil`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glClearStencil_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClearStencil\0", &glClearStencil_p)
    }
    /// Checks if the pointer for [`glClearStencil`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glClearStencil_is_loaded() -> bool {
        !glClearStencil_p.load(RELAX).is_null()
    }

    /// [glClearTexImageEXT](http://docs.gl/es3/glClearTexImageEXT)(texture, level, format, type_, data)
    /// * `texture` class: texture
    /// * `format` group: PixelFormat
    /// * `type_` group: PixelType
    /// * `data` len: COMPSIZE(format,type)
    /// * alias of: [`glClearTexImage`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_clear_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
    pub unsafe fn glClearTexImageEXT(
        texture: GLuint,
        level: GLint,
        format: GLenum,
        type_: GLenum,
        data: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClearTexImageEXT({:?}, {:?}, {:#X}, {:#X}, {:p});",
                texture,
                level,
                format,
                type_,
                data
            );
        }
        let out = call_atomic_ptr_5arg(
            "glClearTexImageEXT",
            &glClearTexImageEXT_p,
            texture,
            level,
            format,
            type_,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearTexImageEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_clear_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
    static glClearTexImageEXT_p: APcv = ap_null();
    /// Tries to load [`glClearTexImageEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_clear_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
    pub unsafe fn glClearTexImageEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glClearTexImageEXT\0",
            &glClearTexImageEXT_p,
        )
    }
    /// Checks if the pointer for [`glClearTexImageEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_clear_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
    pub fn glClearTexImageEXT_is_loaded() -> bool {
        !glClearTexImageEXT_p.load(RELAX).is_null()
    }

    /// [glClearTexSubImageEXT](http://docs.gl/es3/glClearTexSubImageEXT)(texture, level, xoffset, yoffset, zoffset, width, height, depth, format, type_, data)
    /// * `texture` class: texture
    /// * `format` group: PixelFormat
    /// * `type_` group: PixelType
    /// * `data` len: COMPSIZE(format,type)
    /// * alias of: [`glClearTexSubImage`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_clear_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
    pub unsafe fn glClearTexSubImageEXT(
        texture: GLuint,
        level: GLint,
        xoffset: GLint,
        yoffset: GLint,
        zoffset: GLint,
        width: GLsizei,
        height: GLsizei,
        depth: GLsizei,
        format: GLenum,
        type_: GLenum,
        data: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClearTexSubImageEXT({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:#X}, {:p});",
                texture,
                level,
                xoffset,
                yoffset,
                zoffset,
                width,
                height,
                depth,
                format,
                type_,
                data
            );
        }
        let out = call_atomic_ptr_11arg(
            "glClearTexSubImageEXT",
            &glClearTexSubImageEXT_p,
            texture,
            level,
            xoffset,
            yoffset,
            zoffset,
            width,
            height,
            depth,
            format,
            type_,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClearTexSubImageEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_clear_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
    static glClearTexSubImageEXT_p: APcv = ap_null();
    /// Tries to load [`glClearTexSubImageEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_clear_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
    pub unsafe fn glClearTexSubImageEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glClearTexSubImageEXT\0",
            &glClearTexSubImageEXT_p,
        )
    }
    /// Checks if the pointer for [`glClearTexSubImageEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_clear_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clear_texture"))))]
    pub fn glClearTexSubImageEXT_is_loaded() -> bool {
        !glClearTexSubImageEXT_p.load(RELAX).is_null()
    }

    /// [glClientWaitSync](http://docs.gl/es3/glClientWaitSync)(sync, flags, timeout)
    /// * `sync` group: sync
    /// * `sync` class: sync
    /// * `flags` group: SyncObjectMask
    /// * return value group: SyncStatus
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glClientWaitSync(sync: GLsync, flags: GLbitfield, timeout: GLuint64) -> GLenum {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClientWaitSync({:p}, {:?}, {:?});",
                sync,
                flags,
                timeout
            );
        }
        let out = call_atomic_ptr_3arg(
            "glClientWaitSync",
            &glClientWaitSync_p,
            sync,
            flags,
            timeout,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClientWaitSync");
        }
        out
    }
    static glClientWaitSync_p: APcv = ap_null();
    /// Tries to load [`glClientWaitSync`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glClientWaitSync_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClientWaitSync\0", &glClientWaitSync_p)
    }
    /// Checks if the pointer for [`glClientWaitSync`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glClientWaitSync_is_loaded() -> bool {
        !glClientWaitSync_p.load(RELAX).is_null()
    }

    /// [glClientWaitSyncAPPLE](http://docs.gl/es3/glClientWaitSyncAPPLE)(sync, flags, timeout)
    /// * `sync` group: sync
    /// * `sync` class: sync
    /// * `flags` group: SyncObjectMask
    /// * return value group: SyncStatus
    /// * alias of: [`glClientWaitSync`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glClientWaitSyncAPPLE(
        sync: GLsync,
        flags: GLbitfield,
        timeout: GLuint64,
    ) -> GLenum {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glClientWaitSyncAPPLE({:p}, {:?}, {:?});",
                sync,
                flags,
                timeout
            );
        }
        let out = call_atomic_ptr_3arg(
            "glClientWaitSyncAPPLE",
            &glClientWaitSyncAPPLE_p,
            sync,
            flags,
            timeout,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClientWaitSyncAPPLE");
        }
        out
    }
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    static glClientWaitSyncAPPLE_p: APcv = ap_null();
    /// Tries to load [`glClientWaitSyncAPPLE`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glClientWaitSyncAPPLE_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glClientWaitSyncAPPLE\0",
            &glClientWaitSyncAPPLE_p,
        )
    }
    /// Checks if the pointer for [`glClientWaitSyncAPPLE`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub fn glClientWaitSyncAPPLE_is_loaded() -> bool {
        !glClientWaitSyncAPPLE_p.load(RELAX).is_null()
    }

    /// [glClipControlEXT](http://docs.gl/es3/glClipControlEXT)(origin, depth)
    /// * alias of: [`glClipControl`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_clip_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clip_control"))))]
    pub unsafe fn glClipControlEXT(origin: GLenum, depth: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glClipControlEXT({:#X}, {:#X});", origin, depth);
        }
        let out = call_atomic_ptr_2arg("glClipControlEXT", &glClipControlEXT_p, origin, depth);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glClipControlEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_clip_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clip_control"))))]
    static glClipControlEXT_p: APcv = ap_null();
    /// Tries to load [`glClipControlEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_clip_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clip_control"))))]
    pub unsafe fn glClipControlEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glClipControlEXT\0", &glClipControlEXT_p)
    }
    /// Checks if the pointer for [`glClipControlEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_clip_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_clip_control"))))]
    pub fn glClipControlEXT_is_loaded() -> bool {
        !glClipControlEXT_p.load(RELAX).is_null()
    }

    /// [glColorMask](http://docs.gl/es3/glColorMask)(red, green, blue, alpha)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glColorMask(red: GLboolean, green: GLboolean, blue: GLboolean, alpha: GLboolean) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glColorMask({:?}, {:?}, {:?}, {:?});",
                red,
                green,
                blue,
                alpha
            );
        }
        let out = call_atomic_ptr_4arg("glColorMask", &glColorMask_p, red, green, blue, alpha);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glColorMask");
        }
        out
    }
    static glColorMask_p: APcv = ap_null();
    /// Tries to load [`glColorMask`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glColorMask_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glColorMask\0", &glColorMask_p)
    }
    /// Checks if the pointer for [`glColorMask`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glColorMask_is_loaded() -> bool {
        !glColorMask_p.load(RELAX).is_null()
    }

    /// [glColorMaski](http://docs.gl/es3/glColorMask)(index, r, g, b, a)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glColorMaski(
        index: GLuint,
        r: GLboolean,
        g: GLboolean,
        b: GLboolean,
        a: GLboolean,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glColorMaski({:?}, {:?}, {:?}, {:?}, {:?});",
                index,
                r,
                g,
                b,
                a
            );
        }
        let out = call_atomic_ptr_5arg("glColorMaski", &glColorMaski_p, index, r, g, b, a);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glColorMaski");
        }
        out
    }
    static glColorMaski_p: APcv = ap_null();
    /// Tries to load [`glColorMaski`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glColorMaski_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glColorMaski\0", &glColorMaski_p)
    }
    /// Checks if the pointer for [`glColorMaski`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glColorMaski_is_loaded() -> bool {
        !glColorMaski_p.load(RELAX).is_null()
    }

    /// [glColorMaskiEXT](http://docs.gl/es3/glColorMaskiEXT)(index, r, g, b, a)
    /// * alias of: [`glColorMaski`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glColorMaskiEXT(
        index: GLuint,
        r: GLboolean,
        g: GLboolean,
        b: GLboolean,
        a: GLboolean,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glColorMaskiEXT({:?}, {:?}, {:?}, {:?}, {:?});",
                index,
                r,
                g,
                b,
                a
            );
        }
        let out = call_atomic_ptr_5arg("glColorMaskiEXT", &glColorMaskiEXT_p, index, r, g, b, a);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glColorMaskiEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    static glColorMaskiEXT_p: APcv = ap_null();
    /// Tries to load [`glColorMaskiEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glColorMaskiEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glColorMaskiEXT\0", &glColorMaskiEXT_p)
    }
    /// Checks if the pointer for [`glColorMaskiEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub fn glColorMaskiEXT_is_loaded() -> bool {
        !glColorMaskiEXT_p.load(RELAX).is_null()
    }

    /// [glColorMaskiOES](http://docs.gl/es3/glColorMaskiOES)(index, r, g, b, a)
    /// * alias of: [`glColorMaski`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glColorMaskiOES(
        index: GLuint,
        r: GLboolean,
        g: GLboolean,
        b: GLboolean,
        a: GLboolean,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glColorMaskiOES({:?}, {:?}, {:?}, {:?}, {:?});",
                index,
                r,
                g,
                b,
                a
            );
        }
        let out = call_atomic_ptr_5arg("glColorMaskiOES", &glColorMaskiOES_p, index, r, g, b, a);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glColorMaskiOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    static glColorMaskiOES_p: APcv = ap_null();
    /// Tries to load [`glColorMaskiOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub unsafe fn glColorMaskiOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glColorMaskiOES\0", &glColorMaskiOES_p)
    }
    /// Checks if the pointer for [`glColorMaskiOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_buffers_indexed"))))]
    pub fn glColorMaskiOES_is_loaded() -> bool {
        !glColorMaskiOES_p.load(RELAX).is_null()
    }

    /// [glCompileShader](http://docs.gl/es3/glCompileShader)(shader)
    /// * `shader` class: shader
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCompileShader(shader: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCompileShader({:?});", shader);
        }
        let out = call_atomic_ptr_1arg("glCompileShader", &glCompileShader_p, shader);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCompileShader");
        }
        out
    }
    static glCompileShader_p: APcv = ap_null();
    /// Tries to load [`glCompileShader`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCompileShader_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glCompileShader\0", &glCompileShader_p)
    }
    /// Checks if the pointer for [`glCompileShader`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCompileShader_is_loaded() -> bool {
        !glCompileShader_p.load(RELAX).is_null()
    }

    /// [glCompressedTexImage2D](http://docs.gl/es3/glCompressedTexImage2D)(target, level, internalformat, width, height, border, imageSize, data)
    /// * `target` group: TextureTarget
    /// * `level` group: CheckedInt32
    /// * `internalformat` group: InternalFormat
    /// * `border` group: CheckedInt32
    /// * `data` group: CompressedTextureARB
    /// * `data` len: imageSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCompressedTexImage2D(
        target: GLenum,
        level: GLint,
        internalformat: GLenum,
        width: GLsizei,
        height: GLsizei,
        border: GLint,
        imageSize: GLsizei,
        data: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCompressedTexImage2D({:#X}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:p});",
                target,
                level,
                internalformat,
                width,
                height,
                border,
                imageSize,
                data
            );
        }
        let out = call_atomic_ptr_8arg(
            "glCompressedTexImage2D",
            &glCompressedTexImage2D_p,
            target,
            level,
            internalformat,
            width,
            height,
            border,
            imageSize,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCompressedTexImage2D");
        }
        out
    }
    static glCompressedTexImage2D_p: APcv = ap_null();
    /// Tries to load [`glCompressedTexImage2D`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCompressedTexImage2D_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCompressedTexImage2D\0",
            &glCompressedTexImage2D_p,
        )
    }
    /// Checks if the pointer for [`glCompressedTexImage2D`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCompressedTexImage2D_is_loaded() -> bool {
        !glCompressedTexImage2D_p.load(RELAX).is_null()
    }

    /// [glCompressedTexImage3D](http://docs.gl/es3/glCompressedTexImage3D)(target, level, internalformat, width, height, depth, border, imageSize, data)
    /// * `target` group: TextureTarget
    /// * `level` group: CheckedInt32
    /// * `internalformat` group: InternalFormat
    /// * `border` group: CheckedInt32
    /// * `data` group: CompressedTextureARB
    /// * `data` len: imageSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCompressedTexImage3D(
        target: GLenum,
        level: GLint,
        internalformat: GLenum,
        width: GLsizei,
        height: GLsizei,
        depth: GLsizei,
        border: GLint,
        imageSize: GLsizei,
        data: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCompressedTexImage3D({:#X}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:p});",
                target,
                level,
                internalformat,
                width,
                height,
                depth,
                border,
                imageSize,
                data
            );
        }
        let out = call_atomic_ptr_9arg(
            "glCompressedTexImage3D",
            &glCompressedTexImage3D_p,
            target,
            level,
            internalformat,
            width,
            height,
            depth,
            border,
            imageSize,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCompressedTexImage3D");
        }
        out
    }
    static glCompressedTexImage3D_p: APcv = ap_null();
    /// Tries to load [`glCompressedTexImage3D`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCompressedTexImage3D_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCompressedTexImage3D\0",
            &glCompressedTexImage3D_p,
        )
    }
    /// Checks if the pointer for [`glCompressedTexImage3D`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCompressedTexImage3D_is_loaded() -> bool {
        !glCompressedTexImage3D_p.load(RELAX).is_null()
    }

    /// [glCompressedTexImage3DOES](http://docs.gl/es3/glCompressedTexImage3DOES)(target, level, internalformat, width, height, depth, border, imageSize, data)
    /// * `target` group: TextureTarget
    /// * `internalformat` group: InternalFormat
    /// * `data` len: imageSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub unsafe fn glCompressedTexImage3DOES(
        target: GLenum,
        level: GLint,
        internalformat: GLenum,
        width: GLsizei,
        height: GLsizei,
        depth: GLsizei,
        border: GLint,
        imageSize: GLsizei,
        data: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCompressedTexImage3DOES({:#X}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:p});",
                target,
                level,
                internalformat,
                width,
                height,
                depth,
                border,
                imageSize,
                data
            );
        }
        let out = call_atomic_ptr_9arg(
            "glCompressedTexImage3DOES",
            &glCompressedTexImage3DOES_p,
            target,
            level,
            internalformat,
            width,
            height,
            depth,
            border,
            imageSize,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCompressedTexImage3DOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    static glCompressedTexImage3DOES_p: APcv = ap_null();
    /// Tries to load [`glCompressedTexImage3DOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub unsafe fn glCompressedTexImage3DOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCompressedTexImage3DOES\0",
            &glCompressedTexImage3DOES_p,
        )
    }
    /// Checks if the pointer for [`glCompressedTexImage3DOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub fn glCompressedTexImage3DOES_is_loaded() -> bool {
        !glCompressedTexImage3DOES_p.load(RELAX).is_null()
    }

    /// [glCompressedTexSubImage2D](http://docs.gl/es3/glCompressedTexSubImage2D)(target, level, xoffset, yoffset, width, height, format, imageSize, data)
    /// * `target` group: TextureTarget
    /// * `level` group: CheckedInt32
    /// * `xoffset` group: CheckedInt32
    /// * `yoffset` group: CheckedInt32
    /// * `format` group: PixelFormat
    /// * `data` group: CompressedTextureARB
    /// * `data` len: imageSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCompressedTexSubImage2D(
        target: GLenum,
        level: GLint,
        xoffset: GLint,
        yoffset: GLint,
        width: GLsizei,
        height: GLsizei,
        format: GLenum,
        imageSize: GLsizei,
        data: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCompressedTexSubImage2D({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:p});",
                target,
                level,
                xoffset,
                yoffset,
                width,
                height,
                format,
                imageSize,
                data
            );
        }
        let out = call_atomic_ptr_9arg(
            "glCompressedTexSubImage2D",
            &glCompressedTexSubImage2D_p,
            target,
            level,
            xoffset,
            yoffset,
            width,
            height,
            format,
            imageSize,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCompressedTexSubImage2D");
        }
        out
    }
    static glCompressedTexSubImage2D_p: APcv = ap_null();
    /// Tries to load [`glCompressedTexSubImage2D`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCompressedTexSubImage2D_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCompressedTexSubImage2D\0",
            &glCompressedTexSubImage2D_p,
        )
    }
    /// Checks if the pointer for [`glCompressedTexSubImage2D`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCompressedTexSubImage2D_is_loaded() -> bool {
        !glCompressedTexSubImage2D_p.load(RELAX).is_null()
    }

    /// [glCompressedTexSubImage3D](http://docs.gl/es3/glCompressedTexSubImage3D)(target, level, xoffset, yoffset, zoffset, width, height, depth, format, imageSize, data)
    /// * `target` group: TextureTarget
    /// * `level` group: CheckedInt32
    /// * `xoffset` group: CheckedInt32
    /// * `yoffset` group: CheckedInt32
    /// * `zoffset` group: CheckedInt32
    /// * `format` group: PixelFormat
    /// * `data` group: CompressedTextureARB
    /// * `data` len: imageSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCompressedTexSubImage3D(
        target: GLenum,
        level: GLint,
        xoffset: GLint,
        yoffset: GLint,
        zoffset: GLint,
        width: GLsizei,
        height: GLsizei,
        depth: GLsizei,
        format: GLenum,
        imageSize: GLsizei,
        data: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCompressedTexSubImage3D({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:p});", target, level, xoffset, yoffset, zoffset, width, height, depth, format, imageSize, data);
        }
        let out = call_atomic_ptr_11arg(
            "glCompressedTexSubImage3D",
            &glCompressedTexSubImage3D_p,
            target,
            level,
            xoffset,
            yoffset,
            zoffset,
            width,
            height,
            depth,
            format,
            imageSize,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCompressedTexSubImage3D");
        }
        out
    }
    static glCompressedTexSubImage3D_p: APcv = ap_null();
    /// Tries to load [`glCompressedTexSubImage3D`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCompressedTexSubImage3D_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCompressedTexSubImage3D\0",
            &glCompressedTexSubImage3D_p,
        )
    }
    /// Checks if the pointer for [`glCompressedTexSubImage3D`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCompressedTexSubImage3D_is_loaded() -> bool {
        !glCompressedTexSubImage3D_p.load(RELAX).is_null()
    }

    /// [glCompressedTexSubImage3DOES](http://docs.gl/es3/glCompressedTexSubImage3DOES)(target, level, xoffset, yoffset, zoffset, width, height, depth, format, imageSize, data)
    /// * `target` group: TextureTarget
    /// * `format` group: PixelFormat
    /// * `data` len: imageSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub unsafe fn glCompressedTexSubImage3DOES(
        target: GLenum,
        level: GLint,
        xoffset: GLint,
        yoffset: GLint,
        zoffset: GLint,
        width: GLsizei,
        height: GLsizei,
        depth: GLsizei,
        format: GLenum,
        imageSize: GLsizei,
        data: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCompressedTexSubImage3DOES({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:p});", target, level, xoffset, yoffset, zoffset, width, height, depth, format, imageSize, data);
        }
        let out = call_atomic_ptr_11arg(
            "glCompressedTexSubImage3DOES",
            &glCompressedTexSubImage3DOES_p,
            target,
            level,
            xoffset,
            yoffset,
            zoffset,
            width,
            height,
            depth,
            format,
            imageSize,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCompressedTexSubImage3DOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    static glCompressedTexSubImage3DOES_p: APcv = ap_null();
    /// Tries to load [`glCompressedTexSubImage3DOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub unsafe fn glCompressedTexSubImage3DOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCompressedTexSubImage3DOES\0",
            &glCompressedTexSubImage3DOES_p,
        )
    }
    /// Checks if the pointer for [`glCompressedTexSubImage3DOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub fn glCompressedTexSubImage3DOES_is_loaded() -> bool {
        !glCompressedTexSubImage3DOES_p.load(RELAX).is_null()
    }

    /// [glConservativeRasterParameteriNV](http://docs.gl/es3/glConservativeRasterParameteriNV)(pname, param)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles")))
    )]
    pub unsafe fn glConservativeRasterParameteriNV(pname: GLenum, param: GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glConservativeRasterParameteriNV({:#X}, {:?});",
                pname,
                param
            );
        }
        let out = call_atomic_ptr_2arg(
            "glConservativeRasterParameteriNV",
            &glConservativeRasterParameteriNV_p,
            pname,
            param,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glConservativeRasterParameteriNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles")))
    )]
    static glConservativeRasterParameteriNV_p: APcv = ap_null();
    /// Tries to load [`glConservativeRasterParameteriNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles")))
    )]
    pub unsafe fn glConservativeRasterParameteriNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glConservativeRasterParameteriNV\0",
            &glConservativeRasterParameteriNV_p,
        )
    }
    /// Checks if the pointer for [`glConservativeRasterParameteriNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_NV_conservative_raster_pre_snap_triangles")))
    )]
    pub fn glConservativeRasterParameteriNV_is_loaded() -> bool {
        !glConservativeRasterParameteriNV_p.load(RELAX).is_null()
    }

    /// [glCopyBufferSubData](http://docs.gl/es3/glCopyBufferSubData)(readTarget, writeTarget, readOffset, writeOffset, size)
    /// * `readTarget` group: CopyBufferSubDataTarget
    /// * `writeTarget` group: CopyBufferSubDataTarget
    /// * `readOffset` group: BufferOffset
    /// * `writeOffset` group: BufferOffset
    /// * `size` group: BufferSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCopyBufferSubData(
        readTarget: GLenum,
        writeTarget: GLenum,
        readOffset: GLintptr,
        writeOffset: GLintptr,
        size: GLsizeiptr,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCopyBufferSubData({:#X}, {:#X}, {:?}, {:?}, {:?});",
                readTarget,
                writeTarget,
                readOffset,
                writeOffset,
                size
            );
        }
        let out = call_atomic_ptr_5arg(
            "glCopyBufferSubData",
            &glCopyBufferSubData_p,
            readTarget,
            writeTarget,
            readOffset,
            writeOffset,
            size,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyBufferSubData");
        }
        out
    }
    static glCopyBufferSubData_p: APcv = ap_null();
    /// Tries to load [`glCopyBufferSubData`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCopyBufferSubData_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCopyBufferSubData\0",
            &glCopyBufferSubData_p,
        )
    }
    /// Checks if the pointer for [`glCopyBufferSubData`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCopyBufferSubData_is_loaded() -> bool {
        !glCopyBufferSubData_p.load(RELAX).is_null()
    }

    /// [glCopyBufferSubDataNV](http://docs.gl/es3/glCopyBufferSubDataNV)(readTarget, writeTarget, readOffset, writeOffset, size)
    /// * `readTarget` group: CopyBufferSubDataTarget
    /// * `writeTarget` group: CopyBufferSubDataTarget
    /// * `readOffset` group: BufferOffset
    /// * `writeOffset` group: BufferOffset
    /// * `size` group: BufferSize
    /// * alias of: [`glCopyBufferSubData`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_copy_buffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_copy_buffer"))))]
    pub unsafe fn glCopyBufferSubDataNV(
        readTarget: GLenum,
        writeTarget: GLenum,
        readOffset: GLintptr,
        writeOffset: GLintptr,
        size: GLsizeiptr,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCopyBufferSubDataNV({:#X}, {:#X}, {:?}, {:?}, {:?});",
                readTarget,
                writeTarget,
                readOffset,
                writeOffset,
                size
            );
        }
        let out = call_atomic_ptr_5arg(
            "glCopyBufferSubDataNV",
            &glCopyBufferSubDataNV_p,
            readTarget,
            writeTarget,
            readOffset,
            writeOffset,
            size,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyBufferSubDataNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_copy_buffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_copy_buffer"))))]
    static glCopyBufferSubDataNV_p: APcv = ap_null();
    /// Tries to load [`glCopyBufferSubDataNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_copy_buffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_copy_buffer"))))]
    pub unsafe fn glCopyBufferSubDataNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCopyBufferSubDataNV\0",
            &glCopyBufferSubDataNV_p,
        )
    }
    /// Checks if the pointer for [`glCopyBufferSubDataNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_copy_buffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_copy_buffer"))))]
    pub fn glCopyBufferSubDataNV_is_loaded() -> bool {
        !glCopyBufferSubDataNV_p.load(RELAX).is_null()
    }

    /// [glCopyImageSubData](http://docs.gl/es3/glCopyImageSubData)(srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth)
    /// * `srcTarget` group: CopyImageSubDataTarget
    /// * `dstTarget` group: CopyImageSubDataTarget
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCopyImageSubData(
        srcName: GLuint,
        srcTarget: GLenum,
        srcLevel: GLint,
        srcX: GLint,
        srcY: GLint,
        srcZ: GLint,
        dstName: GLuint,
        dstTarget: GLenum,
        dstLevel: GLint,
        dstX: GLint,
        dstY: GLint,
        dstZ: GLint,
        srcWidth: GLsizei,
        srcHeight: GLsizei,
        srcDepth: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCopyImageSubData({:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});", srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth);
        }
        let out = call_atomic_ptr_15arg(
            "glCopyImageSubData",
            &glCopyImageSubData_p,
            srcName,
            srcTarget,
            srcLevel,
            srcX,
            srcY,
            srcZ,
            dstName,
            dstTarget,
            dstLevel,
            dstX,
            dstY,
            dstZ,
            srcWidth,
            srcHeight,
            srcDepth,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyImageSubData");
        }
        out
    }
    static glCopyImageSubData_p: APcv = ap_null();
    /// Tries to load [`glCopyImageSubData`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCopyImageSubData_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCopyImageSubData\0",
            &glCopyImageSubData_p,
        )
    }
    /// Checks if the pointer for [`glCopyImageSubData`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCopyImageSubData_is_loaded() -> bool {
        !glCopyImageSubData_p.load(RELAX).is_null()
    }

    /// [glCopyImageSubDataEXT](http://docs.gl/es3/glCopyImageSubDataEXT)(srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth)
    /// * `srcTarget` group: CopyBufferSubDataTarget
    /// * `dstTarget` group: CopyBufferSubDataTarget
    /// * alias of: [`glCopyImageSubData`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_copy_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_copy_image"))))]
    pub unsafe fn glCopyImageSubDataEXT(
        srcName: GLuint,
        srcTarget: GLenum,
        srcLevel: GLint,
        srcX: GLint,
        srcY: GLint,
        srcZ: GLint,
        dstName: GLuint,
        dstTarget: GLenum,
        dstLevel: GLint,
        dstX: GLint,
        dstY: GLint,
        dstZ: GLint,
        srcWidth: GLsizei,
        srcHeight: GLsizei,
        srcDepth: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCopyImageSubDataEXT({:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});", srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth);
        }
        let out = call_atomic_ptr_15arg(
            "glCopyImageSubDataEXT",
            &glCopyImageSubDataEXT_p,
            srcName,
            srcTarget,
            srcLevel,
            srcX,
            srcY,
            srcZ,
            dstName,
            dstTarget,
            dstLevel,
            dstX,
            dstY,
            dstZ,
            srcWidth,
            srcHeight,
            srcDepth,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyImageSubDataEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_copy_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_copy_image"))))]
    static glCopyImageSubDataEXT_p: APcv = ap_null();
    /// Tries to load [`glCopyImageSubDataEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_copy_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_copy_image"))))]
    pub unsafe fn glCopyImageSubDataEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCopyImageSubDataEXT\0",
            &glCopyImageSubDataEXT_p,
        )
    }
    /// Checks if the pointer for [`glCopyImageSubDataEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_copy_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_copy_image"))))]
    pub fn glCopyImageSubDataEXT_is_loaded() -> bool {
        !glCopyImageSubDataEXT_p.load(RELAX).is_null()
    }

    /// [glCopyImageSubDataOES](http://docs.gl/es3/glCopyImageSubDataOES)(srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth)
    /// * `srcTarget` group: CopyBufferSubDataTarget
    /// * `dstTarget` group: CopyBufferSubDataTarget
    /// * alias of: [`glCopyImageSubData`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_copy_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_copy_image"))))]
    pub unsafe fn glCopyImageSubDataOES(
        srcName: GLuint,
        srcTarget: GLenum,
        srcLevel: GLint,
        srcX: GLint,
        srcY: GLint,
        srcZ: GLint,
        dstName: GLuint,
        dstTarget: GLenum,
        dstLevel: GLint,
        dstX: GLint,
        dstY: GLint,
        dstZ: GLint,
        srcWidth: GLsizei,
        srcHeight: GLsizei,
        srcDepth: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCopyImageSubDataOES({:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});", srcName, srcTarget, srcLevel, srcX, srcY, srcZ, dstName, dstTarget, dstLevel, dstX, dstY, dstZ, srcWidth, srcHeight, srcDepth);
        }
        let out = call_atomic_ptr_15arg(
            "glCopyImageSubDataOES",
            &glCopyImageSubDataOES_p,
            srcName,
            srcTarget,
            srcLevel,
            srcX,
            srcY,
            srcZ,
            dstName,
            dstTarget,
            dstLevel,
            dstX,
            dstY,
            dstZ,
            srcWidth,
            srcHeight,
            srcDepth,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyImageSubDataOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_copy_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_copy_image"))))]
    static glCopyImageSubDataOES_p: APcv = ap_null();
    /// Tries to load [`glCopyImageSubDataOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_copy_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_copy_image"))))]
    pub unsafe fn glCopyImageSubDataOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCopyImageSubDataOES\0",
            &glCopyImageSubDataOES_p,
        )
    }
    /// Checks if the pointer for [`glCopyImageSubDataOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_copy_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_copy_image"))))]
    pub fn glCopyImageSubDataOES_is_loaded() -> bool {
        !glCopyImageSubDataOES_p.load(RELAX).is_null()
    }

    /// [glCopyPathNV](http://docs.gl/es3/glCopyPathNV)(resultPath, srcPath)
    /// * `resultPath` group: Path
    /// * `srcPath` group: Path
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCopyPathNV(resultPath: GLuint, srcPath: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCopyPathNV({:?}, {:?});", resultPath, srcPath);
        }
        let out = call_atomic_ptr_2arg("glCopyPathNV", &glCopyPathNV_p, resultPath, srcPath);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyPathNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glCopyPathNV_p: APcv = ap_null();
    /// Tries to load [`glCopyPathNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCopyPathNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glCopyPathNV\0", &glCopyPathNV_p)
    }
    /// Checks if the pointer for [`glCopyPathNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glCopyPathNV_is_loaded() -> bool {
        !glCopyPathNV_p.load(RELAX).is_null()
    }

    /// [glCopyTexImage2D](http://docs.gl/es3/glCopyTexImage2D)(target, level, internalformat, x, y, width, height, border)
    /// * `target` group: TextureTarget
    /// * `level` group: CheckedInt32
    /// * `internalformat` group: InternalFormat
    /// * `x` group: WinCoord
    /// * `y` group: WinCoord
    /// * `border` group: CheckedInt32
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCopyTexImage2D(
        target: GLenum,
        level: GLint,
        internalformat: GLenum,
        x: GLint,
        y: GLint,
        width: GLsizei,
        height: GLsizei,
        border: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCopyTexImage2D({:#X}, {:?}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?});",
                target,
                level,
                internalformat,
                x,
                y,
                width,
                height,
                border
            );
        }
        let out = call_atomic_ptr_8arg(
            "glCopyTexImage2D",
            &glCopyTexImage2D_p,
            target,
            level,
            internalformat,
            x,
            y,
            width,
            height,
            border,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyTexImage2D");
        }
        out
    }
    static glCopyTexImage2D_p: APcv = ap_null();
    /// Tries to load [`glCopyTexImage2D`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCopyTexImage2D_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glCopyTexImage2D\0", &glCopyTexImage2D_p)
    }
    /// Checks if the pointer for [`glCopyTexImage2D`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCopyTexImage2D_is_loaded() -> bool {
        !glCopyTexImage2D_p.load(RELAX).is_null()
    }

    /// [glCopyTexSubImage2D](http://docs.gl/es3/glCopyTexSubImage2D)(target, level, xoffset, yoffset, x, y, width, height)
    /// * `target` group: TextureTarget
    /// * `level` group: CheckedInt32
    /// * `xoffset` group: CheckedInt32
    /// * `yoffset` group: CheckedInt32
    /// * `x` group: WinCoord
    /// * `y` group: WinCoord
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCopyTexSubImage2D(
        target: GLenum,
        level: GLint,
        xoffset: GLint,
        yoffset: GLint,
        x: GLint,
        y: GLint,
        width: GLsizei,
        height: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCopyTexSubImage2D({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                target,
                level,
                xoffset,
                yoffset,
                x,
                y,
                width,
                height
            );
        }
        let out = call_atomic_ptr_8arg(
            "glCopyTexSubImage2D",
            &glCopyTexSubImage2D_p,
            target,
            level,
            xoffset,
            yoffset,
            x,
            y,
            width,
            height,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyTexSubImage2D");
        }
        out
    }
    static glCopyTexSubImage2D_p: APcv = ap_null();
    /// Tries to load [`glCopyTexSubImage2D`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCopyTexSubImage2D_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCopyTexSubImage2D\0",
            &glCopyTexSubImage2D_p,
        )
    }
    /// Checks if the pointer for [`glCopyTexSubImage2D`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCopyTexSubImage2D_is_loaded() -> bool {
        !glCopyTexSubImage2D_p.load(RELAX).is_null()
    }

    /// [glCopyTexSubImage3D](http://docs.gl/es3/glCopyTexSubImage3D)(target, level, xoffset, yoffset, zoffset, x, y, width, height)
    /// * `target` group: TextureTarget
    /// * `level` group: CheckedInt32
    /// * `xoffset` group: CheckedInt32
    /// * `yoffset` group: CheckedInt32
    /// * `zoffset` group: CheckedInt32
    /// * `x` group: WinCoord
    /// * `y` group: WinCoord
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCopyTexSubImage3D(
        target: GLenum,
        level: GLint,
        xoffset: GLint,
        yoffset: GLint,
        zoffset: GLint,
        x: GLint,
        y: GLint,
        width: GLsizei,
        height: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCopyTexSubImage3D({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                target,
                level,
                xoffset,
                yoffset,
                zoffset,
                x,
                y,
                width,
                height
            );
        }
        let out = call_atomic_ptr_9arg(
            "glCopyTexSubImage3D",
            &glCopyTexSubImage3D_p,
            target,
            level,
            xoffset,
            yoffset,
            zoffset,
            x,
            y,
            width,
            height,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyTexSubImage3D");
        }
        out
    }
    static glCopyTexSubImage3D_p: APcv = ap_null();
    /// Tries to load [`glCopyTexSubImage3D`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCopyTexSubImage3D_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCopyTexSubImage3D\0",
            &glCopyTexSubImage3D_p,
        )
    }
    /// Checks if the pointer for [`glCopyTexSubImage3D`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCopyTexSubImage3D_is_loaded() -> bool {
        !glCopyTexSubImage3D_p.load(RELAX).is_null()
    }

    /// [glCopyTexSubImage3DOES](http://docs.gl/es3/glCopyTexSubImage3DOES)(target, level, xoffset, yoffset, zoffset, x, y, width, height)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub unsafe fn glCopyTexSubImage3DOES(
        target: GLenum,
        level: GLint,
        xoffset: GLint,
        yoffset: GLint,
        zoffset: GLint,
        x: GLint,
        y: GLint,
        width: GLsizei,
        height: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCopyTexSubImage3DOES({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                target,
                level,
                xoffset,
                yoffset,
                zoffset,
                x,
                y,
                width,
                height
            );
        }
        let out = call_atomic_ptr_9arg(
            "glCopyTexSubImage3DOES",
            &glCopyTexSubImage3DOES_p,
            target,
            level,
            xoffset,
            yoffset,
            zoffset,
            x,
            y,
            width,
            height,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyTexSubImage3DOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    static glCopyTexSubImage3DOES_p: APcv = ap_null();
    /// Tries to load [`glCopyTexSubImage3DOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub unsafe fn glCopyTexSubImage3DOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCopyTexSubImage3DOES\0",
            &glCopyTexSubImage3DOES_p,
        )
    }
    /// Checks if the pointer for [`glCopyTexSubImage3DOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub fn glCopyTexSubImage3DOES_is_loaded() -> bool {
        !glCopyTexSubImage3DOES_p.load(RELAX).is_null()
    }

    /// [glCopyTextureLevelsAPPLE](http://docs.gl/es3/glCopyTextureLevelsAPPLE)(destinationTexture, sourceTexture, sourceBaseLevel, sourceLevelCount)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_APPLE_copy_texture_levels"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_copy_texture_levels"))))]
    pub unsafe fn glCopyTextureLevelsAPPLE(
        destinationTexture: GLuint,
        sourceTexture: GLuint,
        sourceBaseLevel: GLint,
        sourceLevelCount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCopyTextureLevelsAPPLE({:?}, {:?}, {:?}, {:?});",
                destinationTexture,
                sourceTexture,
                sourceBaseLevel,
                sourceLevelCount
            );
        }
        let out = call_atomic_ptr_4arg(
            "glCopyTextureLevelsAPPLE",
            &glCopyTextureLevelsAPPLE_p,
            destinationTexture,
            sourceTexture,
            sourceBaseLevel,
            sourceLevelCount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCopyTextureLevelsAPPLE");
        }
        out
    }
    #[cfg(any(feature = "GL_APPLE_copy_texture_levels"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_copy_texture_levels"))))]
    static glCopyTextureLevelsAPPLE_p: APcv = ap_null();
    /// Tries to load [`glCopyTextureLevelsAPPLE`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_copy_texture_levels"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_copy_texture_levels"))))]
    pub unsafe fn glCopyTextureLevelsAPPLE_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCopyTextureLevelsAPPLE\0",
            &glCopyTextureLevelsAPPLE_p,
        )
    }
    /// Checks if the pointer for [`glCopyTextureLevelsAPPLE`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_copy_texture_levels"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_copy_texture_levels"))))]
    pub fn glCopyTextureLevelsAPPLE_is_loaded() -> bool {
        !glCopyTextureLevelsAPPLE_p.load(RELAX).is_null()
    }

    /// [glCoverFillPathInstancedNV](http://docs.gl/es3/glCoverFillPathInstancedNV)(numPaths, pathNameType, paths, pathBase, coverMode, transformType, transformValues)
    /// * `pathNameType` group: PathElementType
    /// * `paths` group: PathElement
    /// * `paths` len: COMPSIZE(numPaths,pathNameType,paths)
    /// * `pathBase` group: Path
    /// * `coverMode` group: PathCoverMode
    /// * `transformType` group: PathTransformType
    /// * `transformValues` len: COMPSIZE(numPaths,transformType)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCoverFillPathInstancedNV(
        numPaths: GLsizei,
        pathNameType: GLenum,
        paths: *const c_void,
        pathBase: GLuint,
        coverMode: GLenum,
        transformType: GLenum,
        transformValues: *const GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCoverFillPathInstancedNV({:?}, {:#X}, {:p}, {:?}, {:#X}, {:#X}, {:p});",
                numPaths,
                pathNameType,
                paths,
                pathBase,
                coverMode,
                transformType,
                transformValues
            );
        }
        let out = call_atomic_ptr_7arg(
            "glCoverFillPathInstancedNV",
            &glCoverFillPathInstancedNV_p,
            numPaths,
            pathNameType,
            paths,
            pathBase,
            coverMode,
            transformType,
            transformValues,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCoverFillPathInstancedNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glCoverFillPathInstancedNV_p: APcv = ap_null();
    /// Tries to load [`glCoverFillPathInstancedNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCoverFillPathInstancedNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCoverFillPathInstancedNV\0",
            &glCoverFillPathInstancedNV_p,
        )
    }
    /// Checks if the pointer for [`glCoverFillPathInstancedNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glCoverFillPathInstancedNV_is_loaded() -> bool {
        !glCoverFillPathInstancedNV_p.load(RELAX).is_null()
    }

    /// [glCoverFillPathNV](http://docs.gl/es3/glCoverFillPathNV)(path, coverMode)
    /// * `path` group: Path
    /// * `coverMode` group: PathCoverMode
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCoverFillPathNV(path: GLuint, coverMode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCoverFillPathNV({:?}, {:#X});", path, coverMode);
        }
        let out = call_atomic_ptr_2arg("glCoverFillPathNV", &glCoverFillPathNV_p, path, coverMode);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCoverFillPathNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glCoverFillPathNV_p: APcv = ap_null();
    /// Tries to load [`glCoverFillPathNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCoverFillPathNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCoverFillPathNV\0",
            &glCoverFillPathNV_p,
        )
    }
    /// Checks if the pointer for [`glCoverFillPathNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glCoverFillPathNV_is_loaded() -> bool {
        !glCoverFillPathNV_p.load(RELAX).is_null()
    }

    /// [glCoverStrokePathInstancedNV](http://docs.gl/es3/glCoverStrokePathInstancedNV)(numPaths, pathNameType, paths, pathBase, coverMode, transformType, transformValues)
    /// * `pathNameType` group: PathElementType
    /// * `paths` group: PathElement
    /// * `paths` len: COMPSIZE(numPaths,pathNameType,paths)
    /// * `pathBase` group: Path
    /// * `coverMode` group: PathCoverMode
    /// * `transformType` group: PathTransformType
    /// * `transformValues` len: COMPSIZE(numPaths,transformType)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCoverStrokePathInstancedNV(
        numPaths: GLsizei,
        pathNameType: GLenum,
        paths: *const c_void,
        pathBase: GLuint,
        coverMode: GLenum,
        transformType: GLenum,
        transformValues: *const GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCoverStrokePathInstancedNV({:?}, {:#X}, {:p}, {:?}, {:#X}, {:#X}, {:p});",
                numPaths,
                pathNameType,
                paths,
                pathBase,
                coverMode,
                transformType,
                transformValues
            );
        }
        let out = call_atomic_ptr_7arg(
            "glCoverStrokePathInstancedNV",
            &glCoverStrokePathInstancedNV_p,
            numPaths,
            pathNameType,
            paths,
            pathBase,
            coverMode,
            transformType,
            transformValues,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCoverStrokePathInstancedNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glCoverStrokePathInstancedNV_p: APcv = ap_null();
    /// Tries to load [`glCoverStrokePathInstancedNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCoverStrokePathInstancedNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCoverStrokePathInstancedNV\0",
            &glCoverStrokePathInstancedNV_p,
        )
    }
    /// Checks if the pointer for [`glCoverStrokePathInstancedNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glCoverStrokePathInstancedNV_is_loaded() -> bool {
        !glCoverStrokePathInstancedNV_p.load(RELAX).is_null()
    }

    /// [glCoverStrokePathNV](http://docs.gl/es3/glCoverStrokePathNV)(path, coverMode)
    /// * `path` group: Path
    /// * `coverMode` group: PathCoverMode
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCoverStrokePathNV(path: GLuint, coverMode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCoverStrokePathNV({:?}, {:#X});", path, coverMode);
        }
        let out = call_atomic_ptr_2arg(
            "glCoverStrokePathNV",
            &glCoverStrokePathNV_p,
            path,
            coverMode,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCoverStrokePathNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glCoverStrokePathNV_p: APcv = ap_null();
    /// Tries to load [`glCoverStrokePathNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glCoverStrokePathNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCoverStrokePathNV\0",
            &glCoverStrokePathNV_p,
        )
    }
    /// Checks if the pointer for [`glCoverStrokePathNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glCoverStrokePathNV_is_loaded() -> bool {
        !glCoverStrokePathNV_p.load(RELAX).is_null()
    }

    /// [glCoverageMaskNV](http://docs.gl/es3/glCoverageMaskNV)(mask)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_coverage_sample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
    pub unsafe fn glCoverageMaskNV(mask: GLboolean) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCoverageMaskNV({:?});", mask);
        }
        let out = call_atomic_ptr_1arg("glCoverageMaskNV", &glCoverageMaskNV_p, mask);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCoverageMaskNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_coverage_sample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
    static glCoverageMaskNV_p: APcv = ap_null();
    /// Tries to load [`glCoverageMaskNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_coverage_sample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
    pub unsafe fn glCoverageMaskNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glCoverageMaskNV\0", &glCoverageMaskNV_p)
    }
    /// Checks if the pointer for [`glCoverageMaskNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_coverage_sample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
    pub fn glCoverageMaskNV_is_loaded() -> bool {
        !glCoverageMaskNV_p.load(RELAX).is_null()
    }

    /// [glCoverageModulationNV](http://docs.gl/es3/glCoverageModulationNV)(components)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    pub unsafe fn glCoverageModulationNV(components: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCoverageModulationNV({:#X});", components);
        }
        let out = call_atomic_ptr_1arg(
            "glCoverageModulationNV",
            &glCoverageModulationNV_p,
            components,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCoverageModulationNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    static glCoverageModulationNV_p: APcv = ap_null();
    /// Tries to load [`glCoverageModulationNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    pub unsafe fn glCoverageModulationNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCoverageModulationNV\0",
            &glCoverageModulationNV_p,
        )
    }
    /// Checks if the pointer for [`glCoverageModulationNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    pub fn glCoverageModulationNV_is_loaded() -> bool {
        !glCoverageModulationNV_p.load(RELAX).is_null()
    }

    /// [glCoverageModulationTableNV](http://docs.gl/es3/glCoverageModulationTableNV)(n, v)
    /// * `v` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    pub unsafe fn glCoverageModulationTableNV(n: GLsizei, v: *const GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCoverageModulationTableNV({:?}, {:p});", n, v);
        }
        let out = call_atomic_ptr_2arg(
            "glCoverageModulationTableNV",
            &glCoverageModulationTableNV_p,
            n,
            v,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCoverageModulationTableNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    static glCoverageModulationTableNV_p: APcv = ap_null();
    /// Tries to load [`glCoverageModulationTableNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    pub unsafe fn glCoverageModulationTableNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCoverageModulationTableNV\0",
            &glCoverageModulationTableNV_p,
        )
    }
    /// Checks if the pointer for [`glCoverageModulationTableNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    pub fn glCoverageModulationTableNV_is_loaded() -> bool {
        !glCoverageModulationTableNV_p.load(RELAX).is_null()
    }

    /// [glCoverageOperationNV](http://docs.gl/es3/glCoverageOperationNV)(operation)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_coverage_sample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
    pub unsafe fn glCoverageOperationNV(operation: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCoverageOperationNV({:#X});", operation);
        }
        let out =
            call_atomic_ptr_1arg("glCoverageOperationNV", &glCoverageOperationNV_p, operation);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCoverageOperationNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_coverage_sample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
    static glCoverageOperationNV_p: APcv = ap_null();
    /// Tries to load [`glCoverageOperationNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_coverage_sample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
    pub unsafe fn glCoverageOperationNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCoverageOperationNV\0",
            &glCoverageOperationNV_p,
        )
    }
    /// Checks if the pointer for [`glCoverageOperationNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_coverage_sample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_coverage_sample"))))]
    pub fn glCoverageOperationNV_is_loaded() -> bool {
        !glCoverageOperationNV_p.load(RELAX).is_null()
    }

    /// [glCreateMemoryObjectsEXT](http://docs.gl/es3/glCreateMemoryObjectsEXT)(n, memoryObjects)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub unsafe fn glCreateMemoryObjectsEXT(n: GLsizei, memoryObjects: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCreateMemoryObjectsEXT({:?}, {:p});",
                n,
                memoryObjects
            );
        }
        let out = call_atomic_ptr_2arg(
            "glCreateMemoryObjectsEXT",
            &glCreateMemoryObjectsEXT_p,
            n,
            memoryObjects,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCreateMemoryObjectsEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    static glCreateMemoryObjectsEXT_p: APcv = ap_null();
    /// Tries to load [`glCreateMemoryObjectsEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub unsafe fn glCreateMemoryObjectsEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCreateMemoryObjectsEXT\0",
            &glCreateMemoryObjectsEXT_p,
        )
    }
    /// Checks if the pointer for [`glCreateMemoryObjectsEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub fn glCreateMemoryObjectsEXT_is_loaded() -> bool {
        !glCreateMemoryObjectsEXT_p.load(RELAX).is_null()
    }

    /// [glCreatePerfQueryINTEL](http://docs.gl/es3/glCreatePerfQueryINTEL)(queryId, queryHandle)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glCreatePerfQueryINTEL(queryId: GLuint, queryHandle: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCreatePerfQueryINTEL({:?}, {:p});",
                queryId,
                queryHandle
            );
        }
        let out = call_atomic_ptr_2arg(
            "glCreatePerfQueryINTEL",
            &glCreatePerfQueryINTEL_p,
            queryId,
            queryHandle,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCreatePerfQueryINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glCreatePerfQueryINTEL_p: APcv = ap_null();
    /// Tries to load [`glCreatePerfQueryINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glCreatePerfQueryINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCreatePerfQueryINTEL\0",
            &glCreatePerfQueryINTEL_p,
        )
    }
    /// Checks if the pointer for [`glCreatePerfQueryINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glCreatePerfQueryINTEL_is_loaded() -> bool {
        !glCreatePerfQueryINTEL_p.load(RELAX).is_null()
    }

    /// [glCreateProgram](http://docs.gl/es3/glCreateProgram)()
    /// * return value class: program
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCreateProgram() -> GLuint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCreateProgram();",);
        }
        let out = call_atomic_ptr_0arg("glCreateProgram", &glCreateProgram_p);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCreateProgram");
        }
        out
    }
    static glCreateProgram_p: APcv = ap_null();
    /// Tries to load [`glCreateProgram`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCreateProgram_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glCreateProgram\0", &glCreateProgram_p)
    }
    /// Checks if the pointer for [`glCreateProgram`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCreateProgram_is_loaded() -> bool {
        !glCreateProgram_p.load(RELAX).is_null()
    }

    /// [glCreateSemaphoresNV](http://docs.gl/es3/glCreateSemaphoresNV)(n, semaphores)
    /// * `semaphores` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
    pub unsafe fn glCreateSemaphoresNV(n: GLsizei, semaphores: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCreateSemaphoresNV({:?}, {:p});", n, semaphores);
        }
        let out = call_atomic_ptr_2arg(
            "glCreateSemaphoresNV",
            &glCreateSemaphoresNV_p,
            n,
            semaphores,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCreateSemaphoresNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
    static glCreateSemaphoresNV_p: APcv = ap_null();
    /// Tries to load [`glCreateSemaphoresNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
    pub unsafe fn glCreateSemaphoresNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCreateSemaphoresNV\0",
            &glCreateSemaphoresNV_p,
        )
    }
    /// Checks if the pointer for [`glCreateSemaphoresNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
    pub fn glCreateSemaphoresNV_is_loaded() -> bool {
        !glCreateSemaphoresNV_p.load(RELAX).is_null()
    }

    /// [glCreateShader](http://docs.gl/es3/glCreateShader)(type_)
    /// * `type_` group: ShaderType
    /// * return value class: shader
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCreateShader(type_: GLenum) -> GLuint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCreateShader({:#X});", type_);
        }
        let out = call_atomic_ptr_1arg("glCreateShader", &glCreateShader_p, type_);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCreateShader");
        }
        out
    }
    static glCreateShader_p: APcv = ap_null();
    /// Tries to load [`glCreateShader`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCreateShader_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glCreateShader\0", &glCreateShader_p)
    }
    /// Checks if the pointer for [`glCreateShader`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCreateShader_is_loaded() -> bool {
        !glCreateShader_p.load(RELAX).is_null()
    }

    /// [glCreateShaderProgramv](http://docs.gl/es3/glCreateShaderProgramv)(type_, count, strings)
    /// * `type_` group: ShaderType
    /// * `strings` len: count
    /// * return value class: program
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCreateShaderProgramv(
        type_: GLenum,
        count: GLsizei,
        strings: *const *const GLchar,
    ) -> GLuint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCreateShaderProgramv({:#X}, {:?}, {:p});",
                type_,
                count,
                strings
            );
        }
        let out = call_atomic_ptr_3arg(
            "glCreateShaderProgramv",
            &glCreateShaderProgramv_p,
            type_,
            count,
            strings,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCreateShaderProgramv");
        }
        out
    }
    static glCreateShaderProgramv_p: APcv = ap_null();
    /// Tries to load [`glCreateShaderProgramv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCreateShaderProgramv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCreateShaderProgramv\0",
            &glCreateShaderProgramv_p,
        )
    }
    /// Checks if the pointer for [`glCreateShaderProgramv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCreateShaderProgramv_is_loaded() -> bool {
        !glCreateShaderProgramv_p.load(RELAX).is_null()
    }

    /// [glCreateShaderProgramvEXT](http://docs.gl/es3/glCreateShaderProgramvEXT)(type_, count, strings)
    /// * `type_` group: ShaderType
    /// * `strings` len: count
    /// * return value class: program
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glCreateShaderProgramvEXT(
        type_: GLenum,
        count: GLsizei,
        strings: *const GLchar,
    ) -> GLuint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glCreateShaderProgramvEXT({:#X}, {:?}, {:p});",
                type_,
                count,
                strings
            );
        }
        let out = call_atomic_ptr_3arg(
            "glCreateShaderProgramvEXT",
            &glCreateShaderProgramvEXT_p,
            type_,
            count,
            strings,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCreateShaderProgramvEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    static glCreateShaderProgramvEXT_p: APcv = ap_null();
    /// Tries to load [`glCreateShaderProgramvEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glCreateShaderProgramvEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glCreateShaderProgramvEXT\0",
            &glCreateShaderProgramvEXT_p,
        )
    }
    /// Checks if the pointer for [`glCreateShaderProgramvEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub fn glCreateShaderProgramvEXT_is_loaded() -> bool {
        !glCreateShaderProgramvEXT_p.load(RELAX).is_null()
    }

    /// [glCullFace](http://docs.gl/es3/glCullFace)(mode)
    /// * `mode` group: CullFaceMode
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glCullFace(mode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glCullFace({:#X});", mode);
        }
        let out = call_atomic_ptr_1arg("glCullFace", &glCullFace_p, mode);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glCullFace");
        }
        out
    }
    static glCullFace_p: APcv = ap_null();
    /// Tries to load [`glCullFace`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glCullFace_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glCullFace\0", &glCullFace_p)
    }
    /// Checks if the pointer for [`glCullFace`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glCullFace_is_loaded() -> bool {
        !glCullFace_p.load(RELAX).is_null()
    }

    /// [glDebugMessageCallback](http://docs.gl/es3/glDebugMessageCallback)(callback, userParam)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDebugMessageCallback(callback: GLDEBUGPROC, userParam: *const c_void) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDebugMessageCallback({:?}, {:p});",
                transmute::<_, Option<fn()>>(callback),
                userParam
            );
        }
        let out = call_atomic_ptr_2arg(
            "glDebugMessageCallback",
            &glDebugMessageCallback_p,
            callback,
            userParam,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDebugMessageCallback");
        }
        out
    }
    static glDebugMessageCallback_p: APcv = ap_null();
    /// Tries to load [`glDebugMessageCallback`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDebugMessageCallback_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDebugMessageCallback\0",
            &glDebugMessageCallback_p,
        )
    }
    /// Checks if the pointer for [`glDebugMessageCallback`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDebugMessageCallback_is_loaded() -> bool {
        !glDebugMessageCallback_p.load(RELAX).is_null()
    }

    /// [glDebugMessageCallbackKHR](http://docs.gl/es3/glDebugMessageCallbackKHR)(callback, userParam)
    /// * alias of: [`glDebugMessageCallback`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glDebugMessageCallbackKHR(callback: GLDEBUGPROCKHR, userParam: *const c_void) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDebugMessageCallbackKHR({:?}, {:p});",
                transmute::<_, Option<fn()>>(callback),
                userParam
            );
        }
        let out = call_atomic_ptr_2arg(
            "glDebugMessageCallbackKHR",
            &glDebugMessageCallbackKHR_p,
            callback,
            userParam,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDebugMessageCallbackKHR");
        }
        out
    }
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    static glDebugMessageCallbackKHR_p: APcv = ap_null();
    /// Tries to load [`glDebugMessageCallbackKHR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glDebugMessageCallbackKHR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDebugMessageCallbackKHR\0",
            &glDebugMessageCallbackKHR_p,
        )
    }
    /// Checks if the pointer for [`glDebugMessageCallbackKHR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub fn glDebugMessageCallbackKHR_is_loaded() -> bool {
        !glDebugMessageCallbackKHR_p.load(RELAX).is_null()
    }

    /// [glDebugMessageControl](http://docs.gl/es3/glDebugMessageControl)(source, type_, severity, count, ids, enabled)
    /// * `source` group: DebugSource
    /// * `type_` group: DebugType
    /// * `severity` group: DebugSeverity
    /// * `ids` len: count
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDebugMessageControl(
        source: GLenum,
        type_: GLenum,
        severity: GLenum,
        count: GLsizei,
        ids: *const GLuint,
        enabled: GLboolean,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDebugMessageControl({:#X}, {:#X}, {:#X}, {:?}, {:p}, {:?});",
                source,
                type_,
                severity,
                count,
                ids,
                enabled
            );
        }
        let out = call_atomic_ptr_6arg(
            "glDebugMessageControl",
            &glDebugMessageControl_p,
            source,
            type_,
            severity,
            count,
            ids,
            enabled,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDebugMessageControl");
        }
        out
    }
    static glDebugMessageControl_p: APcv = ap_null();
    /// Tries to load [`glDebugMessageControl`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDebugMessageControl_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDebugMessageControl\0",
            &glDebugMessageControl_p,
        )
    }
    /// Checks if the pointer for [`glDebugMessageControl`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDebugMessageControl_is_loaded() -> bool {
        !glDebugMessageControl_p.load(RELAX).is_null()
    }

    /// [glDebugMessageControlKHR](http://docs.gl/es3/glDebugMessageControlKHR)(source, type_, severity, count, ids, enabled)
    /// * `source` group: DebugSource
    /// * `type_` group: DebugType
    /// * `severity` group: DebugSeverity
    /// * alias of: [`glDebugMessageControl`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glDebugMessageControlKHR(
        source: GLenum,
        type_: GLenum,
        severity: GLenum,
        count: GLsizei,
        ids: *const GLuint,
        enabled: GLboolean,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDebugMessageControlKHR({:#X}, {:#X}, {:#X}, {:?}, {:p}, {:?});",
                source,
                type_,
                severity,
                count,
                ids,
                enabled
            );
        }
        let out = call_atomic_ptr_6arg(
            "glDebugMessageControlKHR",
            &glDebugMessageControlKHR_p,
            source,
            type_,
            severity,
            count,
            ids,
            enabled,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDebugMessageControlKHR");
        }
        out
    }
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    static glDebugMessageControlKHR_p: APcv = ap_null();
    /// Tries to load [`glDebugMessageControlKHR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glDebugMessageControlKHR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDebugMessageControlKHR\0",
            &glDebugMessageControlKHR_p,
        )
    }
    /// Checks if the pointer for [`glDebugMessageControlKHR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub fn glDebugMessageControlKHR_is_loaded() -> bool {
        !glDebugMessageControlKHR_p.load(RELAX).is_null()
    }

    /// [glDebugMessageInsert](http://docs.gl/es3/glDebugMessageInsert)(source, type_, id, severity, length, buf)
    /// * `source` group: DebugSource
    /// * `type_` group: DebugType
    /// * `severity` group: DebugSeverity
    /// * `buf` len: COMPSIZE(buf,length)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDebugMessageInsert(
        source: GLenum,
        type_: GLenum,
        id: GLuint,
        severity: GLenum,
        length: GLsizei,
        buf: *const GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDebugMessageInsert({:#X}, {:#X}, {:?}, {:#X}, {:?}, {:p});",
                source,
                type_,
                id,
                severity,
                length,
                buf
            );
        }
        let out = call_atomic_ptr_6arg(
            "glDebugMessageInsert",
            &glDebugMessageInsert_p,
            source,
            type_,
            id,
            severity,
            length,
            buf,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDebugMessageInsert");
        }
        out
    }
    static glDebugMessageInsert_p: APcv = ap_null();
    /// Tries to load [`glDebugMessageInsert`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDebugMessageInsert_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDebugMessageInsert\0",
            &glDebugMessageInsert_p,
        )
    }
    /// Checks if the pointer for [`glDebugMessageInsert`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDebugMessageInsert_is_loaded() -> bool {
        !glDebugMessageInsert_p.load(RELAX).is_null()
    }

    /// [glDebugMessageInsertKHR](http://docs.gl/es3/glDebugMessageInsertKHR)(source, type_, id, severity, length, buf)
    /// * `source` group: DebugSource
    /// * `type_` group: DebugType
    /// * `severity` group: DebugSeverity
    /// * alias of: [`glDebugMessageInsert`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glDebugMessageInsertKHR(
        source: GLenum,
        type_: GLenum,
        id: GLuint,
        severity: GLenum,
        length: GLsizei,
        buf: *const GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDebugMessageInsertKHR({:#X}, {:#X}, {:?}, {:#X}, {:?}, {:p});",
                source,
                type_,
                id,
                severity,
                length,
                buf
            );
        }
        let out = call_atomic_ptr_6arg(
            "glDebugMessageInsertKHR",
            &glDebugMessageInsertKHR_p,
            source,
            type_,
            id,
            severity,
            length,
            buf,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDebugMessageInsertKHR");
        }
        out
    }
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    static glDebugMessageInsertKHR_p: APcv = ap_null();
    /// Tries to load [`glDebugMessageInsertKHR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glDebugMessageInsertKHR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDebugMessageInsertKHR\0",
            &glDebugMessageInsertKHR_p,
        )
    }
    /// Checks if the pointer for [`glDebugMessageInsertKHR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub fn glDebugMessageInsertKHR_is_loaded() -> bool {
        !glDebugMessageInsertKHR_p.load(RELAX).is_null()
    }

    /// [glDeleteBuffers](http://docs.gl/es3/glDeleteBuffers)(n, buffers)
    /// * `buffers` len: n
    /// * `buffers` class: buffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteBuffers(n: GLsizei, buffers: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteBuffers({:?}, {:p});", n, buffers);
        }
        let out = call_atomic_ptr_2arg("glDeleteBuffers", &glDeleteBuffers_p, n, buffers);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteBuffers");
        }
        out
    }
    static glDeleteBuffers_p: APcv = ap_null();
    /// Tries to load [`glDeleteBuffers`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteBuffers_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDeleteBuffers\0", &glDeleteBuffers_p)
    }
    /// Checks if the pointer for [`glDeleteBuffers`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteBuffers_is_loaded() -> bool {
        !glDeleteBuffers_p.load(RELAX).is_null()
    }

    /// [glDeleteFencesNV](http://docs.gl/es3/glDeleteFencesNV)(n, fences)
    /// * `fences` group: FenceNV
    /// * `fences` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub unsafe fn glDeleteFencesNV(n: GLsizei, fences: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteFencesNV({:?}, {:p});", n, fences);
        }
        let out = call_atomic_ptr_2arg("glDeleteFencesNV", &glDeleteFencesNV_p, n, fences);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteFencesNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    static glDeleteFencesNV_p: APcv = ap_null();
    /// Tries to load [`glDeleteFencesNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub unsafe fn glDeleteFencesNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDeleteFencesNV\0", &glDeleteFencesNV_p)
    }
    /// Checks if the pointer for [`glDeleteFencesNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub fn glDeleteFencesNV_is_loaded() -> bool {
        !glDeleteFencesNV_p.load(RELAX).is_null()
    }

    /// [glDeleteFramebuffers](http://docs.gl/es3/glDeleteFramebuffers)(n, framebuffers)
    /// * `framebuffers` len: n
    /// * `framebuffers` class: framebuffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteFramebuffers(n: GLsizei, framebuffers: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteFramebuffers({:?}, {:p});", n, framebuffers);
        }
        let out = call_atomic_ptr_2arg(
            "glDeleteFramebuffers",
            &glDeleteFramebuffers_p,
            n,
            framebuffers,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteFramebuffers");
        }
        out
    }
    static glDeleteFramebuffers_p: APcv = ap_null();
    /// Tries to load [`glDeleteFramebuffers`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteFramebuffers_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteFramebuffers\0",
            &glDeleteFramebuffers_p,
        )
    }
    /// Checks if the pointer for [`glDeleteFramebuffers`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteFramebuffers_is_loaded() -> bool {
        !glDeleteFramebuffers_p.load(RELAX).is_null()
    }

    /// [glDeleteMemoryObjectsEXT](http://docs.gl/es3/glDeleteMemoryObjectsEXT)(n, memoryObjects)
    /// * `memoryObjects` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub unsafe fn glDeleteMemoryObjectsEXT(n: GLsizei, memoryObjects: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDeleteMemoryObjectsEXT({:?}, {:p});",
                n,
                memoryObjects
            );
        }
        let out = call_atomic_ptr_2arg(
            "glDeleteMemoryObjectsEXT",
            &glDeleteMemoryObjectsEXT_p,
            n,
            memoryObjects,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteMemoryObjectsEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    static glDeleteMemoryObjectsEXT_p: APcv = ap_null();
    /// Tries to load [`glDeleteMemoryObjectsEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub unsafe fn glDeleteMemoryObjectsEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteMemoryObjectsEXT\0",
            &glDeleteMemoryObjectsEXT_p,
        )
    }
    /// Checks if the pointer for [`glDeleteMemoryObjectsEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub fn glDeleteMemoryObjectsEXT_is_loaded() -> bool {
        !glDeleteMemoryObjectsEXT_p.load(RELAX).is_null()
    }

    /// [glDeletePathsNV](http://docs.gl/es3/glDeletePathsNV)(path, range)
    /// * `path` group: Path
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glDeletePathsNV(path: GLuint, range: GLsizei) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeletePathsNV({:?}, {:?});", path, range);
        }
        let out = call_atomic_ptr_2arg("glDeletePathsNV", &glDeletePathsNV_p, path, range);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeletePathsNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glDeletePathsNV_p: APcv = ap_null();
    /// Tries to load [`glDeletePathsNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glDeletePathsNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDeletePathsNV\0", &glDeletePathsNV_p)
    }
    /// Checks if the pointer for [`glDeletePathsNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glDeletePathsNV_is_loaded() -> bool {
        !glDeletePathsNV_p.load(RELAX).is_null()
    }

    /// [glDeletePerfMonitorsAMD](http://docs.gl/es3/glDeletePerfMonitorsAMD)(n, monitors)
    /// * `monitors` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glDeletePerfMonitorsAMD(n: GLsizei, monitors: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeletePerfMonitorsAMD({:?}, {:p});", n, monitors);
        }
        let out = call_atomic_ptr_2arg(
            "glDeletePerfMonitorsAMD",
            &glDeletePerfMonitorsAMD_p,
            n,
            monitors,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeletePerfMonitorsAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glDeletePerfMonitorsAMD_p: APcv = ap_null();
    /// Tries to load [`glDeletePerfMonitorsAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glDeletePerfMonitorsAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeletePerfMonitorsAMD\0",
            &glDeletePerfMonitorsAMD_p,
        )
    }
    /// Checks if the pointer for [`glDeletePerfMonitorsAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glDeletePerfMonitorsAMD_is_loaded() -> bool {
        !glDeletePerfMonitorsAMD_p.load(RELAX).is_null()
    }

    /// [glDeletePerfQueryINTEL](http://docs.gl/es3/glDeletePerfQueryINTEL)(queryHandle)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glDeletePerfQueryINTEL(queryHandle: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeletePerfQueryINTEL({:?});", queryHandle);
        }
        let out = call_atomic_ptr_1arg(
            "glDeletePerfQueryINTEL",
            &glDeletePerfQueryINTEL_p,
            queryHandle,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeletePerfQueryINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glDeletePerfQueryINTEL_p: APcv = ap_null();
    /// Tries to load [`glDeletePerfQueryINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glDeletePerfQueryINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeletePerfQueryINTEL\0",
            &glDeletePerfQueryINTEL_p,
        )
    }
    /// Checks if the pointer for [`glDeletePerfQueryINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glDeletePerfQueryINTEL_is_loaded() -> bool {
        !glDeletePerfQueryINTEL_p.load(RELAX).is_null()
    }

    /// [glDeleteProgram](http://docs.gl/es3/glDeleteProgram)(program)
    /// * `program` class: program
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteProgram(program: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteProgram({:?});", program);
        }
        let out = call_atomic_ptr_1arg("glDeleteProgram", &glDeleteProgram_p, program);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteProgram");
        }
        out
    }
    static glDeleteProgram_p: APcv = ap_null();
    /// Tries to load [`glDeleteProgram`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteProgram_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDeleteProgram\0", &glDeleteProgram_p)
    }
    /// Checks if the pointer for [`glDeleteProgram`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteProgram_is_loaded() -> bool {
        !glDeleteProgram_p.load(RELAX).is_null()
    }

    /// [glDeleteProgramPipelines](http://docs.gl/es3/glDeleteProgramPipelines)(n, pipelines)
    /// * `pipelines` len: n
    /// * `pipelines` class: program pipeline
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteProgramPipelines(n: GLsizei, pipelines: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDeleteProgramPipelines({:?}, {:p});",
                n,
                pipelines
            );
        }
        let out = call_atomic_ptr_2arg(
            "glDeleteProgramPipelines",
            &glDeleteProgramPipelines_p,
            n,
            pipelines,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteProgramPipelines");
        }
        out
    }
    static glDeleteProgramPipelines_p: APcv = ap_null();
    /// Tries to load [`glDeleteProgramPipelines`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteProgramPipelines_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteProgramPipelines\0",
            &glDeleteProgramPipelines_p,
        )
    }
    /// Checks if the pointer for [`glDeleteProgramPipelines`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteProgramPipelines_is_loaded() -> bool {
        !glDeleteProgramPipelines_p.load(RELAX).is_null()
    }

    /// [glDeleteProgramPipelinesEXT](http://docs.gl/es3/glDeleteProgramPipelinesEXT)(n, pipelines)
    /// * `pipelines` len: n
    /// * `pipelines` class: program pipeline
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glDeleteProgramPipelinesEXT(n: GLsizei, pipelines: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDeleteProgramPipelinesEXT({:?}, {:p});",
                n,
                pipelines
            );
        }
        let out = call_atomic_ptr_2arg(
            "glDeleteProgramPipelinesEXT",
            &glDeleteProgramPipelinesEXT_p,
            n,
            pipelines,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteProgramPipelinesEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    static glDeleteProgramPipelinesEXT_p: APcv = ap_null();
    /// Tries to load [`glDeleteProgramPipelinesEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glDeleteProgramPipelinesEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteProgramPipelinesEXT\0",
            &glDeleteProgramPipelinesEXT_p,
        )
    }
    /// Checks if the pointer for [`glDeleteProgramPipelinesEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub fn glDeleteProgramPipelinesEXT_is_loaded() -> bool {
        !glDeleteProgramPipelinesEXT_p.load(RELAX).is_null()
    }

    /// [glDeleteQueries](http://docs.gl/es3/glDeleteQueries)(n, ids)
    /// * `ids` len: n
    /// * `ids` class: query
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteQueries(n: GLsizei, ids: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteQueries({:?}, {:p});", n, ids);
        }
        let out = call_atomic_ptr_2arg("glDeleteQueries", &glDeleteQueries_p, n, ids);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteQueries");
        }
        out
    }
    static glDeleteQueries_p: APcv = ap_null();
    /// Tries to load [`glDeleteQueries`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteQueries_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDeleteQueries\0", &glDeleteQueries_p)
    }
    /// Checks if the pointer for [`glDeleteQueries`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteQueries_is_loaded() -> bool {
        !glDeleteQueries_p.load(RELAX).is_null()
    }

    /// [glDeleteQueriesEXT](http://docs.gl/es3/glDeleteQueriesEXT)(n, ids)
    /// * `ids` len: n
    /// * `ids` class: query
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glDeleteQueriesEXT(n: GLsizei, ids: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteQueriesEXT({:?}, {:p});", n, ids);
        }
        let out = call_atomic_ptr_2arg("glDeleteQueriesEXT", &glDeleteQueriesEXT_p, n, ids);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteQueriesEXT");
        }
        out
    }
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    static glDeleteQueriesEXT_p: APcv = ap_null();
    /// Tries to load [`glDeleteQueriesEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glDeleteQueriesEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteQueriesEXT\0",
            &glDeleteQueriesEXT_p,
        )
    }
    /// Checks if the pointer for [`glDeleteQueriesEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub fn glDeleteQueriesEXT_is_loaded() -> bool {
        !glDeleteQueriesEXT_p.load(RELAX).is_null()
    }

    /// [glDeleteRenderbuffers](http://docs.gl/es3/glDeleteRenderbuffers)(n, renderbuffers)
    /// * `renderbuffers` len: n
    /// * `renderbuffers` class: renderbuffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteRenderbuffers(n: GLsizei, renderbuffers: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDeleteRenderbuffers({:?}, {:p});",
                n,
                renderbuffers
            );
        }
        let out = call_atomic_ptr_2arg(
            "glDeleteRenderbuffers",
            &glDeleteRenderbuffers_p,
            n,
            renderbuffers,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteRenderbuffers");
        }
        out
    }
    static glDeleteRenderbuffers_p: APcv = ap_null();
    /// Tries to load [`glDeleteRenderbuffers`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteRenderbuffers_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteRenderbuffers\0",
            &glDeleteRenderbuffers_p,
        )
    }
    /// Checks if the pointer for [`glDeleteRenderbuffers`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteRenderbuffers_is_loaded() -> bool {
        !glDeleteRenderbuffers_p.load(RELAX).is_null()
    }

    /// [glDeleteSamplers](http://docs.gl/es3/glDeleteSamplers)(count, samplers)
    /// * `samplers` len: count
    /// * `samplers` class: sampler
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteSamplers(count: GLsizei, samplers: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteSamplers({:?}, {:p});", count, samplers);
        }
        let out = call_atomic_ptr_2arg("glDeleteSamplers", &glDeleteSamplers_p, count, samplers);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteSamplers");
        }
        out
    }
    static glDeleteSamplers_p: APcv = ap_null();
    /// Tries to load [`glDeleteSamplers`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteSamplers_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDeleteSamplers\0", &glDeleteSamplers_p)
    }
    /// Checks if the pointer for [`glDeleteSamplers`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteSamplers_is_loaded() -> bool {
        !glDeleteSamplers_p.load(RELAX).is_null()
    }

    /// [glDeleteSemaphoresEXT](http://docs.gl/es3/glDeleteSemaphoresEXT)(n, semaphores)
    /// * `semaphores` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    pub unsafe fn glDeleteSemaphoresEXT(n: GLsizei, semaphores: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteSemaphoresEXT({:?}, {:p});", n, semaphores);
        }
        let out = call_atomic_ptr_2arg(
            "glDeleteSemaphoresEXT",
            &glDeleteSemaphoresEXT_p,
            n,
            semaphores,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteSemaphoresEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    static glDeleteSemaphoresEXT_p: APcv = ap_null();
    /// Tries to load [`glDeleteSemaphoresEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    pub unsafe fn glDeleteSemaphoresEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteSemaphoresEXT\0",
            &glDeleteSemaphoresEXT_p,
        )
    }
    /// Checks if the pointer for [`glDeleteSemaphoresEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    pub fn glDeleteSemaphoresEXT_is_loaded() -> bool {
        !glDeleteSemaphoresEXT_p.load(RELAX).is_null()
    }

    /// [glDeleteShader](http://docs.gl/es3/glDeleteShader)(shader)
    /// * `shader` class: shader
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteShader(shader: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteShader({:?});", shader);
        }
        let out = call_atomic_ptr_1arg("glDeleteShader", &glDeleteShader_p, shader);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteShader");
        }
        out
    }
    static glDeleteShader_p: APcv = ap_null();
    /// Tries to load [`glDeleteShader`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteShader_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDeleteShader\0", &glDeleteShader_p)
    }
    /// Checks if the pointer for [`glDeleteShader`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteShader_is_loaded() -> bool {
        !glDeleteShader_p.load(RELAX).is_null()
    }

    /// [glDeleteSync](http://docs.gl/es3/glDeleteSync)(sync)
    /// * `sync` group: sync
    /// * `sync` class: sync
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteSync(sync: GLsync) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteSync({:p});", sync);
        }
        let out = call_atomic_ptr_1arg("glDeleteSync", &glDeleteSync_p, sync);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteSync");
        }
        out
    }
    static glDeleteSync_p: APcv = ap_null();
    /// Tries to load [`glDeleteSync`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteSync_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDeleteSync\0", &glDeleteSync_p)
    }
    /// Checks if the pointer for [`glDeleteSync`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteSync_is_loaded() -> bool {
        !glDeleteSync_p.load(RELAX).is_null()
    }

    /// [glDeleteSyncAPPLE](http://docs.gl/es3/glDeleteSyncAPPLE)(sync)
    /// * `sync` class: sync
    /// * alias of: [`glDeleteSync`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glDeleteSyncAPPLE(sync: GLsync) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteSyncAPPLE({:p});", sync);
        }
        let out = call_atomic_ptr_1arg("glDeleteSyncAPPLE", &glDeleteSyncAPPLE_p, sync);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteSyncAPPLE");
        }
        out
    }
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    static glDeleteSyncAPPLE_p: APcv = ap_null();
    /// Tries to load [`glDeleteSyncAPPLE`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glDeleteSyncAPPLE_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteSyncAPPLE\0",
            &glDeleteSyncAPPLE_p,
        )
    }
    /// Checks if the pointer for [`glDeleteSyncAPPLE`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub fn glDeleteSyncAPPLE_is_loaded() -> bool {
        !glDeleteSyncAPPLE_p.load(RELAX).is_null()
    }

    /// [glDeleteTextures](http://docs.gl/es3/glDeleteTextures)(n, textures)
    /// * `textures` group: Texture
    /// * `textures` len: n
    /// * `textures` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteTextures(n: GLsizei, textures: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteTextures({:?}, {:p});", n, textures);
        }
        let out = call_atomic_ptr_2arg("glDeleteTextures", &glDeleteTextures_p, n, textures);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteTextures");
        }
        out
    }
    static glDeleteTextures_p: APcv = ap_null();
    /// Tries to load [`glDeleteTextures`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteTextures_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDeleteTextures\0", &glDeleteTextures_p)
    }
    /// Checks if the pointer for [`glDeleteTextures`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteTextures_is_loaded() -> bool {
        !glDeleteTextures_p.load(RELAX).is_null()
    }

    /// [glDeleteTransformFeedbacks](http://docs.gl/es3/glDeleteTransformFeedbacks)(n, ids)
    /// * `ids` len: n
    /// * `ids` class: transform feedback
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteTransformFeedbacks(n: GLsizei, ids: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteTransformFeedbacks({:?}, {:p});", n, ids);
        }
        let out = call_atomic_ptr_2arg(
            "glDeleteTransformFeedbacks",
            &glDeleteTransformFeedbacks_p,
            n,
            ids,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteTransformFeedbacks");
        }
        out
    }
    static glDeleteTransformFeedbacks_p: APcv = ap_null();
    /// Tries to load [`glDeleteTransformFeedbacks`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteTransformFeedbacks_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteTransformFeedbacks\0",
            &glDeleteTransformFeedbacks_p,
        )
    }
    /// Checks if the pointer for [`glDeleteTransformFeedbacks`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteTransformFeedbacks_is_loaded() -> bool {
        !glDeleteTransformFeedbacks_p.load(RELAX).is_null()
    }

    /// [glDeleteVertexArrays](http://docs.gl/es3/glDeleteVertexArrays)(n, arrays)
    /// * `arrays` len: n
    /// * `arrays` class: vertex array
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDeleteVertexArrays(n: GLsizei, arrays: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteVertexArrays({:?}, {:p});", n, arrays);
        }
        let out = call_atomic_ptr_2arg("glDeleteVertexArrays", &glDeleteVertexArrays_p, n, arrays);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteVertexArrays");
        }
        out
    }
    static glDeleteVertexArrays_p: APcv = ap_null();
    /// Tries to load [`glDeleteVertexArrays`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDeleteVertexArrays_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteVertexArrays\0",
            &glDeleteVertexArrays_p,
        )
    }
    /// Checks if the pointer for [`glDeleteVertexArrays`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDeleteVertexArrays_is_loaded() -> bool {
        !glDeleteVertexArrays_p.load(RELAX).is_null()
    }

    /// [glDeleteVertexArraysOES](http://docs.gl/es3/glDeleteVertexArraysOES)(n, arrays)
    /// * `arrays` len: n
    /// * `arrays` class: vertex array
    /// * alias of: [`glDeleteVertexArrays`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    pub unsafe fn glDeleteVertexArraysOES(n: GLsizei, arrays: *const GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDeleteVertexArraysOES({:?}, {:p});", n, arrays);
        }
        let out = call_atomic_ptr_2arg(
            "glDeleteVertexArraysOES",
            &glDeleteVertexArraysOES_p,
            n,
            arrays,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDeleteVertexArraysOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    static glDeleteVertexArraysOES_p: APcv = ap_null();
    /// Tries to load [`glDeleteVertexArraysOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    pub unsafe fn glDeleteVertexArraysOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDeleteVertexArraysOES\0",
            &glDeleteVertexArraysOES_p,
        )
    }
    /// Checks if the pointer for [`glDeleteVertexArraysOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    pub fn glDeleteVertexArraysOES_is_loaded() -> bool {
        !glDeleteVertexArraysOES_p.load(RELAX).is_null()
    }

    /// [glDepthFunc](http://docs.gl/es3/glDepthFunc)(func)
    /// * `func` group: DepthFunction
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDepthFunc(func: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDepthFunc({:#X});", func);
        }
        let out = call_atomic_ptr_1arg("glDepthFunc", &glDepthFunc_p, func);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDepthFunc");
        }
        out
    }
    static glDepthFunc_p: APcv = ap_null();
    /// Tries to load [`glDepthFunc`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDepthFunc_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDepthFunc\0", &glDepthFunc_p)
    }
    /// Checks if the pointer for [`glDepthFunc`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDepthFunc_is_loaded() -> bool {
        !glDepthFunc_p.load(RELAX).is_null()
    }

    /// [glDepthMask](http://docs.gl/es3/glDepthMask)(flag)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDepthMask(flag: GLboolean) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDepthMask({:?});", flag);
        }
        let out = call_atomic_ptr_1arg("glDepthMask", &glDepthMask_p, flag);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDepthMask");
        }
        out
    }
    static glDepthMask_p: APcv = ap_null();
    /// Tries to load [`glDepthMask`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDepthMask_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDepthMask\0", &glDepthMask_p)
    }
    /// Checks if the pointer for [`glDepthMask`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDepthMask_is_loaded() -> bool {
        !glDepthMask_p.load(RELAX).is_null()
    }

    /// [glDepthRangeArrayfvNV](http://docs.gl/es3/glDepthRangeArrayfvNV)(first, count, v)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glDepthRangeArrayfvNV(first: GLuint, count: GLsizei, v: *const GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDepthRangeArrayfvNV({:?}, {:?}, {:p});",
                first,
                count,
                v
            );
        }
        let out = call_atomic_ptr_3arg(
            "glDepthRangeArrayfvNV",
            &glDepthRangeArrayfvNV_p,
            first,
            count,
            v,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDepthRangeArrayfvNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    static glDepthRangeArrayfvNV_p: APcv = ap_null();
    /// Tries to load [`glDepthRangeArrayfvNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glDepthRangeArrayfvNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDepthRangeArrayfvNV\0",
            &glDepthRangeArrayfvNV_p,
        )
    }
    /// Checks if the pointer for [`glDepthRangeArrayfvNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub fn glDepthRangeArrayfvNV_is_loaded() -> bool {
        !glDepthRangeArrayfvNV_p.load(RELAX).is_null()
    }

    /// [glDepthRangeArrayfvOES](http://docs.gl/es3/glDepthRangeArrayfvOES)(first, count, v)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    pub unsafe fn glDepthRangeArrayfvOES(first: GLuint, count: GLsizei, v: *const GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDepthRangeArrayfvOES({:?}, {:?}, {:p});",
                first,
                count,
                v
            );
        }
        let out = call_atomic_ptr_3arg(
            "glDepthRangeArrayfvOES",
            &glDepthRangeArrayfvOES_p,
            first,
            count,
            v,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDepthRangeArrayfvOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    static glDepthRangeArrayfvOES_p: APcv = ap_null();
    /// Tries to load [`glDepthRangeArrayfvOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    pub unsafe fn glDepthRangeArrayfvOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDepthRangeArrayfvOES\0",
            &glDepthRangeArrayfvOES_p,
        )
    }
    /// Checks if the pointer for [`glDepthRangeArrayfvOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    pub fn glDepthRangeArrayfvOES_is_loaded() -> bool {
        !glDepthRangeArrayfvOES_p.load(RELAX).is_null()
    }

    /// [glDepthRangeIndexedfNV](http://docs.gl/es3/glDepthRangeIndexedfNV)(index, n, f)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glDepthRangeIndexedfNV(index: GLuint, n: GLfloat, f: GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDepthRangeIndexedfNV({:?}, {:?}, {:?});",
                index,
                n,
                f
            );
        }
        let out = call_atomic_ptr_3arg(
            "glDepthRangeIndexedfNV",
            &glDepthRangeIndexedfNV_p,
            index,
            n,
            f,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDepthRangeIndexedfNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    static glDepthRangeIndexedfNV_p: APcv = ap_null();
    /// Tries to load [`glDepthRangeIndexedfNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glDepthRangeIndexedfNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDepthRangeIndexedfNV\0",
            &glDepthRangeIndexedfNV_p,
        )
    }
    /// Checks if the pointer for [`glDepthRangeIndexedfNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub fn glDepthRangeIndexedfNV_is_loaded() -> bool {
        !glDepthRangeIndexedfNV_p.load(RELAX).is_null()
    }

    /// [glDepthRangeIndexedfOES](http://docs.gl/es3/glDepthRangeIndexedfOES)(index, n, f)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    pub unsafe fn glDepthRangeIndexedfOES(index: GLuint, n: GLfloat, f: GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDepthRangeIndexedfOES({:?}, {:?}, {:?});",
                index,
                n,
                f
            );
        }
        let out = call_atomic_ptr_3arg(
            "glDepthRangeIndexedfOES",
            &glDepthRangeIndexedfOES_p,
            index,
            n,
            f,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDepthRangeIndexedfOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    static glDepthRangeIndexedfOES_p: APcv = ap_null();
    /// Tries to load [`glDepthRangeIndexedfOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    pub unsafe fn glDepthRangeIndexedfOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDepthRangeIndexedfOES\0",
            &glDepthRangeIndexedfOES_p,
        )
    }
    /// Checks if the pointer for [`glDepthRangeIndexedfOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    pub fn glDepthRangeIndexedfOES_is_loaded() -> bool {
        !glDepthRangeIndexedfOES_p.load(RELAX).is_null()
    }

    /// [glDepthRangef](http://docs.gl/es3/glDepthRange)(n, f)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDepthRangef(n: GLfloat, f: GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDepthRangef({:?}, {:?});", n, f);
        }
        let out = call_atomic_ptr_2arg("glDepthRangef", &glDepthRangef_p, n, f);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDepthRangef");
        }
        out
    }
    static glDepthRangef_p: APcv = ap_null();
    /// Tries to load [`glDepthRangef`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDepthRangef_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDepthRangef\0", &glDepthRangef_p)
    }
    /// Checks if the pointer for [`glDepthRangef`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDepthRangef_is_loaded() -> bool {
        !glDepthRangef_p.load(RELAX).is_null()
    }

    /// [glDetachShader](http://docs.gl/es3/glDetachShader)(program, shader)
    /// * `program` class: program
    /// * `shader` class: shader
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDetachShader(program: GLuint, shader: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDetachShader({:?}, {:?});", program, shader);
        }
        let out = call_atomic_ptr_2arg("glDetachShader", &glDetachShader_p, program, shader);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDetachShader");
        }
        out
    }
    static glDetachShader_p: APcv = ap_null();
    /// Tries to load [`glDetachShader`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDetachShader_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDetachShader\0", &glDetachShader_p)
    }
    /// Checks if the pointer for [`glDetachShader`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDetachShader_is_loaded() -> bool {
        !glDetachShader_p.load(RELAX).is_null()
    }

    /// [glDisable](http://docs.gl/es3/glDisable)(cap)
    /// * `cap` group: EnableCap
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDisable(cap: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDisable({:#X});", cap);
        }
        let out = call_atomic_ptr_1arg("glDisable", &glDisable_p, cap);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDisable");
        }
        out
    }
    static glDisable_p: APcv = ap_null();
    /// Tries to load [`glDisable`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDisable_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDisable\0", &glDisable_p)
    }
    /// Checks if the pointer for [`glDisable`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDisable_is_loaded() -> bool {
        !glDisable_p.load(RELAX).is_null()
    }

    /// [glDisableDriverControlQCOM](http://docs.gl/es3/glDisableDriverControlQCOM)(driverControl)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub unsafe fn glDisableDriverControlQCOM(driverControl: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDisableDriverControlQCOM({:?});", driverControl);
        }
        let out = call_atomic_ptr_1arg(
            "glDisableDriverControlQCOM",
            &glDisableDriverControlQCOM_p,
            driverControl,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDisableDriverControlQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    static glDisableDriverControlQCOM_p: APcv = ap_null();
    /// Tries to load [`glDisableDriverControlQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub unsafe fn glDisableDriverControlQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDisableDriverControlQCOM\0",
            &glDisableDriverControlQCOM_p,
        )
    }
    /// Checks if the pointer for [`glDisableDriverControlQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub fn glDisableDriverControlQCOM_is_loaded() -> bool {
        !glDisableDriverControlQCOM_p.load(RELAX).is_null()
    }

    /// [glDisableVertexAttribArray](http://docs.gl/es3/glDisableVertexAttribArray)(index)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDisableVertexAttribArray(index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDisableVertexAttribArray({:?});", index);
        }
        let out = call_atomic_ptr_1arg(
            "glDisableVertexAttribArray",
            &glDisableVertexAttribArray_p,
            index,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDisableVertexAttribArray");
        }
        out
    }
    static glDisableVertexAttribArray_p: APcv = ap_null();
    /// Tries to load [`glDisableVertexAttribArray`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDisableVertexAttribArray_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDisableVertexAttribArray\0",
            &glDisableVertexAttribArray_p,
        )
    }
    /// Checks if the pointer for [`glDisableVertexAttribArray`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDisableVertexAttribArray_is_loaded() -> bool {
        !glDisableVertexAttribArray_p.load(RELAX).is_null()
    }

    /// [glDisablei](http://docs.gl/es3/glDisable)(target, index)
    /// * `target` group: EnableCap
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDisablei(target: GLenum, index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDisablei({:#X}, {:?});", target, index);
        }
        let out = call_atomic_ptr_2arg("glDisablei", &glDisablei_p, target, index);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDisablei");
        }
        out
    }
    static glDisablei_p: APcv = ap_null();
    /// Tries to load [`glDisablei`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDisablei_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDisablei\0", &glDisablei_p)
    }
    /// Checks if the pointer for [`glDisablei`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDisablei_is_loaded() -> bool {
        !glDisablei_p.load(RELAX).is_null()
    }

    /// [glDisableiEXT](http://docs.gl/es3/glDisableiEXT)(target, index)
    /// * `target` group: EnableCap
    /// * alias of: [`glDisablei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glDisableiEXT(target: GLenum, index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDisableiEXT({:#X}, {:?});", target, index);
        }
        let out = call_atomic_ptr_2arg("glDisableiEXT", &glDisableiEXT_p, target, index);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDisableiEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    static glDisableiEXT_p: APcv = ap_null();
    /// Tries to load [`glDisableiEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glDisableiEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDisableiEXT\0", &glDisableiEXT_p)
    }
    /// Checks if the pointer for [`glDisableiEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub fn glDisableiEXT_is_loaded() -> bool {
        !glDisableiEXT_p.load(RELAX).is_null()
    }

    /// [glDisableiNV](http://docs.gl/es3/glDisableiNV)(target, index)
    /// * `target` group: EnableCap
    /// * alias of: [`glDisablei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glDisableiNV(target: GLenum, index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDisableiNV({:#X}, {:?});", target, index);
        }
        let out = call_atomic_ptr_2arg("glDisableiNV", &glDisableiNV_p, target, index);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDisableiNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    static glDisableiNV_p: APcv = ap_null();
    /// Tries to load [`glDisableiNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glDisableiNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDisableiNV\0", &glDisableiNV_p)
    }
    /// Checks if the pointer for [`glDisableiNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub fn glDisableiNV_is_loaded() -> bool {
        !glDisableiNV_p.load(RELAX).is_null()
    }

    /// [glDisableiOES](http://docs.gl/es3/glDisableiOES)(target, index)
    /// * `target` group: EnableCap
    /// * alias of: [`glDisablei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(
        feature = "GL_OES_draw_buffers_indexed",
        feature = "GL_OES_viewport_array"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        )))
    )]
    pub unsafe fn glDisableiOES(target: GLenum, index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDisableiOES({:#X}, {:?});", target, index);
        }
        let out = call_atomic_ptr_2arg("glDisableiOES", &glDisableiOES_p, target, index);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDisableiOES");
        }
        out
    }
    #[cfg(any(
        feature = "GL_OES_draw_buffers_indexed",
        feature = "GL_OES_viewport_array"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        )))
    )]
    static glDisableiOES_p: APcv = ap_null();
    /// Tries to load [`glDisableiOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_OES_draw_buffers_indexed",
        feature = "GL_OES_viewport_array"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        )))
    )]
    pub unsafe fn glDisableiOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDisableiOES\0", &glDisableiOES_p)
    }
    /// Checks if the pointer for [`glDisableiOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_OES_draw_buffers_indexed",
        feature = "GL_OES_viewport_array"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        )))
    )]
    pub fn glDisableiOES_is_loaded() -> bool {
        !glDisableiOES_p.load(RELAX).is_null()
    }

    /// [glDiscardFramebufferEXT](http://docs.gl/es3/glDiscardFramebufferEXT)(target, numAttachments, attachments)
    /// * `target` group: FramebufferTarget
    /// * `attachments` group: InvalidateFramebufferAttachment
    /// * `attachments` len: numAttachments
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_discard_framebuffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_discard_framebuffer"))))]
    pub unsafe fn glDiscardFramebufferEXT(
        target: GLenum,
        numAttachments: GLsizei,
        attachments: *const GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDiscardFramebufferEXT({:#X}, {:?}, {:p});",
                target,
                numAttachments,
                attachments
            );
        }
        let out = call_atomic_ptr_3arg(
            "glDiscardFramebufferEXT",
            &glDiscardFramebufferEXT_p,
            target,
            numAttachments,
            attachments,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDiscardFramebufferEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_discard_framebuffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_discard_framebuffer"))))]
    static glDiscardFramebufferEXT_p: APcv = ap_null();
    /// Tries to load [`glDiscardFramebufferEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_discard_framebuffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_discard_framebuffer"))))]
    pub unsafe fn glDiscardFramebufferEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDiscardFramebufferEXT\0",
            &glDiscardFramebufferEXT_p,
        )
    }
    /// Checks if the pointer for [`glDiscardFramebufferEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_discard_framebuffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_discard_framebuffer"))))]
    pub fn glDiscardFramebufferEXT_is_loaded() -> bool {
        !glDiscardFramebufferEXT_p.load(RELAX).is_null()
    }

    /// [glDispatchCompute](http://docs.gl/es3/glDispatchCompute)(num_groups_x, num_groups_y, num_groups_z)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDispatchCompute(
        num_groups_x: GLuint,
        num_groups_y: GLuint,
        num_groups_z: GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDispatchCompute({:?}, {:?}, {:?});",
                num_groups_x,
                num_groups_y,
                num_groups_z
            );
        }
        let out = call_atomic_ptr_3arg(
            "glDispatchCompute",
            &glDispatchCompute_p,
            num_groups_x,
            num_groups_y,
            num_groups_z,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDispatchCompute");
        }
        out
    }
    static glDispatchCompute_p: APcv = ap_null();
    /// Tries to load [`glDispatchCompute`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDispatchCompute_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDispatchCompute\0",
            &glDispatchCompute_p,
        )
    }
    /// Checks if the pointer for [`glDispatchCompute`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDispatchCompute_is_loaded() -> bool {
        !glDispatchCompute_p.load(RELAX).is_null()
    }

    /// [glDispatchComputeIndirect](http://docs.gl/es3/glDispatchComputeIndirect)(indirect)
    /// * `indirect` group: BufferOffset
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDispatchComputeIndirect(indirect: GLintptr) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDispatchComputeIndirect({:?});", indirect);
        }
        let out = call_atomic_ptr_1arg(
            "glDispatchComputeIndirect",
            &glDispatchComputeIndirect_p,
            indirect,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDispatchComputeIndirect");
        }
        out
    }
    static glDispatchComputeIndirect_p: APcv = ap_null();
    /// Tries to load [`glDispatchComputeIndirect`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDispatchComputeIndirect_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDispatchComputeIndirect\0",
            &glDispatchComputeIndirect_p,
        )
    }
    /// Checks if the pointer for [`glDispatchComputeIndirect`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDispatchComputeIndirect_is_loaded() -> bool {
        !glDispatchComputeIndirect_p.load(RELAX).is_null()
    }

    /// [glDrawArrays](http://docs.gl/es3/glDrawArrays)(mode, first, count)
    /// * `mode` group: PrimitiveType
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawArrays(mode: GLenum, first: GLint, count: GLsizei) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawArrays({:#X}, {:?}, {:?});",
                mode,
                first,
                count
            );
        }
        let out = call_atomic_ptr_3arg("glDrawArrays", &glDrawArrays_p, mode, first, count);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawArrays");
        }
        out
    }
    static glDrawArrays_p: APcv = ap_null();
    /// Tries to load [`glDrawArrays`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawArrays_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDrawArrays\0", &glDrawArrays_p)
    }
    /// Checks if the pointer for [`glDrawArrays`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawArrays_is_loaded() -> bool {
        !glDrawArrays_p.load(RELAX).is_null()
    }

    /// [glDrawArraysIndirect](http://docs.gl/es3/glDrawArraysIndirect)(mode, indirect)
    /// * `mode` group: PrimitiveType
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawArraysIndirect(mode: GLenum, indirect: *const c_void) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDrawArraysIndirect({:#X}, {:p});", mode, indirect);
        }
        let out = call_atomic_ptr_2arg(
            "glDrawArraysIndirect",
            &glDrawArraysIndirect_p,
            mode,
            indirect,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawArraysIndirect");
        }
        out
    }
    static glDrawArraysIndirect_p: APcv = ap_null();
    /// Tries to load [`glDrawArraysIndirect`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawArraysIndirect_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawArraysIndirect\0",
            &glDrawArraysIndirect_p,
        )
    }
    /// Checks if the pointer for [`glDrawArraysIndirect`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawArraysIndirect_is_loaded() -> bool {
        !glDrawArraysIndirect_p.load(RELAX).is_null()
    }

    /// [glDrawArraysInstanced](http://docs.gl/es3/glDrawArraysInstanced)(mode, first, count, instancecount)
    /// * `mode` group: PrimitiveType
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawArraysInstanced(
        mode: GLenum,
        first: GLint,
        count: GLsizei,
        instancecount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawArraysInstanced({:#X}, {:?}, {:?}, {:?});",
                mode,
                first,
                count,
                instancecount
            );
        }
        let out = call_atomic_ptr_4arg(
            "glDrawArraysInstanced",
            &glDrawArraysInstanced_p,
            mode,
            first,
            count,
            instancecount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawArraysInstanced");
        }
        out
    }
    static glDrawArraysInstanced_p: APcv = ap_null();
    /// Tries to load [`glDrawArraysInstanced`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawArraysInstanced_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawArraysInstanced\0",
            &glDrawArraysInstanced_p,
        )
    }
    /// Checks if the pointer for [`glDrawArraysInstanced`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawArraysInstanced_is_loaded() -> bool {
        !glDrawArraysInstanced_p.load(RELAX).is_null()
    }

    /// [glDrawArraysInstancedANGLE](http://docs.gl/es3/glDrawArraysInstancedANGLE)(mode, first, count, primcount)
    /// * `mode` group: PrimitiveType
    /// * alias of: [`glDrawArraysInstanced`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
    pub unsafe fn glDrawArraysInstancedANGLE(
        mode: GLenum,
        first: GLint,
        count: GLsizei,
        primcount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawArraysInstancedANGLE({:#X}, {:?}, {:?}, {:?});",
                mode,
                first,
                count,
                primcount
            );
        }
        let out = call_atomic_ptr_4arg(
            "glDrawArraysInstancedANGLE",
            &glDrawArraysInstancedANGLE_p,
            mode,
            first,
            count,
            primcount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawArraysInstancedANGLE");
        }
        out
    }
    #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
    static glDrawArraysInstancedANGLE_p: APcv = ap_null();
    /// Tries to load [`glDrawArraysInstancedANGLE`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
    pub unsafe fn glDrawArraysInstancedANGLE_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawArraysInstancedANGLE\0",
            &glDrawArraysInstancedANGLE_p,
        )
    }
    /// Checks if the pointer for [`glDrawArraysInstancedANGLE`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
    pub fn glDrawArraysInstancedANGLE_is_loaded() -> bool {
        !glDrawArraysInstancedANGLE_p.load(RELAX).is_null()
    }

    /// [glDrawArraysInstancedBaseInstanceEXT](http://docs.gl/es3/glDrawArraysInstancedBaseInstanceEXT)(mode, first, count, instancecount, baseinstance)
    /// * `mode` group: PrimitiveType
    /// * alias of: [`glDrawArraysInstancedBaseInstance`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    pub unsafe fn glDrawArraysInstancedBaseInstanceEXT(
        mode: GLenum,
        first: GLint,
        count: GLsizei,
        instancecount: GLsizei,
        baseinstance: GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawArraysInstancedBaseInstanceEXT({:#X}, {:?}, {:?}, {:?}, {:?});",
                mode,
                first,
                count,
                instancecount,
                baseinstance
            );
        }
        let out = call_atomic_ptr_5arg(
            "glDrawArraysInstancedBaseInstanceEXT",
            &glDrawArraysInstancedBaseInstanceEXT_p,
            mode,
            first,
            count,
            instancecount,
            baseinstance,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawArraysInstancedBaseInstanceEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    static glDrawArraysInstancedBaseInstanceEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawArraysInstancedBaseInstanceEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    pub unsafe fn glDrawArraysInstancedBaseInstanceEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawArraysInstancedBaseInstanceEXT\0",
            &glDrawArraysInstancedBaseInstanceEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawArraysInstancedBaseInstanceEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    pub fn glDrawArraysInstancedBaseInstanceEXT_is_loaded() -> bool {
        !glDrawArraysInstancedBaseInstanceEXT_p.load(RELAX).is_null()
    }

    /// [glDrawArraysInstancedEXT](http://docs.gl/es3/glDrawArraysInstancedEXT)(mode, start, count, primcount)
    /// * `mode` group: PrimitiveType
    /// * alias of: [`glDrawArraysInstanced`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays")))
    )]
    pub unsafe fn glDrawArraysInstancedEXT(
        mode: GLenum,
        start: GLint,
        count: GLsizei,
        primcount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawArraysInstancedEXT({:#X}, {:?}, {:?}, {:?});",
                mode,
                start,
                count,
                primcount
            );
        }
        let out = call_atomic_ptr_4arg(
            "glDrawArraysInstancedEXT",
            &glDrawArraysInstancedEXT_p,
            mode,
            start,
            count,
            primcount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawArraysInstancedEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays")))
    )]
    static glDrawArraysInstancedEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawArraysInstancedEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays")))
    )]
    pub unsafe fn glDrawArraysInstancedEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawArraysInstancedEXT\0",
            &glDrawArraysInstancedEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawArraysInstancedEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays")))
    )]
    pub fn glDrawArraysInstancedEXT_is_loaded() -> bool {
        !glDrawArraysInstancedEXT_p.load(RELAX).is_null()
    }

    /// [glDrawArraysInstancedNV](http://docs.gl/es3/glDrawArraysInstancedNV)(mode, first, count, primcount)
    /// * `mode` group: PrimitiveType
    /// * alias of: [`glDrawArraysInstanced`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_draw_instanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
    pub unsafe fn glDrawArraysInstancedNV(
        mode: GLenum,
        first: GLint,
        count: GLsizei,
        primcount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawArraysInstancedNV({:#X}, {:?}, {:?}, {:?});",
                mode,
                first,
                count,
                primcount
            );
        }
        let out = call_atomic_ptr_4arg(
            "glDrawArraysInstancedNV",
            &glDrawArraysInstancedNV_p,
            mode,
            first,
            count,
            primcount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawArraysInstancedNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_draw_instanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
    static glDrawArraysInstancedNV_p: APcv = ap_null();
    /// Tries to load [`glDrawArraysInstancedNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_draw_instanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
    pub unsafe fn glDrawArraysInstancedNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawArraysInstancedNV\0",
            &glDrawArraysInstancedNV_p,
        )
    }
    /// Checks if the pointer for [`glDrawArraysInstancedNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_draw_instanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
    pub fn glDrawArraysInstancedNV_is_loaded() -> bool {
        !glDrawArraysInstancedNV_p.load(RELAX).is_null()
    }

    /// [glDrawBuffers](http://docs.gl/es3/glDrawBuffers)(n, bufs)
    /// * `bufs` group: DrawBufferMode
    /// * `bufs` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawBuffers(n: GLsizei, bufs: *const GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDrawBuffers({:?}, {:p});", n, bufs);
        }
        let out = call_atomic_ptr_2arg("glDrawBuffers", &glDrawBuffers_p, n, bufs);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawBuffers");
        }
        out
    }
    static glDrawBuffers_p: APcv = ap_null();
    /// Tries to load [`glDrawBuffers`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawBuffers_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDrawBuffers\0", &glDrawBuffers_p)
    }
    /// Checks if the pointer for [`glDrawBuffers`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawBuffers_is_loaded() -> bool {
        !glDrawBuffers_p.load(RELAX).is_null()
    }

    /// [glDrawBuffersEXT](http://docs.gl/es3/glDrawBuffersEXT)(n, bufs)
    /// * `bufs` len: n
    /// * alias of: [`glDrawBuffers`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers"))))]
    pub unsafe fn glDrawBuffersEXT(n: GLsizei, bufs: *const GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDrawBuffersEXT({:?}, {:p});", n, bufs);
        }
        let out = call_atomic_ptr_2arg("glDrawBuffersEXT", &glDrawBuffersEXT_p, n, bufs);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawBuffersEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers"))))]
    static glDrawBuffersEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawBuffersEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers"))))]
    pub unsafe fn glDrawBuffersEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDrawBuffersEXT\0", &glDrawBuffersEXT_p)
    }
    /// Checks if the pointer for [`glDrawBuffersEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers"))))]
    pub fn glDrawBuffersEXT_is_loaded() -> bool {
        !glDrawBuffersEXT_p.load(RELAX).is_null()
    }

    /// [glDrawBuffersIndexedEXT](http://docs.gl/es3/glDrawBuffersIndexedEXT)(n, location, indices)
    /// * `location` len: n
    /// * `indices` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
    pub unsafe fn glDrawBuffersIndexedEXT(
        n: GLint,
        location: *const GLenum,
        indices: *const GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawBuffersIndexedEXT({:?}, {:p}, {:p});",
                n,
                location,
                indices
            );
        }
        let out = call_atomic_ptr_3arg(
            "glDrawBuffersIndexedEXT",
            &glDrawBuffersIndexedEXT_p,
            n,
            location,
            indices,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawBuffersIndexedEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
    static glDrawBuffersIndexedEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawBuffersIndexedEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
    pub unsafe fn glDrawBuffersIndexedEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawBuffersIndexedEXT\0",
            &glDrawBuffersIndexedEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawBuffersIndexedEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
    pub fn glDrawBuffersIndexedEXT_is_loaded() -> bool {
        !glDrawBuffersIndexedEXT_p.load(RELAX).is_null()
    }

    /// [glDrawBuffersNV](http://docs.gl/es3/glDrawBuffersNV)(n, bufs)
    /// * `bufs` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_buffers"))))]
    pub unsafe fn glDrawBuffersNV(n: GLsizei, bufs: *const GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDrawBuffersNV({:?}, {:p});", n, bufs);
        }
        let out = call_atomic_ptr_2arg("glDrawBuffersNV", &glDrawBuffersNV_p, n, bufs);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawBuffersNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_buffers"))))]
    static glDrawBuffersNV_p: APcv = ap_null();
    /// Tries to load [`glDrawBuffersNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_buffers"))))]
    pub unsafe fn glDrawBuffersNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDrawBuffersNV\0", &glDrawBuffersNV_p)
    }
    /// Checks if the pointer for [`glDrawBuffersNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_buffers"))))]
    pub fn glDrawBuffersNV_is_loaded() -> bool {
        !glDrawBuffersNV_p.load(RELAX).is_null()
    }

    /// [glDrawElements](http://docs.gl/es3/glDrawElements)(mode, count, type_, indices)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawElements(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElements({:#X}, {:?}, {:#X}, {:p});",
                mode,
                count,
                type_,
                indices
            );
        }
        let out = call_atomic_ptr_4arg(
            "glDrawElements",
            &glDrawElements_p,
            mode,
            count,
            type_,
            indices,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElements");
        }
        out
    }
    static glDrawElements_p: APcv = ap_null();
    /// Tries to load [`glDrawElements`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawElements_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDrawElements\0", &glDrawElements_p)
    }
    /// Checks if the pointer for [`glDrawElements`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawElements_is_loaded() -> bool {
        !glDrawElements_p.load(RELAX).is_null()
    }

    /// [glDrawElementsBaseVertex](http://docs.gl/es3/glDrawElementsBaseVertex)(mode, count, type_, indices, basevertex)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawElementsBaseVertex(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        basevertex: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsBaseVertex({:#X}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                count,
                type_,
                indices,
                basevertex
            );
        }
        let out = call_atomic_ptr_5arg(
            "glDrawElementsBaseVertex",
            &glDrawElementsBaseVertex_p,
            mode,
            count,
            type_,
            indices,
            basevertex,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsBaseVertex");
        }
        out
    }
    static glDrawElementsBaseVertex_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsBaseVertex`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawElementsBaseVertex_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsBaseVertex\0",
            &glDrawElementsBaseVertex_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsBaseVertex`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawElementsBaseVertex_is_loaded() -> bool {
        !glDrawElementsBaseVertex_p.load(RELAX).is_null()
    }

    /// [glDrawElementsBaseVertexEXT](http://docs.gl/es3/glDrawElementsBaseVertexEXT)(mode, count, type_, indices, basevertex)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    /// * alias of: [`glDrawElementsBaseVertex`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawElementsBaseVertexEXT(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        basevertex: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsBaseVertexEXT({:#X}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                count,
                type_,
                indices,
                basevertex
            );
        }
        let out = call_atomic_ptr_5arg(
            "glDrawElementsBaseVertexEXT",
            &glDrawElementsBaseVertexEXT_p,
            mode,
            count,
            type_,
            indices,
            basevertex,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsBaseVertexEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    static glDrawElementsBaseVertexEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsBaseVertexEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawElementsBaseVertexEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsBaseVertexEXT\0",
            &glDrawElementsBaseVertexEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsBaseVertexEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    pub fn glDrawElementsBaseVertexEXT_is_loaded() -> bool {
        !glDrawElementsBaseVertexEXT_p.load(RELAX).is_null()
    }

    /// [glDrawElementsBaseVertexOES](http://docs.gl/es3/glDrawElementsBaseVertexOES)(mode, count, type_, indices, basevertex)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    /// * alias of: [`glDrawElementsBaseVertex`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawElementsBaseVertexOES(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        basevertex: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsBaseVertexOES({:#X}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                count,
                type_,
                indices,
                basevertex
            );
        }
        let out = call_atomic_ptr_5arg(
            "glDrawElementsBaseVertexOES",
            &glDrawElementsBaseVertexOES_p,
            mode,
            count,
            type_,
            indices,
            basevertex,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsBaseVertexOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    static glDrawElementsBaseVertexOES_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsBaseVertexOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawElementsBaseVertexOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsBaseVertexOES\0",
            &glDrawElementsBaseVertexOES_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsBaseVertexOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    pub fn glDrawElementsBaseVertexOES_is_loaded() -> bool {
        !glDrawElementsBaseVertexOES_p.load(RELAX).is_null()
    }

    /// [glDrawElementsIndirect](http://docs.gl/es3/glDrawElementsIndirect)(mode, type_, indirect)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawElementsIndirect(mode: GLenum, type_: GLenum, indirect: *const c_void) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsIndirect({:#X}, {:#X}, {:p});",
                mode,
                type_,
                indirect
            );
        }
        let out = call_atomic_ptr_3arg(
            "glDrawElementsIndirect",
            &glDrawElementsIndirect_p,
            mode,
            type_,
            indirect,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsIndirect");
        }
        out
    }
    static glDrawElementsIndirect_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsIndirect`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawElementsIndirect_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsIndirect\0",
            &glDrawElementsIndirect_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsIndirect`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawElementsIndirect_is_loaded() -> bool {
        !glDrawElementsIndirect_p.load(RELAX).is_null()
    }

    /// [glDrawElementsInstanced](http://docs.gl/es3/glDrawElementsInstanced)(mode, count, type_, indices, instancecount)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawElementsInstanced(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        instancecount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsInstanced({:#X}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                count,
                type_,
                indices,
                instancecount
            );
        }
        let out = call_atomic_ptr_5arg(
            "glDrawElementsInstanced",
            &glDrawElementsInstanced_p,
            mode,
            count,
            type_,
            indices,
            instancecount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsInstanced");
        }
        out
    }
    static glDrawElementsInstanced_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsInstanced`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawElementsInstanced_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsInstanced\0",
            &glDrawElementsInstanced_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsInstanced`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawElementsInstanced_is_loaded() -> bool {
        !glDrawElementsInstanced_p.load(RELAX).is_null()
    }

    /// [glDrawElementsInstancedANGLE](http://docs.gl/es3/glDrawElementsInstancedANGLE)(mode, count, type_, indices, primcount)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: PrimitiveType
    /// * `indices` len: COMPSIZE(count,type)
    /// * alias of: [`glDrawElementsInstanced`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
    pub unsafe fn glDrawElementsInstancedANGLE(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        primcount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsInstancedANGLE({:#X}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                count,
                type_,
                indices,
                primcount
            );
        }
        let out = call_atomic_ptr_5arg(
            "glDrawElementsInstancedANGLE",
            &glDrawElementsInstancedANGLE_p,
            mode,
            count,
            type_,
            indices,
            primcount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsInstancedANGLE");
        }
        out
    }
    #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
    static glDrawElementsInstancedANGLE_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsInstancedANGLE`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
    pub unsafe fn glDrawElementsInstancedANGLE_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsInstancedANGLE\0",
            &glDrawElementsInstancedANGLE_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsInstancedANGLE`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_ANGLE_instanced_arrays"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_ANGLE_instanced_arrays"))))]
    pub fn glDrawElementsInstancedANGLE_is_loaded() -> bool {
        !glDrawElementsInstancedANGLE_p.load(RELAX).is_null()
    }

    /// [glDrawElementsInstancedBaseInstanceEXT](http://docs.gl/es3/glDrawElementsInstancedBaseInstanceEXT)(mode, count, type_, indices, instancecount, baseinstance)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: PrimitiveType
    /// * `indices` len: count
    /// * alias of: [`glDrawElementsInstancedBaseInstance`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    pub unsafe fn glDrawElementsInstancedBaseInstanceEXT(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        instancecount: GLsizei,
        baseinstance: GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsInstancedBaseInstanceEXT({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?});",
                mode,
                count,
                type_,
                indices,
                instancecount,
                baseinstance
            );
        }
        let out = call_atomic_ptr_6arg(
            "glDrawElementsInstancedBaseInstanceEXT",
            &glDrawElementsInstancedBaseInstanceEXT_p,
            mode,
            count,
            type_,
            indices,
            instancecount,
            baseinstance,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsInstancedBaseInstanceEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    static glDrawElementsInstancedBaseInstanceEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsInstancedBaseInstanceEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    pub unsafe fn glDrawElementsInstancedBaseInstanceEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsInstancedBaseInstanceEXT\0",
            &glDrawElementsInstancedBaseInstanceEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsInstancedBaseInstanceEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    pub fn glDrawElementsInstancedBaseInstanceEXT_is_loaded() -> bool {
        !glDrawElementsInstancedBaseInstanceEXT_p
            .load(RELAX)
            .is_null()
    }

    /// [glDrawElementsInstancedBaseVertex](http://docs.gl/es3/glDrawElementsInstancedBaseVertex)(mode, count, type_, indices, instancecount, basevertex)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawElementsInstancedBaseVertex(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        instancecount: GLsizei,
        basevertex: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsInstancedBaseVertex({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?});",
                mode,
                count,
                type_,
                indices,
                instancecount,
                basevertex
            );
        }
        let out = call_atomic_ptr_6arg(
            "glDrawElementsInstancedBaseVertex",
            &glDrawElementsInstancedBaseVertex_p,
            mode,
            count,
            type_,
            indices,
            instancecount,
            basevertex,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsInstancedBaseVertex");
        }
        out
    }
    static glDrawElementsInstancedBaseVertex_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsInstancedBaseVertex`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawElementsInstancedBaseVertex_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsInstancedBaseVertex\0",
            &glDrawElementsInstancedBaseVertex_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsInstancedBaseVertex`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawElementsInstancedBaseVertex_is_loaded() -> bool {
        !glDrawElementsInstancedBaseVertex_p.load(RELAX).is_null()
    }

    /// [glDrawElementsInstancedBaseVertexBaseInstanceEXT](http://docs.gl/es3/glDrawElementsInstancedBaseVertexBaseInstanceEXT)(mode, count, type_, indices, instancecount, basevertex, baseinstance)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: count
    /// * alias of: [`glDrawElementsInstancedBaseVertexBaseInstance`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    pub unsafe fn glDrawElementsInstancedBaseVertexBaseInstanceEXT(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        instancecount: GLsizei,
        basevertex: GLint,
        baseinstance: GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsInstancedBaseVertexBaseInstanceEXT({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?}, {:?});",
                mode,
                count,
                type_,
                indices,
                instancecount,
                basevertex,
                baseinstance
            );
        }
        let out = call_atomic_ptr_7arg(
            "glDrawElementsInstancedBaseVertexBaseInstanceEXT",
            &glDrawElementsInstancedBaseVertexBaseInstanceEXT_p,
            mode,
            count,
            type_,
            indices,
            instancecount,
            basevertex,
            baseinstance,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsInstancedBaseVertexBaseInstanceEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    static glDrawElementsInstancedBaseVertexBaseInstanceEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsInstancedBaseVertexBaseInstanceEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    pub unsafe fn glDrawElementsInstancedBaseVertexBaseInstanceEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsInstancedBaseVertexBaseInstanceEXT\0",
            &glDrawElementsInstancedBaseVertexBaseInstanceEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsInstancedBaseVertexBaseInstanceEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_base_instance"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_base_instance"))))]
    pub fn glDrawElementsInstancedBaseVertexBaseInstanceEXT_is_loaded() -> bool {
        !glDrawElementsInstancedBaseVertexBaseInstanceEXT_p
            .load(RELAX)
            .is_null()
    }

    /// [glDrawElementsInstancedBaseVertexEXT](http://docs.gl/es3/glDrawElementsInstancedBaseVertexEXT)(mode, count, type_, indices, instancecount, basevertex)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    /// * alias of: [`glDrawElementsInstancedBaseVertex`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawElementsInstancedBaseVertexEXT(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        instancecount: GLsizei,
        basevertex: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsInstancedBaseVertexEXT({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?});",
                mode,
                count,
                type_,
                indices,
                instancecount,
                basevertex
            );
        }
        let out = call_atomic_ptr_6arg(
            "glDrawElementsInstancedBaseVertexEXT",
            &glDrawElementsInstancedBaseVertexEXT_p,
            mode,
            count,
            type_,
            indices,
            instancecount,
            basevertex,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsInstancedBaseVertexEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    static glDrawElementsInstancedBaseVertexEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsInstancedBaseVertexEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawElementsInstancedBaseVertexEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsInstancedBaseVertexEXT\0",
            &glDrawElementsInstancedBaseVertexEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsInstancedBaseVertexEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    pub fn glDrawElementsInstancedBaseVertexEXT_is_loaded() -> bool {
        !glDrawElementsInstancedBaseVertexEXT_p.load(RELAX).is_null()
    }

    /// [glDrawElementsInstancedBaseVertexOES](http://docs.gl/es3/glDrawElementsInstancedBaseVertexOES)(mode, count, type_, indices, instancecount, basevertex)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    /// * alias of: [`glDrawElementsInstancedBaseVertex`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawElementsInstancedBaseVertexOES(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        instancecount: GLsizei,
        basevertex: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsInstancedBaseVertexOES({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?});",
                mode,
                count,
                type_,
                indices,
                instancecount,
                basevertex
            );
        }
        let out = call_atomic_ptr_6arg(
            "glDrawElementsInstancedBaseVertexOES",
            &glDrawElementsInstancedBaseVertexOES_p,
            mode,
            count,
            type_,
            indices,
            instancecount,
            basevertex,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsInstancedBaseVertexOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    static glDrawElementsInstancedBaseVertexOES_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsInstancedBaseVertexOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawElementsInstancedBaseVertexOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsInstancedBaseVertexOES\0",
            &glDrawElementsInstancedBaseVertexOES_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsInstancedBaseVertexOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    pub fn glDrawElementsInstancedBaseVertexOES_is_loaded() -> bool {
        !glDrawElementsInstancedBaseVertexOES_p.load(RELAX).is_null()
    }

    /// [glDrawElementsInstancedEXT](http://docs.gl/es3/glDrawElementsInstancedEXT)(mode, count, type_, indices, primcount)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    /// * alias of: [`glDrawElementsInstanced`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays")))
    )]
    pub unsafe fn glDrawElementsInstancedEXT(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        primcount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsInstancedEXT({:#X}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                count,
                type_,
                indices,
                primcount
            );
        }
        let out = call_atomic_ptr_5arg(
            "glDrawElementsInstancedEXT",
            &glDrawElementsInstancedEXT_p,
            mode,
            count,
            type_,
            indices,
            primcount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsInstancedEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays")))
    )]
    static glDrawElementsInstancedEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsInstancedEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays")))
    )]
    pub unsafe fn glDrawElementsInstancedEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsInstancedEXT\0",
            &glDrawElementsInstancedEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsInstancedEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_draw_instanced", feature = "GL_EXT_instanced_arrays")))
    )]
    pub fn glDrawElementsInstancedEXT_is_loaded() -> bool {
        !glDrawElementsInstancedEXT_p.load(RELAX).is_null()
    }

    /// [glDrawElementsInstancedNV](http://docs.gl/es3/glDrawElementsInstancedNV)(mode, count, type_, indices, primcount)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: PrimitiveType
    /// * `indices` len: COMPSIZE(count,type)
    /// * alias of: [`glDrawElementsInstanced`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_draw_instanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
    pub unsafe fn glDrawElementsInstancedNV(
        mode: GLenum,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        primcount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawElementsInstancedNV({:#X}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                count,
                type_,
                indices,
                primcount
            );
        }
        let out = call_atomic_ptr_5arg(
            "glDrawElementsInstancedNV",
            &glDrawElementsInstancedNV_p,
            mode,
            count,
            type_,
            indices,
            primcount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawElementsInstancedNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_draw_instanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
    static glDrawElementsInstancedNV_p: APcv = ap_null();
    /// Tries to load [`glDrawElementsInstancedNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_draw_instanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
    pub unsafe fn glDrawElementsInstancedNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawElementsInstancedNV\0",
            &glDrawElementsInstancedNV_p,
        )
    }
    /// Checks if the pointer for [`glDrawElementsInstancedNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_draw_instanced"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_instanced"))))]
    pub fn glDrawElementsInstancedNV_is_loaded() -> bool {
        !glDrawElementsInstancedNV_p.load(RELAX).is_null()
    }

    /// [glDrawMeshTasksIndirectNV](http://docs.gl/es3/glDrawMeshTasksIndirectNV)(indirect)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_mesh_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
    pub unsafe fn glDrawMeshTasksIndirectNV(indirect: GLintptr) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDrawMeshTasksIndirectNV({:?});", indirect);
        }
        let out = call_atomic_ptr_1arg(
            "glDrawMeshTasksIndirectNV",
            &glDrawMeshTasksIndirectNV_p,
            indirect,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawMeshTasksIndirectNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_mesh_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
    static glDrawMeshTasksIndirectNV_p: APcv = ap_null();
    /// Tries to load [`glDrawMeshTasksIndirectNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_mesh_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
    pub unsafe fn glDrawMeshTasksIndirectNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawMeshTasksIndirectNV\0",
            &glDrawMeshTasksIndirectNV_p,
        )
    }
    /// Checks if the pointer for [`glDrawMeshTasksIndirectNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_mesh_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
    pub fn glDrawMeshTasksIndirectNV_is_loaded() -> bool {
        !glDrawMeshTasksIndirectNV_p.load(RELAX).is_null()
    }

    /// [glDrawMeshTasksNV](http://docs.gl/es3/glDrawMeshTasksNV)(first, count)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_mesh_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
    pub unsafe fn glDrawMeshTasksNV(first: GLuint, count: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDrawMeshTasksNV({:?}, {:?});", first, count);
        }
        let out = call_atomic_ptr_2arg("glDrawMeshTasksNV", &glDrawMeshTasksNV_p, first, count);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawMeshTasksNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_mesh_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
    static glDrawMeshTasksNV_p: APcv = ap_null();
    /// Tries to load [`glDrawMeshTasksNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_mesh_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
    pub unsafe fn glDrawMeshTasksNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawMeshTasksNV\0",
            &glDrawMeshTasksNV_p,
        )
    }
    /// Checks if the pointer for [`glDrawMeshTasksNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_mesh_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_mesh_shader"))))]
    pub fn glDrawMeshTasksNV_is_loaded() -> bool {
        !glDrawMeshTasksNV_p.load(RELAX).is_null()
    }

    /// [glDrawRangeElements](http://docs.gl/es3/glDrawRangeElements)(mode, start, end, count, type_, indices)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawRangeElements(
        mode: GLenum,
        start: GLuint,
        end: GLuint,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawRangeElements({:#X}, {:?}, {:?}, {:?}, {:#X}, {:p});",
                mode,
                start,
                end,
                count,
                type_,
                indices
            );
        }
        let out = call_atomic_ptr_6arg(
            "glDrawRangeElements",
            &glDrawRangeElements_p,
            mode,
            start,
            end,
            count,
            type_,
            indices,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawRangeElements");
        }
        out
    }
    static glDrawRangeElements_p: APcv = ap_null();
    /// Tries to load [`glDrawRangeElements`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawRangeElements_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawRangeElements\0",
            &glDrawRangeElements_p,
        )
    }
    /// Checks if the pointer for [`glDrawRangeElements`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawRangeElements_is_loaded() -> bool {
        !glDrawRangeElements_p.load(RELAX).is_null()
    }

    /// [glDrawRangeElementsBaseVertex](http://docs.gl/es3/glDrawRangeElementsBaseVertex)(mode, start, end, count, type_, indices, basevertex)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glDrawRangeElementsBaseVertex(
        mode: GLenum,
        start: GLuint,
        end: GLuint,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        basevertex: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawRangeElementsBaseVertex({:#X}, {:?}, {:?}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                start,
                end,
                count,
                type_,
                indices,
                basevertex
            );
        }
        let out = call_atomic_ptr_7arg(
            "glDrawRangeElementsBaseVertex",
            &glDrawRangeElementsBaseVertex_p,
            mode,
            start,
            end,
            count,
            type_,
            indices,
            basevertex,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawRangeElementsBaseVertex");
        }
        out
    }
    static glDrawRangeElementsBaseVertex_p: APcv = ap_null();
    /// Tries to load [`glDrawRangeElementsBaseVertex`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glDrawRangeElementsBaseVertex_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawRangeElementsBaseVertex\0",
            &glDrawRangeElementsBaseVertex_p,
        )
    }
    /// Checks if the pointer for [`glDrawRangeElementsBaseVertex`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glDrawRangeElementsBaseVertex_is_loaded() -> bool {
        !glDrawRangeElementsBaseVertex_p.load(RELAX).is_null()
    }

    /// [glDrawRangeElementsBaseVertexEXT](http://docs.gl/es3/glDrawRangeElementsBaseVertexEXT)(mode, start, end, count, type_, indices, basevertex)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    /// * alias of: [`glDrawRangeElementsBaseVertex`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawRangeElementsBaseVertexEXT(
        mode: GLenum,
        start: GLuint,
        end: GLuint,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        basevertex: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawRangeElementsBaseVertexEXT({:#X}, {:?}, {:?}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                start,
                end,
                count,
                type_,
                indices,
                basevertex
            );
        }
        let out = call_atomic_ptr_7arg(
            "glDrawRangeElementsBaseVertexEXT",
            &glDrawRangeElementsBaseVertexEXT_p,
            mode,
            start,
            end,
            count,
            type_,
            indices,
            basevertex,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawRangeElementsBaseVertexEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    static glDrawRangeElementsBaseVertexEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawRangeElementsBaseVertexEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawRangeElementsBaseVertexEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawRangeElementsBaseVertexEXT\0",
            &glDrawRangeElementsBaseVertexEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawRangeElementsBaseVertexEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_elements_base_vertex"))))]
    pub fn glDrawRangeElementsBaseVertexEXT_is_loaded() -> bool {
        !glDrawRangeElementsBaseVertexEXT_p.load(RELAX).is_null()
    }

    /// [glDrawRangeElementsBaseVertexOES](http://docs.gl/es3/glDrawRangeElementsBaseVertexOES)(mode, start, end, count, type_, indices, basevertex)
    /// * `mode` group: PrimitiveType
    /// * `type_` group: DrawElementsType
    /// * `indices` len: COMPSIZE(count,type)
    /// * alias of: [`glDrawRangeElementsBaseVertex`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawRangeElementsBaseVertexOES(
        mode: GLenum,
        start: GLuint,
        end: GLuint,
        count: GLsizei,
        type_: GLenum,
        indices: *const c_void,
        basevertex: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawRangeElementsBaseVertexOES({:#X}, {:?}, {:?}, {:?}, {:#X}, {:p}, {:?});",
                mode,
                start,
                end,
                count,
                type_,
                indices,
                basevertex
            );
        }
        let out = call_atomic_ptr_7arg(
            "glDrawRangeElementsBaseVertexOES",
            &glDrawRangeElementsBaseVertexOES_p,
            mode,
            start,
            end,
            count,
            type_,
            indices,
            basevertex,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawRangeElementsBaseVertexOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    static glDrawRangeElementsBaseVertexOES_p: APcv = ap_null();
    /// Tries to load [`glDrawRangeElementsBaseVertexOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    pub unsafe fn glDrawRangeElementsBaseVertexOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawRangeElementsBaseVertexOES\0",
            &glDrawRangeElementsBaseVertexOES_p,
        )
    }
    /// Checks if the pointer for [`glDrawRangeElementsBaseVertexOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_draw_elements_base_vertex"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_draw_elements_base_vertex"))))]
    pub fn glDrawRangeElementsBaseVertexOES_is_loaded() -> bool {
        !glDrawRangeElementsBaseVertexOES_p.load(RELAX).is_null()
    }

    /// [glDrawTransformFeedbackEXT](http://docs.gl/es3/glDrawTransformFeedbackEXT)(mode, id)
    /// * `mode` group: PrimitiveType
    /// * `id` class: transform feedback
    /// * alias of: [`glDrawTransformFeedback`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
    pub unsafe fn glDrawTransformFeedbackEXT(mode: GLenum, id: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glDrawTransformFeedbackEXT({:#X}, {:?});", mode, id);
        }
        let out = call_atomic_ptr_2arg(
            "glDrawTransformFeedbackEXT",
            &glDrawTransformFeedbackEXT_p,
            mode,
            id,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawTransformFeedbackEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
    static glDrawTransformFeedbackEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawTransformFeedbackEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
    pub unsafe fn glDrawTransformFeedbackEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawTransformFeedbackEXT\0",
            &glDrawTransformFeedbackEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawTransformFeedbackEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
    pub fn glDrawTransformFeedbackEXT_is_loaded() -> bool {
        !glDrawTransformFeedbackEXT_p.load(RELAX).is_null()
    }

    /// [glDrawTransformFeedbackInstancedEXT](http://docs.gl/es3/glDrawTransformFeedbackInstancedEXT)(mode, id, instancecount)
    /// * `mode` group: PrimitiveType
    /// * `id` class: transform feedback
    /// * alias of: [`glDrawTransformFeedbackInstanced`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
    pub unsafe fn glDrawTransformFeedbackInstancedEXT(
        mode: GLenum,
        id: GLuint,
        instancecount: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawTransformFeedbackInstancedEXT({:#X}, {:?}, {:?});",
                mode,
                id,
                instancecount
            );
        }
        let out = call_atomic_ptr_3arg(
            "glDrawTransformFeedbackInstancedEXT",
            &glDrawTransformFeedbackInstancedEXT_p,
            mode,
            id,
            instancecount,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawTransformFeedbackInstancedEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
    static glDrawTransformFeedbackInstancedEXT_p: APcv = ap_null();
    /// Tries to load [`glDrawTransformFeedbackInstancedEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
    pub unsafe fn glDrawTransformFeedbackInstancedEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glDrawTransformFeedbackInstancedEXT\0",
            &glDrawTransformFeedbackInstancedEXT_p,
        )
    }
    /// Checks if the pointer for [`glDrawTransformFeedbackInstancedEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_transform_feedback"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_transform_feedback"))))]
    pub fn glDrawTransformFeedbackInstancedEXT_is_loaded() -> bool {
        !glDrawTransformFeedbackInstancedEXT_p.load(RELAX).is_null()
    }

    /// [glDrawVkImageNV](http://docs.gl/es3/glDrawVkImageNV)(vkImage, sampler, x0, y0, x1, y1, z, s0, t0, s1, t1)
    /// * `sampler` class: sampler
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
    pub unsafe fn glDrawVkImageNV(
        vkImage: GLuint64,
        sampler: GLuint,
        x0: GLfloat,
        y0: GLfloat,
        x1: GLfloat,
        y1: GLfloat,
        z: GLfloat,
        s0: GLfloat,
        t0: GLfloat,
        s1: GLfloat,
        t1: GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glDrawVkImageNV({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                vkImage,
                sampler,
                x0,
                y0,
                x1,
                y1,
                z,
                s0,
                t0,
                s1,
                t1
            );
        }
        let out = call_atomic_ptr_11arg(
            "glDrawVkImageNV",
            &glDrawVkImageNV_p,
            vkImage,
            sampler,
            x0,
            y0,
            x1,
            y1,
            z,
            s0,
            t0,
            s1,
            t1,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glDrawVkImageNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
    static glDrawVkImageNV_p: APcv = ap_null();
    /// Tries to load [`glDrawVkImageNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
    pub unsafe fn glDrawVkImageNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glDrawVkImageNV\0", &glDrawVkImageNV_p)
    }
    /// Checks if the pointer for [`glDrawVkImageNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_draw_vulkan_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_draw_vulkan_image"))))]
    pub fn glDrawVkImageNV_is_loaded() -> bool {
        !glDrawVkImageNV_p.load(RELAX).is_null()
    }

    /// [glEGLImageTargetRenderbufferStorageOES](http://docs.gl/es3/glEGLImageTargetRenderbufferStorageOES)(target, image)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_EGL_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
    pub unsafe fn glEGLImageTargetRenderbufferStorageOES(target: GLenum, image: GLeglImageOES) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glEGLImageTargetRenderbufferStorageOES({:#X}, {:?});",
                target,
                image
            );
        }
        let out = call_atomic_ptr_2arg(
            "glEGLImageTargetRenderbufferStorageOES",
            &glEGLImageTargetRenderbufferStorageOES_p,
            target,
            image,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEGLImageTargetRenderbufferStorageOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_EGL_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
    static glEGLImageTargetRenderbufferStorageOES_p: APcv = ap_null();
    /// Tries to load [`glEGLImageTargetRenderbufferStorageOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_EGL_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
    pub unsafe fn glEGLImageTargetRenderbufferStorageOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEGLImageTargetRenderbufferStorageOES\0",
            &glEGLImageTargetRenderbufferStorageOES_p,
        )
    }
    /// Checks if the pointer for [`glEGLImageTargetRenderbufferStorageOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_EGL_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
    pub fn glEGLImageTargetRenderbufferStorageOES_is_loaded() -> bool {
        !glEGLImageTargetRenderbufferStorageOES_p
            .load(RELAX)
            .is_null()
    }

    /// [glEGLImageTargetTexStorageEXT](http://docs.gl/es3/glEGLImageTargetTexStorageEXT)(target, image, attrib_list)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
    pub unsafe fn glEGLImageTargetTexStorageEXT(
        target: GLenum,
        image: GLeglImageOES,
        attrib_list: *const GLin,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glEGLImageTargetTexStorageEXT({:#X}, {:?}, {:p});",
                target,
                image,
                attrib_list
            );
        }
        let out = call_atomic_ptr_3arg(
            "glEGLImageTargetTexStorageEXT",
            &glEGLImageTargetTexStorageEXT_p,
            target,
            image,
            attrib_list,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEGLImageTargetTexStorageEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
    static glEGLImageTargetTexStorageEXT_p: APcv = ap_null();
    /// Tries to load [`glEGLImageTargetTexStorageEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
    pub unsafe fn glEGLImageTargetTexStorageEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEGLImageTargetTexStorageEXT\0",
            &glEGLImageTargetTexStorageEXT_p,
        )
    }
    /// Checks if the pointer for [`glEGLImageTargetTexStorageEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
    pub fn glEGLImageTargetTexStorageEXT_is_loaded() -> bool {
        !glEGLImageTargetTexStorageEXT_p.load(RELAX).is_null()
    }

    /// [glEGLImageTargetTexture2DOES](http://docs.gl/es3/glEGLImageTargetTexture2DOES)(target, image)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_EGL_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
    pub unsafe fn glEGLImageTargetTexture2DOES(target: GLenum, image: GLeglImageOES) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glEGLImageTargetTexture2DOES({:#X}, {:?});",
                target,
                image
            );
        }
        let out = call_atomic_ptr_2arg(
            "glEGLImageTargetTexture2DOES",
            &glEGLImageTargetTexture2DOES_p,
            target,
            image,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEGLImageTargetTexture2DOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_EGL_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
    static glEGLImageTargetTexture2DOES_p: APcv = ap_null();
    /// Tries to load [`glEGLImageTargetTexture2DOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_EGL_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
    pub unsafe fn glEGLImageTargetTexture2DOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEGLImageTargetTexture2DOES\0",
            &glEGLImageTargetTexture2DOES_p,
        )
    }
    /// Checks if the pointer for [`glEGLImageTargetTexture2DOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_EGL_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_EGL_image"))))]
    pub fn glEGLImageTargetTexture2DOES_is_loaded() -> bool {
        !glEGLImageTargetTexture2DOES_p.load(RELAX).is_null()
    }

    /// [glEGLImageTargetTextureStorageEXT](http://docs.gl/es3/glEGLImageTargetTextureStorageEXT)(texture, image, attrib_list)
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
    pub unsafe fn glEGLImageTargetTextureStorageEXT(
        texture: GLuint,
        image: GLeglImageOES,
        attrib_list: *const GLin,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glEGLImageTargetTextureStorageEXT({:?}, {:?}, {:p});",
                texture,
                image,
                attrib_list
            );
        }
        let out = call_atomic_ptr_3arg(
            "glEGLImageTargetTextureStorageEXT",
            &glEGLImageTargetTextureStorageEXT_p,
            texture,
            image,
            attrib_list,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEGLImageTargetTextureStorageEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
    static glEGLImageTargetTextureStorageEXT_p: APcv = ap_null();
    /// Tries to load [`glEGLImageTargetTextureStorageEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
    pub unsafe fn glEGLImageTargetTextureStorageEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEGLImageTargetTextureStorageEXT\0",
            &glEGLImageTargetTextureStorageEXT_p,
        )
    }
    /// Checks if the pointer for [`glEGLImageTargetTextureStorageEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_EGL_image_storage"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_EGL_image_storage"))))]
    pub fn glEGLImageTargetTextureStorageEXT_is_loaded() -> bool {
        !glEGLImageTargetTextureStorageEXT_p.load(RELAX).is_null()
    }

    /// [glEnable](http://docs.gl/es3/glEnable)(cap)
    /// * `cap` group: EnableCap
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glEnable(cap: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEnable({:#X});", cap);
        }
        let out = call_atomic_ptr_1arg("glEnable", &glEnable_p, cap);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEnable");
        }
        out
    }
    static glEnable_p: APcv = ap_null();
    /// Tries to load [`glEnable`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glEnable_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glEnable\0", &glEnable_p)
    }
    /// Checks if the pointer for [`glEnable`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glEnable_is_loaded() -> bool {
        !glEnable_p.load(RELAX).is_null()
    }

    /// [glEnableDriverControlQCOM](http://docs.gl/es3/glEnableDriverControlQCOM)(driverControl)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub unsafe fn glEnableDriverControlQCOM(driverControl: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEnableDriverControlQCOM({:?});", driverControl);
        }
        let out = call_atomic_ptr_1arg(
            "glEnableDriverControlQCOM",
            &glEnableDriverControlQCOM_p,
            driverControl,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEnableDriverControlQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    static glEnableDriverControlQCOM_p: APcv = ap_null();
    /// Tries to load [`glEnableDriverControlQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub unsafe fn glEnableDriverControlQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEnableDriverControlQCOM\0",
            &glEnableDriverControlQCOM_p,
        )
    }
    /// Checks if the pointer for [`glEnableDriverControlQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub fn glEnableDriverControlQCOM_is_loaded() -> bool {
        !glEnableDriverControlQCOM_p.load(RELAX).is_null()
    }

    /// [glEnableVertexAttribArray](http://docs.gl/es3/glEnableVertexAttribArray)(index)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glEnableVertexAttribArray(index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEnableVertexAttribArray({:?});", index);
        }
        let out = call_atomic_ptr_1arg(
            "glEnableVertexAttribArray",
            &glEnableVertexAttribArray_p,
            index,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEnableVertexAttribArray");
        }
        out
    }
    static glEnableVertexAttribArray_p: APcv = ap_null();
    /// Tries to load [`glEnableVertexAttribArray`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glEnableVertexAttribArray_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEnableVertexAttribArray\0",
            &glEnableVertexAttribArray_p,
        )
    }
    /// Checks if the pointer for [`glEnableVertexAttribArray`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glEnableVertexAttribArray_is_loaded() -> bool {
        !glEnableVertexAttribArray_p.load(RELAX).is_null()
    }

    /// [glEnablei](http://docs.gl/es3/glEnable)(target, index)
    /// * `target` group: EnableCap
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glEnablei(target: GLenum, index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEnablei({:#X}, {:?});", target, index);
        }
        let out = call_atomic_ptr_2arg("glEnablei", &glEnablei_p, target, index);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEnablei");
        }
        out
    }
    static glEnablei_p: APcv = ap_null();
    /// Tries to load [`glEnablei`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glEnablei_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glEnablei\0", &glEnablei_p)
    }
    /// Checks if the pointer for [`glEnablei`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glEnablei_is_loaded() -> bool {
        !glEnablei_p.load(RELAX).is_null()
    }

    /// [glEnableiEXT](http://docs.gl/es3/glEnableiEXT)(target, index)
    /// * `target` group: EnableCap
    /// * alias of: [`glEnablei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glEnableiEXT(target: GLenum, index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEnableiEXT({:#X}, {:?});", target, index);
        }
        let out = call_atomic_ptr_2arg("glEnableiEXT", &glEnableiEXT_p, target, index);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEnableiEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    static glEnableiEXT_p: APcv = ap_null();
    /// Tries to load [`glEnableiEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub unsafe fn glEnableiEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glEnableiEXT\0", &glEnableiEXT_p)
    }
    /// Checks if the pointer for [`glEnableiEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_draw_buffers_indexed"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_draw_buffers_indexed"))))]
    pub fn glEnableiEXT_is_loaded() -> bool {
        !glEnableiEXT_p.load(RELAX).is_null()
    }

    /// [glEnableiNV](http://docs.gl/es3/glEnableiNV)(target, index)
    /// * `target` group: EnableCap
    /// * alias of: [`glEnablei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glEnableiNV(target: GLenum, index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEnableiNV({:#X}, {:?});", target, index);
        }
        let out = call_atomic_ptr_2arg("glEnableiNV", &glEnableiNV_p, target, index);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEnableiNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    static glEnableiNV_p: APcv = ap_null();
    /// Tries to load [`glEnableiNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glEnableiNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glEnableiNV\0", &glEnableiNV_p)
    }
    /// Checks if the pointer for [`glEnableiNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub fn glEnableiNV_is_loaded() -> bool {
        !glEnableiNV_p.load(RELAX).is_null()
    }

    /// [glEnableiOES](http://docs.gl/es3/glEnableiOES)(target, index)
    /// * `target` group: EnableCap
    /// * alias of: [`glEnablei`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(
        feature = "GL_OES_draw_buffers_indexed",
        feature = "GL_OES_viewport_array"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        )))
    )]
    pub unsafe fn glEnableiOES(target: GLenum, index: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEnableiOES({:#X}, {:?});", target, index);
        }
        let out = call_atomic_ptr_2arg("glEnableiOES", &glEnableiOES_p, target, index);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEnableiOES");
        }
        out
    }
    #[cfg(any(
        feature = "GL_OES_draw_buffers_indexed",
        feature = "GL_OES_viewport_array"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        )))
    )]
    static glEnableiOES_p: APcv = ap_null();
    /// Tries to load [`glEnableiOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_OES_draw_buffers_indexed",
        feature = "GL_OES_viewport_array"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        )))
    )]
    pub unsafe fn glEnableiOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glEnableiOES\0", &glEnableiOES_p)
    }
    /// Checks if the pointer for [`glEnableiOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_OES_draw_buffers_indexed",
        feature = "GL_OES_viewport_array"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_OES_draw_buffers_indexed",
            feature = "GL_OES_viewport_array"
        )))
    )]
    pub fn glEnableiOES_is_loaded() -> bool {
        !glEnableiOES_p.load(RELAX).is_null()
    }

    /// [glEndConditionalRenderNV](http://docs.gl/es3/glEndConditionalRenderNV)()
    /// * alias of: [`glEndConditionalRender`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_conditional_render"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
    pub unsafe fn glEndConditionalRenderNV() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEndConditionalRenderNV();",);
        }
        let out = call_atomic_ptr_0arg("glEndConditionalRenderNV", &glEndConditionalRenderNV_p);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEndConditionalRenderNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_conditional_render"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
    static glEndConditionalRenderNV_p: APcv = ap_null();
    /// Tries to load [`glEndConditionalRenderNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_conditional_render"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
    pub unsafe fn glEndConditionalRenderNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEndConditionalRenderNV\0",
            &glEndConditionalRenderNV_p,
        )
    }
    /// Checks if the pointer for [`glEndConditionalRenderNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_conditional_render"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_conditional_render"))))]
    pub fn glEndConditionalRenderNV_is_loaded() -> bool {
        !glEndConditionalRenderNV_p.load(RELAX).is_null()
    }

    /// [glEndPerfMonitorAMD](http://docs.gl/es3/glEndPerfMonitorAMD)(monitor)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glEndPerfMonitorAMD(monitor: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEndPerfMonitorAMD({:?});", monitor);
        }
        let out = call_atomic_ptr_1arg("glEndPerfMonitorAMD", &glEndPerfMonitorAMD_p, monitor);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEndPerfMonitorAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glEndPerfMonitorAMD_p: APcv = ap_null();
    /// Tries to load [`glEndPerfMonitorAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glEndPerfMonitorAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEndPerfMonitorAMD\0",
            &glEndPerfMonitorAMD_p,
        )
    }
    /// Checks if the pointer for [`glEndPerfMonitorAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glEndPerfMonitorAMD_is_loaded() -> bool {
        !glEndPerfMonitorAMD_p.load(RELAX).is_null()
    }

    /// [glEndPerfQueryINTEL](http://docs.gl/es3/glEndPerfQueryINTEL)(queryHandle)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glEndPerfQueryINTEL(queryHandle: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEndPerfQueryINTEL({:?});", queryHandle);
        }
        let out = call_atomic_ptr_1arg("glEndPerfQueryINTEL", &glEndPerfQueryINTEL_p, queryHandle);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEndPerfQueryINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glEndPerfQueryINTEL_p: APcv = ap_null();
    /// Tries to load [`glEndPerfQueryINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glEndPerfQueryINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEndPerfQueryINTEL\0",
            &glEndPerfQueryINTEL_p,
        )
    }
    /// Checks if the pointer for [`glEndPerfQueryINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glEndPerfQueryINTEL_is_loaded() -> bool {
        !glEndPerfQueryINTEL_p.load(RELAX).is_null()
    }

    /// [glEndQuery](http://docs.gl/es3/glEndQuery)(target)
    /// * `target` group: QueryTarget
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glEndQuery(target: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEndQuery({:#X});", target);
        }
        let out = call_atomic_ptr_1arg("glEndQuery", &glEndQuery_p, target);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEndQuery");
        }
        out
    }
    static glEndQuery_p: APcv = ap_null();
    /// Tries to load [`glEndQuery`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glEndQuery_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glEndQuery\0", &glEndQuery_p)
    }
    /// Checks if the pointer for [`glEndQuery`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glEndQuery_is_loaded() -> bool {
        !glEndQuery_p.load(RELAX).is_null()
    }

    /// [glEndQueryEXT](http://docs.gl/es3/glEndQueryEXT)(target)
    /// * `target` group: QueryTarget
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glEndQueryEXT(target: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEndQueryEXT({:#X});", target);
        }
        let out = call_atomic_ptr_1arg("glEndQueryEXT", &glEndQueryEXT_p, target);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEndQueryEXT");
        }
        out
    }
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    static glEndQueryEXT_p: APcv = ap_null();
    /// Tries to load [`glEndQueryEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glEndQueryEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glEndQueryEXT\0", &glEndQueryEXT_p)
    }
    /// Checks if the pointer for [`glEndQueryEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub fn glEndQueryEXT_is_loaded() -> bool {
        !glEndQueryEXT_p.load(RELAX).is_null()
    }

    /// [glEndTilingQCOM](http://docs.gl/es3/glEndTilingQCOM)(preserveMask)
    /// * `preserveMask` group: BufferBitQCOM
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
    pub unsafe fn glEndTilingQCOM(preserveMask: GLbitfield) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEndTilingQCOM({:?});", preserveMask);
        }
        let out = call_atomic_ptr_1arg("glEndTilingQCOM", &glEndTilingQCOM_p, preserveMask);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEndTilingQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
    static glEndTilingQCOM_p: APcv = ap_null();
    /// Tries to load [`glEndTilingQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
    pub unsafe fn glEndTilingQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glEndTilingQCOM\0", &glEndTilingQCOM_p)
    }
    /// Checks if the pointer for [`glEndTilingQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_tiled_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_tiled_rendering"))))]
    pub fn glEndTilingQCOM_is_loaded() -> bool {
        !glEndTilingQCOM_p.load(RELAX).is_null()
    }

    /// [glEndTransformFeedback](http://docs.gl/es3/glEndTransformFeedback)()
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glEndTransformFeedback() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glEndTransformFeedback();",);
        }
        let out = call_atomic_ptr_0arg("glEndTransformFeedback", &glEndTransformFeedback_p);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glEndTransformFeedback");
        }
        out
    }
    static glEndTransformFeedback_p: APcv = ap_null();
    /// Tries to load [`glEndTransformFeedback`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glEndTransformFeedback_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glEndTransformFeedback\0",
            &glEndTransformFeedback_p,
        )
    }
    /// Checks if the pointer for [`glEndTransformFeedback`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glEndTransformFeedback_is_loaded() -> bool {
        !glEndTransformFeedback_p.load(RELAX).is_null()
    }

    /// [glExtGetBufferPointervQCOM](http://docs.gl/es3/glExtGetBufferPointervQCOM)(target, params)
    /// * `params` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetBufferPointervQCOM(target: GLenum, params: *mut *mut c_void) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetBufferPointervQCOM({:#X}, {:p});",
                target,
                params
            );
        }
        let out = call_atomic_ptr_2arg(
            "glExtGetBufferPointervQCOM",
            &glExtGetBufferPointervQCOM_p,
            target,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetBufferPointervQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    static glExtGetBufferPointervQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetBufferPointervQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetBufferPointervQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetBufferPointervQCOM\0",
            &glExtGetBufferPointervQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetBufferPointervQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub fn glExtGetBufferPointervQCOM_is_loaded() -> bool {
        !glExtGetBufferPointervQCOM_p.load(RELAX).is_null()
    }

    /// [glExtGetBuffersQCOM](http://docs.gl/es3/glExtGetBuffersQCOM)(buffers, maxBuffers, numBuffers)
    /// * `buffers` len: maxBuffers
    /// * `buffers` class: buffer
    /// * `numBuffers` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetBuffersQCOM(
        buffers: *mut GLuint,
        maxBuffers: GLint,
        numBuffers: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetBuffersQCOM({:p}, {:?}, {:p});",
                buffers,
                maxBuffers,
                numBuffers
            );
        }
        let out = call_atomic_ptr_3arg(
            "glExtGetBuffersQCOM",
            &glExtGetBuffersQCOM_p,
            buffers,
            maxBuffers,
            numBuffers,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetBuffersQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    static glExtGetBuffersQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetBuffersQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetBuffersQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetBuffersQCOM\0",
            &glExtGetBuffersQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetBuffersQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub fn glExtGetBuffersQCOM_is_loaded() -> bool {
        !glExtGetBuffersQCOM_p.load(RELAX).is_null()
    }

    /// [glExtGetFramebuffersQCOM](http://docs.gl/es3/glExtGetFramebuffersQCOM)(framebuffers, maxFramebuffers, numFramebuffers)
    /// * `framebuffers` len: maxFramebuffers
    /// * `framebuffers` class: framebuffer
    /// * `numFramebuffers` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetFramebuffersQCOM(
        framebuffers: *mut GLuint,
        maxFramebuffers: GLint,
        numFramebuffers: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetFramebuffersQCOM({:p}, {:?}, {:p});",
                framebuffers,
                maxFramebuffers,
                numFramebuffers
            );
        }
        let out = call_atomic_ptr_3arg(
            "glExtGetFramebuffersQCOM",
            &glExtGetFramebuffersQCOM_p,
            framebuffers,
            maxFramebuffers,
            numFramebuffers,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetFramebuffersQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    static glExtGetFramebuffersQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetFramebuffersQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetFramebuffersQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetFramebuffersQCOM\0",
            &glExtGetFramebuffersQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetFramebuffersQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub fn glExtGetFramebuffersQCOM_is_loaded() -> bool {
        !glExtGetFramebuffersQCOM_p.load(RELAX).is_null()
    }

    /// [glExtGetProgramBinarySourceQCOM](http://docs.gl/es3/glExtGetProgramBinarySourceQCOM)(program, shadertype, source, length)
    /// * `program` class: program
    /// * `shadertype` group: ShaderType
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub unsafe fn glExtGetProgramBinarySourceQCOM(
        program: GLuint,
        shadertype: GLenum,
        source: *mut GLchar,
        length: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetProgramBinarySourceQCOM({:?}, {:#X}, {:p}, {:p});",
                program,
                shadertype,
                source,
                length
            );
        }
        let out = call_atomic_ptr_4arg(
            "glExtGetProgramBinarySourceQCOM",
            &glExtGetProgramBinarySourceQCOM_p,
            program,
            shadertype,
            source,
            length,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetProgramBinarySourceQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    static glExtGetProgramBinarySourceQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetProgramBinarySourceQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub unsafe fn glExtGetProgramBinarySourceQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetProgramBinarySourceQCOM\0",
            &glExtGetProgramBinarySourceQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetProgramBinarySourceQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub fn glExtGetProgramBinarySourceQCOM_is_loaded() -> bool {
        !glExtGetProgramBinarySourceQCOM_p.load(RELAX).is_null()
    }

    /// [glExtGetProgramsQCOM](http://docs.gl/es3/glExtGetProgramsQCOM)(programs, maxPrograms, numPrograms)
    /// * `programs` len: maxPrograms
    /// * `programs` class: program
    /// * `numPrograms` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub unsafe fn glExtGetProgramsQCOM(
        programs: *mut GLuint,
        maxPrograms: GLint,
        numPrograms: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetProgramsQCOM({:p}, {:?}, {:p});",
                programs,
                maxPrograms,
                numPrograms
            );
        }
        let out = call_atomic_ptr_3arg(
            "glExtGetProgramsQCOM",
            &glExtGetProgramsQCOM_p,
            programs,
            maxPrograms,
            numPrograms,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetProgramsQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    static glExtGetProgramsQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetProgramsQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub unsafe fn glExtGetProgramsQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetProgramsQCOM\0",
            &glExtGetProgramsQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetProgramsQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub fn glExtGetProgramsQCOM_is_loaded() -> bool {
        !glExtGetProgramsQCOM_p.load(RELAX).is_null()
    }

    /// [glExtGetRenderbuffersQCOM](http://docs.gl/es3/glExtGetRenderbuffersQCOM)(renderbuffers, maxRenderbuffers, numRenderbuffers)
    /// * `renderbuffers` len: maxRenderbuffers
    /// * `renderbuffers` class: renderbuffer
    /// * `numRenderbuffers` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetRenderbuffersQCOM(
        renderbuffers: *mut GLuint,
        maxRenderbuffers: GLint,
        numRenderbuffers: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetRenderbuffersQCOM({:p}, {:?}, {:p});",
                renderbuffers,
                maxRenderbuffers,
                numRenderbuffers
            );
        }
        let out = call_atomic_ptr_3arg(
            "glExtGetRenderbuffersQCOM",
            &glExtGetRenderbuffersQCOM_p,
            renderbuffers,
            maxRenderbuffers,
            numRenderbuffers,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetRenderbuffersQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    static glExtGetRenderbuffersQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetRenderbuffersQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetRenderbuffersQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetRenderbuffersQCOM\0",
            &glExtGetRenderbuffersQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetRenderbuffersQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub fn glExtGetRenderbuffersQCOM_is_loaded() -> bool {
        !glExtGetRenderbuffersQCOM_p.load(RELAX).is_null()
    }

    /// [glExtGetShadersQCOM](http://docs.gl/es3/glExtGetShadersQCOM)(shaders, maxShaders, numShaders)
    /// * `shaders` len: maxShaders
    /// * `shaders` class: shader
    /// * `numShaders` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub unsafe fn glExtGetShadersQCOM(
        shaders: *mut GLuint,
        maxShaders: GLint,
        numShaders: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetShadersQCOM({:p}, {:?}, {:p});",
                shaders,
                maxShaders,
                numShaders
            );
        }
        let out = call_atomic_ptr_3arg(
            "glExtGetShadersQCOM",
            &glExtGetShadersQCOM_p,
            shaders,
            maxShaders,
            numShaders,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetShadersQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    static glExtGetShadersQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetShadersQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub unsafe fn glExtGetShadersQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetShadersQCOM\0",
            &glExtGetShadersQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetShadersQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub fn glExtGetShadersQCOM_is_loaded() -> bool {
        !glExtGetShadersQCOM_p.load(RELAX).is_null()
    }

    /// [glExtGetTexLevelParameterivQCOM](http://docs.gl/es3/glExtGetTexLevelParameterivQCOM)(texture, face, level, pname, params)
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetTexLevelParameterivQCOM(
        texture: GLuint,
        face: GLenum,
        level: GLint,
        pname: GLenum,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetTexLevelParameterivQCOM({:?}, {:#X}, {:?}, {:#X}, {:p});",
                texture,
                face,
                level,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_5arg(
            "glExtGetTexLevelParameterivQCOM",
            &glExtGetTexLevelParameterivQCOM_p,
            texture,
            face,
            level,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetTexLevelParameterivQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    static glExtGetTexLevelParameterivQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetTexLevelParameterivQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetTexLevelParameterivQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetTexLevelParameterivQCOM\0",
            &glExtGetTexLevelParameterivQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetTexLevelParameterivQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub fn glExtGetTexLevelParameterivQCOM_is_loaded() -> bool {
        !glExtGetTexLevelParameterivQCOM_p.load(RELAX).is_null()
    }

    /// [glExtGetTexSubImageQCOM](http://docs.gl/es3/glExtGetTexSubImageQCOM)(target, level, xoffset, yoffset, zoffset, width, height, depth, format, type_, texels)
    /// * `format` group: PixelFormat
    /// * `type_` group: PixelType
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetTexSubImageQCOM(
        target: GLenum,
        level: GLint,
        xoffset: GLint,
        yoffset: GLint,
        zoffset: GLint,
        width: GLsizei,
        height: GLsizei,
        depth: GLsizei,
        format: GLenum,
        type_: GLenum,
        texels: *mut c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetTexSubImageQCOM({:#X}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:#X}, {:#X}, {:p});",
                target,
                level,
                xoffset,
                yoffset,
                zoffset,
                width,
                height,
                depth,
                format,
                type_,
                texels
            );
        }
        let out = call_atomic_ptr_11arg(
            "glExtGetTexSubImageQCOM",
            &glExtGetTexSubImageQCOM_p,
            target,
            level,
            xoffset,
            yoffset,
            zoffset,
            width,
            height,
            depth,
            format,
            type_,
            texels,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetTexSubImageQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    static glExtGetTexSubImageQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetTexSubImageQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetTexSubImageQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetTexSubImageQCOM\0",
            &glExtGetTexSubImageQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetTexSubImageQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub fn glExtGetTexSubImageQCOM_is_loaded() -> bool {
        !glExtGetTexSubImageQCOM_p.load(RELAX).is_null()
    }

    /// [glExtGetTexturesQCOM](http://docs.gl/es3/glExtGetTexturesQCOM)(textures, maxTextures, numTextures)
    /// * `textures` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetTexturesQCOM(
        textures: *mut GLuint,
        maxTextures: GLint,
        numTextures: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtGetTexturesQCOM({:p}, {:?}, {:p});",
                textures,
                maxTextures,
                numTextures
            );
        }
        let out = call_atomic_ptr_3arg(
            "glExtGetTexturesQCOM",
            &glExtGetTexturesQCOM_p,
            textures,
            maxTextures,
            numTextures,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtGetTexturesQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    static glExtGetTexturesQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtGetTexturesQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtGetTexturesQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtGetTexturesQCOM\0",
            &glExtGetTexturesQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtGetTexturesQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub fn glExtGetTexturesQCOM_is_loaded() -> bool {
        !glExtGetTexturesQCOM_p.load(RELAX).is_null()
    }

    /// [glExtIsProgramBinaryQCOM](http://docs.gl/es3/glExtIsProgramBinaryQCOM)(program)
    /// * `program` class: program
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub unsafe fn glExtIsProgramBinaryQCOM(program: GLuint) -> GLboolean {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glExtIsProgramBinaryQCOM({:?});", program);
        }
        let out = call_atomic_ptr_1arg(
            "glExtIsProgramBinaryQCOM",
            &glExtIsProgramBinaryQCOM_p,
            program,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtIsProgramBinaryQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    static glExtIsProgramBinaryQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtIsProgramBinaryQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub unsafe fn glExtIsProgramBinaryQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtIsProgramBinaryQCOM\0",
            &glExtIsProgramBinaryQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtIsProgramBinaryQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get2"))))]
    pub fn glExtIsProgramBinaryQCOM_is_loaded() -> bool {
        !glExtIsProgramBinaryQCOM_p.load(RELAX).is_null()
    }

    /// [glExtTexObjectStateOverrideiQCOM](http://docs.gl/es3/glExtTexObjectStateOverrideiQCOM)(target, pname, param)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtTexObjectStateOverrideiQCOM(target: GLenum, pname: GLenum, param: GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtTexObjectStateOverrideiQCOM({:#X}, {:#X}, {:?});",
                target,
                pname,
                param
            );
        }
        let out = call_atomic_ptr_3arg(
            "glExtTexObjectStateOverrideiQCOM",
            &glExtTexObjectStateOverrideiQCOM_p,
            target,
            pname,
            param,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtTexObjectStateOverrideiQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    static glExtTexObjectStateOverrideiQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtTexObjectStateOverrideiQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub unsafe fn glExtTexObjectStateOverrideiQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtTexObjectStateOverrideiQCOM\0",
            &glExtTexObjectStateOverrideiQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtTexObjectStateOverrideiQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_extended_get"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_extended_get"))))]
    pub fn glExtTexObjectStateOverrideiQCOM_is_loaded() -> bool {
        !glExtTexObjectStateOverrideiQCOM_p.load(RELAX).is_null()
    }

    /// [glExtrapolateTex2DQCOM](http://docs.gl/es3/glExtrapolateTex2DQCOM)(src1, src2, output, scaleFactor)
    /// * `src1` class: texture
    /// * `src2` class: texture
    /// * `output` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_frame_extrapolation"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_frame_extrapolation"))))]
    pub unsafe fn glExtrapolateTex2DQCOM(
        src1: GLuint,
        src2: GLuint,
        output: GLuint,
        scaleFactor: GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glExtrapolateTex2DQCOM({:?}, {:?}, {:?}, {:?});",
                src1,
                src2,
                output,
                scaleFactor
            );
        }
        let out = call_atomic_ptr_4arg(
            "glExtrapolateTex2DQCOM",
            &glExtrapolateTex2DQCOM_p,
            src1,
            src2,
            output,
            scaleFactor,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glExtrapolateTex2DQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_frame_extrapolation"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_frame_extrapolation"))))]
    static glExtrapolateTex2DQCOM_p: APcv = ap_null();
    /// Tries to load [`glExtrapolateTex2DQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_frame_extrapolation"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_frame_extrapolation"))))]
    pub unsafe fn glExtrapolateTex2DQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glExtrapolateTex2DQCOM\0",
            &glExtrapolateTex2DQCOM_p,
        )
    }
    /// Checks if the pointer for [`glExtrapolateTex2DQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_frame_extrapolation"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_frame_extrapolation"))))]
    pub fn glExtrapolateTex2DQCOM_is_loaded() -> bool {
        !glExtrapolateTex2DQCOM_p.load(RELAX).is_null()
    }

    /// [glFenceSync](http://docs.gl/es3/glFenceSync)(condition, flags)
    /// * `condition` group: SyncCondition
    /// * `flags` group: SyncBehaviorFlags
    /// * return value group: sync
    /// * return value class: sync
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFenceSync(condition: GLenum, flags: GLbitfield) -> GLsync {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glFenceSync({:#X}, {:?});", condition, flags);
        }
        let out = call_atomic_ptr_2arg("glFenceSync", &glFenceSync_p, condition, flags);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFenceSync");
        }
        out
    }
    static glFenceSync_p: APcv = ap_null();
    /// Tries to load [`glFenceSync`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFenceSync_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glFenceSync\0", &glFenceSync_p)
    }
    /// Checks if the pointer for [`glFenceSync`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFenceSync_is_loaded() -> bool {
        !glFenceSync_p.load(RELAX).is_null()
    }

    /// [glFenceSyncAPPLE](http://docs.gl/es3/glFenceSyncAPPLE)(condition, flags)
    /// * `condition` group: SyncCondition
    /// * `flags` group: SyncBehaviorFlags
    /// * return value group: sync
    /// * return value class: sync
    /// * alias of: [`glFenceSync`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glFenceSyncAPPLE(condition: GLenum, flags: GLbitfield) -> GLsync {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glFenceSyncAPPLE({:#X}, {:?});", condition, flags);
        }
        let out = call_atomic_ptr_2arg("glFenceSyncAPPLE", &glFenceSyncAPPLE_p, condition, flags);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFenceSyncAPPLE");
        }
        out
    }
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    static glFenceSyncAPPLE_p: APcv = ap_null();
    /// Tries to load [`glFenceSyncAPPLE`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glFenceSyncAPPLE_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glFenceSyncAPPLE\0", &glFenceSyncAPPLE_p)
    }
    /// Checks if the pointer for [`glFenceSyncAPPLE`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub fn glFenceSyncAPPLE_is_loaded() -> bool {
        !glFenceSyncAPPLE_p.load(RELAX).is_null()
    }

    /// [glFinish](http://docs.gl/es3/glFinish)()
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFinish() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glFinish();",);
        }
        let out = call_atomic_ptr_0arg("glFinish", &glFinish_p);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFinish");
        }
        out
    }
    static glFinish_p: APcv = ap_null();
    /// Tries to load [`glFinish`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFinish_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glFinish\0", &glFinish_p)
    }
    /// Checks if the pointer for [`glFinish`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFinish_is_loaded() -> bool {
        !glFinish_p.load(RELAX).is_null()
    }

    /// [glFinishFenceNV](http://docs.gl/es3/glFinishFenceNV)(fence)
    /// * `fence` group: FenceNV
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub unsafe fn glFinishFenceNV(fence: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glFinishFenceNV({:?});", fence);
        }
        let out = call_atomic_ptr_1arg("glFinishFenceNV", &glFinishFenceNV_p, fence);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFinishFenceNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    static glFinishFenceNV_p: APcv = ap_null();
    /// Tries to load [`glFinishFenceNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub unsafe fn glFinishFenceNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glFinishFenceNV\0", &glFinishFenceNV_p)
    }
    /// Checks if the pointer for [`glFinishFenceNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub fn glFinishFenceNV_is_loaded() -> bool {
        !glFinishFenceNV_p.load(RELAX).is_null()
    }

    /// [glFlush](http://docs.gl/es3/glFlush)()
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFlush() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glFlush();",);
        }
        let out = call_atomic_ptr_0arg("glFlush", &glFlush_p);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFlush");
        }
        out
    }
    static glFlush_p: APcv = ap_null();
    /// Tries to load [`glFlush`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFlush_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glFlush\0", &glFlush_p)
    }
    /// Checks if the pointer for [`glFlush`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFlush_is_loaded() -> bool {
        !glFlush_p.load(RELAX).is_null()
    }

    /// [glFlushMappedBufferRange](http://docs.gl/es3/glFlushMappedBufferRange)(target, offset, length)
    /// * `target` group: BufferTargetARB
    /// * `offset` group: BufferOffset
    /// * `length` group: BufferSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFlushMappedBufferRange(target: GLenum, offset: GLintptr, length: GLsizeiptr) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFlushMappedBufferRange({:#X}, {:?}, {:?});",
                target,
                offset,
                length
            );
        }
        let out = call_atomic_ptr_3arg(
            "glFlushMappedBufferRange",
            &glFlushMappedBufferRange_p,
            target,
            offset,
            length,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFlushMappedBufferRange");
        }
        out
    }
    static glFlushMappedBufferRange_p: APcv = ap_null();
    /// Tries to load [`glFlushMappedBufferRange`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFlushMappedBufferRange_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFlushMappedBufferRange\0",
            &glFlushMappedBufferRange_p,
        )
    }
    /// Checks if the pointer for [`glFlushMappedBufferRange`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFlushMappedBufferRange_is_loaded() -> bool {
        !glFlushMappedBufferRange_p.load(RELAX).is_null()
    }

    /// [glFlushMappedBufferRangeEXT](http://docs.gl/es3/glFlushMappedBufferRangeEXT)(target, offset, length)
    /// * `target` group: BufferTargetARB
    /// * alias of: [`glFlushMappedBufferRange`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
    pub unsafe fn glFlushMappedBufferRangeEXT(
        target: GLenum,
        offset: GLintptr,
        length: GLsizeiptr,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFlushMappedBufferRangeEXT({:#X}, {:?}, {:?});",
                target,
                offset,
                length
            );
        }
        let out = call_atomic_ptr_3arg(
            "glFlushMappedBufferRangeEXT",
            &glFlushMappedBufferRangeEXT_p,
            target,
            offset,
            length,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFlushMappedBufferRangeEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
    static glFlushMappedBufferRangeEXT_p: APcv = ap_null();
    /// Tries to load [`glFlushMappedBufferRangeEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
    pub unsafe fn glFlushMappedBufferRangeEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFlushMappedBufferRangeEXT\0",
            &glFlushMappedBufferRangeEXT_p,
        )
    }
    /// Checks if the pointer for [`glFlushMappedBufferRangeEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_map_buffer_range"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_map_buffer_range"))))]
    pub fn glFlushMappedBufferRangeEXT_is_loaded() -> bool {
        !glFlushMappedBufferRangeEXT_p.load(RELAX).is_null()
    }

    /// [glFragmentCoverageColorNV](http://docs.gl/es3/glFragmentCoverageColorNV)(color)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_fragment_coverage_to_color"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fragment_coverage_to_color"))))]
    pub unsafe fn glFragmentCoverageColorNV(color: GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glFragmentCoverageColorNV({:?});", color);
        }
        let out = call_atomic_ptr_1arg(
            "glFragmentCoverageColorNV",
            &glFragmentCoverageColorNV_p,
            color,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFragmentCoverageColorNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_fragment_coverage_to_color"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fragment_coverage_to_color"))))]
    static glFragmentCoverageColorNV_p: APcv = ap_null();
    /// Tries to load [`glFragmentCoverageColorNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fragment_coverage_to_color"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fragment_coverage_to_color"))))]
    pub unsafe fn glFragmentCoverageColorNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFragmentCoverageColorNV\0",
            &glFragmentCoverageColorNV_p,
        )
    }
    /// Checks if the pointer for [`glFragmentCoverageColorNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fragment_coverage_to_color"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fragment_coverage_to_color"))))]
    pub fn glFragmentCoverageColorNV_is_loaded() -> bool {
        !glFragmentCoverageColorNV_p.load(RELAX).is_null()
    }

    /// [glFramebufferFetchBarrierEXT](http://docs.gl/es3/glFramebufferFetchBarrierEXT)()
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent")))
    )]
    pub unsafe fn glFramebufferFetchBarrierEXT() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glFramebufferFetchBarrierEXT();",);
        }
        let out = call_atomic_ptr_0arg(
            "glFramebufferFetchBarrierEXT",
            &glFramebufferFetchBarrierEXT_p,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferFetchBarrierEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent")))
    )]
    static glFramebufferFetchBarrierEXT_p: APcv = ap_null();
    /// Tries to load [`glFramebufferFetchBarrierEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent")))
    )]
    pub unsafe fn glFramebufferFetchBarrierEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferFetchBarrierEXT\0",
            &glFramebufferFetchBarrierEXT_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferFetchBarrierEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_shader_framebuffer_fetch_non_coherent")))
    )]
    pub fn glFramebufferFetchBarrierEXT_is_loaded() -> bool {
        !glFramebufferFetchBarrierEXT_p.load(RELAX).is_null()
    }

    /// [glFramebufferFetchBarrierQCOM](http://docs.gl/es3/glFramebufferFetchBarrierQCOM)()
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent")))
    )]
    pub unsafe fn glFramebufferFetchBarrierQCOM() {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glFramebufferFetchBarrierQCOM();",);
        }
        let out = call_atomic_ptr_0arg(
            "glFramebufferFetchBarrierQCOM",
            &glFramebufferFetchBarrierQCOM_p,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferFetchBarrierQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent")))
    )]
    static glFramebufferFetchBarrierQCOM_p: APcv = ap_null();
    /// Tries to load [`glFramebufferFetchBarrierQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent")))
    )]
    pub unsafe fn glFramebufferFetchBarrierQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferFetchBarrierQCOM\0",
            &glFramebufferFetchBarrierQCOM_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferFetchBarrierQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_QCOM_shader_framebuffer_fetch_noncoherent")))
    )]
    pub fn glFramebufferFetchBarrierQCOM_is_loaded() -> bool {
        !glFramebufferFetchBarrierQCOM_p.load(RELAX).is_null()
    }

    /// [glFramebufferFoveationConfigQCOM](http://docs.gl/es3/glFramebufferFoveationConfigQCOM)(framebuffer, numLayers, focalPointsPerLayer, requestedFeatures, providedFeatures)
    /// * `framebuffer` group: Framebuffer
    /// * `framebuffer` class: framebuffer
    /// * `providedFeatures` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
    pub unsafe fn glFramebufferFoveationConfigQCOM(
        framebuffer: GLuint,
        numLayers: GLuint,
        focalPointsPerLayer: GLuint,
        requestedFeatures: GLuint,
        providedFeatures: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferFoveationConfigQCOM({:?}, {:?}, {:?}, {:?}, {:p});",
                framebuffer,
                numLayers,
                focalPointsPerLayer,
                requestedFeatures,
                providedFeatures
            );
        }
        let out = call_atomic_ptr_5arg(
            "glFramebufferFoveationConfigQCOM",
            &glFramebufferFoveationConfigQCOM_p,
            framebuffer,
            numLayers,
            focalPointsPerLayer,
            requestedFeatures,
            providedFeatures,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferFoveationConfigQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
    static glFramebufferFoveationConfigQCOM_p: APcv = ap_null();
    /// Tries to load [`glFramebufferFoveationConfigQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
    pub unsafe fn glFramebufferFoveationConfigQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferFoveationConfigQCOM\0",
            &glFramebufferFoveationConfigQCOM_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferFoveationConfigQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
    pub fn glFramebufferFoveationConfigQCOM_is_loaded() -> bool {
        !glFramebufferFoveationConfigQCOM_p.load(RELAX).is_null()
    }

    /// [glFramebufferFoveationParametersQCOM](http://docs.gl/es3/glFramebufferFoveationParametersQCOM)(framebuffer, layer, focalPoint, focalX, focalY, gainX, gainY, foveaArea)
    /// * `framebuffer` group: Framebuffer
    /// * `framebuffer` class: framebuffer
    /// * `focalX` group: CheckedFloat32
    /// * `focalY` group: CheckedFloat32
    /// * `gainX` group: CheckedFloat32
    /// * `gainY` group: CheckedFloat32
    /// * `foveaArea` group: CheckedFloat32
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
    pub unsafe fn glFramebufferFoveationParametersQCOM(
        framebuffer: GLuint,
        layer: GLuint,
        focalPoint: GLuint,
        focalX: GLfloat,
        focalY: GLfloat,
        gainX: GLfloat,
        gainY: GLfloat,
        foveaArea: GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferFoveationParametersQCOM({:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?}, {:?});",
                framebuffer,
                layer,
                focalPoint,
                focalX,
                focalY,
                gainX,
                gainY,
                foveaArea
            );
        }
        let out = call_atomic_ptr_8arg(
            "glFramebufferFoveationParametersQCOM",
            &glFramebufferFoveationParametersQCOM_p,
            framebuffer,
            layer,
            focalPoint,
            focalX,
            focalY,
            gainX,
            gainY,
            foveaArea,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferFoveationParametersQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
    static glFramebufferFoveationParametersQCOM_p: APcv = ap_null();
    /// Tries to load [`glFramebufferFoveationParametersQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
    pub unsafe fn glFramebufferFoveationParametersQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferFoveationParametersQCOM\0",
            &glFramebufferFoveationParametersQCOM_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferFoveationParametersQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_framebuffer_foveated"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_framebuffer_foveated"))))]
    pub fn glFramebufferFoveationParametersQCOM_is_loaded() -> bool {
        !glFramebufferFoveationParametersQCOM_p.load(RELAX).is_null()
    }

    /// [glFramebufferParameteri](http://docs.gl/es3/glFramebufferParameter)(target, pname, param)
    /// * `target` group: FramebufferTarget
    /// * `pname` group: FramebufferParameterName
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFramebufferParameteri(target: GLenum, pname: GLenum, param: GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferParameteri({:#X}, {:#X}, {:?});",
                target,
                pname,
                param
            );
        }
        let out = call_atomic_ptr_3arg(
            "glFramebufferParameteri",
            &glFramebufferParameteri_p,
            target,
            pname,
            param,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferParameteri");
        }
        out
    }
    static glFramebufferParameteri_p: APcv = ap_null();
    /// Tries to load [`glFramebufferParameteri`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFramebufferParameteri_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferParameteri\0",
            &glFramebufferParameteri_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferParameteri`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFramebufferParameteri_is_loaded() -> bool {
        !glFramebufferParameteri_p.load(RELAX).is_null()
    }

    /// [glFramebufferParameteriMESA](http://docs.gl/es3/glFramebufferParameteriMESA)(target, pname, param)
    /// * `target` group: FramebufferTarget
    /// * `pname` group: FramebufferParameterName
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
    pub unsafe fn glFramebufferParameteriMESA(target: GLenum, pname: GLenum, param: GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferParameteriMESA({:#X}, {:#X}, {:?});",
                target,
                pname,
                param
            );
        }
        let out = call_atomic_ptr_3arg(
            "glFramebufferParameteriMESA",
            &glFramebufferParameteriMESA_p,
            target,
            pname,
            param,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferParameteriMESA");
        }
        out
    }
    #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
    static glFramebufferParameteriMESA_p: APcv = ap_null();
    /// Tries to load [`glFramebufferParameteriMESA`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
    pub unsafe fn glFramebufferParameteriMESA_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferParameteriMESA\0",
            &glFramebufferParameteriMESA_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferParameteriMESA`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
    pub fn glFramebufferParameteriMESA_is_loaded() -> bool {
        !glFramebufferParameteriMESA_p.load(RELAX).is_null()
    }

    /// [glFramebufferPixelLocalStorageSizeEXT](http://docs.gl/es3/glFramebufferPixelLocalStorageSizeEXT)(target, size)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    pub unsafe fn glFramebufferPixelLocalStorageSizeEXT(target: GLuint, size: GLsizei) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferPixelLocalStorageSizeEXT({:?}, {:?});",
                target,
                size
            );
        }
        let out = call_atomic_ptr_2arg(
            "glFramebufferPixelLocalStorageSizeEXT",
            &glFramebufferPixelLocalStorageSizeEXT_p,
            target,
            size,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferPixelLocalStorageSizeEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    static glFramebufferPixelLocalStorageSizeEXT_p: APcv = ap_null();
    /// Tries to load [`glFramebufferPixelLocalStorageSizeEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    pub unsafe fn glFramebufferPixelLocalStorageSizeEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferPixelLocalStorageSizeEXT\0",
            &glFramebufferPixelLocalStorageSizeEXT_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferPixelLocalStorageSizeEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    pub fn glFramebufferPixelLocalStorageSizeEXT_is_loaded() -> bool {
        !glFramebufferPixelLocalStorageSizeEXT_p
            .load(RELAX)
            .is_null()
    }

    /// [glFramebufferRenderbuffer](http://docs.gl/es3/glFramebufferRenderbuffer)(target, attachment, renderbuffertarget, renderbuffer)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `renderbuffertarget` group: RenderbufferTarget
    /// * `renderbuffer` class: renderbuffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFramebufferRenderbuffer(
        target: GLenum,
        attachment: GLenum,
        renderbuffertarget: GLenum,
        renderbuffer: GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferRenderbuffer({:#X}, {:#X}, {:#X}, {:?});",
                target,
                attachment,
                renderbuffertarget,
                renderbuffer
            );
        }
        let out = call_atomic_ptr_4arg(
            "glFramebufferRenderbuffer",
            &glFramebufferRenderbuffer_p,
            target,
            attachment,
            renderbuffertarget,
            renderbuffer,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferRenderbuffer");
        }
        out
    }
    static glFramebufferRenderbuffer_p: APcv = ap_null();
    /// Tries to load [`glFramebufferRenderbuffer`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFramebufferRenderbuffer_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferRenderbuffer\0",
            &glFramebufferRenderbuffer_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferRenderbuffer`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFramebufferRenderbuffer_is_loaded() -> bool {
        !glFramebufferRenderbuffer_p.load(RELAX).is_null()
    }

    /// [glFramebufferSampleLocationsfvNV](http://docs.gl/es3/glFramebufferSampleLocationsfvNV)(target, start, count, v)
    /// * `target` group: FramebufferTarget
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_sample_locations"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
    pub unsafe fn glFramebufferSampleLocationsfvNV(
        target: GLenum,
        start: GLuint,
        count: GLsizei,
        v: *const GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferSampleLocationsfvNV({:#X}, {:?}, {:?}, {:p});",
                target,
                start,
                count,
                v
            );
        }
        let out = call_atomic_ptr_4arg(
            "glFramebufferSampleLocationsfvNV",
            &glFramebufferSampleLocationsfvNV_p,
            target,
            start,
            count,
            v,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferSampleLocationsfvNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_sample_locations"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
    static glFramebufferSampleLocationsfvNV_p: APcv = ap_null();
    /// Tries to load [`glFramebufferSampleLocationsfvNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_sample_locations"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
    pub unsafe fn glFramebufferSampleLocationsfvNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferSampleLocationsfvNV\0",
            &glFramebufferSampleLocationsfvNV_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferSampleLocationsfvNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_sample_locations"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_sample_locations"))))]
    pub fn glFramebufferSampleLocationsfvNV_is_loaded() -> bool {
        !glFramebufferSampleLocationsfvNV_p.load(RELAX).is_null()
    }

    /// [glFramebufferTexture](http://docs.gl/es3/glFramebufferTexture)(target, attachment, texture, level)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFramebufferTexture(
        target: GLenum,
        attachment: GLenum,
        texture: GLuint,
        level: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTexture({:#X}, {:#X}, {:?}, {:?});",
                target,
                attachment,
                texture,
                level
            );
        }
        let out = call_atomic_ptr_4arg(
            "glFramebufferTexture",
            &glFramebufferTexture_p,
            target,
            attachment,
            texture,
            level,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTexture");
        }
        out
    }
    static glFramebufferTexture_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTexture`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFramebufferTexture_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTexture\0",
            &glFramebufferTexture_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTexture`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFramebufferTexture_is_loaded() -> bool {
        !glFramebufferTexture_p.load(RELAX).is_null()
    }

    /// [glFramebufferTexture2D](http://docs.gl/es3/glFramebufferTexture2D)(target, attachment, textarget, texture, level)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `textarget` group: TextureTarget
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFramebufferTexture2D(
        target: GLenum,
        attachment: GLenum,
        textarget: GLenum,
        texture: GLuint,
        level: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTexture2D({:#X}, {:#X}, {:#X}, {:?}, {:?});",
                target,
                attachment,
                textarget,
                texture,
                level
            );
        }
        let out = call_atomic_ptr_5arg(
            "glFramebufferTexture2D",
            &glFramebufferTexture2D_p,
            target,
            attachment,
            textarget,
            texture,
            level,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTexture2D");
        }
        out
    }
    static glFramebufferTexture2D_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTexture2D`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFramebufferTexture2D_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTexture2D\0",
            &glFramebufferTexture2D_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTexture2D`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFramebufferTexture2D_is_loaded() -> bool {
        !glFramebufferTexture2D_p.load(RELAX).is_null()
    }

    /// [glFramebufferTexture2DDownsampleIMG](http://docs.gl/es3/glFramebufferTexture2DDownsampleIMG)(target, attachment, textarget, texture, level, xscale, yscale)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `textarget` group: TextureTarget
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
    pub unsafe fn glFramebufferTexture2DDownsampleIMG(
        target: GLenum,
        attachment: GLenum,
        textarget: GLenum,
        texture: GLuint,
        level: GLint,
        xscale: GLint,
        yscale: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTexture2DDownsampleIMG({:#X}, {:#X}, {:#X}, {:?}, {:?}, {:?}, {:?});",
                target,
                attachment,
                textarget,
                texture,
                level,
                xscale,
                yscale
            );
        }
        let out = call_atomic_ptr_7arg(
            "glFramebufferTexture2DDownsampleIMG",
            &glFramebufferTexture2DDownsampleIMG_p,
            target,
            attachment,
            textarget,
            texture,
            level,
            xscale,
            yscale,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTexture2DDownsampleIMG");
        }
        out
    }
    #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
    static glFramebufferTexture2DDownsampleIMG_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTexture2DDownsampleIMG`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
    pub unsafe fn glFramebufferTexture2DDownsampleIMG_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTexture2DDownsampleIMG\0",
            &glFramebufferTexture2DDownsampleIMG_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTexture2DDownsampleIMG`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
    pub fn glFramebufferTexture2DDownsampleIMG_is_loaded() -> bool {
        !glFramebufferTexture2DDownsampleIMG_p.load(RELAX).is_null()
    }

    /// [glFramebufferTexture2DMultisampleEXT](http://docs.gl/es3/glFramebufferTexture2DMultisampleEXT)(target, attachment, textarget, texture, level, samples)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `textarget` group: TextureTarget
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
    )]
    pub unsafe fn glFramebufferTexture2DMultisampleEXT(
        target: GLenum,
        attachment: GLenum,
        textarget: GLenum,
        texture: GLuint,
        level: GLint,
        samples: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTexture2DMultisampleEXT({:#X}, {:#X}, {:#X}, {:?}, {:?}, {:?});",
                target,
                attachment,
                textarget,
                texture,
                level,
                samples
            );
        }
        let out = call_atomic_ptr_6arg(
            "glFramebufferTexture2DMultisampleEXT",
            &glFramebufferTexture2DMultisampleEXT_p,
            target,
            attachment,
            textarget,
            texture,
            level,
            samples,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTexture2DMultisampleEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
    )]
    static glFramebufferTexture2DMultisampleEXT_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTexture2DMultisampleEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
    )]
    pub unsafe fn glFramebufferTexture2DMultisampleEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTexture2DMultisampleEXT\0",
            &glFramebufferTexture2DMultisampleEXT_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTexture2DMultisampleEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_EXT_multisampled_render_to_texture")))
    )]
    pub fn glFramebufferTexture2DMultisampleEXT_is_loaded() -> bool {
        !glFramebufferTexture2DMultisampleEXT_p.load(RELAX).is_null()
    }

    /// [glFramebufferTexture2DMultisampleIMG](http://docs.gl/es3/glFramebufferTexture2DMultisampleIMG)(target, attachment, textarget, texture, level, samples)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `textarget` group: TextureTarget
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
    )]
    pub unsafe fn glFramebufferTexture2DMultisampleIMG(
        target: GLenum,
        attachment: GLenum,
        textarget: GLenum,
        texture: GLuint,
        level: GLint,
        samples: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTexture2DMultisampleIMG({:#X}, {:#X}, {:#X}, {:?}, {:?}, {:?});",
                target,
                attachment,
                textarget,
                texture,
                level,
                samples
            );
        }
        let out = call_atomic_ptr_6arg(
            "glFramebufferTexture2DMultisampleIMG",
            &glFramebufferTexture2DMultisampleIMG_p,
            target,
            attachment,
            textarget,
            texture,
            level,
            samples,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTexture2DMultisampleIMG");
        }
        out
    }
    #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
    )]
    static glFramebufferTexture2DMultisampleIMG_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTexture2DMultisampleIMG`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
    )]
    pub unsafe fn glFramebufferTexture2DMultisampleIMG_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTexture2DMultisampleIMG\0",
            &glFramebufferTexture2DMultisampleIMG_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTexture2DMultisampleIMG`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_IMG_multisampled_render_to_texture")))
    )]
    pub fn glFramebufferTexture2DMultisampleIMG_is_loaded() -> bool {
        !glFramebufferTexture2DMultisampleIMG_p.load(RELAX).is_null()
    }

    /// [glFramebufferTexture3DOES](http://docs.gl/es3/glFramebufferTexture3DOES)(target, attachment, textarget, texture, level, zoffset)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `textarget` group: TextureTarget
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub unsafe fn glFramebufferTexture3DOES(
        target: GLenum,
        attachment: GLenum,
        textarget: GLenum,
        texture: GLuint,
        level: GLint,
        zoffset: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTexture3DOES({:#X}, {:#X}, {:#X}, {:?}, {:?}, {:?});",
                target,
                attachment,
                textarget,
                texture,
                level,
                zoffset
            );
        }
        let out = call_atomic_ptr_6arg(
            "glFramebufferTexture3DOES",
            &glFramebufferTexture3DOES_p,
            target,
            attachment,
            textarget,
            texture,
            level,
            zoffset,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTexture3DOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    static glFramebufferTexture3DOES_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTexture3DOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub unsafe fn glFramebufferTexture3DOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTexture3DOES\0",
            &glFramebufferTexture3DOES_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTexture3DOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_3D"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_3D"))))]
    pub fn glFramebufferTexture3DOES_is_loaded() -> bool {
        !glFramebufferTexture3DOES_p.load(RELAX).is_null()
    }

    /// [glFramebufferTextureEXT](http://docs.gl/es3/glFramebufferTextureEXT)(target, attachment, texture, level)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `texture` group: Texture
    /// * `texture` class: texture
    /// * `level` group: CheckedInt32
    /// * alias of: [`glFramebufferTexture`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_geometry_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_geometry_shader"))))]
    pub unsafe fn glFramebufferTextureEXT(
        target: GLenum,
        attachment: GLenum,
        texture: GLuint,
        level: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTextureEXT({:#X}, {:#X}, {:?}, {:?});",
                target,
                attachment,
                texture,
                level
            );
        }
        let out = call_atomic_ptr_4arg(
            "glFramebufferTextureEXT",
            &glFramebufferTextureEXT_p,
            target,
            attachment,
            texture,
            level,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTextureEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_geometry_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_geometry_shader"))))]
    static glFramebufferTextureEXT_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTextureEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_geometry_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_geometry_shader"))))]
    pub unsafe fn glFramebufferTextureEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTextureEXT\0",
            &glFramebufferTextureEXT_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTextureEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_geometry_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_geometry_shader"))))]
    pub fn glFramebufferTextureEXT_is_loaded() -> bool {
        !glFramebufferTextureEXT_p.load(RELAX).is_null()
    }

    /// [glFramebufferTextureLayer](http://docs.gl/es3/glFramebufferTextureLayer)(target, attachment, texture, level, layer)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `texture` group: Texture
    /// * `texture` class: texture
    /// * `level` group: CheckedInt32
    /// * `layer` group: CheckedInt32
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFramebufferTextureLayer(
        target: GLenum,
        attachment: GLenum,
        texture: GLuint,
        level: GLint,
        layer: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTextureLayer({:#X}, {:#X}, {:?}, {:?}, {:?});",
                target,
                attachment,
                texture,
                level,
                layer
            );
        }
        let out = call_atomic_ptr_5arg(
            "glFramebufferTextureLayer",
            &glFramebufferTextureLayer_p,
            target,
            attachment,
            texture,
            level,
            layer,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTextureLayer");
        }
        out
    }
    static glFramebufferTextureLayer_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTextureLayer`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFramebufferTextureLayer_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTextureLayer\0",
            &glFramebufferTextureLayer_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTextureLayer`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFramebufferTextureLayer_is_loaded() -> bool {
        !glFramebufferTextureLayer_p.load(RELAX).is_null()
    }

    /// [glFramebufferTextureLayerDownsampleIMG](http://docs.gl/es3/glFramebufferTextureLayerDownsampleIMG)(target, attachment, texture, level, layer, xscale, yscale)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `texture` group: Texture
    /// * `texture` class: texture
    /// * `level` group: CheckedInt32
    /// * `layer` group: CheckedInt32
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
    pub unsafe fn glFramebufferTextureLayerDownsampleIMG(
        target: GLenum,
        attachment: GLenum,
        texture: GLuint,
        level: GLint,
        layer: GLint,
        xscale: GLint,
        yscale: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTextureLayerDownsampleIMG({:#X}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?});",
                target,
                attachment,
                texture,
                level,
                layer,
                xscale,
                yscale
            );
        }
        let out = call_atomic_ptr_7arg(
            "glFramebufferTextureLayerDownsampleIMG",
            &glFramebufferTextureLayerDownsampleIMG_p,
            target,
            attachment,
            texture,
            level,
            layer,
            xscale,
            yscale,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTextureLayerDownsampleIMG");
        }
        out
    }
    #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
    static glFramebufferTextureLayerDownsampleIMG_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTextureLayerDownsampleIMG`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
    pub unsafe fn glFramebufferTextureLayerDownsampleIMG_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTextureLayerDownsampleIMG\0",
            &glFramebufferTextureLayerDownsampleIMG_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTextureLayerDownsampleIMG`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_framebuffer_downsample"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_framebuffer_downsample"))))]
    pub fn glFramebufferTextureLayerDownsampleIMG_is_loaded() -> bool {
        !glFramebufferTextureLayerDownsampleIMG_p
            .load(RELAX)
            .is_null()
    }

    /// [glFramebufferTextureMultisampleMultiviewOVR](http://docs.gl/es3/glFramebufferTextureMultisampleMultiviewOVR)(target, attachment, texture, level, samples, baseViewIndex, numViews)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `texture` group: Texture
    /// * `texture` class: texture
    /// * `level` group: CheckedInt32
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture")))
    )]
    pub unsafe fn glFramebufferTextureMultisampleMultiviewOVR(
        target: GLenum,
        attachment: GLenum,
        texture: GLuint,
        level: GLint,
        samples: GLsizei,
        baseViewIndex: GLint,
        numViews: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTextureMultisampleMultiviewOVR({:#X}, {:#X}, {:?}, {:?}, {:?}, {:?}, {:?});",
                target,
                attachment,
                texture,
                level,
                samples,
                baseViewIndex,
                numViews
            );
        }
        let out = call_atomic_ptr_7arg(
            "glFramebufferTextureMultisampleMultiviewOVR",
            &glFramebufferTextureMultisampleMultiviewOVR_p,
            target,
            attachment,
            texture,
            level,
            samples,
            baseViewIndex,
            numViews,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTextureMultisampleMultiviewOVR");
        }
        out
    }
    #[cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture")))
    )]
    static glFramebufferTextureMultisampleMultiviewOVR_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTextureMultisampleMultiviewOVR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture")))
    )]
    pub unsafe fn glFramebufferTextureMultisampleMultiviewOVR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTextureMultisampleMultiviewOVR\0",
            &glFramebufferTextureMultisampleMultiviewOVR_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTextureMultisampleMultiviewOVR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture"))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(feature = "GL_OVR_multiview_multisampled_render_to_texture")))
    )]
    pub fn glFramebufferTextureMultisampleMultiviewOVR_is_loaded() -> bool {
        !glFramebufferTextureMultisampleMultiviewOVR_p
            .load(RELAX)
            .is_null()
    }

    /// [glFramebufferTextureMultiviewOVR](http://docs.gl/es3/glFramebufferTextureMultiviewOVR)(target, attachment, texture, level, baseViewIndex, numViews)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `texture` group: Texture
    /// * `texture` class: texture
    /// * `level` group: CheckedInt32
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OVR_multiview"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OVR_multiview"))))]
    pub unsafe fn glFramebufferTextureMultiviewOVR(
        target: GLenum,
        attachment: GLenum,
        texture: GLuint,
        level: GLint,
        baseViewIndex: GLint,
        numViews: GLsizei,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTextureMultiviewOVR({:#X}, {:#X}, {:?}, {:?}, {:?}, {:?});",
                target,
                attachment,
                texture,
                level,
                baseViewIndex,
                numViews
            );
        }
        let out = call_atomic_ptr_6arg(
            "glFramebufferTextureMultiviewOVR",
            &glFramebufferTextureMultiviewOVR_p,
            target,
            attachment,
            texture,
            level,
            baseViewIndex,
            numViews,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTextureMultiviewOVR");
        }
        out
    }
    #[cfg(any(feature = "GL_OVR_multiview"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OVR_multiview"))))]
    static glFramebufferTextureMultiviewOVR_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTextureMultiviewOVR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OVR_multiview"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OVR_multiview"))))]
    pub unsafe fn glFramebufferTextureMultiviewOVR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTextureMultiviewOVR\0",
            &glFramebufferTextureMultiviewOVR_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTextureMultiviewOVR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OVR_multiview"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OVR_multiview"))))]
    pub fn glFramebufferTextureMultiviewOVR_is_loaded() -> bool {
        !glFramebufferTextureMultiviewOVR_p.load(RELAX).is_null()
    }

    /// [glFramebufferTextureOES](http://docs.gl/es3/glFramebufferTextureOES)(target, attachment, texture, level)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `texture` group: Texture
    /// * `texture` class: texture
    /// * `level` group: CheckedInt32
    /// * alias of: [`glFramebufferTexture`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_geometry_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_geometry_shader"))))]
    pub unsafe fn glFramebufferTextureOES(
        target: GLenum,
        attachment: GLenum,
        texture: GLuint,
        level: GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glFramebufferTextureOES({:#X}, {:#X}, {:?}, {:?});",
                target,
                attachment,
                texture,
                level
            );
        }
        let out = call_atomic_ptr_4arg(
            "glFramebufferTextureOES",
            &glFramebufferTextureOES_p,
            target,
            attachment,
            texture,
            level,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFramebufferTextureOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_geometry_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_geometry_shader"))))]
    static glFramebufferTextureOES_p: APcv = ap_null();
    /// Tries to load [`glFramebufferTextureOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_geometry_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_geometry_shader"))))]
    pub unsafe fn glFramebufferTextureOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glFramebufferTextureOES\0",
            &glFramebufferTextureOES_p,
        )
    }
    /// Checks if the pointer for [`glFramebufferTextureOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_geometry_shader"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_geometry_shader"))))]
    pub fn glFramebufferTextureOES_is_loaded() -> bool {
        !glFramebufferTextureOES_p.load(RELAX).is_null()
    }

    /// [glFrontFace](http://docs.gl/es3/glFrontFace)(mode)
    /// * `mode` group: FrontFaceDirection
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glFrontFace(mode: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glFrontFace({:#X});", mode);
        }
        let out = call_atomic_ptr_1arg("glFrontFace", &glFrontFace_p, mode);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glFrontFace");
        }
        out
    }
    static glFrontFace_p: APcv = ap_null();
    /// Tries to load [`glFrontFace`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glFrontFace_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glFrontFace\0", &glFrontFace_p)
    }
    /// Checks if the pointer for [`glFrontFace`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glFrontFace_is_loaded() -> bool {
        !glFrontFace_p.load(RELAX).is_null()
    }

    /// [glGenBuffers](http://docs.gl/es3/glGenBuffers)(n, buffers)
    /// * `buffers` len: n
    /// * `buffers` class: buffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenBuffers(n: GLsizei, buffers: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenBuffers({:?}, {:p});", n, buffers);
        }
        let out = call_atomic_ptr_2arg("glGenBuffers", &glGenBuffers_p, n, buffers);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenBuffers");
        }
        out
    }
    static glGenBuffers_p: APcv = ap_null();
    /// Tries to load [`glGenBuffers`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenBuffers_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGenBuffers\0", &glGenBuffers_p)
    }
    /// Checks if the pointer for [`glGenBuffers`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenBuffers_is_loaded() -> bool {
        !glGenBuffers_p.load(RELAX).is_null()
    }

    /// [glGenFencesNV](http://docs.gl/es3/glGenFencesNV)(n, fences)
    /// * `fences` group: FenceNV
    /// * `fences` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub unsafe fn glGenFencesNV(n: GLsizei, fences: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenFencesNV({:?}, {:p});", n, fences);
        }
        let out = call_atomic_ptr_2arg("glGenFencesNV", &glGenFencesNV_p, n, fences);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenFencesNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    static glGenFencesNV_p: APcv = ap_null();
    /// Tries to load [`glGenFencesNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub unsafe fn glGenFencesNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGenFencesNV\0", &glGenFencesNV_p)
    }
    /// Checks if the pointer for [`glGenFencesNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub fn glGenFencesNV_is_loaded() -> bool {
        !glGenFencesNV_p.load(RELAX).is_null()
    }

    /// [glGenFramebuffers](http://docs.gl/es3/glGenFramebuffers)(n, framebuffers)
    /// * `framebuffers` len: n
    /// * `framebuffers` class: framebuffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenFramebuffers(n: GLsizei, framebuffers: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenFramebuffers({:?}, {:p});", n, framebuffers);
        }
        let out = call_atomic_ptr_2arg("glGenFramebuffers", &glGenFramebuffers_p, n, framebuffers);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenFramebuffers");
        }
        out
    }
    static glGenFramebuffers_p: APcv = ap_null();
    /// Tries to load [`glGenFramebuffers`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenFramebuffers_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGenFramebuffers\0",
            &glGenFramebuffers_p,
        )
    }
    /// Checks if the pointer for [`glGenFramebuffers`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenFramebuffers_is_loaded() -> bool {
        !glGenFramebuffers_p.load(RELAX).is_null()
    }

    /// [glGenPathsNV](http://docs.gl/es3/glGenPathsNV)(range)
    /// * return value group: Path
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGenPathsNV(range: GLsizei) -> GLuint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenPathsNV({:?});", range);
        }
        let out = call_atomic_ptr_1arg("glGenPathsNV", &glGenPathsNV_p, range);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenPathsNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGenPathsNV_p: APcv = ap_null();
    /// Tries to load [`glGenPathsNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGenPathsNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGenPathsNV\0", &glGenPathsNV_p)
    }
    /// Checks if the pointer for [`glGenPathsNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGenPathsNV_is_loaded() -> bool {
        !glGenPathsNV_p.load(RELAX).is_null()
    }

    /// [glGenPerfMonitorsAMD](http://docs.gl/es3/glGenPerfMonitorsAMD)(n, monitors)
    /// * `monitors` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGenPerfMonitorsAMD(n: GLsizei, monitors: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenPerfMonitorsAMD({:?}, {:p});", n, monitors);
        }
        let out =
            call_atomic_ptr_2arg("glGenPerfMonitorsAMD", &glGenPerfMonitorsAMD_p, n, monitors);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenPerfMonitorsAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glGenPerfMonitorsAMD_p: APcv = ap_null();
    /// Tries to load [`glGenPerfMonitorsAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGenPerfMonitorsAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGenPerfMonitorsAMD\0",
            &glGenPerfMonitorsAMD_p,
        )
    }
    /// Checks if the pointer for [`glGenPerfMonitorsAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glGenPerfMonitorsAMD_is_loaded() -> bool {
        !glGenPerfMonitorsAMD_p.load(RELAX).is_null()
    }

    /// [glGenProgramPipelines](http://docs.gl/es3/glGenProgramPipelines)(n, pipelines)
    /// * `pipelines` len: n
    /// * `pipelines` class: program pipeline
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenProgramPipelines(n: GLsizei, pipelines: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenProgramPipelines({:?}, {:p});", n, pipelines);
        }
        let out = call_atomic_ptr_2arg(
            "glGenProgramPipelines",
            &glGenProgramPipelines_p,
            n,
            pipelines,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenProgramPipelines");
        }
        out
    }
    static glGenProgramPipelines_p: APcv = ap_null();
    /// Tries to load [`glGenProgramPipelines`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenProgramPipelines_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGenProgramPipelines\0",
            &glGenProgramPipelines_p,
        )
    }
    /// Checks if the pointer for [`glGenProgramPipelines`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenProgramPipelines_is_loaded() -> bool {
        !glGenProgramPipelines_p.load(RELAX).is_null()
    }

    /// [glGenProgramPipelinesEXT](http://docs.gl/es3/glGenProgramPipelinesEXT)(n, pipelines)
    /// * `pipelines` len: n
    /// * `pipelines` class: program pipeline
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glGenProgramPipelinesEXT(n: GLsizei, pipelines: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGenProgramPipelinesEXT({:?}, {:p});",
                n,
                pipelines
            );
        }
        let out = call_atomic_ptr_2arg(
            "glGenProgramPipelinesEXT",
            &glGenProgramPipelinesEXT_p,
            n,
            pipelines,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenProgramPipelinesEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    static glGenProgramPipelinesEXT_p: APcv = ap_null();
    /// Tries to load [`glGenProgramPipelinesEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glGenProgramPipelinesEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGenProgramPipelinesEXT\0",
            &glGenProgramPipelinesEXT_p,
        )
    }
    /// Checks if the pointer for [`glGenProgramPipelinesEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub fn glGenProgramPipelinesEXT_is_loaded() -> bool {
        !glGenProgramPipelinesEXT_p.load(RELAX).is_null()
    }

    /// [glGenQueries](http://docs.gl/es3/glGenQueries)(n, ids)
    /// * `ids` len: n
    /// * `ids` class: query
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenQueries(n: GLsizei, ids: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenQueries({:?}, {:p});", n, ids);
        }
        let out = call_atomic_ptr_2arg("glGenQueries", &glGenQueries_p, n, ids);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenQueries");
        }
        out
    }
    static glGenQueries_p: APcv = ap_null();
    /// Tries to load [`glGenQueries`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenQueries_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGenQueries\0", &glGenQueries_p)
    }
    /// Checks if the pointer for [`glGenQueries`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenQueries_is_loaded() -> bool {
        !glGenQueries_p.load(RELAX).is_null()
    }

    /// [glGenQueriesEXT](http://docs.gl/es3/glGenQueriesEXT)(n, ids)
    /// * `ids` len: n
    /// * `ids` class: query
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glGenQueriesEXT(n: GLsizei, ids: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenQueriesEXT({:?}, {:p});", n, ids);
        }
        let out = call_atomic_ptr_2arg("glGenQueriesEXT", &glGenQueriesEXT_p, n, ids);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenQueriesEXT");
        }
        out
    }
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    static glGenQueriesEXT_p: APcv = ap_null();
    /// Tries to load [`glGenQueriesEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glGenQueriesEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGenQueriesEXT\0", &glGenQueriesEXT_p)
    }
    /// Checks if the pointer for [`glGenQueriesEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub fn glGenQueriesEXT_is_loaded() -> bool {
        !glGenQueriesEXT_p.load(RELAX).is_null()
    }

    /// [glGenRenderbuffers](http://docs.gl/es3/glGenRenderbuffers)(n, renderbuffers)
    /// * `renderbuffers` len: n
    /// * `renderbuffers` class: renderbuffer
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenRenderbuffers(n: GLsizei, renderbuffers: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenRenderbuffers({:?}, {:p});", n, renderbuffers);
        }
        let out = call_atomic_ptr_2arg(
            "glGenRenderbuffers",
            &glGenRenderbuffers_p,
            n,
            renderbuffers,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenRenderbuffers");
        }
        out
    }
    static glGenRenderbuffers_p: APcv = ap_null();
    /// Tries to load [`glGenRenderbuffers`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenRenderbuffers_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGenRenderbuffers\0",
            &glGenRenderbuffers_p,
        )
    }
    /// Checks if the pointer for [`glGenRenderbuffers`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenRenderbuffers_is_loaded() -> bool {
        !glGenRenderbuffers_p.load(RELAX).is_null()
    }

    /// [glGenSamplers](http://docs.gl/es3/glGenSamplers)(count, samplers)
    /// * `samplers` len: count
    /// * `samplers` class: sampler
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenSamplers(count: GLsizei, samplers: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenSamplers({:?}, {:p});", count, samplers);
        }
        let out = call_atomic_ptr_2arg("glGenSamplers", &glGenSamplers_p, count, samplers);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenSamplers");
        }
        out
    }
    static glGenSamplers_p: APcv = ap_null();
    /// Tries to load [`glGenSamplers`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenSamplers_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGenSamplers\0", &glGenSamplers_p)
    }
    /// Checks if the pointer for [`glGenSamplers`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenSamplers_is_loaded() -> bool {
        !glGenSamplers_p.load(RELAX).is_null()
    }

    /// [glGenSemaphoresEXT](http://docs.gl/es3/glGenSemaphoresEXT)(n, semaphores)
    /// * `semaphores` len: n
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    pub unsafe fn glGenSemaphoresEXT(n: GLsizei, semaphores: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenSemaphoresEXT({:?}, {:p});", n, semaphores);
        }
        let out = call_atomic_ptr_2arg("glGenSemaphoresEXT", &glGenSemaphoresEXT_p, n, semaphores);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenSemaphoresEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    static glGenSemaphoresEXT_p: APcv = ap_null();
    /// Tries to load [`glGenSemaphoresEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    pub unsafe fn glGenSemaphoresEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGenSemaphoresEXT\0",
            &glGenSemaphoresEXT_p,
        )
    }
    /// Checks if the pointer for [`glGenSemaphoresEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    pub fn glGenSemaphoresEXT_is_loaded() -> bool {
        !glGenSemaphoresEXT_p.load(RELAX).is_null()
    }

    /// [glGenTextures](http://docs.gl/es3/glGenTextures)(n, textures)
    /// * `textures` group: Texture
    /// * `textures` len: n
    /// * `textures` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenTextures(n: GLsizei, textures: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenTextures({:?}, {:p});", n, textures);
        }
        let out = call_atomic_ptr_2arg("glGenTextures", &glGenTextures_p, n, textures);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenTextures");
        }
        out
    }
    static glGenTextures_p: APcv = ap_null();
    /// Tries to load [`glGenTextures`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenTextures_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGenTextures\0", &glGenTextures_p)
    }
    /// Checks if the pointer for [`glGenTextures`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenTextures_is_loaded() -> bool {
        !glGenTextures_p.load(RELAX).is_null()
    }

    /// [glGenTransformFeedbacks](http://docs.gl/es3/glGenTransformFeedbacks)(n, ids)
    /// * `ids` len: n
    /// * `ids` class: transform feedback
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenTransformFeedbacks(n: GLsizei, ids: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenTransformFeedbacks({:?}, {:p});", n, ids);
        }
        let out = call_atomic_ptr_2arg(
            "glGenTransformFeedbacks",
            &glGenTransformFeedbacks_p,
            n,
            ids,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenTransformFeedbacks");
        }
        out
    }
    static glGenTransformFeedbacks_p: APcv = ap_null();
    /// Tries to load [`glGenTransformFeedbacks`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenTransformFeedbacks_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGenTransformFeedbacks\0",
            &glGenTransformFeedbacks_p,
        )
    }
    /// Checks if the pointer for [`glGenTransformFeedbacks`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenTransformFeedbacks_is_loaded() -> bool {
        !glGenTransformFeedbacks_p.load(RELAX).is_null()
    }

    /// [glGenVertexArrays](http://docs.gl/es3/glGenVertexArrays)(n, arrays)
    /// * `arrays` len: n
    /// * `arrays` class: vertex array
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenVertexArrays(n: GLsizei, arrays: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenVertexArrays({:?}, {:p});", n, arrays);
        }
        let out = call_atomic_ptr_2arg("glGenVertexArrays", &glGenVertexArrays_p, n, arrays);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenVertexArrays");
        }
        out
    }
    static glGenVertexArrays_p: APcv = ap_null();
    /// Tries to load [`glGenVertexArrays`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenVertexArrays_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGenVertexArrays\0",
            &glGenVertexArrays_p,
        )
    }
    /// Checks if the pointer for [`glGenVertexArrays`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenVertexArrays_is_loaded() -> bool {
        !glGenVertexArrays_p.load(RELAX).is_null()
    }

    /// [glGenVertexArraysOES](http://docs.gl/es3/glGenVertexArraysOES)(n, arrays)
    /// * `arrays` len: n
    /// * `arrays` class: vertex array
    /// * alias of: [`glGenVertexArrays`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    pub unsafe fn glGenVertexArraysOES(n: GLsizei, arrays: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenVertexArraysOES({:?}, {:p});", n, arrays);
        }
        let out = call_atomic_ptr_2arg("glGenVertexArraysOES", &glGenVertexArraysOES_p, n, arrays);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenVertexArraysOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    static glGenVertexArraysOES_p: APcv = ap_null();
    /// Tries to load [`glGenVertexArraysOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    pub unsafe fn glGenVertexArraysOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGenVertexArraysOES\0",
            &glGenVertexArraysOES_p,
        )
    }
    /// Checks if the pointer for [`glGenVertexArraysOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_vertex_array_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_vertex_array_object"))))]
    pub fn glGenVertexArraysOES_is_loaded() -> bool {
        !glGenVertexArraysOES_p.load(RELAX).is_null()
    }

    /// [glGenerateMipmap](http://docs.gl/es3/glGenerateMipmap)(target)
    /// * `target` group: TextureTarget
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGenerateMipmap(target: GLenum) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGenerateMipmap({:#X});", target);
        }
        let out = call_atomic_ptr_1arg("glGenerateMipmap", &glGenerateMipmap_p, target);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGenerateMipmap");
        }
        out
    }
    static glGenerateMipmap_p: APcv = ap_null();
    /// Tries to load [`glGenerateMipmap`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGenerateMipmap_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGenerateMipmap\0", &glGenerateMipmap_p)
    }
    /// Checks if the pointer for [`glGenerateMipmap`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGenerateMipmap_is_loaded() -> bool {
        !glGenerateMipmap_p.load(RELAX).is_null()
    }

    /// [glGetActiveAttrib](http://docs.gl/es3/glGetActiveAttrib)(program, index, bufSize, length, size, type_, name)
    /// * `program` class: program
    /// * `length` len: 1
    /// * `size` len: 1
    /// * `type_` group: AttributeType
    /// * `type_` len: 1
    /// * `name` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetActiveAttrib(
        program: GLuint,
        index: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        size: *mut GLint,
        type_: *mut GLenum,
        name: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetActiveAttrib({:?}, {:?}, {:?}, {:p}, {:p}, {:p}, {:p});",
                program,
                index,
                bufSize,
                length,
                size,
                type_,
                name
            );
        }
        let out = call_atomic_ptr_7arg(
            "glGetActiveAttrib",
            &glGetActiveAttrib_p,
            program,
            index,
            bufSize,
            length,
            size,
            type_,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetActiveAttrib");
        }
        out
    }
    static glGetActiveAttrib_p: APcv = ap_null();
    /// Tries to load [`glGetActiveAttrib`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetActiveAttrib_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetActiveAttrib\0",
            &glGetActiveAttrib_p,
        )
    }
    /// Checks if the pointer for [`glGetActiveAttrib`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetActiveAttrib_is_loaded() -> bool {
        !glGetActiveAttrib_p.load(RELAX).is_null()
    }

    /// [glGetActiveUniform](http://docs.gl/es3/glGetActiveUniform)(program, index, bufSize, length, size, type_, name)
    /// * `program` class: program
    /// * `length` len: 1
    /// * `size` len: 1
    /// * `type_` group: UniformType
    /// * `type_` len: 1
    /// * `name` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetActiveUniform(
        program: GLuint,
        index: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        size: *mut GLint,
        type_: *mut GLenum,
        name: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetActiveUniform({:?}, {:?}, {:?}, {:p}, {:p}, {:p}, {:p});",
                program,
                index,
                bufSize,
                length,
                size,
                type_,
                name
            );
        }
        let out = call_atomic_ptr_7arg(
            "glGetActiveUniform",
            &glGetActiveUniform_p,
            program,
            index,
            bufSize,
            length,
            size,
            type_,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetActiveUniform");
        }
        out
    }
    static glGetActiveUniform_p: APcv = ap_null();
    /// Tries to load [`glGetActiveUniform`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetActiveUniform_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetActiveUniform\0",
            &glGetActiveUniform_p,
        )
    }
    /// Checks if the pointer for [`glGetActiveUniform`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetActiveUniform_is_loaded() -> bool {
        !glGetActiveUniform_p.load(RELAX).is_null()
    }

    /// [glGetActiveUniformBlockName](http://docs.gl/es3/glGetActiveUniformBlockName)(program, uniformBlockIndex, bufSize, length, uniformBlockName)
    /// * `program` class: program
    /// * `length` len: 1
    /// * `uniformBlockName` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetActiveUniformBlockName(
        program: GLuint,
        uniformBlockIndex: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        uniformBlockName: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetActiveUniformBlockName({:?}, {:?}, {:?}, {:p}, {:p});",
                program,
                uniformBlockIndex,
                bufSize,
                length,
                uniformBlockName
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetActiveUniformBlockName",
            &glGetActiveUniformBlockName_p,
            program,
            uniformBlockIndex,
            bufSize,
            length,
            uniformBlockName,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetActiveUniformBlockName");
        }
        out
    }
    static glGetActiveUniformBlockName_p: APcv = ap_null();
    /// Tries to load [`glGetActiveUniformBlockName`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetActiveUniformBlockName_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetActiveUniformBlockName\0",
            &glGetActiveUniformBlockName_p,
        )
    }
    /// Checks if the pointer for [`glGetActiveUniformBlockName`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetActiveUniformBlockName_is_loaded() -> bool {
        !glGetActiveUniformBlockName_p.load(RELAX).is_null()
    }

    /// [glGetActiveUniformBlockiv](http://docs.gl/es3/glGetActiveUniformBlockiv)(program, uniformBlockIndex, pname, params)
    /// * `program` class: program
    /// * `pname` group: UniformBlockPName
    /// * `params` len: COMPSIZE(program,uniformBlockIndex,pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetActiveUniformBlockiv(
        program: GLuint,
        uniformBlockIndex: GLuint,
        pname: GLenum,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetActiveUniformBlockiv({:?}, {:?}, {:#X}, {:p});",
                program,
                uniformBlockIndex,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetActiveUniformBlockiv",
            &glGetActiveUniformBlockiv_p,
            program,
            uniformBlockIndex,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetActiveUniformBlockiv");
        }
        out
    }
    static glGetActiveUniformBlockiv_p: APcv = ap_null();
    /// Tries to load [`glGetActiveUniformBlockiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetActiveUniformBlockiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetActiveUniformBlockiv\0",
            &glGetActiveUniformBlockiv_p,
        )
    }
    /// Checks if the pointer for [`glGetActiveUniformBlockiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetActiveUniformBlockiv_is_loaded() -> bool {
        !glGetActiveUniformBlockiv_p.load(RELAX).is_null()
    }

    /// [glGetActiveUniformsiv](http://docs.gl/es3/glGetActiveUniformsiv)(program, uniformCount, uniformIndices, pname, params)
    /// * `program` class: program
    /// * `uniformIndices` len: uniformCount
    /// * `pname` group: UniformPName
    /// * `params` len: COMPSIZE(uniformCount,pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetActiveUniformsiv(
        program: GLuint,
        uniformCount: GLsizei,
        uniformIndices: *const GLuint,
        pname: GLenum,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetActiveUniformsiv({:?}, {:?}, {:p}, {:#X}, {:p});",
                program,
                uniformCount,
                uniformIndices,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetActiveUniformsiv",
            &glGetActiveUniformsiv_p,
            program,
            uniformCount,
            uniformIndices,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetActiveUniformsiv");
        }
        out
    }
    static glGetActiveUniformsiv_p: APcv = ap_null();
    /// Tries to load [`glGetActiveUniformsiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetActiveUniformsiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetActiveUniformsiv\0",
            &glGetActiveUniformsiv_p,
        )
    }
    /// Checks if the pointer for [`glGetActiveUniformsiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetActiveUniformsiv_is_loaded() -> bool {
        !glGetActiveUniformsiv_p.load(RELAX).is_null()
    }

    /// [glGetAttachedShaders](http://docs.gl/es3/glGetAttachedShaders)(program, maxCount, count, shaders)
    /// * `program` class: program
    /// * `count` len: 1
    /// * `shaders` len: maxCount
    /// * `shaders` class: shader
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetAttachedShaders(
        program: GLuint,
        maxCount: GLsizei,
        count: *mut GLsizei,
        shaders: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetAttachedShaders({:?}, {:?}, {:p}, {:p});",
                program,
                maxCount,
                count,
                shaders
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetAttachedShaders",
            &glGetAttachedShaders_p,
            program,
            maxCount,
            count,
            shaders,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetAttachedShaders");
        }
        out
    }
    static glGetAttachedShaders_p: APcv = ap_null();
    /// Tries to load [`glGetAttachedShaders`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetAttachedShaders_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetAttachedShaders\0",
            &glGetAttachedShaders_p,
        )
    }
    /// Checks if the pointer for [`glGetAttachedShaders`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetAttachedShaders_is_loaded() -> bool {
        !glGetAttachedShaders_p.load(RELAX).is_null()
    }

    /// [glGetAttribLocation](http://docs.gl/es3/glGetAttribLocation)(program, name)
    /// * `program` class: program
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetAttribLocation(program: GLuint, name: *const GLchar) -> GLint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetAttribLocation({:?}, {:p});", program, name);
        }
        let out =
            call_atomic_ptr_2arg("glGetAttribLocation", &glGetAttribLocation_p, program, name);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetAttribLocation");
        }
        out
    }
    static glGetAttribLocation_p: APcv = ap_null();
    /// Tries to load [`glGetAttribLocation`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetAttribLocation_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetAttribLocation\0",
            &glGetAttribLocation_p,
        )
    }
    /// Checks if the pointer for [`glGetAttribLocation`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetAttribLocation_is_loaded() -> bool {
        !glGetAttribLocation_p.load(RELAX).is_null()
    }

    /// [glGetBooleani_v](http://docs.gl/es3/glGet)(target, index, data)
    /// * `target` group: BufferTargetARB
    /// * `data` len: COMPSIZE(target)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetBooleani_v(target: GLenum, index: GLuint, data: *mut GLboolean) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetBooleani_v({:#X}, {:?}, {:p});",
                target,
                index,
                data
            );
        }
        let out = call_atomic_ptr_3arg("glGetBooleani_v", &glGetBooleani_v_p, target, index, data);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetBooleani_v");
        }
        out
    }
    static glGetBooleani_v_p: APcv = ap_null();
    /// Tries to load [`glGetBooleani_v`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetBooleani_v_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetBooleani_v\0", &glGetBooleani_v_p)
    }
    /// Checks if the pointer for [`glGetBooleani_v`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetBooleani_v_is_loaded() -> bool {
        !glGetBooleani_v_p.load(RELAX).is_null()
    }

    /// [glGetBooleanv](http://docs.gl/es3/glGet)(pname, data)
    /// * `pname` group: GetPName
    /// * `data` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetBooleanv(pname: GLenum, data: *mut GLboolean) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetBooleanv({:#X}, {:p});", pname, data);
        }
        let out = call_atomic_ptr_2arg("glGetBooleanv", &glGetBooleanv_p, pname, data);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetBooleanv");
        }
        out
    }
    static glGetBooleanv_p: APcv = ap_null();
    /// Tries to load [`glGetBooleanv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetBooleanv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetBooleanv\0", &glGetBooleanv_p)
    }
    /// Checks if the pointer for [`glGetBooleanv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetBooleanv_is_loaded() -> bool {
        !glGetBooleanv_p.load(RELAX).is_null()
    }

    /// [glGetBufferParameteri64v](http://docs.gl/es3/glGetBufferParameter)(target, pname, params)
    /// * `target` group: BufferTargetARB
    /// * `pname` group: BufferPNameARB
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetBufferParameteri64v(target: GLenum, pname: GLenum, params: *mut GLint64) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetBufferParameteri64v({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetBufferParameteri64v",
            &glGetBufferParameteri64v_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetBufferParameteri64v");
        }
        out
    }
    static glGetBufferParameteri64v_p: APcv = ap_null();
    /// Tries to load [`glGetBufferParameteri64v`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetBufferParameteri64v_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetBufferParameteri64v\0",
            &glGetBufferParameteri64v_p,
        )
    }
    /// Checks if the pointer for [`glGetBufferParameteri64v`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetBufferParameteri64v_is_loaded() -> bool {
        !glGetBufferParameteri64v_p.load(RELAX).is_null()
    }

    /// [glGetBufferParameteriv](http://docs.gl/es3/glGetBufferParameter)(target, pname, params)
    /// * `target` group: BufferTargetARB
    /// * `pname` group: BufferPNameARB
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetBufferParameteriv(target: GLenum, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetBufferParameteriv({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetBufferParameteriv",
            &glGetBufferParameteriv_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetBufferParameteriv");
        }
        out
    }
    static glGetBufferParameteriv_p: APcv = ap_null();
    /// Tries to load [`glGetBufferParameteriv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetBufferParameteriv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetBufferParameteriv\0",
            &glGetBufferParameteriv_p,
        )
    }
    /// Checks if the pointer for [`glGetBufferParameteriv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetBufferParameteriv_is_loaded() -> bool {
        !glGetBufferParameteriv_p.load(RELAX).is_null()
    }

    /// [glGetBufferPointerv](http://docs.gl/es3/glGetBufferPointerv)(target, pname, params)
    /// * `target` group: BufferTargetARB
    /// * `pname` group: BufferPointerNameARB
    /// * `params` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetBufferPointerv(target: GLenum, pname: GLenum, params: *mut *mut c_void) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetBufferPointerv({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetBufferPointerv",
            &glGetBufferPointerv_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetBufferPointerv");
        }
        out
    }
    static glGetBufferPointerv_p: APcv = ap_null();
    /// Tries to load [`glGetBufferPointerv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetBufferPointerv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetBufferPointerv\0",
            &glGetBufferPointerv_p,
        )
    }
    /// Checks if the pointer for [`glGetBufferPointerv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetBufferPointerv_is_loaded() -> bool {
        !glGetBufferPointerv_p.load(RELAX).is_null()
    }

    /// [glGetBufferPointervOES](http://docs.gl/es3/glGetBufferPointervOES)(target, pname, params)
    /// * `target` group: BufferTargetARB
    /// * `pname` group: BufferPointerNameARB
    /// * `params` len: 1
    /// * alias of: [`glGetBufferPointerv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_mapbuffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
    pub unsafe fn glGetBufferPointervOES(target: GLenum, pname: GLenum, params: *mut *mut c_void) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetBufferPointervOES({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetBufferPointervOES",
            &glGetBufferPointervOES_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetBufferPointervOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_mapbuffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
    static glGetBufferPointervOES_p: APcv = ap_null();
    /// Tries to load [`glGetBufferPointervOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_mapbuffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
    pub unsafe fn glGetBufferPointervOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetBufferPointervOES\0",
            &glGetBufferPointervOES_p,
        )
    }
    /// Checks if the pointer for [`glGetBufferPointervOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_mapbuffer"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_mapbuffer"))))]
    pub fn glGetBufferPointervOES_is_loaded() -> bool {
        !glGetBufferPointervOES_p.load(RELAX).is_null()
    }

    /// [glGetCoverageModulationTableNV](http://docs.gl/es3/glGetCoverageModulationTableNV)(bufSize, v)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    pub unsafe fn glGetCoverageModulationTableNV(bufSize: GLsizei, v: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetCoverageModulationTableNV({:?}, {:p});",
                bufSize,
                v
            );
        }
        let out = call_atomic_ptr_2arg(
            "glGetCoverageModulationTableNV",
            &glGetCoverageModulationTableNV_p,
            bufSize,
            v,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetCoverageModulationTableNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    static glGetCoverageModulationTableNV_p: APcv = ap_null();
    /// Tries to load [`glGetCoverageModulationTableNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    pub unsafe fn glGetCoverageModulationTableNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetCoverageModulationTableNV\0",
            &glGetCoverageModulationTableNV_p,
        )
    }
    /// Checks if the pointer for [`glGetCoverageModulationTableNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_framebuffer_mixed_samples"))))]
    pub fn glGetCoverageModulationTableNV_is_loaded() -> bool {
        !glGetCoverageModulationTableNV_p.load(RELAX).is_null()
    }

    /// [glGetDebugMessageLog](http://docs.gl/es3/glGetDebugMessageLog)(count, bufSize, sources, types, ids, severities, lengths, messageLog)
    /// * `sources` group: DebugSource
    /// * `sources` len: count
    /// * `types` group: DebugType
    /// * `types` len: count
    /// * `ids` len: count
    /// * `severities` group: DebugSeverity
    /// * `severities` len: count
    /// * `lengths` len: count
    /// * `messageLog` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetDebugMessageLog(
        count: GLuint,
        bufSize: GLsizei,
        sources: *mut GLenum,
        types: *mut GLenum,
        ids: *mut GLuint,
        severities: *mut GLenum,
        lengths: *mut GLsizei,
        messageLog: *mut GLchar,
    ) -> GLuint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetDebugMessageLog({:?}, {:?}, {:p}, {:p}, {:p}, {:p}, {:p}, {:p});",
                count,
                bufSize,
                sources,
                types,
                ids,
                severities,
                lengths,
                messageLog
            );
        }
        let out = call_atomic_ptr_8arg(
            "glGetDebugMessageLog",
            &glGetDebugMessageLog_p,
            count,
            bufSize,
            sources,
            types,
            ids,
            severities,
            lengths,
            messageLog,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetDebugMessageLog");
        }
        out
    }
    static glGetDebugMessageLog_p: APcv = ap_null();
    /// Tries to load [`glGetDebugMessageLog`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetDebugMessageLog_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetDebugMessageLog\0",
            &glGetDebugMessageLog_p,
        )
    }
    /// Checks if the pointer for [`glGetDebugMessageLog`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetDebugMessageLog_is_loaded() -> bool {
        !glGetDebugMessageLog_p.load(RELAX).is_null()
    }

    /// [glGetDebugMessageLogKHR](http://docs.gl/es3/glGetDebugMessageLogKHR)(count, bufSize, sources, types, ids, severities, lengths, messageLog)
    /// * `sources` group: DebugSource
    /// * `sources` len: count
    /// * `types` group: DebugType
    /// * `types` len: count
    /// * `ids` len: count
    /// * `severities` group: DebugSeverity
    /// * `severities` len: count
    /// * `lengths` len: count
    /// * `messageLog` len: bufSize
    /// * alias of: [`glGetDebugMessageLog`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glGetDebugMessageLogKHR(
        count: GLuint,
        bufSize: GLsizei,
        sources: *mut GLenum,
        types: *mut GLenum,
        ids: *mut GLuint,
        severities: *mut GLenum,
        lengths: *mut GLsizei,
        messageLog: *mut GLchar,
    ) -> GLuint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetDebugMessageLogKHR({:?}, {:?}, {:p}, {:p}, {:p}, {:p}, {:p}, {:p});",
                count,
                bufSize,
                sources,
                types,
                ids,
                severities,
                lengths,
                messageLog
            );
        }
        let out = call_atomic_ptr_8arg(
            "glGetDebugMessageLogKHR",
            &glGetDebugMessageLogKHR_p,
            count,
            bufSize,
            sources,
            types,
            ids,
            severities,
            lengths,
            messageLog,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetDebugMessageLogKHR");
        }
        out
    }
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    static glGetDebugMessageLogKHR_p: APcv = ap_null();
    /// Tries to load [`glGetDebugMessageLogKHR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glGetDebugMessageLogKHR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetDebugMessageLogKHR\0",
            &glGetDebugMessageLogKHR_p,
        )
    }
    /// Checks if the pointer for [`glGetDebugMessageLogKHR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub fn glGetDebugMessageLogKHR_is_loaded() -> bool {
        !glGetDebugMessageLogKHR_p.load(RELAX).is_null()
    }

    /// [glGetDriverControlStringQCOM](http://docs.gl/es3/glGetDriverControlStringQCOM)(driverControl, bufSize, length, driverControlString)
    /// * `driverControlString` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub unsafe fn glGetDriverControlStringQCOM(
        driverControl: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        driverControlString: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetDriverControlStringQCOM({:?}, {:?}, {:p}, {:p});",
                driverControl,
                bufSize,
                length,
                driverControlString
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetDriverControlStringQCOM",
            &glGetDriverControlStringQCOM_p,
            driverControl,
            bufSize,
            length,
            driverControlString,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetDriverControlStringQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    static glGetDriverControlStringQCOM_p: APcv = ap_null();
    /// Tries to load [`glGetDriverControlStringQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub unsafe fn glGetDriverControlStringQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetDriverControlStringQCOM\0",
            &glGetDriverControlStringQCOM_p,
        )
    }
    /// Checks if the pointer for [`glGetDriverControlStringQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub fn glGetDriverControlStringQCOM_is_loaded() -> bool {
        !glGetDriverControlStringQCOM_p.load(RELAX).is_null()
    }

    /// [glGetDriverControlsQCOM](http://docs.gl/es3/glGetDriverControlsQCOM)(num, size, driverControls)
    /// * `driverControls` len: size
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub unsafe fn glGetDriverControlsQCOM(
        num: *mut GLint,
        size: GLsizei,
        driverControls: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetDriverControlsQCOM({:p}, {:?}, {:p});",
                num,
                size,
                driverControls
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetDriverControlsQCOM",
            &glGetDriverControlsQCOM_p,
            num,
            size,
            driverControls,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetDriverControlsQCOM");
        }
        out
    }
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    static glGetDriverControlsQCOM_p: APcv = ap_null();
    /// Tries to load [`glGetDriverControlsQCOM`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub unsafe fn glGetDriverControlsQCOM_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetDriverControlsQCOM\0",
            &glGetDriverControlsQCOM_p,
        )
    }
    /// Checks if the pointer for [`glGetDriverControlsQCOM`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_QCOM_driver_control"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_QCOM_driver_control"))))]
    pub fn glGetDriverControlsQCOM_is_loaded() -> bool {
        !glGetDriverControlsQCOM_p.load(RELAX).is_null()
    }

    /// [glGetError](http://docs.gl/es3/glGetError)()
    /// * return value group: ErrorCode
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetError() -> GLenum {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetError();",);
        }
        let out = call_atomic_ptr_0arg("glGetError", &glGetError_p);

        out
    }
    static glGetError_p: APcv = ap_null();
    /// Tries to load [`glGetError`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetError_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetError\0", &glGetError_p)
    }
    /// Checks if the pointer for [`glGetError`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetError_is_loaded() -> bool {
        !glGetError_p.load(RELAX).is_null()
    }

    /// [glGetFenceivNV](http://docs.gl/es3/glGetFenceivNV)(fence, pname, params)
    /// * `fence` group: FenceNV
    /// * `pname` group: FenceParameterNameNV
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub unsafe fn glGetFenceivNV(fence: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetFenceivNV({:?}, {:#X}, {:p});",
                fence,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg("glGetFenceivNV", &glGetFenceivNV_p, fence, pname, params);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFenceivNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    static glGetFenceivNV_p: APcv = ap_null();
    /// Tries to load [`glGetFenceivNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub unsafe fn glGetFenceivNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetFenceivNV\0", &glGetFenceivNV_p)
    }
    /// Checks if the pointer for [`glGetFenceivNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_fence"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_fence"))))]
    pub fn glGetFenceivNV_is_loaded() -> bool {
        !glGetFenceivNV_p.load(RELAX).is_null()
    }

    /// [glGetFirstPerfQueryIdINTEL](http://docs.gl/es3/glGetFirstPerfQueryIdINTEL)(queryId)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetFirstPerfQueryIdINTEL(queryId: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetFirstPerfQueryIdINTEL({:p});", queryId);
        }
        let out = call_atomic_ptr_1arg(
            "glGetFirstPerfQueryIdINTEL",
            &glGetFirstPerfQueryIdINTEL_p,
            queryId,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFirstPerfQueryIdINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glGetFirstPerfQueryIdINTEL_p: APcv = ap_null();
    /// Tries to load [`glGetFirstPerfQueryIdINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetFirstPerfQueryIdINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetFirstPerfQueryIdINTEL\0",
            &glGetFirstPerfQueryIdINTEL_p,
        )
    }
    /// Checks if the pointer for [`glGetFirstPerfQueryIdINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glGetFirstPerfQueryIdINTEL_is_loaded() -> bool {
        !glGetFirstPerfQueryIdINTEL_p.load(RELAX).is_null()
    }

    /// [glGetFloati_vNV](http://docs.gl/es3/glGetFloati_vNV)(target, index, data)
    /// * `target` group: GetPName
    /// * `data` len: COMPSIZE(target)
    /// * alias of: [`glGetFloati_v`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glGetFloati_vNV(target: GLenum, index: GLuint, data: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetFloati_vNV({:#X}, {:?}, {:p});",
                target,
                index,
                data
            );
        }
        let out = call_atomic_ptr_3arg("glGetFloati_vNV", &glGetFloati_vNV_p, target, index, data);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFloati_vNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    static glGetFloati_vNV_p: APcv = ap_null();
    /// Tries to load [`glGetFloati_vNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub unsafe fn glGetFloati_vNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetFloati_vNV\0", &glGetFloati_vNV_p)
    }
    /// Checks if the pointer for [`glGetFloati_vNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_viewport_array"))))]
    pub fn glGetFloati_vNV_is_loaded() -> bool {
        !glGetFloati_vNV_p.load(RELAX).is_null()
    }

    /// [glGetFloati_vOES](http://docs.gl/es3/glGetFloati_vOES)(target, index, data)
    /// * `target` group: GetPName
    /// * `data` len: COMPSIZE(target)
    /// * alias of: [`glGetFloati_v`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    pub unsafe fn glGetFloati_vOES(target: GLenum, index: GLuint, data: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetFloati_vOES({:#X}, {:?}, {:p});",
                target,
                index,
                data
            );
        }
        let out =
            call_atomic_ptr_3arg("glGetFloati_vOES", &glGetFloati_vOES_p, target, index, data);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFloati_vOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    static glGetFloati_vOES_p: APcv = ap_null();
    /// Tries to load [`glGetFloati_vOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    pub unsafe fn glGetFloati_vOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetFloati_vOES\0", &glGetFloati_vOES_p)
    }
    /// Checks if the pointer for [`glGetFloati_vOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_viewport_array"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_viewport_array"))))]
    pub fn glGetFloati_vOES_is_loaded() -> bool {
        !glGetFloati_vOES_p.load(RELAX).is_null()
    }

    /// [glGetFloatv](http://docs.gl/es3/glGet)(pname, data)
    /// * `pname` group: GetPName
    /// * `data` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetFloatv(pname: GLenum, data: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetFloatv({:#X}, {:p});", pname, data);
        }
        let out = call_atomic_ptr_2arg("glGetFloatv", &glGetFloatv_p, pname, data);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFloatv");
        }
        out
    }
    static glGetFloatv_p: APcv = ap_null();
    /// Tries to load [`glGetFloatv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetFloatv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetFloatv\0", &glGetFloatv_p)
    }
    /// Checks if the pointer for [`glGetFloatv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetFloatv_is_loaded() -> bool {
        !glGetFloatv_p.load(RELAX).is_null()
    }

    /// [glGetFragDataIndexEXT](http://docs.gl/es3/glGetFragDataIndexEXT)(program, name)
    /// * `program` class: program
    /// * alias of: [`glGetFragDataIndex`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub unsafe fn glGetFragDataIndexEXT(program: GLuint, name: *const GLchar) -> GLint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetFragDataIndexEXT({:?}, {:p});", program, name);
        }
        let out = call_atomic_ptr_2arg(
            "glGetFragDataIndexEXT",
            &glGetFragDataIndexEXT_p,
            program,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFragDataIndexEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    static glGetFragDataIndexEXT_p: APcv = ap_null();
    /// Tries to load [`glGetFragDataIndexEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub unsafe fn glGetFragDataIndexEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetFragDataIndexEXT\0",
            &glGetFragDataIndexEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetFragDataIndexEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub fn glGetFragDataIndexEXT_is_loaded() -> bool {
        !glGetFragDataIndexEXT_p.load(RELAX).is_null()
    }

    /// [glGetFragDataLocation](http://docs.gl/es3/glGetFragDataLocation)(program, name)
    /// * `program` class: program
    /// * `name` len: COMPSIZE(name)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetFragDataLocation(program: GLuint, name: *const GLchar) -> GLint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetFragDataLocation({:?}, {:p});", program, name);
        }
        let out = call_atomic_ptr_2arg(
            "glGetFragDataLocation",
            &glGetFragDataLocation_p,
            program,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFragDataLocation");
        }
        out
    }
    static glGetFragDataLocation_p: APcv = ap_null();
    /// Tries to load [`glGetFragDataLocation`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetFragDataLocation_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetFragDataLocation\0",
            &glGetFragDataLocation_p,
        )
    }
    /// Checks if the pointer for [`glGetFragDataLocation`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetFragDataLocation_is_loaded() -> bool {
        !glGetFragDataLocation_p.load(RELAX).is_null()
    }

    /// [glGetFramebufferAttachmentParameteriv](http://docs.gl/es3/glGetFramebufferAttachmentParameter)(target, attachment, pname, params)
    /// * `target` group: FramebufferTarget
    /// * `attachment` group: FramebufferAttachment
    /// * `pname` group: FramebufferAttachmentParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetFramebufferAttachmentParameteriv(
        target: GLenum,
        attachment: GLenum,
        pname: GLenum,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetFramebufferAttachmentParameteriv({:#X}, {:#X}, {:#X}, {:p});",
                target,
                attachment,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetFramebufferAttachmentParameteriv",
            &glGetFramebufferAttachmentParameteriv_p,
            target,
            attachment,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFramebufferAttachmentParameteriv");
        }
        out
    }
    static glGetFramebufferAttachmentParameteriv_p: APcv = ap_null();
    /// Tries to load [`glGetFramebufferAttachmentParameteriv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetFramebufferAttachmentParameteriv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetFramebufferAttachmentParameteriv\0",
            &glGetFramebufferAttachmentParameteriv_p,
        )
    }
    /// Checks if the pointer for [`glGetFramebufferAttachmentParameteriv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetFramebufferAttachmentParameteriv_is_loaded() -> bool {
        !glGetFramebufferAttachmentParameteriv_p
            .load(RELAX)
            .is_null()
    }

    /// [glGetFramebufferParameteriv](http://docs.gl/es3/glGetFramebufferParameter)(target, pname, params)
    /// * `target` group: FramebufferTarget
    /// * `pname` group: FramebufferAttachmentParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetFramebufferParameteriv(target: GLenum, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetFramebufferParameteriv({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetFramebufferParameteriv",
            &glGetFramebufferParameteriv_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFramebufferParameteriv");
        }
        out
    }
    static glGetFramebufferParameteriv_p: APcv = ap_null();
    /// Tries to load [`glGetFramebufferParameteriv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetFramebufferParameteriv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetFramebufferParameteriv\0",
            &glGetFramebufferParameteriv_p,
        )
    }
    /// Checks if the pointer for [`glGetFramebufferParameteriv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetFramebufferParameteriv_is_loaded() -> bool {
        !glGetFramebufferParameteriv_p.load(RELAX).is_null()
    }

    /// [glGetFramebufferParameterivMESA](http://docs.gl/es3/glGetFramebufferParameterivMESA)(target, pname, params)
    /// * `target` group: FramebufferTarget
    /// * `pname` group: FramebufferAttachmentParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
    pub unsafe fn glGetFramebufferParameterivMESA(
        target: GLenum,
        pname: GLenum,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetFramebufferParameterivMESA({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetFramebufferParameterivMESA",
            &glGetFramebufferParameterivMESA_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFramebufferParameterivMESA");
        }
        out
    }
    #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
    static glGetFramebufferParameterivMESA_p: APcv = ap_null();
    /// Tries to load [`glGetFramebufferParameterivMESA`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
    pub unsafe fn glGetFramebufferParameterivMESA_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetFramebufferParameterivMESA\0",
            &glGetFramebufferParameterivMESA_p,
        )
    }
    /// Checks if the pointer for [`glGetFramebufferParameterivMESA`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_MESA_framebuffer_flip_y"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_MESA_framebuffer_flip_y"))))]
    pub fn glGetFramebufferParameterivMESA_is_loaded() -> bool {
        !glGetFramebufferParameterivMESA_p.load(RELAX).is_null()
    }

    /// [glGetFramebufferPixelLocalStorageSizeEXT](http://docs.gl/es3/glGetFramebufferPixelLocalStorageSizeEXT)(target)
    /// * `target` group: FramebufferTarget
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    pub unsafe fn glGetFramebufferPixelLocalStorageSizeEXT(target: GLuint) -> GLsizei {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetFramebufferPixelLocalStorageSizeEXT({:?});",
                target
            );
        }
        let out = call_atomic_ptr_1arg(
            "glGetFramebufferPixelLocalStorageSizeEXT",
            &glGetFramebufferPixelLocalStorageSizeEXT_p,
            target,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetFramebufferPixelLocalStorageSizeEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    static glGetFramebufferPixelLocalStorageSizeEXT_p: APcv = ap_null();
    /// Tries to load [`glGetFramebufferPixelLocalStorageSizeEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    pub unsafe fn glGetFramebufferPixelLocalStorageSizeEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetFramebufferPixelLocalStorageSizeEXT\0",
            &glGetFramebufferPixelLocalStorageSizeEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetFramebufferPixelLocalStorageSizeEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_shader_pixel_local_storage2"))))]
    pub fn glGetFramebufferPixelLocalStorageSizeEXT_is_loaded() -> bool {
        !glGetFramebufferPixelLocalStorageSizeEXT_p
            .load(RELAX)
            .is_null()
    }

    /// [glGetGraphicsResetStatus](http://docs.gl/es3/glGetGraphicsResetStatus)()
    /// * return value group: GraphicsResetStatus
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetGraphicsResetStatus() -> GLenum {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetGraphicsResetStatus();",);
        }
        let out = call_atomic_ptr_0arg("glGetGraphicsResetStatus", &glGetGraphicsResetStatus_p);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetGraphicsResetStatus");
        }
        out
    }
    static glGetGraphicsResetStatus_p: APcv = ap_null();
    /// Tries to load [`glGetGraphicsResetStatus`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetGraphicsResetStatus_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetGraphicsResetStatus\0",
            &glGetGraphicsResetStatus_p,
        )
    }
    /// Checks if the pointer for [`glGetGraphicsResetStatus`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetGraphicsResetStatus_is_loaded() -> bool {
        !glGetGraphicsResetStatus_p.load(RELAX).is_null()
    }

    /// [glGetGraphicsResetStatusEXT](http://docs.gl/es3/glGetGraphicsResetStatusEXT)()
    /// * return value group: GraphicsResetStatus
    /// * alias of: [`glGetGraphicsResetStatus`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_robustness"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
    pub unsafe fn glGetGraphicsResetStatusEXT() -> GLenum {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetGraphicsResetStatusEXT();",);
        }
        let out = call_atomic_ptr_0arg(
            "glGetGraphicsResetStatusEXT",
            &glGetGraphicsResetStatusEXT_p,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetGraphicsResetStatusEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_robustness"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
    static glGetGraphicsResetStatusEXT_p: APcv = ap_null();
    /// Tries to load [`glGetGraphicsResetStatusEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_robustness"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
    pub unsafe fn glGetGraphicsResetStatusEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetGraphicsResetStatusEXT\0",
            &glGetGraphicsResetStatusEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetGraphicsResetStatusEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_robustness"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_robustness"))))]
    pub fn glGetGraphicsResetStatusEXT_is_loaded() -> bool {
        !glGetGraphicsResetStatusEXT_p.load(RELAX).is_null()
    }

    /// [glGetGraphicsResetStatusKHR](http://docs.gl/es3/glGetGraphicsResetStatusKHR)()
    /// * return value group: GraphicsResetStatus
    /// * alias of: [`glGetGraphicsResetStatus`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_KHR_robustness"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
    pub unsafe fn glGetGraphicsResetStatusKHR() -> GLenum {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetGraphicsResetStatusKHR();",);
        }
        let out = call_atomic_ptr_0arg(
            "glGetGraphicsResetStatusKHR",
            &glGetGraphicsResetStatusKHR_p,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetGraphicsResetStatusKHR");
        }
        out
    }
    #[cfg(any(feature = "GL_KHR_robustness"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
    static glGetGraphicsResetStatusKHR_p: APcv = ap_null();
    /// Tries to load [`glGetGraphicsResetStatusKHR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_robustness"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
    pub unsafe fn glGetGraphicsResetStatusKHR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetGraphicsResetStatusKHR\0",
            &glGetGraphicsResetStatusKHR_p,
        )
    }
    /// Checks if the pointer for [`glGetGraphicsResetStatusKHR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_robustness"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_robustness"))))]
    pub fn glGetGraphicsResetStatusKHR_is_loaded() -> bool {
        !glGetGraphicsResetStatusKHR_p.load(RELAX).is_null()
    }

    /// [glGetImageHandleNV](http://docs.gl/es3/glGetImageHandleNV)(texture, level, layered, layer, format)
    /// * `texture` class: texture
    /// * `format` group: PixelFormat
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    pub unsafe fn glGetImageHandleNV(
        texture: GLuint,
        level: GLint,
        layered: GLboolean,
        layer: GLint,
        format: GLenum,
    ) -> GLuint64 {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetImageHandleNV({:?}, {:?}, {:?}, {:?}, {:#X});",
                texture,
                level,
                layered,
                layer,
                format
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetImageHandleNV",
            &glGetImageHandleNV_p,
            texture,
            level,
            layered,
            layer,
            format,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetImageHandleNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    static glGetImageHandleNV_p: APcv = ap_null();
    /// Tries to load [`glGetImageHandleNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    pub unsafe fn glGetImageHandleNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetImageHandleNV\0",
            &glGetImageHandleNV_p,
        )
    }
    /// Checks if the pointer for [`glGetImageHandleNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    pub fn glGetImageHandleNV_is_loaded() -> bool {
        !glGetImageHandleNV_p.load(RELAX).is_null()
    }

    /// [glGetInteger64i_v](http://docs.gl/es3/glGet)(target, index, data)
    /// * `target` group: GetPName
    /// * `data` len: COMPSIZE(target)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetInteger64i_v(target: GLenum, index: GLuint, data: *mut GLint64) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetInteger64i_v({:#X}, {:?}, {:p});",
                target,
                index,
                data
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetInteger64i_v",
            &glGetInteger64i_v_p,
            target,
            index,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetInteger64i_v");
        }
        out
    }
    static glGetInteger64i_v_p: APcv = ap_null();
    /// Tries to load [`glGetInteger64i_v`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetInteger64i_v_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetInteger64i_v\0",
            &glGetInteger64i_v_p,
        )
    }
    /// Checks if the pointer for [`glGetInteger64i_v`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetInteger64i_v_is_loaded() -> bool {
        !glGetInteger64i_v_p.load(RELAX).is_null()
    }

    /// [glGetInteger64v](http://docs.gl/es3/glGet)(pname, data)
    /// * `pname` group: GetPName
    /// * `data` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetInteger64v(pname: GLenum, data: *mut GLint64) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetInteger64v({:#X}, {:p});", pname, data);
        }
        let out = call_atomic_ptr_2arg("glGetInteger64v", &glGetInteger64v_p, pname, data);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetInteger64v");
        }
        out
    }
    static glGetInteger64v_p: APcv = ap_null();
    /// Tries to load [`glGetInteger64v`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetInteger64v_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetInteger64v\0", &glGetInteger64v_p)
    }
    /// Checks if the pointer for [`glGetInteger64v`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetInteger64v_is_loaded() -> bool {
        !glGetInteger64v_p.load(RELAX).is_null()
    }

    /// [glGetInteger64vAPPLE](http://docs.gl/es3/glGetInteger64vAPPLE)(pname, params)
    /// * `pname` group: GetPName
    /// * alias of: [`glGetInteger64v`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glGetInteger64vAPPLE(pname: GLenum, params: *mut GLint64) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetInteger64vAPPLE({:#X}, {:p});", pname, params);
        }
        let out = call_atomic_ptr_2arg(
            "glGetInteger64vAPPLE",
            &glGetInteger64vAPPLE_p,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetInteger64vAPPLE");
        }
        out
    }
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    static glGetInteger64vAPPLE_p: APcv = ap_null();
    /// Tries to load [`glGetInteger64vAPPLE`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glGetInteger64vAPPLE_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetInteger64vAPPLE\0",
            &glGetInteger64vAPPLE_p,
        )
    }
    /// Checks if the pointer for [`glGetInteger64vAPPLE`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub fn glGetInteger64vAPPLE_is_loaded() -> bool {
        !glGetInteger64vAPPLE_p.load(RELAX).is_null()
    }

    /// [glGetInteger64vEXT](http://docs.gl/es3/glGetInteger64vEXT)(pname, data)
    /// * `pname` group: GetPName
    /// * `data` len: COMPSIZE(pname)
    /// * alias of: [`glGetInteger64v`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub unsafe fn glGetInteger64vEXT(pname: GLenum, data: *mut GLint64) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetInteger64vEXT({:#X}, {:p});", pname, data);
        }
        let out = call_atomic_ptr_2arg("glGetInteger64vEXT", &glGetInteger64vEXT_p, pname, data);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetInteger64vEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    static glGetInteger64vEXT_p: APcv = ap_null();
    /// Tries to load [`glGetInteger64vEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub unsafe fn glGetInteger64vEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetInteger64vEXT\0",
            &glGetInteger64vEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetInteger64vEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub fn glGetInteger64vEXT_is_loaded() -> bool {
        !glGetInteger64vEXT_p.load(RELAX).is_null()
    }

    /// [glGetIntegeri_v](http://docs.gl/es3/glGet)(target, index, data)
    /// * `target` group: GetPName
    /// * `data` len: COMPSIZE(target)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetIntegeri_v(target: GLenum, index: GLuint, data: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetIntegeri_v({:#X}, {:?}, {:p});",
                target,
                index,
                data
            );
        }
        let out = call_atomic_ptr_3arg("glGetIntegeri_v", &glGetIntegeri_v_p, target, index, data);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetIntegeri_v");
        }
        out
    }
    static glGetIntegeri_v_p: APcv = ap_null();
    /// Tries to load [`glGetIntegeri_v`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetIntegeri_v_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetIntegeri_v\0", &glGetIntegeri_v_p)
    }
    /// Checks if the pointer for [`glGetIntegeri_v`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetIntegeri_v_is_loaded() -> bool {
        !glGetIntegeri_v_p.load(RELAX).is_null()
    }

    /// [glGetIntegeri_vEXT](http://docs.gl/es3/glGetIntegeri_vEXT)(target, index, data)
    /// * `target` group: GetPName
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
    pub unsafe fn glGetIntegeri_vEXT(target: GLenum, index: GLuint, data: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetIntegeri_vEXT({:#X}, {:?}, {:p});",
                target,
                index,
                data
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetIntegeri_vEXT",
            &glGetIntegeri_vEXT_p,
            target,
            index,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetIntegeri_vEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
    static glGetIntegeri_vEXT_p: APcv = ap_null();
    /// Tries to load [`glGetIntegeri_vEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
    pub unsafe fn glGetIntegeri_vEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetIntegeri_vEXT\0",
            &glGetIntegeri_vEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetIntegeri_vEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_multiview_draw_buffers"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_multiview_draw_buffers"))))]
    pub fn glGetIntegeri_vEXT_is_loaded() -> bool {
        !glGetIntegeri_vEXT_p.load(RELAX).is_null()
    }

    /// [glGetIntegerv](http://docs.gl/es3/glGet)(pname, data)
    /// * `pname` group: GetPName
    /// * `data` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetIntegerv(pname: GLenum, data: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetIntegerv({:#X}, {:p});", pname, data);
        }
        let out = call_atomic_ptr_2arg("glGetIntegerv", &glGetIntegerv_p, pname, data);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetIntegerv");
        }
        out
    }
    static glGetIntegerv_p: APcv = ap_null();
    /// Tries to load [`glGetIntegerv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetIntegerv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetIntegerv\0", &glGetIntegerv_p)
    }
    /// Checks if the pointer for [`glGetIntegerv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetIntegerv_is_loaded() -> bool {
        !glGetIntegerv_p.load(RELAX).is_null()
    }

    /// [glGetInternalformatSampleivNV](http://docs.gl/es3/glGetInternalformatSampleivNV)(target, internalformat, samples, pname, count, params)
    /// * `target` group: TextureTarget
    /// * `internalformat` group: InternalFormat
    /// * `pname` group: InternalFormatPName
    /// * `params` len: count
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_internalformat_sample_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_internalformat_sample_query"))))]
    pub unsafe fn glGetInternalformatSampleivNV(
        target: GLenum,
        internalformat: GLenum,
        samples: GLsizei,
        pname: GLenum,
        count: GLsizei,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetInternalformatSampleivNV({:#X}, {:#X}, {:?}, {:#X}, {:?}, {:p});",
                target,
                internalformat,
                samples,
                pname,
                count,
                params
            );
        }
        let out = call_atomic_ptr_6arg(
            "glGetInternalformatSampleivNV",
            &glGetInternalformatSampleivNV_p,
            target,
            internalformat,
            samples,
            pname,
            count,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetInternalformatSampleivNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_internalformat_sample_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_internalformat_sample_query"))))]
    static glGetInternalformatSampleivNV_p: APcv = ap_null();
    /// Tries to load [`glGetInternalformatSampleivNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_internalformat_sample_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_internalformat_sample_query"))))]
    pub unsafe fn glGetInternalformatSampleivNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetInternalformatSampleivNV\0",
            &glGetInternalformatSampleivNV_p,
        )
    }
    /// Checks if the pointer for [`glGetInternalformatSampleivNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_internalformat_sample_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_internalformat_sample_query"))))]
    pub fn glGetInternalformatSampleivNV_is_loaded() -> bool {
        !glGetInternalformatSampleivNV_p.load(RELAX).is_null()
    }

    /// [glGetInternalformativ](http://docs.gl/es3/glGetInternalformativ)(target, internalformat, pname, count, params)
    /// * `target` group: TextureTarget
    /// * `internalformat` group: InternalFormat
    /// * `pname` group: InternalFormatPName
    /// * `params` len: count
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetInternalformativ(
        target: GLenum,
        internalformat: GLenum,
        pname: GLenum,
        count: GLsizei,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetInternalformativ({:#X}, {:#X}, {:#X}, {:?}, {:p});",
                target,
                internalformat,
                pname,
                count,
                params
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetInternalformativ",
            &glGetInternalformativ_p,
            target,
            internalformat,
            pname,
            count,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetInternalformativ");
        }
        out
    }
    static glGetInternalformativ_p: APcv = ap_null();
    /// Tries to load [`glGetInternalformativ`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetInternalformativ_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetInternalformativ\0",
            &glGetInternalformativ_p,
        )
    }
    /// Checks if the pointer for [`glGetInternalformativ`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetInternalformativ_is_loaded() -> bool {
        !glGetInternalformativ_p.load(RELAX).is_null()
    }

    /// [glGetMemoryObjectDetachedResourcesuivNV](http://docs.gl/es3/glGetMemoryObjectDetachedResourcesuivNV)(memory, pname, first, count, params)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_memory_attachment"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
    pub unsafe fn glGetMemoryObjectDetachedResourcesuivNV(
        memory: GLuint,
        pname: GLenum,
        first: GLint,
        count: GLsizei,
        params: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetMemoryObjectDetachedResourcesuivNV({:?}, {:#X}, {:?}, {:?}, {:p});",
                memory,
                pname,
                first,
                count,
                params
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetMemoryObjectDetachedResourcesuivNV",
            &glGetMemoryObjectDetachedResourcesuivNV_p,
            memory,
            pname,
            first,
            count,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetMemoryObjectDetachedResourcesuivNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_memory_attachment"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
    static glGetMemoryObjectDetachedResourcesuivNV_p: APcv = ap_null();
    /// Tries to load [`glGetMemoryObjectDetachedResourcesuivNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_memory_attachment"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
    pub unsafe fn glGetMemoryObjectDetachedResourcesuivNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetMemoryObjectDetachedResourcesuivNV\0",
            &glGetMemoryObjectDetachedResourcesuivNV_p,
        )
    }
    /// Checks if the pointer for [`glGetMemoryObjectDetachedResourcesuivNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_memory_attachment"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_memory_attachment"))))]
    pub fn glGetMemoryObjectDetachedResourcesuivNV_is_loaded() -> bool {
        !glGetMemoryObjectDetachedResourcesuivNV_p
            .load(RELAX)
            .is_null()
    }

    /// [glGetMemoryObjectParameterivEXT](http://docs.gl/es3/glGetMemoryObjectParameterivEXT)(memoryObject, pname, params)
    /// * `pname` group: MemoryObjectParameterName
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub unsafe fn glGetMemoryObjectParameterivEXT(
        memoryObject: GLuint,
        pname: GLenum,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetMemoryObjectParameterivEXT({:?}, {:#X}, {:p});",
                memoryObject,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetMemoryObjectParameterivEXT",
            &glGetMemoryObjectParameterivEXT_p,
            memoryObject,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetMemoryObjectParameterivEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    static glGetMemoryObjectParameterivEXT_p: APcv = ap_null();
    /// Tries to load [`glGetMemoryObjectParameterivEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub unsafe fn glGetMemoryObjectParameterivEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetMemoryObjectParameterivEXT\0",
            &glGetMemoryObjectParameterivEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetMemoryObjectParameterivEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_memory_object"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_memory_object"))))]
    pub fn glGetMemoryObjectParameterivEXT_is_loaded() -> bool {
        !glGetMemoryObjectParameterivEXT_p.load(RELAX).is_null()
    }

    /// [glGetMultisamplefv](http://docs.gl/es3/glGetMultisample)(pname, index, val)
    /// * `pname` group: GetMultisamplePNameNV
    /// * `val` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetMultisamplefv(pname: GLenum, index: GLuint, val: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetMultisamplefv({:#X}, {:?}, {:p});",
                pname,
                index,
                val
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetMultisamplefv",
            &glGetMultisamplefv_p,
            pname,
            index,
            val,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetMultisamplefv");
        }
        out
    }
    static glGetMultisamplefv_p: APcv = ap_null();
    /// Tries to load [`glGetMultisamplefv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetMultisamplefv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetMultisamplefv\0",
            &glGetMultisamplefv_p,
        )
    }
    /// Checks if the pointer for [`glGetMultisamplefv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetMultisamplefv_is_loaded() -> bool {
        !glGetMultisamplefv_p.load(RELAX).is_null()
    }

    /// [glGetNextPerfQueryIdINTEL](http://docs.gl/es3/glGetNextPerfQueryIdINTEL)(queryId, nextQueryId)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetNextPerfQueryIdINTEL(queryId: GLuint, nextQueryId: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetNextPerfQueryIdINTEL({:?}, {:p});",
                queryId,
                nextQueryId
            );
        }
        let out = call_atomic_ptr_2arg(
            "glGetNextPerfQueryIdINTEL",
            &glGetNextPerfQueryIdINTEL_p,
            queryId,
            nextQueryId,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetNextPerfQueryIdINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glGetNextPerfQueryIdINTEL_p: APcv = ap_null();
    /// Tries to load [`glGetNextPerfQueryIdINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetNextPerfQueryIdINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetNextPerfQueryIdINTEL\0",
            &glGetNextPerfQueryIdINTEL_p,
        )
    }
    /// Checks if the pointer for [`glGetNextPerfQueryIdINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glGetNextPerfQueryIdINTEL_is_loaded() -> bool {
        !glGetNextPerfQueryIdINTEL_p.load(RELAX).is_null()
    }

    /// [glGetObjectLabel](http://docs.gl/es3/glGetObjectLabel)(identifier, name, bufSize, length, label)
    /// * `identifier` group: ObjectIdentifier
    /// * `length` len: 1
    /// * `label` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetObjectLabel(
        identifier: GLenum,
        name: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        label: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetObjectLabel({:#X}, {:?}, {:?}, {:p}, {:p});",
                identifier,
                name,
                bufSize,
                length,
                label
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetObjectLabel",
            &glGetObjectLabel_p,
            identifier,
            name,
            bufSize,
            length,
            label,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetObjectLabel");
        }
        out
    }
    static glGetObjectLabel_p: APcv = ap_null();
    /// Tries to load [`glGetObjectLabel`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetObjectLabel_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetObjectLabel\0", &glGetObjectLabel_p)
    }
    /// Checks if the pointer for [`glGetObjectLabel`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetObjectLabel_is_loaded() -> bool {
        !glGetObjectLabel_p.load(RELAX).is_null()
    }

    /// [glGetObjectLabelEXT](http://docs.gl/es3/glGetObjectLabelEXT)(type_, object, bufSize, length, label)
    /// * `length` len: 1
    /// * `label` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_debug_label"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
    pub unsafe fn glGetObjectLabelEXT(
        type_: GLenum,
        object: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        label: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetObjectLabelEXT({:#X}, {:?}, {:?}, {:p}, {:p});",
                type_,
                object,
                bufSize,
                length,
                label
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetObjectLabelEXT",
            &glGetObjectLabelEXT_p,
            type_,
            object,
            bufSize,
            length,
            label,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetObjectLabelEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_debug_label"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
    static glGetObjectLabelEXT_p: APcv = ap_null();
    /// Tries to load [`glGetObjectLabelEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_debug_label"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
    pub unsafe fn glGetObjectLabelEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetObjectLabelEXT\0",
            &glGetObjectLabelEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetObjectLabelEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_debug_label"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_debug_label"))))]
    pub fn glGetObjectLabelEXT_is_loaded() -> bool {
        !glGetObjectLabelEXT_p.load(RELAX).is_null()
    }

    /// [glGetObjectLabelKHR](http://docs.gl/es3/glGetObjectLabelKHR)(identifier, name, bufSize, length, label)
    /// * `label` len: bufSize
    /// * alias of: [`glGetObjectLabel`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glGetObjectLabelKHR(
        identifier: GLenum,
        name: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        label: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetObjectLabelKHR({:#X}, {:?}, {:?}, {:p}, {:p});",
                identifier,
                name,
                bufSize,
                length,
                label
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetObjectLabelKHR",
            &glGetObjectLabelKHR_p,
            identifier,
            name,
            bufSize,
            length,
            label,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetObjectLabelKHR");
        }
        out
    }
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    static glGetObjectLabelKHR_p: APcv = ap_null();
    /// Tries to load [`glGetObjectLabelKHR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glGetObjectLabelKHR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetObjectLabelKHR\0",
            &glGetObjectLabelKHR_p,
        )
    }
    /// Checks if the pointer for [`glGetObjectLabelKHR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub fn glGetObjectLabelKHR_is_loaded() -> bool {
        !glGetObjectLabelKHR_p.load(RELAX).is_null()
    }

    /// [glGetObjectPtrLabel](http://docs.gl/es3/glGetObjectPtrLabel)(ptr, bufSize, length, label)
    /// * `length` len: 1
    /// * `label` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetObjectPtrLabel(
        ptr: *const c_void,
        bufSize: GLsizei,
        length: *mut GLsizei,
        label: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetObjectPtrLabel({:p}, {:?}, {:p}, {:p});",
                ptr,
                bufSize,
                length,
                label
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetObjectPtrLabel",
            &glGetObjectPtrLabel_p,
            ptr,
            bufSize,
            length,
            label,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetObjectPtrLabel");
        }
        out
    }
    static glGetObjectPtrLabel_p: APcv = ap_null();
    /// Tries to load [`glGetObjectPtrLabel`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetObjectPtrLabel_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetObjectPtrLabel\0",
            &glGetObjectPtrLabel_p,
        )
    }
    /// Checks if the pointer for [`glGetObjectPtrLabel`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetObjectPtrLabel_is_loaded() -> bool {
        !glGetObjectPtrLabel_p.load(RELAX).is_null()
    }

    /// [glGetObjectPtrLabelKHR](http://docs.gl/es3/glGetObjectPtrLabelKHR)(ptr, bufSize, length, label)
    /// * `length` len: 1
    /// * `label` len: bufSize
    /// * alias of: [`glGetObjectPtrLabel`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glGetObjectPtrLabelKHR(
        ptr: *const c_void,
        bufSize: GLsizei,
        length: *mut GLsizei,
        label: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetObjectPtrLabelKHR({:p}, {:?}, {:p}, {:p});",
                ptr,
                bufSize,
                length,
                label
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetObjectPtrLabelKHR",
            &glGetObjectPtrLabelKHR_p,
            ptr,
            bufSize,
            length,
            label,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetObjectPtrLabelKHR");
        }
        out
    }
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    static glGetObjectPtrLabelKHR_p: APcv = ap_null();
    /// Tries to load [`glGetObjectPtrLabelKHR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glGetObjectPtrLabelKHR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetObjectPtrLabelKHR\0",
            &glGetObjectPtrLabelKHR_p,
        )
    }
    /// Checks if the pointer for [`glGetObjectPtrLabelKHR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub fn glGetObjectPtrLabelKHR_is_loaded() -> bool {
        !glGetObjectPtrLabelKHR_p.load(RELAX).is_null()
    }

    /// [glGetPathCommandsNV](http://docs.gl/es3/glGetPathCommandsNV)(path, commands)
    /// * `path` group: Path
    /// * `commands` group: PathCommand
    /// * `commands` len: COMPSIZE(path)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathCommandsNV(path: GLuint, commands: *mut GLubyte) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetPathCommandsNV({:?}, {:p});", path, commands);
        }
        let out = call_atomic_ptr_2arg(
            "glGetPathCommandsNV",
            &glGetPathCommandsNV_p,
            path,
            commands,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPathCommandsNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetPathCommandsNV_p: APcv = ap_null();
    /// Tries to load [`glGetPathCommandsNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathCommandsNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPathCommandsNV\0",
            &glGetPathCommandsNV_p,
        )
    }
    /// Checks if the pointer for [`glGetPathCommandsNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetPathCommandsNV_is_loaded() -> bool {
        !glGetPathCommandsNV_p.load(RELAX).is_null()
    }

    /// [glGetPathCoordsNV](http://docs.gl/es3/glGetPathCoordsNV)(path, coords)
    /// * `path` group: Path
    /// * `coords` len: COMPSIZE(path)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathCoordsNV(path: GLuint, coords: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetPathCoordsNV({:?}, {:p});", path, coords);
        }
        let out = call_atomic_ptr_2arg("glGetPathCoordsNV", &glGetPathCoordsNV_p, path, coords);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPathCoordsNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetPathCoordsNV_p: APcv = ap_null();
    /// Tries to load [`glGetPathCoordsNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathCoordsNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPathCoordsNV\0",
            &glGetPathCoordsNV_p,
        )
    }
    /// Checks if the pointer for [`glGetPathCoordsNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetPathCoordsNV_is_loaded() -> bool {
        !glGetPathCoordsNV_p.load(RELAX).is_null()
    }

    /// [glGetPathDashArrayNV](http://docs.gl/es3/glGetPathDashArrayNV)(path, dashArray)
    /// * `path` group: Path
    /// * `dashArray` len: COMPSIZE(path)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathDashArrayNV(path: GLuint, dashArray: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetPathDashArrayNV({:?}, {:p});", path, dashArray);
        }
        let out = call_atomic_ptr_2arg(
            "glGetPathDashArrayNV",
            &glGetPathDashArrayNV_p,
            path,
            dashArray,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPathDashArrayNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetPathDashArrayNV_p: APcv = ap_null();
    /// Tries to load [`glGetPathDashArrayNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathDashArrayNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPathDashArrayNV\0",
            &glGetPathDashArrayNV_p,
        )
    }
    /// Checks if the pointer for [`glGetPathDashArrayNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetPathDashArrayNV_is_loaded() -> bool {
        !glGetPathDashArrayNV_p.load(RELAX).is_null()
    }

    /// [glGetPathLengthNV](http://docs.gl/es3/glGetPathLengthNV)(path, startSegment, numSegments)
    /// * `path` group: Path
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathLengthNV(
        path: GLuint,
        startSegment: GLsizei,
        numSegments: GLsizei,
    ) -> GLfloat {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPathLengthNV({:?}, {:?}, {:?});",
                path,
                startSegment,
                numSegments
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetPathLengthNV",
            &glGetPathLengthNV_p,
            path,
            startSegment,
            numSegments,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPathLengthNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetPathLengthNV_p: APcv = ap_null();
    /// Tries to load [`glGetPathLengthNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathLengthNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPathLengthNV\0",
            &glGetPathLengthNV_p,
        )
    }
    /// Checks if the pointer for [`glGetPathLengthNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetPathLengthNV_is_loaded() -> bool {
        !glGetPathLengthNV_p.load(RELAX).is_null()
    }

    /// [glGetPathMetricRangeNV](http://docs.gl/es3/glGetPathMetricRangeNV)(metricQueryMask, firstPathName, numPaths, stride, metrics)
    /// * `metricQueryMask` group: PathMetricMask
    /// * `firstPathName` group: Path
    /// * `metrics` len: COMPSIZE(metricQueryMask,numPaths,stride)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathMetricRangeNV(
        metricQueryMask: GLbitfield,
        firstPathName: GLuint,
        numPaths: GLsizei,
        stride: GLsizei,
        metrics: *mut GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPathMetricRangeNV({:?}, {:?}, {:?}, {:?}, {:p});",
                metricQueryMask,
                firstPathName,
                numPaths,
                stride,
                metrics
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetPathMetricRangeNV",
            &glGetPathMetricRangeNV_p,
            metricQueryMask,
            firstPathName,
            numPaths,
            stride,
            metrics,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPathMetricRangeNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetPathMetricRangeNV_p: APcv = ap_null();
    /// Tries to load [`glGetPathMetricRangeNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathMetricRangeNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPathMetricRangeNV\0",
            &glGetPathMetricRangeNV_p,
        )
    }
    /// Checks if the pointer for [`glGetPathMetricRangeNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetPathMetricRangeNV_is_loaded() -> bool {
        !glGetPathMetricRangeNV_p.load(RELAX).is_null()
    }

    /// [glGetPathMetricsNV](http://docs.gl/es3/glGetPathMetricsNV)(metricQueryMask, numPaths, pathNameType, paths, pathBase, stride, metrics)
    /// * `metricQueryMask` group: PathMetricMask
    /// * `pathNameType` group: PathElementType
    /// * `paths` group: PathElement
    /// * `paths` len: COMPSIZE(numPaths,pathNameType,paths)
    /// * `pathBase` group: Path
    /// * `metrics` len: COMPSIZE(metricQueryMask,numPaths,stride)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathMetricsNV(
        metricQueryMask: GLbitfield,
        numPaths: GLsizei,
        pathNameType: GLenum,
        paths: *const c_void,
        pathBase: GLuint,
        stride: GLsizei,
        metrics: *mut GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPathMetricsNV({:?}, {:?}, {:#X}, {:p}, {:?}, {:?}, {:p});",
                metricQueryMask,
                numPaths,
                pathNameType,
                paths,
                pathBase,
                stride,
                metrics
            );
        }
        let out = call_atomic_ptr_7arg(
            "glGetPathMetricsNV",
            &glGetPathMetricsNV_p,
            metricQueryMask,
            numPaths,
            pathNameType,
            paths,
            pathBase,
            stride,
            metrics,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPathMetricsNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetPathMetricsNV_p: APcv = ap_null();
    /// Tries to load [`glGetPathMetricsNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathMetricsNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPathMetricsNV\0",
            &glGetPathMetricsNV_p,
        )
    }
    /// Checks if the pointer for [`glGetPathMetricsNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetPathMetricsNV_is_loaded() -> bool {
        !glGetPathMetricsNV_p.load(RELAX).is_null()
    }

    /// [glGetPathParameterfvNV](http://docs.gl/es3/glGetPathParameterfvNV)(path, pname, value)
    /// * `path` group: Path
    /// * `pname` group: PathParameter
    /// * `value` len: 4
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathParameterfvNV(path: GLuint, pname: GLenum, value: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPathParameterfvNV({:?}, {:#X}, {:p});",
                path,
                pname,
                value
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetPathParameterfvNV",
            &glGetPathParameterfvNV_p,
            path,
            pname,
            value,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPathParameterfvNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetPathParameterfvNV_p: APcv = ap_null();
    /// Tries to load [`glGetPathParameterfvNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathParameterfvNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPathParameterfvNV\0",
            &glGetPathParameterfvNV_p,
        )
    }
    /// Checks if the pointer for [`glGetPathParameterfvNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetPathParameterfvNV_is_loaded() -> bool {
        !glGetPathParameterfvNV_p.load(RELAX).is_null()
    }

    /// [glGetPathParameterivNV](http://docs.gl/es3/glGetPathParameterivNV)(path, pname, value)
    /// * `path` group: Path
    /// * `pname` group: PathParameter
    /// * `value` len: 4
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathParameterivNV(path: GLuint, pname: GLenum, value: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPathParameterivNV({:?}, {:#X}, {:p});",
                path,
                pname,
                value
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetPathParameterivNV",
            &glGetPathParameterivNV_p,
            path,
            pname,
            value,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPathParameterivNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetPathParameterivNV_p: APcv = ap_null();
    /// Tries to load [`glGetPathParameterivNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathParameterivNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPathParameterivNV\0",
            &glGetPathParameterivNV_p,
        )
    }
    /// Checks if the pointer for [`glGetPathParameterivNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetPathParameterivNV_is_loaded() -> bool {
        !glGetPathParameterivNV_p.load(RELAX).is_null()
    }

    /// [glGetPathSpacingNV](http://docs.gl/es3/glGetPathSpacingNV)(pathListMode, numPaths, pathNameType, paths, pathBase, advanceScale, kerningScale, transformType, returnedSpacing)
    /// * `pathListMode` group: PathListMode
    /// * `pathNameType` group: PathElementType
    /// * `paths` group: PathElement
    /// * `paths` len: COMPSIZE(numPaths,pathNameType,paths)
    /// * `pathBase` group: Path
    /// * `transformType` group: PathTransformType
    /// * `returnedSpacing` len: COMPSIZE(pathListMode,numPaths)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathSpacingNV(
        pathListMode: GLenum,
        numPaths: GLsizei,
        pathNameType: GLenum,
        paths: *const c_void,
        pathBase: GLuint,
        advanceScale: GLfloat,
        kerningScale: GLfloat,
        transformType: GLenum,
        returnedSpacing: *mut GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPathSpacingNV({:#X}, {:?}, {:#X}, {:p}, {:?}, {:?}, {:?}, {:#X}, {:p});",
                pathListMode,
                numPaths,
                pathNameType,
                paths,
                pathBase,
                advanceScale,
                kerningScale,
                transformType,
                returnedSpacing
            );
        }
        let out = call_atomic_ptr_9arg(
            "glGetPathSpacingNV",
            &glGetPathSpacingNV_p,
            pathListMode,
            numPaths,
            pathNameType,
            paths,
            pathBase,
            advanceScale,
            kerningScale,
            transformType,
            returnedSpacing,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPathSpacingNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetPathSpacingNV_p: APcv = ap_null();
    /// Tries to load [`glGetPathSpacingNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetPathSpacingNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPathSpacingNV\0",
            &glGetPathSpacingNV_p,
        )
    }
    /// Checks if the pointer for [`glGetPathSpacingNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetPathSpacingNV_is_loaded() -> bool {
        !glGetPathSpacingNV_p.load(RELAX).is_null()
    }

    /// [glGetPerfCounterInfoINTEL](http://docs.gl/es3/glGetPerfCounterInfoINTEL)(queryId, counterId, counterNameLength, counterName, counterDescLength, counterDesc, counterOffset, counterDataSize, counterTypeEnum, counterDataTypeEnum, rawCounterMaxValue)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetPerfCounterInfoINTEL(
        queryId: GLuint,
        counterId: GLuint,
        counterNameLength: GLuint,
        counterName: *mut GLchar,
        counterDescLength: GLuint,
        counterDesc: *mut GLchar,
        counterOffset: *mut GLuint,
        counterDataSize: *mut GLuint,
        counterTypeEnum: *mut GLuint,
        counterDataTypeEnum: *mut GLuint,
        rawCounterMaxValue: *mut GLuint64,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfCounterInfoINTEL({:?}, {:?}, {:?}, {:p}, {:?}, {:p}, {:p}, {:p}, {:p}, {:p}, {:p});",
                queryId,
                counterId,
                counterNameLength,
                counterName,
                counterDescLength,
                counterDesc,
                counterOffset,
                counterDataSize,
                counterTypeEnum,
                counterDataTypeEnum,
                rawCounterMaxValue
            );
        }
        let out = call_atomic_ptr_11arg(
            "glGetPerfCounterInfoINTEL",
            &glGetPerfCounterInfoINTEL_p,
            queryId,
            counterId,
            counterNameLength,
            counterName,
            counterDescLength,
            counterDesc,
            counterOffset,
            counterDataSize,
            counterTypeEnum,
            counterDataTypeEnum,
            rawCounterMaxValue,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfCounterInfoINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glGetPerfCounterInfoINTEL_p: APcv = ap_null();
    /// Tries to load [`glGetPerfCounterInfoINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetPerfCounterInfoINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfCounterInfoINTEL\0",
            &glGetPerfCounterInfoINTEL_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfCounterInfoINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glGetPerfCounterInfoINTEL_is_loaded() -> bool {
        !glGetPerfCounterInfoINTEL_p.load(RELAX).is_null()
    }

    /// [glGetPerfMonitorCounterDataAMD](http://docs.gl/es3/glGetPerfMonitorCounterDataAMD)(monitor, pname, dataSize, data, bytesWritten)
    /// * `data` len: dataSize / 4
    /// * `bytesWritten` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorCounterDataAMD(
        monitor: GLuint,
        pname: GLenum,
        dataSize: GLsizei,
        data: *mut GLuint,
        bytesWritten: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfMonitorCounterDataAMD({:?}, {:#X}, {:?}, {:p}, {:p});",
                monitor,
                pname,
                dataSize,
                data,
                bytesWritten
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetPerfMonitorCounterDataAMD",
            &glGetPerfMonitorCounterDataAMD_p,
            monitor,
            pname,
            dataSize,
            data,
            bytesWritten,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfMonitorCounterDataAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glGetPerfMonitorCounterDataAMD_p: APcv = ap_null();
    /// Tries to load [`glGetPerfMonitorCounterDataAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorCounterDataAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfMonitorCounterDataAMD\0",
            &glGetPerfMonitorCounterDataAMD_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfMonitorCounterDataAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glGetPerfMonitorCounterDataAMD_is_loaded() -> bool {
        !glGetPerfMonitorCounterDataAMD_p.load(RELAX).is_null()
    }

    /// [glGetPerfMonitorCounterInfoAMD](http://docs.gl/es3/glGetPerfMonitorCounterInfoAMD)(group, counter, pname, data)
    /// * `data` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorCounterInfoAMD(
        group: GLuint,
        counter: GLuint,
        pname: GLenum,
        data: *mut c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfMonitorCounterInfoAMD({:?}, {:?}, {:#X}, {:p});",
                group,
                counter,
                pname,
                data
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetPerfMonitorCounterInfoAMD",
            &glGetPerfMonitorCounterInfoAMD_p,
            group,
            counter,
            pname,
            data,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfMonitorCounterInfoAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glGetPerfMonitorCounterInfoAMD_p: APcv = ap_null();
    /// Tries to load [`glGetPerfMonitorCounterInfoAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorCounterInfoAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfMonitorCounterInfoAMD\0",
            &glGetPerfMonitorCounterInfoAMD_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfMonitorCounterInfoAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glGetPerfMonitorCounterInfoAMD_is_loaded() -> bool {
        !glGetPerfMonitorCounterInfoAMD_p.load(RELAX).is_null()
    }

    /// [glGetPerfMonitorCounterStringAMD](http://docs.gl/es3/glGetPerfMonitorCounterStringAMD)(group, counter, bufSize, length, counterString)
    /// * `length` len: 1
    /// * `counterString` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorCounterStringAMD(
        group: GLuint,
        counter: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        counterString: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfMonitorCounterStringAMD({:?}, {:?}, {:?}, {:p}, {:p});",
                group,
                counter,
                bufSize,
                length,
                counterString
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetPerfMonitorCounterStringAMD",
            &glGetPerfMonitorCounterStringAMD_p,
            group,
            counter,
            bufSize,
            length,
            counterString,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfMonitorCounterStringAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glGetPerfMonitorCounterStringAMD_p: APcv = ap_null();
    /// Tries to load [`glGetPerfMonitorCounterStringAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorCounterStringAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfMonitorCounterStringAMD\0",
            &glGetPerfMonitorCounterStringAMD_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfMonitorCounterStringAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glGetPerfMonitorCounterStringAMD_is_loaded() -> bool {
        !glGetPerfMonitorCounterStringAMD_p.load(RELAX).is_null()
    }

    /// [glGetPerfMonitorCountersAMD](http://docs.gl/es3/glGetPerfMonitorCountersAMD)(group, numCounters, maxActiveCounters, counterSize, counters)
    /// * `numCounters` len: 1
    /// * `maxActiveCounters` len: 1
    /// * `counters` len: counterSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorCountersAMD(
        group: GLuint,
        numCounters: *mut GLint,
        maxActiveCounters: *mut GLint,
        counterSize: GLsizei,
        counters: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfMonitorCountersAMD({:?}, {:p}, {:p}, {:?}, {:p});",
                group,
                numCounters,
                maxActiveCounters,
                counterSize,
                counters
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetPerfMonitorCountersAMD",
            &glGetPerfMonitorCountersAMD_p,
            group,
            numCounters,
            maxActiveCounters,
            counterSize,
            counters,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfMonitorCountersAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glGetPerfMonitorCountersAMD_p: APcv = ap_null();
    /// Tries to load [`glGetPerfMonitorCountersAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorCountersAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfMonitorCountersAMD\0",
            &glGetPerfMonitorCountersAMD_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfMonitorCountersAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glGetPerfMonitorCountersAMD_is_loaded() -> bool {
        !glGetPerfMonitorCountersAMD_p.load(RELAX).is_null()
    }

    /// [glGetPerfMonitorGroupStringAMD](http://docs.gl/es3/glGetPerfMonitorGroupStringAMD)(group, bufSize, length, groupString)
    /// * `length` len: 1
    /// * `groupString` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorGroupStringAMD(
        group: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        groupString: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfMonitorGroupStringAMD({:?}, {:?}, {:p}, {:p});",
                group,
                bufSize,
                length,
                groupString
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetPerfMonitorGroupStringAMD",
            &glGetPerfMonitorGroupStringAMD_p,
            group,
            bufSize,
            length,
            groupString,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfMonitorGroupStringAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glGetPerfMonitorGroupStringAMD_p: APcv = ap_null();
    /// Tries to load [`glGetPerfMonitorGroupStringAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorGroupStringAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfMonitorGroupStringAMD\0",
            &glGetPerfMonitorGroupStringAMD_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfMonitorGroupStringAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glGetPerfMonitorGroupStringAMD_is_loaded() -> bool {
        !glGetPerfMonitorGroupStringAMD_p.load(RELAX).is_null()
    }

    /// [glGetPerfMonitorGroupsAMD](http://docs.gl/es3/glGetPerfMonitorGroupsAMD)(numGroups, groupsSize, groups)
    /// * `numGroups` len: 1
    /// * `groups` len: groupsSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorGroupsAMD(
        numGroups: *mut GLint,
        groupsSize: GLsizei,
        groups: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfMonitorGroupsAMD({:p}, {:?}, {:p});",
                numGroups,
                groupsSize,
                groups
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetPerfMonitorGroupsAMD",
            &glGetPerfMonitorGroupsAMD_p,
            numGroups,
            groupsSize,
            groups,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfMonitorGroupsAMD");
        }
        out
    }
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    static glGetPerfMonitorGroupsAMD_p: APcv = ap_null();
    /// Tries to load [`glGetPerfMonitorGroupsAMD`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub unsafe fn glGetPerfMonitorGroupsAMD_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfMonitorGroupsAMD\0",
            &glGetPerfMonitorGroupsAMD_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfMonitorGroupsAMD`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_AMD_performance_monitor"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_AMD_performance_monitor"))))]
    pub fn glGetPerfMonitorGroupsAMD_is_loaded() -> bool {
        !glGetPerfMonitorGroupsAMD_p.load(RELAX).is_null()
    }

    /// [glGetPerfQueryDataINTEL](http://docs.gl/es3/glGetPerfQueryDataINTEL)(queryHandle, flags, dataSize, data, bytesWritten)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetPerfQueryDataINTEL(
        queryHandle: GLuint,
        flags: GLuint,
        dataSize: GLsizei,
        data: *mut c_void,
        bytesWritten: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfQueryDataINTEL({:?}, {:?}, {:?}, {:p}, {:p});",
                queryHandle,
                flags,
                dataSize,
                data,
                bytesWritten
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetPerfQueryDataINTEL",
            &glGetPerfQueryDataINTEL_p,
            queryHandle,
            flags,
            dataSize,
            data,
            bytesWritten,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfQueryDataINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glGetPerfQueryDataINTEL_p: APcv = ap_null();
    /// Tries to load [`glGetPerfQueryDataINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetPerfQueryDataINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfQueryDataINTEL\0",
            &glGetPerfQueryDataINTEL_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfQueryDataINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glGetPerfQueryDataINTEL_is_loaded() -> bool {
        !glGetPerfQueryDataINTEL_p.load(RELAX).is_null()
    }

    /// [glGetPerfQueryIdByNameINTEL](http://docs.gl/es3/glGetPerfQueryIdByNameINTEL)(queryName, queryId)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetPerfQueryIdByNameINTEL(queryName: *mut GLchar, queryId: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfQueryIdByNameINTEL({:p}, {:p});",
                queryName,
                queryId
            );
        }
        let out = call_atomic_ptr_2arg(
            "glGetPerfQueryIdByNameINTEL",
            &glGetPerfQueryIdByNameINTEL_p,
            queryName,
            queryId,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfQueryIdByNameINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glGetPerfQueryIdByNameINTEL_p: APcv = ap_null();
    /// Tries to load [`glGetPerfQueryIdByNameINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetPerfQueryIdByNameINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfQueryIdByNameINTEL\0",
            &glGetPerfQueryIdByNameINTEL_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfQueryIdByNameINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glGetPerfQueryIdByNameINTEL_is_loaded() -> bool {
        !glGetPerfQueryIdByNameINTEL_p.load(RELAX).is_null()
    }

    /// [glGetPerfQueryInfoINTEL](http://docs.gl/es3/glGetPerfQueryInfoINTEL)(queryId, queryNameLength, queryName, dataSize, noCounters, noInstances, capsMask)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetPerfQueryInfoINTEL(
        queryId: GLuint,
        queryNameLength: GLuint,
        queryName: *mut GLchar,
        dataSize: *mut GLuint,
        noCounters: *mut GLuint,
        noInstances: *mut GLuint,
        capsMask: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetPerfQueryInfoINTEL({:?}, {:?}, {:p}, {:p}, {:p}, {:p}, {:p});",
                queryId,
                queryNameLength,
                queryName,
                dataSize,
                noCounters,
                noInstances,
                capsMask
            );
        }
        let out = call_atomic_ptr_7arg(
            "glGetPerfQueryInfoINTEL",
            &glGetPerfQueryInfoINTEL_p,
            queryId,
            queryNameLength,
            queryName,
            dataSize,
            noCounters,
            noInstances,
            capsMask,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPerfQueryInfoINTEL");
        }
        out
    }
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    static glGetPerfQueryInfoINTEL_p: APcv = ap_null();
    /// Tries to load [`glGetPerfQueryInfoINTEL`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub unsafe fn glGetPerfQueryInfoINTEL_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetPerfQueryInfoINTEL\0",
            &glGetPerfQueryInfoINTEL_p,
        )
    }
    /// Checks if the pointer for [`glGetPerfQueryInfoINTEL`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_INTEL_performance_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_INTEL_performance_query"))))]
    pub fn glGetPerfQueryInfoINTEL_is_loaded() -> bool {
        !glGetPerfQueryInfoINTEL_p.load(RELAX).is_null()
    }

    /// [glGetPointerv](http://docs.gl/es3/glGetPointerv)(pname, params)
    /// * `pname` group: GetPointervPName
    /// * `params` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetPointerv(pname: GLenum, params: *mut *mut c_void) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetPointerv({:#X}, {:p});", pname, params);
        }
        let out = call_atomic_ptr_2arg("glGetPointerv", &glGetPointerv_p, pname, params);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPointerv");
        }
        out
    }
    static glGetPointerv_p: APcv = ap_null();
    /// Tries to load [`glGetPointerv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetPointerv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetPointerv\0", &glGetPointerv_p)
    }
    /// Checks if the pointer for [`glGetPointerv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetPointerv_is_loaded() -> bool {
        !glGetPointerv_p.load(RELAX).is_null()
    }

    /// [glGetPointervKHR](http://docs.gl/es3/glGetPointervKHR)(pname, params)
    /// * `params` len: 1
    /// * alias of: [`glGetPointerv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glGetPointervKHR(pname: GLenum, params: *mut *mut c_void) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetPointervKHR({:#X}, {:p});", pname, params);
        }
        let out = call_atomic_ptr_2arg("glGetPointervKHR", &glGetPointervKHR_p, pname, params);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetPointervKHR");
        }
        out
    }
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    static glGetPointervKHR_p: APcv = ap_null();
    /// Tries to load [`glGetPointervKHR`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub unsafe fn glGetPointervKHR_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetPointervKHR\0", &glGetPointervKHR_p)
    }
    /// Checks if the pointer for [`glGetPointervKHR`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_KHR_debug"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_KHR_debug"))))]
    pub fn glGetPointervKHR_is_loaded() -> bool {
        !glGetPointervKHR_p.load(RELAX).is_null()
    }

    /// [glGetProgramBinary](http://docs.gl/es3/glGetProgramBinary)(program, bufSize, length, binaryFormat, binary)
    /// * `program` class: program
    /// * `length` len: 1
    /// * `binaryFormat` len: 1
    /// * `binary` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramBinary(
        program: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        binaryFormat: *mut GLenum,
        binary: *mut c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramBinary({:?}, {:?}, {:p}, {:p}, {:p});",
                program,
                bufSize,
                length,
                binaryFormat,
                binary
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetProgramBinary",
            &glGetProgramBinary_p,
            program,
            bufSize,
            length,
            binaryFormat,
            binary,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramBinary");
        }
        out
    }
    static glGetProgramBinary_p: APcv = ap_null();
    /// Tries to load [`glGetProgramBinary`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramBinary_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramBinary\0",
            &glGetProgramBinary_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramBinary`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramBinary_is_loaded() -> bool {
        !glGetProgramBinary_p.load(RELAX).is_null()
    }

    /// [glGetProgramBinaryOES](http://docs.gl/es3/glGetProgramBinaryOES)(program, bufSize, length, binaryFormat, binary)
    /// * `program` class: program
    /// * `length` len: 1
    /// * `binaryFormat` len: 1
    /// * `binary` len: bufSize
    /// * alias of: [`glGetProgramBinary`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_get_program_binary"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
    pub unsafe fn glGetProgramBinaryOES(
        program: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        binaryFormat: *mut GLenum,
        binary: *mut c_void,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramBinaryOES({:?}, {:?}, {:p}, {:p}, {:p});",
                program,
                bufSize,
                length,
                binaryFormat,
                binary
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetProgramBinaryOES",
            &glGetProgramBinaryOES_p,
            program,
            bufSize,
            length,
            binaryFormat,
            binary,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramBinaryOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_get_program_binary"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
    static glGetProgramBinaryOES_p: APcv = ap_null();
    /// Tries to load [`glGetProgramBinaryOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_get_program_binary"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
    pub unsafe fn glGetProgramBinaryOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramBinaryOES\0",
            &glGetProgramBinaryOES_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramBinaryOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_get_program_binary"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_get_program_binary"))))]
    pub fn glGetProgramBinaryOES_is_loaded() -> bool {
        !glGetProgramBinaryOES_p.load(RELAX).is_null()
    }

    /// [glGetProgramInfoLog](http://docs.gl/es3/glGetProgramInfoLog)(program, bufSize, length, infoLog)
    /// * `program` class: program
    /// * `length` len: 1
    /// * `infoLog` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramInfoLog(
        program: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        infoLog: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramInfoLog({:?}, {:?}, {:p}, {:p});",
                program,
                bufSize,
                length,
                infoLog
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetProgramInfoLog",
            &glGetProgramInfoLog_p,
            program,
            bufSize,
            length,
            infoLog,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramInfoLog");
        }
        out
    }
    static glGetProgramInfoLog_p: APcv = ap_null();
    /// Tries to load [`glGetProgramInfoLog`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramInfoLog_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramInfoLog\0",
            &glGetProgramInfoLog_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramInfoLog`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramInfoLog_is_loaded() -> bool {
        !glGetProgramInfoLog_p.load(RELAX).is_null()
    }

    /// [glGetProgramInterfaceiv](http://docs.gl/es3/glGetProgramInterface)(program, programInterface, pname, params)
    /// * `program` class: program
    /// * `programInterface` group: ProgramInterface
    /// * `pname` group: ProgramInterfacePName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramInterfaceiv(
        program: GLuint,
        programInterface: GLenum,
        pname: GLenum,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramInterfaceiv({:?}, {:#X}, {:#X}, {:p});",
                program,
                programInterface,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetProgramInterfaceiv",
            &glGetProgramInterfaceiv_p,
            program,
            programInterface,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramInterfaceiv");
        }
        out
    }
    static glGetProgramInterfaceiv_p: APcv = ap_null();
    /// Tries to load [`glGetProgramInterfaceiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramInterfaceiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramInterfaceiv\0",
            &glGetProgramInterfaceiv_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramInterfaceiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramInterfaceiv_is_loaded() -> bool {
        !glGetProgramInterfaceiv_p.load(RELAX).is_null()
    }

    /// [glGetProgramPipelineInfoLog](http://docs.gl/es3/glGetProgramPipelineInfoLog)(pipeline, bufSize, length, infoLog)
    /// * `pipeline` class: program pipeline
    /// * `length` len: 1
    /// * `infoLog` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramPipelineInfoLog(
        pipeline: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        infoLog: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramPipelineInfoLog({:?}, {:?}, {:p}, {:p});",
                pipeline,
                bufSize,
                length,
                infoLog
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetProgramPipelineInfoLog",
            &glGetProgramPipelineInfoLog_p,
            pipeline,
            bufSize,
            length,
            infoLog,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramPipelineInfoLog");
        }
        out
    }
    static glGetProgramPipelineInfoLog_p: APcv = ap_null();
    /// Tries to load [`glGetProgramPipelineInfoLog`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramPipelineInfoLog_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramPipelineInfoLog\0",
            &glGetProgramPipelineInfoLog_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramPipelineInfoLog`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramPipelineInfoLog_is_loaded() -> bool {
        !glGetProgramPipelineInfoLog_p.load(RELAX).is_null()
    }

    /// [glGetProgramPipelineInfoLogEXT](http://docs.gl/es3/glGetProgramPipelineInfoLogEXT)(pipeline, bufSize, length, infoLog)
    /// * `pipeline` class: program pipeline
    /// * `length` len: 1
    /// * `infoLog` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glGetProgramPipelineInfoLogEXT(
        pipeline: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        infoLog: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramPipelineInfoLogEXT({:?}, {:?}, {:p}, {:p});",
                pipeline,
                bufSize,
                length,
                infoLog
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetProgramPipelineInfoLogEXT",
            &glGetProgramPipelineInfoLogEXT_p,
            pipeline,
            bufSize,
            length,
            infoLog,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramPipelineInfoLogEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    static glGetProgramPipelineInfoLogEXT_p: APcv = ap_null();
    /// Tries to load [`glGetProgramPipelineInfoLogEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glGetProgramPipelineInfoLogEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramPipelineInfoLogEXT\0",
            &glGetProgramPipelineInfoLogEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramPipelineInfoLogEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub fn glGetProgramPipelineInfoLogEXT_is_loaded() -> bool {
        !glGetProgramPipelineInfoLogEXT_p.load(RELAX).is_null()
    }

    /// [glGetProgramPipelineiv](http://docs.gl/es3/glGetProgramPipeline)(pipeline, pname, params)
    /// * `pipeline` class: program pipeline
    /// * `pname` group: PipelineParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramPipelineiv(pipeline: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramPipelineiv({:?}, {:#X}, {:p});",
                pipeline,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetProgramPipelineiv",
            &glGetProgramPipelineiv_p,
            pipeline,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramPipelineiv");
        }
        out
    }
    static glGetProgramPipelineiv_p: APcv = ap_null();
    /// Tries to load [`glGetProgramPipelineiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramPipelineiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramPipelineiv\0",
            &glGetProgramPipelineiv_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramPipelineiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramPipelineiv_is_loaded() -> bool {
        !glGetProgramPipelineiv_p.load(RELAX).is_null()
    }

    /// [glGetProgramPipelineivEXT](http://docs.gl/es3/glGetProgramPipelineivEXT)(pipeline, pname, params)
    /// * `pipeline` class: program pipeline
    /// * `pname` group: PipelineParameterName
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glGetProgramPipelineivEXT(pipeline: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramPipelineivEXT({:?}, {:#X}, {:p});",
                pipeline,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetProgramPipelineivEXT",
            &glGetProgramPipelineivEXT_p,
            pipeline,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramPipelineivEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    static glGetProgramPipelineivEXT_p: APcv = ap_null();
    /// Tries to load [`glGetProgramPipelineivEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub unsafe fn glGetProgramPipelineivEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramPipelineivEXT\0",
            &glGetProgramPipelineivEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramPipelineivEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_separate_shader_objects"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_separate_shader_objects"))))]
    pub fn glGetProgramPipelineivEXT_is_loaded() -> bool {
        !glGetProgramPipelineivEXT_p.load(RELAX).is_null()
    }

    /// [glGetProgramResourceIndex](http://docs.gl/es3/glGetProgramResourceIndex)(program, programInterface, name)
    /// * `program` class: program
    /// * `programInterface` group: ProgramInterface
    /// * `name` len: COMPSIZE(name)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramResourceIndex(
        program: GLuint,
        programInterface: GLenum,
        name: *const GLchar,
    ) -> GLuint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramResourceIndex({:?}, {:#X}, {:p});",
                program,
                programInterface,
                name
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetProgramResourceIndex",
            &glGetProgramResourceIndex_p,
            program,
            programInterface,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramResourceIndex");
        }
        out
    }
    static glGetProgramResourceIndex_p: APcv = ap_null();
    /// Tries to load [`glGetProgramResourceIndex`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramResourceIndex_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramResourceIndex\0",
            &glGetProgramResourceIndex_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramResourceIndex`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramResourceIndex_is_loaded() -> bool {
        !glGetProgramResourceIndex_p.load(RELAX).is_null()
    }

    /// [glGetProgramResourceLocation](http://docs.gl/es3/glGetProgramResourceLocation)(program, programInterface, name)
    /// * `program` class: program
    /// * `programInterface` group: ProgramInterface
    /// * `name` len: COMPSIZE(name)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramResourceLocation(
        program: GLuint,
        programInterface: GLenum,
        name: *const GLchar,
    ) -> GLint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramResourceLocation({:?}, {:#X}, {:p});",
                program,
                programInterface,
                name
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetProgramResourceLocation",
            &glGetProgramResourceLocation_p,
            program,
            programInterface,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramResourceLocation");
        }
        out
    }
    static glGetProgramResourceLocation_p: APcv = ap_null();
    /// Tries to load [`glGetProgramResourceLocation`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramResourceLocation_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramResourceLocation\0",
            &glGetProgramResourceLocation_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramResourceLocation`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramResourceLocation_is_loaded() -> bool {
        !glGetProgramResourceLocation_p.load(RELAX).is_null()
    }

    /// [glGetProgramResourceLocationIndexEXT](http://docs.gl/es3/glGetProgramResourceLocationIndexEXT)(program, programInterface, name)
    /// * `program` class: program
    /// * `programInterface` group: ProgramInterface
    /// * `name` len: COMPSIZE(name)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub unsafe fn glGetProgramResourceLocationIndexEXT(
        program: GLuint,
        programInterface: GLenum,
        name: *const GLchar,
    ) -> GLint {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramResourceLocationIndexEXT({:?}, {:#X}, {:p});",
                program,
                programInterface,
                name
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetProgramResourceLocationIndexEXT",
            &glGetProgramResourceLocationIndexEXT_p,
            program,
            programInterface,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramResourceLocationIndexEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    static glGetProgramResourceLocationIndexEXT_p: APcv = ap_null();
    /// Tries to load [`glGetProgramResourceLocationIndexEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub unsafe fn glGetProgramResourceLocationIndexEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramResourceLocationIndexEXT\0",
            &glGetProgramResourceLocationIndexEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramResourceLocationIndexEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_blend_func_extended"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_blend_func_extended"))))]
    pub fn glGetProgramResourceLocationIndexEXT_is_loaded() -> bool {
        !glGetProgramResourceLocationIndexEXT_p.load(RELAX).is_null()
    }

    /// [glGetProgramResourceName](http://docs.gl/es3/glGetProgramResourceName)(program, programInterface, index, bufSize, length, name)
    /// * `program` class: program
    /// * `programInterface` group: ProgramInterface
    /// * `length` len: 1
    /// * `name` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramResourceName(
        program: GLuint,
        programInterface: GLenum,
        index: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        name: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramResourceName({:?}, {:#X}, {:?}, {:?}, {:p}, {:p});",
                program,
                programInterface,
                index,
                bufSize,
                length,
                name
            );
        }
        let out = call_atomic_ptr_6arg(
            "glGetProgramResourceName",
            &glGetProgramResourceName_p,
            program,
            programInterface,
            index,
            bufSize,
            length,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramResourceName");
        }
        out
    }
    static glGetProgramResourceName_p: APcv = ap_null();
    /// Tries to load [`glGetProgramResourceName`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramResourceName_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramResourceName\0",
            &glGetProgramResourceName_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramResourceName`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramResourceName_is_loaded() -> bool {
        !glGetProgramResourceName_p.load(RELAX).is_null()
    }

    /// [glGetProgramResourcefvNV](http://docs.gl/es3/glGetProgramResourcefvNV)(program, programInterface, index, propCount, props, count, length, params)
    /// * `program` class: program
    /// * `programInterface` group: ProgramInterface
    /// * `length` len: 1
    /// * `params` len: count
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetProgramResourcefvNV(
        program: GLuint,
        programInterface: GLenum,
        index: GLuint,
        propCount: GLsizei,
        props: *const GLenum,
        count: GLsizei,
        length: *mut GLsizei,
        params: *mut GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramResourcefvNV({:?}, {:#X}, {:?}, {:?}, {:p}, {:?}, {:p}, {:p});",
                program,
                programInterface,
                index,
                propCount,
                props,
                count,
                length,
                params
            );
        }
        let out = call_atomic_ptr_8arg(
            "glGetProgramResourcefvNV",
            &glGetProgramResourcefvNV_p,
            program,
            programInterface,
            index,
            propCount,
            props,
            count,
            length,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramResourcefvNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    static glGetProgramResourcefvNV_p: APcv = ap_null();
    /// Tries to load [`glGetProgramResourcefvNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub unsafe fn glGetProgramResourcefvNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramResourcefvNV\0",
            &glGetProgramResourcefvNV_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramResourcefvNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_path_rendering"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_path_rendering"))))]
    pub fn glGetProgramResourcefvNV_is_loaded() -> bool {
        !glGetProgramResourcefvNV_p.load(RELAX).is_null()
    }

    /// [glGetProgramResourceiv](http://docs.gl/es3/glGetProgramResource)(program, programInterface, index, propCount, props, count, length, params)
    /// * `program` class: program
    /// * `programInterface` group: ProgramInterface
    /// * `props` group: ProgramResourceProperty
    /// * `props` len: propCount
    /// * `length` len: 1
    /// * `params` len: count
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramResourceiv(
        program: GLuint,
        programInterface: GLenum,
        index: GLuint,
        propCount: GLsizei,
        props: *const GLenum,
        count: GLsizei,
        length: *mut GLsizei,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramResourceiv({:?}, {:#X}, {:?}, {:?}, {:p}, {:?}, {:p}, {:p});",
                program,
                programInterface,
                index,
                propCount,
                props,
                count,
                length,
                params
            );
        }
        let out = call_atomic_ptr_8arg(
            "glGetProgramResourceiv",
            &glGetProgramResourceiv_p,
            program,
            programInterface,
            index,
            propCount,
            props,
            count,
            length,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramResourceiv");
        }
        out
    }
    static glGetProgramResourceiv_p: APcv = ap_null();
    /// Tries to load [`glGetProgramResourceiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramResourceiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetProgramResourceiv\0",
            &glGetProgramResourceiv_p,
        )
    }
    /// Checks if the pointer for [`glGetProgramResourceiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramResourceiv_is_loaded() -> bool {
        !glGetProgramResourceiv_p.load(RELAX).is_null()
    }

    /// [glGetProgramiv](http://docs.gl/es3/glGetProgram)(program, pname, params)
    /// * `program` class: program
    /// * `pname` group: ProgramPropertyARB
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetProgramiv(program: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetProgramiv({:?}, {:#X}, {:p});",
                program,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg("glGetProgramiv", &glGetProgramiv_p, program, pname, params);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetProgramiv");
        }
        out
    }
    static glGetProgramiv_p: APcv = ap_null();
    /// Tries to load [`glGetProgramiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetProgramiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetProgramiv\0", &glGetProgramiv_p)
    }
    /// Checks if the pointer for [`glGetProgramiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetProgramiv_is_loaded() -> bool {
        !glGetProgramiv_p.load(RELAX).is_null()
    }

    /// [glGetQueryObjecti64vEXT](http://docs.gl/es3/glGetQueryObjecti64vEXT)(id, pname, params)
    /// * `id` class: query
    /// * `pname` group: QueryObjectParameterName
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetQueryObjecti64v`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub unsafe fn glGetQueryObjecti64vEXT(id: GLuint, pname: GLenum, params: *mut GLint64) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetQueryObjecti64vEXT({:?}, {:#X}, {:p});",
                id,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetQueryObjecti64vEXT",
            &glGetQueryObjecti64vEXT_p,
            id,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetQueryObjecti64vEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    static glGetQueryObjecti64vEXT_p: APcv = ap_null();
    /// Tries to load [`glGetQueryObjecti64vEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub unsafe fn glGetQueryObjecti64vEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetQueryObjecti64vEXT\0",
            &glGetQueryObjecti64vEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetQueryObjecti64vEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub fn glGetQueryObjecti64vEXT_is_loaded() -> bool {
        !glGetQueryObjecti64vEXT_p.load(RELAX).is_null()
    }

    /// [glGetQueryObjectivEXT](http://docs.gl/es3/glGetQueryObjectivEXT)(id, pname, params)
    /// * `id` class: query
    /// * `pname` group: QueryObjectParameterName
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetQueryObjectiv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub unsafe fn glGetQueryObjectivEXT(id: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetQueryObjectivEXT({:?}, {:#X}, {:p});",
                id,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetQueryObjectivEXT",
            &glGetQueryObjectivEXT_p,
            id,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetQueryObjectivEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    static glGetQueryObjectivEXT_p: APcv = ap_null();
    /// Tries to load [`glGetQueryObjectivEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub unsafe fn glGetQueryObjectivEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetQueryObjectivEXT\0",
            &glGetQueryObjectivEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetQueryObjectivEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub fn glGetQueryObjectivEXT_is_loaded() -> bool {
        !glGetQueryObjectivEXT_p.load(RELAX).is_null()
    }

    /// [glGetQueryObjectui64vEXT](http://docs.gl/es3/glGetQueryObjectui64vEXT)(id, pname, params)
    /// * `id` class: query
    /// * `pname` group: QueryObjectParameterName
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetQueryObjectui64v`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub unsafe fn glGetQueryObjectui64vEXT(id: GLuint, pname: GLenum, params: *mut GLuint64) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetQueryObjectui64vEXT({:?}, {:#X}, {:p});",
                id,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetQueryObjectui64vEXT",
            &glGetQueryObjectui64vEXT_p,
            id,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetQueryObjectui64vEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    static glGetQueryObjectui64vEXT_p: APcv = ap_null();
    /// Tries to load [`glGetQueryObjectui64vEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub unsafe fn glGetQueryObjectui64vEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetQueryObjectui64vEXT\0",
            &glGetQueryObjectui64vEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetQueryObjectui64vEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_disjoint_timer_query"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_disjoint_timer_query"))))]
    pub fn glGetQueryObjectui64vEXT_is_loaded() -> bool {
        !glGetQueryObjectui64vEXT_p.load(RELAX).is_null()
    }

    /// [glGetQueryObjectuiv](http://docs.gl/es3/glGetQueryObject)(id, pname, params)
    /// * `id` class: query
    /// * `pname` group: QueryObjectParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetQueryObjectuiv(id: GLuint, pname: GLenum, params: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetQueryObjectuiv({:?}, {:#X}, {:p});",
                id,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetQueryObjectuiv",
            &glGetQueryObjectuiv_p,
            id,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetQueryObjectuiv");
        }
        out
    }
    static glGetQueryObjectuiv_p: APcv = ap_null();
    /// Tries to load [`glGetQueryObjectuiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetQueryObjectuiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetQueryObjectuiv\0",
            &glGetQueryObjectuiv_p,
        )
    }
    /// Checks if the pointer for [`glGetQueryObjectuiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetQueryObjectuiv_is_loaded() -> bool {
        !glGetQueryObjectuiv_p.load(RELAX).is_null()
    }

    /// [glGetQueryObjectuivEXT](http://docs.gl/es3/glGetQueryObjectuivEXT)(id, pname, params)
    /// * `id` class: query
    /// * `pname` group: QueryObjectParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glGetQueryObjectuivEXT(id: GLuint, pname: GLenum, params: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetQueryObjectuivEXT({:?}, {:#X}, {:p});",
                id,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetQueryObjectuivEXT",
            &glGetQueryObjectuivEXT_p,
            id,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetQueryObjectuivEXT");
        }
        out
    }
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    static glGetQueryObjectuivEXT_p: APcv = ap_null();
    /// Tries to load [`glGetQueryObjectuivEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glGetQueryObjectuivEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetQueryObjectuivEXT\0",
            &glGetQueryObjectuivEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetQueryObjectuivEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub fn glGetQueryObjectuivEXT_is_loaded() -> bool {
        !glGetQueryObjectuivEXT_p.load(RELAX).is_null()
    }

    /// [glGetQueryiv](http://docs.gl/es3/glGetQuery)(target, pname, params)
    /// * `target` group: QueryTarget
    /// * `pname` group: QueryParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetQueryiv(target: GLenum, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetQueryiv({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg("glGetQueryiv", &glGetQueryiv_p, target, pname, params);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetQueryiv");
        }
        out
    }
    static glGetQueryiv_p: APcv = ap_null();
    /// Tries to load [`glGetQueryiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetQueryiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetQueryiv\0", &glGetQueryiv_p)
    }
    /// Checks if the pointer for [`glGetQueryiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetQueryiv_is_loaded() -> bool {
        !glGetQueryiv_p.load(RELAX).is_null()
    }

    /// [glGetQueryivEXT](http://docs.gl/es3/glGetQueryivEXT)(target, pname, params)
    /// * `target` group: QueryTarget
    /// * `pname` group: QueryParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glGetQueryivEXT(target: GLenum, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetQueryivEXT({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out =
            call_atomic_ptr_3arg("glGetQueryivEXT", &glGetQueryivEXT_p, target, pname, params);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetQueryivEXT");
        }
        out
    }
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    static glGetQueryivEXT_p: APcv = ap_null();
    /// Tries to load [`glGetQueryivEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub unsafe fn glGetQueryivEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetQueryivEXT\0", &glGetQueryivEXT_p)
    }
    /// Checks if the pointer for [`glGetQueryivEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(
        feature = "GL_EXT_disjoint_timer_query",
        feature = "GL_EXT_occlusion_query_boolean"
    ))]
    #[cfg_attr(
        docs_rs,
        doc(cfg(any(
            feature = "GL_EXT_disjoint_timer_query",
            feature = "GL_EXT_occlusion_query_boolean"
        )))
    )]
    pub fn glGetQueryivEXT_is_loaded() -> bool {
        !glGetQueryivEXT_p.load(RELAX).is_null()
    }

    /// [glGetRenderbufferParameteriv](http://docs.gl/es3/glGetRenderbufferParameter)(target, pname, params)
    /// * `target` group: RenderbufferTarget
    /// * `pname` group: RenderbufferParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetRenderbufferParameteriv(target: GLenum, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetRenderbufferParameteriv({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetRenderbufferParameteriv",
            &glGetRenderbufferParameteriv_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetRenderbufferParameteriv");
        }
        out
    }
    static glGetRenderbufferParameteriv_p: APcv = ap_null();
    /// Tries to load [`glGetRenderbufferParameteriv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetRenderbufferParameteriv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetRenderbufferParameteriv\0",
            &glGetRenderbufferParameteriv_p,
        )
    }
    /// Checks if the pointer for [`glGetRenderbufferParameteriv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetRenderbufferParameteriv_is_loaded() -> bool {
        !glGetRenderbufferParameteriv_p.load(RELAX).is_null()
    }

    /// [glGetSamplerParameterIiv](http://docs.gl/es3/glGetSamplerParameter)(sampler, pname, params)
    /// * `sampler` class: sampler
    /// * `pname` group: SamplerParameterI
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetSamplerParameterIiv(sampler: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSamplerParameterIiv({:?}, {:#X}, {:p});",
                sampler,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSamplerParameterIiv",
            &glGetSamplerParameterIiv_p,
            sampler,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSamplerParameterIiv");
        }
        out
    }
    static glGetSamplerParameterIiv_p: APcv = ap_null();
    /// Tries to load [`glGetSamplerParameterIiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetSamplerParameterIiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSamplerParameterIiv\0",
            &glGetSamplerParameterIiv_p,
        )
    }
    /// Checks if the pointer for [`glGetSamplerParameterIiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetSamplerParameterIiv_is_loaded() -> bool {
        !glGetSamplerParameterIiv_p.load(RELAX).is_null()
    }

    /// [glGetSamplerParameterIivEXT](http://docs.gl/es3/glGetSamplerParameterIivEXT)(sampler, pname, params)
    /// * `sampler` class: sampler
    /// * `pname` group: SamplerParameterI
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetSamplerParameterIiv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub unsafe fn glGetSamplerParameterIivEXT(sampler: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSamplerParameterIivEXT({:?}, {:#X}, {:p});",
                sampler,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSamplerParameterIivEXT",
            &glGetSamplerParameterIivEXT_p,
            sampler,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSamplerParameterIivEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    static glGetSamplerParameterIivEXT_p: APcv = ap_null();
    /// Tries to load [`glGetSamplerParameterIivEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub unsafe fn glGetSamplerParameterIivEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSamplerParameterIivEXT\0",
            &glGetSamplerParameterIivEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetSamplerParameterIivEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub fn glGetSamplerParameterIivEXT_is_loaded() -> bool {
        !glGetSamplerParameterIivEXT_p.load(RELAX).is_null()
    }

    /// [glGetSamplerParameterIivOES](http://docs.gl/es3/glGetSamplerParameterIivOES)(sampler, pname, params)
    /// * `sampler` class: sampler
    /// * `pname` group: SamplerParameterI
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetSamplerParameterIiv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub unsafe fn glGetSamplerParameterIivOES(sampler: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSamplerParameterIivOES({:?}, {:#X}, {:p});",
                sampler,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSamplerParameterIivOES",
            &glGetSamplerParameterIivOES_p,
            sampler,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSamplerParameterIivOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    static glGetSamplerParameterIivOES_p: APcv = ap_null();
    /// Tries to load [`glGetSamplerParameterIivOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub unsafe fn glGetSamplerParameterIivOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSamplerParameterIivOES\0",
            &glGetSamplerParameterIivOES_p,
        )
    }
    /// Checks if the pointer for [`glGetSamplerParameterIivOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub fn glGetSamplerParameterIivOES_is_loaded() -> bool {
        !glGetSamplerParameterIivOES_p.load(RELAX).is_null()
    }

    /// [glGetSamplerParameterIuiv](http://docs.gl/es3/glGetSamplerParameter)(sampler, pname, params)
    /// * `sampler` class: sampler
    /// * `pname` group: SamplerParameterI
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetSamplerParameterIuiv(sampler: GLuint, pname: GLenum, params: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSamplerParameterIuiv({:?}, {:#X}, {:p});",
                sampler,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSamplerParameterIuiv",
            &glGetSamplerParameterIuiv_p,
            sampler,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSamplerParameterIuiv");
        }
        out
    }
    static glGetSamplerParameterIuiv_p: APcv = ap_null();
    /// Tries to load [`glGetSamplerParameterIuiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetSamplerParameterIuiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSamplerParameterIuiv\0",
            &glGetSamplerParameterIuiv_p,
        )
    }
    /// Checks if the pointer for [`glGetSamplerParameterIuiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetSamplerParameterIuiv_is_loaded() -> bool {
        !glGetSamplerParameterIuiv_p.load(RELAX).is_null()
    }

    /// [glGetSamplerParameterIuivEXT](http://docs.gl/es3/glGetSamplerParameterIuivEXT)(sampler, pname, params)
    /// * `sampler` class: sampler
    /// * `pname` group: SamplerParameterI
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetSamplerParameterIuiv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub unsafe fn glGetSamplerParameterIuivEXT(
        sampler: GLuint,
        pname: GLenum,
        params: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSamplerParameterIuivEXT({:?}, {:#X}, {:p});",
                sampler,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSamplerParameterIuivEXT",
            &glGetSamplerParameterIuivEXT_p,
            sampler,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSamplerParameterIuivEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    static glGetSamplerParameterIuivEXT_p: APcv = ap_null();
    /// Tries to load [`glGetSamplerParameterIuivEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub unsafe fn glGetSamplerParameterIuivEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSamplerParameterIuivEXT\0",
            &glGetSamplerParameterIuivEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetSamplerParameterIuivEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub fn glGetSamplerParameterIuivEXT_is_loaded() -> bool {
        !glGetSamplerParameterIuivEXT_p.load(RELAX).is_null()
    }

    /// [glGetSamplerParameterIuivOES](http://docs.gl/es3/glGetSamplerParameterIuivOES)(sampler, pname, params)
    /// * `sampler` class: sampler
    /// * `pname` group: SamplerParameterI
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetSamplerParameterIuiv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub unsafe fn glGetSamplerParameterIuivOES(
        sampler: GLuint,
        pname: GLenum,
        params: *mut GLuint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSamplerParameterIuivOES({:?}, {:#X}, {:p});",
                sampler,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSamplerParameterIuivOES",
            &glGetSamplerParameterIuivOES_p,
            sampler,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSamplerParameterIuivOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    static glGetSamplerParameterIuivOES_p: APcv = ap_null();
    /// Tries to load [`glGetSamplerParameterIuivOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub unsafe fn glGetSamplerParameterIuivOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSamplerParameterIuivOES\0",
            &glGetSamplerParameterIuivOES_p,
        )
    }
    /// Checks if the pointer for [`glGetSamplerParameterIuivOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub fn glGetSamplerParameterIuivOES_is_loaded() -> bool {
        !glGetSamplerParameterIuivOES_p.load(RELAX).is_null()
    }

    /// [glGetSamplerParameterfv](http://docs.gl/es3/glGetSamplerParameter)(sampler, pname, params)
    /// * `sampler` class: sampler
    /// * `pname` group: SamplerParameterF
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetSamplerParameterfv(sampler: GLuint, pname: GLenum, params: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSamplerParameterfv({:?}, {:#X}, {:p});",
                sampler,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSamplerParameterfv",
            &glGetSamplerParameterfv_p,
            sampler,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSamplerParameterfv");
        }
        out
    }
    static glGetSamplerParameterfv_p: APcv = ap_null();
    /// Tries to load [`glGetSamplerParameterfv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetSamplerParameterfv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSamplerParameterfv\0",
            &glGetSamplerParameterfv_p,
        )
    }
    /// Checks if the pointer for [`glGetSamplerParameterfv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetSamplerParameterfv_is_loaded() -> bool {
        !glGetSamplerParameterfv_p.load(RELAX).is_null()
    }

    /// [glGetSamplerParameteriv](http://docs.gl/es3/glGetSamplerParameter)(sampler, pname, params)
    /// * `sampler` class: sampler
    /// * `pname` group: SamplerParameterI
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetSamplerParameteriv(sampler: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSamplerParameteriv({:?}, {:#X}, {:p});",
                sampler,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSamplerParameteriv",
            &glGetSamplerParameteriv_p,
            sampler,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSamplerParameteriv");
        }
        out
    }
    static glGetSamplerParameteriv_p: APcv = ap_null();
    /// Tries to load [`glGetSamplerParameteriv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetSamplerParameteriv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSamplerParameteriv\0",
            &glGetSamplerParameteriv_p,
        )
    }
    /// Checks if the pointer for [`glGetSamplerParameteriv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetSamplerParameteriv_is_loaded() -> bool {
        !glGetSamplerParameteriv_p.load(RELAX).is_null()
    }

    /// [glGetSemaphoreParameterivNV](http://docs.gl/es3/glGetSemaphoreParameterivNV)(semaphore, pname, params)
    /// * `pname` group: SemaphoreParameterName
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
    pub unsafe fn glGetSemaphoreParameterivNV(
        semaphore: GLuint,
        pname: GLenum,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSemaphoreParameterivNV({:?}, {:#X}, {:p});",
                semaphore,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSemaphoreParameterivNV",
            &glGetSemaphoreParameterivNV_p,
            semaphore,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSemaphoreParameterivNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
    static glGetSemaphoreParameterivNV_p: APcv = ap_null();
    /// Tries to load [`glGetSemaphoreParameterivNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
    pub unsafe fn glGetSemaphoreParameterivNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSemaphoreParameterivNV\0",
            &glGetSemaphoreParameterivNV_p,
        )
    }
    /// Checks if the pointer for [`glGetSemaphoreParameterivNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_timeline_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_timeline_semaphore"))))]
    pub fn glGetSemaphoreParameterivNV_is_loaded() -> bool {
        !glGetSemaphoreParameterivNV_p.load(RELAX).is_null()
    }

    /// [glGetSemaphoreParameterui64vEXT](http://docs.gl/es3/glGetSemaphoreParameterui64vEXT)(semaphore, pname, params)
    /// * `pname` group: SemaphoreParameterName
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    pub unsafe fn glGetSemaphoreParameterui64vEXT(
        semaphore: GLuint,
        pname: GLenum,
        params: *mut GLuint64,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSemaphoreParameterui64vEXT({:?}, {:#X}, {:p});",
                semaphore,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetSemaphoreParameterui64vEXT",
            &glGetSemaphoreParameterui64vEXT_p,
            semaphore,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSemaphoreParameterui64vEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    static glGetSemaphoreParameterui64vEXT_p: APcv = ap_null();
    /// Tries to load [`glGetSemaphoreParameterui64vEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    pub unsafe fn glGetSemaphoreParameterui64vEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetSemaphoreParameterui64vEXT\0",
            &glGetSemaphoreParameterui64vEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetSemaphoreParameterui64vEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_semaphore"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_semaphore"))))]
    pub fn glGetSemaphoreParameterui64vEXT_is_loaded() -> bool {
        !glGetSemaphoreParameterui64vEXT_p.load(RELAX).is_null()
    }

    /// [glGetShaderInfoLog](http://docs.gl/es3/glGetShaderInfoLog)(shader, bufSize, length, infoLog)
    /// * `shader` class: shader
    /// * `length` len: 1
    /// * `infoLog` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetShaderInfoLog(
        shader: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        infoLog: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetShaderInfoLog({:?}, {:?}, {:p}, {:p});",
                shader,
                bufSize,
                length,
                infoLog
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetShaderInfoLog",
            &glGetShaderInfoLog_p,
            shader,
            bufSize,
            length,
            infoLog,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetShaderInfoLog");
        }
        out
    }
    static glGetShaderInfoLog_p: APcv = ap_null();
    /// Tries to load [`glGetShaderInfoLog`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetShaderInfoLog_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetShaderInfoLog\0",
            &glGetShaderInfoLog_p,
        )
    }
    /// Checks if the pointer for [`glGetShaderInfoLog`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetShaderInfoLog_is_loaded() -> bool {
        !glGetShaderInfoLog_p.load(RELAX).is_null()
    }

    /// [glGetShaderPrecisionFormat](http://docs.gl/es3/glGetShaderPrecisionFormat)(shadertype, precisiontype, range, precision)
    /// * `shadertype` group: ShaderType
    /// * `precisiontype` group: PrecisionType
    /// * `range` len: 2
    /// * `precision` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetShaderPrecisionFormat(
        shadertype: GLenum,
        precisiontype: GLenum,
        range: *mut GLint,
        precision: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetShaderPrecisionFormat({:#X}, {:#X}, {:p}, {:p});",
                shadertype,
                precisiontype,
                range,
                precision
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetShaderPrecisionFormat",
            &glGetShaderPrecisionFormat_p,
            shadertype,
            precisiontype,
            range,
            precision,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetShaderPrecisionFormat");
        }
        out
    }
    static glGetShaderPrecisionFormat_p: APcv = ap_null();
    /// Tries to load [`glGetShaderPrecisionFormat`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetShaderPrecisionFormat_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetShaderPrecisionFormat\0",
            &glGetShaderPrecisionFormat_p,
        )
    }
    /// Checks if the pointer for [`glGetShaderPrecisionFormat`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetShaderPrecisionFormat_is_loaded() -> bool {
        !glGetShaderPrecisionFormat_p.load(RELAX).is_null()
    }

    /// [glGetShaderSource](http://docs.gl/es3/glGetShaderSource)(shader, bufSize, length, source)
    /// * `shader` class: shader
    /// * `length` len: 1
    /// * `source` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetShaderSource(
        shader: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        source: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetShaderSource({:?}, {:?}, {:p}, {:p});",
                shader,
                bufSize,
                length,
                source
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetShaderSource",
            &glGetShaderSource_p,
            shader,
            bufSize,
            length,
            source,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetShaderSource");
        }
        out
    }
    static glGetShaderSource_p: APcv = ap_null();
    /// Tries to load [`glGetShaderSource`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetShaderSource_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetShaderSource\0",
            &glGetShaderSource_p,
        )
    }
    /// Checks if the pointer for [`glGetShaderSource`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetShaderSource_is_loaded() -> bool {
        !glGetShaderSource_p.load(RELAX).is_null()
    }

    /// [glGetShaderiv](http://docs.gl/es3/glGetShaderiv)(shader, pname, params)
    /// * `shader` class: shader
    /// * `pname` group: ShaderParameterName
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetShaderiv(shader: GLuint, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetShaderiv({:?}, {:#X}, {:p});",
                shader,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg("glGetShaderiv", &glGetShaderiv_p, shader, pname, params);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetShaderiv");
        }
        out
    }
    static glGetShaderiv_p: APcv = ap_null();
    /// Tries to load [`glGetShaderiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetShaderiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetShaderiv\0", &glGetShaderiv_p)
    }
    /// Checks if the pointer for [`glGetShaderiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetShaderiv_is_loaded() -> bool {
        !glGetShaderiv_p.load(RELAX).is_null()
    }

    /// [glGetShadingRateImagePaletteNV](http://docs.gl/es3/glGetShadingRateImagePaletteNV)(viewport, entry, rate)
    /// * `rate` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    pub unsafe fn glGetShadingRateImagePaletteNV(
        viewport: GLuint,
        entry: GLuint,
        rate: *mut GLenum,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetShadingRateImagePaletteNV({:?}, {:?}, {:p});",
                viewport,
                entry,
                rate
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetShadingRateImagePaletteNV",
            &glGetShadingRateImagePaletteNV_p,
            viewport,
            entry,
            rate,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetShadingRateImagePaletteNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    static glGetShadingRateImagePaletteNV_p: APcv = ap_null();
    /// Tries to load [`glGetShadingRateImagePaletteNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    pub unsafe fn glGetShadingRateImagePaletteNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetShadingRateImagePaletteNV\0",
            &glGetShadingRateImagePaletteNV_p,
        )
    }
    /// Checks if the pointer for [`glGetShadingRateImagePaletteNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    pub fn glGetShadingRateImagePaletteNV_is_loaded() -> bool {
        !glGetShadingRateImagePaletteNV_p.load(RELAX).is_null()
    }

    /// [glGetShadingRateSampleLocationivNV](http://docs.gl/es3/glGetShadingRateSampleLocationivNV)(rate, samples, index, location)
    /// * `location` len: 3
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    pub unsafe fn glGetShadingRateSampleLocationivNV(
        rate: GLenum,
        samples: GLuint,
        index: GLuint,
        location: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetShadingRateSampleLocationivNV({:#X}, {:?}, {:?}, {:p});",
                rate,
                samples,
                index,
                location
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetShadingRateSampleLocationivNV",
            &glGetShadingRateSampleLocationivNV_p,
            rate,
            samples,
            index,
            location,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetShadingRateSampleLocationivNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    static glGetShadingRateSampleLocationivNV_p: APcv = ap_null();
    /// Tries to load [`glGetShadingRateSampleLocationivNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    pub unsafe fn glGetShadingRateSampleLocationivNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetShadingRateSampleLocationivNV\0",
            &glGetShadingRateSampleLocationivNV_p,
        )
    }
    /// Checks if the pointer for [`glGetShadingRateSampleLocationivNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_shading_rate_image"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_shading_rate_image"))))]
    pub fn glGetShadingRateSampleLocationivNV_is_loaded() -> bool {
        !glGetShadingRateSampleLocationivNV_p.load(RELAX).is_null()
    }

    /// [glGetString](http://docs.gl/es3/glGetString)(name)
    /// * `name` group: StringName
    /// * return value group: String
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetString(name: GLenum) -> *const GLubyte {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetString({:#X});", name);
        }
        let out = call_atomic_ptr_1arg("glGetString", &glGetString_p, name);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetString");
        }
        out
    }
    static glGetString_p: APcv = ap_null();
    /// Tries to load [`glGetString`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetString_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetString\0", &glGetString_p)
    }
    /// Checks if the pointer for [`glGetString`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetString_is_loaded() -> bool {
        !glGetString_p.load(RELAX).is_null()
    }

    /// [glGetStringi](http://docs.gl/es3/glGetString)(name, index)
    /// * `name` group: StringName
    /// * return value group: String
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetStringi(name: GLenum, index: GLuint) -> *const GLubyte {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetStringi({:#X}, {:?});", name, index);
        }
        let out = call_atomic_ptr_2arg("glGetStringi", &glGetStringi_p, name, index);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetStringi");
        }
        out
    }
    static glGetStringi_p: APcv = ap_null();
    /// Tries to load [`glGetStringi`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetStringi_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetStringi\0", &glGetStringi_p)
    }
    /// Checks if the pointer for [`glGetStringi`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetStringi_is_loaded() -> bool {
        !glGetStringi_p.load(RELAX).is_null()
    }

    /// [glGetSynciv](http://docs.gl/es3/glGetSync)(sync, pname, count, length, values)
    /// * `sync` group: sync
    /// * `sync` class: sync
    /// * `pname` group: SyncParameterName
    /// * `length` len: 1
    /// * `values` len: count
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetSynciv(
        sync: GLsync,
        pname: GLenum,
        count: GLsizei,
        length: *mut GLsizei,
        values: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSynciv({:p}, {:#X}, {:?}, {:p}, {:p});",
                sync,
                pname,
                count,
                length,
                values
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetSynciv",
            &glGetSynciv_p,
            sync,
            pname,
            count,
            length,
            values,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSynciv");
        }
        out
    }
    static glGetSynciv_p: APcv = ap_null();
    /// Tries to load [`glGetSynciv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetSynciv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetSynciv\0", &glGetSynciv_p)
    }
    /// Checks if the pointer for [`glGetSynciv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetSynciv_is_loaded() -> bool {
        !glGetSynciv_p.load(RELAX).is_null()
    }

    /// [glGetSyncivAPPLE](http://docs.gl/es3/glGetSyncivAPPLE)(sync, pname, count, length, values)
    /// * `sync` class: sync
    /// * `pname` group: SyncParameterName
    /// * `values` len: count
    /// * alias of: [`glGetSynciv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glGetSyncivAPPLE(
        sync: GLsync,
        pname: GLenum,
        count: GLsizei,
        length: *mut GLsizei,
        values: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetSyncivAPPLE({:p}, {:#X}, {:?}, {:p}, {:p});",
                sync,
                pname,
                count,
                length,
                values
            );
        }
        let out = call_atomic_ptr_5arg(
            "glGetSyncivAPPLE",
            &glGetSyncivAPPLE_p,
            sync,
            pname,
            count,
            length,
            values,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetSyncivAPPLE");
        }
        out
    }
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    static glGetSyncivAPPLE_p: APcv = ap_null();
    /// Tries to load [`glGetSyncivAPPLE`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub unsafe fn glGetSyncivAPPLE_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(get_proc_address, b"glGetSyncivAPPLE\0", &glGetSyncivAPPLE_p)
    }
    /// Checks if the pointer for [`glGetSyncivAPPLE`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_APPLE_sync"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_APPLE_sync"))))]
    pub fn glGetSyncivAPPLE_is_loaded() -> bool {
        !glGetSyncivAPPLE_p.load(RELAX).is_null()
    }

    /// [glGetTexLevelParameterfv](http://docs.gl/es3/glGetTexLevelParameter)(target, level, pname, params)
    /// * `target` group: TextureTarget
    /// * `level` group: CheckedInt32
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetTexLevelParameterfv(
        target: GLenum,
        level: GLint,
        pname: GLenum,
        params: *mut GLfloat,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexLevelParameterfv({:#X}, {:?}, {:#X}, {:p});",
                target,
                level,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetTexLevelParameterfv",
            &glGetTexLevelParameterfv_p,
            target,
            level,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexLevelParameterfv");
        }
        out
    }
    static glGetTexLevelParameterfv_p: APcv = ap_null();
    /// Tries to load [`glGetTexLevelParameterfv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetTexLevelParameterfv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexLevelParameterfv\0",
            &glGetTexLevelParameterfv_p,
        )
    }
    /// Checks if the pointer for [`glGetTexLevelParameterfv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetTexLevelParameterfv_is_loaded() -> bool {
        !glGetTexLevelParameterfv_p.load(RELAX).is_null()
    }

    /// [glGetTexLevelParameteriv](http://docs.gl/es3/glGetTexLevelParameter)(target, level, pname, params)
    /// * `target` group: TextureTarget
    /// * `level` group: CheckedInt32
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetTexLevelParameteriv(
        target: GLenum,
        level: GLint,
        pname: GLenum,
        params: *mut GLint,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexLevelParameteriv({:#X}, {:?}, {:#X}, {:p});",
                target,
                level,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_4arg(
            "glGetTexLevelParameteriv",
            &glGetTexLevelParameteriv_p,
            target,
            level,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexLevelParameteriv");
        }
        out
    }
    static glGetTexLevelParameteriv_p: APcv = ap_null();
    /// Tries to load [`glGetTexLevelParameteriv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetTexLevelParameteriv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexLevelParameteriv\0",
            &glGetTexLevelParameteriv_p,
        )
    }
    /// Checks if the pointer for [`glGetTexLevelParameteriv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetTexLevelParameteriv_is_loaded() -> bool {
        !glGetTexLevelParameteriv_p.load(RELAX).is_null()
    }

    /// [glGetTexParameterIiv](http://docs.gl/es3/glGetTexParameter)(target, pname, params)
    /// * `target` group: TextureTarget
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetTexParameterIiv(target: GLenum, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexParameterIiv({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetTexParameterIiv",
            &glGetTexParameterIiv_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexParameterIiv");
        }
        out
    }
    static glGetTexParameterIiv_p: APcv = ap_null();
    /// Tries to load [`glGetTexParameterIiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetTexParameterIiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexParameterIiv\0",
            &glGetTexParameterIiv_p,
        )
    }
    /// Checks if the pointer for [`glGetTexParameterIiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetTexParameterIiv_is_loaded() -> bool {
        !glGetTexParameterIiv_p.load(RELAX).is_null()
    }

    /// [glGetTexParameterIivEXT](http://docs.gl/es3/glGetTexParameterIivEXT)(target, pname, params)
    /// * `target` group: TextureTarget
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetTexParameterIiv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub unsafe fn glGetTexParameterIivEXT(target: GLenum, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexParameterIivEXT({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetTexParameterIivEXT",
            &glGetTexParameterIivEXT_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexParameterIivEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    static glGetTexParameterIivEXT_p: APcv = ap_null();
    /// Tries to load [`glGetTexParameterIivEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub unsafe fn glGetTexParameterIivEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexParameterIivEXT\0",
            &glGetTexParameterIivEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetTexParameterIivEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub fn glGetTexParameterIivEXT_is_loaded() -> bool {
        !glGetTexParameterIivEXT_p.load(RELAX).is_null()
    }

    /// [glGetTexParameterIivOES](http://docs.gl/es3/glGetTexParameterIivOES)(target, pname, params)
    /// * `target` group: TextureTarget
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetTexParameterIiv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub unsafe fn glGetTexParameterIivOES(target: GLenum, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexParameterIivOES({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetTexParameterIivOES",
            &glGetTexParameterIivOES_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexParameterIivOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    static glGetTexParameterIivOES_p: APcv = ap_null();
    /// Tries to load [`glGetTexParameterIivOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub unsafe fn glGetTexParameterIivOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexParameterIivOES\0",
            &glGetTexParameterIivOES_p,
        )
    }
    /// Checks if the pointer for [`glGetTexParameterIivOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub fn glGetTexParameterIivOES_is_loaded() -> bool {
        !glGetTexParameterIivOES_p.load(RELAX).is_null()
    }

    /// [glGetTexParameterIuiv](http://docs.gl/es3/glGetTexParameter)(target, pname, params)
    /// * `target` group: TextureTarget
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetTexParameterIuiv(target: GLenum, pname: GLenum, params: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexParameterIuiv({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetTexParameterIuiv",
            &glGetTexParameterIuiv_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexParameterIuiv");
        }
        out
    }
    static glGetTexParameterIuiv_p: APcv = ap_null();
    /// Tries to load [`glGetTexParameterIuiv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetTexParameterIuiv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexParameterIuiv\0",
            &glGetTexParameterIuiv_p,
        )
    }
    /// Checks if the pointer for [`glGetTexParameterIuiv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetTexParameterIuiv_is_loaded() -> bool {
        !glGetTexParameterIuiv_p.load(RELAX).is_null()
    }

    /// [glGetTexParameterIuivEXT](http://docs.gl/es3/glGetTexParameterIuivEXT)(target, pname, params)
    /// * `target` group: TextureTarget
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetTexParameterIuiv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub unsafe fn glGetTexParameterIuivEXT(target: GLenum, pname: GLenum, params: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexParameterIuivEXT({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetTexParameterIuivEXT",
            &glGetTexParameterIuivEXT_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexParameterIuivEXT");
        }
        out
    }
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    static glGetTexParameterIuivEXT_p: APcv = ap_null();
    /// Tries to load [`glGetTexParameterIuivEXT`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub unsafe fn glGetTexParameterIuivEXT_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexParameterIuivEXT\0",
            &glGetTexParameterIuivEXT_p,
        )
    }
    /// Checks if the pointer for [`glGetTexParameterIuivEXT`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_EXT_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_EXT_texture_border_clamp"))))]
    pub fn glGetTexParameterIuivEXT_is_loaded() -> bool {
        !glGetTexParameterIuivEXT_p.load(RELAX).is_null()
    }

    /// [glGetTexParameterIuivOES](http://docs.gl/es3/glGetTexParameterIuivOES)(target, pname, params)
    /// * `target` group: TextureTarget
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    /// * alias of: [`glGetTexParameterIuiv`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub unsafe fn glGetTexParameterIuivOES(target: GLenum, pname: GLenum, params: *mut GLuint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexParameterIuivOES({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetTexParameterIuivOES",
            &glGetTexParameterIuivOES_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexParameterIuivOES");
        }
        out
    }
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    static glGetTexParameterIuivOES_p: APcv = ap_null();
    /// Tries to load [`glGetTexParameterIuivOES`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub unsafe fn glGetTexParameterIuivOES_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexParameterIuivOES\0",
            &glGetTexParameterIuivOES_p,
        )
    }
    /// Checks if the pointer for [`glGetTexParameterIuivOES`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_OES_texture_border_clamp"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_OES_texture_border_clamp"))))]
    pub fn glGetTexParameterIuivOES_is_loaded() -> bool {
        !glGetTexParameterIuivOES_p.load(RELAX).is_null()
    }

    /// [glGetTexParameterfv](http://docs.gl/es3/glGetTexParameter)(target, pname, params)
    /// * `target` group: TextureTarget
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetTexParameterfv(target: GLenum, pname: GLenum, params: *mut GLfloat) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexParameterfv({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetTexParameterfv",
            &glGetTexParameterfv_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexParameterfv");
        }
        out
    }
    static glGetTexParameterfv_p: APcv = ap_null();
    /// Tries to load [`glGetTexParameterfv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetTexParameterfv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexParameterfv\0",
            &glGetTexParameterfv_p,
        )
    }
    /// Checks if the pointer for [`glGetTexParameterfv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetTexParameterfv_is_loaded() -> bool {
        !glGetTexParameterfv_p.load(RELAX).is_null()
    }

    /// [glGetTexParameteriv](http://docs.gl/es3/glGetTexParameter)(target, pname, params)
    /// * `target` group: TextureTarget
    /// * `pname` group: GetTextureParameter
    /// * `params` len: COMPSIZE(pname)
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetTexParameteriv(target: GLenum, pname: GLenum, params: *mut GLint) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTexParameteriv({:#X}, {:#X}, {:p});",
                target,
                pname,
                params
            );
        }
        let out = call_atomic_ptr_3arg(
            "glGetTexParameteriv",
            &glGetTexParameteriv_p,
            target,
            pname,
            params,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTexParameteriv");
        }
        out
    }
    static glGetTexParameteriv_p: APcv = ap_null();
    /// Tries to load [`glGetTexParameteriv`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetTexParameteriv_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTexParameteriv\0",
            &glGetTexParameteriv_p,
        )
    }
    /// Checks if the pointer for [`glGetTexParameteriv`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetTexParameteriv_is_loaded() -> bool {
        !glGetTexParameteriv_p.load(RELAX).is_null()
    }

    /// [glGetTextureHandleIMG](http://docs.gl/es3/glGetTextureHandleIMG)(texture)
    /// * `texture` class: texture
    /// * alias of: [`glGetTextureHandleARB`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_IMG_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
    pub unsafe fn glGetTextureHandleIMG(texture: GLuint) -> GLuint64 {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetTextureHandleIMG({:?});", texture);
        }
        let out = call_atomic_ptr_1arg("glGetTextureHandleIMG", &glGetTextureHandleIMG_p, texture);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTextureHandleIMG");
        }
        out
    }
    #[cfg(any(feature = "GL_IMG_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
    static glGetTextureHandleIMG_p: APcv = ap_null();
    /// Tries to load [`glGetTextureHandleIMG`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
    pub unsafe fn glGetTextureHandleIMG_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTextureHandleIMG\0",
            &glGetTextureHandleIMG_p,
        )
    }
    /// Checks if the pointer for [`glGetTextureHandleIMG`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
    pub fn glGetTextureHandleIMG_is_loaded() -> bool {
        !glGetTextureHandleIMG_p.load(RELAX).is_null()
    }

    /// [glGetTextureHandleNV](http://docs.gl/es3/glGetTextureHandleNV)(texture)
    /// * `texture` class: texture
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    pub unsafe fn glGetTextureHandleNV(texture: GLuint) -> GLuint64 {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!("calling glGetTextureHandleNV({:?});", texture);
        }
        let out = call_atomic_ptr_1arg("glGetTextureHandleNV", &glGetTextureHandleNV_p, texture);
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTextureHandleNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    static glGetTextureHandleNV_p: APcv = ap_null();
    /// Tries to load [`glGetTextureHandleNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    pub unsafe fn glGetTextureHandleNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTextureHandleNV\0",
            &glGetTextureHandleNV_p,
        )
    }
    /// Checks if the pointer for [`glGetTextureHandleNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    pub fn glGetTextureHandleNV_is_loaded() -> bool {
        !glGetTextureHandleNV_p.load(RELAX).is_null()
    }

    /// [glGetTextureSamplerHandleIMG](http://docs.gl/es3/glGetTextureSamplerHandleIMG)(texture, sampler)
    /// * `texture` class: texture
    /// * `sampler` class: sampler
    /// * alias of: [`glGetTextureSamplerHandleARB`]
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_IMG_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
    pub unsafe fn glGetTextureSamplerHandleIMG(texture: GLuint, sampler: GLuint) -> GLuint64 {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTextureSamplerHandleIMG({:?}, {:?});",
                texture,
                sampler
            );
        }
        let out = call_atomic_ptr_2arg(
            "glGetTextureSamplerHandleIMG",
            &glGetTextureSamplerHandleIMG_p,
            texture,
            sampler,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTextureSamplerHandleIMG");
        }
        out
    }
    #[cfg(any(feature = "GL_IMG_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
    static glGetTextureSamplerHandleIMG_p: APcv = ap_null();
    /// Tries to load [`glGetTextureSamplerHandleIMG`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
    pub unsafe fn glGetTextureSamplerHandleIMG_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTextureSamplerHandleIMG\0",
            &glGetTextureSamplerHandleIMG_p,
        )
    }
    /// Checks if the pointer for [`glGetTextureSamplerHandleIMG`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_IMG_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_IMG_bindless_texture"))))]
    pub fn glGetTextureSamplerHandleIMG_is_loaded() -> bool {
        !glGetTextureSamplerHandleIMG_p.load(RELAX).is_null()
    }

    /// [glGetTextureSamplerHandleNV](http://docs.gl/es3/glGetTextureSamplerHandleNV)(texture, sampler)
    /// * `texture` class: texture
    /// * `sampler` class: sampler
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    pub unsafe fn glGetTextureSamplerHandleNV(texture: GLuint, sampler: GLuint) -> GLuint64 {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTextureSamplerHandleNV({:?}, {:?});",
                texture,
                sampler
            );
        }
        let out = call_atomic_ptr_2arg(
            "glGetTextureSamplerHandleNV",
            &glGetTextureSamplerHandleNV_p,
            texture,
            sampler,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTextureSamplerHandleNV");
        }
        out
    }
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    static glGetTextureSamplerHandleNV_p: APcv = ap_null();
    /// Tries to load [`glGetTextureSamplerHandleNV`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    pub unsafe fn glGetTextureSamplerHandleNV_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTextureSamplerHandleNV\0",
            &glGetTextureSamplerHandleNV_p,
        )
    }
    /// Checks if the pointer for [`glGetTextureSamplerHandleNV`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    #[cfg(any(feature = "GL_NV_bindless_texture"))]
    #[cfg_attr(docs_rs, doc(cfg(any(feature = "GL_NV_bindless_texture"))))]
    pub fn glGetTextureSamplerHandleNV_is_loaded() -> bool {
        !glGetTextureSamplerHandleNV_p.load(RELAX).is_null()
    }

    /// [glGetTransformFeedbackVarying](http://docs.gl/es3/glGetTransformFeedbackVarying)(program, index, bufSize, length, size, type_, name)
    /// * `program` class: program
    /// * `length` len: 1
    /// * `size` len: 1
    /// * `type_` group: AttributeType
    /// * `type_` len: 1
    /// * `name` len: bufSize
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    pub unsafe fn glGetTransformFeedbackVarying(
        program: GLuint,
        index: GLuint,
        bufSize: GLsizei,
        length: *mut GLsizei,
        size: *mut GLsizei,
        type_: *mut GLenum,
        name: *mut GLchar,
    ) {
        #[cfg(all(debug_assertions, feature = "debug_trace_calls"))]
        {
            trace!(
                "calling glGetTransformFeedbackVarying({:?}, {:?}, {:?}, {:p}, {:p}, {:p}, {:p});",
                program,
                index,
                bufSize,
                length,
                size,
                type_,
                name
            );
        }
        let out = call_atomic_ptr_7arg(
            "glGetTransformFeedbackVarying",
            &glGetTransformFeedbackVarying_p,
            program,
            index,
            bufSize,
            length,
            size,
            type_,
            name,
        );
        #[cfg(all(debug_assertions, feature = "debug_automatic_glGetError"))]
        {
            global_automatic_glGetError("glGetTransformFeedbackVarying");
        }
        out
    }
    static glGetTransformFeedbackVarying_p: APcv = ap_null();
    /// Tries to load [`glGetTransformFeedbackVarying`], returns if a non-null pointer was obtained.
    #[doc(hidden)]
    pub unsafe fn glGetTransformFeedbackVarying_load_with_dyn(
        get_proc_address: &mut dyn FnMut(*const c_char) -> *mut c_void,
    ) -> bool {
        load_dyn_name_atomic_ptr(
            get_proc_address,
            b"glGetTransformFeedbackVarying\0",
            &glGetTransformFeedbackVarying_p,
        )
    }
    /// Checks if the pointer for [`glGetTransformFeedbackVarying`] is loaded (non-null).
    #[inline]
    #[doc(hidden)]
    pub fn glGetTransformFeedbackVarying_is_loaded() -> bool {
        !glGetTransformFeedbackVarying_p.load(RELAX).is_null()
    }

    /// [glGetTranslatedShaderSourceANGLE](http://docs.gl/es3/glGetTranslatedShaderSourceANGLE)(shader, bufSize, length, source)
    /// * `shader` class: shader
    /// * `length` len: 1
    #[cfg_attr(feature = "inline", inline)]
    #[cfg_attr(feature = "inline_always", inline(always))]
    #[cfg(any(feature = "GL_ANGLE_translated_shader_source"))]
    #[cfg_attr(docs_rs, doc(cfg(