use core::mem::MaybeUninit;

use simd_abstraction::generic::hex as sa_hex;
use simd_abstraction::{Bytes32, Load, SIMD256};

use crate::polyfill::SIMD256Ext;
use crate::{AsciiCase, Error, ERROR};

macro_rules! specialize_for {
    ($feature:literal, $ty: ty) => {
        use crate::{AsciiCase, Error};

        #[inline]
        #[target_feature(enable = $feature)]
        pub unsafe fn parse(src: &[u8]) -> Result<[u8; 16], Error> {
            let s = <$ty as simd_abstraction::InstructionSet>::new_unchecked();
            crate::generic::parse(s, src)
        }

        #[inline]
        #[target_feature(enable = $feature)]
        pub unsafe fn parse_simple(src: &[u8]) -> Result<[u8; 16], Error> {
            let s = <$ty as simd_abstraction::InstructionSet>::new_unchecked();
            crate::generic::parse_simple(s, src)
        }

        #[inline]
        #[target_feature(enable = $feature)]
        pub unsafe fn parse_hyphenated(src: &[u8]) -> Result<[u8; 16], Error> {
            let s = <$ty as simd_abstraction::InstructionSet>::new_unchecked();
            crate::generic::parse_hyphenated(s, src)
        }

        #[inline]
        #[target_feature(enable = $feature)]
        pub unsafe fn format_simple(src: &[u8; 16], case: AsciiCase) -> [u8; 32] {
            let s = <$ty as simd_abstraction::InstructionSet>::new_unchecked();
            crate::generic::format_simple(s, src, case)
        }

        #[inline]
        #[target_feature(enable = $feature)]
        pub unsafe fn format_hyphenated(src: &[u8; 16], case: AsciiCase) -> [u8; 36] {
            let s = <$ty as simd_abstraction::InstructionSet>::new_unchecked();
            crate::generic::format_hyphenated(s, src, case)
        }
    };
}

#[inline]
pub(crate) fn parse<S: SIMD256Ext>(s: S, mut src: &[u8]) -> Result<[u8; 16], Error> {
    #[inline]
    fn judge_other(src: &[u8]) -> Result<&[u8], Error> {
        match src.len() {
            // Microsoft GUID
            38 => {
                if src[0] == b'{' && src[37] == b'}' {
                    Ok(&src[1..37])
                } else {
                    Err(ERROR)
                }
            }
            // URN prefixed UUID
            45 => match src.strip_prefix(b"urn:uuid:") {
                Some(s) => Ok(s),
                None => Err(ERROR),
            },
            _ => Err(ERROR),
        }
    }

    let n = src.len();
    if n == 32 {
        return parse_simple(s, src);
    }
    if n != 36 {
        src = judge_other(src)?;
    }
    parse_hyphenated(s, src)
}

#[inline]
pub(crate) fn parse_simple<S: SIMD256>(s: S, src: &[u8]) -> Result<[u8; 16], Error> {
    if src.len() != 32 {
        return Err(ERROR);
    }
    let a = unsafe { s.v256_loadu(src.as_ptr()) };
    hex_decode(s, a)
}

#[inline]
pub(crate) fn parse_hyphenated<S: SIMD256Ext>(s: S, src: &[u8]) -> Result<[u8; 16], Error> {
    if src.len() != 36 {
        return Err(ERROR);
    }

    let x = unsafe {
        [
            *src.get_unchecked(8),
            *src.get_unchecked(13),
            *src.get_unchecked(18),
            *src.get_unchecked(23),
        ]
    };
    if !matches!(x, [b'-', b'-', b'-', b'-']) {
        return Err(ERROR);
    }

    const SHUFFLE: &Bytes32 = &Bytes32([
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, //
        0x09, 0x0a, 0x0b, 0x0c, 0x0e, 0x0f, 0x80, 0x80, //
        0x03, 0x04, 0x05, 0x06, 0x08, 0x09, 0x0a, 0x0b, //
        0x0c, 0x0d, 0x0e, 0x0f, 0x80, 0x80, 0x80, 0x80, //
    ]);

    let base: *const u8 = src.as_ptr();
    unsafe {
        let a0 = s.v256_loadu(base.cast());
        let a1 = s.u8x32_shuffle(a0, s.load(SHUFFLE));
        let a2 = s.i16x16_write_index7(a1, base.add(16).cast::<i16>().read_unaligned());
        let a3 = s.i32x8_write_index7(a2, base.add(32).cast::<i32>().read_unaligned());
        hex_decode(s, a3)
    }
}

#[inline]
pub(crate) fn format_simple<S: SIMD256>(s: S, src: &[u8; 16], case: AsciiCase) -> [u8; 32] {
    unsafe {
        let a = s.v128_loadu(src.as_ptr());
        s.v256_to_bytes(hex_encode(s, a, case))
    }
}

#[inline(always)]
unsafe fn write_unaligned<T>(base: *mut u8, offset: usize, value: T) {
    base.add(offset).cast::<T>().write_unaligned(value)
}

#[inline]
pub(crate) fn format_hyphenated<S: SIMD256>(s: S, src: &[u8; 16], case: AsciiCase) -> [u8; 36] {
    const SHUFFLE: &Bytes32 = &Bytes32([
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, //
        0x80, 0x08, 0x09, 0x0a, 0x0b, 0x80, 0x0c, 0x0d, //
        0x80, 0x80, 0x80, 0x00, 0x01, 0x02, 0x03, 0x80, //
        0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, //
    ]);

    const DASH: &Bytes32 = &Bytes32([
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //
        0x2d, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x00, //
        0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0x00, 0x2d, //
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, //
    ]);

    let mut buf: MaybeUninit<[u8; 36]> = MaybeUninit::uninit();
    let dst = buf.as_mut_ptr().cast::<u8>();

    let a = unsafe { hex_encode(s, s.v128_loadu(src.as_ptr()), case) };

    {
        let a1 = s.u8x32_shuffle(a, s.load(SHUFFLE));
        let a2 = s.v256_or(a1, s.load(DASH));
        unsafe { s.v256_storeu(dst, a2) };
    }

    unsafe {
        let bytes_28_31 = s.i32x4_extract::<3>(s.v128_from_high_v256(a)) as u32;
        let bytes_14_15 = s.i16x8_extract::<7>(s.v128_from_low_v256(a)) as u16;
        write_unaligned(dst, 16, bytes_14_15);
        write_unaligned(dst, 32, bytes_28_31);
        buf.assume_init()
    }
}

#[inline]
fn hex_decode<S: SIMD256>(s: S, a: S::V256) -> Result<[u8; 16], Error> {
    match sa_hex::decode_u8x32(s, a) {
        Ok(ans) => Ok(s.v128_to_bytes(ans)),
        Err(()) => Err(ERROR),
    }
}

const fn char_lut(case: AsciiCase) -> &'static Bytes32 {
    match case {
        AsciiCase::Lower => sa_hex::ENCODE_LOWER_LUT,
        AsciiCase::Upper => sa_hex::ENCODE_UPPER_LUT,
    }
}

#[inline]
fn hex_encode<S: SIMD256>(s: S, a: S::V128, case: AsciiCase) -> S::V256 {
    let lut = s.load(char_lut(case));
    sa_hex::encode_u8x16(s, a, lut)
}
