use simd_abstraction::SIMD256;

pub unsafe trait SIMD256Ext: SIMD256 {
    fn i16x16_write_index7(self, a: Self::V256, x: i16) -> Self::V256;
    fn i32x8_write_index7(self, a: Self::V256, x: i32) -> Self::V256;
}

#[cfg(any(target_arch = "x86", target_arch = "x86_64"))]
mod x86 {
    use super::*;

    #[cfg(target_arch = "x86")]
    use core::arch::x86::*;

    #[cfg(target_arch = "x86_64")]
    use core::arch::x86_64::*;

    unsafe impl SIMD256Ext for simd_abstraction::AVX2 {
        #[inline(always)]
        fn i16x16_write_index7(self, a: Self::V256, x: i16) -> Self::V256 {
            unsafe { _mm256_insert_epi16::<7>(a, x) }
        }

        #[inline(always)]
        fn i32x8_write_index7(self, a: Self::V256, x: i32) -> Self::V256 {
            unsafe { _mm256_insert_epi32::<7>(a, x) }
        }
    }

    unsafe impl SIMD256Ext for simd_abstraction::SSE41 {
        #[inline(always)]
        fn i16x16_write_index7(self, a: Self::V256, x: i16) -> Self::V256 {
            unsafe { (_mm_insert_epi16::<7>(a.0, x as i32), a.1) }
        }

        #[inline(always)]
        fn i32x8_write_index7(self, a: Self::V256, x: i32) -> Self::V256 {
            unsafe { (a.0, _mm_insert_epi32::<3>(a.1, x)) }
        }
    }
}
