use crate::{Error, ERROR};

const HEX_TABLE: [u8; 256] = {
    let mut buf = [0; 256];
    let mut i: u8 = 0;
    loop {
        buf[i as usize] = match i {
            b'0'..=b'9' => i - b'0',
            b'a'..=b'f' => i - b'a' + 10,
            b'A'..=b'F' => i - b'A' + 10,
            _ => 0xff,
        };
        if i == 255 {
            break buf;
        }
        i += 1
    }
};

const SHL4_TABLE: [u8; 256] = {
    let mut buf = [0; 256];
    let mut i: u8 = 0;
    loop {
        buf[i as usize] = i.wrapping_shl(4);
        if i == 255 {
            break buf;
        }
        i += 1;
    }
};

const UPPER_TABLE: [u8; 16] = [
    b'0', b'1', b'2', b'3', b'4', b'5', b'6', b'7', //
    b'8', b'9', b'A', b'B', b'C', b'D', b'E', b'F', //
];

const LOWER_TABLE: [u8; 16] = [
    b'0', b'1', b'2', b'3', b'4', b'5', b'6', b'7', //
    b'8', b'9', b'a', b'b', b'c', b'd', b'e', b'f', //
];

#[inline]
pub const fn parse(s: &[u8]) -> Result<[u8; 16], Error> {
    match (s.len(), s) {
        (32, _) => parse_simple(s),
        (36, s)
        | (38, [b'{', s @ .., b'}'])
        | (45, [b'u', b'r', b'n', b':', b'u', b'u', b'i', b'd', b':', s @ ..]) => {
            parse_hyphenated(s)
        }
        _ => Err(ERROR),
    }
}

#[inline]
pub const fn parse_simple(s: &[u8]) -> Result<[u8; 16], Error> {
    if s.len() != 32 {
        return Err(ERROR);
    }

    let mut buf: [u8; 16] = [0; 16];
    let mut i = 0;
    while i < 16 {
        let h1 = HEX_TABLE[s[i * 2] as usize];
        let h2 = HEX_TABLE[s[i * 2 + 1] as usize];
        if h1 | h2 == 0xff {
            return Err(ERROR);
        }
        buf[i] = SHL4_TABLE[h1 as usize] | h2;
        i += 1;
    }
    Ok(buf)
}

#[inline]
pub const fn parse_hyphenated(s: &[u8]) -> Result<[u8; 16], Error> {
    if s.len() != 36 {
        return Err(ERROR);
    }

    match [s[8], s[13], s[18], s[23]] {
        [b'-', b'-', b'-', b'-'] => {}
        _ => return Err(ERROR),
    }

    let positions: [u8; 8] = [0, 4, 9, 14, 19, 24, 28, 32];
    let mut buf: [u8; 16] = [0; 16];
    let mut j = 0;
    while j < 8 {
        let i = positions[j];
        let h1 = HEX_TABLE[s[i as usize] as usize];
        let h2 = HEX_TABLE[s[(i + 1) as usize] as usize];
        let h3 = HEX_TABLE[s[(i + 2) as usize] as usize];
        let h4 = HEX_TABLE[s[(i + 3) as usize] as usize];
        if h1 | h2 | h3 | h4 == 0xff {
            return Err(ERROR);
        }
        buf[j * 2] = SHL4_TABLE[h1 as usize] | h2;
        buf[j * 2 + 1] = SHL4_TABLE[h3 as usize] | h4;
        j += 1;
    }

    Ok(buf)
}

#[inline]
pub const fn format_simple(src: &[u8; 16], upper: bool) -> [u8; 32] {
    let lut = if upper { &UPPER_TABLE } else { &LOWER_TABLE };
    let mut dst = [0; 32];
    let mut i = 0;
    while i < 16 {
        let x = src[i];
        dst[i * 2] = lut[(x >> 4) as usize];
        dst[i * 2 + 1] = lut[(x & 0x0f) as usize];
        i += 1;
    }
    dst
}

#[inline]
pub const fn format_hyphenated(src: &[u8; 16], upper: bool) -> [u8; 36] {
    let lut = if upper { &UPPER_TABLE } else { &LOWER_TABLE };
    let groups = [(0, 8), (9, 13), (14, 18), (19, 23), (24, 36)];
    let mut dst = [0; 36];

    let mut group_idx = 0;
    let mut i = 0;
    while group_idx < 5 {
        let (start, end) = groups[group_idx];
        let mut j = start;
        while j < end {
            let x = src[i];
            i += 1;

            dst[j] = lut[(x >> 4) as usize];
            dst[j + 1] = lut[(x & 0x0f) as usize];
            j += 2;
        }
        if group_idx < 4 {
            dst[end] = b'-';
        }
        group_idx += 1;
    }
    dst
}

#[test]
fn test_parse() {
    super::tests::test_parse_ok(|s| parse(s.as_bytes()));
    super::tests::test_parse_err(|s| parse(s.as_bytes()));
}

#[test]
fn test_format() {
    super::tests::test_format_simple(format_simple);
    super::tests::test_format_hypenated(format_hyphenated);
}
