use simd_abstraction::InstructionSet;

use crate::Error;

macro_rules! try_simd {
    ($f:ident($($args:tt)*)) => {
        #[cfg(any(target_arch = "x86", target_arch = "x86_64"))]
        {
            if simd_abstraction::AVX2::detect().is_some() {
                return unsafe { $crate::avx2::$f($($args)*) };
            }
            if simd_abstraction::SSE41::detect().is_some() {
                return unsafe { $crate::sse41::$f($($args)*) };
            }
        }
    };
}

#[inline]
pub fn parse(s: &[u8]) -> Result<[u8; 16], Error> {
    try_simd!(parse(s));
    crate::fallback::parse(s)
}

#[inline]
pub fn parse_simple(s: &[u8]) -> Result<[u8; 16], Error> {
    try_simd!(parse_simple(s));
    crate::fallback::parse_simple(s)
}

#[inline]
pub fn parse_hyphenated(s: &[u8]) -> Result<[u8; 16], Error> {
    try_simd!(parse_hyphenated(s));
    crate::fallback::parse_hyphenated(s)
}

#[inline]
pub fn format_simple(src: &[u8; 16], upper: bool) -> [u8; 32] {
    try_simd!(format_simple(src, upper));
    crate::fallback::format_simple(src, upper)
}

#[inline]
pub fn format_hyphenated(src: &[u8; 16], upper: bool) -> [u8; 36] {
    try_simd!(format_hyphenated(src, upper));
    crate::fallback::format_hyphenated(src, upper)
}

#[test]
fn test_parse() {
    crate::tests::test_parse_ok(|s| parse(s.as_bytes()));
    crate::tests::test_parse_err(|s| parse(s.as_bytes()));
}

#[test]
fn test_format() {
    crate::tests::test_format_simple(format_simple);
    crate::tests::test_format_hypenated(format_hyphenated);
}
