use criterion::{black_box, criterion_group, criterion_main, BenchmarkId, Criterion};
use uuid::Uuid;

pub fn bench_parse(c: &mut Criterion) {
    let mut group = c.benchmark_group("uuid-simd-parse");

    let inputs = [
        ("simple", "67e5504410b1426f9247bb680e5fe0c8"),
        ("hyphenated", "67e55044-10b1-426f-9247-bb680e5fe0c8"),
        ("guid", "{67e55044-10b1-426f-9247-bb680e5fe0c8}"),
        ("urn", "urn:uuid:67e55044-10b1-426f-9247-bb680e5fe0c8"),
    ];

    #[allow(clippy::type_complexity)]
    let functions: &[(&str, fn(&str))] = &[
        #[cfg(target_feature = "avx2")]
        ("avx2", |s: &str| unsafe {
            uuid_simd::avx2::parse(s.as_bytes()).unwrap();
        }),
        #[cfg(target_feature = "sse4.1")]
        ("sse4.1", |s: &str| unsafe {
            uuid_simd::sse41::parse(s.as_bytes()).unwrap();
        }),
        ("fallback", |s: &str| {
            uuid_simd::fallback::parse(s.as_bytes()).unwrap();
        }),
        ("uuid-latest", |s: &str| {
            Uuid::try_parse(s).unwrap();
        }),
    ];

    for &(name, f) in functions {
        for (tag, input) in inputs {
            group.bench_with_input(BenchmarkId::new(name, tag), input, |b, s| {
                b.iter(|| f(black_box(s)))
            });
        }
    }
}

criterion_group!(benches, bench_parse);
criterion_main!(benches);
