//! help

extern crate proc_macro;

use proc_macro2::TokenStream;
use quote::quote;

use syn::{
    parse_macro_input, Data, DeriveInput, Field, Fields, FieldsNamed, GenericArgument, Ident,
    PathArguments, Type, TypePath,
};

const PLUGIN: &str = "Plugin";

struct PluginField {
    name: Ident,
    generic: Type,
}

fn get_plugin_field(field: &Field) -> Option<PluginField> {
    let generic = match &field.ty {
        Type::Path(TypePath { path, .. }) => {
            let ty = path.segments.first().unwrap();
            let args = if ty.ident != PLUGIN {
                return None;
            } else {
                &ty.arguments
            };
            if let PathArguments::AngleBracketed(generic) = args {
                if let GenericArgument::Type(ty) = generic.args.first().unwrap() {
                    ty.to_owned()
                } else {
                    return None;
                }
            } else {
                return None;
            }
        }
        _ => {
            return None;
        }
    };

    let name = field.ident.to_owned().unwrap();

    let plugin_field = PluginField { name, generic };

    Some(plugin_field)
}

fn concentric_impl(ast: DeriveInput) -> TokenStream {
    let name = &ast.ident;

    let fields = match &ast.data {
        Data::Struct(v) => &v.fields,
        _ => unimplemented!(),
    };
    let fields = match fields {
        Fields::Named(FieldsNamed { ref named, .. }) => named,
        _ => unimplemented!(),
    };

    let plugins: Vec<PluginField> = fields
        .into_iter()
        .filter_map(|f| get_plugin_field(f))
        .collect();

    let plugins: Vec<(Ident, Type)> = plugins.into_iter().map(|f| (f.name, f.generic)).collect();

    let (plugin_names, plugin_generics): (Vec<Ident>, Vec<Type>) = plugins.into_iter().unzip();

    quote! {

        impl Concentric for #name {
            fn concentric<T>(&mut self, _plugin: &Plugin<T>) -> &mut Self {
                match std::any::type_name::<T>() {
                    #(
                        t if t.eq(std::any::type_name::<#plugin_generics>()) =>
                                self.#plugin_names.copy_from(_plugin),
                    )*
                    _ => {}
                }
                self
            }
        }

    }
}

#[proc_macro_derive(Concentric)]
pub fn concentric_derive(input: proc_macro::TokenStream) -> proc_macro::TokenStream {
    let ast = parse_macro_input!(input as DeriveInput);

    let output = concentric_impl(ast);

    proc_macro::TokenStream::from(output)
}
