use std::collections::HashMap;

use crate::pos::new_pos;

use super::{Color, Context, GameError, Mesh, TypeInter, create_simple_rectangle, graphics};
use super::Interfacing;
use super::{Pos, Size};


pub struct Container {
    pos: Pos,
    size: Size,
    mesh: Mesh,
    t: TypeInter,

    pub clickable: bool,
    contains: HashMap<&'static str, Box<&'static dyn Interfacing>>,
    color: Color,
}

impl Container {
    pub fn new(ctx: &mut Context, pos: Pos, size: Size, color: Color, button: bool) -> Self {
        Self {
            pos, size,
            mesh: create_simple_rectangle(ctx, size),
            t: TypeInter::Container,
            clickable: button,
            contains: HashMap::new(),
            color
        }
    }
}

impl Interfacing for Container {
    fn draw_self(&self, ctx: &mut Context) -> Result<(), GameError> {
        graphics::draw(ctx, &self.mesh, (self.pos, self.color))
    }
    fn draw_recurs(&self, ctx: &mut Context, i: u8) -> Result<(), GameError> {
        self.draw_self(ctx)?;
        if i == 0 { return Ok(()) }
        for c in self.contains.values() {
            c.draw_recurs(ctx, i-1)?;
        }
        Ok(())
    }

    fn draw_all(&self, ctx: &mut Context) -> Result<(), GameError> {
        self.draw_self(ctx)?;
        for c in self.contains.values() {
            c.draw_all(ctx)?;
        }
        Ok(())
    }

    fn get_type(&self) -> &TypeInter {
        &self.t
    }
    fn get_pos(&self) -> &Pos {
        &self.pos
    }
    fn set_pos(&mut self, pos: Pos) -> () {
        self.pos = pos;
    }

    fn get_size(&self) -> &Size {
        &self.size
    }
    fn set_size(&mut self, ctx: &mut Context, size: Size) -> () {
        self.size = size;
        self.mesh = create_simple_rectangle(ctx, size);
    }

    fn add_composant(&mut self, name: &'static str, comp: &'static mut dyn Interfacing) -> () {
        comp.set_pos(new_pos(comp.get_pos().x + self.pos.x, comp.get_pos().y + self.pos.y));
        self.contains.insert(name, Box::new(comp));
    }

    fn is_click(&self, pos: Pos) -> bool {
        pos.x >= self.pos.x && pos.x <= self.pos.x + self.size.x && pos.y >= self.pos.y && pos.y <= self.pos.y + self.size.y && self.clickable
    }

    fn get_text(&self) -> &'static str { "" }
    fn set_text(&mut self, _v: &'static str) -> () { }
}