import LRU from "lru-cache";
import { UpListing, UpObject } from "upend";
import type {
  Address,
  IJob,
  InEntry,
  IValue,
  ListingResult,
  PutResult,
  VaultInfo
} from "upend/types";
import type { EntityListing } from "./entity";

export async function fetchEntity(address: string): Promise<UpObject> {
  const entityFetch = await fetch(`api/obj/${address}`);
  const entityResult = (await entityFetch.json()) as EntityListing;
  const entityListing = new UpListing(entityResult.entries);
  return entityListing.getObject(address);
}

export async function fetchEntry(address: string) {
  const response = await fetch(`api/raw/${address}`);
  const data = await response.json();
  const listing = new UpListing({ address: data });
  return listing.entries[0];
}

const queryOnceLRU = new LRU<string, UpListing>(128);
const inFlightRequests: { [key: string]: Promise<UpListing> } = {};

export async function queryOnce(query: string): Promise<UpListing> {
  const cacheResult = queryOnceLRU.get(query);
  if (!cacheResult) {
    if (!inFlightRequests[query]) {
      console.debug(`Querying: ${query}`);
      inFlightRequests[query] = new Promise((resolve, reject) => {
        fetch("api/query", { method: "POST", body: query, keepalive: true })
          .then(async (response) => {
            resolve(new UpListing(await response.json()));
          })
          .catch((err) => reject(err));
      });
    } else {
      console.debug(`Chaining request for ${query}...`);
    }
    return await inFlightRequests[query];
  } else {
    console.debug(`Returning cached: ${query}`);
    return cacheResult;
  }
}

export async function putEntry(entry: InEntry): Promise<PutResult> {
  const response = await fetch(`api/obj`, {
    method: "PUT",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(entry),
  });

  return await response.json();
}

export async function putEntityAttribute(
  entity: Address,
  attribute: string,
  value: IValue
): Promise<Address> {
  const response = await fetch(`api/obj/${entity}/${attribute}`, {
    method: "PUT",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(value),
  });

  return await response.json();
}

export async function uploadFile(file: File): Promise<PutResult> {
  const formData = new FormData();
  formData.append("file", file);

  const response = await fetch("api/obj", {
    method: "PUT",
    body: formData,
  });

  if (!response.ok) {
    throw Error(await response.text());
  }

  return await response.json();
}

export async function deleteEntry(address: Address): Promise<void> {
  await fetch(`api/obj/${address}`, { method: "DELETE" });
}

export async function getRaw(address: Address, preview = false) {
  return await fetch(`api/${preview ? "thumb" : "raw"}/${address}`);
}

export async function refreshVault() {
  return await fetch("api/refresh", { method: "POST" });
}

export async function nativeOpen(address: Address) {
  return fetch(`api/raw/${address}?native=1`);
}

export async function fetchRoots(): Promise<ListingResult> {
  const response = await fetch("api/hier_roots");
  return await response.json();
}

export async function fetchJobs(): Promise<IJob[]> {
  const response = await fetch("api/jobs");
  return await response.json();
}

export async function fetchAllAttributes(): Promise<string[]> {
  const response = await fetch("api/all/attributes");
  return await response.json();
}

export async function fetchInfo(): Promise<VaultInfo> {
  const response = await fetch("api/info");
  return await response.json();
}
