use crate::util::hash::Hash;
use crate::util::jobs::{Job, JobContainer, State};
use crate::{database::UpEndDatabase, util::hash::b58_encode};
use anyhow::{anyhow, Result};
use log::{debug, trace};
use std::sync::RwLock;
use std::{
    collections::HashMap,
    fs::File,
    io::Write,
    path::{Path, PathBuf},
    sync::{Arc, Mutex},
};

use self::audio::AudioPath;
use self::image::ImagePath;
use self::text::TextPath;
use self::video::VideoPath;

pub mod audio;
pub mod image;
pub mod text;
pub mod video;

pub trait Previewable {
    fn get_thumbnail(&self) -> Result<Option<Vec<u8>>>;
}
pub struct PreviewStore {
    path: PathBuf,
    db: Arc<UpEndDatabase>,

    locks: Mutex<HashMap<Hash, Arc<Mutex<PathBuf>>>>,
}

#[cfg(feature = "previews")]
impl PreviewStore {
    pub fn new<P: AsRef<Path>>(path: P, db: Arc<UpEndDatabase>) -> Self {
        PreviewStore {
            path: PathBuf::from(path.as_ref()),
            db,
            locks: Mutex::new(HashMap::new()),
        }
    }

    fn get_path(&self, hash: &Hash) -> Arc<Mutex<PathBuf>> {
        let mut locks = self.locks.lock().unwrap();
        if let Some(path) = locks.get(hash) {
            path.clone()
        } else {
            let thumbpath = self.path.join(b58_encode(hash));
            let path = Arc::new(Mutex::new(thumbpath));
            locks.insert(hash.clone(), path.clone());
            path
        }
    }

    pub fn get<S>(
        &self,
        hash: Hash,
        mime_type: S,
        job_container: Arc<RwLock<JobContainer>>,
    ) -> Result<Option<PathBuf>>
    where
        S: Into<Option<String>>,
    {
        debug!("Preview for {hash:?} requested...");
        let path_mutex = self.get_path(&hash);
        let thumbpath = path_mutex.lock().unwrap();
        if thumbpath.exists() {
            trace!("Preview for {hash:?} already exists, returning {thumbpath:?}");
            Ok(Some(thumbpath.clone()))
        } else {
            trace!("Calculating preview for {hash:?}...");
            let connection = self.db.connection()?;
            let files = connection.retrieve_file(&hash)?;
            if let Some(file) = files.get(0) {
                let job_id = job_container
                    .write()
                    .unwrap()
                    .add_job(Job::new(
                        None,
                        &format!("Creating preview for {:?}", file.path.file_name().unwrap()),
                    ))
                    .unwrap();

                let mime_type = mime_type.into();

                let mime_type: Option<String> = if mime_type.is_some() {
                    mime_type
                } else {
                    tree_magic_mini::from_filepath(&file.path).map(|m| m.into())
                };

                let preview = match mime_type {
                    Some(tm) if tm.starts_with("text") => TextPath(&file.path).get_thumbnail(),
                    Some(tm) if tm.starts_with("video") || tm == "application/x-matroska" => {
                        VideoPath(&file.path).get_thumbnail()
                    }
                    Some(tm) if tm.starts_with("audio") || tm == "application/x-riff" => {
                        AudioPath(&file.path).get_thumbnail()
                    }
                    Some(tm) if tm.starts_with("image") => ImagePath(&file.path).get_thumbnail(),
                    Some(unknown) => Err(anyhow!("No capability for {:?} thumbnails.", unknown)),
                    _ => Err(anyhow!("Unknown file type, or file doesn't exist.")),
                };

                match preview {
                    Ok(preview) => {
                        trace!("Got preview for {hash:?}.");

                        let _ = job_container
                            .write()
                            .unwrap()
                            .update_state(&job_id, State::Done);

                        if let Some(data) = preview {
                            std::fs::create_dir_all(&self.path)?;
                            let mut file = File::create(&*thumbpath)?;
                            file.write_all(&data)?;
                            Ok(Some(thumbpath.clone()))
                        } else {
                            Ok(None)
                        }
                    }
                    Err(err) => {
                        let _ = job_container
                            .write()
                            .unwrap()
                            .update_state(&job_id, State::Failed);
                        Err(err)
                    }
                }
            } else {
                Err(anyhow!("Object not found, or is not a file."))
            }
        }
    }
}
