// import { useSWR } from "sswr";
import LRU from "lru-cache";
import { derived, Readable } from "svelte/store";
import { UpListing, UpObject } from "upend";
import type { ListingResult } from "upend/types";
import { useSWR } from "../util/fetch";

const queryOnceLRU = new LRU<string, UpListing>(128);
const inFlightRequests: { [key: string]: Promise<UpListing> } = {};

export function useEntity(address: string) {
  const { data, error, revalidate } = useSWR<ListingResult, unknown>(
    `api/obj/${address}`
  );

  const entity: Readable<UpObject | undefined> = derived(data, ($listing) => {
    if ($listing) {
      const listing = new UpListing($listing);
      return listing.getObject(address);
    }
  });

  return {
    entity,
    error,
    revalidate,
  };
}

export async function fetchEntry(address: string) {
  const response = await fetch(`api/raw/${address}`);
  const data = await response.json();
  const listing = new UpListing({ address: data });
  return listing.entries[0];
}

export function query(query: string) {
  console.debug(`Querying: ${query}`);
  const { data, error, revalidate } = useSWR<ListingResult, unknown>(
    "api/query",
    { method: "POST", body: query }
  );

  const result = derived(data, ($values) => {
    return $values ? new UpListing($values) : undefined;
  });

  return {
    result,
    error,
    revalidate,
  };
}

export async function queryOnce(query: string): Promise<UpListing> {
  const cacheResult = queryOnceLRU.get(query);
  if (!cacheResult) {
    if (!inFlightRequests[query]) {
      console.debug(`Querying: ${query}`);
      inFlightRequests[query] = new Promise((resolve, reject) => {
        fetch("api/query", { method: "POST", body: query, keepalive: true })
          .then(async (response) => {
            resolve(new UpListing(await response.json()));
          })
          .catch((err) => reject(err));
      });
    } else {
      console.debug(`Chaining request for ${query}...`);
    }
    return await inFlightRequests[query];
  } else {
    console.debug(`Returning cached: ${query}`);
    return cacheResult;
  }
}
