use anyhow::anyhow;
use std::io::Read;
use std::path::Path;
use std::process::Command;

use anyhow::Result;

use super::Previewable;

pub struct VideoPath<'a>(pub &'a Path);

impl<'a> Previewable for VideoPath<'a> {
    fn get_thumbnail(&self) -> Result<Option<Vec<u8>>> {
        let duration_cmd = Command::new("ffprobe")
            .args(["-v", "error"])
            .args(["-show_entries", "format=duration"])
            .args(["-of", "default=noprint_wrappers=1:nokey=1"])
            .arg(self.0)
            .output()?;
        if !duration_cmd.status.success() {
            return Err(anyhow!(
                "Failed to retrieve file duration: {:?}",
                String::from_utf8_lossy(&duration_cmd.stderr)
            ));
        }
        let duration = String::from_utf8_lossy(&duration_cmd.stdout)
            .trim()
            .parse::<f64>()?;
        let outfile = tempfile::Builder::new().suffix(".png").tempfile()?;
        let thumbnail_cmd = Command::new("ffmpeg")
            .args(["-i", &self.0.to_string_lossy()])
            .args(["-vframes", "1"])
            .args(["-ss", &(duration / 2.0).to_string()])
            .arg(&*outfile.path().to_string_lossy())
            .arg("-y")
            .output()?;

        if !thumbnail_cmd.status.success() {
            return Err(anyhow!(
                "Failed to render thumbnail: {:?}",
                String::from_utf8_lossy(&thumbnail_cmd.stderr)
            ));
        }

        let mut buffer = Vec::new();
        outfile.as_file().read_to_end(&mut buffer)?;
        Ok(Some(buffer))
    }
}
