use crate::addressing::{Address, Addressable};
use crate::database::entry::{Entry, InEntry};
use crate::database::hierarchies::{list_roots, resolve_path, UHierPath};
use crate::database::lang::Query;
use crate::database::UpEndDatabase;
use crate::previews::PreviewStore;
use crate::util::hash::{decode, encode};
use crate::util::jobs::JobContainer;
use actix_files::NamedFile;
use actix_web::error::{ErrorBadRequest, ErrorInternalServerError, ErrorNotFound};
use actix_web::http;
use actix_web::{delete, error, get, post, put, web, Either, Error, HttpResponse};
use anyhow::Result;
use futures_util::StreamExt;
use log::{debug, info, trace};
use serde::Deserialize;
use serde_json::json;
use std::collections::HashMap;
use std::convert::TryFrom;
use std::sync::{Arc, RwLock};

#[cfg(feature = "desktop")]
use is_executable::IsExecutable;

const VERSION: &str = env!("CARGO_PKG_VERSION");

#[derive(Clone)]
pub struct State {
    pub upend: Arc<UpEndDatabase>,
    pub vault_name: Option<String>,
    pub job_container: Arc<RwLock<JobContainer>>,
    pub preview_store: Option<Arc<PreviewStore>>,
}

#[derive(Deserialize)]
pub struct RawRequest {
    native: Option<String>,
}

#[get("/api/raw/{hash}")]
pub async fn get_raw(
    state: web::Data<State>,
    web::Query(query): web::Query<RawRequest>,
    hash: web::Path<String>,
) -> Result<Either<NamedFile, HttpResponse>, Error> {
    let address = Address::decode(&decode(hash.into_inner()).map_err(ErrorInternalServerError)?)
        .map_err(ErrorInternalServerError)?;
    if let Address::Hash(hash) = address {
        let connection = state.upend.connection().map_err(ErrorInternalServerError)?;
        let files = connection
            .retrieve_file(hash)
            .map_err(ErrorInternalServerError)?;
        if let Some(file) = files.get(0) {
            let file_path = state.upend.vault_path.join(&file.path);

            if query.native.is_none() {
                Ok(Either::A(NamedFile::open(file_path)?))
            } else if cfg!(feature = "desktop") {
                #[cfg(feature = "desktop")]
                {
                    info!("Opening {:?}...", file_path);
                    let mut response = HttpResponse::NoContent();
                    let path = if !file_path.is_executable() {
                        file_path
                    } else {
                        response
                            .header(
                                http::header::WARNING,
                                "199 - Opening parent directory due to file being executable.",
                            )
                            .header(
                                http::header::ACCESS_CONTROL_EXPOSE_HEADERS,
                                http::header::WARNING.to_string(),
                            );

                        file_path
                            .parent()
                            .ok_or_else(|| {
                                ErrorInternalServerError("No parent to open as fallback.")
                            })?
                            .to_path_buf()
                    };
                    opener::open(path).map_err(error::ErrorServiceUnavailable)?;
                    Ok(Either::B(response.finish()))
                }

                #[cfg(not(feature = "desktop"))]
                unreachable!()
            } else {
                Err(error::ErrorNotImplemented("Desktop features not enabled."))
            }
        } else {
            Err(error::ErrorNotFound("NOT FOUND"))
        }
    } else {
        Err(ErrorBadRequest(
            "Address does not refer to a rawable object.",
        ))
    }
}

#[derive(Deserialize)]
pub struct QueryRequest {
    query: String,
}

#[get("/api/obj")]
pub async fn get_query(
    state: web::Data<State>,
    web::Query(info): web::Query<QueryRequest>,
) -> Result<HttpResponse, Error> {
    let connection = state.upend.connection().map_err(ErrorInternalServerError)?;

    let in_query: Query = info.query.as_str().parse().map_err(ErrorBadRequest)?;
    let entries = connection
        .query(in_query)
        .map_err(ErrorInternalServerError)?;
    let mut result: HashMap<String, Entry> = HashMap::new();
    for entry in entries {
        result.insert(
            encode(
                entry
                    .address()
                    .map_err(ErrorInternalServerError)?
                    .encode()
                    .map_err(ErrorInternalServerError)?,
            ),
            entry,
        );
    }

    Ok(HttpResponse::Ok().json(&result))
}

trait EntriesAsHash {
    fn as_hash(&self) -> Result<HashMap<String, &Entry>>;
}

impl EntriesAsHash for Vec<Entry> {
    fn as_hash(&self) -> Result<HashMap<String, &Entry>> {
        let mut result: HashMap<String, &Entry> = HashMap::new();
        for entry in self {
            result.insert(encode(entry.address()?.encode()?), entry);
        }

        Ok(result)
    }
}

#[get("/api/obj/{address_str}")]
pub async fn get_object(
    state: web::Data<State>,
    address_str: web::Path<String>,
) -> Result<HttpResponse, Error> {
    let connection = state.upend.connection().map_err(ErrorInternalServerError)?;
    let result: Vec<Entry> = connection
        .retrieve_object(
            Address::decode(&decode(address_str.into_inner()).map_err(ErrorBadRequest)?)
                .map_err(ErrorBadRequest)?,
        )
        .map_err(ErrorInternalServerError)?;

    debug!("{:?}", result);
    Ok(HttpResponse::Ok().json(result.as_hash().map_err(ErrorInternalServerError)?))
}

const MAX_SIZE: usize = 1_000_000;

#[put("/api/obj")]
pub async fn put_object(
    state: web::Data<State>,
    mut payload: web::Payload,
) -> Result<HttpResponse, Error> {
    let connection = state.upend.connection().map_err(ErrorInternalServerError)?;

    let mut body = web::BytesMut::new();
    while let Some(chunk) = payload.next().await {
        let chunk = chunk?;
        // limit max size of in-memory payload
        if (body.len() + chunk.len()) > MAX_SIZE {
            return Err(error::ErrorBadRequest("overflow."));
        }
        body.extend_from_slice(&chunk);
    }

    let in_entry = serde_json::from_slice::<InEntry>(&body).map_err(ErrorBadRequest)?;
    let entry = Entry::try_from(in_entry).map_err(ErrorInternalServerError)?;

    let result_address = connection
        .insert_entry(entry.clone())
        .map_err(ErrorInternalServerError)?;

    Ok(HttpResponse::Ok().json(
        [(
            encode(result_address.encode().map_err(ErrorInternalServerError)?),
            entry,
        )]
        .iter()
        .cloned()
        .collect::<HashMap<String, Entry>>(),
    ))
}

#[delete("/api/obj/{address_str}")]
pub async fn delete_object(
    state: web::Data<State>,
    address_str: web::Path<String>,
) -> Result<HttpResponse, Error> {
    let connection = state.upend.connection().map_err(ErrorInternalServerError)?;
    let _ = connection
        .remove_object(
            Address::decode(&decode(address_str.into_inner()).map_err(ErrorBadRequest)?)
                .map_err(ErrorInternalServerError)?,
        )
        .map_err(ErrorInternalServerError)?;

    Ok(HttpResponse::Ok().finish())
}

#[get("/api/hier/{path:.*}")]
pub async fn list_hier(
    state: web::Data<State>,
    path: web::Path<String>,
) -> Result<HttpResponse, Error> {
    let connection = state.upend.connection().map_err(ErrorInternalServerError)?;
    if path.is_empty() {
        Ok(HttpResponse::MovedPermanently()
            .header("Location", "/api/hier_roots")
            .finish())
    } else {
        let upath: UHierPath = path.into_inner().parse().map_err(ErrorBadRequest)?;
        trace!("Listing path \"{}\"", upath);

        // todo: 500 if actual error occurs
        let path = resolve_path(&connection, &upath, false).map_err(ErrorNotFound)?;
        match path.last() {
            Some(addr) => Ok(HttpResponse::Found()
                .header("Location", format!("/api/obj/{}", addr))
                .finish()),
            None => Ok(HttpResponse::NotFound().finish()),
        }
    }
}

#[get("/api/hier_roots")]
pub async fn list_hier_roots(state: web::Data<State>) -> Result<HttpResponse, Error> {
    let connection = state.upend.connection().map_err(ErrorInternalServerError)?;

    let result = list_roots(&connection)
        .map_err(ErrorInternalServerError)?
        .into_iter()
        .map(|root| connection.retrieve_object(root))
        .collect::<Result<Vec<Vec<Entry>>>>()
        .map_err(ErrorInternalServerError)?
        .concat();

    Ok(HttpResponse::Ok().json(result.as_hash().map_err(ErrorInternalServerError)?))
}

#[post("/api/refresh")]
pub async fn api_refresh(state: web::Data<State>) -> Result<HttpResponse, Error> {
    actix::spawn(crate::filesystem::rescan_vault(
        state.upend.clone(),
        state.job_container.clone(),
    ));
    Ok(HttpResponse::Ok().finish())
}

#[get("/api/files/{hash}")]
pub async fn get_file(
    state: web::Data<State>,
    hash: web::Path<String>,
) -> Result<HttpResponse, Error> {
    let address = Address::decode(&decode(hash.into_inner()).map_err(ErrorInternalServerError)?)
        .map_err(ErrorInternalServerError)?;

    if let Address::Hash(hash) = address {
        let connection = state.upend.connection().map_err(ErrorInternalServerError)?;
        let response = connection
            .retrieve_file(hash)
            .map_err(ErrorInternalServerError)?;

        Ok(HttpResponse::Ok().json(response))
    } else {
        Err(ErrorBadRequest("Address does not refer to a file."))
    }
}

#[get("/api/files/latest")]
pub async fn latest_files(state: web::Data<State>) -> Result<HttpResponse, Error> {
    let connection = state.upend.connection().map_err(ErrorInternalServerError)?;
    let files = connection
        .get_latest_files(100)
        .map_err(ErrorInternalServerError)?;
    Ok(HttpResponse::Ok().json(&files))
}

#[get("/api/jobs")]
pub async fn get_jobs(state: web::Data<State>) -> Result<HttpResponse, Error> {
    let jobs = state.job_container.read().unwrap().get_jobs();
    Ok(HttpResponse::Ok().json(&jobs))
}

#[get("/api/info")]
pub async fn get_info(state: web::Data<State>) -> Result<HttpResponse, Error> {
    Ok(HttpResponse::Ok().json(json!({
        "name": state.vault_name,
        "location": &*state.upend.vault_path,
        "version": VERSION
    })))
}

#[get("/api/thumb/{hash}")]
pub async fn get_thumbnail(
    state: web::Data<State>,
    hash: web::Path<String>,
) -> Result<NamedFile, Error> {
    if let Some(preview_store) = &state.preview_store {
        let address =
            Address::decode(&decode(hash.into_inner()).map_err(ErrorInternalServerError)?)
                .map_err(ErrorInternalServerError)?;
        if let Address::Hash(hash) = address {
            let preview_path = preview_store
                .get(hash)
                .map_err(error::ErrorInternalServerError)?;

            let mut file = NamedFile::open(&preview_path)?.disable_content_disposition();
            if let Some(mime_type) = tree_magic_mini::from_filepath(&preview_path) {
                if let Ok(mime) = mime_type.parse() {
                    file = file.set_content_type(mime);
                }
            }
            Ok(file)
        } else {
            Err(ErrorBadRequest(
                "Address does not refer to a previewable object.",
            ))
        }
    } else {
        Err(error::ErrorNotImplemented("Previews not enabled."))
    }
}
