// Unseemly is a "core" typed language with (typed!) macros.
// You shouldn't write code in Unseemly.
// Instead, you should implement your programming language as Unseemly macros.

#![allow(dead_code, unused_macros, non_snake_case, non_upper_case_globals, deprecated)]
// dead_code and unused_macros are hopefully temporary allowances
// non_snake_case is stylistic, so we can write `non__snake_case`.
// non_upper_case_globals is stylistic ... but maybe thread_locals really ought to be upper case.
// deprecated is temporary, until `Sky` replaces `EnvMBE` (and the deprecated calls are cleaned up)
#![recursion_limit = "128"] // Yikes.

// for testing; requires `cargo +nightly`
// #![feature(log_syntax, trace_macros)]
// trace_macros!(true);

// TODO: turn these into `use` statements in the appropriate places
#[macro_use]
extern crate custom_derive;

use std::{fs::File, io::Read, path::Path};

mod macros;

mod name; // should maybe be moved to `util`; `mbe` needs it

mod util;

mod alpha;
mod ast;
mod beta;
mod read;

mod earley;
mod grammar;
mod unparse;

mod form;

mod ast_walk;
mod expand;
mod ty;
mod ty_compare;
mod walk_mode;

mod runtime;

mod core_extra_forms;
mod core_forms;
mod core_macro_forms;
mod core_qq_forms;
mod core_type_forms;

use crate::{
    ast::Ast,
    name::{n, Name},
    runtime::{
        core_values,
        eval::{eval, Value},
    },
    util::assoc::Assoc,
};
use std::{borrow::Cow, cell::RefCell, io::BufRead};

thread_local! {
    pub static ty_env : RefCell<Assoc<Name, Ast>> = RefCell::new(core_values::core_types());
    pub static val_env : RefCell<Assoc<Name, Value>> = RefCell::new(core_values::core_values());
}

struct LineHelper {
    highlighter: rustyline::highlight::MatchingBracketHighlighter,
    // Braket-matching isn't exactly right,
    //  but running the whole parser to decide whether more lines are needed is probably ... bad.
    validator: rustyline::validate::MatchingBracketValidator,
}

impl LineHelper {
    fn new() -> LineHelper {
        LineHelper {
            highlighter: rustyline::highlight::MatchingBracketHighlighter::new(),
            validator: rustyline::validate::MatchingBracketValidator::new(),
        }
    }
}

impl rustyline::completion::Completer for LineHelper {
    type Candidate = String;

    fn complete(
        &self,
        line: &str,
        pos: usize,
        _ctxt: &rustyline::Context,
    ) -> Result<(usize, Vec<String>), rustyline::error::ReadlineError> {
        let mut res = vec![];
        let (start, word_so_far) = rustyline::completion::extract_word(line, pos, None, b"[({ })]");
        val_env.with(|vals| {
            let vals = vals.borrow();
            for k in vals.iter_keys() {
                if k.sp().starts_with(word_so_far) {
                    res.push(k.sp());
                }
            }
        });
        Ok((start, res))
    }
}

impl rustyline::hint::Hinter for LineHelper {
    type Hint = String;
    fn hint(&self, _line: &str, _pos: usize, _ctxt: &rustyline::Context) -> Option<String> { None }
}

impl rustyline::highlight::Highlighter for LineHelper {
    fn highlight<'l>(&self, line: &'l str, pos: usize) -> Cow<'l, str> {
        self.highlighter.highlight(line, pos)
    }
    fn highlight_prompt<'b, 's: 'b, 'p: 'b>(
        &'s self,
        prompt: &'p str,
        default: bool,
    ) -> Cow<'b, str> {
        self.highlighter.highlight_prompt(prompt, default)
    }
    fn highlight_hint<'h>(&self, hint: &'h str) -> Cow<'h, str> {
        self.highlighter.highlight_hint(hint)
    }
    fn highlight_candidate<'c>(
        &self,
        candidate: &'c str,
        completion: rustyline::config::CompletionType,
    ) -> Cow<'c, str> {
        self.highlighter.highlight_candidate(candidate, completion)
    }
    fn highlight_char(&self, line: &str, pos: usize) -> bool {
        self.highlighter.highlight_char(line, pos)
    }
}

impl rustyline::validate::Validator for LineHelper {
    fn validate(
        &self,
        ctx: &mut rustyline::validate::ValidationContext,
    ) -> rustyline::Result<rustyline::validate::ValidationResult> {
        self.validator.validate(ctx)
    }

    fn validate_while_typing(&self) -> bool { self.validator.validate_while_typing() }
}

impl rustyline::Helper for LineHelper {}

#[cfg_attr(tarpaulin, skip)]
fn main() {
    let arguments: Vec<String> = std::env::args().collect();

    if arguments.len() == 1 {
        repl();
    } else if arguments.len() == 2 {
        let filename = Path::new(&arguments[1]);

        let mut raw_input = String::new();
        File::open(&filename)
            .expect("Error opening file")
            .read_to_string(&mut raw_input)
            .expect("Error reading file");

        // So the file can import (etc.) relative to its own location:
        if let Some(dir) = filename.parent() {
            if dir.is_dir() {
                std::env::set_current_dir(dir).unwrap();
            }
        }
        let result = eval_unseemly_program(&raw_input);

        match result {
            Ok(v) => println!("{}", v),
            Err(e) => println!("\x1b[1;31m✘\x1b[0m {}", e),
        }
    } else if arguments.len() == 3 {
        let (pc, type_env, type_env__phaseless, value_env) =
            core_extra_forms::language_from_file(&std::path::Path::new(&arguments[1]));

        // Run the second program in the language defined by the first:
        let mut second_program = String::new();
        File::open(&Path::new(&arguments[2]))
            .expect("Error opening file")
            .read_to_string(&mut second_program)
            .expect("Error reading file");

        if let Some(dir) = Path::new(&arguments[2]).parent() {
            if dir.is_dir() {
                std::env::set_current_dir(dir).unwrap();
            }
        }
        match eval_program(&second_program, pc, type_env, type_env__phaseless, value_env) {
            Ok(v) => println!("{}", v),
            Err(e) => println!("\x1b[1;31m✘\x1b[0m {}", e),
        }
    }
}

fn repl() {
    let prelude_filename = dirs::home_dir().unwrap().join(".unseemly_prelude");
    let history_filename = dirs::home_dir().unwrap().join(".unseemly_history");

    let mut rl = rustyline::Editor::<LineHelper>::new();
    rl.set_helper(Some(LineHelper::new()));

    let quit = regex::Regex::new(r"\s*quit\s*").unwrap();
    let just_parse = regex::Regex::new(r"^:p (.*)$").unwrap();
    let just_parse_debug_print = regex::Regex::new(r"^:pd (.*)$").unwrap();
    let just_type = regex::Regex::new(r"^:t (.*)$").unwrap();
    let just_eval = regex::Regex::new(r"^:e (.*)$").unwrap();
    let type_and_expand = regex::Regex::new(r"^:x (.*)$").unwrap();
    let canon_type = regex::Regex::new(r"^:tt (.*)$").unwrap();
    let assign_value = regex::Regex::new(r"^(\w+)\s*:=(.*)$").unwrap();
    let save_value = regex::Regex::new(r"^:s +((\w+)\s*:=(.*))$").unwrap();
    let assign_type = regex::Regex::new(r"^(\w+)\s*t=(.*)$").unwrap();
    let save_type = regex::Regex::new(r"^:s +((\w+)\s*t=(.*))$").unwrap();
    let comment = regex::Regex::new(r"^#").unwrap();

    println!();
    println!("                    \x1b[1;38mUnseemly\x1b[0m");
    println!("    `<expr>` to (typecheck and expand and) evaluate `<expr>`.");
    println!("    `:x <expr>` to (typecheck and) expand `<expr>`.");
    println!("    `:e <expr>` to (expand and) evaluate `<expr>` without typechecking.");
    println!("    `<name> := <expr>` to bind a name for this session.");
    println!("    `:t <expr>` to synthesize the type of <expr>.");
    println!("    `:tt <type>` to canonicalize <type>.");
    println!("    `<name> t= <type>` to bind a type for this session.");
    println!("    `:s <name> := <expr>` to save a binding to the prelude for the future.");
    println!("    `:s <name> t= <expr>` to save a type binding to the prelude.");
    println!("    `:p <expr>` to parse `<expr>` and pretty-print its AST output.");
    println!("    `:pd <expr>` to parse `<expr>` and debug-print its AST output.");
    println!("    Command history is saved over sessions.");
    println!("    Tab-completion works on variables, and lots of Bash-isms work.");
    println!();
    println!("This virtual machine kills cyber-fascists.");

    if let Ok(prelude_file) = File::open(&prelude_filename) {
        let prelude = std::io::BufReader::new(prelude_file);
        for line in prelude.lines() {
            let line = line.unwrap();
            if comment.captures(&line).is_some() {
                // comment
            } else if let Some(caps) = assign_value.captures(&line) {
                if let Err(e) = assign_variable(&caps[1], &caps[2]) {
                    println!("    Error in prelude line: {}\n    {}", line, e);
                }
            } else if let Some(caps) = assign_type.captures(&line) {
                if let Err(e) = assign_t_var(&caps[1], &caps[2]) {
                    println!("    Error in prelude line: {}\n    {}", line, e);
                }
            }
        }
        println!("    [prelude loaded from {}]", prelude_filename.display());
    }

    let _ = rl.load_history(&history_filename);
    while let Ok(line) = rl.readline("\x1b[1;36m≫\x1b[0m ") {
        // TODO: count delimiters, and allow line continuation!
        rl.add_history_entry(line.clone());

        if quit.captures(&line).is_some() {
            break;
        }

        let result_display = if let Some(caps) = just_parse.captures(&line) {
            parse_unseemly_program(&caps[1], true)
        } else if let Some(caps) = just_parse_debug_print.captures(&line) {
            parse_unseemly_program(&caps[1], false)
        } else if let Some(caps) = just_type.captures(&line) {
            type_unseemly_program(&caps[1]).map(|x| format!("{}", x))
        } else if let Some(caps) = just_eval.captures(&line) {
            eval_unseemly_program_without_typechecking(&caps[1]).map(|x| format!("{}", x))
        } else if let Some(caps) = type_and_expand.captures(&line) {
            type_and_expand_unseemly_program(&caps[1]).map(|x| format!("{}", x))
        } else if let Some(caps) = canon_type.captures(&line) {
            canonicalize_type(&caps[1]).map(|x| format!("{}", x))
        } else if let Some(caps) = assign_value.captures(&line) {
            assign_variable(&caps[1], &caps[2]).map(|x| format!("{}", x))
        } else if let Some(caps) = save_value.captures(&line) {
            match assign_variable(&caps[2], &caps[3]) {
                Ok(_) => {
                    use std::io::Write;
                    let mut prel_file = std::fs::OpenOptions::new()
                        .create(true)
                        .append(true)
                        .open(&prelude_filename)
                        .unwrap();
                    writeln!(prel_file, "{}", &caps[1]).unwrap();
                    Ok(format!("[saved to {}]", &prelude_filename.display()))
                }
                Err(e) => Err(e),
            }
        } else if let Some(caps) = assign_type.captures(&line) {
            assign_t_var(&caps[1], &caps[2]).map(|x| format!("{}", x))
        } else if let Some(caps) = save_type.captures(&line) {
            match assign_t_var(&caps[2], &caps[3]) {
                Ok(_) => {
                    use std::io::Write;
                    let mut prel_file = std::fs::OpenOptions::new()
                        .create(true)
                        .append(true)
                        .open(&prelude_filename)
                        .unwrap();
                    writeln!(prel_file, "{}", &caps[1]).unwrap();
                    Ok(format!("[saved to {}]", &prelude_filename.display()))
                }
                Err(e) => Err(e),
            }
        } else {
            eval_unseemly_program(&line).map(|x| format!("{}", x))
        };

        match result_display {
            Ok(v) => println!("\x1b[1;32m≉\x1b[0m {}", v),
            Err(s) => println!("\x1b[1;31m✘\x1b[0m {}", s),
        }
    }
    println!("Bye! Saving history to {}", &history_filename.display());
    rl.save_history(&history_filename).unwrap();
}

fn assign_variable(name: &str, expr: &str) -> Result<Value, String> {
    let res = eval_unseemly_program(expr);

    if let Ok(ref v) = res {
        let ty = type_unseemly_program(expr).unwrap();
        ty_env.with(|tys| {
            val_env.with(|vals| {
                let new_tys = tys.borrow().set(n(name), ty);
                let new_vals = vals.borrow().set(n(name), v.clone());
                *tys.borrow_mut() = new_tys;
                *vals.borrow_mut() = new_vals;
            })
        })
    }
    res
}

fn assign_t_var(name: &str, t: &str) -> Result<Ast, String> {
    let ast = grammar::parse(
        &grammar::FormPat::Call(n("Type")),
        core_forms::outermost__parse_context(),
        t,
    )
    .map_err(|e| e.msg)?;

    let res =
        ty_env.with(|tys| ty::synth_type(&ast, tys.borrow().clone()).map_err(|e| format!("{}", e)));

    if let Ok(ref t) = res {
        ty_env.with(|tys| {
            let new_tys = tys.borrow().set(n(name), t.clone());
            *tys.borrow_mut() = new_tys;
        })
    }

    res
}

fn canonicalize_type(t: &str) -> Result<Ast, String> {
    let ast = grammar::parse(
        &grammar::FormPat::Call(n("Type")),
        core_forms::outermost__parse_context(),
        t,
    )
    .map_err(|e| e.msg)?;

    ty_env.with(|tys| ty::synth_type(&ast, tys.borrow().clone()).map_err(|e| format!("{}", e)))
}

fn parse_unseemly_program(program: &str, pretty: bool) -> Result<String, String> {
    let ast = grammar::parse(
        &core_forms::outermost_form(),
        core_forms::outermost__parse_context(),
        program,
    )
    .map_err(|e| e.msg)?;

    if pretty {
        Ok(format!("{}", ast))
    } else {
        Ok(format!("{:#?}", ast))
    }
}

fn eval_program(
    program: &str,
    pc: crate::earley::ParseContext,
    type_env: Assoc<Name, Ast>,
    type_env__phaseless: Assoc<Name, Ast>,
    value_env: Assoc<Name, Value>,
) -> Result<Value, String> {
    let ast: Ast = grammar::parse(&core_forms::outermost_form(), pc, program).map_err(|e| e.msg)?;

    let _type = ast_walk::walk::<ty::SynthTy>(
        &ast,
        &ast_walk::LazyWalkReses::new(type_env, type_env__phaseless, ast.node_parts(), ast.clone()),
    )
    .map_err(|e| format!("{}", e))?;

    let core_ast = crate::expand::expand(&ast).map_err(|_| "???".to_string())?;

    eval(&core_ast, value_env).map_err(|_| "???".to_string())
}

fn type_unseemly_program(program: &str) -> Result<Ast, String> {
    let ast = grammar::parse(
        &core_forms::outermost_form(),
        core_forms::outermost__parse_context(),
        program,
    )
    .map_err(|e| e.msg)?;

    ty_env.with(|tys| ty::synth_type(&ast, tys.borrow().clone()).map_err(|e| format!("{}", e)))
}

fn eval_unseemly_program_without_typechecking(program: &str) -> Result<Value, String> {
    let ast: Ast = grammar::parse(
        &core_forms::outermost_form(),
        core_forms::outermost__parse_context(),
        program,
    )
    .map_err(|e| e.msg)?;

    let core_ast = crate::expand::expand(&ast).map_err(|_| "error".to_owned())?;

    val_env.with(|vals| eval(&core_ast, vals.borrow().clone()).map_err(|_| "???".to_string()))
}

fn eval_unseemly_program(program: &str) -> Result<Value, String> {
    let ast: Ast = grammar::parse(
        &core_forms::outermost_form(),
        core_forms::outermost__parse_context(),
        program,
    )
    .map_err(|e| e.msg)?;

    let _type = ty_env
        .with(|tys| ty::synth_type(&ast, tys.borrow().clone()).map_err(|e| format!("{}", e)))?;

    let core_ast = crate::expand::expand(&ast).map_err(|_| "error".to_owned())?;

    val_env.with(|vals| eval(&core_ast, vals.borrow().clone()).map_err(|_| "???".to_string()))
}

fn type_and_expand_unseemly_program(program: &str) -> Result<ast::Ast, String> {
    let ast: Ast = grammar::parse(
        &core_forms::outermost_form(),
        core_forms::outermost__parse_context(),
        program,
    )
    .map_err(|e| e.msg)?;

    let _type = ty_env
        .with(|tys| ty::synth_type(&ast, tys.borrow().clone()).map_err(|e| format!("{}", e)))?;

    crate::expand::expand(&ast).map_err(|_| "error".to_owned())
}

#[test]
fn simple_end_to_end_eval() {
    assert_eq!(eval_unseemly_program("(zero? zero)"), Ok(val!(b true)));

    assert_eq!(eval_unseemly_program("(plus one one)"), Ok(val!(i 2)));

    assert_eq!(eval_unseemly_program("(.[x : Int  y : Int . (plus x y)]. one one)"), Ok(val!(i 2)));

    assert_eq!(
        eval_unseemly_program(
            "((fix .[ again : [ -> [ Int -> Int ]] .
            .[ n : Int .
                match (zero? n) {
                    +[True]+ => one
                    +[False]+ => (times n ((again) (minus n one))) } ]. ].) five)"
        ),
        Ok(val!(i 120))
    );
}

#[test]
fn end_to_end_int_list_tools() {
    assert_m!(assign_t_var("IntList", "mu_type IntList . { +[Nil]+ +[Cons Int IntList]+ }"), Ok(_));

    assert_m!(assign_t_var("IntListUF", "{ +[Nil]+ +[Cons Int IntList]+ }"), Ok(_));

    assert_m!(
        assign_variable("mt_ilist", "fold +[Nil]+ : { +[Nil]+ +[Cons Int IntList]+ } : IntList"),
        Ok(_)
    );

    assert_m!(
        assign_variable("ilist_3", "fold +[Cons three mt_ilist]+ : IntListUF : IntList"),
        Ok(_)
    );

    assert_m!(
        assign_variable("ilist_23", "fold +[Cons two ilist_3]+ : IntListUF : IntList"),
        Ok(_)
    );

    assert_m!(
        assign_variable("ilist_123", "fold +[Cons one ilist_23]+ : IntListUF : IntList"),
        Ok(_)
    );

    assert_m!(
        assign_variable(
            "sum_int_list",
            "(fix .[again : [-> [IntList -> Int]] .
             .[ lst : IntList .
                 match unfold lst {
                     +[Nil]+ => zero +[Cons hd tl]+ => (plus hd ((again) tl))} ]. ]. )"
        ),
        Ok(_)
    );

    assert_eq!(eval_unseemly_program("(sum_int_list ilist_123)"), Ok(val!(i 6)));

    assert_m!(
        assign_variable(
            "int_list_len",
            "(fix .[again : [-> [IntList -> Int]] .
             .[ lst : IntList .
                 match unfold lst {
                     +[Nil]+ => zero +[Cons hd tl]+ => (plus one ((again) tl))} ]. ].)"
        ),
        Ok(_)
    );

    assert_eq!(eval_unseemly_program("(int_list_len ilist_123)"), Ok(val!(i 3)));
}

#[test]
fn end_to_end_list_tools() {
    assert_m!(
        assign_t_var("List", "forall T . mu_type List . { +[Nil]+ +[Cons T List<T> ]+ }"),
        Ok(_)
    );

    assert_m!(assign_t_var("ListUF", "forall T . { +[Nil]+ +[Cons T List<T> ]+ }"), Ok(_));

    assert_m!(
        assign_variable(
            "mt_list",
            "fold +[Nil]+ : { +[Nil]+ +[Cons Int List<Int> ]+ } : List < Int > "
        ),
        Ok(_)
    );

    assert_m!(
        assign_variable("list_3", "fold +[Cons three mt_list]+ : ListUF<Int> : List<Int>"),
        Ok(_)
    );

    assert_m!(
        assign_variable("list_23", "fold +[Cons two list_3]+ : ListUF<Int> : List<Int>"),
        Ok(_)
    );

    assert_m!(
        assign_variable("list_123", "fold +[Cons one list_23]+ : ListUF<Int> : List<Int>"),
        Ok(_)
    );

    assert_m!(
        assign_variable(
            "list_len",
            "forall S . (fix .[again : [-> [List<S> -> Int]] .
            .[ lst : List<S> .
                match unfold lst {
                    +[Nil]+ => zero
                    +[Cons hd tl]+ => (plus one ((again) tl))} ]. ].)"
        ),
        Ok(_)
    );

    assert_eq!(eval_unseemly_program("(list_len list_123)"), Ok(val!(i 3)));

    assert_m!(
        assign_variable(
            "map",
            "forall T S . (fix  .[again : [-> [List<T>  [T -> S] -> List<S> ]] .
            .[ lst : List<T>   f : [T -> S] .
                match unfold lst {
                    +[Nil]+ => fold +[Nil]+ : ListUF<S> : List<S>
                    +[Cons hd tl]+ =>
                      fold +[Cons (f hd) ((again) tl f)]+ : ListUF<S> : List<S> } ]. ].)"
        ),
        Ok(_)
    );
    // TODO: what should even happen if you have `forall` not on the "outside"?
    // It should probably be an error to have a value typed with an underdetermined type.

    // TODO: it's way too much of a pain to define each different expected result list.
    assert_m!(eval_unseemly_program("(map list_123 .[x : Int . (plus x one)]. )"), Ok(_));

    assert_m!(eval_unseemly_program("(map list_123 .[x : Int . (equal? x two)]. )"), Ok(_));
}

#[test]
fn end_to_end_quotation_basic() {
    assert_m!(eval_unseemly_program("'[Expr | .[ x : Int . x ]. ]'"), Ok(_));

    assert_m!(eval_unseemly_program("'[Expr | (plus five five) ]'"), Ok(_));

    assert_m!(eval_unseemly_program("'[Expr | '[Expr | (plus five five) ]' ]'"), Ok(_));

    //≫ .[s : Expr<Int> . '[Expr | ( ,[Expr | s], '[Expr | ,[Expr | s], ]')]' ].
}
#[test]
fn subtyping_direction() {
    // Let's check to make sure that "supertype" and "subtype" never got mixed up:

    assert_m!(assign_variable("ident", "forall T . .[ a : T . a ]."), Ok(_));

    assert_eq!(eval_unseemly_program("(ident five)"), Ok(val!(i 5)));

    assert_m!(eval_unseemly_program("( .[ a : [Int -> Int] . a]. ident)"), Ok(_));

    assert_m!(eval_unseemly_program("( .[ a : forall T . [T -> T] . a]. .[a : Int . a].)"), Err(_));

    assert_m!(eval_unseemly_program(".[ a : *[]* . a]."), Ok(_));

    assert_m!(
        eval_unseemly_program("( .[ a : *[normal : Int extra : Int]* . a]. *[normal : one]*)"),
        Err(_)
    );

    assert_m!(
        eval_unseemly_program("( .[ a : *[normal : Int]* . a]. *[normal : one extra : five]*)"),
        Ok(_)
    );
}

#[test]
fn end_to_end_quotation_advanced() {
    assert_eq!(
        eval_unseemly_program(
            "(.[five_e : Expr < Int >.
                '[Expr | (plus five ,[five_e],) ]' ].
                '[Expr | five]')"
        ),
        eval_unseemly_program("'[Expr | (plus five five) ]'")
    );

    // Pass the wrong type (not really a test of quotation)
    assert_m!(
        type_unseemly_program(
            "(.[five_e : Expr<Int> .
                '[Expr | (plus five ,[five_e],) ]' ].
                '[Expr | true]')"
        ),
        Err(_)
    );

    // Interpolate the wrong type
    assert_m!(
        type_unseemly_program(
            "(.[five_e : Expr<Bool> .
                '[Expr | (plus five ,[five_e],) ]' ].
                '[Expr | true]')"
        ),
        Err(_)
    );

    // Interpolate the wrong type (no application needed to find the error)
    assert_m!(
        type_unseemly_program(".[five_e : Expr<Bool> . '[Expr | (plus five ,[five_e],) ]' ]."),
        Err(_)
    );

    assert_m!(
        eval_unseemly_program(
            "forall T . .[type : Type<T>   rhs : Expr<T>
                . '[Expr | (.[x : ,[Type<T> | type], . eight].  ,[rhs], )]' ]."
        ),
        Ok(_)
    );

    assert_m!(eval_unseemly_program("'[Pat<Nat> | x]'"), Ok(_));

    // Actually import a pattern of quoted syntax:
    assert_eq!(
        eval_unseemly_program(
            "match '[Expr | (plus one two) ]' {
                 '[Expr<Int> | (plus ,[Expr<Int> | e], two) ]' => e }"
        ),
        Ok(val!(ast (vr "one")))
    );

    // In order to have "traditional", non-type-annotated `let`, we want to ... reify T, I guess?
    // But the whole language has parametricity kinda baked in, and that seems to make it hard?
    // I think the solution is to build `let` into the language;
    //  if a macro wants to have non-annotated binding, it's probably expandable to `let` anyways.
    assert_m!(
        assign_variable(
            "let",
            "forall T S . .[binder : Pat<T>
                        type : Type<T>
                        rhs : Expr<T>
                        body : Expr<S> .
             '[ Expr | (.[x : ,[type],
                     . match x { ,[Pat<T> | binder], => ,[body], } ].
                 ,[rhs],)]' ]."
        ),
        Ok(_)
    );

    without_freshening! {
        assert_eq!(
            eval_unseemly_program(
                "(let  '[Pat<Int> | y]'
                       '[Type<Int> | Int]'
                       '[Expr<Int> | eight]'
                       '[Expr<Int> | five]')"),
            eval_unseemly_program("'[Expr<Int> | (.[x : Int . match x {y => five}].  eight)]'"));
    }

    //  // We need tuple literals before we can test this:
    //  assert_m!(assign_variable("let-multi",
    //      "forall T . .[ binder : **[ :::[T >> Ident<T> ]::: ]**
    //                     type : **[ :::[T >> Type<T> ]::: ]**
    //                     rhs : **[ :::[T >> Expr<T> ]::: ]**
    //                     body : Expr<S> .
    //          '[Expr | (.[ ...[, binder , >> ,[Ident | binder],]...
    //                       : ...[, type , >> ,[Type | type], ]... .
    //                    ,[body], ].
    //                      ...[, Expr , | ,[rhs], ]... ) ]'
    //                       "),
    //       Ok(_));

    //  without_freshening! {
    //      assert_eq!(
    //          eval_unseemly_program(
    //              "(let-multi  '[Ident<Int> | y]'
    //                     '[Type<Int> | Int]'
    //                     '[Expr<Int> | eight]'
    //                     '[Expr<Int> | five]')"),
    //          eval_unseemly_program("'[Expr<Int> | (.[x : Int . match x {y => five}].  eight)]'"));
    //  }
}

#[test]
fn language_building() {
    assert_eq!(
        eval_unseemly_program(
            r"extend_syntax
                DefaultSeparator ::= /((?:\s|#[^\n]*)*)/ ;
            in
                # Now we have comments! (just not after the last token)
            five"
        ),
        Ok(val!(i 5))
    );

    let bound_wrong_prog = "extend_syntax
            Expr ::=also forall T S . '{
                [
                    lit ,{ DefaultToken }, = 'let'
                    [
                        pat := ( ,{ Pat<S> }, )
                        lit ,{ DefaultToken }, = '='
                        value := ( ,{ Expr<S> }, )
                        lit ,{ DefaultToken }, = ';'
                    ] *
                    lit ,{ DefaultToken }, = 'in'
                    body := ( ,{ Expr<T> }, <-- ...[pat = value]... )
                ]
            }' let_macro -> .{
                '[Expr |
                    match ...[,value, >> ,[value], ]...
                        { ...[,pat, >> ,[pat],]... => ,[body], } ]'
            }. ;
        in
        let x = eight ;
            y = times ;
        in (plus x y)";
    let bound_wrong_ast = grammar::parse(
        &core_forms::outermost_form(),
        core_forms::outermost__parse_context(),
        bound_wrong_prog,
    )
    .unwrap();

    assert_m!(
        ty::synth_type(&bound_wrong_ast, crate::runtime::core_values::core_types()),
        ty_err_p!(Mismatch(x, y)) => {
            assert_eq!(x, uty!({Int :}));
            assert_eq!(y, uty!({fn : [{Int :}; {Int :}] {Int :}}));
        }
    );

    let inner_expr_wrong_prog = "extend_syntax
            Expr ::=also forall T S . '{
                [
                    lit ,{ DefaultToken }, = 'let'
                    [
                        pat := ( ,{ Pat<S> }, )
                        lit ,{ DefaultToken }, = '='
                        value := ( ,{ Expr<S> }, )
                        lit ,{ DefaultToken }, = ';'
                    ] *
                    lit ,{ DefaultToken }, = 'in'
                    body := ( ,{ Expr< T > }, <-- ...[pat = value]... )
                ]
            }' let_macro -> .{
                '[Expr |
                    match ...[,value, >> ,[value], ]...
                        { ...[,pat, >> ,[pat],]... => ,[body], } ]'
            }. ;
        in
        let x = eight ;
            y = four ;
        in (plus x times)";
    let inner_expr_wrong_ast = grammar::parse(
        &core_forms::outermost_form(),
        core_forms::outermost__parse_context(),
        inner_expr_wrong_prog,
    )
    .unwrap();

    assert_m!(
        ty::synth_type(&inner_expr_wrong_ast, crate::runtime::core_values::core_types()),
        ty_err_p!(Mismatch(x, times)) => {
            assert_eq!(x, uty!({Int :}));
            assert_eq!(times, uty!({fn : [{Int :}; {Int :}] {Int :}}));
        }
    );

    // TODO: leaving out the `**[ ]**` results in an ICP; it should be a static error.

    let let_macro_prog = "extend_syntax
            Expr ::=also forall T S . '{
                [
                    lit ,{ DefaultToken }, = 'let'
                    [
                        pat := ( ,{ Pat<S> }, )
                        lit ,{ DefaultToken }, = '='
                        value := ( ,{ Expr<S> }, )
                        lit ,{ DefaultToken }, = ';'
                    ] *
                    lit ,{ DefaultToken }, = 'in'
                    body := ( ,{ Expr<T> }, <-- ...[pat = value]... )
                ]
            }' let_macro -> .{
                '[Expr |
                    match **[...[,value, >> ,[value], ]... ]**
                        { **[...[,pat, >> ,[pat],]... ]** => ,[body], } ]'
            }. ;
        in
        let x = eight ;
            y = four ;
        in (plus y (plus x y))";
    assert_eq!(eval_unseemly_program(let_macro_prog), Ok(val!(i 16)));
}
