#[cfg(feature = "tokio")]
extern crate runtime_tokio as tokio;
#[cfg(feature = "async-std")]
extern crate runtime_async_std as async_std;

use std::path::Path;
use std::path::PathBuf;

use futures::stream::StreamExt;
use clap::Parser;

use unrar_async::Archive;

#[derive(Debug, Parser)]
struct Options {
	/// List of rar files to process
	#[clap(required = true, min_values = 1)]
	rar_file: Vec<PathBuf>,

	/// Target directory
	#[clap(long, default_value = ".")]
	target: PathBuf
}

/// This is the "real" entrypoint; the main() functions below are just thin wrappers that the different executors can utilize
async fn run() {
	let options = Options::parse();

	for file in options.rar_file.iter() {
		println!("--- {}", file.display());
		process(file, &options.target).await.unwrap();
		println!();
	}
}

async fn process(rar_file: &Path, target: &Path) -> anyhow::Result<()> {
	match Archive::new(rar_file).unwrap().extract_to(target).await {
		// Everything okay, just list the archive
		Ok(mut archive) => {
			while let Some(entry) = archive.next().await {
				match entry {
					Ok(entry) => println!("{}", entry),
					Err(e) => println!("Error: {}", e)
				};
			}
		},
		// Irrecoverable failure, do nothing.
		Err(e) => eprintln!("Error: {}", e)
	}

	Ok(())
}

#[cfg(feature = "tokio")]
#[tokio::main]
async fn main() {
	run().await
}

#[cfg(feature = "async-std")]
#[async_std::main]
async fn main() {
	run().await
}

