use std::fs::File;
use std::io::prelude::*;

use futures::stream::StreamExt;

use tempdir::TempDir;

use unrar_async::Archive;
use unrar_async::Error;
use unrar_async::error::RarError;

#[cfg(feature = "tokio")]
#[tokio::test]
async fn list_tokio() {
	// No password needed in order to list contents
	let mut entries = Archive::new("data/crypted.rar".into()).unwrap().list().await.unwrap();
	assert_eq!(entries.next().await.unwrap().unwrap().filename, ".gitignore");
}

#[cfg(feature = "async-std")]
#[async_std::test]
async fn list_async_std() {
	// No password needed in order to list contents
	let mut entries = Archive::new("data/crypted.rar".into()).unwrap().list().await.unwrap();
	assert_eq!(entries.next().await.unwrap().unwrap().filename, ".gitignore");
}

#[cfg(feature = "tokio")]
#[tokio::test]
async fn no_password_tokio() {
	let t = TempDir::new("unrar").unwrap();
	let mut arc = Archive::new("data/crypted.rar".into())
		.unwrap()
		.extract_to(t.path())
		.await
		.unwrap();
	let err = arc.next().await.unwrap().unwrap_err();
	assert!(matches!(err, Error::Rar(RarError::MissingPassword)));
}

#[cfg(feature = "async-std")]
#[async_std::test]
async fn no_password_async_std() {
	let t = TempDir::new("unrar").unwrap();
	let mut arc = Archive::new("data/crypted.rar".into())
		.unwrap()
		.extract_to(t.path())
		.await
		.unwrap();
	let err = arc.next().await.unwrap().unwrap_err();
	assert!(matches!(err, Error::Rar(RarError::MissingPassword)));
}

#[cfg(feature = "tokio")]
#[tokio::test]
async fn version_cat_tokio() {
	let t = TempDir::new("unrar").unwrap();
	Archive::with_password("data/crypted.rar".into(), "unrar".into())
		.unwrap()
		.extract_to(t.path())
		.await
		.unwrap()
		.process()
		.await
		.unwrap();
	let mut file = File::open(t.path().join(".gitignore")).unwrap();
	let mut s = String::new();
	file.read_to_string(&mut s).unwrap();
	assert_eq!(s, "target\nCargo.lock\n");
}

#[cfg(feature = "async-std")]
#[async_std::test]
async fn version_cat_async_std() {
	let t = TempDir::new("unrar").unwrap();
	Archive::with_password("data/crypted.rar".into(), "unrar".into())
		.unwrap()
		.extract_to(t.path())
		.await
		.unwrap()
		.process()
		.await
		.unwrap();
	let mut file = File::open(t.path().join(".gitignore")).unwrap();
	let mut s = String::new();
	file.read_to_string(&mut s).unwrap();
	assert_eq!(s, "target\nCargo.lock\n");
}

