use std::ffi::CStr;
use std::ffi::CString;
use std::fmt;
use std::ptr;
use std::os::raw::c_char;

pub(crate) struct Handle(Box<unrar_sys::HANDLE>);
impl Handle {
	pub fn from_ffi(handle: *mut unrar_sys::HANDLE) -> Self {
		unsafe { Self(Box::from_raw(handle)) }
	}

	pub fn as_ffi(&self) -> unrar_sys::Handle {
		&*self.0
	}
}
impl fmt::Debug for Handle {
	fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
		fmt::Pointer::fmt(&self.0, f)
	}
}

#[derive(Debug)]
pub(crate) struct OpenArchiveData {
	pub archive_name: CString,
	pub open_mode: u32,
	pub open_result: u32,
	pub comment_buffer: Option<CString>,
	pub comment_buffer_size: u32,
	pub comment_state: u32
}

impl OpenArchiveData {
	pub fn new(archive_name: CString, open_mode: u32) -> Self {
		Self{
			archive_name,
			open_mode,
			open_result: 0,
			comment_buffer: None,
			comment_buffer_size: 0,
			comment_state: 0
		}
	}

	pub fn as_ffi(&self) -> unrar_sys::OpenArchiveData {
		unrar_sys::OpenArchiveData{
			archive_name: self.archive_name.as_ptr(),
			open_mode: self.open_mode,
			open_result: self.open_result,
			comment_buffer: self.comment_buffer.as_ref().map(|s| s.as_ptr() as *mut _).unwrap_or(ptr::null_mut()),
			comment_buffer_size: self.comment_buffer_size,
			comment_size: self.comment_buffer.as_ref().map(|s| s.to_bytes().len() as u32).unwrap_or(0),
			comment_state: self.comment_state
		}
	}
}

/*
impl<'a> Drop for OpenArchiveData<'a> {
	fn drop(&mut self) {
		self.comment_buffer.into_raw();
	}
}
*/

#[derive(Debug, Default)]
pub(crate) struct HeaderData {
	pub archive_name: CString,
	pub filename: CString,
	pub flags: u32,
	pub pack_size: u32,
	pub unp_size: u32,
	pub host_os: u32,
	pub file_crc: u32,
	pub file_time: u32,
	pub unp_ver: u32,
	pub method: u32,
	pub file_attr: u32,
	pub comment_buffer: Option<CString>,
	pub comment_buffer_size: u32,
	pub comment_state: u32
}

impl TryFrom<unrar_sys::HeaderData> for HeaderData {
	type Error = std::ffi::FromVecWithNulError;
	fn try_from(other: unrar_sys::HeaderData) -> Result<Self, Self::Error> {
		let archive_name_len = unsafe { libc::strlen(&other.archive_name as *const _) };
		let filename_len = unsafe { libc::strlen(&other.filename as *const _) };
		Ok(Self{
			archive_name: CString::from_vec_with_nul(other.archive_name[..archive_name_len + 1].iter().map(|&c| unsafe { std::mem::transmute(c) }).collect::<Vec<u8>>())?,
			filename: CString::from_vec_with_nul(other.filename[..filename_len + 1].iter().map(|&c| unsafe { std::mem::transmute(c) }).collect::<Vec<u8>>())?,
			flags: other.flags,
			pack_size: other.pack_size,
			unp_size: other.unp_size,
			host_os: other.host_os,
			file_crc: other.file_crc,
			file_time: other.file_time,
			unp_ver: other.unp_ver,
			method: other.method,
			file_attr: other.file_attr,
			comment_buffer: match other.comment_buffer.is_null() {
				true => None,
				false => Some(unsafe { CStr::from_ptr(other.comment_buffer).into() })
			},
			comment_buffer_size: other.comment_buffer_size,
			comment_state: other.comment_state
		})
	}
}

impl TryFrom<HeaderData> for unrar_sys::HeaderData {
	type Error = std::array::TryFromSliceError;
	fn try_from(this: HeaderData) -> Result<Self, Self::Error> {
		let mut archive_name = this.archive_name.as_bytes_with_nul().iter().map(|&c| unsafe { std::mem::transmute(c) }).collect::<Vec<c_char>>();
		archive_name.resize_with(260, Default::default);
		let mut filename = this.filename.as_bytes_with_nul().iter().map(|&c| unsafe { std::mem::transmute(c) }).collect::<Vec<c_char>>();
		filename.resize_with(260, Default::default);
		Ok(Self{
			archive_name: archive_name[..].try_into()?,
			filename: filename[..].try_into()?,
			flags: this.flags,
			pack_size: this.pack_size,
			unp_size: this.unp_size,
			host_os: this.host_os,
			file_crc: this.file_crc,
			file_time: this.file_time,
			unp_ver: this.unp_ver,
			method: this.method,
			file_attr: this.file_attr,
			comment_buffer: this.comment_buffer.as_ref().map(|s| s.as_ptr() as *mut _).unwrap_or(ptr::null_mut()),
			comment_buffer_size: this.comment_buffer_size,
			comment_size: this.comment_buffer.as_ref().map(|s| s.to_bytes().len() as u32).unwrap_or(0),
			comment_state: this.comment_state
		})
	}
}

/// A single entry contained in an archive - usually, a file or directory
#[derive(Debug)]
pub struct Entry {
	pub filename: String,
	pub flags: Flags,
	pub unpacked_size: u32,
	pub file_crc: u32,
	pub file_time: u32,
	pub method: u32,
	pub file_attr: u32,
	pub next_volume: Option<String>
}

impl Entry {
	/// Whether or not an entry is split across multiple volumes in a multi-part archive
	pub fn is_split(&self) -> bool {
		self.flags.contains(Flags::SPLIT_BEFORE) || self.flags.contains(Flags::SPLIT_AFTER)
	}

	pub fn is_directory(&self) -> bool {
		self.flags.contains(Flags::DIRECTORY)
	}

	pub fn is_encrypted(&self) -> bool {
		self.flags.contains(Flags::ENCRYPTED)
	}

	pub fn is_file(&self) -> bool {
		!self.is_directory()
	}
}

impl fmt::Display for Entry {
	fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
		write!(f, "{}", self.filename)?;
		if(self.is_directory()) {
			write!(f, "/")?;
		}
		if(self.is_split()) {
			write!(f, " (partial)")?;
		}
		Ok(())
	}
}

impl TryFrom<HeaderData> for Entry {
	type Error = std::str::Utf8Error;
	fn try_from(header: HeaderData) -> Result<Self, Self::Error> {
		Ok(Self{
			filename: std::str::from_utf8(header.filename.as_bytes())?.into(),
			flags: Flags::from_bits(header.flags).unwrap(),
			unpacked_size: header.unp_size,
			file_crc: header.file_crc,
			file_time: header.file_time,
			method: header.method,
			file_attr: header.file_attr,
			next_volume: None
		})
	}
}

bitflags::bitflags! {
	pub struct Flags: u32 {
		const SPLIT_BEFORE = 0x01;
		const SPLIT_AFTER = 0x02;
		const ENCRYPTED = 0x04;
		//const RESERVED = 0x08;
		const SOLID = 0x10;
		const DIRECTORY = 0x20;
	}
}

