#![allow(unused_parens)]

#[cfg(not(any(feature = "tokio", feature = "async-std")))]
compile_error!("Either feature \"tokio\" or \"async-std\" must be enabled for unrar-async.");

use std::borrow::Cow;
use std::ffi::CString;
use std::path::Path;
use std::path::PathBuf;

use widestring::WideCString;

pub mod error;
pub use error::Error;
use error::NulError;

mod consts;
pub use consts::RE_MULTIPART_EXTENSION;
pub use consts::RE_EXTENSION;
mod flags;
pub use flags::ArchiveFlags;
pub use flags::OpenMode;
pub use flags::Operation;
pub use flags::VolumeInfo;
mod open_archive;
pub use open_archive::Entry;
pub use open_archive::OpenArchive;

pub struct Archive<'a> {
	filename: Cow<'a, Path>,
	password: Option<CString>,
	comments: Option<&'a mut [u8]>
}

impl<'a> Archive<'a> {
	/// Creates an `Archive` object to operate on a plain RAR archive
	pub fn new<F>(file: &'a F) -> Result<Self, Error>
	where
		F: AsRef<Path> + ?Sized
	{
		WideCString::from_os_str(file.as_ref()).map_err(|e| NulError::from(e))?;
		Ok(Self{
			filename: Cow::Borrowed(file.as_ref()),
			password: None,
			comments: None
		})
	}

	/// Creates an `Archive` object to operate on a password-encrypted RAR archive.
	pub fn with_password<F, P>(file: &'a F, password: &'a P) -> Result<Self, Error>
	where
		F: AsRef<Path> + ?Sized,
		P: AsRef<str> + ?Sized
	{
		let mut this = Self::new(file)?;
		this.password = Some(CString::new(password.as_ref()).map_err(|e| NulError::from(e))?);
		Ok(this)
	}

	/// Set the comment buffer of the underlying archive.
	/// Note:  Comments are not yet implemented; this function will have no effect at this time.
	pub fn set_comments(&mut self, comments: &'a mut [u8]) {
		self.comments = Some(comments)
	}

	/// Returns `true` if the filename matches a RAR archive; `false` otherwise.
	///
	/// This method does not perform any filesystem operations; it operates purely on the string filename.
	#[inline]
	pub fn is_archive(&self) -> bool {
		is_archive(&self.filename)
	}

	/// Returns `true` if the filename matches a part of a multipart collection; `false` otherwise.
	///
	/// This method does not perform any filesystem operations; it operates purely on the string filename.
	#[inline]
	pub fn is_multipart(&self) -> bool {
		is_multipart(&self.filename)
	}

	/// Returns a glob string covering all parts of the multipart collection, or `None` if the archive is single-part.
	///
	/// This method does not make any filesystem operations; it operates purely on the string filename.
	pub fn try_all_parts(&self) -> Option<PathBuf> {
		get_rar_extension(&self.filename).and_then(|full_ext| {
			RE_MULTIPART_EXTENSION.captures(&full_ext).map(|captures| {
				let mut replacement = String::from(captures.get(1).unwrap().as_str());
				replacement.push_str(&"?".repeat(captures.get(2).unwrap().as_str().len()));
				replacement.push_str(captures.get(3).unwrap().as_str());
				full_ext.replace(captures.get(0).unwrap().as_str(), &replacement)
			})
		}).and_then(|new_ext| {
			self.filename.file_stem().map(|s| Path::new(s).with_extension(&new_ext[1..]))
		})
	}

	/// Returns a glob string covering all parts of the multipart collection, or a copy of the archive's entire filename if it's a single-part archive.
	///
	/// This method does not make any filesystem operations; it operates purely on the string filename.
	pub fn all_parts(&self) -> PathBuf {
		self.try_all_parts().unwrap_or_else(|| self.filename.to_path_buf())
	}

	/// Returns the nth part of this multi-part collection, or `None` if the archive is single-part.
	///
	/// This method does not make any filesystem operations; it operates purely on the string filename.
	pub fn nth_part(&self, n: usize) -> Option<PathBuf> {
		get_rar_extension(&self.filename).and_then(|full_ext| {
			RE_MULTIPART_EXTENSION.captures(&full_ext).map(|captures| {
				let mut replacement = String::from(captures.get(1).unwrap().as_str());
				// `n` left-padded with zeroes to the length of the archive's numbers (e.g. 3 for part001.rar)
				replacement.push_str(&format!("{:01$}", n, captures.get(2).unwrap().as_str().len()));
				replacement.push_str(captures.get(3).unwrap().as_str());
				full_ext.replace(captures.get(0).unwrap().as_str(), &replacement)
			})
		}).and_then(|new_ext| {
			self.filename.file_stem().map(|s| Path::new(s).with_extension(&new_ext[1..]))
		})
	}

	/// Returns the first part of the multi-part collection, or a copy of the underlying archive's filename if the archive is single-part.
	///
	/// This method does not make any filesystem operations; it operates purely on the string filename.
	pub fn first_part(&self) -> PathBuf {
		self.nth_part(1).unwrap_or_else(|| self.filename.to_path_buf())
	}

	/// Changes the filename to point to the first part of the multipart collection.  Does nothing if the archive is single-part.
	///
	/// This method does not make any filesystem operations; it operates purely on the string filename.
	pub fn as_first_part(&mut self) {
		if let Some(first_part) = self.nth_part(1) {
			self.filename = Cow::Owned(first_part);
		}
	}

	/// Opens the underlying archive for listing its contents
	pub async fn list(self) -> Result<OpenArchive, Error> {
		self.open(OpenMode::List, None, Operation::Skip).await
	}

	/// Opens the underlying archive for listing its contents without omitting or pooling split entries
	pub async fn list_split(self) -> Result<OpenArchive, Error> {
		self.open(OpenMode::ListSplit, None, Operation::Skip).await
	}

	/// Opens the underlying archive for extracting to the given directory
	pub async fn extract_to(self, path: &Path) -> Result<OpenArchive, Error> {
		self.open(OpenMode::Extract, Some(path), Operation::Extract).await
	}

	/// Opens the underlying archive for testing
	pub async fn test(self) -> Result<OpenArchive, Error> {
		self.open(OpenMode::Extract, None, Operation::Test).await
	}

	/// Opens the underlying archive
	pub async fn open(self, mode: OpenMode, path: Option<&Path>, operation: Operation) -> Result<OpenArchive, Error> {
		OpenArchive::open(&self.filename, mode, self.password, path, operation).await
	}
}

fn get_rar_extension(path: &Path) -> Option<String> {
	path.extension().map(|ext| {
		match path.file_stem().and_then(|s| Path::new(s).extension()) {
			Some(pre_ext) => format!(".{}.{}", pre_ext.to_string_lossy(), ext.to_string_lossy()),
			None => format!(".{}", ext.to_string_lossy())
		}
	})
}

fn is_archive(path: &Path) -> bool {
	get_rar_extension(path).and_then(|full_ext| {
		RE_EXTENSION.find(&full_ext).map(|_| ())
	}).is_some()
}

fn is_multipart(path: &Path) -> bool {
	get_rar_extension(path).and_then(|full_ext| {
		RE_MULTIPART_EXTENSION.find(&full_ext).map(|_| ())
	}).is_some()
}

#[cfg(test)]
mod tests {
	use std::path::PathBuf;
	use super::Archive;

	#[test]
	fn glob() {
		assert_eq!(Archive::new("arc.part0010.rar".into()).unwrap().all_parts(), PathBuf::from("arc.part????.rar"));
		assert_eq!(Archive::new("archive.r100".into()).unwrap().all_parts(), PathBuf::from("archive.r???"));
		assert_eq!(Archive::new("archive.r9".into()).unwrap().all_parts(), PathBuf::from("archive.r?"));
		assert_eq!(Archive::new("archive.999".into()).unwrap().all_parts(), PathBuf::from("archive.???"));
		assert_eq!(Archive::new("archive.rar".into()).unwrap().all_parts(), PathBuf::from("archive.rar"));
		assert_eq!(Archive::new("random_string".into()).unwrap().all_parts(), PathBuf::from("random_string"));
		assert_eq!(Archive::new("v8/v8.rar".into()).unwrap().all_parts(), PathBuf::from("v8/v8.rar"));
		assert_eq!(Archive::new("v8/v8".into()).unwrap().all_parts(), PathBuf::from("v8/v8"));
	}

	#[test]
	fn first_part() {
		assert_eq!(Archive::new("arc.part0010.rar".into()).unwrap().first_part(), PathBuf::from("arc.part0001.rar"));
		assert_eq!(Archive::new("archive.r100".into()).unwrap().first_part(), PathBuf::from("archive.r001"));
		assert_eq!(Archive::new("archive.r9".into()).unwrap().first_part(), PathBuf::from("archive.r1"));
		assert_eq!(Archive::new("archive.999".into()).unwrap().first_part(), PathBuf::from("archive.001"));
		assert_eq!(Archive::new("archive.rar".into()).unwrap().first_part(), PathBuf::from("archive.rar"));
		assert_eq!(Archive::new("random_string".into()).unwrap().first_part(), PathBuf::from("random_string"));
		assert_eq!(Archive::new("v8/v8.rar".into()).unwrap().first_part(), PathBuf::from("v8/v8.rar"));
		assert_eq!(Archive::new("v8/v8".into()).unwrap().first_part(), PathBuf::from("v8/v8"));
	}

	#[test]
	fn is_archive() {
		assert_eq!(super::is_archive(&PathBuf::from("archive.rar")), true);
		assert_eq!(super::is_archive(&PathBuf::from("archive.part1.rar")), true);
		assert_eq!(super::is_archive(&PathBuf::from("archive.part100.rar")), true);
		assert_eq!(super::is_archive(&PathBuf::from("archive.r10")), true);
		assert_eq!(super::is_archive(&PathBuf::from("archive.part1rar")), false);
		assert_eq!(super::is_archive(&PathBuf::from("archive.rar\n")), false);
		assert_eq!(super::is_archive(&PathBuf::from("archive.zip")), false);
	}
}

