use enum_primitive::enum_from_primitive;
use enum_primitive::enum_from_primitive_impl;
use enum_primitive::enum_from_primitive_impl_ty;
use enum_primitive::FromPrimitive;

/// The main error type for `unrar-async`
#[derive(Clone, Debug, thiserror::Error)]
pub enum Error {
	#[error("rar error: {0}")]
	Rar(#[from] RarError),
	#[error("string nul error: {0}")]
	NulString(#[from] NulError),
	#[error("error converting to an FFI string: {0}")]
	FFIStringConversionBytes(#[from] std::ffi::FromBytesWithNulError),
	#[error("error converting to an FFI string: {0}")]
	FFIStringConversionVec(#[from] std::ffi::FromVecWithNulError),
	#[cfg(feature = "tokio")]
	#[error("tokio error")]
	Join,
	#[error("nul handle returned by core")]
	NulHandle,
	#[error("input string too long for buffer ({0}/{1})")]
	BufferOverflow(usize, usize),
	#[error("failed to pack into an array for FFI: {0}")]
	TryFromSlice(#[from] std::array::TryFromSliceError),
	#[error("string from FFI was not valid UTF-8: {0}")]
	UTF8(#[from] std::str::Utf8Error)
}

#[cfg(feature = "tokio")]
impl From<tokio::task::JoinError> for Error {
	fn from(_e: tokio::task::JoinError) -> Self {
		Self::Join
	}
}

/// The various errors that can occur while processing a `.rar` archive
#[derive(Clone, Debug, thiserror::Error)]
pub enum RarError {
	#[error("Archive header damaged")]
	ArchiveHeader,
	#[error("File header damaged")]
	FileHeader,
	#[error("File checksum mismatch")]
	FileChecksum,
	#[error("Unknown encryption")]
	UnknownEncryption,
	#[error("Could not open next volume")]
	OpenVolume,
	#[error("Unknown archive format")]
	UnknownFormat,
	#[error("could not open archive")]
	OpenArchive,
	#[error("Not enough memory")]
	OOM,
	#[error("Not a RAR archive")]
	BadArchive,
	#[error("Could not create file")]
	CreateFile,
	#[error("Could not close file")]
	CloseFile,
	#[error("Read error")]
	Read,
	#[error("Write error")]
	Write,
	#[error("Archive comment was truncated to fit buffer")]
	CommentTruncated,
	#[error("Password for encrypted archive not specified")]
	MissingPassword,
	#[error("Could not open file source for reference record")]
	Reference,
	#[error("Wrong password was specified")]
	BadPassword,
	#[error("Archive end")]
	EndArchive,
	#[error("Unknown error")]
	Unknown,
	#[error("Invalid return code: {0}")]
	InvalidCode(u32)
}

impl From<(Code, When)> for RarError {
	fn from(input: (Code, When)) -> Self {
		use self::Code::*;
		use self::When::*;
		match (input.0, input.1) {
			(BadData, Open) => Self::ArchiveHeader,
			(BadData, Read) => Self::FileHeader,
			(BadData, Process) => Self::FileChecksum,
			(UnknownFormat, Open) => Self::UnknownEncryption,
			(OpenError, Process) => Self::OpenVolume,
			(UnknownFormat, _) => Self::UnknownFormat,
			(OpenError, _) => Self::OpenArchive,
			(NoMemory, _) => Self::OOM,
			(BadArchive, _) => Self::BadArchive,
			(CreateError, _) => Self::CreateFile,
			(CloseError, _) => Self::CloseFile,
			(ReadError, _) => Self::Read,
			(WriteError, _) => Self::Write,
			(SmallBuffer, _) => Self::CommentTruncated,
			(Unknown, _) => Self::Unknown,
			(MissingPassword, _) => Self::MissingPassword,
			(ReferenceError, _) => Self::Reference,
			(BadPassword, _) => Self::BadPassword,
			(EndArchive, _) => Self::EndArchive,
			(Success, _) => unreachable!()
		}
	}
}

enum_from_primitive! {
	/// Return codes from the C++ unrar function calls
	#[derive(Clone, Copy, Debug, Eq, PartialEq)]
	#[repr(u8)]
	pub enum Code {
		Success = unrar_sys::ERAR_SUCCESS as u8,
		EndArchive = unrar_sys::ERAR_END_ARCHIVE as u8,
		NoMemory = unrar_sys::ERAR_NO_MEMORY as u8,
		BadData = unrar_sys::ERAR_BAD_DATA as u8,
		BadArchive = unrar_sys::ERAR_BAD_ARCHIVE as u8,
		UnknownFormat = unrar_sys::ERAR_UNKNOWN_FORMAT as u8,
		OpenError = unrar_sys::ERAR_EOPEN as u8,
		CreateError = unrar_sys::ERAR_ECREATE as u8,
		CloseError = unrar_sys::ERAR_ECLOSE as u8,
		ReadError = unrar_sys::ERAR_EREAD as u8,
		WriteError = unrar_sys::ERAR_EWRITE as u8,
		SmallBuffer = unrar_sys::ERAR_SMALL_BUF as u8,
		Unknown = unrar_sys::ERAR_UNKNOWN as u8,
		MissingPassword = unrar_sys::ERAR_MISSING_PASSWORD as u8,
		ReferenceError = unrar_sys::ERAR_EREFERENCE as u8,
		BadPassword = unrar_sys::ERAR_BAD_PASSWORD as u8
	}
}

impl TryFrom<u32> for Code {
	type Error = RarError;
	fn try_from(code: u32) -> Result<Self, Self::Error> {
		match code.try_into() {
			Ok(c) => Self::from_u8(c).ok_or(RarError::InvalidCode(code)),
			Err(_) => Err(RarError::InvalidCode(code))
		}
	}
}

/// Errors that can occur around nul bytes while processing strings
#[derive(Clone, Debug, thiserror::Error)]
pub enum NulError {
	#[error("nul value found at position {0}")]
	ContainsNul(usize),
	#[error("")]
	MissingNulTerminator
}

impl<C> From<widestring::error::ContainsNul<C>> for NulError {
	fn from(e: widestring::error::ContainsNul<C>) -> Self {
		Self::ContainsNul(e.nul_position())
	}
}

/*
impl<C> From<widestring::error::NulError<C>> for NulError {
	fn from(e: widestring::error::NulError<C>) -> Self {
		match e {
			widestring::error::NulError::MissingNulTerminator(_) => Self::MissingNulTerminator,
			widestring::error::NulError::ContainsNul(c) => Self::ContainsNul(c.nul_position())
		}
	}
}
*/

impl From<std::ffi::NulError> for NulError {
	fn from(e: std::ffi::NulError) -> Self {
		Self::ContainsNul(e.nul_position())
	}
}

/*
impl From<std::ffi::FromBytesWithNulError> for NulError {
	fn from(e: std::ffi::FromBytesWithNulError) -> Self {
		match e.kind {
			std::ffi::FromBytesWithNulErrorKind::InteriorNul(pos) => Self::ContainsNul(pos),
			std::ffi::FromBytesWithNulErrorKind::NotNulTerminated => Self::MissingNulTerminator
		}
	}
}
*/

/// Codes for the phases of handling a `.rar` archive
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
pub(crate) enum When {
	Open,
	Read,
	Process
}

