//! 路由表

use crate::consts::*;
use crate::id::*;
use unmp_link::{ErrorKind as LinkError, Link};
cfg_if::cfg_if! {
    if #[cfg(feature = "role_center")] {
        use alloc::vec::Vec;
        use alloc::collections::BTreeMap;
    } else{
        use crate::consts::{CHILDREN_LEN, PARENT_LEN};
        use fixed_queue::Vec as FixedVec;
        type Vec<T> = FixedVec<T, PARENT_LEN>;
        use fixed_queue::LinearMap;
        type BTreeMap<K, V> = LinearMap<K, V, CHILDREN_LEN>;
    }
}

pub type DisconnectCb = fn(&Id);

/// 错误类型
#[derive(Debug)]
pub enum ErrorKind {
    /// 链路错误
    LinkErr(LinkError),
    /// 无效地址
    InvalidAddress,
    /// 无路由
    NoRoute,
}

/// 路由表
pub struct Router {
    parents: Vec<Link>,
    children: BTreeMap<Id, Link>,
    disconnect_cb: Vec<DisconnectCb>,
}
impl Router {
    pub fn new() -> Router {
        Router {
            parents: Vec::new(),
            children: BTreeMap::new(),
            disconnect_cb: Vec::new(),
        }
    }
    /// 更新路由到路由表上
    pub fn add(&mut self, id: &Id, link: Link) -> bool {
        if self.parents.contains(&link) {
            return true;
        }
        if id.kind() == IdKind::Special {
            if id != &ID_PARENT {
                return false;
            }
            self.parents.push(link);
            return true;
        }

        self.children.insert(id.clone(), link);
        return true;
    }
    /// 链路断开
    pub fn when_link_disconnect(&mut self, link: &Link) {
        // 移除children
        let mut rm_queue: Vec<Id> = Vec::new();
        for (id, route) in self.children.iter() {
            if route == link {
                rm_queue.push(id.clone());
            }
        }
        while let Some(id) = rm_queue.pop() {
            self.children.remove(&id);
            for cb in self.disconnect_cb.iter() {
                cb(&id);
            }
        }

        // 移除parent
        let parent_len = self.parents.len();
        for i in 0..parent_len {
            let index = parent_len - i - 1;
            if &self.parents[index] == link {
                self.parents.swap_remove(index);
            }
        }
    }
    pub fn on_disconnect(&mut self, cb: DisconnectCb) {
        let _err = self.disconnect_cb.push(cb);
    }
    /// 获取链路
    pub fn get_link(&self, id: &Id) -> Option<Link> {
        if id.kind() == IdKind::Special {
            return None;
        }
        self.children.get(id).cloned()
    }

    /// 向下发送数据给目标地址
    pub async fn send_to(
        &mut self,
        buf: &VecPacket,
        id: &Id,
        exclude: Option<&Link>,
    ) -> Result<(), ErrorKind> {
        if id.kind() == IdKind::Special {
            return Err(ErrorKind::InvalidAddress);
        }

        if let Some(route) = self.children.get(id) {
            if exclude == Some(&route) {
                return Err(ErrorKind::NoRoute);
            }
            match route.send(buf).await {
                Ok(_) => {
                    return Ok(());
                }
                Err(LinkError::NotFound) => {
                    self.children.remove(id);
                    return Err(ErrorKind::NoRoute);
                }
                Err(err) => {
                    return Err(ErrorKind::LinkErr(err));
                }
            }
        } else {
            return Err(ErrorKind::NoRoute);
        }
    }

    /// 向上发送数据给父设备
    pub async fn send_to_parent(
        &mut self,
        buf: &VecPacket,
        exclude: Option<&Link>,
    ) -> Result<(), ErrorKind> {
        let mut success: u8 = 0;
        let mut rm_queue: Vec<usize> = Vec::new();
        for i in 0..self.parents.len() {
            if exclude == Some(&self.parents[i]) {
                continue;
            }
            match self.parents[i].send(buf).await {
                Ok(_) => {
                    success += 1;
                }
                Err(LinkError::NotFound) => {
                    rm_queue.push(i);
                }
                Err(_) => {}
            }
        }
        while let Some(i) = rm_queue.pop() {
            self.parents.swap_remove(i);
        }

        if success == 0 {
            return Err(ErrorKind::NoRoute);
        } else {
            return Ok(());
        }
    }
}
