//! 路由表

use crate::consts::*;
use crate::id::*;
use spin::RwLock;
use unmp_link::{ErrorKind as LinkError, Link};
cfg_if::cfg_if! {
    if #[cfg(feature = "role_center")] {
        use alloc::vec::Vec;
        use alloc::collections::BTreeMap;
    } else{
        use crate::consts::{CHILDREN_LEN, PARENT_LEN};
        use fixed_queue::Vec as FixedVec;
        type Vec<T> = FixedVec<T, PARENT_LEN>;
        use fixed_queue::LinearMap;
        type BTreeMap<K, V> = LinearMap<K, V, CHILDREN_LEN>;
    }
}

pub type DisconnectCb = fn(&Id);

static PARENT: RwLock<Vec<Link>> = RwLock::new(Vec::new());
static CHILDREN: RwLock<BTreeMap<Id, Link>> = RwLock::new(BTreeMap::new());
static DISCONNECT_CB: RwLock<Vec<DisconnectCb>> = RwLock::new(Vec::new());

/// 错误类型
#[derive(Debug)]
pub enum ErrorKind {
    /// 链路错误
    LinkErr(LinkError),
    /// 无效地址
    InvalidAddress,
    /// 无路由
    NoRoute,
}

/// 更新路由到路由表上
pub fn add(id: &Id, link: Link) -> bool {
    if PARENT.read().contains(&link) {
        return true;
    }
    if id.kind() == IdKind::Special {
        if id != &ID_PARENT {
            return false;
        }
        PARENT.write().push(link);
        return true;
    }
    {
        CHILDREN.write().insert(id.clone(), link);
    };
    return true;
}
/// 链路断开
pub fn when_link_disconnect(link: &Link) {
    // 移除children
    let mut rm_queue: Vec<Id> = Vec::new();
    let mut children = CHILDREN.write();
    for (id, route) in children.iter() {
        if route == link {
            rm_queue.push(id.clone());
        }
    }
    while let Some(id) = rm_queue.pop() {
        children.remove(&id);
        for cb in DISCONNECT_CB.read().iter() {
            cb(&id);
        }
    }

    // 移除parent
    let mut parent = PARENT.write();
    let parent_len = parent.len();
    for i in 0..parent_len {
        let index = parent_len - i - 1;
        if &parent[index] == link {
            parent.swap_remove(index);
        }
    }
}
pub fn on_disconnect(cb: DisconnectCb) {
    let _err = DISCONNECT_CB.write().push(cb);
}
/// 获取链路
pub fn get_link(id: &Id) -> Option<Link> {
    if id.kind() == IdKind::Special {
        return None;
    }
    CHILDREN.write().get(id).cloned()
}

/// 向下发送数据给目标地址
pub async fn send_to(buf: &VecPacket, id: &Id, exclude: Option<&Link>) -> Result<(), ErrorKind> {
    if id.kind() == IdKind::Special {
        return Err(ErrorKind::InvalidAddress);
    }

    if let Some(route) = CHILDREN.read().get(id) {
        if exclude == Some(&route) {
            return Err(ErrorKind::NoRoute);
        }
        match route.send(buf).await {
            Ok(_) => {
                return Ok(());
            }
            Err(LinkError::NotFound) => {
                CHILDREN.write().remove(id);
                return Err(ErrorKind::NoRoute);
            }
            Err(err) => {
                return Err(ErrorKind::LinkErr(err));
            }
        }
    } else {
        return Err(ErrorKind::NoRoute);
    }
}

/// 向上发送数据给父设备
pub async fn send_to_parent(buf: &VecPacket, exclude: Option<&Link>) -> Result<(), ErrorKind> {
    let mut success: u8 = 0;
    let mut rm_queue: Vec<usize> = Vec::new();
    let mut parent = PARENT.write();
    for i in 0..parent.len() {
        if exclude == Some(&parent[i]) {
            continue;
        }
        match parent[i].send(buf).await {
            Ok(_) => {
                success += 1;
            }
            Err(LinkError::NotFound) => {
                rm_queue.push(i);
            }
            Err(_) => {}
        }
    }
    while let Some(i) = rm_queue.pop() {
        parent.swap_remove(i);
    }

    if success == 0 {
        return Err(ErrorKind::NoRoute);
    } else {
        return Ok(());
    }
}
