//! 数据包格式

use crate::consts::*;
use crate::id::*;
use core::sync::atomic::{AtomicU8, Ordering};
use spin::Lazy;

/// 版本号
pub(crate) const VERSION: u8 = 0x03;
/// 数据包最小长度(数据长度为0)
pub const MIN_LEN: usize = 5;

/// 序列号
static SN: Lazy<AtomicU8> = Lazy::new(|| {
    AtomicU8::new({
        let mut data: [u8; 1] = [0; 1];
        if let Ok(_) = getrandom::getrandom(&mut data) {
            data[0]
        } else {
            0
        }
    })
});

/// 设置SN
pub fn set_sn(sn: u8) {
    SN.store(sn, Ordering::Relaxed);
}

/// 数据包头
#[derive(Clone)]
pub struct PacketHeader {
    /// 序列号
    sn: u8,
    /// 源地址
    src: Id,
    /// 目标地址
    dst: Id,
    /// 生存时间
    ttl: u8,
    /// 协议类型
    protocol: u8,
}
impl PacketHeader {
    pub fn new(src: Id, dst: Id) -> Self {
        PacketHeader {
            sn: SN.fetch_add(1, Ordering::Relaxed),
            src: src,
            dst: dst,
            ttl: 8,
            protocol: 0,
        }
    }
    pub fn sn(&self) -> u8 {
        self.sn
    }
    pub fn src(&self) -> &Id {
        &self.src
    }
    pub fn dst(&self) -> &Id {
        &self.dst
    }
    pub fn ttl(&self) -> u8 {
        self.ttl
    }
    pub fn set_ttl(&mut self, ttl: u8) {
        self.ttl = ttl;
    }
    pub fn protocol(&self) -> u8 {
        self.protocol
    }
    pub fn set_protocol(&mut self, protocol: u8) {
        self.protocol = protocol;
    }
    // output长度应等于PACKET_LEN
    pub fn generate(&self, data: &[u8]) -> VecPacket {
        let head = self;
        let mut output: VecPacket = VecPacket::new();
        output.push(VERSION);
        output.push(head.sn);
        output.extend_from_slice(&head.src);
        output.extend_from_slice(&head.dst);
        output.push(head.ttl);
        output.push(head.protocol);
        output.extend_from_slice(data);
        return output;
    }
}

/// 数据包
pub struct Packet<'a> {
    /// 头
    head: PacketHeader,
    /// 数据
    data: &'a [u8],
}
impl<'a> Packet<'a> {
    pub fn head(&self) -> &PacketHeader {
        &self.head
    }
    pub fn data(&self) -> &[u8] {
        &self.data
    }
    pub fn parse(buf: &'a [u8]) -> Result<Packet<'a>, &'static str> {
        let buf_len = buf.len();
        if buf_len < MIN_LEN {
            return Err("Packet buf too short.");
        } else if buf_len > PACKET_LEN {
            return Err("Packet buf too long.");
        }

        if buf[0] == VERSION {
            let sn = buf[1];
            let (src, rest) = Id::split_buf(&buf[2..]);
            let (dst, rest) = Id::split_buf(rest);
            let ttl = rest[0];
            let protocol = rest[1];
            let data = &rest[2..];
            if data.len() > DATA_LEN {
                return Err("Packet data too long.");
            }
            let head = PacketHeader {
                sn,
                src,
                dst,
                ttl,
                protocol,
            };
            let pkt = Packet { head, data };
            return Ok(pkt);
        } else {
            return Err("Packet no version.");
        }
    }
}
