//! # 网络层
//!
//! 网络层(net)规定了数据包格式(Packet)，负责维护路由表

mod history;
pub mod packet;
mod route;

use crate::consts::*;
use crate::id::*;
use crate::protocol;
use crate::Connection;
use history::History;
use log::{info, warn};
use packet::{Packet, PacketHeader};
pub use route::{get_link, DisconnectCb};
use spin::RwLock;
use unmp_link::Link;

static CFG: RwLock<Config> = RwLock::new(Config {
    id: Id::new(&[]),
    relay: false,
});
static HISTORY: History = History::new();

/// 配置
pub struct Config {
    /// 本设备ID
    id: Id,
    /// 转发开关
    relay: bool,
}
impl Config {
    pub fn new(id: Id) -> Self {
        Config {
            id: id,
            relay: false,
        }
    }
    pub fn id(&self) -> &Id {
        &self.id
    }
    pub fn set_id(&mut self, id: Id) {
        self.id = id;
    }
    pub fn relay(&self) -> bool {
        self.relay
    }
    pub fn set_relay(&mut self, relay: bool) {
        self.relay = relay;
    }
}

/// 初始化
pub fn init(cfg: Config) {
    *CFG.write() = cfg;
    unmp_link::on_destroy(when_link_disconnect);
}
/// 读取本机ID
pub fn get_id() -> Id {
    return CFG.read().id.clone();
}
/// 通用发送函数
async fn send_common(
    head: &PacketHeader,
    data: &[u8],
    dst: Option<&Id>,
    dst_link: Option<&Link>,
    exclude: Option<&Link>,
) -> Result<(), ()> {
    let buf = head.generate(data);
    info!("net send: {:02X?}.", buf);
    let id: &Id = if let Some(id) = dst { id } else { head.dst() };
    if let Some(link) = dst_link {
        // 指定了目标链路
        if let Some(origin) = exclude {
            if origin == link {
                return Err(());
            }
        }
        if let Err(_) = link.send(&buf).await {
            return Err(());
        } else {
            return Ok(());
        }
    } else {
        // 查找目标链路
        if dst == Some(&ID_ALL) {
            return unmp_link::broadcast(&buf).await;
        }
        let result_child = route::send_to(&buf, &id, exclude).await;
        let result_parent = route::send_to_parent(&buf, exclude).await;
        match (result_child, result_parent) {
            (Err(_), Err(_)) => return Err(()),
            _ => return Ok(()),
        }
    }
}
/// 发送数据包到指定设备
pub(crate) async fn send(protocol: u8, data: &[u8], conn: &Connection) -> Result<(), ()> {
    let mut head = PacketHeader::new(get_id(), conn.id().clone());
    head.set_ttl(if conn.id() == &ID_ALL { 0 } else { 8 });
    head.set_protocol(protocol);
    return send_common(&head, &data, Some(conn.id()), conn.link(), None).await;
}
/// 收到了数据包
pub fn when_recv(link: &Link, buf: &[u8]) {
    info!("net recv: {:02X?}, from {}.", buf, link);
    let pkt = match Packet::parse(buf) {
        Ok(result) => result,
        Err(err) => {
            warn!("net recv err: {}", err);
            return;
        }
    };
    if !HISTORY.add(&pkt) {
        info!("net packet repeat.");
        return;
    }
    let mut head = pkt.head().clone();
    let data = pkt.data();

    if head.src() != &ID_PARENT && head.src() != &ID_ALL {
        connect(&head.src(), link.clone());
    }

    if head.dst() == &CFG.read().id || head.dst() == &ID_PARENT || head.dst() == &ID_ALL {
        let mut conn = Connection::new(head.src().clone());
        conn.set_link(link.clone());
        let protocol_id = head.protocol();
        let data = VecData::from(data);
        task_stream::spawn(async move {
            protocol::distribute(protocol_id, conn, &data).await;
        });
    } else if CFG.read().relay {
        if head.ttl() > 0 {
            head.set_ttl(head.ttl() - 1);
            let data = VecData::from(data);
            let link = link.clone();
            task_stream::spawn(async move {
                if let Err(()) = send_common(&head, &data, None, None, Some(&link)).await {
                    info!("can't forward");
                }
            });
        } else {
            info!("unmp TTL is 0.");
        }
    }
}
/// 更新路由
pub fn connect(id: &Id, link: Link) {
    route::add(id, link);
}
/// 链路断开
fn when_link_disconnect(link: &Link) {
    route::when_link_disconnect(link);
}
/// 注册设备断开回调
pub fn on_disconnect(cb: DisconnectCb) {
    route::on_disconnect(cb);
}
