//! 发送器

use crate::consts::*;
use log::*;
use spin::Mutex;
use unmp::id::Id;
use unmp::net;
cfg_if::cfg_if! {
    if #[cfg(feature = "role_center")] {
        use alloc::vec::Vec;
        use alloc::collections::BTreeMap;
    } else{
        use fixed_queue::Vec as FixedVec;
        type Vec<T> = FixedVec<T, MAX_LEN>;
        use fixed_queue::LinearMap;
        type BTreeMap<K, V> = LinearMap<K, V, RETRY_DATA_LEN>;
    }
}

/// 待处理数据
struct Data {
    // 待发送数据
    buf: Vec<u8>,
    // 目标地址
    id: Id,
    // 剩余重传次数
    retry: u8,
    /// 计时器
    t: usize,
    /// 超时等待
    timeout: usize,
}

static DATAS: Mutex<BTreeMap<u16, Data>> = Mutex::new(BTreeMap::new());
static TIMEOUTS: Mutex<BTreeMap<Id, (usize, u8)>> = Mutex::new(BTreeMap::new());

/// 发送数据
pub fn send(index: u16, buf: &[u8], id: Id, retry: u8) {
    net::send(PROTOCOL_ID, &buf, &id, None);
    if retry == 0 {
        return;
    }
    let timeout: usize = {
        let mut timeouts = TIMEOUTS.lock();
        if let Some(timeout) = timeouts.get(&id) {
            timeout.0
        } else {
            timeouts.insert(id.clone(), (10_000, 1));
            10_000
        }
    };
    let data = Data {
        buf: Vec::from(buf),
        id: id,
        retry: retry,
        t: 0,
        timeout: timeout,
    };
    DATAS.lock().insert(index, data);
}
/// 进行重发
pub fn resend() {
    let mut del_queue: Vec<u16> = Vec::new();
    let mut datas = DATAS.lock();
    for (index, data) in datas.iter_mut() {
        data.t += 100;
        if data.retry > 0 {
            if data.t >= data.timeout {
                data.t = 0;
                data.timeout *= 2;
                data.retry -= 1;
                if data.timeout > 60_000 || data.retry == 0 {
                    del_queue.push(*index);
                }
                net::send(PROTOCOL_ID, &data.buf, &data.id, None);
            }
        } else {
            del_queue.push(*index);
        }
    }
    for index in del_queue.iter() {
        info!("etp msg {} timeout.", index);
        datas.remove(&index);
    }
}
/// 收到响应数据
pub fn finish(index: u16) {
    if let Some(data) = DATAS.lock().remove(&index) {
        let mut timeouts = TIMEOUTS.lock();
        let timeout = timeouts.get_mut(&data.id).unwrap();
        if timeout.1 < 200 {
            timeout.1 += 1;
            timeout.0 =
                (timeout.0 * (timeout.1 as usize - 1) + 2 * (data.t + 100)) / timeout.1 as usize;
        }
    }
}
/// 路由断开
pub(crate) fn when_disconnect(id: &Id) {
    TIMEOUTS.lock().remove(id);

    let mut del_queue: Vec<u16> = Vec::new();
    let mut datas = DATAS.lock();
    for (index, data) in datas.iter() {
        if &data.id == id {
            del_queue.push(*index);
        }
    }
    while let Some(index) = del_queue.pop() {
        datas.remove(&index);
    }
}
