//! 简易传输协议

#![no_std]
#![feature(const_btree_new)]

#[cfg(feature = "role_center")]
extern crate alloc;

mod consts;
mod datagram;
mod history;
mod sender;

use consts::*;
use core::time::Duration;
pub use datagram::set_index;
use datagram::{Datagram, DatagramHeader};
use history::History;
use spin::RwLock;
use unmp::id::Id;
use unmp::link::Link;
use unmp::net;
use unmp::protocol;
use unmp::Connection;
cfg_if::cfg_if! {
    if #[cfg(feature = "role_center")] {
        use alloc::collections::BTreeMap;
    } else{
        use fixed_queue::LinearMap;
        type BTreeMap<K, V> = LinearMap<K, V, LISTENERS_LEN>;
    }
}

static HISTORY: History = History::new();

/// socket地址
#[derive(Clone)]
pub struct Addr {
    /// 设备ID
    id: Id,
    /// 端口号
    port: u8,
    /// 链路
    link: Option<Link>,
}
impl Addr {
    pub const fn new(id: Id, port: u8) -> Self {
        Self {
            id: id,
            port: port,
            link: None,
        }
    }
    pub fn id(&self) -> &Id {
        &self.id
    }
    pub fn link(&self) -> &Option<Link> {
        &self.link
    }
    pub fn set_link(&mut self, link: Link) {
        self.link = Some(link);
    }
    pub fn port(&self) -> &u8 {
        &self.port
    }
}

type RecvCallback = fn(remote: Addr, data: &[u8]);

static LISTENERS: RwLock<BTreeMap<u8, RecvCallback>> = RwLock::new(BTreeMap::new());

/// 简易传输协议接收数据
fn when_recv(conn: Connection, data: &[u8]) {
    let datagram = Datagram::parse(data);
    if let Ok(datagram) = datagram {
        let head = datagram.head();
        let body = datagram.body();
        if head.control() < 0x80 {
            // 响应
            let mut is_repeat = false;
            if head.control() & (1 << 1) == (1 << 1) {
                // 需要去重
                if !HISTORY.add(&datagram) {
                    is_repeat = true;
                }
            }
            if head.control() & (1 << 0) == (1 << 0) {
                // 需要响应
                let mut res =
                    DatagramHeader::new(head.dstport(), head.srcport(), Some(head.index()));
                res.set_control(0x81 | (is_repeat as u8) << 1);
                let buf = res.generate(&[]);
                net::send(PROTOCOL_ID, &buf, &conn.id(), conn.link().as_ref());
            }
            // 处理
            if is_repeat {
                return;
            }
            let addr = Addr {
                id: conn.id().clone(),
                link: conn.link().clone(),
                port: head.srcport(),
            };
            if let Some(f) = LISTENERS.read().get(&head.dstport()) {
                f(addr, body);
            }
        } else {
            // 收到响应
            sender::finish(head.index());
        }
    }
}

/// 创建一个简易传输协议实例
pub fn init() {
    protocol::add_protocol(PROTOCOL_ID, when_recv);
    task_stream::spawn(async {
        loop {
            sender::resend();
            task_stream::sleep(Duration::from_millis(100)).await;
        }
    });
    net::on_disconnect(sender::when_disconnect);
}
/// 创建一个socket
pub fn listen(port: u8, f: RecvCallback) {
    LISTENERS.write().insert(port, f);
}
/// 发送数据到指定设备，失败时重试retry次
pub fn send(port: u8, dst: &Addr, body: &[u8], retry: u8) {
    let mut head = DatagramHeader::new(port, dst.port, None);
    head.set_control(if retry == 0 { 2 } else { 3 });
    let buf = head.generate(body);
    sender::send(head.index(), &buf, dst.id.clone(), retry);
}
