//! 服务端

use async_std::net::{TcpListener, TcpStream};
use async_std::sync::Mutex;
use async_std::task;
use async_tungstenite::accept_async;
use async_tungstenite::tungstenite::Message;
use async_tungstenite::WebSocketStream;
use futures_util::{stream::SplitSink, SinkExt, StreamExt};
use log::*;
use std::sync::Arc;
use unmp::net;
use unmp_link::{Driver, ErrorKind, Link, ResultFuture};

struct ServerDriver {
    identifier: String,
    socket: Mutex<SplitSink<WebSocketStream<TcpStream>, Message>>,
}
impl Driver for ServerDriver {
    fn name(&self) -> &str {
        &self.identifier
    }
    fn send(self: Arc<Self>, buf: &[u8]) -> ResultFuture {
        trace!("websocket_server send: {:02X?}.", buf);
        let (result_future, sender) = ResultFuture::new();
        let msg = Message::binary(buf);
        task::spawn(async move {
            let mut ws = self.socket.lock().await;
            match ws.send(msg).await {
                Ok(_) => {
                    let _err = sender.send(Ok(()));
                }
                Err(_) => {
                    warn!("websocket_server send error.");
                    let _err = sender.send(Err(ErrorKind::TimedOut));
                }
            }
        });
        return result_future;
    }
}

/// 创建一个链路实例，并监听指定端口
pub async fn start(ip: &str, port: u16) {
    let listener = TcpListener::bind((ip, port)).await.expect("Failed to bind");
    task::spawn(async move {
        while let Ok((stream, _)) = listener.accept().await {
            let identifier = format!("{}", stream.peer_addr().unwrap());
            let ws = accept_async(stream)
                .await
                .expect("Error during the websocket handshake occurred");
            let (sender, mut recver) = ws.split();
            let driver = Arc::new(ServerDriver {
                identifier: identifier,
                socket: Mutex::new(sender),
            });
            let link_driver = Arc::downgrade(&driver);
            let link = Link::new(link_driver);
            info!("websocket_server new {:?}.", link);
            task::spawn(async move {
                let _driver = driver;
                while let Some(message) = recver.next().await {
                    match message {
                        Ok(m) => match m {
                            Message::Binary(buf) => {
                                trace!("websocket_server recv: {:02X?}.", buf);

                                // 接收数据帧
                                net::when_recv(&link, &buf);
                            }
                            _ => {}
                        },
                        Err(e) => {
                            warn!("websocket_server recv error: {:?}", e);
                            info!("websocket_server rm {}.", link);
                            link.destroy();
                            return;
                        }
                    };
                }
            });
        }
    });
}
