//! 客户端

use async_std::net::TcpStream;
use async_std::sync::Mutex;
use async_tungstenite::async_std::connect_async;
use async_tungstenite::tungstenite::Message;
use async_tungstenite::WebSocketStream;
use futures_util::{stream::SplitSink, SinkExt, StreamExt};
use log::*;
use std::collections::BTreeMap;
use std::sync::Arc;
use unmp::net;
use unmp_link::{Driver, ErrorKind, Link, ResultFuture};

static LINKS: Mutex<BTreeMap<String, Link>> = Mutex::new(BTreeMap::new());

struct ClientDriver {
    identifier: String,
    socket: Mutex<SplitSink<WebSocketStream<TcpStream>, Message>>,
}
impl Driver for ClientDriver {
    fn name(&self) -> &str {
        &self.identifier
    }
    fn send(self: Arc<Self>, buf: &[u8]) -> ResultFuture {
        trace!("websocket_client send: {:02X?}.", buf);
        let (result_future, sender) = ResultFuture::new();
        let msg = Message::binary(buf);
        async_std::task::spawn(async move {
            let mut ws = self.socket.lock().await;
            match ws.send(msg).await {
                Ok(_) => {
                    let _err = sender.send(Ok(()));
                }
                Err(_) => {
                    warn!("websocket_client send error.");
                    let _err = sender.send(Err(ErrorKind::TimedOut));
                }
            }
        });
        return result_future;
    }
}

/// 创建一个链路实例，并连接到指定地址端口
pub async fn start(url: &str) -> Result<Link, ()> {
    let identifier = format!("{}", url);
    if let Some(_) = LINKS.lock().await.get(&identifier) {
        return Err(());
    }

    let (ws, _) = connect_async(url).await.unwrap();
    let (sender, mut recver) = ws.split();

    let driver = Arc::new(ClientDriver {
        identifier: identifier.clone(),
        socket: Mutex::new(sender),
    });
    let link_driver = Arc::downgrade(&driver);
    let link = Link::new(link_driver);
    info!("websocket_client new {:?}.", link);
    LINKS.lock().await.insert(identifier.clone(), link.clone());

    let link_tmp = link.clone();
    async_std::task::spawn(async move {
        let link = link_tmp;
        let _driver = driver;
        while let Some(message) = recver.next().await {
            match message {
                Ok(m) => match m {
                    Message::Binary(buf) => {
                        trace!("websocket_client recv: {:02X?}.", buf);

                        // 接收数据帧
                        net::when_recv(&link, &buf);
                    }
                    _ => {}
                },
                Err(e) => {
                    warn!("websocket_client {:?} recv error: {:?}", link, e);
                    LINKS.lock().await.remove(&identifier);
                    link.destroy();
                    return;
                }
            };
        }
    });
    return Ok(link);
}
