//! 服务端

use eds::consts::FRAME_LEN;
use eds::{Reader, Writer};
use log::*;
use spin::{Mutex, RwLock};
use std::collections::BTreeMap;
use std::net::UdpSocket;
use std::sync::atomic::{AtomicI64, Ordering};
use std::sync::{Arc, Weak};
use std::thread;
use std::time::Duration;
use unmp::link::{Driver, ErrorKind, Link, SendResultFuture};
use unmp::net;

static LINKS: Mutex<BTreeMap<String, Link>> = Mutex::new(BTreeMap::new());
static DRIVERS: RwLock<BTreeMap<Link, Arc<ServerDriver>>> = RwLock::new(BTreeMap::new());

struct ServerDriver {
    identifier: String,
    address: String,
    port: u16,
    socket: Weak<UdpSocket>,
    last_time: AtomicI64,
    eds: bool,
}
impl Driver for ServerDriver {
    fn name(&self) -> &str {
        &self.identifier
    }
    fn send(self: Arc<Self>, buf: &[u8]) -> SendResultFuture {
        trace!("udp_server send: {:02X?}.", buf);
        let (result_future, sender) = SendResultFuture::new();
        if let Some(socket) = self.socket.upgrade() {
            let mut writer = Writer::new(4);
            let buf = if self.eds { writer.get_frame(buf) } else { buf };

            match socket.send_to(buf, (&self.address[..], self.port)) {
                Ok(_) => {
                    let _err = sender.send(Ok(()));
                }
                Err(_) => {
                    warn!("udp_server send error.");
                    let _err = sender.send(Err(ErrorKind::TimedOut));
                }
            };
        } else {
            let _err = sender.send(Err(ErrorKind::NotConnected));
        }
        return result_future;
    }
}

/// 创建一个UDP链路实例，并监听指定端口
pub fn start(ip: &str, port: u16, eds: bool) {
    let socket = UdpSocket::bind((ip, port)).expect("udp_server bind error");
    socket
        .set_write_timeout(Some(Duration::new(0, 500_000)))
        .expect("udp_server set write timeout");
    let socket = Arc::new(socket);

    thread::spawn(move || loop {
        thread::sleep(Duration::from_secs(3 * 60));
        let now = chrono::Local::now().timestamp_millis();
        let links = LINKS.lock().clone();
        *LINKS.lock() = links
            .into_iter()
            .filter(|(_, link)| {
                if let Some(driver) = DRIVERS.read().get(link) {
                    if driver.last_time.load(Ordering::Relaxed) < now - 10 * 60 * 1000 {
                        return true;
                    }
                }
                info!("udp_server rm {}.", link);
                link.destroy();
                false
            })
            .collect();
    });
    thread::spawn(move || {
        let mut reader = Reader::new();
        let mut buf = [0; FRAME_LEN];
        loop {
            match socket.recv_from(&mut buf) {
                Ok((len, src)) => {
                    let identifier = format!("{}:{}", src.ip(), src.port());
                    let now = chrono::Local::now().timestamp_millis();
                    let link = LINKS
                        .lock()
                        .entry(identifier.clone())
                        .and_modify(|v| {
                            DRIVERS
                                .write()
                                .get_mut(v)
                                .unwrap()
                                .last_time
                                .store(now, Ordering::Relaxed);
                        })
                        .or_insert_with(|| {
                            let driver = Arc::new(ServerDriver {
                                identifier: identifier,
                                address: src.ip().to_string(),
                                port: src.port(),
                                socket: Arc::downgrade(&socket),
                                last_time: AtomicI64::new(now),
                                eds: eds,
                            });
                            let link_driver = Arc::downgrade(&driver);
                            let link = Link::new(link_driver);
                            info!("udp_server new {:?}.", link);
                            DRIVERS.write().insert(link.clone(), driver);
                            link
                        })
                        .clone();
                    let buf: &[u8] = &buf[..len];
                    trace!("udp_server recv: {:02X?}.", buf);

                    // 接收数据帧
                    if eds {
                        reader.recv(buf);
                        if reader.is_ready() {
                            net::when_recv(&link, reader.get_load());
                        }
                        while !reader.is_finish() {
                            reader.recv(&[]);
                            if reader.is_ready() {
                                net::when_recv(&link, reader.get_load());
                            }
                        }
                    } else {
                        net::when_recv(&link, buf);
                    }
                }
                Err(e) => {
                    warn!("udp_server recv error: {:?}", e.kind());
                    return;
                }
            }
        }
    });
}
