//! 客户端

use eds::consts::FRAME_LEN;
use eds::{Reader, Writer};
use log::*;
use spin::Mutex;
use std::collections::BTreeMap;
use std::net::UdpSocket;
use std::sync::Arc;
use std::thread;
use std::time::Duration;
use unmp::link::{Driver, ErrorKind, Link, SendResultFuture};
use unmp::net;

static LINKS: Mutex<BTreeMap<String, Link>> = Mutex::new(BTreeMap::new());

struct ClientDriver {
    identifier: String,
    socket: UdpSocket,
    eds: bool,
}
impl Driver for ClientDriver {
    fn name(&self) -> &str {
        &self.identifier
    }
    fn send(self: Arc<Self>, buf: &[u8]) -> SendResultFuture {
        trace!("udp_client send: {:02X?}.", buf);
        let (result_future, sender) = SendResultFuture::new();
        let mut writer = Writer::new(4);
        let buf = if self.eds { writer.get_frame(buf) } else { buf };

        match self.socket.send(buf) {
            Ok(_) => {
                let _err = sender.send(Ok(()));
            }
            Err(_) => {
                warn!("udp_client send error.");
                let _err = sender.send(Err(ErrorKind::TimedOut));
            }
        };
        return result_future;
    }
}

/// 创建一个UDP链路实例，并连接到指定地址端口
pub fn start(ip: &str, port: u16, eds: bool) -> Result<Link, ()> {
    let identifier = format!("{}:{}", ip, port);
    if let Some(_) = LINKS.lock().get(&identifier) {
        return Err(());
    }

    let socket = UdpSocket::bind(("0.0.0.0", 0)).expect("udp_client bind error");
    socket
        .set_write_timeout(Some(Duration::new(0, 500_000)))
        .expect("udp_client set write timeout");
    socket
        .connect((ip, port))
        .expect("udp_client connect error");
    let driver = Arc::new(ClientDriver {
        identifier: identifier.clone(),
        socket: socket,
        eds: eds,
    });
    let link_driver = Arc::downgrade(&driver);
    let link = Link::new(link_driver);
    info!("udp_client new {:?}.", link);
    LINKS.lock().insert(identifier.clone(), link.clone());

    //设置数据接收接口
    let link_tmp = link.clone();
    thread::spawn(move || {
        let link = link_tmp;
        let mut reader = Reader::new();
        let mut buf = [0; FRAME_LEN];
        loop {
            match driver.socket.recv(&mut buf) {
                Ok(len) => {
                    let buf: &[u8] = &buf[..len];
                    trace!("udp_client recv: {:02X?}.", buf);

                    // 接收数据帧
                    if eds {
                        reader.recv(buf);
                        if reader.is_ready() {
                            net::when_recv(&link, reader.get_load());
                        }
                        while !reader.is_finish() {
                            reader.recv(&[]);
                            if reader.is_ready() {
                                net::when_recv(&link, reader.get_load());
                            }
                        }
                    } else {
                        net::when_recv(&link, buf);
                    }
                }
                Err(e) => {
                    warn!("udp_client {:?} recv error: {:?}", link, e.kind());
                    LINKS.lock().remove(&identifier);
                    link.destroy();
                    return;
                }
            }
        }
    });
    return Ok(link);
}
