//! Redis链路驱动层
#![feature(const_btree_new)]

use futures_util::stream::StreamExt;
use log::*;
use redis::{AsyncCommands, IntoConnectionInfo};
use spin::Mutex;
use std::collections::BTreeMap;
use std::sync::Arc;
use unmp::consts::*;
use unmp::link::{Driver, ErrorKind, Link, SendResultFuture};
use unmp::net;

static LINKS: Mutex<BTreeMap<String, Link>> = Mutex::new(BTreeMap::new());

pub struct ClientDriver {
    identifier: String,
    client: redis::Client,
    topic: String,
}
impl Driver for ClientDriver {
    fn name(&self) -> &str {
        &self.identifier
    }
    fn send(self: Arc<Self>, buf: &[u8]) -> SendResultFuture {
        trace!("redis send: {:02X?}.", buf);
        let (result_future, sender) = SendResultFuture::new();
        let buf = VecPacket::from(buf);
        task_stream::spawn(async move {
            let mut tx = self.client.get_async_connection().await.unwrap();
            match tx.publish::<_, _, u8>(&self.topic, buf.as_slice()).await {
                Ok(_) => {
                    let _err = sender.send(Ok(()));
                }
                Err(_) => {
                    warn!("redis send error.");
                    let _err = sender.send(Err(ErrorKind::TimedOut));
                }
            };
        });
        return result_future;
    }
}

/// 创建一个UDP链路实例，并连接到指定地址端口
pub fn start(url: &str, topic: &str) -> Result<Link, ()> {
    let url = {
        if let Ok(url) = url.into_connection_info() {
            url
        } else {
            return Err(());
        }
    };
    let identifier = format!("{}/{}", url.addr, topic);
    if let Some(_) = LINKS.lock().get(&identifier) {
        return Err(());
    }

    let client = redis::Client::open(url).unwrap();
    let driver = Arc::new(ClientDriver {
        identifier: identifier.clone(),
        client: client,
        topic: String::from(topic),
    });
    let link_driver = Arc::downgrade(&driver);
    let link = Link::new(link_driver);
    info!("redis new {:?}.", link);
    LINKS.lock().insert(identifier.clone(), link.clone());

    //设置数据接收接口
    let link_tmp = link.clone();
    task_stream::spawn(async move {
        let link = link_tmp;
        let rx = driver.client.get_async_connection().await.unwrap();
        let mut pubsub = rx.into_pubsub();
        pubsub.subscribe(&driver.topic).await.unwrap();
        while let Some(msg) = pubsub.on_message().next().await {
            let buf = msg.get_payload_bytes();
            trace!("redis recv: {:02X?}.", buf);

            // 接收数据帧
            net::when_recv(&link, &buf);
        }
        warn!("redis {:?} recv error.", link);
        LINKS.lock().remove(&identifier);
        link.destroy();
    });

    return Ok(link);
}
