//
// Copyright (c) 2022 Oleg Lelenkov <o.lelenkov@gmail.com>
// Distributed under terms of the BSD 3-Clause license.
//

mod accessor;
mod error;
#[cfg(feature = "toml")]
pub mod toml;
#[cfg(feature = "yaml")]
pub mod yaml;

pub use self::error::{UniNodeFmtError, UniNodeLoadError, UniNodeIoError};
pub use self::accessor::{StdFileAccessor, FileAccesor, AsyncFileAccesor};

use std::collections::HashMap;
use std::sync::RwLock;
use std::path::Path;

use once_cell::sync::Lazy;

use crate::value::UniNode;

type GlobalFormat = Box<dyn UniNodeFormat + Sync + Send + 'static>;
type FormatMap = HashMap<String, GlobalFormat>;

static FORMATS: Lazy<RwLock<FormatMap>> = Lazy::new(|| {
    let mut cont = FormatMap::new();
    #[cfg(feature = "yaml")]
    add_format(&mut cont, yaml::YamlFormat::default());
    #[cfg(feature = "toml")]
    add_format(&mut cont, toml::TomlFormat::default());
    RwLock::new(cont)
});

pub trait UniNodeFormat {
    fn extensions(&self) -> &[&str];

    fn parse(&self, text: &str) -> Result<UniNode, UniNodeFmtError>;

    fn compile(&self, node: &UniNode) -> Result<String, UniNodeFmtError>;
}

fn add_format<L>(cont: &mut FormatMap, format: L)
where
    L: UniNodeFormat + Clone + Sync + Send + 'static,
{
    for ext in format.extensions() {
        cont.insert(ext.to_string(), Box::new(format.clone()));
    }
}

pub fn register_format<F>(format: F)
where
    F: UniNodeFormat + Clone + Sync + Send + 'static,
{
    let mut formats = FORMATS.write().unwrap();
    add_format(&mut formats, format);
}

pub fn available_extensions() -> Vec<String> {
    let formats = FORMATS.read().unwrap();
    formats.keys().cloned().collect()
}

impl UniNode {
    pub fn parse_with_acc<P, A>(
        path: P, accessor: A,
    ) -> Result<UniNode, UniNodeLoadError>
    where
        P: AsRef<Path>,
        A: FileAccesor,
    {
        let path = path.as_ref();
        match path.extension().and_then(|e| e.to_str()) {
            Some(ext) => {
                let formats = FORMATS.read().unwrap();
                if !formats.contains_key(ext) {
                    let err =
                        UniNodeFmtError::NotDefinedFormat(ext.to_string());
                    return Err(UniNodeLoadError::Fmt(err));
                }
                let content = accessor
                    .read(path)
                    .map_err(|e| UniNodeIoError::new(path, e))?;
                formats[ext].parse(&content).map_err(UniNodeLoadError::Fmt)
            },
            None => {
                for (ext, fmt) in FORMATS.read().unwrap().iter() {
                    let full_path = path.with_extension(&ext);
                    if accessor.exists(&full_path) {
                        let content = accessor
                            .read(&full_path)
                            .map_err(|e| UniNodeIoError::new(path, e))?;
                        return fmt
                            .parse(&content)
                            .map_err(UniNodeLoadError::Fmt);
                    }
                }
                Err(UniNodeLoadError::NotFoundLoader)
            },
        }
    }

    pub fn parse<P>(
        path: P
    ) -> Result<UniNode, UniNodeLoadError>
    where
        P: AsRef<Path>,
    {
        Self::parse_with_acc(path, StdFileAccessor)
    }

    pub async fn parse_async<P, A>(
        path: P, accessor: A,
    ) -> Result<UniNode, UniNodeLoadError>
    where
        P: AsRef<Path>,
        A: AsyncFileAccesor,
    {
        let path = path.as_ref();
        match path.extension().and_then(|e| e.to_str()) {
            Some(ext) => {
                let formats = FORMATS.read().unwrap();
                if !formats.contains_key(ext) {
                    let err =
                        UniNodeFmtError::NotDefinedFormat(ext.to_string());
                    return Err(UniNodeLoadError::Fmt(err));
                }
                let content = accessor
                    .read(path)
                    .await
                    .map_err(|e| UniNodeIoError::new(path, e))?;
                formats[ext].parse(&content).map_err(UniNodeLoadError::Fmt)
            },
            None => {
                for (ext, fmt) in FORMATS.read().unwrap().iter() {
                    let full_path = path.with_extension(&ext);
                    if accessor.exists(&full_path).await {
                        let content = accessor
                            .read(&full_path)
                            .await
                            .map_err(|e| UniNodeIoError::new(path, e))?;
                        return fmt
                            .parse(&content)
                            .map_err(UniNodeLoadError::Fmt);
                    }
                }
                Err(UniNodeLoadError::NotFoundLoader)
            },
        }
    }
}
