//
// Copyright (c) 2022 Oleg Lelenkov <o.lelenkov@gmail.com>
// Distributed under terms of the BSD 3-Clause license.
//

use std::path::{PathBuf, Path};
use std::io::Error as IoError;
use std::num::{ParseFloatError, TryFromIntError};

use thiserror::Error;

use crate::value::Expected;

#[derive(Debug, thiserror::Error)]
pub enum UniNodeFmtError {
    #[error("{0}")]
    Format(anyhow::Error),
    #[error("Not defined format '{0}'")]
    NotDefinedFormat(String),
    #[error("Not supported '{0}' value")]
    NotSupportedType(Expected),
    #[error("{0}")]
    Conv(String),
}

impl From<ParseFloatError> for UniNodeFmtError {
    fn from(err: ParseFloatError) -> Self {
        Self::Conv(format!("Failed conversion to float: {}", err))
    }
}

impl From<TryFromIntError> for UniNodeFmtError {
    fn from(err: TryFromIntError) -> Self {
        Self::Conv(format!("Failed conversion to integer: {}", err))
    }
}

#[derive(Debug, Error)]
#[error("{path}: {source}")]
pub struct UniNodeIoError {
    path: PathBuf,
    #[source]
    source: std::io::Error,
}

impl UniNodeIoError {
    pub fn new<E, P>(path: P, err: E) -> Self
    where
        E: Into<IoError>,
        P: Into<PathBuf>,
    {
        Self {
            path: path.into(),
            source: err.into(),
        }
    }

    pub fn source(&self) -> &IoError {
        &self.source
    }

    pub fn path(&self) -> &Path {
        &self.path
    }
}

#[derive(Debug, thiserror::Error)]
pub enum UniNodeLoadError {
    #[error(transparent)]
    Io(#[from] UniNodeIoError),
    #[error(transparent)]
    Fmt(#[from] UniNodeFmtError),
    #[error("Not found loader")]
    NotFoundLoader,
}
