// mod.rs
// Copyright: (c) 2015-2021, Oleg Lelenkov
// Distributed under terms of the BSD 3-Clause license.
//

mod error;
#[macro_use]
mod macros;

pub use error::UniNodeError;

use std::collections::HashMap;
use std::convert::TryFrom;
use std::fmt::{self, Display};

use error::{Result, Expected};

const PATH_DELIMITER: char = '.';

/// Type representing a UniNode array, payload of the `UniNode::Array` variant
pub(crate) type Array = Vec<UniNode>;

/// Type representing a UniNode object, payload of the `UniNode::Object` variant.
pub(crate) type Object = HashMap<String, UniNode>;

/// Representation of a UniNode value.
#[derive(Debug, Clone, PartialEq)]
pub enum UniNode {
    /// Represents a UniNode null
    Null,
    /// Represents a UniNode bool
    Boolean(bool),
    /// Represents a UniNode integer
    Integer(i64),
    /// Represents a UniNode unsigned integer
    UInteger(u64),
    /// Represents a UniNode float
    Float(f64),
    /// Represents a UniNode bytes array
    Bytes(Vec<u8>),
    /// Represents a UniNode string
    String(String),
    /// Represents a UniNode array
    Array(Array),
    /// Represents a UniNode object
    Object(Object),
}

impl UniNode {
    impl_is_function!(is_bool, Boolean);

    impl_is_function!(is_int, Integer);

    impl_is_function!(is_uint, UInteger);

    impl_is_function!(is_float, Float);

    impl_is_function!(is_bytes, Bytes);

    impl_is_function!(is_string, String);

    impl_is_function!(is_object, Object);

    impl_is_function!(is_array, Array);

    impl_get_function!(get_bool, as_bool, bool);

    impl_get_function!(get_int, as_int, i64);

    impl_get_function!(get_uint, as_uint, u64);

    impl_get_function!(get_float, as_float, f64);

    impl_get_function!(get_str, as_str, &str);

    impl_get_function!(get_bytes, as_bytes, &[u8]);

    pub fn empty_array() -> Self {
        UniNode::Array(Array::new())
    }

    pub fn empty_object() -> Self {
        UniNode::Object(Object::new())
    }

    pub fn is_null(&self) -> bool {
        matches!(self, UniNode::Null)
    }

    pub fn is_some_int(&self) -> bool {
        matches!(self, UniNode::Integer(_) | UniNode::UInteger(_))
    }

    fn invalid_error(&self, expected: Expected) -> UniNodeError {
        UniNodeError::invalid_type(self, expected)
    }

    pub fn as_bool(&self) -> Result<bool> {
        match self {
            UniNode::Boolean(val) => Ok(*val),
            UniNode::Integer(val) => Ok(*val != 0),
            UniNode::UInteger(val) => Ok(*val != 0),
            UniNode::Float(val) => Ok(*val != 0.0),
            UniNode::String(val) => {
                match val.to_lowercase().as_ref() {
                    "1" | "true" | "on" | "yes" => Ok(true),
                    "0" | "false" | "off" | "no" => Ok(false),
                    _ => Err(self.invalid_error(Expected::Bool)),
                }
            },
            _ => Err(self.invalid_error(Expected::Bool)),
        }
    }

    pub fn as_int(&self) -> Result<i64> {
        match self {
            UniNode::Boolean(val) => Ok(if *val { 1 } else { 0 }),
            UniNode::Integer(val) => Ok(*val),
            UniNode::UInteger(val) => {
                i64::try_from(*val)
                    .map_err(|_| self.invalid_error(Expected::Integer))
            },
            UniNode::Float(val) => Ok(val.round() as i64),
            UniNode::String(val) => {
                match val.to_lowercase().as_ref() {
                    "1" | "true" | "on" | "yes" => Ok(1),
                    "0" | "false" | "off" | "no" => Ok(0),
                    _ => {
                        val.parse()
                            .map_err(|_| self.invalid_error(Expected::Integer))
                    },
                }
            },
            _ => Err(self.invalid_error(Expected::Integer)),
        }
    }

    pub fn as_uint(&self) -> Result<u64> {
        match self {
            UniNode::Boolean(val) => Ok(if *val { 1 } else { 0 }),
            UniNode::UInteger(val) => Ok(*val),
            UniNode::Integer(val) => {
                u64::try_from(*val)
                    .map_err(|_| self.invalid_error(Expected::UInteger))
            },
            UniNode::Float(val) => {
                u64::try_from(val.round() as i64)
                    .map_err(|_| self.invalid_error(Expected::UInteger))
            },
            UniNode::String(val) => {
                match val.to_lowercase().as_ref() {
                    "1" | "true" | "on" | "yes" => Ok(1),
                    "0" | "false" | "off" | "no" => Ok(0),
                    _ => {
                        val.parse()
                            .map_err(|_| self.invalid_error(Expected::UInteger))
                    },
                }
            },
            _ => Err(self.invalid_error(Expected::UInteger)),
        }
    }

    pub fn as_float(&self) -> Result<f64> {
        match self {
            UniNode::Boolean(val) => Ok(if *val { 1.0 } else { 0.0 }),
            UniNode::Integer(val) => Ok(*val as f64),
            UniNode::UInteger(val) => Ok(*val as f64),
            UniNode::Float(val) => Ok(*val),
            UniNode::String(val) => {
                match val.to_lowercase().as_ref() {
                    "1" | "true" | "on" | "yes" => Ok(1.0),
                    "0" | "false" | "off" | "no" => Ok(0.0),
                    _ => {
                        val.parse()
                            .map_err(|_| self.invalid_error(Expected::Float))
                    },
                }
            },
            _ => Err(self.invalid_error(Expected::Float)),
        }
    }

    pub fn as_str(&self) -> Result<&str> {
        match self {
            UniNode::String(val) => Ok(val),
            _ => Err(self.invalid_error(Expected::String)),
        }
    }

    pub fn as_bytes(&self) -> Result<&[u8]> {
        match self {
            UniNode::Bytes(val) => Ok(val),
            UniNode::String(val) => Ok(val.as_bytes()),
            _ => Err(self.invalid_error(Expected::Bytes)),
        }
    }

    pub fn to_bytes(self) -> Result<Vec<u8>> {
        match self {
            UniNode::Bytes(val) => Ok(val),
            UniNode::String(val) => Ok(val.into_bytes()),
            _ => Err(self.invalid_error(Expected::Bytes)),
        }
    }

    pub fn to_string(self) -> Result<String> {
        match self {
            UniNode::String(val) => Ok(val),
            UniNode::Boolean(val) => Ok(val.to_string()),
            UniNode::Integer(val) => Ok(val.to_string()),
            UniNode::UInteger(val) => Ok(val.to_string()),
            UniNode::Float(val) => Ok(val.to_string()),
            _ => Err(self.invalid_error(Expected::String)),
        }
    }

    pub fn as_array(&self) -> Result<&Vec<UniNode>> {
        match self {
            UniNode::Array(val) => Ok(val),
            _ => Err(self.invalid_error(Expected::Array)),
        }
    }

    pub fn as_array_mut(&mut self) -> Result<&mut Vec<UniNode>> {
        match self {
            UniNode::Array(ref mut val) => Ok(val),
            _ => Err(self.invalid_error(Expected::Array)),
        }
    }

    pub fn to_array(self) -> UniNode {
        match self {
            UniNode::Array(_) => self,
            UniNode::Object(mut o) => {
                UniNode::Array(o.drain().map(|(_k, v)| v).collect())
            },
            _ => UniNode::Array(vec![self]),
        }
    }

    pub fn as_object(&self) -> Result<&HashMap<String, UniNode>> {
        match self {
            UniNode::Object(val) => Ok(val),
            _ => Err(self.invalid_error(Expected::Object)),
        }
    }

    pub fn as_object_mut(&mut self) -> Result<&mut HashMap<String, UniNode>> {
        match self {
            UniNode::Object(ref mut val) => Ok(val),
            _ => Err(self.invalid_error(Expected::Object)),
        }
    }

    pub fn find(&self, path: &str) -> Option<&Self> {
        let names = path.split(PATH_DELIMITER);

        fn find_path<'a, 'b, I>(
            root: &'a UniNode, mut names: I,
        ) -> Option<&'a UniNode>
        where
            I: Iterator<Item = &'b str>,
        {
            if let Some(name) = names.next() {
                if !name.is_empty() {
                    if let UniNode::Object(tbl) = root {
                        if let Some(chd) = tbl.get(name) {
                            return find_path(chd, names);
                        }
                    }
                    return None;
                }
            };
            Some(root)
        }

        find_path(self, names)
    }

    pub fn find_mut(&mut self, path: &str) -> Option<&mut Self> {
        let names = path.split(PATH_DELIMITER);

        fn find_path<'a, 'b, I>(
            root: &'a mut UniNode, mut names: I,
        ) -> Option<&'a mut UniNode>
        where
            I: Iterator<Item = &'b str>,
        {
            if let Some(name) = names.next() {
                if !name.is_empty() {
                    if let UniNode::Object(tbl) = root {
                        if let Some(chd) = tbl.get_mut(name) {
                            return find_path(chd, names);
                        }
                    }
                    return None;
                }
            };
            Some(root)
        }

        find_path(self, names)
    }

    pub fn get_array(&self, path: &str) -> Option<&Vec<UniNode>> {
        self.find(path).and_then(|v| v.as_array().ok())
    }

    pub fn get_object(&self, path: &str) -> Option<&HashMap<String, UniNode>> {
        self.find(path).and_then(|v| v.as_object().ok())
    }

    pub fn merge(&mut self, other: UniNode) {
        fn merge_node(dst: &mut UniNode, src: UniNode) {
            match dst {
                UniNode::Null => *dst = src,
                UniNode::Array(a) => {
                    if src.is_array() {
                        if let UniNode::Array(src) = src {
                            a.extend(src);
                        }
                    } else {
                        a.push(src);
                    }
                },
                UniNode::Object(o) if src.is_object() => {
                    if let UniNode::Object(src) = src {
                        #[allow(clippy::map_entry)]
                        for (key, node) in src {
                            if o.contains_key(&key) {
                                merge_node(o.get_mut(&key).unwrap(), node);
                            } else {
                                o.insert(key, node);
                            }
                        }
                    }
                },
                _ => {
                    if !src.is_null() {
                        let v = match src {
                            a @ UniNode::Array(_) => a,
                            a => UniNode::Array(vec![a]),
                        };
                        let v = std::mem::replace(dst, v);
                        dst.as_array_mut().unwrap().insert(0, v);
                    }
                },
            }
        }

        merge_node(self, other)
    }
}

#[test]
fn value_merge_test() {
    fn test_merge(mut root: UniNode, other: UniNode, res: UniNode) {
        root.merge(other);
        assert_eq!(root, res);
    }
    test_merge(unode!(), unode!(), unode!());
    test_merge(unode!(), unode![1, 2], unode![1, 2]);
    test_merge(unode!(1), unode!(2), unode![1, 2]);
    test_merge(unode!(1), unode![2, 3], unode![1, 2, 3]);
    test_merge(unode!(1), unode! {"1" => 1}, unode![1, unode! {"1" => 1}]);
    test_merge(unode![1, 2], unode![3, 4], unode![1, 2, 3, 4]);
    test_merge(unode![1, 2], unode!(3), unode![1, 2, 3]);
    test_merge(unode![1, 2], unode! {"3" => 3}, unode![
        1,
        2,
        unode! {"3" => 3}
    ]);
    test_merge(unode! {"1" => 1}, unode!(2), unode![unode! {"1" => 1}, 2]);
    test_merge(unode! {"1" => 1}, unode![2, 3], unode![
        unode! {"1" => 1},
        2,
        3
    ]);
    test_merge(
        unode! {"1" => 1},
        unode! {"2" => 2},
        unode! {"1" => 1, "2" => 2},
    );
    test_merge(
        unode! {"zero" => unode!(), "one" => 1, "two" => unode!{"right" => 2}},
        unode! {"zero" => 0, "one" => 1.1, "two" => unode!{"left" => 1}, "three" => 3},
        unode! {"zero" => 0, "one" => unode![1, 1.1], "two" => unode!{"right" => 2, "left" => 1}, "three" => 3},
    );
}

#[test]
fn value_to_array_test() {
    assert_eq!(unode!(1).to_array(), UniNode::Array(vec![unode!(1)]));
    assert_eq!(
        unode!("hello").to_array(),
        UniNode::Array(vec![unode!("hello")])
    );
    assert_eq!(unode![1, 2].to_array(), unode![1, 2]);
    let v = unode! {"1" => 1, "2" => 2}.to_array();
    assert!(v.is_array());
    assert_eq!(v.as_array().unwrap().len(), 2);
    assert!(v.as_array().unwrap().contains(&unode!(1)));
    assert!(v.as_array().unwrap().contains(&unode!(2)));
    assert!(!v.as_array().unwrap().contains(&unode!(3)));
}

#[test]
fn value_is_functions_test() {
    assert!(UniNode::Null.is_null());
    assert!(UniNode::Boolean(false).is_bool());
    assert!(UniNode::Integer(1).is_int());
    assert!(UniNode::UInteger(1).is_uint());
    assert!(UniNode::Float(1.1).is_float());
    assert!(UniNode::String("".to_string()).is_string());
    assert!(UniNode::Bytes(vec![]).is_bytes());
    assert!(UniNode::empty_array().is_array());
    assert!(UniNode::empty_object().is_object());
    assert!(UniNode::Integer(1).is_some_int());
    assert!(UniNode::UInteger(1).is_some_int());
}

pub trait InnerType {}

macro_rules! impl_value_into {
    ($($values:ty),+ $(,)?) => {
        $(impl InnerType for $values {})+
    }
}

impl_value_into!(
    bool, i8, i16, i32, i64, u16, u32, u64, f32, f64, String, &str
);

impl<T: Into<UniNode>> InnerType for Option<T> {}

impl<T: Into<UniNode>> InnerType for Vec<T> {}

impl<T: Into<UniNode>> InnerType for HashMap<String, T> {}

impl From<bool> for UniNode {
    fn from(value: bool) -> Self {
        UniNode::Boolean(value)
    }
}

impl From<i8> for UniNode {
    fn from(value: i8) -> Self {
        UniNode::Integer(value as i64)
    }
}

impl From<i16> for UniNode {
    fn from(value: i16) -> Self {
        UniNode::Integer(value as i64)
    }
}

impl From<i32> for UniNode {
    fn from(value: i32) -> Self {
        UniNode::Integer(value as i64)
    }
}

impl From<i64> for UniNode {
    fn from(value: i64) -> Self {
        UniNode::Integer(value)
    }
}

impl From<u8> for UniNode {
    fn from(value: u8) -> Self {
        UniNode::UInteger(value as u64)
    }
}

impl From<u16> for UniNode {
    fn from(value: u16) -> Self {
        UniNode::UInteger(value as u64)
    }
}

impl From<u32> for UniNode {
    fn from(value: u32) -> Self {
        UniNode::UInteger(value as u64)
    }
}

impl From<u64> for UniNode {
    fn from(value: u64) -> Self {
        UniNode::UInteger(value)
    }
}

impl From<f32> for UniNode {
    fn from(value: f32) -> Self {
        UniNode::Float(value as f64)
    }
}

impl From<f64> for UniNode {
    fn from(value: f64) -> Self {
        UniNode::Float(value)
    }
}

impl From<String> for UniNode {
    fn from(value: String) -> Self {
        UniNode::String(value)
    }
}

impl From<&str> for UniNode {
    fn from(value: &str) -> Self {
        UniNode::String(value.into())
    }
}

impl<T> From<Option<T>> for UniNode
where
    T: Into<UniNode>,
{
    fn from(value: Option<T>) -> Self {
        match value {
            Some(value) => value.into(),
            None => UniNode::Null,
        }
    }
}

impl From<Vec<u8>> for UniNode {
    fn from(value: Vec<u8>) -> Self {
        UniNode::Bytes(value)
    }
}

impl From<&[u8]> for UniNode {
    fn from(value: &[u8]) -> Self {
        UniNode::Bytes(value.to_vec())
    }
}

impl<T> From<Vec<T>> for UniNode
where
    T: Into<UniNode> + InnerType,
{
    fn from(value: Vec<T>) -> Self {
        let mut data = Vec::new();
        for v in value {
            data.push(v.into());
        }
        UniNode::Array(data)
    }
}

impl<T> From<HashMap<String, T>> for UniNode
where
    T: Into<UniNode>,
{
    fn from(value: HashMap<String, T>) -> Self {
        let mut data = HashMap::new();
        for (k, v) in value {
            let _ = data.insert(k.clone(), v.into());
        }
        UniNode::Object(data)
    }
}

#[test]
fn value_construct_test() {
    assert_eq!(unode!(), UniNode::Null);
    assert_eq!(unode!(true), UniNode::Boolean(true));
    assert_eq!(unode!(5u8), UniNode::UInteger(5));
    assert_eq!(unode!("one"), UniNode::String("one".to_string()));
    assert_eq!(
        unode!(1, 2.4, "one", false),
        UniNode::Array(vec![
            UniNode::Integer(1),
            UniNode::Float(2.4),
            UniNode::String("one".to_string()),
            UniNode::Boolean(false),
        ])
    );
    assert_eq!(UniNode::from(Some(unode!(11))), unode!(11));
    assert_eq!(unode! {"one" => 1.0}, {
        let mut hm = HashMap::new();
        hm.insert("one".to_string(), UniNode::Float(1.0));
        UniNode::Object(hm)
    });
    assert_eq!(unode!(1, 2, 3), UniNode::from(vec![1, 2, 3]));
    assert_eq!(
        unode! {
            "one" => 1,
            "two" => 2
        },
        {
            let mut hm = HashMap::new();
            hm.insert("one".to_string(), UniNode::Integer(1));
            hm.insert("two".to_string(), UniNode::Integer(2));
            UniNode::Object(hm)
        }
    );
}

impl Default for UniNode {
    fn default() -> Self {
        UniNode::Null
    }
}

#[test]
fn value_default_test() {
    assert_eq!(UniNode::default(), UniNode::Null);
}

#[test]
fn value_as_boolean_test() {
    fn bool_test(val: UniNode, s: bool) {
        assert_eq!(val.as_bool().unwrap(), s);
    }
    bool_test(unode!(true), true);
    bool_test(unode!(false), false);
    bool_test(unode!(1), true);
    bool_test(unode!(0), false);
    bool_test(unode!(1u8), true);
    bool_test(unode!(1.1), true);
    bool_test(unode!(0.0), false);
    bool_test(unode!("1"), true);
    bool_test(unode!("true"), true);
    bool_test(unode!("on"), true);
    bool_test(unode!("yes"), true);
    bool_test(unode!("0"), false);
    bool_test(unode!("false"), false);
    bool_test(unode!("off"), false);
    bool_test(unode!("no"), false);

    assert_eq!(
        format!("{}", unode!().as_bool().unwrap_err()),
        "invalid type: null value, expected a boolean"
    );
    assert_eq!(
        format!("{}", unode!("maybe").as_bool().unwrap_err()),
        "invalid type: string \"maybe\", expected a boolean"
    );
    assert_eq!(
        format!("{}", unode!(1, 2).as_bool().unwrap_err()),
        "invalid type: array, expected a boolean"
    );
    assert_eq!(
        format!("{}", unode!("one" => 1).as_bool().unwrap_err()),
        "invalid type: object, expected a boolean"
    );
}

#[test]
fn value_as_integer_test() {
    fn integer_test(value: UniNode, val: i64) {
        assert_eq!(value.as_int().unwrap(), val);
    }
    integer_test(unode!(true), 1);
    integer_test(unode!(false), 0);
    integer_test(unode!(4), 4);
    integer_test(unode!(5u8), 5);
    assert_eq!(
        format!("{}", unode!(u64::MAX).as_int().unwrap_err()),
        "invalid type: unsigned integer `18446744073709551615`, expected a \
         integer"
    );
    integer_test(unode!(1.1), 1);
    integer_test(unode!(1.6), 2);
    integer_test(unode!(-1.6), -2);
    integer_test(unode!("1"), 1);
    integer_test(unode!("true"), 1);
    integer_test(unode!("on"), 1);
    integer_test(unode!("yes"), 1);
    integer_test(unode!("0"), 0);
    integer_test(unode!("false"), 0);
    integer_test(unode!("off"), 0);
    integer_test(unode!("no"), 0);
    assert_eq!(
        format!("{}", unode!().as_int().unwrap_err()),
        "invalid type: null value, expected a integer"
    );
    assert_eq!(
        format!("{}", unode!("one").as_int().unwrap_err()),
        "invalid type: string \"one\", expected a integer"
    );
    assert_eq!(
        format!("{}", unode!(1, 2).as_int().unwrap_err()),
        "invalid type: array, expected a integer"
    );
    assert_eq!(
        format!("{}", unode!("one" => 1).as_int().unwrap_err()),
        "invalid type: object, expected a integer"
    );
}

#[test]
fn value_as_uinteger_test() {
    fn integer_test(value: UniNode, val: u64) {
        assert_eq!(value.as_uint().unwrap(), val);
    }
    integer_test(unode!(true), 1);
    integer_test(unode!(false), 0);
    integer_test(unode!(4i32), 4);
    integer_test(unode!(5u8), 5);
    assert_eq!(
        format!("{}", unode!(i64::MIN).as_uint().unwrap_err()),
        "invalid type: integer `-9223372036854775808`, expected a unsigned \
         integer"
    );
    integer_test(unode!(1.1), 1);
    assert_eq!(
        format!("{}", unode!(-2.6).as_uint().unwrap_err()),
        "invalid type: floating point `-2.6`, expected a unsigned integer"
    );

    integer_test(unode!("1"), 1);
    integer_test(unode!("true"), 1);
    integer_test(unode!("on"), 1);
    integer_test(unode!("yes"), 1);
    integer_test(unode!("0"), 0);
    integer_test(unode!("false"), 0);
    integer_test(unode!("off"), 0);
    integer_test(unode!("no"), 0);

    assert_eq!(
        format!("{}", unode!().as_uint().unwrap_err()),
        "invalid type: null value, expected a unsigned integer"
    );
    assert_eq!(
        format!("{}", unode!("one").as_uint().unwrap_err()),
        "invalid type: string \"one\", expected a unsigned integer"
    );
    assert_eq!(
        format!("{}", unode!(1, 2).as_uint().unwrap_err()),
        "invalid type: array, expected a unsigned integer"
    );
    assert_eq!(
        format!("{}", unode!("one" => 1).as_uint().unwrap_err()),
        "invalid type: object, expected a unsigned integer"
    );
}

#[test]
fn value_as_float_test() {
    fn float_test(value: UniNode, val: f64) {
        assert!((value.as_float().unwrap() - val).abs() < f64::EPSILON);
    }
    float_test(unode!(true), 1.0);
    float_test(unode!(false), 0.0);
    float_test(unode!(4i32), 4.0);
    float_test(unode!(5u8), 5.0);
    float_test(unode!(i64::MIN), -9223372036854776000.0);
    float_test(unode!(i64::MAX), 9223372036854776000.0);
    float_test(unode!(u64::MAX), 18446744073709552000.0);
    float_test(unode!(1.1), 1.1);
    float_test(unode!("1"), 1.0);
    float_test(unode!("true"), 1.0);
    float_test(unode!("on"), 1.0);
    float_test(unode!("yes"), 1.0);
    float_test(unode!("0"), 0.0);
    float_test(unode!("false"), 0.0);
    float_test(unode!("off"), 0.0);
    float_test(unode!("no"), 0.0);
    assert_eq!(
        format!("{}", unode!().as_float().unwrap_err()),
        "invalid type: null value, expected a floating point"
    );
    assert_eq!(
        format!("{}", unode!("one").as_float().unwrap_err()),
        "invalid type: string \"one\", expected a floating point"
    );
    assert_eq!(
        format!("{}", unode!(1, 2).as_float().unwrap_err()),
        "invalid type: array, expected a floating point"
    );
    assert_eq!(
        format!("{}", unode!("one" => 1).as_float().unwrap_err()),
        "invalid type: object, expected a floating point"
    );
}

#[test]
fn value_as_array_test() {
    assert_eq!(unode!("one", 1).as_array().unwrap(), &vec![
        unode!("one"),
        unode!(1)
    ]);

    let mut arr = unode!("one", 1);
    arr.as_array_mut().unwrap().push(unode!("two"));
    assert_eq!(arr.as_array().unwrap(), &vec![
        unode!("one"),
        unode!(1),
        unode!("two"),
    ]);
}

#[test]
fn value_as_object_test() {
    assert_eq!(unode!("one" => 1).as_object().unwrap(), &{
        let mut hm = HashMap::new();
        hm.insert("one".to_string(), UniNode::Integer(1));
        hm
    });

    let mut tbl = unode!("one" => 1);
    tbl.as_object_mut()
        .unwrap()
        .insert("two".to_string(), UniNode::Integer(2));
    assert_eq!(tbl.as_object().unwrap(), &{
        let mut hm = HashMap::new();
        hm.insert("one".to_string(), UniNode::Integer(1));
        hm.insert("two".to_string(), UniNode::Integer(2));
        hm
    });
}

#[test]
fn value_find_test() {
    let mut val = unode!(
        "server" => unode!("host" => "localhost", "port" => 999),
        "two" => 2);

    assert!(val.find("tw").is_none());
    assert!(val.find("server.url").is_none());
    assert!(val.find("server.").is_some());
    assert!(val.find("server.").unwrap().is_object());
    assert!(val.find("server.host").is_some());
    assert!(val.find("server.host").unwrap().is_string());
    assert_eq!(
        val.find("server.host").unwrap().as_str().unwrap(),
        "localhost"
    );

    assert!(val.find("server.timeout").is_none());
    let n = val.find_mut("server").unwrap().as_object_mut().unwrap();
    n.insert("timeout".to_string(), UniNode::Integer(10));

    assert!(val.find("server.timeout").is_some());
    assert_eq!(
        val.find("server.timeout")
            .and_then(|v| v.as_int().ok())
            .unwrap(),
        10
    );
}

#[test]
fn value_get_test() {
    let val = unode!(
        "server" => unode!(
            "host" => "localhost",
            "port" => 999,
            "keys" => unode!(1, 2, 4, 7),
        ),
        "bool" => true,
        "int" => 33,
        "uint" => 12u32,
        "float" => 1.2,
        "str" => "hello",
        "bytes" => vec![1u8,2u8,3u8],
        "opt" => Some("opt"),
        "none" => Option::<u8>::None,
        "arr" => unode!["one", 2, 3.3],
    );

    assert!(val.get_bool("bool").unwrap());
    assert_eq!(val.get_int("int").unwrap(), 33);
    assert_eq!(val.get_uint("uint").unwrap(), 12);
    assert!((val.get_float("float").unwrap() - 1.2).abs() < f64::EPSILON);
    assert_eq!(val.get_str("str").unwrap(), "hello");
    assert_eq!(val.get_bytes("bytes").unwrap(), vec![1, 2, 3]);
    assert_eq!(val.get_str("opt").unwrap(), "opt");
    assert!(val.find("none").unwrap().is_null());

    let keys = val
        .get_array("server.keys")
        .unwrap()
        .iter()
        .map(|v| v.as_int().unwrap())
        .collect::<Vec<_>>();
    assert_eq!(keys, vec![1, 2, 4, 7]);
}

impl Display for UniNode {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            UniNode::Null => write!(f, "null"),
            UniNode::Boolean(val) => write!(f, "{}", val),
            UniNode::Integer(val) => write!(f, "{}", val),
            UniNode::UInteger(val) => write!(f, "{}", val),
            UniNode::Float(val) => write!(f, "{}", val),
            UniNode::String(val) => write!(f, "{}", val),
            UniNode::Bytes(val) => write!(f, "{:?}", val),
            UniNode::Array(val) => {
                write!(f, "[")?;
                for (i, item) in val.iter().enumerate() {
                    if i > 0 {
                        write!(f, ", ")?;
                    }
                    write!(f, "{}", item)?;
                }
                write!(f, "]")
            },
            UniNode::Object(val) => {
                write!(f, "{{")?;
                for (i, item) in val.iter().enumerate() {
                    if i > 0 {
                        write!(f, ", ")?;
                    }
                    write!(f, "{}: {}", item.0, item.1)?;
                }
                write!(f, "}}")
            },
        }
    }
}

#[test]
fn value_display_test() {
    assert_eq!(format!("{}", UniNode::Null), "null");
    assert_eq!(format!("{}", UniNode::Boolean(true)), "true");
    assert_eq!(format!("{}", UniNode::Integer(23)), "23");
    assert_eq!(format!("{}", UniNode::UInteger(55)), "55");
    assert_eq!(format!("{}", UniNode::Float(5.6)), "5.6");
    assert_eq!(format!("{}", UniNode::String("hello".to_string())), "hello");
    let conf = unode!(false, UniNode::Array(vec![unode!(3.3)]));
    assert_eq!(format!("{}", conf), "[false, [3.3]]");
    let conf = unode! { "one" => true };
    assert_eq!(format!("{}", conf), "{one: true}");
}
