// error.rs
// Copyright: (c) 2015-2021, Prelum Project.
// Author: <o.lelenkov@gmail.com> Oleg Lelenkov
// Distributed under terms of the Subject to the terms of the BSD 3-Clause License, as written in the included LICENSE.md file. license.
//

use std::{fmt, result, error::Error};
use crate::value::UniNode;
use serde::{ser, de};
use serde::de::Unexpected;

#[derive(Debug)]
pub enum UniNodeSerError {
    KeyMustBeAString,
    Custom(String),
}

pub type Result<T> = result::Result<T, UniNodeSerError>;

impl fmt::Display for UniNodeSerError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            UniNodeSerError::KeyMustBeAString => {
                f.write_str("key must be a string")
            },
            UniNodeSerError::Custom(msg) => f.write_str(msg),
        }
    }
}

impl Error for UniNodeSerError {}

impl ser::Error for UniNodeSerError {
    fn custom<T: fmt::Display>(msg: T) -> Self {
        UniNodeSerError::Custom(msg.to_string())
    }
}

impl de::Error for UniNodeSerError {
    fn custom<T: fmt::Display>(msg: T) -> Self {
        UniNodeSerError::Custom(msg.to_string())
    }
}

impl UniNode {
    pub fn invalid_type(&self, exp: &dyn de::Expected) -> UniNodeSerError {
        <UniNodeSerError as serde::de::Error>::invalid_type(self.into(), exp)
    }
}

impl<'a> From<&'a UniNode> for Unexpected<'a> {
    fn from(val: &'a UniNode) -> Unexpected<'a> {
        match val {
            UniNode::Null => Unexpected::Unit,
            UniNode::Boolean(val) => Unexpected::Bool(*val),
            UniNode::Integer(val) => Unexpected::Signed(*val),
            UniNode::UInteger(val) => Unexpected::Unsigned(*val),
            UniNode::Float(val) => Unexpected::Float(*val),
            UniNode::String(val) => Unexpected::Str(val),
            UniNode::Bytes(val) => {
                Unexpected::Bytes(&val[0..val.len().min(10)])
            },
            UniNode::Array(_) => Unexpected::Seq,
            UniNode::Object(_) => Unexpected::Map,
        }
    }
}
