// deserialize.rs
// Copyright (C) 2021 Oleg Lelenkov <o.lelenkov@gmail.com>
// Distributed under terms of the Subject to the terms of the BSD 3-Clause License, as written in the included LICENSE.md file. license.
//

use serde::de::{self, DeserializeSeed, Visitor};
use std::{fmt, result, slice};
use std::borrow::Cow;

use serde::de::IntoDeserializer;
use crate::value::{UniNode, Object, Array};
use super::{UniNodeSerError, Result};

#[cfg(test)]
mod tests {
    use super::*;

    pub fn deserialize_test<T>(node: UniNode, val: T)
    where
        T: de::DeserializeOwned + fmt::Debug + PartialEq,
    {
        assert_eq!(T::deserialize(&node).unwrap(), val);
        assert_eq!(T::deserialize(node).unwrap(), val);
    }
}

#[cfg(test)]
use {
    serde::Deserialize, crate::unode, serde_bytes::ByteBuf,
    std::collections::HashMap,
};

impl<'de> de::Deserialize<'de> for UniNode {
    fn deserialize<D>(deserializer: D) -> result::Result<UniNode, D::Error>
    where
        D: de::Deserializer<'de>,
    {
        struct UniNodeVisitor;

        type VResult<E> = result::Result<UniNode, E>;

        impl<'de> de::Visitor<'de> for UniNodeVisitor {
            type Value = UniNode;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("any valid UniNode value")
            }

            fn visit_bool<E>(self, v: bool) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_i8<E>(self, v: i8) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_i16<E>(self, v: i16) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_i32<E>(self, v: i32) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_i64<E>(self, v: i64) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_u8<E>(self, v: u8) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_u16<E>(self, v: u16) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_u32<E>(self, v: u32) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_u64<E>(self, v: u64) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_f32<E>(self, v: f32) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_f64<E>(self, v: f64) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_str<E>(self, v: &str) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_string<E>(self, v: String) -> VResult<E> {
                Ok(UniNode::from(v))
            }

            fn visit_none<E>(self) -> VResult<E> {
                Ok(UniNode::Null)
            }

            fn visit_unit<E>(self) -> VResult<E> {
                Ok(UniNode::Null)
            }

            fn visit_some<D>(self, deserializer: D) -> VResult<D::Error>
            where
                D: de::Deserializer<'de>,
            {
                de::Deserialize::deserialize(deserializer)
            }

            fn visit_seq<A>(self, mut seq: A) -> VResult<A::Error>
            where
                A: de::SeqAccess<'de>,
            {
                let mut vec =
                    Array::with_capacity(seq.size_hint().unwrap_or(0));
                while let Some(el) = seq.next_element()? {
                    vec.push(el);
                }
                Ok(UniNode::Array(vec))
            }

            fn visit_map<A>(self, mut map: A) -> VResult<A::Error>
            where
                A: de::MapAccess<'de>,
            {
                let mut obj =
                    Object::with_capacity(map.size_hint().unwrap_or(0));
                while let Some((key, val)) = map.next_entry()? {
                    obj.insert(key, val);
                }
                Ok(UniNode::Object(obj))
            }
        }

        deserializer.deserialize_any(UniNodeVisitor)
    }
}

#[test]
fn deserialize_trait_test() {
    let v = UniNode::deserialize(unode!(1)).unwrap();
    assert_eq!(v, unode!(1));
}

fn visit_array<'de, V>(array: Array, visitor: V) -> Result<V::Value>
where
    V: Visitor<'de>,
{
    let len = array.len();
    let mut deserializer = SeqDeserializer::new(array);
    let seq = visitor.visit_seq(&mut deserializer)?;
    let remaining = deserializer.iter.len();
    if remaining == 0 {
        Ok(seq)
    } else {
        Err(de::Error::invalid_length(len, &"fewer elements in array"))
    }
}

fn visit_object<'de, V>(object: Object, visitor: V) -> Result<V::Value>
where
    V: Visitor<'de>,
{
    let len = object.len();
    let mut deserializer = MapDeserializer::new(object);
    let map = visitor.visit_map(&mut deserializer)?;
    let remaining = deserializer.iter.len();
    if remaining == 0 {
        Ok(map)
    } else {
        Err(de::Error::invalid_length(len, &"fewer elements in map"))
    }
}

impl<'de> de::Deserializer<'de> for UniNode {
    type Error = UniNodeSerError;

    serde::forward_to_deserialize_any! {
        i8 i16 i32 i64 u8 u16 u32 u64 f32 f64 bool
    }

    fn deserialize_any<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Null => visitor.visit_unit(),
            UniNode::Boolean(v) => visitor.visit_bool(v),
            UniNode::Integer(v) => visitor.visit_i64(v),
            UniNode::UInteger(v) => visitor.visit_u64(v),
            UniNode::Float(v) => visitor.visit_f64(v),
            UniNode::String(v) => visitor.visit_string(v),
            UniNode::Bytes(v) => visitor.visit_byte_buf(v),
            UniNode::Array(v) => visit_array(v, visitor),
            UniNode::Object(v) => visit_object(v, visitor),
        }
    }

    fn deserialize_char<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_string(visitor)
    }

    fn deserialize_str<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_string(visitor)
    }

    fn deserialize_string<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::String(v) => visitor.visit_string(v),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_bytes<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_byte_buf(visitor)
    }

    fn deserialize_byte_buf<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::String(v) => visitor.visit_string(v),
            UniNode::Array(v) => visit_array(v, visitor),
            UniNode::Bytes(v) => visitor.visit_byte_buf(v),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_option<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Null => visitor.visit_none(),
            _ => visitor.visit_some(self),
        }
    }

    fn deserialize_unit<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Null => visitor.visit_unit(),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_unit_struct<V>(
        self, _name: &'static str, visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_unit(visitor)
    }

    fn deserialize_seq<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Array(v) => visit_array(v, visitor),
            UniNode::Bytes(v) => {
                let mut deserializer =
                    de::value::SeqDeserializer::new(v.into_iter());
                let seq = visitor.visit_seq(&mut deserializer)?;
                deserializer.end()?;
                Ok(seq)
            },
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_tuple<V>(self, _len: usize, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_seq(visitor)
    }

    fn deserialize_tuple_struct<V>(
        self, _name: &'static str, _len: usize, visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_seq(visitor)
    }

    fn deserialize_map<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Object(v) => visit_object(v, visitor),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_struct<V>(
        self, _name: &'static str, _fields: &'static [&'static str], visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Array(v) => visit_array(v, visitor),
            UniNode::Object(v) => visit_object(v, visitor),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_newtype_struct<V>(
        self, _name: &'static str, visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        visitor.visit_newtype_struct(self)
    }

    fn deserialize_identifier<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_string(visitor)
    }

    fn deserialize_ignored_any<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        drop(self);
        visitor.visit_unit()
    }

    fn deserialize_enum<V>(
        self, _name: &str, _variants: &'static [&'static str], visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let (variant, value) = match self {
            UniNode::Object(value) => {
                let mut iter = value.into_iter();
                let (variant, value) = match iter.next() {
                    Some(v) => v,
                    None => {
                        return Err(de::Error::invalid_value(
                            de::Unexpected::Map,
                            &"map with a single key",
                        ));
                    },
                };
                if iter.next().is_some() {
                    return Err(de::Error::invalid_value(
                        de::Unexpected::Map,
                        &"map with a single key",
                    ));
                }
                (variant, Some(value))
            },
            UniNode::String(variant) => (variant, None),
            other => return Err(other.invalid_type(&"string or map")),
        };

        visitor.visit_enum(EnumDeserializer { variant, value })
    }
}

struct SeqDeserializer {
    iter: std::vec::IntoIter<UniNode>,
}

impl SeqDeserializer {
    fn new(vec: Vec<UniNode>) -> Self {
        SeqDeserializer {
            iter: vec.into_iter(),
        }
    }
}

impl<'de> de::Deserializer<'de> for SeqDeserializer {
    type Error = UniNodeSerError;

    serde::forward_to_deserialize_any! {
        bool i8 i16 i32 i64 i128 u8 u16 u32 u64 u128 f32 f64 char str string
        bytes byte_buf option unit unit_struct newtype_struct seq tuple
        tuple_struct map struct enum identifier ignored_any
    }

    fn deserialize_any<V>(mut self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let len = self.iter.len();
        if len == 0 {
            visitor.visit_unit()
        } else {
            let ret = visitor.visit_seq(&mut self)?;
            let remaining = self.iter.len();
            if remaining == 0 {
                Ok(ret)
            } else {
                Err(de::Error::invalid_length(len, &"fewer elements in array"))
            }
        }
    }
}

impl<'de> de::SeqAccess<'de> for SeqDeserializer {
    type Error = UniNodeSerError;

    fn next_element_seed<T>(&mut self, seed: T) -> Result<Option<T::Value>>
    where
        T: DeserializeSeed<'de>,
    {
        match self.iter.next() {
            Some(value) => seed.deserialize(value).map(Some),
            None => Ok(None),
        }
    }

    fn size_hint(&self) -> Option<usize> {
        match self.iter.size_hint() {
            (lower, Some(upper)) if lower == upper => Some(upper),
            _ => None,
        }
    }
}

struct MapDeserializer {
    iter: <Object as IntoIterator>::IntoIter,
    value: Option<UniNode>,
}

impl MapDeserializer {
    fn new(map: Object) -> Self {
        MapDeserializer {
            iter: map.into_iter(),
            value: None,
        }
    }
}

impl<'de> de::Deserializer<'de> for MapDeserializer {
    type Error = UniNodeSerError;

    serde::forward_to_deserialize_any! {
        bool i8 i16 i32 i64 i128 u8 u16 u32 u64 u128 f32 f64 char str string
        bytes byte_buf option unit unit_struct newtype_struct seq tuple
        tuple_struct map struct enum identifier ignored_any
    }

    #[inline]
    fn deserialize_any<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        visitor.visit_map(self)
    }
}

impl<'de> de::MapAccess<'de> for MapDeserializer {
    type Error = UniNodeSerError;

    fn next_key_seed<T>(&mut self, seed: T) -> Result<Option<T::Value>>
    where
        T: de::DeserializeSeed<'de>,
    {
        match self.iter.next() {
            Some((key, value)) => {
                self.value = Some(value);
                let key_de = MapKeyDeserializer {
                    key: Cow::Owned(key),
                };
                seed.deserialize(key_de).map(Some)
            },
            None => Ok(None),
        }
    }

    fn next_value_seed<T>(&mut self, seed: T) -> Result<T::Value>
    where
        T: de::DeserializeSeed<'de>,
    {
        match self.value.take() {
            Some(value) => seed.deserialize(value),
            None => Err(de::Error::custom("value is missing")),
        }
    }

    fn size_hint(&self) -> Option<usize> {
        match self.iter.size_hint() {
            (lower, Some(upper)) if lower == upper => Some(upper),
            _ => None,
        }
    }
}

struct EnumDeserializer {
    variant: String,
    value: Option<UniNode>,
}

impl<'de> de::EnumAccess<'de> for EnumDeserializer {
    type Error = UniNodeSerError;
    type Variant = VariantDeserializer;

    fn variant_seed<V>(self, seed: V) -> Result<(V::Value, VariantDeserializer)>
    where
        V: DeserializeSeed<'de>,
    {
        let variant = self.variant.into_deserializer();
        let visitor = VariantDeserializer { value: self.value };
        seed.deserialize(variant).map(|v| (v, visitor))
    }
}

struct VariantDeserializer {
    value: Option<UniNode>,
}

impl<'de> de::VariantAccess<'de> for VariantDeserializer {
    type Error = UniNodeSerError;

    fn unit_variant(self) -> Result<()> {
        match self.value {
            Some(value) => de::Deserialize::deserialize(value),
            None => Ok(()),
        }
    }

    fn newtype_variant_seed<T>(self, seed: T) -> Result<T::Value>
    where
        T: DeserializeSeed<'de>,
    {
        match self.value {
            Some(value) => seed.deserialize(value),
            None => {
                Err(de::Error::invalid_type(
                    de::Unexpected::UnitVariant,
                    &"newtype variant",
                ))
            },
        }
    }

    fn tuple_variant<V>(self, _len: usize, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self.value {
            Some(UniNode::Array(v)) => {
                serde::Deserializer::deserialize_any(
                    SeqDeserializer::new(v),
                    visitor,
                )
            },
            Some(other) => Err(other.invalid_type(&"tuple variant")),
            None => {
                Err(de::Error::invalid_type(
                    de::Unexpected::UnitVariant,
                    &"tuple variant",
                ))
            },
        }
    }

    fn struct_variant<V>(
        self, _fields: &'static [&'static str], visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self.value {
            Some(UniNode::Object(v)) => {
                serde::Deserializer::deserialize_any(
                    MapDeserializer::new(v),
                    visitor,
                )
            },
            Some(other) => Err(other.invalid_type(&"struct variant")),
            None => {
                Err(de::Error::invalid_type(
                    de::Unexpected::UnitVariant,
                    &"struct variant",
                ))
            },
        }
    }
}

fn visit_array_ref<'de, V>(arr: &'de [UniNode], visitor: V) -> Result<V::Value>
where
    V: de::Visitor<'de>,
{
    let len = arr.len();
    let mut deserializer = SeqRefDeserializer::new(arr);
    let seq = visitor.visit_seq(&mut deserializer)?;
    let remaining = deserializer.iter.len();
    if remaining == 0 {
        Ok(seq)
    } else {
        Err(de::Error::invalid_length(len, &"fewer elements in array"))
    }
}

fn visit_object_ref<'de, V>(object: &'de Object, visitor: V) -> Result<V::Value>
where
    V: de::Visitor<'de>,
{
    let len = object.len();
    let mut deserializer = MapRefDeserializer::new(object);
    let map = visitor.visit_map(&mut deserializer)?;
    let remaining = deserializer.iter.len();
    if remaining == 0 {
        Ok(map)
    } else {
        Err(de::Error::invalid_length(len, &"fewer elements in map"))
    }
}

impl<'de> de::Deserializer<'de> for &'de UniNode {
    type Error = UniNodeSerError;

    serde::forward_to_deserialize_any! {
        i8 i16 i32 i64 u8 u16 u32 u64 f32 f64 bool
    }

    fn deserialize_any<V>(self, visitor: V) -> Result<V::Value>
    where
        V: de::Visitor<'de>,
    {
        match self {
            UniNode::Null => visitor.visit_unit(),
            UniNode::Boolean(v) => visitor.visit_bool(*v),
            UniNode::Integer(v) => visitor.visit_i64(*v),
            UniNode::UInteger(v) => visitor.visit_u64(*v),
            UniNode::Float(v) => visitor.visit_f64(*v),
            UniNode::String(v) => visitor.visit_borrowed_str(v),
            UniNode::Bytes(v) => visitor.visit_bytes(v),
            UniNode::Array(v) => visit_array_ref(v, visitor),
            UniNode::Object(v) => visit_object_ref(v, visitor),
        }
    }

    fn deserialize_char<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_str(visitor)
    }

    fn deserialize_str<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::String(v) => visitor.visit_borrowed_str(v),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_string<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_str(visitor)
    }

    fn deserialize_bytes<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::String(v) => visitor.visit_borrowed_str(v),
            UniNode::Array(v) => visit_array_ref(v, visitor),
            UniNode::Bytes(v) => visitor.visit_borrowed_bytes(v),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_byte_buf<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_bytes(visitor)
    }

    fn deserialize_option<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Null => visitor.visit_none(),
            _ => visitor.visit_some(self),
        }
    }

    fn deserialize_unit<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Null => visitor.visit_unit(),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_unit_struct<V>(
        self, _name: &'static str, visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_unit(visitor)
    }

    fn deserialize_seq<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Array(v) => visit_array_ref(v, visitor),
            UniNode::Bytes(v) => {
                let len = v.len();
                let mut deserializer = BytesAccessor::new(v);
                let seq = visitor.visit_seq(&mut deserializer)?;
                let remaining = deserializer.iter.len();
                if remaining == 0 {
                    Ok(seq)
                } else {
                    Err(de::Error::invalid_length(
                        len,
                        &"fewer elements in array",
                    ))
                }
            },
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_tuple<V>(self, _len: usize, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_seq(visitor)
    }

    fn deserialize_tuple_struct<V>(
        self, _name: &'static str, _len: usize, visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_seq(visitor)
    }

    fn deserialize_map<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Object(v) => visit_object_ref(v, visitor),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_struct<V>(
        self, _name: &'static str, _fields: &'static [&'static str], visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self {
            UniNode::Array(v) => visit_array_ref(v, visitor),
            UniNode::Object(v) => visit_object_ref(v, visitor),
            _ => Err(self.invalid_type(&visitor)),
        }
    }

    fn deserialize_newtype_struct<V>(
        self, _name: &'static str, visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        visitor.visit_newtype_struct(self)
    }

    fn deserialize_identifier<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_str(visitor)
    }

    fn deserialize_ignored_any<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        visitor.visit_unit()
    }

    fn deserialize_enum<V>(
        self, _name: &str, _variants: &'static [&'static str], visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let (variant, value) = match self {
            UniNode::Object(value) => {
                let mut iter = value.iter();
                let (variant, value) = match iter.next() {
                    Some(v) => v,
                    None => {
                        return Err(de::Error::invalid_value(
                            de::Unexpected::Map,
                            &"map with a single key",
                        ));
                    },
                };
                // enums are encoded in json as maps with a single key:value pair
                if iter.next().is_some() {
                    return Err(de::Error::invalid_value(
                        de::Unexpected::Map,
                        &"map with a single key",
                    ));
                }
                (variant, Some(value))
            },
            UniNode::String(variant) => (variant, None),
            other => return Err(other.invalid_type(&"string or map")),
        };

        visitor.visit_enum(EnumRefDeserializer { variant, value })
    }
}

struct SeqRefDeserializer<'de> {
    iter: slice::Iter<'de, UniNode>,
}

impl<'de> SeqRefDeserializer<'de> {
    fn new(slice: &'de [UniNode]) -> Self {
        SeqRefDeserializer { iter: slice.iter() }
    }
}

impl<'de> de::SeqAccess<'de> for SeqRefDeserializer<'de> {
    type Error = UniNodeSerError;

    fn next_element_seed<T>(&mut self, seed: T) -> Result<Option<T::Value>>
    where
        T: DeserializeSeed<'de>,
    {
        match self.iter.next() {
            Some(value) => seed.deserialize(value).map(Some),
            None => Ok(None),
        }
    }

    fn size_hint(&self) -> Option<usize> {
        match self.iter.size_hint() {
            (lower, Some(upper)) if lower == upper => Some(upper),
            _ => None,
        }
    }
}

struct BytesAccessor<'de> {
    iter: slice::Iter<'de, u8>,
}

impl<'de> BytesAccessor<'de> {
    fn new(slice: &'de [u8]) -> Self {
        Self { iter: slice.iter() }
    }
}

impl<'de> de::SeqAccess<'de> for BytesAccessor<'de> {
    type Error = UniNodeSerError;

    fn next_element_seed<T>(&mut self, seed: T) -> Result<Option<T::Value>>
    where
        T: DeserializeSeed<'de>,
    {
        match self.iter.next() {
            Some(value) => {
                let deserializer = (*value).into_deserializer();
                seed.deserialize(deserializer).map(Some)
            },
            _ => Ok(None),
        }
    }

    fn size_hint(&self) -> Option<usize> {
        match self.iter.size_hint() {
            (lower, Some(upper)) if lower == upper => Some(upper),
            _ => None,
        }
    }
}

struct MapRefDeserializer<'de> {
    iter: <&'de Object as IntoIterator>::IntoIter,
    value: Option<&'de UniNode>,
}

impl<'de> MapRefDeserializer<'de> {
    fn new(map: &'de Object) -> Self {
        MapRefDeserializer {
            iter: map.iter(),
            value: None,
        }
    }
}

impl<'de> de::Deserializer<'de> for MapRefDeserializer<'de> {
    type Error = UniNodeSerError;

    serde::forward_to_deserialize_any! {
        bool i8 i16 i32 i64 i128 u8 u16 u32 u64 u128 f32 f64 char str string
        bytes byte_buf option unit unit_struct newtype_struct seq tuple
        tuple_struct map struct enum identifier ignored_any
    }

    fn deserialize_any<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        visitor.visit_map(self)
    }
}

impl<'de> de::MapAccess<'de> for MapRefDeserializer<'de> {
    type Error = UniNodeSerError;

    fn next_key_seed<T>(&mut self, seed: T) -> Result<Option<T::Value>>
    where
        T: DeserializeSeed<'de>,
    {
        match self.iter.next() {
            Some((key, value)) => {
                self.value = Some(value);
                let key_de = MapKeyDeserializer {
                    key: Cow::Borrowed(&**key),
                };
                seed.deserialize(key_de).map(Some)
            },
            None => Ok(None),
        }
    }

    fn next_value_seed<T>(&mut self, seed: T) -> Result<T::Value>
    where
        T: DeserializeSeed<'de>,
    {
        match self.value.take() {
            Some(value) => seed.deserialize(value),
            None => Err(de::Error::custom("value is missing")),
        }
    }

    fn size_hint(&self) -> Option<usize> {
        match self.iter.size_hint() {
            (lower, Some(upper)) if lower == upper => Some(upper),
            _ => None,
        }
    }
}

impl<'de> de::Deserializer<'de> for SeqRefDeserializer<'de> {
    type Error = UniNodeSerError;

    serde::forward_to_deserialize_any! {
        bool i8 i16 i32 i64 i128 u8 u16 u32 u64 u128 f32 f64 char str string
        bytes byte_buf option unit unit_struct newtype_struct seq tuple
        tuple_struct map struct enum identifier ignored_any
    }

    fn deserialize_any<V>(mut self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let len = self.iter.len();
        if len == 0 {
            visitor.visit_unit()
        } else {
            let ret = visitor.visit_seq(&mut self)?;
            let remaining = self.iter.len();
            if remaining == 0 {
                Ok(ret)
            } else {
                Err(de::Error::invalid_length(len, &"fewer elements in array"))
            }
        }
    }
}

struct EnumRefDeserializer<'de> {
    variant: &'de str,
    value: Option<&'de UniNode>,
}

impl<'de> de::EnumAccess<'de> for EnumRefDeserializer<'de> {
    type Error = UniNodeSerError;
    type Variant = VariantRefDeserializer<'de>;

    fn variant_seed<V>(self, seed: V) -> Result<(V::Value, Self::Variant)>
    where
        V: DeserializeSeed<'de>,
    {
        let variant = self.variant.into_deserializer();
        let visitor = VariantRefDeserializer { value: self.value };
        seed.deserialize(variant).map(|v| (v, visitor))
    }
}

struct VariantRefDeserializer<'de> {
    value: Option<&'de UniNode>,
}

impl<'de> de::VariantAccess<'de> for VariantRefDeserializer<'de> {
    type Error = UniNodeSerError;

    fn unit_variant(self) -> Result<()> {
        match self.value {
            Some(value) => de::Deserialize::deserialize(value),
            None => Ok(()),
        }
    }

    fn newtype_variant_seed<T>(self, seed: T) -> Result<T::Value>
    where
        T: DeserializeSeed<'de>,
    {
        match self.value {
            Some(value) => seed.deserialize(value),
            None => {
                Err(de::Error::invalid_type(
                    de::Unexpected::UnitVariant,
                    &"newtype variant",
                ))
            },
        }
    }

    fn tuple_variant<V>(self, _len: usize, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self.value {
            Some(&UniNode::Array(ref v)) => {
                de::Deserializer::deserialize_any(
                    SeqRefDeserializer::new(v),
                    visitor,
                )
            },
            Some(other) => Err(other.invalid_type(&"tuple variant")),
            None => {
                Err(de::Error::invalid_type(
                    de::Unexpected::UnitVariant,
                    &"tuple variant",
                ))
            },
        }
    }

    fn struct_variant<V>(
        self, _fields: &'static [&'static str], visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        match self.value {
            Some(&UniNode::Object(ref v)) => {
                serde::Deserializer::deserialize_any(
                    MapRefDeserializer::new(v),
                    visitor,
                )
            },
            Some(other) => Err(other.invalid_type(&"struct variant")),
            None => {
                Err(de::Error::invalid_type(
                    de::Unexpected::UnitVariant,
                    &"struct variant",
                ))
            },
        }
    }
}

struct MapKeyDeserializer<'de> {
    key: Cow<'de, str>,
}

macro_rules! deserialize_integer_key {
    ($method:ident => $visit:ident) => {
        fn $method<V>(self, visitor: V) -> Result<V::Value>
        where
            V: Visitor<'de>,
        {
            match (self.key.parse(), self.key) {
                (Ok(integer), _) => visitor.$visit(integer),
                (Err(_), Cow::Borrowed(s)) => visitor.visit_borrowed_str(s),
                (Err(_), Cow::Owned(s)) => visitor.visit_string(s),
            }
        }
    };
}

impl<'de> de::Deserializer<'de> for MapKeyDeserializer<'de> {
    type Error = UniNodeSerError;

    deserialize_integer_key!(deserialize_i8 => visit_i8);

    deserialize_integer_key!(deserialize_i16 => visit_i16);

    deserialize_integer_key!(deserialize_i32 => visit_i32);

    deserialize_integer_key!(deserialize_i64 => visit_i64);

    deserialize_integer_key!(deserialize_u8 => visit_u8);

    deserialize_integer_key!(deserialize_u16 => visit_u16);

    deserialize_integer_key!(deserialize_u32 => visit_u32);

    deserialize_integer_key!(deserialize_u64 => visit_u64);

    serde::forward_to_deserialize_any! {
        bool f32 f64 char str string bytes byte_buf unit unit_struct seq tuple
        tuple_struct map struct identifier ignored_any
    }

    fn deserialize_any<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        BorrowedCowStrDeserializer::new(self.key).deserialize_any(visitor)
    }

    fn deserialize_option<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        // Map keys cannot be null.
        visitor.visit_some(self)
    }

    fn deserialize_newtype_struct<V>(
        self, _name: &'static str, visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        visitor.visit_newtype_struct(self)
    }

    fn deserialize_enum<V>(
        self, name: &'static str, variants: &'static [&'static str], visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.key
            .into_deserializer()
            .deserialize_enum(name, variants, visitor)
    }
}

struct BorrowedCowStrDeserializer<'de> {
    value: Cow<'de, str>,
}

impl<'de> BorrowedCowStrDeserializer<'de> {
    fn new(value: Cow<'de, str>) -> Self {
        BorrowedCowStrDeserializer { value }
    }
}

impl<'de> de::Deserializer<'de> for BorrowedCowStrDeserializer<'de> {
    type Error = UniNodeSerError;

    serde::forward_to_deserialize_any! {
        bool i8 i16 i32 i64 i128 u8 u16 u32 u64 u128 f32 f64 char str string
        bytes byte_buf option unit unit_struct newtype_struct seq tuple
        tuple_struct map struct identifier ignored_any
    }

    fn deserialize_any<V>(self, visitor: V) -> Result<V::Value>
    where
        V: de::Visitor<'de>,
    {
        match self.value {
            Cow::Borrowed(string) => visitor.visit_borrowed_str(string),
            Cow::Owned(string) => visitor.visit_string(string),
        }
    }

    fn deserialize_enum<V>(
        self, _name: &str, _variants: &'static [&'static str], visitor: V,
    ) -> Result<V::Value>
    where
        V: de::Visitor<'de>,
    {
        visitor.visit_enum(self)
    }
}

impl<'de> de::EnumAccess<'de> for BorrowedCowStrDeserializer<'de> {
    type Error = UniNodeSerError;
    type Variant = UnitOnly;

    fn variant_seed<T>(self, seed: T) -> Result<(T::Value, Self::Variant)>
    where
        T: de::DeserializeSeed<'de>,
    {
        let value = seed.deserialize(self)?;
        Ok((value, UnitOnly))
    }
}

struct UnitOnly;

impl<'de> de::VariantAccess<'de> for UnitOnly {
    type Error = UniNodeSerError;

    fn unit_variant(self) -> Result<()> {
        Ok(())
    }

    fn newtype_variant_seed<T>(self, _seed: T) -> Result<T::Value>
    where
        T: de::DeserializeSeed<'de>,
    {
        Err(de::Error::invalid_type(
            de::Unexpected::UnitVariant,
            &"newtype variant",
        ))
    }

    fn tuple_variant<V>(self, _len: usize, _visitor: V) -> Result<V::Value>
    where
        V: de::Visitor<'de>,
    {
        Err(de::Error::invalid_type(
            de::Unexpected::UnitVariant,
            &"tuple variant",
        ))
    }

    fn struct_variant<V>(
        self, _fields: &'static [&'static str], _visitor: V,
    ) -> Result<V::Value>
    where
        V: de::Visitor<'de>,
    {
        Err(de::Error::invalid_type(
            de::Unexpected::UnitVariant,
            &"struct variant",
        ))
    }
}

#[test]
fn deserialize_primitive_test() {
    tests::deserialize_test(unode!(4u8), 4u8);
    tests::deserialize_test(unode!(4u16), 4u16);
    tests::deserialize_test(unode!(4u32), 4u32);
    tests::deserialize_test(unode!(4u64), 4u64);
    tests::deserialize_test(unode!(4i8), 4i8);
    tests::deserialize_test(unode!(4i16), 4i16);
    tests::deserialize_test(unode!(4i32), 4i32);
    tests::deserialize_test(unode!(4i64), 4i64);
    tests::deserialize_test(unode!(4.2f32), 4.2f32);
    tests::deserialize_test(unode!(4.2f64), 4.2f64);
    tests::deserialize_test(unode!(true), true);
    tests::deserialize_test(unode!("a"), 'a');
    tests::deserialize_test(unode!("hello"), "hello".to_string());
    let bytes = ByteBuf::from(vec![1u8, 2u8]);
    tests::deserialize_test::<ByteBuf>(unode!(vec![1u8, 2u8]), bytes);
    let bytes = ByteBuf::from(vec![104, 101, 108, 108, 111]);
    tests::deserialize_test::<ByteBuf>(unode!("hello"), bytes);
    let bytes = ByteBuf::from(vec![1u8, 2u8, 3u8]);
    tests::deserialize_test::<ByteBuf>(unode!(1, 2, 3), bytes);
    tests::deserialize_test(unode!(42), Some(42));
    tests::deserialize_test::<Option<u32>>(unode!(), None);
    tests::deserialize_test::<()>(unode!(), ());
    tests::deserialize_test::<std::marker::PhantomData<u8>>(
        unode!(),
        std::marker::PhantomData,
    );
}

#[test]
fn deserialize_seq_test() {
    let arr = vec!["one", "two", "three"]
        .into_iter()
        .map(|i| i.to_string())
        .collect();
    tests::deserialize_test::<Vec<String>>(unode!("one", "two", "three"), arr);
    let arr = vec![1u8, 2u8, 3u8];
    tests::deserialize_test::<Vec<u8>>(unode!(vec![1u8, 2u8, 3u8]), arr);
}

#[test]
fn deserialize_tuple_test() {
    let tup = ("one".to_string(), 1, 3.18);
    tests::deserialize_test(unode!("one", 1, 3.18), tup);
}

#[test]
fn deserialize_tuple_struct_test() {
    #[derive(Deserialize, PartialEq, Debug)]
    struct Rgb(u8, u8, u8);

    let tup = Rgb(2, 3, 4);
    tests::deserialize_test(unode!(2u8, 3u8, 4u8), tup);
}

#[test]
fn deserialize_newtype_struct_test() {
    #[derive(Debug, Deserialize, PartialEq)]
    struct NewTypeStruct(u8);
    tests::deserialize_test(unode!(2u8), NewTypeStruct(2));

    #[derive(Debug, Deserialize, PartialEq)]
    enum NewTypeEnum {
        V(u8),
    }
    tests::deserialize_test(unode!("V" => 4u8), NewTypeEnum::V(4));
}

#[test]
fn deserialize_tuple_variant_test() {
    #[derive(Deserialize, Debug, PartialEq)]
    enum E {
        T(u8, u8),
        U(String, u32, u32),
    }
    let tup = E::T(2, 3);
    tests::deserialize_test(unode!("T" => unode![2u8, 3u8]), tup);
}

#[test]
fn deserialize_map_test() {
    #[derive(PartialEq, Eq, Hash, Deserialize, Debug)]
    struct Key(u8);
    let mut map = HashMap::new();
    map.insert(Key(1), "One".to_string());
    map.insert(Key(2), "Two".to_string());
    map.insert(Key(3), "Three".to_string());
    tests::deserialize_test(
        unode! {"1" => "One", "2" => "Two", "3" => "Three"},
        map,
    );
}

#[test]
fn deserialize_struct_test() {
    #[derive(Debug, Deserialize, PartialEq)]
    struct Point {
        x: i32,
        y: i32,
    }

    impl Point {
        fn new(x: i32, y: i32) -> Self {
            Self { x, y }
        }
    }

    #[derive(Debug, Deserialize, PartialEq)]
    struct Test {
        int: u32,
        float: f64,
        text: String,
        data: Vec<Point>,
        node: UniNode,
        bytes: Vec<u8>,
    }

    let data = Test {
        int: 42,
        float: 3.18,
        text: "hello".to_string(),
        data: vec![Point::new(1, 2), Point::new(2, 3), Point::new(3, 4)],
        node: 10.into(),
        bytes: vec![97, 98, 99],
    };

    let node = unode! {
        "int" => 42u32,
        "float" => 3.18,
        "text" => "hello",
        "data" => unode![
            unode!{"x" => 1, "y" => 2},
            unode!{"x" => 2, "y" => 3},
            unode!{"x" => 3, "y" => 4},
        ],
        "node" => 10,
        "bytes" => vec![97u8, 98u8, 99u8],
    };

    tests::deserialize_test(node, data);
}

#[test]
fn deserialize_enum_test() {
    use std::ops::Bound;
    tests::deserialize_test(unode!("Unbounded"), Bound::<usize>::Unbounded);
}

#[test]
fn deserialize_struct_variant_test() {
    #[derive(Deserialize, PartialEq, Debug)]
    enum E {
        S { r: u8, g: u8, b: u8 },
    }
    let data = E::S { r: 3, g: 5, b: 6 };
    let node = unode! {
        "S" => unode!{"r" => 3u8, "g" => 5u8, "b" => 6u8}
    };
    tests::deserialize_test(node, data);
}
