// yaml.rs
// Copyright: (c) 2015-2021, Oleg Lelenkov
// Distributed under terms of the BSD 3-Clause license.
//

use crate::value::{Array, Object, UniNode};
use super::{UniLoader, Result};
use super::error::{UniNodeLoadError, Marker};

use unicode_normalization::UnicodeNormalization;
use yaml_rust::{yaml, ScanError, Yaml, YamlLoader};

#[derive(Clone, Default)]
pub struct UniYamlLoader;

impl UniLoader for UniYamlLoader {
    fn extensions(&self) -> &[&str] {
        &["yaml", "yml"]
    }

    fn load(&self, text: &str) -> Result {
        parse(text)
    }
}

impl From<ScanError> for UniNodeLoadError {
    fn from(err: ScanError) -> Self {
        let m = err.marker();
        UniNodeLoadError::ParseError {
            mark: Marker::new(m.index(), m.line(), m.col()),
            info: err.to_string(),
        }
    }
}

fn parse(text: &str) -> Result {
    let mut root = YamlLoader::load_from_str(text)?;

    fn convert(node: &Yaml) -> Result {
        match node {
            Yaml::Null => Ok(UniNode::Null),
            Yaml::Boolean(val) => Ok(UniNode::Boolean(*val)),
            Yaml::Integer(val) => Ok(UniNode::Integer(*val)),
            Yaml::String(val) => Ok(UniNode::String(val.nfc().collect())),
            Yaml::Real(val) => Ok(UniNode::Float(val.parse::<f64>()?)),
            Yaml::Array(val) => {
                let mut data = Array::with_capacity(val.len());
                for node in val {
                    data.push(convert(node)?);
                }
                Ok(UniNode::Array(data))
            },
            Yaml::Hash(val) => {
                let mut data = Object::with_capacity(val.len());
                for (key, node) in val {
                    let key = key.as_str().ok_or_else(|| {
                        UniNodeLoadError::Conv("Non-string key".to_string())
                    })?;
                    data.insert(key.to_string(), convert(node)?);
                }
                Ok(UniNode::Object(data))
            },
            Yaml::Alias(_) => {
                Err(UniNodeLoadError::Conv(
                    "Yaml alias, not fully supported yet".to_string(),
                ))
            },
            _ => Err(UniNodeLoadError::Conv("Yaml invalid type".to_string())),
        }
    }

    let root = match root.len() {
        0 => Yaml::Hash(yaml::Hash::new()),
        1 => std::mem::replace(&mut root[0], Yaml::Null),
        n => {
            return Err(UniNodeLoadError::ParseError {
                mark: Marker::new(0, 0, 0),
                info: format!("Got {} YAML documents, expected 1", n),
            })
        },
    };

    convert(&root)
}

#[test]
fn loaders_yaml_test() {
    let text = r#"
            client:
                host: "localhost"
                port: 404
                can: 2001-12-15T02:59:43.1Z
                collection:
                - "one"
                - "two"
        "#;

    let node = parse(text).unwrap();
    assert_eq!(node.get_int("client.port").unwrap(), 404);
    assert_eq!(node.get_str("client.host").unwrap(), "localhost");
}
