// mod.rs
// Copyright: (c) 2015-2021, Oleg Lelenkov
// Distributed under terms of the BSD 3-Clause license.
//

mod error;
#[cfg(feature = "toml")]
mod toml;
#[cfg(feature = "yaml")]
mod yaml;

#[cfg(test)]
use crate::unode;

pub use error::UniNodeLoadError;

use std::collections::HashMap;
use std::sync::RwLock;
use std::path::Path;

use once_cell::sync::Lazy;
use crate::value::UniNode;

#[cfg(feature = "yaml")]
use yaml::UniYamlLoader;

#[cfg(feature = "toml")]
use toml::UniTomlLoader;

pub type Result = std::result::Result<UniNode, UniNodeLoadError>;

pub trait UniLoader {
    fn extensions(&self) -> &[&str];

    fn load(&self, text: &str) -> Result;
}

pub trait FileReader {
    fn read(&self, path: &Path) -> std::io::Result<String>;

    fn exists(&self, path: &Path) -> bool;
}

struct StdFileReader;

impl FileReader for StdFileReader {
    fn read(&self, path: &Path) -> std::io::Result<String> {
        use std::fs::File;
        use std::io::Read;

        let mut file = File::open(path)?;
        let mut contents = String::new();
        file.read_to_string(&mut contents)?;
        Ok(contents)
    }

    fn exists(&self, path: &Path) -> bool {
        path.is_file()
    }
}

type GlobalFileReader = Box<dyn FileReader + Sync + Send>;

static READER: Lazy<RwLock<GlobalFileReader>> =
    Lazy::new(|| RwLock::new(Box::new(StdFileReader {})));

pub fn set_default_reader<R>(reader: R)
where
    R: FileReader + Send + Sync + 'static,
{
    let reader = Box::new(reader);
    *(READER.write().unwrap()) = reader;
}

type GlobalUniLoader = Box<dyn UniLoader + Sync + Send>;
type LoadersMap = HashMap<String, GlobalUniLoader>;

fn add_loader<L: UniLoader + Clone + Sync + Send + 'static>(
    cont: &mut LoadersMap, loader: L,
) {
    for ext in loader.extensions() {
        log::info!("register loader: '{}'", ext);
        cont.insert(ext.to_string(), Box::new(loader.clone()));
    }
}

static LOADERS: Lazy<RwLock<LoadersMap>> = Lazy::new(|| {
    let mut cont = LoadersMap::new();
    #[cfg(feature = "yaml")]
    add_loader(&mut cont, UniYamlLoader::default());
    #[cfg(feature = "toml")]
    add_loader(&mut cont, UniTomlLoader::default());
    RwLock::new(cont)
});

pub fn register_loader<L: UniLoader + Clone + Sync + Send + 'static>(
    loader: L,
) {
    let mut loaders = LOADERS.write().unwrap();
    add_loader(&mut loaders, loader);
}

pub fn is_available_loader(ext: &str) -> bool {
    let loaders = LOADERS.read().unwrap();
    loaders.contains_key(ext)
}

#[test]
fn loaders_register_test() {
    #[derive(Clone)]
    struct TestLoader {}
    impl UniLoader for TestLoader {
        fn extensions(&self) -> &[&str] {
            &["tst", "test"]
        }

        fn load(&self, _text: &str) -> Result {
            Ok(UniNode::Null)
        }
    }

    register_loader(TestLoader {});

    assert!(is_available_loader("tst"));
    assert!(is_available_loader("test"));
    assert!(!is_available_loader("any"));
}

#[test]
fn loaders_register_default() {
    #[cfg(feature = "yaml")]
    {
        assert!(is_available_loader("yaml"));
        assert!(is_available_loader("yml"));
    }

    #[cfg(feature = "toml")]
    assert!(is_available_loader("toml"));
}

fn load_uninode<F, R>(ext: &str, text_fn: F) -> Result
where
    F: FnOnce() -> std::result::Result<R, UniNodeLoadError>,
    R: AsRef<str>,
{
    let loaders = LOADERS.read().unwrap();
    if !loaders.contains_key(ext) {
        return Err(UniNodeLoadError::NotFoundLoader(ext.to_string()));
    }
    loaders[ext].load(text_fn()?.as_ref())
}

pub fn from_string<T: AsRef<str>>(ext: &str, text: T) -> Result {
    load_uninode(ext, move || Ok(text))
}

#[test]
fn loaders_from_string() {
    let node = from_string("toml", "[node]").unwrap();
    assert_eq!(node, unode!("node" => UniNode::empty_object()));
}

fn from_reader<P>(ext: &str, path: P, reader: &GlobalFileReader) -> Result
where
    P: AsRef<Path>,
{
    load_uninode(ext, move || Ok(reader.read(path.as_ref())?))
}

pub fn from_file<P: AsRef<Path>>(path: P) -> Result {
    let path = path.as_ref();
    let ext = path.extension().and_then(|e| e.to_str()).ok_or_else(|| {
        UniNodeLoadError::FileExtension(path.display().to_string())
    })?;

    let reader = READER.read().unwrap();
    from_reader(ext, path, &reader)
}

pub fn from_file_base<P: AsRef<Path>>(base_path: P) -> Result {
    let loaders = LOADERS.read().unwrap();
    let reader = READER.read().unwrap();
    let base_path = base_path.as_ref();

    for ext in loaders.keys() {
        let full_path = base_path.with_extension(ext);
        if reader.exists(&full_path) {
            return from_reader(ext, full_path, &reader);
        }
    }

    return Err(UniNodeLoadError::NotFoundBaseLoader(
        base_path.display().to_string(),
    ));
}

#[test]
fn loaders_from_file() {
    struct SimpleReader;
    impl FileReader for SimpleReader {
        fn read(&self, _path: &Path) -> std::io::Result<String> {
            Ok("val = 1".to_string())
        }

        fn exists(&self, path: &Path) -> bool {
            path == Path::new("text.toml")
        }
    }
    set_default_reader(SimpleReader {});
    assert_eq!(from_file("text.toml").unwrap(), unode! { "val" => 1 });
    assert_eq!(from_file_base("text.json").unwrap(), unode! { "val" => 1 });
    assert_eq!(from_file_base("text").unwrap(), unode! { "val" => 1 });
}
