// mod.rs
// Copyright: (c) 2015-2021, Oleg Lelenkov
// Distributed under terms of the BSD 3-Clause license.
//

mod de;
mod error;
mod ser;

pub use ser::UniNodeSerializer;
pub use error::UniNodeSerError;

use error::Result;
use crate::value::UniNode;

pub fn serialize<T>(value: &T) -> Result<UniNode>
where
    T: serde::Serialize,
{
    value.serialize(UniNodeSerializer)
}

#[test]
fn serialize_bytes_test() {
    use serde::{Serialize, Deserialize, de::DeserializeOwned};
    use std::fmt::Debug;
    use serde_bytes::Bytes;

    fn test_owned_data<T>(data: T)
    where
        T: Serialize + DeserializeOwned + Debug + PartialEq,
    {
        let node = serialize(&data).unwrap();
        assert_eq!(T::deserialize(&node).unwrap(), data);
        assert_eq!(T::deserialize(node).unwrap(), data);
    }

    test_owned_data(vec![1, 2, 3]);
    test_owned_data(String::from("hello"));

    let data = "abc";
    let node = serialize(&data).unwrap();
    assert_eq!(<&str>::deserialize(&node).unwrap(), data);

    let data = Bytes::new(b"abc");
    let node = serialize(&data).unwrap();
    assert_eq!(<&Bytes>::deserialize(&node).unwrap(), data);

    #[derive(Debug, PartialEq, Serialize, Deserialize)]
    struct Data<'a> {
        #[serde(with = "serde_bytes")]
        data_slice: &'a [u8],
        #[serde(with = "serde_bytes")]
        data_buf: Vec<u8>,
    }
    let data = Data {
        data_slice: b"abc",
        data_buf: vec![1, 2, 3],
    };
    let node = serialize(&data).unwrap();
    assert_eq!(Data::deserialize(&node).unwrap(), data);
}
