// toml.rs
// Copyright: (c) 2015-2021, Oleg Lelenkov
// Distributed under terms of the BSD 3-Clause license.
//

use crate::value::{Array, Object, UniNode};
use super::{UniLoader, Result};
use super::error::{UniNodeLoadError, Marker};

use unicode_normalization::UnicodeNormalization;
use toml_rs::{self as toml, Value};

#[derive(Clone, Default)]
pub struct UniTomlLoader;

impl UniLoader for UniTomlLoader {
    fn extensions(&self) -> &[&str] {
        &["toml"]
    }

    fn load(&self, text: &str) -> Result {
        parse(text)
    }
}

impl From<toml::de::Error> for UniNodeLoadError {
    fn from(err: toml::de::Error) -> UniNodeLoadError {
        let (line, col) = err.line_col().unwrap_or((0, 0));
        UniNodeLoadError::ParseError {
            mark: Marker::new(0, line, col),
            info: err.to_string(),
        }
    }
}

fn parse(text: &str) -> Result {
    let root = toml::from_str(text)?;

    fn convert(node: &Value) -> Result {
        match node {
            Value::Boolean(val) => Ok(UniNode::Boolean(*val)),
            Value::Integer(val) => Ok(UniNode::Integer(*val)),
            Value::Float(val) => Ok(UniNode::Float(*val)),
            Value::String(val) => Ok(UniNode::String(val.nfc().collect())),
            Value::Array(val) => {
                let mut data = Array::with_capacity(val.len());
                for node in val {
                    data.push(convert(node)?);
                }
                Ok(UniNode::Array(data))
            },
            Value::Table(val) => {
                let mut data = Object::with_capacity(val.len());
                for (key, node) in val {
                    data.insert(key.clone(), convert(node)?);
                }
                Ok(UniNode::Object(data))
            },
            Value::Datetime(val) => Ok(UniNode::String(val.to_string())),
        }
    }

    convert(&root)
}

#[test]
fn loaders_toml_test() {
    let text = r#"
[client]
host = "localhost"
port = 404
can = 2001-12-15T02:59:43.1Z
collection = ["one", "two"]
"#;

    let node = parse(text).unwrap();
    assert_eq!(node.get_int("client.port").unwrap(), 404);
    assert_eq!(node.get_str("client.host").unwrap(), "localhost");
}
