// error.rs
// Copyright: (c) 2015-2021, Oleg Lelenkov
// Distributed under terms of the BSD 3-Clause license.
//

use std::num::ParseFloatError;
use thiserror::Error;

#[derive(Clone, Copy, PartialEq, Debug, Eq)]
pub struct Marker {
    index: usize,
    line: usize,
    col: usize,
}

impl Marker {
    pub fn new(index: usize, line: usize, col: usize) -> Marker {
        Marker { index, line, col }
    }

    pub fn index(&self) -> usize {
        self.index
    }

    pub fn line(&self) -> usize {
        self.line
    }

    pub fn col(&self) -> usize {
        self.col
    }
}

#[derive(Debug, Error)]
pub enum UniNodeLoadError {
    #[error(transparent)]
    Io(#[from] std::io::Error),
    #[error("Not found loader for format '{0}'")]
    NotFoundLoader(String),
    #[error("Not found loader for file '{0}'")]
    NotFoundBaseLoader(String),
    #[error("Failed extracts the extension of '{0}'")]
    FileExtension(String),
    #[error("{info} at line {} column {}", mark.line, mark.col)]
    ParseError { mark: Marker, info: String },
    #[error("{0}")]
    Conv(String),
}

impl From<ParseFloatError> for UniNodeLoadError {
    fn from(err: ParseFloatError) -> Self {
        Self::Conv(format!("Failed conversion to float: {}", err))
    }
}
