use cookie::Cookie;
use reqwest::{Client, RequestBuilder, StatusCode};
use serde::{Deserialize, Serialize};

use crate::UnifiedError;

#[derive(Serialize)]
struct Credentials {
    username: String,
    password: String,
    remember: bool,
    strict: bool,
}

#[derive(Deserialize)]
pub(crate) struct Response<T> {
    pub(crate) data: T,
}

#[derive(Debug)]
pub struct Unified {
    pub host: String,
    pub token: String,
}

impl Unified {
    pub fn from_token(host: &str, token: &str) -> Unified {
        Unified {
            host: host.to_string(),
            token: token.to_string(),
        }
    }

    pub async fn from_credentials(
        host: &str,
        username: &str,
        password: &str,
    ) -> Result<Unified, UnifiedError> {
        let host = host.to_string();
        let credentials = Credentials {
            username: username.to_string(),
            password: password.to_string(),
            remember: true,
            strict: true,
        };

        let client = reqwest::Client::new();
        let response = client
            .post("https://unifi.appscho.com/api/login")
            .json(&credentials)
            .send()
            .await?;

        if response.status() != StatusCode::OK {
            return Err(UnifiedError::InvalidCredentials);
        }

        let cookie = response
            .headers()
            .get_all("set-cookie")
            .into_iter()
            .map(|cookie| Cookie::parse(cookie.to_str().unwrap_or_default()).ok())
            .flatten()
            .find(|cookie| cookie.name() == "unifises");

        match cookie {
            Some(cookie) => Ok(Unified {
                host,
                token: cookie.value().to_owned(),
            }),

            None => Err(UnifiedError::Unknown),
        }
    }

    pub(crate) fn request(&self, path: &str) -> RequestBuilder {
        let client = Client::new();

        client
            .get(&format!("{}{}", self.host, path))
            .header("cookie", &format!("unifises={}", self.token))
    }
}
