use std::str::FromStr;

use ipnet::IpNet;
use serde::Deserialize;

use crate::{unified::Response, Unified, UnifiedError};

#[derive(Debug)]
pub struct Network {
    pub id: String,
    pub name: String,
    pub enabled: bool,
    pub purpose: String,
    pub subnet: Option<IpNet>,
    pub domain: Option<String>,
    pub vlan: Option<u16>,
}

#[derive(Deserialize)]
pub struct RemoteNetwork {
    #[serde(rename = "_id")]
    pub id: String,
    pub name: String,
    #[serde(default = "crate::util::is_true")]
    pub enabled: bool,
    pub purpose: String,
    pub ip_subnet: Option<String>,
    pub domain_name: Option<String>,
    pub vlan: Option<String>,
}

impl Unified {
    pub async fn networks(&self, site: &str) -> Result<Vec<Network>, UnifiedError> {
        let response = self
            .request(&format!("/api/s/{}/rest/networkconf", site))
            .send()
            .await?
            .json::<Response<Vec<RemoteNetwork>>>()
            .await?;

        let networks = response
            .data
            .into_iter()
            .map(|network| {
                let subnet = network
                    .ip_subnet
                    .map(|ip| IpNet::from_str(&ip).ok())
                    .flatten();

                Network {
                    id: network.id,
                    name: network.name,
                    enabled: network.enabled,
                    subnet,
                    purpose: network.purpose,
                    domain: network.domain_name,
                    vlan: network.vlan.map(|vlan| u16::from_str(&vlan).ok()).flatten(),
                }
            })
            .collect();

        Ok(networks)
    }
}
