use std::{net::IpAddr, str::FromStr, time::Duration};

use macaddr::MacAddr;
use num_derive::FromPrimitive;
use num_traits::FromPrimitive;
use serde::Deserialize;

use crate::{unified::Response, Unified, UnifiedError};

#[derive(Debug)]
pub struct Device {
    pub id: String,
    pub name: String,
    pub model: String,
    pub mac: MacAddr,
    pub ip: Option<IpAddr>,
    pub version: String,
    pub upgradable: bool,
    pub state: DeviceState,
    pub uptime: Duration,
    pub rx_bytes: u64,
    pub tx_bytes: u64,
}

#[derive(Deserialize)]
struct RemoteDevice {
    #[serde(rename = "_id")]
    id: String,
    name: String,
    model: String,
    mac: String,
    ip: String,
    #[serde(default)]
    network_table: Vec<Network>,
    version: String,
    upgradable: bool,
    state: u32,
    uptime: u64,
    rx_bytes: u64,
    tx_bytes: u64,
}

#[derive(Deserialize)]
pub struct Network {
    #[serde(rename = "attr_no_delete", default)]
    persistent: bool,
    ip: String,
}

#[derive(Debug, FromPrimitive)]
pub enum DeviceState {
    Disconnected = 0,
    Connected = 1,
    PendingAdoption = 2,
    PendingUpgrade = 3,
    Upgrading = 4,
    Provisionning = 5,
    HeartbeatMissed = 6,
    Adopting = 7,
    Deleting = 8,
    InformError = 9,
    AdoptionRequired = 10,
    AdoptionFailed = 11,
    Isolated = 12,
    RFScanning = 13,
    ManagedByOther = 14,
    Unknown = 15,
}

impl Unified {
    pub async fn devices(&self, site: &str) -> Result<Vec<Device>, UnifiedError> {
        let response = self
            .request(&format!("/api/s/{}/stat/device", site))
            .send()
            .await?
            .json::<Response<Vec<RemoteDevice>>>()
            .await?;

        let devices = response
            .data
            .into_iter()
            .map(|device| {
                let ip = if !device.network_table.is_empty() {
                    device.network_table.into_iter().find_map(|network| {
                        if network.persistent {
                            IpAddr::from_str(&network.ip).ok()
                        } else {
                            None
                        }
                    })
                } else {
                    IpAddr::from_str(&device.ip).ok()
                };

                let state = FromPrimitive::from_u32(device.state).unwrap_or(DeviceState::Unknown);

                Device {
                    id: device.id,
                    name: device.name,
                    model: device.model,
                    mac: MacAddr::from_str(&device.mac).unwrap(),
                    ip,
                    version: device.version,
                    upgradable: device.upgradable,
                    state,
                    uptime: Duration::from_secs(device.uptime),
                    rx_bytes: device.rx_bytes,
                    tx_bytes: device.tx_bytes,
                }
            })
            .collect();

        Ok(devices)
    }
}
