use std::{net::IpAddr, str::FromStr};

use chrono::NaiveDateTime;
use macaddr::MacAddr;
use serde::Deserialize;

use crate::{unified::Response, Unified, UnifiedError};

#[derive(Debug)]
pub struct Client {
    pub name: Option<String>,
    pub mac: MacAddr,
    pub oui: String,
    pub hostname: String,
    pub ip: Option<IpAddr>,
    pub identity: Option<String>,
    pub last_seen: Option<NaiveDateTime>,
    pub wired: bool,
    pub guest: bool,
    pub authorized: bool,
    pub rx_bytes: u64,
    pub tx_bytes: u64,
    pub wired_rx_bytes: u64,
    pub wired_tx_bytes: u64,
}

#[derive(Deserialize)]
struct RemoteClient {
    name: Option<String>,
    mac: String,
    oui: String,
    hostname: String,
    ip: String,
    #[serde(rename = "1x_identity")]
    identity: Option<String>,
    last_seen: Option<i64>,
    is_wired: bool,
    is_guest: bool,
    authorized: bool,
    pub rx_bytes: u64,
    pub tx_bytes: u64,
    #[serde(rename = "wired-rx_bytes", default)]
    pub wired_rx_bytes: u64,
    #[serde(rename = "wired-tx_bytes", default)]
    pub wired_tx_bytes: u64,
}

impl Unified {
    pub async fn clients(&self, site: &str) -> Result<Vec<Client>, UnifiedError> {
        let response = self
            .request(&format!("/api/s/{}/stat/sta", site))
            .send()
            .await?
            .json::<Response<Vec<RemoteClient>>>()
            .await?;

        let clients = response
            .data
            .into_iter()
            .map(|client| {
                let seen = client
                    .last_seen
                    .map(|ts| NaiveDateTime::from_timestamp(ts, 0));

                Client {
                    name: client.name,
                    mac: MacAddr::from_str(&client.mac).unwrap(),
                    oui: client.oui,
                    hostname: client.hostname,
                    ip: IpAddr::from_str(&client.ip).ok(),
                    identity: client.identity,
                    last_seen: seen,
                    wired: client.is_wired,
                    guest: client.is_guest,
                    authorized: client.authorized,
                    rx_bytes: client.rx_bytes,
                    tx_bytes: client.tx_bytes,
                    wired_rx_bytes: client.wired_rx_bytes,
                    wired_tx_bytes: client.wired_tx_bytes,
                }
            })
            .collect();

        Ok(clients)
    }
}
