/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

use std::{
    fs::File,
    io::Write,
    path::{Path, PathBuf},
};

use anyhow::{Context, Result};

pub mod gen_gecko_js;
mod webidl;
pub use gen_gecko_js::{
    Config, Interface, InterfaceHeader, Namespace, NamespaceHeader, SharedHeader, WebIDL,
};

use super::super::interface::ComponentInterface;

pub struct Binding {
    name: String,
    contents: String,
}

/// Generate uniffi component bindings for Firefox.
///
/// Firefox's WebIDL binding declarations, generated by `Codegen.py` in m-c,
/// expect to find a `.h`/`.cpp` pair per interface, even if those interfaces
/// are declared in a single WebIDL file. Dictionaries and enums are
/// autogenerated by `Codegen.py`, so we don't need to worry about them...but
/// we do need to emit serialization code for them, plus the actual interface
/// and top-level function implementations, in the UniFFI bindings.
///
/// So the Gecko backend generates:
///
/// * A single WebIDL file with the component interface. This is similar to the
///   UniFFI UDL format, but the names of some types are different.
/// * A shared C++ header, with serialization helpers for all built-in and
///   interface types.
/// * A header and source file for the namespace, if the component defines any
///   top-level functions.
/// * A header and source file for each `interface` declaration in the UDL.
///
/// These files should be checked in to the Firefox source tree. The WebIDL
/// file goes in `dom/chrome-webidl`, and the header and source files can be
/// added to any directory and referenced in `moz.build`. The Rust component
/// library must also be added as a dependency to `gkrust-shared` (in
/// `toolkit/library/rust/shared`), so that the FFI symbols are linked into
/// libxul.
pub fn write_bindings(
    config: &Config,
    ci: &ComponentInterface,
    out_dir: &Path,
    _try_format_code: bool,
    _is_testing: bool,
) -> Result<()> {
    let out_path = PathBuf::from(out_dir);
    let bindings = generate_bindings(config, ci)?;
    for binding in bindings {
        let mut file = out_path.clone();
        file.push(&binding.name);
        let mut f = File::create(&file)
            .with_context(|| format!("Failed to create file `{}`", binding.name))?;
        write!(f, "{}", binding.contents)?;
    }
    Ok(())
}

/// Generate Gecko bindings for the given ComponentInterface, as a string.
pub fn generate_bindings(config: &Config, ci: &ComponentInterface) -> Result<Vec<Binding>> {
    use askama::Template;

    let mut bindings = Vec::new();

    let context = gen_gecko_js::Context::new(config, ci);

    let webidl = WebIDL::new(context, ci)
        .render()
        .context("Failed to render WebIDL bindings")?;
    bindings.push(Binding {
        name: format!("{}.webidl", context.header_name(context.namespace())),
        contents: webidl,
    });

    let shared_header = SharedHeader::new(context, ci)
        .render()
        .context("Failed to render shared header")?;
    bindings.push(Binding {
        name: format!("{}Shared.h", context.header_name(context.namespace())),
        contents: shared_header,
    });

    // Top-level functions go in one namespace, which needs its own header and
    // source file.
    let functions = ci.iter_function_definitions();
    if !functions.is_empty() {
        let header = NamespaceHeader::new(context, functions.as_slice())
            .render()
            .context("Failed to render top-level namespace header")?;
        bindings.push(Binding {
            name: format!("{}.h", context.header_name(context.namespace())),
            contents: header,
        });

        let source = Namespace::new(context, functions.as_slice())
            .render()
            .context("Failed to render top-level namespace binding")?;
        bindings.push(Binding {
            name: format!("{}.cpp", context.header_name(context.namespace())),
            contents: source,
        });
    }

    // Now generate one header/source pair for each interface.
    let objects = ci.iter_object_definitions();
    for obj in objects {
        let header = InterfaceHeader::new(context, &obj)
            .render()
            .with_context(|| format!("Failed to render {} header", obj.name()))?;
        bindings.push(Binding {
            name: format!("{}.h", context.header_name(obj.name())),
            contents: header,
        });

        let source = Interface::new(context, &obj)
            .render()
            .with_context(|| format!("Failed to render {} binding", obj.name()))?;
        bindings.push(Binding {
            name: format!("{}.cpp", context.header_name(obj.name())),
            contents: source,
        })
    }

    Ok(bindings)
}
