
use postgres::{NoTls};
use r2d2_postgres::PostgresConnectionManager;
use r2d2::{Pool, PooledConnection};
use parking_lot::{ RwLock};
use std::time::Duration;
use std::collections::HashMap;
use unidb::{get_db_properties};
use unidb::build_sql;

pub mod pg_dao;

#[macro_use]
extern crate lazy_static;
#[macro_use]
extern crate log;

lazy_static!{

    static ref PG_POOL: RwLock<Pool<PostgresConnectionManager<NoTls>>> = {
        RwLock::new(create_postgres_connection_manager())
    };
    static ref DATA_TYPE_CACHE: HashMap<String, String> = {

        let sql = build_sql("select_table_columns", "", &HashMap::new());

        let map = match connect_db() {
            Ok(mut conn) => {

                let mut map : HashMap<String, String> = HashMap::new();

                for row in conn.query(sql.as_str(), &[]).unwrap() {

                    let table_name : &str = row.get("table_name");
                    let column_name : &str = row.get("column_name");
                    let udt_name : &str = row.get("udt_name");

                    map.insert(format!("{}.{}", table_name, column_name), udt_name.to_string());
                }
                map
            }
            Err(e) => {
                println!("DATA_TYPE_CACHE error: {}", e);
                HashMap::new()
            }
        };
        map
    };
}
pub fn connect_db() -> Result<PooledConnection<PostgresConnectionManager<NoTls>>, r2d2::Error>{

    let conn = PG_POOL.read();
    return match conn.get() {

        Ok(t) => Ok(t),
        Err(e) => {
            error!("==> {:?}", e);
            drop(conn);

            let mut conn = PG_POOL.write();
            *conn = create_postgres_connection_manager();
            conn.get()
        }
    };
}
pub fn create_postgres_connection_manager() -> Pool<PostgresConnectionManager<NoTls>> {

    let properties = get_db_properties();

    let pool_size = match properties.get("db.pgsql.pool_size") {
        Some(t) =>   t.parse::<u32>().unwrap(),
        None => panic!("No such property: db.pgsql.pool_size")
    };
    let idle_timeout_sec = match properties.get("db.pgsql.idle_timeout_sec") {
        Some(t) =>   t.parse::<u64>().unwrap_or(15),
        None => 15,
    };

    let manager = PostgresConnectionManager::new( get_data_source().parse().unwrap(),NoTls,);
    let pool = r2d2::Pool::builder()
        .idle_timeout(Some(Duration::from_secs(idle_timeout_sec)))
        .max_size(pool_size)
        .build(manager).unwrap();
    pool
}

fn get_data_source() -> String{

    let properties = get_db_properties();

    let data_source
        = format!("host={} port={} user={} password={} dbname={}",
                  match properties.get("db.pgsql.host") {
                      Some(v) => v,
                      _ => panic!("No such property: db.pgsql.host"),
                  },
                  match properties.get("db.pgsql.port") {
                      Some(v) => v,
                      _ => panic!("No such property: db.pgsql.port"),
                  },
                  match properties.get("db.pgsql.user") {
                      Some(v) => v,
                      _ => panic!("No such property: db.pgsql.user"),
                  },
                  match properties.get("db.pgsql.password") {
                      Some(v) => v,
                      _ => panic!("No such property: db.pgsql.password"),
                  },
                  match properties.get("db.pgsql.dbname") {
                      Some(v) => v,
                      _ => panic!("No such property: db.pgsql.dbname"),
                  }
    );
    data_source
}

pub fn get_table_column(){

    // println!("user_api_auth.update_datetime={:?}", DATA_TYPE_CACHE.get("user_api_auth.update_datetime"));
    // println!("user_api_auth.update_datetime={:?}", SQL_CACHE.get("select_table_columns"));
}
#[cfg(test)]
mod tests {
    #[test]
    fn it_works() {
        assert_eq!(2 + 2, 4);
    }
}
