use serde::{Serialize};
use std::fmt::{Display, Formatter, Result as FmtResult};
use serde_json::{to_string_pretty};
use std::collections::HashMap;
use postgres::Transaction;
use unidb::model::Value;
use unidb::build_sql;
use crate::connect_db;

#[derive(Debug, Serialize)]
pub enum PgDaoError {

    ConnectionError(String),

    ExecutionError(String),

    TransactionCreationError(String),

    CommitError(String),

    RowError(String),

    Error(String),

}

impl Display for PgDaoError {
    fn fmt(&self, f: &mut Formatter) -> FmtResult {
        let error = to_string_pretty(self).unwrap();
        error!("{}", &error);
        write!(f, "{}", error)
    }
}

pub struct UpsertMapInput {
    pub sql_id: String,
    pub data: HashMap<String, Value>,
}
fn execute_uncommitted(sql_id: &str, transaction: &mut Transaction, item: &HashMap<String, Value>) -> Result<usize, PgDaoError>  {
    let sql = build_sql(sql_id, "", item);
    // info!("{}",sql);
    Ok(match transaction.execute(sql.as_str(), &[]) {
        Ok(t) => t as usize,
        Err(e) => {
            error!("{}\n{:?}", sql, e);
            return Err(PgDaoError::ExecutionError(e.to_string()))
        }
    })
}
pub fn execute_multi_tables_dao(input: Vec<UpsertMapInput>) -> Result<usize, PgDaoError>{

    let count = match connect_db() {
        Ok(mut conn) => {
            let mut transaction =
                match conn.transaction() {
                    Ok(t) => t,
                    Err(e) => {
                        error!("{:?}", e);
                        return Err(PgDaoError::TransactionCreationError(e.to_string()))
                    }
                };

            let mut count = 0;
            for item in input {
                count = count + execute_uncommitted(&item.sql_id, &mut transaction, &item.data)?;
            }

            match &transaction.commit() {
                Ok(_t) => {},
                Err(e) => {
                    error!("{:?}", e);
                    return Err(PgDaoError::CommitError(e.to_string()))
                }
            }
            count
        }
        Err(e) => {
            error!("{:?}", e);
            return Err(PgDaoError::ConnectionError(e.to_string()))
        }
    };
    Ok(count)
}

pub fn execute_dao(sql_id: &str, item: &HashMap<String, Value>) -> Result<usize, PgDaoError>{

    let count = match connect_db() {
        Ok(mut conn) => {
            let mut count = 0;
            let mut transaction =
                match conn.transaction() {
                    Ok(t) => t,
                    Err(e) => {
                        error!("{:?}", e);
                        return Err(PgDaoError::TransactionCreationError(e.to_string()))
                    }
                };
            match execute_uncommitted(sql_id, &mut transaction, item) {
                Ok(t) => {
                    count = count + t;
                    count
                },
                Err(e) => {
                    error!("{:?}", e);
                    &transaction.rollback().unwrap_or_default();
                    return Err(e);
                }
            };
            match &transaction.commit() {
                Ok(_t) => {},
                Err(e) => {
                    error!("{:?}", e);
                    return Err(PgDaoError::CommitError(e.to_string()))
                }
            }
            count
        }
        Err(e) => {
            error!("{:?}", e);
            return Err(PgDaoError::ConnectionError(e.to_string()))
        }
    };
    Ok(count)
}
pub fn execute_multi_entries_dao(sql_id: &str, list : Vec<HashMap<String, Value>>) -> Result<usize, PgDaoError>{

    let count = match connect_db() {
        Ok(mut conn) => {
            let mut transaction =
                match conn.transaction() {
                    Ok(t) => t,
                    Err(e) => {
                        error!("{:?}", e);
                        return Err(PgDaoError::TransactionCreationError(e.to_string()))
                    }
                };
            let mut count = 0;
            for item in list {
                match execute_uncommitted(sql_id, &mut transaction, &item) {
                    Ok(t) => {
                        // info!("upsert_multi_map_dao: {}", t);
                        count = count + t;
                    },
                    Err(e) => {
                        error!("{:?}", e);
                        &transaction.rollback().unwrap_or_default();
                        return Err(e);
                    }
                };
            }
            match &transaction.commit() {
                Ok(_t) => {},
                Err(e) => {
                    error!("{:?}", e);
                    return Err(PgDaoError::CommitError(e.to_string()))
                }
            }
            count
        }
        Err(e) => {
            error!("{:?}", e);
            return Err(PgDaoError::ConnectionError(e.to_string()))
        }
    };
    Ok(count)
}